Program Picdraw;

{$UNDEF DEBUG}

uses
  crt,               {common/general stuff}
  PD_Opcodes,        {all the opcodes and related functions}
  PD_DrawPrimitives; {drawing primitives--this was put in a seperate
                      file so that I could use any drawing method or library I
                      wanted to without cluttering up this code.
                      I use VGA's Mode X (320x200x256) for lowres, and will probably use
                      VESA (640x480) for high-res.}

type
  interpret_type=(interpret_only,draw);
  {
  interpret_only:  Just print a listing of commands and data; used for debugging
  draw:            Draw the picture normally
  }

const
  {user options}
  drawmode:interpret_type=draw;
  simulate_apple_speed:boolean=true;

  picdata:pointer=nil;
  picsize:word=0; {total size of picture file}
  loc:word=0; {offset into picture file}

procedure Pause_for_user;
begin
  while keypressed do readkey; {empty keyboard buffer}
  readkey;
end;

Procedure Wait_for_VertRetrace;assembler;
asm
  mov  dx,03dah
@l1:
  in   al,dx
  test al,8
  jz   @l1
@l2:
  in   al,dx
  test al,8
  jnz  @l2
end;

procedure chirp;
begin
  sound(400);
  delay(100);
  sound(600);
  delay(100);
  sound(800);
  delay(100);
  nosound;
end;

procedure chirp_bad;
begin
  sound(100);
  delay(50);
  nosound;
end;

Procedure fatalerror(s:String);
{allows a means of reporting errors and stopping the system}
Begin
  asm
    mov  ax,03h
    int  10h
  end;
  TextColor(lightRed);
  WriteLn('Fatal error: ',s);
  writeln('(Errors can be reported to trixter@hornet.org, if necessary.)');
  Halt;
End;

Function FileExists (filename:String):Boolean;
Var
  f:File;
Begin
  Assign(f,filename);
  {$I-}
  Reset(f);
  {Close(f);}
  {$I+}
  FileExists:=(IOResult = 0);
End;

Procedure PD_Init;
{Initialize ourselves and our display mode}

var
  ch:char;

  procedure display_options;
  begin
    clrscr;
    writeln('Picdraw Interpreter v0.1  --  Jim Leonard (trixter@hornet.org)');
    writeln;
    writeln('Filename: ',paramstr(1));
    writeln;
    writeln('Interpret Options:  (hit number keys to toggle)');
    writeln;
    write('1. Interpret mode:        ');
    if drawmode=draw
      then writeln('Interpret and Draw')
      else writeln('Interpret only');

    if drawmode=draw then begin
    writeln('2. Simulate Apple Speed:  ',simulate_apple_speed);
    writeln;
    writeln('Draw Options:  (hit letter keys to toggle)');
    writeln;
    write('a. Graphic mode:          ');
    if graphmode=modex
      then writeln('320x200x256')
      else writeln('VESA');
    write('b. Line Thickness:        ');
    case thickness of
      normal:writeln('Normal');
      thick:writeln('Wide');
      doublethick:writeln('Double-Thick');
    end;
    writeln('c. Pattern Fills Enabled: ',filling);
    writeln('d. Line Color Enabled:    ',color);
    if graphmode=vesa
      then begin
        write('e. VESA resolution:       ');
        case vesa_multiplier of
          1:writeln('320x200');
          2:writeln('640x480');
          3:writeln('800x600');
          4:writeln('1024x768');
        end;
      end;
    end;

    writeln;
    writeln('Hit ENTER to start.');
  end;

begin
  clrscr;

  if not fileexists(paramstr(1))
    then fatalerror(paramstr(1)+' was not found.');

  repeat
    display_options;
    ch:=upcase(readkey);
    case ch of
      '1':if drawmode=draw
            then drawmode:=interpret_only
            else drawmode:=draw;
      '2':simulate_apple_speed:=not simulate_apple_speed;
      'A':if graphmode=modex
            then graphmode:=vesa
            else graphmode:=modex;
      'B':begin
            if thickness=normal
              then thickness:=thick
              else if thickness=thick
                then thickness:=doublethick
                else if thickness=doublethick
                  then thickness:=normal;
          end;
      'C':filling:=not filling;
      'D':color:=not color;
      'E':if vesa_multiplier<4
            then inc(vesa_multiplier)
            else vesa_multiplier:=1;
    end;
    if filling then color:=true;
  until ch=#13;

  {actually init}

  case drawmode of
    interpret_only:clrscr;
    draw:pd_graphinit;
  end;
end;

Function hex(c:Word):String;
{decimal to hex string}
Const
  hexs:String[16]='0123456789ABCDEF';
Var
  s:String;
Begin
  s:='';
  If c=0 Then s:='0' Else
    While c<>0 Do Begin
      s:=hexs[c And $f+1]+s;
      c:=c ShR 4;
    End;
  hex:=s;
End;

procedure Update_Current_Point(x:word;y:byte);
begin
  current_point.x:=x;
  current_point.y:=y;
end;

procedure Update_Text_Cursor(x:word;y:byte);
begin
  text_cursor.x:=x;
  text_cursor.y:=y;
end;

procedure Advance_Text_Cursor;
begin
  inc(text_cursor.x,7);
end;

Function Perform_Opcode(var p:pointer):byte;
{performes the opcode at pointer, increments the pointer,
then returns the number of bytes the pointer advanced}
var
  size:byte;
  b:^byte;
  x:word;
  y:byte;

  function get_x_and_y(var _x:word;var _y:byte):boolean;
  begin
    (*
    The old, step-by-step way:

    _x:=(b^ and $0f);
    _x:=_x shl 8;
    inc(b);
    _x:=_x or b^;
    inc(b);
    _y:=b^;

    The more readable way:    *)

    _x:=(b^ and $0f) shl 8; inc(b); {move the opcode low nybble over to high word of x}
    _x:=_x or b^;           inc(b); {next byte is lower word of x}
    _y:=b^;                         {next byte is y byte, then we're done}

    {Yeah, yeah; I know the above would look much better in assembler,
    but I don't hink non-asm people would be able to understand it.}

    {Now we test for out-of-bound wacko stuff, because if it's out
    of bounds, somebody made a mistake.}

    if (_x>apple_xmax) or (_y>apple_ymax)
      then get_x_and_y:=false
      else get_x_and_y:=true;
  end;

begin
  size:=0;
  b:=p; {can't typecast pointers in pascal, so point a "byte" pointer to the real one to access the value as a byte}

  case opcode(p) of

    not_yet_implemented:begin
      size:=1;
      case drawmode of
        interpret_only:begin
          writeln('Not yet implemented: ',hex(b^));
          chirp_bad;
          pause_for_user;
        end;
      end;
    end;

    reserved:begin
      size:=1;
      case drawmode of
        interpret_only:begin
          writeln('Reserved opcode: ',hex(b^));
          chirp_bad;
          pause_for_user;
        end;
      end;
    end;

    end_of_picture:begin
      size:=1;
      case drawmode of
        interpret_only:begin
          writeln('End of Picture');
        end;
      end;
    end;

    set_line_color:begin
      size:=1;
      line_color:=b^ and $0F;
      case drawmode of
        interpret_only:begin
          writeln('Set line color (',hex(b^),') -- Line color now ',line_color);
        end;
      end;
    end;

    set_brush_number:begin
      size:=1;
      brush_number:=b^ and $0F;
      case drawmode of
        interpret_only:begin
          writeln('Set brush number (',hex(b^),') -- Brush number now ',brush_number);
        end;
      end;
    end;

    set_text_cursor:begin
      size:=3;
      if get_x_and_y(x,y) then begin
        update_text_cursor(x,y);
        case drawmode of
          interpret_only:begin
            writeln('Set text cursor -- Text cursor now ',text_cursor.x,',',text_cursor.y);
          end;
        end;
      end
      else chirp_bad;
    end;

    set_point:begin
      size:=3;
      if get_x_and_y(x,y) then begin
        update_current_point(x,y);
        case drawmode of
          interpret_only:begin
            writeln('Set point -- Current point now ',current_point.x,',',current_point.y);
          end;
        end;
      end
      else chirp_bad;
    end;

    set_fill_color:begin
      size:=2;
      inc(b);
      fill_color:=b^;
      case drawmode of
        interpret_only:begin
          writeln('Set fill color -- Current fill color now ',fill_color);
        end;
      end;
    end;

    draw_circle:begin
      size:=2;
      inc(b);
      case drawmode of
        interpret_only:begin
          writeln('Draw circle at ',current_point.x,',',current_point.y,
                  ' with radius ',b^,
                  ' and color ',line_color);
        end;
        draw:pd_draw_circle(b^);
      end;
    end;

    wait:begin
      size:=2;
      inc(b);
      case drawmode of
        interpret_only:begin
          writeln('Wait for ',b^,' 1/16ths of a second');
        end;
        draw:delay(round(b^ * (1000/16)));
      end;
    end;

    xor_text_character_at_cursor:begin
      size:=2;
      inc(b);
      case drawmode of
        interpret_only:begin
          write('XOR character "',chr(b^),'" at ',text_cursor.x,',',text_cursor.y);
          Advance_text_cursor;
          writeln(' -- text cursor now ',text_cursor.x,',',text_cursor.y);
        end;
        draw:begin
          pd_xor_character(chr(b^));
          advance_text_cursor;
        end;
      end;
    end;

    color_text_character_at_cursor:begin
      size:=2;
      inc(b);
      case drawmode of
        interpret_only:begin
          write('Plot character "',chr(b^),'" at ',text_cursor.x,',',text_cursor.y,
                ' using fill color ',fill_color);
          Advance_text_cursor;
          writeln(' -- text cursor now ',text_cursor.x,',',text_cursor.y);
        end;
        draw:begin
          pd_put_character(chr(b^));
          advance_text_cursor;
        end;
      end;
    end;

    draw_box:begin
      size:=3;
      if get_x_and_y(x,y) then begin
        case drawmode of
          interpret_only:begin
            writeln('Draw box from ',current_point.x,',',current_point.y,
                    ' to point ',x,',',y,' using line color ',line_color);
          end;
          draw:pd_draw_box(x,y);
        end;
        update_current_point(x,y);
      end else chirp_bad;
    end;

    draw_line:begin
      size:=3;
      if get_x_and_y(x,y) then begin
        case drawmode of
          interpret_only:begin
            writeln('Draw line from ',current_point.x,',',current_point.y,
                    ' to point ',x,',',y,' using line color ',line_color);
          end;
          draw:pd_draw_line(x,y);
        end;
        update_current_point(x,y);
      end else chirp_bad;
    end;

    plot_brush:begin
      size:=3;
      if get_x_and_y(x,y) then begin
        case drawmode of
          interpret_only:begin
            writeln('Plot brush number ',brush_number,' at ',current_point.x,',',current_point.y,
                    ' using fill color ',fill_color);
          end;
          draw:pd_plot_brush;
        end;
        update_current_point(x,y);
        {$IFDEF DEBUG}
        pause_for_user;
        {$ENDIF}
      end else chirp_bad;
    end;

    fill:begin
      size:=3;
      if get_x_and_y(x,y) then begin
        case drawmode of
          interpret_only:begin
            writeln('Fill at ',current_point.x,',',current_point.y,
                    ' using fill color ',fill_color);
          end;
          draw:pd_fill;
        end;
        update_current_point(x,y);
      end else chirp_bad;
    end;

    else begin
      size:=1;
      case drawmode of
        interpret_only:begin
          writeln('Unknown opcode: ',hex(b^));
          chirp_bad;
          pause_for_user;
        end;
      end;
    end;

  end;
  inc(longint(p),size);
  Perform_Opcode:=size;
end;

Procedure PD_Draw(filename:string);
var
  f:file;
  walk:pointer;

begin
  assign(f,filename);
  reset(f,1);
  picsize:=filesize(f);
  getmem(picdata,picsize);
  blockread(f,picdata^,picsize);
  close(f);

  walk:=picdata;
  loc:=0;

  while (loc<picsize) do begin
    inc(loc,Perform_Opcode(walk));
    {let's slow down the drawing, to simulate the feel of an apple}
    if simulate_apple_speed
      then if loc and 3=1
        then wait_for_vertretrace;
  end;

  chirp;
  pause_for_user;
end;

Procedure PD_Done;
{Pause, then exit when <ESC> is hit.  Save the screen if <S> is hit.}
begin
  {free memory}
  freemem(picdata,picsize);
  case drawmode of
    draw:pd_graphshutdown;
  end;
end;

begin
  PD_init;
  PD_draw(paramstr(1));
  PD_done;
end.
