
{*******************************************************}
{                                                       }
{       Turbo Pascal Version 7.0                        }
{       Graphics Vision Unit                            }
{                                                       }
{       Copyright (c) 1992-95 by Solar Designer         }
{                                                       }
{*******************************************************}

unit GraphDrv;
{$B-,I-,P+}
interface

{ GraphDrv for Turbo Pascal 7.0.
  Supports loadable graphics drivers for different graphics modes.
  The drivers must be in PGI format.

  Where it is written that the identifier is same as in standard
  Borland's units, you can use Ctrl+F1 for more information.

  You should use LoadDriver/SetGraphMode procedures to initialize graphics
  mode. For some drivers you need to call InitDriver after LoadDriver.

  This unit requires Turbo Vision's Objects and Memory units (any version). }

   Uses
      Objects, Memory, UniFont;

   Type
      PathStr =    String[79];
{ Same as in the standard DOS unit }

   Var
{ Graphics procedures. Most of them are same as in the standard
  Graph unit, but parameter C is added. It is used to specify the color. }

      LowLevelInitDriver     :Procedure;
{ Used internally by InitDriver }
      LowLevelSetGraphMode   :Procedure;
{ You'd better use SetGraphMode instead of this procedure ! }

      ClearDevice            :Procedure (c                 :Integer);

      PutPixel               :Procedure (x, y, c           :Integer);
      Line                   :Procedure (x1, y1, x2, y2, c :Integer);
      HorizLine              :Procedure (y, x1, x2, c      :Integer);
      Bar                    :Procedure (x1, y1, x2, y2, c :Integer);
      QuickBar               :Procedure (x1, y1, x2, y2, c :Integer);
      Rectangle              :Procedure (x1, y1, x2, y2, c :Integer);
      GetPixel               :Function  (x, y              :Integer) :Integer;

      GetImage               :Procedure (x1, y1, x2, y2    :Integer;
                                         Buf               :Pointer);
      PutImage               :Procedure (x, y              :Integer;
                                         Buf               :Pointer);
      ImageSize              :Function  (x1, y1, x2, y2    :Integer) :Word;

      PutBlock               :Procedure (x, y              :Integer;
                                         p                 :Pointer);
      ConvertBlock           :Procedure (p1, p2            :Pointer);

      BufferToScreen         :Procedure;
      ScreenToBuffer         :Procedure;

      ChangeColor            :Procedure (x1,y1,x2,y2,c1,c2 :Integer);

      DrawImageLine          :Procedure (x, y, l, c        :Integer;
                                         Buf               :Pointer);
      DrawImageLineK         :Procedure (x, y, l, k, c     :Integer;
                                         Buf               :Pointer);

      Text_Init              :Procedure (x, y, Height      :Integer);
      Text_Advance           :Procedure; { AX = DeltaX }
      Text_Show8xX           :Procedure; { DS:SI -> Font; AL = Char; DL = Color }

   Const
      ProcCount =  23;
   Var
      ProcPtrs     :Array [0..ProcCount-1] of Pointer
      absolute LowLevelInitDriver;

   Const
{ Put modes (same as in standard Graph unit) }
      NormalPut =  0;
      AndPut =     1;
      OrPut =      2;
      XorPut =     3;

      DriverId=    'PGI'#9;

   Type
      TDriver=
      record
         Id                  :Array [1..4 ] of Char;
         Header              :Array [1..21] of Char; { Driver's name }
         Copyright           :Array [1..41] of Char;
         HeaderSize          :Word;
         VideoSeg, VideoSize :Word;
         SizeX, SizeY        :Word;
         MaxColor            :Byte;
         BufferSeg           :Word;
         AspX, AspY          :Byte;
         PutMode             :Byte;                  { These two fields can }
         LinePattern         :Byte;                  { be changed directly }
         LineBreak           :Boolean;
         FastSwitch          :Boolean;               { For Realtek SVGA bugs }

         Extra               :Array [1..8] of Byte; { Reserved }

         ExtraSize           :Word;
         ExtraPtr            :Pointer;

         ModesList           :Word;
         ModesCount          :Byte;
         ModeParams          :Word;

         MouseConvX,
         MouseConvY          :Byte;                  { Shifts for mouse }
                                                     { position conversions }
         PutModeSupport      :Array [NormalPut..XorPut] of Boolean;

         ProcCount           :Byte;
         ProcOfs             :Array [0..ProcCount-1] of Word;
      end;

      TDriverOutput=         (ToScreen, ToBuffer);

   Const
      DriverPtr              :^TDriver=nil;
   Var
      DriverSize             :Word;
      DriverVideoSeg         :Word;
      BufferPtr              :Pointer;

   Type
      TModesList =
      Array [1..256] of
      Record
         Mode, SizeX, SizeY, HBytes    :Word;
         AspX, AspY, MouseX, MouseY    :Byte;
      end;

      TModeParams =
      Record
         ModeNum, HBytes               :Word;
      end;

   Var
      ModesList              :^TModesList;
      ModeParams             :^TModeParams;

   Const
      TextModeNumber         :Word=    $FFFF;
      { Used by SetGraphMode/RestoreTextMode and CloseGraph }

   Const
      { Default color palette. Same as in CRT and Graph units }
      Black     = 0;  DarkGray=      8;
      Blue      = 1;  LightBlue=     9;
      Green     = 2;  LightGreen=   10;
      Cyan      = 3;  LightCyan=    11;
      Red       = 4;  LightRed=     12;
      Magenta   = 5;  LightMagenta= 13;
      Brown     = 6;  Yellow=       14;
      LightGray = 7;  White=        15;

      { A special color for PutBlock and DrawImageLine }
      NullColor =  255;

      gdOk =       0;
      gdNoMem =    1;
      gdIOError =  2;
      gdInvMode =  3;

   function  LoadDriver(Const d        :PathStr) :Integer;
      { Loads the driver into memory. Returnes an error code.
        If the driver is invalid, terminates the program.
        If you enable the extended syntax in your program ($X+ directive),
        you can call it as a procedure }

   procedure FreeDriverMem;
      { Releases the memory used by the driver. Used internally. }

   procedure InitDriver(Mode           :Word);
      { Initializes graphics driver for the specified mode }

   function  SetGraphMode                        :Integer;
      { Stores current video mode number and
        sets graphics mode of the currently loaded driver.
        Returns an error code. }

   procedure RestoreTextMode;
      { Restores text mode without removing the driver from memory }
   procedure CloseGraph;
      { Same as in the standard Graph unit. Restores text mode and
        releases memory. Use it at the end of your program. }

   function  Supported(var Proc                 ):Boolean;
      { Determines if the specified procedure or function is
        supported by the currently loaded graphics driver }

   function  PutModeSupported(Mode      :Integer):Boolean;
      { Same as previous, but for put modes }

   procedure GetBufferMem;
   procedure FreeBufferMem;
      { Allocate/release memory for the screen buffer.
        Some drivers don't support buffering. }

   procedure SetDriverOutput(o         :TDriverOutput);
   function  GetDriverOutput           :TDriverOutput;
      { Set/query if graphics is written to screen or buffer }

   function  GetMaxX         :Integer;
   function  GetMaxY         :Integer;
      { Same as in the standard Graph unit }

   Type
      TPolyLine=
      record
         y, x1, x2 :Integer;
      end;

      TPoly=       Array [0..8192] of TPolyLine;

   procedure LFillPoly(var pv;
                       n, c            :Integer);
      { Draws a filled polygon. N - number of points, C - color }

   procedure LDrawPoly(var pv;
                       n, c            :Integer);
      { Same as the previous, but draws only the border }

   procedure Ellipse(x, y,
                     sa, ea,
                     xr, yr,
                     c                 :Integer);
      { Draws an elliptical arc. X, Y - center,
        SA, EA - start and end angles (in degrees),
        XR, YR - horizontal and vertical radii, C - color }

   procedure Arc    (x, y,
                     sa, ea,
                     r,
                     c                 :Integer);
      { Same as the previous, but draws a circular arc }

   procedure Circle(x, y, r, c         :Integer);
      { Draws a circle }

   procedure EllipseRadius(x, y,
                           a,
                           xr, yr,
                           c           :Integer);

   procedure CircleRadius (x, y,
                           a,
                           r,
                           c           :Integer);
      { These two procedures are used to draw lines with given angle }

   Type
      TTextSettings=
      record
         SizeX, DivX, SizeY, DivY      :Integer;
         Horiz, Vert                   :Word;
      end;

   Const           { These constants are same as in the standard Graph unit }
      LeftText=    0;
      CenterText=  1;
      RightText=   2;
      BottomText=  0;
      TopText=     2;

      DTextSettings          :TTextSettings=
      (SizeX:1 ; DivX:1 ; SizeY:1 ; DivY:1 ; Horiz:LeftText ; Vert:TopText);

   procedure SetTextJustify(h, v                 :Word);
      { Same as in the standard Graph unit }

   procedure SetUserCharSize(mx, dx, my, dy      :Integer);
      { Same as in the standard Graph unit }

   procedure OutTextXY(x, y                      :Integer;
                       Const s                   :String;
                       c                         :Integer);
      { Writes text S at X,Y with color C using current justify and size.
        Warning: PutMode=XorPut is NOT supported here. }

   procedure SetTextAngle(a                      :Integer);
      { Sets the angle (in degrees) for OutTextXY and OutColoredTextXY }

   Const
      FillBufSize  :Word=    4096;
   { Size of the buffer used by FloodFill in bytes }

   procedure FloodFill(x, y, cf, cb              :Integer);
      { Fills a bounded region with color CF using current put mode.
        As soon as it reaches a point with color different than CB,
        it stops filling. }

   Procedure TurnBlockX(Source, Target           :Pointer);
   Procedure TurnBlockY(Source, Target           :Pointer);

   Procedure UText_PutPixel;
   Procedure UText_Init(x, y, Height             :Integer);
   Procedure UText_Advance;
   Procedure UText_Show8xX;
      { For use by the GViews unit only }

   procedure Abort(const Msg           :String);

implementation
{$L graphfun.obj}

   procedure Abort;
   begin
      if TextModeNumber<>$FFFF then RestoreTextMode else
      asm
         mov  ah,0Fh
         int  10h
         cbw
         or   al,80h
         int  10h
      end;
      WriteLn(Msg); Halt($FF);
   end;

   procedure GraphNotLoadedMsg; far;
   begin
      Abort('Graphics driver not loaded');
   end;

   procedure NotSupportedMsg; far;
   begin
      Abort('The requested operation isn''t supported by '+
         'the current graphics driver');
   end;

   procedure NotInitMsg; near;
   begin
      Abort('Graphics driver not initialized');
   end;

   function LoadDriver;
   var
      f            :File;
      InvOffset    :Boolean;
      Offset       :Word;
      fs           :LongInt;
      i            :Integer;
   begin
      LoadDriver:=gdOk;

      FreeDriverMem;

      Assign(f, d); Reset(f, 1);
      fs:=FileSize(f);

      if IOResult<>0 then
      begin
         LoadDriver:=gdIOError; Exit;
      end;

      if fs>$FFE0 then fs:=$FFE0;
      DriverSize:=fs;

      DriverPtr:=MemAllocSeg(DriverSize);
      if DriverPtr=nil then
      begin
         Close(f); LoadDriver:=gdNoMem; Exit;
      end;

      BlockRead(f, DriverPtr^, DriverSize);
      Close(f);

      if IOResult<>0 then
      begin
         FreeDriverMem; LoadDriver:=gdIOError; Exit;
      end;

      if DriverPtr^.Id<>DriverId then
         Abort('Invalid graphics driver');

      InvOffset:=False;
      for i:=0 to ProcCount-1 do
      begin
         Offset:=DriverPtr^.ProcOfs[i];
         if (Offset<>0) and (i<DriverPtr^.ProcCount) then
         begin
            ProcPtrs[i]:=Ptr(Seg(DriverPtr^)-$10, Offset);
            if Offset>DriverSize+$100 then InvOffset:=True;
         end
         else
            ProcPtrs[i]:=Addr(NotSupportedMsg);
      end;
      if InvOffset or (DriverPtr^.HeaderSize<>SizeOf(TDriver)-2*ProcCount) then
         Abort('Incorrect graphics driver version');

      if DriverPtr^.ModesCount<>0 then
      begin
         ModesList:=Ptr(Seg(DriverPtr^)-$10, DriverPtr^.ModesList);
         ModeParams:=Ptr(Seg(DriverPtr^)-$10, DriverPtr^.ModeParams);
      end
      else
      begin
         ModesList:=nil; ModeParams:=nil;
      end;

      DriverVideoSeg:=DriverPtr^.VideoSeg;

      if DriverPtr^.ExtraSize<>0 then
      begin
         DriverPtr^.ExtraPtr:=MemAlloc(DriverPtr^.ExtraSize);
         if DriverPtr^.ExtraPtr=nil then
         begin
            FreeDriverMem; LoadDriver:=gdNoMem; Exit;
         end;
      end;
   end;

   procedure FreeDriverMem;
   var
      i            :Integer;
   begin
      FreeBufferMem;
      for i:=0 to ProcCount-1 do ProcPtrs[i]:=Addr(GraphNotLoadedMsg);
      if DriverPtr<>nil then
      begin
         if DriverPtr^.ExtraPtr<>nil then
            FreeMem(DriverPtr^.ExtraPtr, DriverPtr^.ExtraSize);
         FreeMem(DriverPtr, DriverSize);
      end;
      DriverPtr:=nil; BufferPtr:=nil;
   end;

   procedure InitDriver;
   var
      i            :Integer;
   begin
      if DriverPtr=nil then GraphNotLoadedMsg else
      for i:=1 to DriverPtr^.ModesCount do
      begin
         if ModesList^[i].Mode = Mode then
         begin
            ModeParams^.ModeNum:=Mode;
            ModeParams^.HBytes:=ModesList^[i].HBytes;
            DriverPtr^.SizeX:=ModesList^[i].SizeX;
            DriverPtr^.SizeY:=ModesList^[i].SizeY;
            DriverPtr^.AspX:=ModesList^[i].AspX;
            DriverPtr^.AspY:=ModesList^[i].AspY;
            DriverPtr^.MouseConvX:=ModesList^[i].MouseX;
            DriverPtr^.MouseConvY:=ModesList^[i].MouseY;
            if Supported(LowLevelInitDriver) then LowLevelInitDriver;
            Exit;
         end;
      end;
      if ModesList<>nil then
         Abort('Invalid graphics mode for selected driver');
   end;

   function  SetGraphMode;
   assembler;
   asm
      les  di,ModeParams
      mov  ax,es
      or   ax,di
      jz   @@2
      cmp  es:[di].TModeParams.HBytes,0
      jne  @@2
      call NotInitMsg
@@2:
      mov  ax,TextModeNumber
      inc  ax
      jnz  @@AlreadyGfx
      mov  es,ax
      mov  al,es:[449h]
      mov  TextModeNumber,ax
@@AlreadyGfx:

      call dword ptr LowLevelSetGraphMode
      jz   @@Ok
      call RestoreTextMode
      mov  ax,gdInvMode
      jmp  @@1
@@Ok:
      xor  ax,ax
@@1:
   end;

   procedure RestoreTextMode;
   assembler;
   asm
      mov  ax,TextModeNumber
      cmp  ax,0FFFFh
      je   @@1
      cbw
      int  10h
@@1:
      mov  TextModeNumber,0FFFFh
   end;

   procedure CloseGraph;
   begin
      RestoreTextMode; FreeDriverMem;
   end;

   function Supported;
   begin
      Supported:=(DriverPtr<>nil) and (Pointer(Proc)<>Addr(NotSupportedMsg));
   end;

   function PutModeSupported;
   begin
      PutModeSupported:=(DriverPtr<>nil) and DriverPtr^.PutModeSupport[Mode];
   end;

   procedure SetDriverOutput;
   begin
      if DriverPtr=nil then GraphNotLoadedMsg;
      if o=ToScreen then DriverPtr^.VideoSeg:=DriverVideoSeg else
         if BufferPtr<>nil then DriverPtr^.VideoSeg:=DriverPtr^.BufferSeg;
   end;

   function  GetDriverOutput;
   begin
      if (DriverPtr^.VideoSeg=DriverPtr^.BufferSeg) and
         (DriverPtr^.BufferSeg<>DriverVideoSeg)
      then GetDriverOutput:=ToBuffer else GetDriverOutput:=ToScreen;
   end;

   function  GetMaxX;
   begin
      GetMaxX:=DriverPtr^.SizeX-1;
   end;

   function  GetMaxY;
   begin
      GetMaxY:=DriverPtr^.SizeY-1;
   end;

   procedure GetBufferMem;
   begin
      if BufferPtr=nil then
      begin
         if DriverPtr=nil then GraphNotLoadedMsg;

         if DriverPtr^.VideoSize=0 then Exit;
         BufferPtr:=MemAllocSeg(DriverPtr^.VideoSize);
         if BufferPtr<>nil then
         begin
            FillChar(BufferPtr^, DriverPtr^.VideoSize, 0);
            DriverPtr^.BufferSeg:=Seg(BufferPtr^);
         end;
      end;
   end;

   procedure FreeBufferMem;
   begin
      if (BufferPtr<>nil) and (DriverPtr<>nil) then
      begin
         SetDriverOutput(ToScreen);
         FreeMem(BufferPtr, DriverPtr^.VideoSize);
         DriverPtr^.BufferSeg:=DriverVideoSeg;
         BufferPtr:=nil;
      end;
   end;

   procedure LFillPoly;
   var
      i, y, ly, x1, x2                 :Integer;
      p                                :TPoly absolute pv;
   begin
      for i:=1 to n-1 do
      begin
         if i=1 then ly:=p[i-1].y else ly:=p[i-1].y+1;
         for y:=ly to p[i].y do
         begin
            if p[i].y=ly then
            begin
               x1:=p[i].x1; x2:=p[i].x2;
            end
            else
            begin
               x1:=p[i-1].x1+LongDiv(LongMul(p[i].x1-p[i-1].x1, y-ly), p[i].y-ly);
               x2:=p[i-1].x2+LongDiv(LongMul(p[i].x2-p[i-1].x2, y-ly), p[i].y-ly);
            end;
            HorizLine(y, x1, x2, c);
         end;
      end;
   end;

   procedure LDrawPoly;
   var
      p            :TPoly absolute pv;
      i            :Integer;
   begin
      HorizLine(p[0].y, p[0].x1, p[0].x2, c);
      for i:=1 to n-1 do
      begin
         Line(p[i-1].x1, p[i-1].y, p[i].x1, p[i].y, c);
         Line(p[i-1].x2, p[i-1].y, p[i].x2, p[i].y, c);
      end;
      HorizLine(p[n-1].y, p[n-1].x1, p[n-1].x2, c);
   end;

   procedure Ellipse;
   var
      i, n, a, cx, cy, lx, ly          :Integer;
   begin
      DriverPtr^.LineBreak:=True;
      if xr>yr then n:=yr else n:=xr;
      if n<=5 then n:=n shl 1+1 else Inc(n, 6);
      for i:=0 to n do
      begin
         a:=sa+LongDiv(LongMul(i, ea-sa), n);
         cx:=x+LongDiv(LongMul(QSin(a), xr), MaxInt);
         cy:=y+LongDiv(LongMul(QCos(a), yr), MaxInt);
         if a<>sa then Line(lx,ly,cx,cy,c);
         if (a<>sa) or (i=0) then
         begin
            lx:=cx; ly:=cy;
         end;
      end;
      DriverPtr^.LineBreak:=False;
   end;

   procedure Arc;
   begin
      Ellipse(x, y, sa, ea,
         LongDiv(LongMul(r, DriverPtr^.AspX), DriverPtr^.AspY), r, c);
   end;

   procedure Circle;
   begin
      Arc(x, y, 0, 360, r, c);
   end;

   procedure EllipseRadius;
   begin
      Line(x, y,
         x+LongDiv(LongMul(QSin(a), xr), MaxInt),
         y+LongDiv(LongMul(QCos(a), yr), MaxInt), c);
   end;

   procedure CircleRadius;
   begin
      EllipseRadius(x, y, a,
         LongDiv(LongMul(r, DriverPtr^.AspX), DriverPtr^.AspY), r, c);
   end;

   procedure SetTextJustify;
   begin
      DTextSettings.Horiz:=h; DTextSettings.Vert:=v;
   end;

   procedure SetUserCharSize;
   begin
      with DTextSettings do
      begin
         SizeX:=mx; DivX:=dx; SizeY:=my; DivY:=dy;
      end;
   end;

   procedure OutTextXY;
   var
      LastPutMode  :Byte;
      sx, sy, i,
      stp          :Integer;
   begin
      with DTextSettings do
      begin
         sx:=(SizeX shl 3)*Byte(s[0]) div DivX; sy:=(SizeY shl 3) div DivY;
         case Horiz of
            RightText:  Dec(x, sx);
            CenterText: Dec(x, sx shr 1);
         end;
         case Vert of
            BottomText: Dec(y, sy);
            CenterText: Dec(y, sy shr 1);
         end;
         stp:=(SizeX shl 3) div DivX;

         LastPutMode:=DriverPtr^.PutMode;
         if DriverPtr^.PutMode=XorPut then DriverPtr^.PutMode:=NormalPut;

         for i:=1 to Length(s) do
         begin
            DrawChar(UniFont.Fonts, s[i], x, y, c, SizeX, DivX, SizeY, DivY,
               @Line, @Rectangle, @Bar);
            Inc(x, stp);
         end;
         DriverPtr^.PutMode:=LastPutMode;
      end;
   end;

   procedure SetTextAngle;
   begin
      SetFontAngle(a);
      VectorParams.AspX:=DriverPtr^.AspX; VectorParams.AspY:=DriverPtr^.AspY;
   end;

   procedure FloodFill;
   type
      TCoordArray= Array [1..$4000] of Integer;
   var
      sx, sy,
      sxnew, synew,
      temp, MarkMem                    :^TCoordArray;
      i, n, nnew, maxx, maxy, MaxIndex :Integer;
      ArrayMem                         :Word;

      procedure FillPixel(x, y         :Integer);
      begin
         if nnew>=MaxIndex then Exit;
         PutPixel(x, y, cf); Inc(nnew); sxnew^[nnew]:=x; synew^[nnew]:=y;
      end;

   begin
      if cf>DriverPtr^.MaxColor then cf:=DriverPtr^.MaxColor;
      if cf=cb then Exit;
      if (FillBufSize<32) or (FillBufSize>MaxAvail) then Exit;

      MaxIndex:=FillBufSize shr 3; ArrayMem:=MaxIndex shl 1;

      Mark(MarkMem);
      GetMem(sx, ArrayMem);    GetMem(sy, ArrayMem);
      GetMem(sxnew, ArrayMem); GetMem(synew, ArrayMem);

      maxx:=DriverPtr^.SizeX-1; maxy:=DriverPtr^.SizeY-1;
      nnew:=0;
      if GetPixel(x, y)=cb then FillPixel(x, y);
      repeat
         n:=nnew; nnew:=0;
         temp:=sx; sx:=sxnew; sxnew:=temp; temp:=sy; sy:=synew; synew:=temp;
         for i:=1 to n do
         begin
            x:=sx^[i]; y:=sy^[i];
            if (x>0) and (GetPixel(x-1, y)=cb) then FillPixel(x-1, y);
            if (x<maxx) and (GetPixel(x+1, y)=cb) then FillPixel(x+1, y);
            if (y>0) and (GetPixel(x, y-1)=cb) then FillPixel(x, y-1);
            if (y<maxy) and (GetPixel(x, y+1)=cb) then FillPixel(x, y+1);
         end;
      until nnew=0;

      Release(MarkMem);
   end;

   Procedure TurnBlockX; External;
   Procedure TurnBlockY; External;

   Procedure UText_PutPixel; External;
   Procedure UText_Init; External;
   Procedure UText_Advance; External;
   Procedure UText_Show8xX; External;

begin
   FreeDriverMem;
end.
