UNIT VOCTOOL;
{
 ********************************************************************
 * Unit para controlar una tarjeta SoundBlaster en Borland Pascal   *
 *       mediante utilizacin del controlador CT-VOICE.DRV.         *
 ********************************************************************
 *                 (C) 1994 Data Becker GmbH & Co.                  *
 *                       MARCOMBO S.A.                              *
 *                    Autor : Axel Stolz                            *
 ********************************************************************
}
INTERFACE

TYPE
   VOCFileTyp = File;

CONST
   VOCToolVersion  = 'v1.5';  { N de versin de la VOCTOOL-Unit}
   VOCBreakEnd     = 0;       { Constante para interr. de bucle }
   VOCBreakNow     = 1;       { Constante para interr. de bucle }
   VOCCardNames    : ARRAY[0..6] OF STRING[22] =  ('desconocida',
                                                   'Sound Blaster 1.0/1.5',
                                                   'Sound Blaster Pro',
                                                   'Sound Blaster 2.0',
                                                   'Sound Blaster Pro 2.0',
                                                   'Sound Blaster Pro MCV',
                                                   'Sound Blaster 16');


VAR
   VOCStatusWord        : WORD;     { Variable para Sound-Blaster-Status }
   VOCErrStat           : WORD;     { Variable para n de error de contr.}
   VOCFileHeader        : STRING;   { Variable para cabecera CT          }
   VOCFileHeaderLength  : BYTE;     { Longitud de la cabecera CT         }
   VOCPaused            : BOOLEAN;  { Flag para VoiceStatus Pause        }
   VOCDriverInstalled   : BOOLEAN;  { Flag, si controlador instalado     }
   VOCDriverVersion     : WORD;     { N de versin de controlador       }
   VOCPtrToDriver       : Pointer;  { Puntero al contr. en la memoria    }
   OldExitProc          : Pointer;  { Puntero a la vieja Unit-ExitProc   }


PROCEDURE PrintVOCErrMessage;
FUNCTION  VOCGetBuffer(VAR VoiceBuff : Pointer; Voicefile : STRING):BOOLEAN;
FUNCTION  VOCFreeBuffer(VAR VoiceBuff : Pointer):BOOLEAN;
FUNCTION  VOCGetVersion:WORD;
PROCEDURE VOCSetPort(PortNumber : WORD);
PROCEDURE VOCSetIRQ(IRQNumber : WORD);
FUNCTION  VOCInitDriver:BOOLEAN;
PROCEDURE VOCDeInstallDriver;
PROCEDURE VOCSetSpeaker(OnOff:BOOLEAN);
PROCEDURE VOCOutput(BufferAddress : Pointer);
PROCEDURE VOCOutputLoop (BufferAddress : Pointer);
PROCEDURE VOCStop;
PROCEDURE VOCPause;
PROCEDURE VOCContinue;
PROCEDURE VOCBreakLoop(BreakMode : WORD);

IMPLEMENTATION

USES DOS,Crt;

TYPE
   TypeCastType = ARRAY [0..6000] of Char;

VAR
   Regs : Registers;

PROCEDURE PrintVOCErrMessage;
{
 * ENTRADA  : ninguna
 * SALIDA   : ninguna
 * FUNCION  : Devuelve el error SB como texto en la pantalla, sin modificar
 *            el estado de error.
}
BEGIN
   CASE VOCErrStat OF
      100 : Write(' No se encuentra el controlador CT-VOICE.DRV ');
      110 : Write(' no hay memoria libre para el controlador ');
      120 : Write(' controlador errneo ');

      200 : Write(' archivo VOC no encontrado ');
      210 : Write(' no hay memoria libre para archivo VOC ');
      220 : Write(' archivo no est en formato VOC ');

      300 : Write(' Error de asignacin de memoria ');

      400 : Write(' no se encontr tarjeta Sound-Blaster ');
      410 : Write(' direccin de port errnea ');
      420 : Write(' interrupcin errnea ');

      500 : Write(' no hay bucle en proceso ');
      510 : Write(' no hay sampling en reproduccin ');
      520 : Write(' no hay sampling parado ');
      END;
   END;

FUNCTION Exists (Filename : STRING):BOOLEAN;
{
 * ENTRADA  : Nombre de archivo como String
 * SALIDA   : TRUE, si existe el archivo, sino FALSE
 * FUNCION  : Comprueba si un archivo existe, y devuelve la expresin
              booleana correspondiente.
}
VAR
   F : File;
BEGIN
   Assign(F,Filename);
{$I-}
   Reset(F);
   Close(F);
{$I+}
   Exists := (IoResult = 0) AND (Filename <> '');
   END;

PROCEDURE AllocateMem (VAR Pt : Pointer; Size : LongInt);
{
 * ENTRADA  : Variable para bfer como Pointer,
              tamao del bfer como LongInt
 * SALIDA   : Puntero al bfer en variable o NIL
 * FUNCION  : Reserva los bytes que indica Size y coloca un
              puntero a ellos en la variable Pt. Si no hay memoria
              suficiente, Pt apuntar a NIL
}
VAR
   SizeIntern : WORD;     { Tamao del bfer para clculos internos }
BEGIN
   Inc(Size,15);                 { Inrementar tamao de bfer en 15}
   SizeIntern := (Size shr 4);   { y dividir por 16.               }
   Regs.AH := $48;               { Cargar funcin MS-DOS $48 en AH }
   Regs.BX := SizeIntern;        { Cargar tamao interno en BX     }
   MsDos(Regs);                  { Reservar memoria                }
   IF (Regs.BX <> SizeIntern) THEN Pt := NIL
   ELSE Pt := Ptr(Regs.AX,0);
   END;

FUNCTION  CheckFreeMem (VAR VoiceBuff : Pointer; VoiceSize : LongInt):BOOLEAN;
{
 * ENTRADA  : Variable para bfer como Pointer, tamao deseado como LongInt
 * SALIDA   : Puntero a bfer, TRUE/FALSE, segn AllocateMem
 * FUNCION  : Comprueba si se puede ocupar memoria suficiente.
}
BEGIN
   AllocateMem(VoiceBuff,VoiceSize);
   CheckFreeMem := VoiceBuff <> NIL;
   END;

FUNCTION  VOCGetBuffer (VAR VoiceBuff : Pointer; Voicefile : STRING):BOOLEAN;
{
 * ENTRADA  : Variable para bfer como Pointer, nombre de archivo como String
 * SALIDA   : Puntero a bfer con datos VOC, TRUE/FALSE
 * FUNCION  : Carga un archivo en memoria, y en caso de xito devuelve
              el valor TRUE, sino FALSE.
}
VAR
   SampleSize : LongInt;
   FPresent   : BOOLEAN;
   VFile      : VOCFileTyp;
   Segs       : WORD;
   Read       : WORD;

BEGIN
   FPresent := Exists(VoiceFile);

{ El archivo VOC no se encontr }
   IF Not(FPresent) THEN BEGIN
      VOCGetBuffer := FALSE;
      VOCErrStat   := 200;
      EXIT
      END;

   Assign(VFile,Voicefile);
   Reset(VFile,1);
   SampleSize := Filesize(VFile);
   AllocateMem(VoiceBuff,SampleSize);

{ No hay memoria suficiente para el archivo VOC }
   IF (VoiceBuff = NIL) THEN BEGIN
      Close(VFile);
      VOCGetBuffer := FALSE;
      VOCErrStat   := 210;
      EXIT;
      END;

   Segs := 0;
   REPEAT
      Blockread(VFile,Ptr(seg(VoiceBuff^)+4096*Segs,Ofs(VoiceBuff^))^,$FFFF,Read);
      Inc(Segs);
      UNTIL Read = 0;
   Close(VFile);

{ El archivo no est en formato VOC }
   IF (TypeCastType(VoiceBuff^)[0]<>'C') OR
      (TypeCastType(VoiceBuff^)[1]<>'r') THEN BEGIN
      VOCGetBuffer := FALSE;
      VOCErrStat := 220;
      EXIT;
      END;

{ La carga ha tenido xito }
   VOCGetBuffer := TRUE;
   VOCErrStat   := 0;

{ Leer longitud de la cabecera del archivo }
   VOCFileHeaderLength := Ord(TypeCastType(VoiceBuff^)[20]);
   END;

FUNCTION VOCFreeBuffer (VAR VoiceBuff : Pointer):BOOLEAN;
{
 * ENTRADA  : Puntero a bfer como Pointer
 * SALIDA   : ninguna
 * FUNCION  : Libera la memoria ocupada por los datos VOC
}
BEGIN
   Regs.AH := $49;              { Cargar funcin MS-DOS $49 en AH   }
   Regs.ES := seg(VoiceBuff^);  { Segmento de memoria en ES         }
   MsDos(Regs);                 { Liberar memoria de nuevo          }
   VOCFreeBuffer := TRUE;
   IF (Regs.AX = 7) OR (Regs.AX = 9) THEN BEGIN
      VOCFreeBuffer := FALSE;
      VOCErrStat := 300         { al liberar ha ocurrido un error   }
      END;                      { de DOS.                           }
   END;

FUNCTION VOCGetVersion:WORD;
{
 * ENTRADA  : ninguna
 * SALIDA   : N de versin del controlador
 * FUNCION  : Obtener el n de versin del controlador
}
VAR
   VDummy : WORD;
BEGIN
   ASM
      MOV       BX,0
      CALL      VOCPtrToDriver
      MOV       VDummy, AX
      END;
   VOCGetVersion := VDummy;
   END;

PROCEDURE VOCSetPort(PortNumber : WORD);
{
 * ENTRADA  : N de direccin de port
 * SALIDA   : ninguna
 * FUNCION  : Asignar la direccin de port antes de inicializar
}
BEGIN
   ASM
      MOV    BX,1
      MOV    AX,PortNumber
      CALL   VOCPtrToDriver
      END;
   END;

PROCEDURE VOCSetIRQ(IRQNumber : WORD);
{
 * ENTRADA  : N de interrupcin
 * SALIDA   : ninguna
 * FUNCION  : Asignar la interrupcin antes de inicializar
}
BEGIN
   ASM
      MOV    BX,2
      MOV    AX,IRQNumber
      CALL   VOCPtrToDriver
      END;
   END;

FUNCTION  VOCInitDriver: BOOLEAN;
{
 * ENTRADA  : ninguna
 * SALIDA   : N de mensaje de error, segn resultado de inicializacin
 * FUNCION  : Inicializacin del software del controlador
}
VAR
   Out, VSeg, VOfs : WORD;
   F   : File;
   Drivername,
   Pdir        : DirStr;
   Pnam        : NameStr;
   Pext        : ExtStr;

BEGIN
{ Primero se busca el controlador CT-VOICE.DRV en la va de acceso en la
  que se encontr el programa que lo quiere emplear.                    }
   Pdir := ParamStr(0);
   Fsplit(ParamStr(0),Pdir,Pnam,Pext);
   Drivername := Pdir + 'CT-VOICE.DRV';

   VOCInitDriver := TRUE;

{ Controlador no se encontr            }
{ Buscar en la va de acceso del sistema}
   IF Not Exists(Drivername) THEN BEGIN
      DriverName := GetEnv('SOUND')+'\DRV\CT-VOICE.DRV';
      IF Not Exists(Drivername) THEN BEGIN
         VOCInitDriver := FALSE;
         VOCErrStat    := 100;
         EXIT;
         END;
      END;

{ Cargar el controlador }
   Assign(F,Drivername);
   Reset(F,1);
   AllocateMem(VOCPtrToDriver,Filesize(F));

{ No se pudo ocupar memoria para  }
{ el controlador                  }
   IF VOCPtrToDriver = NIL THEN BEGIN
      VOCInitDriver := FALSE;
      VOCErrStat    := 110;
      EXIT;
      END;

   Blockread(F,VOCPtrToDriver^,Filesize(F));
   Close(F);

{ Archivo controlador no comienza con }
{ "CT", de modo que es errneo        }
   IF (TypeCastType(VOCPtrToDriver^)[3]<>'C') OR
      (TypeCastType(VOCPtrToDriver^)[4]<>'T') THEN BEGIN
         VOCInitDriver := FALSE;
         VOCErrStat    := 120;
         EXIT;
         END;

{ Obtener n de versin y pasar a variable global }
   VOCDriverVersion := VOCGetVersion;

{ Inicial el controlador }
   Vseg := Seg(VOCStatusWord);
   VOfs := Ofs(VOCStatusWord);
   ASM
      MOV       BX,3
      CALL      VOCPtrToDriver
      MOV       Out,AX
      MOV       BX,5
      MOV       ES,VSeg
      MOV       DI,VOfs
      CALL      VOCPtrToDriver
      END;

{ No se pudo encontrar una tarjeta}
{ SoundBlaster                    }
   IF Out = 1 THEN BEGIN
      VOCInitDriver := FALSE;
      VOCErrStat    := 400;
      EXIT;
      END;

{ Direccin de port errnea }
   IF Out = 2 THEN BEGIN
      VOCInitDriver := FALSE;
      VOCErrStat    := 410;
      EXIT;
      END;

{ Interrupcin errnea }
   IF Out = 3 THEN BEGIN
      VOCInitDriver := FALSE;
      VOCErrStat    := 420;
      EXIT;
      END;
   END;

PROCEDURE VOCDeInstallDriver;
{
 * ENTRADA  : ninguna
 * SALIDA   : ninguna
 * FUNCION  : Desactivar el controlador y liberar la memoria
}
VAR
   Check : BOOLEAN;
BEGIN
   IF VOCDriverInstalled THEN
   ASM
      MOV       BX,9
      CALL      VOCPtrToDriver
      END;
   Check := VOCFreeBuffer(VOCPtrToDriver);
   END;

PROCEDURE VOCSetSpeaker(OnOff:BOOLEAN);
{
 * ENTRADA  : TRUE para altavoz activo, FALSE para inactivo
 * SALIDA   : ninguna
 * FUNCION  : ON/OFF - Conmutar el altavoz de la tarjeta SB
}
VAR
   Switch : BYTE;
BEGIN
   Switch := Ord(OnOff) AND $01;
   ASM
      MOV       BX,4
      MOV       AL,Switch
      CALL      VOCPtrToDriver
      END;
   END;

PROCEDURE VOCOutput (BufferAddress : Pointer);
{
 * ENTRADA  : Puntero a los datos de sampling como Pointer
 * SALIDA   : ninguna
 * FUNCION  : Reproducir un sampling
}
VAR
   VSeg, VOfs : WORD;
BEGIN
   if VOCStatusWord<>0 then VOCStop;
   VOCSetSpeaker(TRUE);
   VSeg := Seg(BufferAddress^);
   VOfs := Ofs(BufferAddress^)+VOCFileHeaderLength;
   ASM
      MOV       BX,6
      MOV       ES,VSeg
      MOV       DI,VOfs
      CALL      VOCPtrToDriver
      END;
   END;

PROCEDURE VOCOutputLoop (BufferAddress : Pointer);
{
 *    Diferencias con VOCOutput :
 *    aqu no se activa el altavoz antes de cada reproduccin de un sampling,
 *    ya que esto lleva a un crujido en algunas tarjetas SoundBlaster al
 *    reproducir un bucle.
}
VAR
   VSeg, VOfs : WORD;
BEGIN
   VSeg := Seg(BufferAddress^);
   VOfs := Ofs(BufferAddress^)+VOCFileHeaderLength;
   ASM
      MOV       BX,6
      MOV       ES,VSeg
      MOV       DI,VOfs
      CALL      VOCPtrToDriver
      END;
   END;

PROCEDURE VOCStop;
{
 * ENTRADA  : ninguna
 * SALIDA   : ninguna
 * FUNCION  : Parar un sampling
}
BEGIN
   ASM
      MOV       BX,8
      CALL      VOCPtrToDriver
      END;
   END;

PROCEDURE VOCPause;
{
 * ENTRADA  : ninguna
 * SALIDA   : ninguna
 * FUNCION  : Detener un sampling
}
VAR
   Switch : WORD;
BEGIN
   VOCPaused := TRUE;
   ASM
      MOV       BX,10
      CALL      VOCPtrToDriver
      MOV       Switch,AX
      END;
   IF (Switch = 1) THEN BEGIN
      VOCPaused := FALSE;
      VOCErrStat := 510;
      END;
   END;

PROCEDURE VOCContinue;
{
 * ENTRADA  : ninguna
 * SALIDA   : ninguna
 * FUNCION  : Continuar un sample detenido
}
VAR
   Switch : WORD;
BEGIN
   ASM
      MOV       BX,11
      CALL      VOCPtrToDriver
      MOV       Switch,AX
      END;
   IF (Switch = 1) THEN BEGIN
      VOCPaused := FALSE;
      VOCErrStat := 520;
      END;
   END;

PROCEDURE VOCBreakLoop(BreakMode : WORD);
{
 * ENTRADA  : Modo de cancelacin
 * SALIDA   : ninguna
 * FUNCION  : Interrumpir un bucle de sampling
}
BEGIN
   ASM
      MOV       BX,12
      MOV       AX,BreakMode
      CALL      VOCPtrToDriver
      MOV       BreakMode,AX
      END;
   IF (BreakMode = 1) THEN VOCErrStat := 500;
   END;

{$F+}
PROCEDURE VoiceToolsExitProc;
{$F-}
{
 * ENTRADA  : ninguna
 * SALIDA   : ninguna
 * FUNCION  : En la nueva ExitProc se desinstala de nuevo el
              Voice-Driver al final del programa.
}
BEGIN
   VOCDeInstallDriver;
   ExitProc := OldExitProc;
   END;

BEGIN
{
 * Las siguientes instrucciones se ejecutan automticamente, en cuanto
 * se incluye la Unit en un programa, y se ejecuta ste.
}
{ Reasignar la antigua ExitProc a la nueva Tool-Unit }
   OldExitProc := ExitProc;
   ExitProc := @VoiceToolsExitProc;
{ Inicializar valores bsicos }
   VOCStatusWord := 0;
   VOCErrStat    := 0;
   VOCPaused     := FALSE;
   VOCFileHeaderLength := $1A;
   VOCFileHeader :=
      'Creative Voice File'+#$1A+#$1A+#$00+#$0A+#$01+#$29+#$11+#$01;
{
 * VOCDriverInstalled contiene, despus de la instalacin, o TRUE o
 * FALSE, segn si el controlador se pudo instalar adecuadamente o, no
}
   VOCDriverInstalled := VOCInitDriver;
   END.
