
;
;   SQ_CODE.ASM
;
;   Simon Hern     (Wolfson College, Cambridge, CB3 9BB, England)
;   Summer 1995    (Email: sdh20@cam.ac.uk)
;
;   Assembler routines for "Squidgy Wars" game
;



    public _MapToViewBuffer     ; Graphics routines for generating and
    public _ViewBufferToScr     ;   displaying windows onto the game
    public _DrawObject          ;   play arena

    public _InstallKeyboard     ; Keyboard handler
    public _RemoveKeyboard      ;
    public _KeyTable            ;
    public _KeyPressed          ;

    public _ReadJoystick        ; Joystick routine

    public _SBSetReg            ; Basic sound card functions
    public _SBReadStatus        ;

    public _DrawSprite          ; General Mode X sprite routine

    public _DisplayStatic       ; Function used by title page code



SCR_WIDTH       equ 320     ; Mode X screen dimensions
SCR_HEIGHT      equ 240

MAP_HEIGHT      equ 20      ; Size of Map array of tiles
MAP_WIDTH       equ 20

TILE_WIDTH      equ 16      ; Size of tile sprites
TILE_SIZE       equ TILE_WIDTH*TILE_WIDTH   ; Panic if not 256!

VIEW_WIDTH_TILE equ 9       ; Width of player's viewing window in tiles
VIEW_WIDTH      equ VIEW_WIDTH_TILE*TILE_WIDTH  ; Width of viewing window
VIEW_BUFF_WIDTH equ (VIEW_WIDTH+TILE_WIDTH)     ; Width of viewing buffer
VIEW_BUFF_SIZE  equ VIEW_BUFF_WIDTH*VIEW_BUFF_WIDTH     ; Size of buffer

SEQU_ADDR   equ 003c4h      ; Base port of the Sequencer (ModeX plane select)
VID_SEG     equ 0a000h      ; Segment of video memory

KEYTABLE_LENGTH equ 256     ; Number of entries (possible keys) in KeyTable
KEY_PORT        equ 060h    ; Port for reading keyboard
KEY_ACK         equ 020h    ; Acknowledge code for keyboard hardware
EXTEND_CODE     equ 224     ; Extended keystroke code

JOY_PORT        equ 00201h  ; Port for reading joystick
SB_STATUS_PORT  equ 00388h  ; Port for programming soundcard



_Data   SEGMENT BYTE PUBLIC 'Data'

; When keyboard handler is installed, KeyPressed is the code of the key
;  currently held down, or 0 if no keys are being pressed
; KeyTable has one entry for each possible key code and that entry is
;  1 if the corresponding key is being pressed, and 0 if it isn't
; Key codes 0-127 are the standard codes used by the keyboard handler
; Extended key strokes are given codes in the range 128-255

_KeyPressed:    dw ?
_KeyTable:      db KEYTABLE_LENGTH dup(?)

_Data   ENDS



_Text   SEGMENT BYTE PUBLIC 'Code'



; Each player has a viewing window onto the game play area which shows
;  pieces of wall and floor (as positioned in the Map array) together with
;  other objects (squidgies, missiles, etc)
; The viewing window is initially generated in a buffer area of memory and
;  is then copied to the screen
; To simplify the programming, the viewing buffer is always aligned to the
;  edges of the tiles in the map so that fractions of tiles are never drawn
;  in the buffer. This positioning is corrected for when the buffer is
;  copied to the viewing window on screen.
; The viewing buffer is one tile-width wider than the viewing window


; This function draws the floor and wall tiles shown in the Map array
;  in the viewing buffer pointed to by <buff>
; The tile drawn in the top-left corner of the buffer is that which has
;  map position (LeftEdge, TopEdge)

_MapToViewBuffer:   ; MapToView(int LeftEdge, int TopEdge, char far * Buff)

    push bp
    mov bp,sp
    push es
    push si
    push di
    les di,d[bp+8]      ; es:di = start of ViewBuffer
    mov dl,b[bp+4]
    mov dh,b[bp+6]
    mov al,MAP_WIDTH
    mul dh
    add al,dl
    adc ah,0
    add ax,_Map         ; (dh,dl) = position on map of top-left tile
    mov bx,ax           ; bx = address in map array of top-left tile

    mov ch,VIEW_WIDTH_TILE+1

mtv01:
    push bx
    push dx
    push di
    mov cl,VIEW_WIDTH_TILE+1

mtv02:
    push di
    mov ah,b[bx]
    mov al,0
    add ax,w[_TileSprites]
    mov si,ax
    mov al,TILE_WIDTH
mtv07:
    movsw                       ; Copy tile to buffer
    movsw
    movsw
    movsw                       ; TILE_WIDTH/2 repeats
    movsw
    movsw
    movsw
    movsw
    add di,VIEW_BUFF_WIDTH - TILE_WIDTH
    dec al
    jnz mtv07
    pop di

    inc bx
    inc dl
    cmp dl,MAP_WIDTH
    jc mtv04
    sub dl,MAP_WIDTH            ; Wrap around map coords
    sub bx,MAP_WIDTH
mtv04:
    add di,TILE_WIDTH
    dec cl
    jnz mtv02

    pop di
    pop dx
    pop bx
    add bx,MAP_WIDTH
    inc dh
    cmp dh,MAP_HEIGHT
    jc mtv05
    sub dh,MAP_HEIGHT           ; Wrap around map coords
    sub bx,MAP_HEIGHT*MAP_WIDTH
mtv05:
    add di,TILE_WIDTH*VIEW_BUFF_WIDTH
    dec ch
    jz mtv06
    jmp mtv01
mtv06:

    pop di
    pop si
    pop es
    pop bp
    ret



; ViewBufferToScr() copies part of a viewing buffer to a viewing window
; <scr_addr> is the offset into the video segment at which the window is drawn
; <buff> points to the pixel in the viewing buffer that is to be drawn at
;  the very top left of the viewing window (nb, this will usually not be the
;  first byte in the buffer since the buffer always aligns to tile boundaries)
; The central loops of the function are unrolled for speed

UNROLL MACRO            ; Unrolled central loop
#RX1(VIEW_WIDTH/8)          ; i.e. Repeat VIEW_WIDTH/8 times
    mov al,ds:b[si]
    add si,dx               ; dx = 4
    mov ah,ds:b[si]
    add si,dx               ; dx = 4
    stosw
#ER
#EM

_ViewBufferToScr:   ; ViewBufferToScr(char far * buff, unsigned scr_addr)

    push bp
    mov bp,sp
    push si
    push di
    push ds
    push es
    mov si,w[bp+4]
    mov ax,w[bp+6]
    mov ds,ax           ; ds:si = view buffer
    mov di,w[bp+8]
    push VID_SEG
    pop es              ; es:di = screen address

    mov ax,00102h       ; video plane 1
    mov dx,SEQU_ADDR
    out dx,ax
    mov dx,4
    mov bx,di
    push si

    mov cl,VIEW_WIDTH
vts01:

    UNROLL

    add si,TILE_WIDTH
    add di,(SCR_WIDTH - VIEW_WIDTH)/4
    dec cl
    jz vts02
    jmp vts01
vts02:

    mov ax,00202h       ; video plane 2
    mov dx,SEQU_ADDR
    out dx,ax
    mov dx,4
    mov di,bx
    pop si
    inc si
    push si

    mov cl,VIEW_WIDTH
vts03:

    UNROLL

    add si,TILE_WIDTH
    add di,(SCR_WIDTH - VIEW_WIDTH)/4
    dec cl
    jz vts04
    jmp vts03
vts04:

    mov ax,00402h       ; video plane 3
    mov dx,SEQU_ADDR
    out dx,ax
    mov dx,4
    mov di,bx
    pop si
    inc si
    push si

    mov cl,VIEW_WIDTH
vts05:

    UNROLL

    add si,TILE_WIDTH
    add di,(SCR_WIDTH - VIEW_WIDTH)/4
    dec cl
    jz vts06
    jmp vts05
vts06:

    mov ax,00802h       ; video plane 4
    mov dx,SEQU_ADDR
    out dx,ax
    mov dx,4
    mov di,bx
    pop si
    inc si

    mov cl,VIEW_WIDTH
vts07:

    UNROLL

    add si,TILE_WIDTH
    add di,(SCR_WIDTH - VIEW_WIDTH)/4
    dec cl
    jz vts08
    jmp vts07
vts08:

    pop es
    pop ds
    pop di
    pop si
    pop bp
    ret



; Copies any tile sprite into the viewing buffer, masking pixels with value 0
;  and clipping at the window boundaries
; <view> points to the pixel in the viewing buffer that is to be drawn at the
;  very top-left of the viewing window (see the comments above)
; (x,y) is the displacement relative to this pixel of the top-left corner of
;  the destination of the sprite in the viewing window
; <tile> is the tile number of the sprite

_DrawObject:        ; void DrawObject(char far * view, int x, int y, int tile)

    push bp
    mov bp,sp
    push si
    push di
    push es

    mov ah,b[bp+12]
    xor al,al
    add ax,[_TileSprites]
    mov si,ax               ; ds:si = start of sprite
    mov di,w[bp+4]
    mov ax,w[bp+6]
    mov es,ax               ; es:di = destination in viewbuffer

    mov cl,TILE_WIDTH
    mov bx,w[bp+8]
    cmp bx,0                ; clip to the left
    jge dro01
    add bx,TILE_WIDTH
    cmp bx,0
    jg dro08
    jmp dro00
dro08:
    mov cl,bl
    add si,TILE_WIDTH
    sub si,bx
    mov bx,0
    jmp dro02

dro01:
    cmp bx,VIEW_WIDTH-TILE_WIDTH    ; clip to the right
    jle dro02
    cmp bx,VIEW_WIDTH
    jge dro00
    mov cx,VIEW_WIDTH
    sub cx,bx

dro02:                      ; bx = x-position, cl = width to draw
    mov ch,TILE_WIDTH
    mov dx,w[bp+10]
    cmp dx,0                ; clip above
    jge dro03
    add dx,TILE_WIDTH
    cmp dx,0
    jle dro00
    mov ch,dl
    sub dl,TILE_WIDTH
    neg dl
    shl dx,4
    add si,dx
    mov dx,0
    jmp dro04

dro03:
    cmp dx,VIEW_WIDTH-TILE_WIDTH    ; clip below
    jle dro04
    cmp dx,VIEW_WIDTH
    jge dro00
    mov ax,VIEW_WIDTH
    sub ax,dx
    mov ch,al

dro04:                      ; dx = y-position, ch = height to draw
    mov al,VIEW_BUFF_WIDTH
    mul dl
    add ax,bx
    add di,ax

    mov ah,cl
    mov dl,VIEW_BUFF_WIDTH
    sub dl,cl
    xor dh,dh               ; dx = distance to next line in viewbuffer
    mov bl,TILE_WIDTH
    sub bl,cl
    xor bh,bh               ; bx = distance to next line in sprite
dro05:
    mov cl,ah
dro06:
    lodsb
    or al,al
    jz dro07
    mov es:b[di],al
dro07:
    inc di
    dec cl
    jnz dro06
    add di,dx
    add si,bx
    dec ch
    jnz dro05

dro00:
    pop es
    pop di
    pop si
    pop bp
    ret



; Installs a keyboard handler to updates KeyPressed and KeyTable
; Normal key operations are suspended

KeyVect:    dd 0    ; Address of previous keyboard handler
KeyExtend:  db 0    ; Flags whether an extended keystroke is being read

_InstallKeyboard:

    push di
    push es

    mov ax,03509
    int 021
    mov cs:w[KeyVect],bx
    mov cs:w[KeyVect+2],es
    cli
    push ds
    mov ax,02509
    mov dx,_KeyboardInt
    mov ds,cs
    int 21h
    pop ds
    sti

    mov w[_KeyPressed],0        ; Clear KeyTable
    mov cs:b[KeyExtend],0
    mov di,_KeyTable
    mov es,ds
    xor ax,ax
    mov cx,KEYTABLE_LENGTH/2
    rep stosw

    pop es
    pop di
    ret



; Restores original keyboard interupt

_RemoveKeyboard:

    cli
    push ds
    mov ax,02509
    mov dx,cs:w[KeyVect]
    mov ds,cs:w[KeyVect+2]
    int 21h
    pop ds
    sti
    ret



; The keyboard handler - called when any key is pressed or released

_KeyboardInt:

    push ax
    push bx
    push cx
    push ds
    pushf

    push SEG _Data
    pop ds

    xor     cx,cx
kin02:
    in      al,64h
    test    al,2
    loopnz  kin02

    cli
    in al,KEY_PORT              ; Read key data
    sti

    cmp cs:b[KeyExtend],0
    jnz kin10

    cmp al,EXTEND_CODE
    jz kin20

    xor ah,ah                   ; Normal keystroke
    mov bx,ax
    and bl,07Fh
    shl ax,1
    xor ah,1
    mov al,bl                   ; bl = key code, 0-127
    add bx,_KeyTable
    mov b[bx],ah
    cmp ah,0
    jz kin01
    mov b[_KeyPressed],al
    jmp kin00

kin10:                          ; Second part of an extended keystroke
    xor ah,ah
    mov bx,ax
    and bl,07Fh
    shl ax,1
    xor ah,1
    mov al,bl
    add bx,_KeyTable+128        ; bl = extended key code, 128-255
    mov b[bx],ah
    mov cs:b[KeyExtend],0
    cmp ah,0
    jz kin01
    or al,128
    mov b[_KeyPressed],al
    jmp kin00

kin20:
    mov cs:b[KeyExtend],1       ; First part of an extended keystroke
    jmp kin00

kin01:
    mov b[_KeyPressed],0        ; Key was released not pressed

kin00:
    mov al,KEY_ACK              ; Acknowledge and finish
    out KEY_ACK,al

    popf
    pop ds
    pop cx
    pop bx
    pop ax
    iret



; Writes the state of joystick <stk> (1 or 2) to the addresses given
; (x,y) is the position the joystick is held in
; b1, b2 are the states (0 or 1) of the buttons 1 and 2
; The timing measurements which give the values for x and y are performed
;  together, one timer giving out before the other (messy code!)
; If either of the x, y values are zero the joystick is not present

_ReadJoystick:  ; void ReadJoystick(int stk, int *x, int *y, int *b1, int *b2)

    push bp
    mov bp,sp
    push si

    mov ah,3
    cmp b[bp+4],2       ; first or second joystick?
    jnz rjo00
    mov ah,12
rjo00:
    mov dx,JOY_PORT

    mov bl,255          ; bl used to check buttons pressed
    mov cx,0
    cli
    out dx,al
rjo01:
    in al,dx            ; time both x- and y-axes simultaneously
    and bl,al
    not al
    test al,ah
    loopz rjo01
    push cx
    mov bh,al           ; one of the axes has timed-out
rjo02:
    not al              ; dummy instruction for timing purposes
    in al,dx
    and bl,al
    test al,ah
    loopnz rjo02
    sti

    mov ah,1
    cmp b[bp+4],2
    jnz rjo03
    mov ah,4
rjo03:
    test bh,ah
    jnz rjo04
    mov si,w[bp+6]      ; y-axis went first
    neg cx
    mov w[si],cx
    pop cx
    mov si,w[bp+8]
    neg cx
    mov w[si],cx
    jmp rjo05
rjo04:
    mov si,w[bp+8]      ; x-axis went first
    neg cx
    mov w[si],cx
    pop cx
    mov si,w[bp+6]
    neg cx
    mov w[si],cx
rjo05:

    shl ah,4            ; set buttons
    mov si,w[bp+10]
    mov w[si],0
    test bl,ah
    jnz rjo06
    mov w[si],1
rjo06:
    shl ah,1
    mov si,w[bp+12]
    mov w[si],0
    test bl,ah
    jnz rjo07
    mov w[si],1
rjo07:

    pop si
    pop bp
    ret

    db 84, 104, 105, 115, 32, 103, 97, 109, 101, 32, 105, 115
    db 32, 100, 101, 100, 105, 99, 97, 116, 101, 100, 32, 116
    db 111, 32, 99, 111, 110, 115, 112, 105, 114, 97, 99, 121
    db 32, 116, 104, 101, 111, 114, 105, 115, 116, 115, 32
    db 101, 118, 101, 114, 121, 119, 104, 101, 114, 101, 0



; Routine for copying a sprite to Mode X screen memory
; No clipping is performed and the sprite is not masked
; The sprite is drawn at coordinates (x, y) from source address <bmap>
;  and is <width> pixels by <height> pixels
; <offset> is the start address of the Mode X page drawn to
; WARNING: The sprite cannot be wider than 255 pixels

_DrawSprite:    ; void DrawSprite(int x, int y, char far * bmap,
                ;                 int width, int height, unsigned offset);
    push bp
    mov bp,sp
    push es
    push ds
    push si
    push di

    mov si,w[bp+8]
    mov ax,w[bp+10]
    mov ds,ax               ; ds:si = start of bitmap
    mov di,w[bp+16]
    mov ax,VID_SEG
    mov es,ax               ; es:di = start of video page
    mov al,10
    mul b[bp+6]
    shl ax,3
    add di,ax               ; es:di = start of screen line
    mov ax,w[bp+4]
    mov bl,al
    and bl,3                ; bl = first video plane
    shr ax,2
    add di,ax               ; es:di = start address of image

    mov ax,00102        ; Plane 0
    mov dx,SEQU_ADDR
    out dx,ax

    mov ch,b[bp+14]         ; ch = rows to draw
    push si
    push di
    cmp bl,0
    jz dsp01
    inc di
dsp01:
    mov al,4
    sub al,bl
    and al,3
    xor ah,ah
    add si,ax
    mov bh,b[bp+12]
    sub bh,al
    dec bh
    shr bh,2
    inc bh                  ; bh = columns to draw
    mov dx,SCR_WIDTH/4
    sub dl,bh               ; dx = columns to skip

dsp02:
    mov cl,bh
    push si
dsp03:
    mov al,ds:b[si]
    add si,4
    mov es:b[di],al
    inc di
    dec cl
    jnz dsp03
    add di,dx
    pop si
    add si,w[bp+12]
    dec ch
    jnz dsp02

    pop di
    pop si

    mov ax,00202        ; Plane 1
    mov dx,SEQU_ADDR
    out dx,ax

    mov ch,b[bp+14]         ; ch = rows to draw
    push si
    push di
    cmp bl,2
    jc dsp11
    inc di
dsp11:
    mov al,5
    sub al,bl
    and al,3
    xor ah,ah
    add si,ax
    mov bh,b[bp+12]
    sub bh,al
    dec bh
    shr bh,2
    inc bh                  ; bh = columns to draw
    mov dx,SCR_WIDTH/4
    sub dl,bh               ; dx = columns to skip

dsp12:
    mov cl,bh
    push si
dsp13:
    mov al,ds:b[si]
    add si,4
    mov es:b[di],al
    inc di
    dec cl
    jnz dsp13
    add di,dx
    pop si
    add si,w[bp+12]
    dec ch
    jnz dsp12

    pop di
    pop si

    mov ax,00402        ; Plane 2
    mov dx,SEQU_ADDR
    out dx,ax

    mov ch,b[bp+14]         ; ch = rows to draw
    push si
    push di
    cmp bl,3
    jc dsp21
    inc di
dsp21:
    mov al,6
    sub al,bl
    and al,3
    xor ah,ah
    add si,ax
    mov bh,b[bp+12]
    sub bh,al
    dec bh
    shr bh,2
    inc bh                  ; bh = columns to draw
    mov dx,SCR_WIDTH/4
    sub dl,bh               ; dx = columns to skip

dsp22:
    mov cl,bh
    push si
dsp23:
    mov al,ds:b[si]
    add si,4
    mov es:b[di],al
    inc di
    dec cl
    jnz dsp23
    add di,dx
    pop si
    add si,w[bp+12]
    dec ch
    jnz dsp22

    pop di
    pop si

    mov ax,00802        ; Plane 3
    mov dx,SEQU_ADDR
    out dx,ax

    mov ch,b[bp+14]         ; ch = rows to draw
    mov al,7
    sub al,bl
    and al,3
    xor ah,ah
    add si,ax
    mov bh,b[bp+12]
    sub bh,al
    dec bh
    shr bh,2
    inc bh                  ; bh = columns to draw
    mov dx,SCR_WIDTH/4
    sub dl,bh               ; dx = columns to skip

dsp32:
    mov cl,bh
    push si
dsp33:
    mov al,ds:b[si]
    add si,4
    mov es:b[di],al
    inc di
    dec cl
    jnz dsp33
    add di,dx
    pop si
    add si,w[bp+12]
    dec ch
    jnz dsp32

    pop di
    pop si
    pop ds
    pop es
    pop bp
    ret



; Basic function to set a SoundBlaster/Adlib soundcard register
; Register <reg> is loaded with value <val>
; The appropriate delays are made after accessing the ports

_SBSetReg:          ; void SBSetReg(int reg, int val)

    push bp
    mov bp,sp
    push ax
    push cx
    push dx

    mov dx,SB_STATUS_PORT
    mov al,b[bp+4]
    out dx,al

    mov cx,7
srg01:
    in al,dx
    loop srg01

    mov al,b[bp+6]
    inc dx
    out dx,al

    mov cx,35
srg02:
    in al,dx
    loop srg02

    pop dx
    pop cx
    pop ax
    pop bp
    ret



; Very basic function for reading the SoundBlaster/Adlib status byte

_SBReadStatus:      ; int SBReadStatus()

    push dx
    mov dx,SB_STATUS_PORT
    in al,dx
    xor ah,ah
    pop dx
    ret



; Routine to fill up the right-hand viewing window with 'blue static'
; The 'static' (starting at address <stat> and going on for <length> bytes)
;  is copied repeatedly to the display window until the window is filled up
; The four Mode X planes are handled in turn

_DisplayStatic:     ; void DisplayStatic(char far * stat, int length);

    push bp
    mov bp,sp
    push di
    push si
    push es
    push ds

    mov di,w[_HiddenPageOffs]
    add di,SCR_WIDTH*48/4+2+SCR_WIDTH/8
    mov ax,VID_SEG
    mov es,ax               ; es:di = screen destination
    mov ax,w[bp+6]
    mov ds,ax
    mov si,w[bp+4]          ; ds:si = static
    mov bx,w[bp+8]          ; bx = length of static

    mov dx,SEQU_ADDR
    mov ax,00102
    out dx,ax
    push di
    call dst00              ; video plane 0
    pop di

    mov dx,SEQU_ADDR
    mov ax,00202
    out dx,ax
    push di
    call dst00              ; video plane 1
    pop di

    mov dx,SEQU_ADDR
    mov ax,00402
    out dx,ax
    push di
    call dst00              ; video plane 2
    pop di

    mov dx,SEQU_ADDR
    mov ax,00802
    out dx,ax
    call dst00              ; video plane 3

    pop ds
    pop es
    pop si
    pop di
    pop bp
    ret


dst00:                          ; Draw one pageful of static
    mov dh,VIEW_WIDTH
dst01:
    cmp bx,VIEW_WIDTH/4 + 1             ; bx = static remaining
    jnc dst02
    mov cx,bx                           ; Static runs out this line...
    rep movsb
    mov si,w[bp+4]
    mov cx,VIEW_WIDTH/4
    sub cx,bx
    mov bx,w[bp+8]                      ; ...so go back for more static
    sub bx,cx
    rep movsb
    jmp dst03
dst02:
    mov cx,VIEW_WIDTH/4                 ; Enough static to do this line
    sub bx,cx
    rep movsb
dst03:
    add di,(SCR_WIDTH - VIEW_WIDTH)/4
    dec dh
    jnz dst01
    ret



_Text   ENDS

