
/*
 *  SQ_FRONT.C
 *
 *  Simon Hern     (Wolfson College, Cambridge, CB3 9BB, England)
 *  Summer 1995    (Email: sdh20@cam.ac.uk)
 *
 *  Code for "Squidgy Wars" game: title page routines
 *
 */


/* Squidgy Header */
#include "squidgy.h"



/* The amount of random static data used for the 3 frames of 'blue static' */
#define STATIC_SIZE_1 5*7*7*7
#define STATIC_SIZE_2 5*5*5*7
#define STATIC_SIZE_3 5*5*7*7
/* The colours making up the static */
#define STATIC_COL_1 0x12
#define STATIC_COL_2 0x13
#define STATIC_COL_3 0x14
#define STATIC_COL_4 0x11

/* Colours making up the double helix patterns on the introduction page */
#define RED_HELIX_COL 0xA0
#define BLUE_HELIX_COL 0x15
#define RED_LIGHT_COL 0xD6
#define BLUE_LIGHT_COL 0x93

/* The number of pieces of text that can be buffered to be printed */
/* on screen at any one time                                       */
#define PRINT_BUFFER_ENTRIES 30
/* The amount of memory allocated in the print buffer to each character */
/* that is to be printed                                                */
#define PRINT_BUFF_PER_CHAR 40

/* Number of squidgies used in the 'blob demo' */
#define TOTAL_BLOBS 16
/* Maximum displacement of the flying blobs from the centre of the window */
#define AMPLITUDE 56
/* Initial values for FirstBlobX and FirstBlobY */
#define FIRST_BLOB_X 0
#define FIRST_BLOB_Y (TOTAL_ANGLES/4)
/* Limits on values taken by NextBlobX and NextBlobY */
#define MAX_NEXT (TOTAL_ANGLES/TOTAL_BLOBS)
#define MIN_NEXT (TOTAL_ANGLES/TOTAL_BLOBS)/2
/* Initial values for BlobVelX and BlobVelY */
#define BLOB_VEL_X 15
#define BLOB_VEL_Y 15
/* Range of values for BlobVelX and BlobVelY */
#define MIN_VEL 9
#define MAX_VEL 21
/* Delay till BlobVelX, BlobVelY next vary */
#define VEL_CHANGE_TIME 8

/* Music - number of available channels */
#define MUSIC_CHANNELS 9
/* Number of instruments that can be defined in the music script */
#define MAX_INSTRUMENTS 9

/* The authors */
#define NAME_CHECK "*** WRITTEN BY SIMON HERN AND ROLAND HIGGINSON ***"



/* STRUCTURE DECLARATIONS */

/* Information stored about one channel of music */
typedef struct {
    int Instrument;  /* Number of 'instrument' this channel is to play */
    int Volume;  /* Volume of channel - 0=quiet, 9=loud */
    int DataAlign;  /* Column at which this channel's data appears in script */
    int Freq;  /* Frequency (f-number) of note currently being played */
    int Octave;  /* Octave of current note */
} ChannelData;

/* Information defining one instrument, match with Adlib registers */
typedef struct {
    int Volume1;  /* Volume of Operator 1, 0-63 */
    int Attack1, Attack2;  /* Attack values for both operators, 0-15 */
    int Decay1, Decay2;  /* Decay values, 0-15 */
    int Sustain1, Sustain2;  /* Sustain values, 0-15 */
    int Release1, Release2;  /* Release values, 0-15 */
    int Harmonic1, Harmonic2;  /* Operator harmonics, 0-15 */
    int Wave1, Wave2;  /* Wave forms, 0-3 */
    int Trem1, Trem2;  /* Tremolo settings, 0 or 1 */
    int Vib1, Vib2;  /* Vibrato settings, 0 or 1 */
    int Feedback;  /* Feedback value, 0-7 */
} InstrumentData;



/* EXTERNAL FUNCTIONS */

/* SQ_MAIN.C */

extern void Error(char * comment);

/* SQ_GAME.C */

extern void UndrawSprite(int x, int y, int width, int height, unsigned offset);

/* SQ_CODE.ASM */

extern void DrawSprite(int x, int y, char far * bmap,
                       int width, int height, unsigned offset);
extern void ReadJoystick(int stick, int * x, int * y, int * b1, int * b2);
extern void SBSetReg(int reg, int val);
extern void DisplayStatic(char far * stat, int length);

/* XLib */

extern void x_line(WORD x0, WORD y0, WORD x1, WORD y1, WORD col, WORD offs);
extern void x_rect_pattern(WORD StartX, WORD StartY, WORD EndX, WORD EndY,
                           WORD PageBase, BYTE far * Pattern);
extern int x_compile_bitmap(WORD lsw, char far * bitmap, char far * dest);
extern int x_sizeof_cbitmap(WORD lsw, char far * bitmap);
extern void x_put_cbitmap(WORD X, WORD Y, WORD offs, char far * bitmap);




/* EXTERNAL VARIABLES */

/* SQ_MAIN.C */

extern char * TileSprites;
extern int Joysticks;
extern int SoundOn;
extern int MusicSwitch;
extern int IntroSwitch;

/* SQ_GAME.C */

extern PlayerStruct Player[2];

/* .ASM */

extern char far Font[CHAR_HEIGHT];
extern char far Logo[LOGO_WIDTH*LOGO_HEIGHT];
extern char Map[MAP_HEIGHT][MAP_WIDTH];
extern char KeyTable[256];
extern int KeyPressed;
extern int far Sines[TOTAL_ANGLES];
extern char far MusicScript[100];

/* XLib */

extern long VsyncIntTicks;
extern WORD HiddenPageOffs;
extern WORD VisiblePageOffs;
extern WORD Page0_Offs;
extern WORD Page1_Offs;
extern WORD Page2_Offs;



/* FUNCTION DECLARATIONS */

int Titles();

void NextFrame();

void GenerateStatic(char far * stat, unsigned size);
void StaticDisplay();

void PrepareText(int across, int down, char * text, int col);
void PrintText();
void WipePrintBuffer();

void ControlsMenu(int plyr);
void RedefineKeysMenu(int plyr);
void CalibJoystickMenu(int plyr, int stick);

void CompileBlobs();
void BlobDemo();
void BlobBackground();

void ReadInstruments();
void GetInstrument(InstrumentData * instr, char far * p);
void PrepareInstrument(int i);
void PlayInstruments();
void MuteChannel(int i);
void KillChannel(int i);

void Prelude();
void DrawHelices();

char far * SkipBlankLines(char far * p);
char far * SkipCommentLines(char far * p);
char far * NextLine(char far * p);
char far * FindColumn(char far * p, int column);
int GetNumber(char far * p);




/* GLOBAL VARIABLES */

int DemoOn;
int MusicOn;

char far * Static[3];  /* Pointers to the 3 frames of static data */

PrintStruct PrintBuffer[PRINT_BUFFER_ENTRIES];  /* The text printing buffer */

char far * BlobSprite1;  /* Xlib compiled sprite for Blue Squidgy blob */
char far * BlobSprite2;  /* Xlib compiled sprite for Red Squidgy blob */
int FirstBlobX=FIRST_BLOB_X, FirstBlobY=FIRST_BLOB_Y;  /* x and y positions */
        /* of blobs vary on a sine wave. These values are the 'angles' of   */
        /* the first blob in the string                                     */
int NextBlobX, NextBlobY;  /* Change in 'angles' between subsequent blobs */
int BlobVelX=BLOB_VEL_X, BlobVelY=BLOB_VEL_Y;  /* Rate of change of */
                                       /* FirstBlobX and FirstBlobY */

char far * MusicPointer;  /* Pointer to current position in music script */
InstrumentData Instruments[MAX_INSTRUMENTS];  /* Instrument settings */
ChannelData Channel[MUSIC_CHANNELS];  /* Array of music channel settings */
int TotalInstruments;  /* Number of instruments defined */
int ActiveChannels;  /* Number of channels used for music */
int FramesPerBeat;  /* Speed of music relative to video update */

/* Frequency values for notes A to G */
int NaturalFreqTable[7] = { 0x11F, 0x141, 0x155, 0x181, 0x1B0, 0x1CA, 0x202 };
/* Sharpened frequency values for notes A# to G# */
int SharpFreqTable[7] =   { 0x12E, 0x155, 0x16B, 0x198, 0x1CA, 0x1E5, 0x220 };
/* Flattened frequency values for notes Ab to Gb */
int FlatFreqTable[7] =    { 0x112, 0x12E, 0x141, 0x16B, 0x198, 0x1B0, 0x1E5 };
/* Adlib register values for channels 0-8, operators 0&1 */
int ChannelTable[9][2] = { {0x00, 0x03}, {0x01, 0x04}, {0x02, 0x05},
                           {0x08, 0x0B}, {0x09, 0x0C}, {0x0A, 0x0D},
                           {0x10, 0x13}, {0x11, 0x14}, {0x12, 0x15} };

/* Four frames of animation for the background pattern to the blobs demo */
unsigned char Pattern1[16] = { 0x00, 0x00, 0x00, 0x7E,
                               0x7E, 0x00, 0x00, 0x00,
                               0x00, 0x00, 0x5B, 0x00,
                               0x00, 0x5B, 0x00, 0x00 };

unsigned char Pattern2[16] = { 0x00, 0x5B, 0x00, 0x00,
                               0x00, 0x00, 0x00, 0x7E,
                               0x7E, 0x00, 0x00, 0x00,
                               0x00, 0x00, 0x5B, 0x00 };

unsigned char Pattern3[16] = { 0x00, 0x00, 0x5B, 0x00,
                               0x00, 0x5B, 0x00, 0x00,
                               0x00, 0x00, 0x00, 0x7E,
                               0x7E, 0x00, 0x00, 0x00 };

unsigned char Pattern4[16] = { 0x7E, 0x00, 0x00, 0x00,
                               0x00, 0x00, 0x5B, 0x00,
                               0x00, 0x5B, 0x00, 0x00,
                               0x00, 0x00, 0x00, 0x7E };

/* Helix pattern, used down sides of introduction screen */
unsigned char HelixPattern[60] =
    { 0xA0, 0xA0, 0xA0, 0x00, 0x00, 0x00, 0x15, 0x15, 0x15, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0xA0, 0x00, 0x15, 0x00, 0x00, 0x00, 0x15, 0x00, 0xA0,
      0x00, 0x00, 0x00, 0x00, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0xA0, 0x00,
      0x00, 0x00, 0x00, 0x15, 0x00, 0xA0, 0x00, 0x00, 0x00, 0xA0, 0x00, 0x15,
      0x15, 0x15, 0x15, 0x00, 0x00, 0x00, 0xA0, 0xA0, 0xA0, 0x00, 0x00, 0x00 };
/* Positions of lights on helix; four frames           */
/* 3 for red then 3 for blue; -1 means 'no light here' */
int HelixLights[4][6] = { { 2,34,54, 6,50,-1 }, { 15,23,55, 17,21,49 },
                          { 0,56,-1, 8,28,48 }, { 1,41,45,  7,39,47 } };



/* THE CODE */

/* Display the title page, menus, demo, etc                  */
/* Static data is generated and the blob demo is initialised */
/* Window borders are drawn and the main menu is presented   */
/* On exit appropriate regions of the display are cleared    */

int Titles() {
    int i;
    int exit_code;
    static int first_time=1;

    DemoOn = 0;
    MusicOn = 0;
    MusicPointer = (char far *) MusicScript;

  /* Prepare the blob demo */
    CompileBlobs();
    NextBlobX = MIN_NEXT + random(MAX_NEXT-MIN_NEXT);
    NextBlobY = MIN_NEXT + random(MAX_NEXT-MIN_NEXT);

  /* Prepare the 'blue static' window background */
    Static[0] = (char far *) farmalloc(STATIC_SIZE_1);
    Static[1] = (char far *) farmalloc(STATIC_SIZE_2);
    Static[2] = (char far *) farmalloc(STATIC_SIZE_3);
    GenerateStatic(Static[0], STATIC_SIZE_1);
    GenerateStatic(Static[1], STATIC_SIZE_2);
    GenerateStatic(Static[2], STATIC_SIZE_3);

  /* Ensure the print buffer is clear */
    for ( i = 0 ; i < PRINT_BUFFER_ENTRIES ; i++ ) PrintBuffer[i].Times = 0;

  /* Display introduction page */
    if ( first_time ) {
        if ( SoundOn && !MusicSwitch ) MusicOn = 1;
        ReadInstruments();
        Prelude();
        first_time = 0;
    }

  /* Print credits at bottom of screen (buffered; only appears later) */
    PrepareText(3,27, NAME_CHECK, CYAN_TEXT);

  /* Three times (one for each video buffer) draw window borders */
  /* and clear the windows and bottom section of screen          */
    for ( i = 0 ; i < 3 ; i++ ) {

      /* Border to left window */
        x_line(5,45, 5,45+VIEW_WIDTH+5,
               OUTLINE_COLOUR, HiddenPageOffs);
        x_line(5,45, 5+VIEW_WIDTH+5,45,
               OUTLINE_COLOUR, HiddenPageOffs);
        x_line(5+VIEW_WIDTH+5,45+VIEW_WIDTH+5, 5,45+VIEW_WIDTH+5,
               OUTLINE_COLOUR, HiddenPageOffs);
        x_line(5+VIEW_WIDTH+5,45+VIEW_WIDTH+5, 5+VIEW_WIDTH+5,45,
               OUTLINE_COLOUR, HiddenPageOffs);

      /* Border to right window */
        x_line(160+5,45, 160+5,45+VIEW_WIDTH+5,
               OUTLINE_COLOUR, HiddenPageOffs);
        x_line(160+5,45, 160+5+VIEW_WIDTH+5,45,
               OUTLINE_COLOUR, HiddenPageOffs);
        x_line(160+5+VIEW_WIDTH+5,45+VIEW_WIDTH+5, 160+5,45+VIEW_WIDTH+5,
               OUTLINE_COLOUR, HiddenPageOffs);
        x_line(160+5+VIEW_WIDTH+5,45+VIEW_WIDTH+5, 160+5+VIEW_WIDTH+5,45,
               OUTLINE_COLOUR, HiddenPageOffs);

      /* Clear left window, right window and bottom of screen */
        UndrawSprite(8, 48, VIEW_WIDTH, VIEW_WIDTH, HiddenPageOffs);
        UndrawSprite(168, 48, VIEW_WIDTH, VIEW_WIDTH, HiddenPageOffs);
        UndrawSprite(0, 195, 320, 240-195, HiddenPageOffs);

        NextFrame();
    }

  /* Display the menus */
    do {

        DemoOn = 1;
        exit_code = RET_CONTINUE;

      /* Clear menu window with 'blue static' */
        StaticDisplay();

      /* Main menu */
        PrepareText(62,9, "1:", WHITE_TEXT);
        PrepareText(68,9, "SELECT CONTROLS", GREY_TEXT);
        PrepareText(68,10, "FOR PLAYER ONE", GREY_TEXT);

        PrepareText(62,13, "2:", WHITE_TEXT);
        PrepareText(68,13, "SELECT CONTROLS", GREY_TEXT);
        PrepareText(68,14, "FOR PLAYER TWO", GREY_TEXT);

        PrepareText(62,17, "SPACE:", WHITE_TEXT);
        PrepareText(76,17, "START GAME", GREY_TEXT);

        PrepareText(62,20, "ESCAPE:", WHITE_TEXT);
        PrepareText(78,20, "EXIT", GREY_TEXT);

      /* Consider key presses; 1 or 2 calls subroutine ControlsMenu() */
      /* whilst 'space' begins the game (RET_GAME) and 'abort' kills  */
      /* the program (RET_ESCAPE)                                     */
        while ( exit_code == RET_CONTINUE ) {

            NextFrame();

          /* Abort pressed */
            if ( AbortPressed() ) exit_code = RET_ESCAPE;
          /* Space pressed */
            if ( KeyTable[57] ) exit_code = RET_GAME;
          /* '1' pressed */
            if ( KeyTable[2] ) {
                ControlsMenu(0);
                exit_code = RET_RENEW;
            }
          /* '2' pressed */
            if ( KeyTable[3] ) {
                ControlsMenu(1);
                exit_code = RET_RENEW;
            }

        }

    } while ( exit_code == RET_RENEW );

  /* Stop music and demo */
    DemoOn = 0;
    if ( MusicOn ) {
        MusicOn = 0;
        for ( i = 0 ; i < ActiveChannels ; i++ ) KillChannel(i);
    }

  /* Clear the windows and the bottom of the screen */
    for ( i = 0 ; i < 3 ; i++ ) {
        UndrawSprite(8, 48, VIEW_WIDTH, VIEW_WIDTH, HiddenPageOffs);
        UndrawSprite(168, 48, VIEW_WIDTH, VIEW_WIDTH, HiddenPageOffs);
        UndrawSprite(0, 195, 320, 240-195, HiddenPageOffs);
        NextFrame();
    }

  /* Free-up memory */

    WipePrintBuffer();

    farfree(Static[0]);
    farfree(Static[1]);
    farfree(Static[2]);

    farfree(BlobSprite1);
    farfree(BlobSprite2);

    return exit_code;
}



/* Flip buffered video page                                          */
/* But in addition produce the next frame of the blob demo and print */
/*  this frame's buffered text                                       */

void NextFrame() {
    PrintText();
    if ( DemoOn ) BlobDemo();
    x_page_flip(0, 0);
    if ( SoundOn && MusicOn ) PlayInstruments();
}



/* Fill up the array <stat> with <size> bytes of random 'static' */

void GenerateStatic(char far * stat, unsigned size) {
    char far * p;
    unsigned j;

    p = stat;
    for ( j = 0 ; j < size ; j++ ) {
        switch( random(4) ) {
          case 0:
            *p++ = STATIC_COL_1;
            break;
          case 1:
            *p++ = STATIC_COL_2;
            break;
          case 2:
            *p++ = STATIC_COL_3;
            break;
          case 3:
            *p++ = STATIC_COL_4;
            break;
        }
    }

}



/* Draw the 'blue static' display in the right-hand window for all      */
/*  three video buffers (still animating blob demo whilst this happens) */
/* The static is drawn using an assembler routine                       */

void StaticDisplay() {
	int i;

	for ( i = 0 ; i < 3 ; i++ ) {
        PrintText();
        if ( DemoOn ) BlobDemo();

        if ( HiddenPageOffs == Page0_Offs )
			DisplayStatic(Static[0], STATIC_SIZE_1);
		if ( HiddenPageOffs == Page1_Offs )
			DisplayStatic(Static[1], STATIC_SIZE_2);
		if ( HiddenPageOffs == Page2_Offs )
			DisplayStatic(Static[2], STATIC_SIZE_3);

        x_page_flip(0, 0);
        if ( SoundOn && MusicOn ) PlayInstruments();
	}

}



/* Clear the print buffer and free-up its allocated memory */

void WipePrintBuffer() {
    int i;
    for ( i = 0 ; i < PRINT_BUFFER_ENTRIES ; i++ ) {
        if ( PrintBuffer[i].Times ) {
            PrintBuffer[i].Times = 0;
            free(PrintBuffer[i].Data);
        }
    }
}



/* Add string <text> to the print buffer, to be displayed at position */
/*  <across> (measured in half character widths), <down> (measured in */
/*  character heights)                                                */
/* The text is displayed using colour value <col>                     */
/*                                                                    */
/* A text buffer must be used because of the triple buffered video -  */
/*  anything that's not being redrawn each frame must be drawn        */
/*  initially three times                                             */
/* For each frame the contents of the print buffer are displayed,     */
/*  entries in the buffer being erased after three frames             */
/*                                                                    */
/* The data in the buffer for each piece of text takes the form of    */
/*  four list of values, one list for each of the Mode X video planes */
/* The values in the list are the changes in screen address between   */
/*  each pixel to be plotted and the next, starting with the          */
/*  displacement from the first address in the screen memory          */
/* The lists terminate with the value -1 (0xFFFF)                     */
/*                                                                    */
/* The font contains 59 characters: ascii ' ' to 'Z'                  */

void PrepareText(int across, int down, char * text, int col) {
    int ent;  /* Entry position in print buffer */
    int x, y;  /* Screen position text printed at */
    unsigned scr_addr;  /* Screen address text printed at */
    int len;  /* Length of text */
    char far * source;  /* Pointer into font data */
    int buff_width;  /* Width of mini display buffer */
    char * buff;  /* Mini display buffer text expanded into */
    char * dest;  /* Pointer into mini display buffer */
    unsigned space;  /* Distance between pixels to be plotted */
    unsigned * write;  /* Pointer into print buffer plotting list */
    char * read;  /* Pointer into mini display buffer */
    unsigned char c;
    int i, j, k;

  /* Find an unused entry in the print buffer */
    for ( ent = 0 ; PrintBuffer[ent].Times != 0 ; ent++ ) {
        if ( ent == PRINT_BUFFER_ENTRIES-1 ) return;
    }

  /* Work out screen address text is to be printed at */
    x = across*(CHAR_WIDTH+1)/2;
    y = down*(CHAR_HEIGHT+1);
    scr_addr = y*(SCR_WIDTH/4) + x/4;

  /* Create mini display buffer and draw the text to it using the font data */
  /* The text is aligned in the buffer so that the first column in the      */
  /*  buffer is always matched up to video plane 0 in Mode X screen memory  */

    len = strlen(text);
    buff_width = (CHAR_WIDTH+1)*len + 3;
    buff = (char *) malloc( buff_width*CHAR_HEIGHT );

    dest = buff;  /* Clean the buffer */
    for ( i = 0 ; i < buff_width*CHAR_HEIGHT ; i++ ) *dest++ = 0;

    dest = buff + (x & 3);  /* First column of buff matches plane 0 */
    for ( i = 0 ; i < len ; i++ ) {
        source = Font + CHAR_HEIGHT*(text[i] - ' ');
        for ( j = 0 ; j < CHAR_HEIGHT ; j++ ) {
            c = *source++;
            for ( k = 0 ; k < CHAR_WIDTH ; k++ ) {
                if ( c & 128 ) *dest = 1;
                c = c << 1;
                dest++;
            }
            dest += buff_width - CHAR_WIDTH;
        }
        dest -= CHAR_HEIGHT*buff_width - (CHAR_WIDTH+1);
    }

  /* Create a print buffer entry */
    PrintBuffer[ent].Times = 3;
    PrintBuffer[ent].Colour = col;
  /* Memory is allocated based on the number of characters to be drawn */
  /* One word of memory is needed for each pixel drawn                 */
    PrintBuffer[ent].Data = (unsigned *) malloc(len*PRINT_BUFF_PER_CHAR);

  /* Make the four lists of data - one for each video plane           */
  /* Each value is a difference in screen address between the pixel   */
  /*  just plotted and the pixel next to plot; -1 terminates the list */

    write = PrintBuffer[ent].Data;
    for ( i = 0 ; i < 4 ; i++ ) {
        space = scr_addr;
        for ( j = 0 ; j < CHAR_HEIGHT ; j++ ) {
            read = buff + i + j*buff_width;
            for ( k = 0 ; k < (buff_width - i + 3)/4 ; k++ ) {
                if ( *read ) {
                    *write++ = space;
                    space = 0;
                }
                space++;
                read += 4;
            }
            space += SCR_WIDTH/4 - (buff_width - i + 3)/4;
        }
        *write++ = 0xFFFF;
    }

    free(buff);
}



/* Plot the contents of the print buffer and reduce the number of        */
/*  times each entry is to be shown (freeing released memory when ready) */

void PrintText() {
    int ent;
    char col;
    char far * p;
    unsigned * source;
    unsigned s;
    int i;

    for ( ent = 0 ; ent < PRINT_BUFFER_ENTRIES ; ent++ ) {
        if ( PrintBuffer[ent].Times ) {

            col = PrintBuffer[ent].Colour;
            source = PrintBuffer[ent].Data;
            for ( i = 1 ; i < 16 ; (i=i<<1) ) {
                outport(0x03C4, 256*i + 2);
                p = MK_FP(0xA000, HiddenPageOffs);
                while ( (s = *source++) != 0xFFFF ) {
                    p += s;
                    *p = col;
                }
            }

            if ( --PrintBuffer[ent].Times == 0 ) {
                free(PrintBuffer[ent].Data);
            }

        }
    }
}



/* Present the controls selection menu system                            */
/* (The form of the controls menu depends on the number of joysticks     */
/*  that are available)                                                  */
/* The key definition and joystick definition menus are called from here */

void ControlsMenu(int plyr) {
    int exit_code;

    exit_code = RET_CONTINUE;

  /* No joysticks attached - go straight to redefining keys */
    if ( (Joysticks & 3) == 0 ) {
        RedefineKeysMenu(plyr);
        return;
    }

    StaticDisplay();

    if ( plyr == 0 ) PrepareText(70,9, "PLAYER ONE", WHITE_TEXT);
    else PrepareText(70,9, "PLAYER TWO", WHITE_TEXT);

    PrepareText(65,11, "SELECT CONTROLS", WHITE_TEXT);

    PrepareText(67,15, "A:", WHITE_TEXT);
    PrepareText(73,15, "KEYBOARD", GREY_TEXT);

  /* Joystick 1 at least is present */
    if ( Joysticks & 1 ) {

        PrepareText(67,17, "B:", WHITE_TEXT);
        PrepareText(73,17, "JOYSTICK 1", GREY_TEXT);

      /* Also Joystick 2 */
        if ( Joysticks & 2 ) {
            PrepareText(67,19, "C:", WHITE_TEXT);
            PrepareText(73,19, "JOYSTICK 2", GREY_TEXT);
        }

        while ( exit_code == RET_CONTINUE ) {

            NextFrame();

            if ( AbortPressed() ) exit_code = RET_ESCAPE;
            if ( KeyTable[30] ) {
                RedefineKeysMenu(plyr);  /* 'A' pressed */
                exit_code = RET_ESCAPE;
            }
            if ( KeyTable[48] ) {
                CalibJoystickMenu(plyr, 1);  /* 'B' pressed */
                exit_code = RET_ESCAPE;
            }
            if ( KeyTable[46] && (Joysticks & 2) ) {
                CalibJoystickMenu(plyr, 2);  /* 'C' pressed */
                exit_code = RET_ESCAPE;
            }

        }

        return;

  /* Joystick 2 only */
    } else {

        PrepareText(67,17, "B:", WHITE_TEXT);
        PrepareText(73,17, "JOYSTICK 2", GREY_TEXT);

        while ( exit_code == RET_CONTINUE ) {

            NextFrame();

            if ( AbortPressed() ) exit_code = RET_ESCAPE;
            if ( KeyTable[30] ) {
                RedefineKeysMenu(plyr);  /* 'A' pressed */
                exit_code = RET_ESCAPE;
            }
            if ( KeyTable[48] ) {
                CalibJoystickMenu(plyr, 2);  /* 'B' pressed */
                exit_code = RET_ESCAPE;
            }

        }

        return;

    }

}



/* Menu asking for key redefinitions */

void RedefineKeysMenu(int plyr) {
    int last_key;

    StaticDisplay();

    while ( KeyPressed != 0 ) NextFrame();

    Player[plyr].ControlDevice = KEYBOARD;

    if ( plyr == 0 ) PrepareText(70,8, "PLAYER ONE", WHITE_TEXT);
    else PrepareText(70,8, "PLAYER TWO", WHITE_TEXT);

    PrepareText(69,10, "DEFINE KEYS", WHITE_TEXT);

    PrepareText(71,13, "UP?", GREY_TEXT);
    do {
        NextFrame();
    } while ( (last_key = KeyPressed) == 0 );
    Player[plyr].KeyUp = last_key;
    PrepareText(80,13, "OKAY", MAGENTA_TEXT);

    PrepareText(71,15, "DOWN?", GREY_TEXT);
    do {
        NextFrame();
    } while ( KeyPressed == last_key || (last_key = KeyPressed) == 0 );
    Player[plyr].KeyDown = last_key;
    PrepareText(84,15, "OKAY", MAGENTA_TEXT);


    PrepareText(71,17, "LEFT?", GREY_TEXT);
    do {
        NextFrame();
    } while ( KeyPressed == last_key || (last_key = KeyPressed) == 0 );
    Player[plyr].KeyLeft = last_key;
    PrepareText(84,17, "OKAY", MAGENTA_TEXT);

    PrepareText(71,19, "RIGHT?", GREY_TEXT);
    do {
        NextFrame();
    } while ( KeyPressed == last_key || (last_key = KeyPressed) == 0 );
    Player[plyr].KeyRight = last_key;
    PrepareText(86,19, "OKAY", MAGENTA_TEXT);

    PrepareText(71,21, "FIRE?", GREY_TEXT);
    do {
        NextFrame();
    } while ( KeyPressed == last_key || (last_key = KeyPressed) == 0 );
    Player[plyr].KeyFire = last_key;
    PrepareText(84,21, "OKAY", MAGENTA_TEXT);

    NextFrame();
    NextFrame();
    NextFrame();
}



/* Menu for recalibrating joystick                         */
/* (Exits and resets to keyboard control if abort pressed) */

void CalibJoystickMenu(int plyr, int stick) {
    int x_min, x_mid, x_max;
    int y_min, y_mid, y_max;
    int b1, b2;

    if ( stick == 1 ) Player[plyr].ControlDevice = JOYSTICK1;
    else Player[plyr].ControlDevice = JOYSTICK2;

    StaticDisplay();
    while ( ReadJoystick(stick, &x_min, &y_min, &b1, &b2), b1+b2 != 0 ) {
        NextFrame();
        if ( AbortPressed() ) {
            Player[plyr].ControlDevice = KEYBOARD;
            return;
        }
    }

    PrepareText(64,11, "MOVE JOYSTICK TO", GREY_TEXT);
    PrepareText(63,12, "TOP-LEFT POSITION", GREY_TEXT);
    PrepareText(66,13, "AND PRESS FIRE", GREY_TEXT);
    while ( ReadJoystick(stick, &x_min, &y_min, &b1, &b2), b1+b2 == 0 ) {
        NextFrame();
        if ( AbortPressed() ) {
            Player[plyr].ControlDevice = KEYBOARD;
            return;
        }
    }

    StaticDisplay();
    while ( ReadJoystick(stick, &x_max, &y_max, &b1, &b2), b1+b2 != 0 ) {
        NextFrame();
        if ( AbortPressed() ) {
            Player[plyr].ControlDevice = KEYBOARD;
            return;
        }
    }

    PrepareText(64,11, "MOVE JOYSTICK TO", GREY_TEXT);
    PrepareText(59,12, "BOTTOM-RIGHT POSITION", GREY_TEXT);
    PrepareText(66,13, "AND PRESS FIRE", GREY_TEXT);
    while ( ReadJoystick(stick, &x_max, &y_max, &b1, &b2), b1+b2 == 0 ) {
        NextFrame();
        if ( AbortPressed() ) {
            Player[plyr].ControlDevice = KEYBOARD;
            return;
        }
    }

    StaticDisplay();
    while ( ReadJoystick(stick, &x_mid, &y_mid, &b1, &b2), b1+b2 != 0 ) {
        NextFrame();
        if ( AbortPressed() ) {
            Player[plyr].ControlDevice = KEYBOARD;
            return;
        }
    }

    PrepareText(64,11, "MOVE JOYSTICK TO", GREY_TEXT);
    PrepareText(65,12, "CENTRE POSITION", GREY_TEXT);
    PrepareText(66,13, "AND PRESS FIRE", GREY_TEXT);
    while ( ReadJoystick(stick, &x_mid, &y_mid, &b1, &b2), b1+b2 == 0 ) {
        NextFrame();
        if ( AbortPressed() ) {
            Player[plyr].ControlDevice = KEYBOARD;
            return;
        }
    }

    Player[plyr].JoyXUpper = x_mid + (x_max - x_mid)*JOYSTICK_FIDGE_FACTOR;
    Player[plyr].JoyYUpper = y_mid + (y_max - y_mid)*JOYSTICK_FIDGE_FACTOR;
    Player[plyr].JoyXLower = x_mid + (x_min - x_mid)*JOYSTICK_FIDGE_FACTOR;
    Player[plyr].JoyYLower = y_mid + (y_min - y_mid)*JOYSTICK_FIDGE_FACTOR;

}



/* Manufactures two XLib compiled sprites (very fast to use!) based on */
/*  the (cleaned up) blue and red squidgy tiles                        */

void CompileBlobs() {
    char far * buff;
    char far * p;
    char far * q;
    int len;
    int i;

  /* Temporary buffer contains sprites in XLib 'linear' format */
    buff = (char far *) farmalloc(TILE_SIZE+2);
    *buff = (char) TILE_WIDTH;
    *(buff+1) = (char) TILE_WIDTH;

  /* Copy blue squidgy sprite */
    p = buff+2;
    q = TileSprites + TILE_SIZE*CLEAN_BLUE_TILE;
    for ( i = 0 ; i < TILE_SIZE ; i++ ) *p++ = *q++;

  /* Compile blue squidgy sprite */
    len = x_sizeof_cbitmap((unsigned)SCR_WIDTH/4, buff);
    BlobSprite1 = (char far *) farmalloc(len);
    x_compile_bitmap((unsigned)SCR_WIDTH/4, buff, BlobSprite1);

  /* Copy red squidgy sprite */
    p = buff+2;
    q = TileSprites + TILE_SIZE*CLEAN_RED_TILE;
    for ( i = 0 ; i < TILE_SIZE ; i++ ) *p++ = *q++;

  /* Compile red squidgy sprite */
    len = x_sizeof_cbitmap((unsigned)SCR_WIDTH/4, buff);
    BlobSprite2 = (char far *) farmalloc(len);
    x_compile_bitmap((unsigned)SCR_WIDTH/4, buff, BlobSprite2);

    farfree(buff);
}



/* Draw a frame of the 'blob demo'                                           */
/* Sixteen compiled sprites are thrown onto a patterned background           */
/* The paths followed by the blobs are changed slightly at random intervals  */
/* Blob positions are determined using a sine table                          */
/*                                                                           */
/* The blobs form a 'Lissajous' figure                                       */
/* Hello to Matt Spinner - from whom this demo was shamelessly stolen ;)     */

void BlobDemo() {
    int x, y;
    int xpos, ypos;
    char far * blob;
    int i;
    static int vel_change=VEL_CHANGE_TIME;  /* Time till next random */
                              /* change of blobs' 'angular velocity' */

    BlobBackground();

  /* Draw the blobs */
    x = FirstBlobX;
    y = FirstBlobY;
    for ( i = 0 ; i < TOTAL_BLOBS ; i++ ) {
        xpos = ( 8 + VIEW_WIDTH/2 - TILE_WIDTH/2 )
               + ( ( AMPLITUDE * (Sines[x]>>6) ) >> 9 );
        ypos = ( 48 + VIEW_WIDTH/2 - TILE_WIDTH/2 )
               + ( ( AMPLITUDE * (Sines[y]>>6) ) >> 9 );
        if ( i & 1 ) blob = BlobSprite1;
        else blob = BlobSprite2;
        x_put_cbitmap(xpos, ypos, HiddenPageOffs, blob);
        x = (x + NextBlobX) % TOTAL_ANGLES;
        y = (y + NextBlobY) % TOTAL_ANGLES;
    }

  /* Update blobs for next frame */
    FirstBlobX = (FirstBlobX + BlobVelX) % TOTAL_ANGLES;
    FirstBlobY = (FirstBlobY + BlobVelY) % TOTAL_ANGLES;

  /* Fiddle with the demo parameters at random intervals */
    if ( --vel_change == 0 ) {
        vel_change = VEL_CHANGE_TIME + random(VEL_CHANGE_TIME);
        switch ( random(4) ) {
          case 0:
            if ( BlobVelX < MAX_VEL ) BlobVelX++;
            break;
          case 1:
            if ( BlobVelY < MAX_VEL ) BlobVelY++;
            break;
          case 2:
            if ( BlobVelX > MIN_VEL ) BlobVelX--;
            break;
          case 3:
            if ( BlobVelY > MIN_VEL ) BlobVelY--;
            break;
        }
    }

}



/* Draw a patterned background for the blob demo                */
/* The pattern is changed each frame to give a scrolling effect */
/* XLib's patterned-fill function is used                       */

void BlobBackground() {
    static int frame = 0;
    char * pattern;

    frame = (frame + 1) & 3;
    switch (frame) {
      case 0:
        pattern = Pattern1;
        break;
      case 1:
        pattern = Pattern2;
        break;
      case 2:
        pattern = Pattern3;
        break;
      case 3:
        pattern = Pattern4;
        break;
    }

    x_rect_pattern(8, 48, 8+VIEW_WIDTH, 48+VIEW_WIDTH, HiddenPageOffs,
                   (BYTE far *) pattern);
}



/* Display an introduction page identifying some of the game characters */
/* Exit when a key is pressed, leaving the screen blank except for the  */
/*  games logo at the top of the screen                                 */

void Prelude() {
    long t;
    int i;

    for ( i = 0 ; i < PRINT_BUFFER_ENTRIES ; i++ ) PrintBuffer[i].Times = 0;

  /* Draw the introduction page */
    DrawSprite( (SCR_WIDTH-LOGO_WIDTH)/2, 6, (char far *)Logo,
        LOGO_WIDTH, LOGO_HEIGHT, (unsigned)HiddenPageOffs);

    DrawSprite(62, 64,
        (char far *)(TileSprites+TILE_SIZE*CLEAN_BLUE_TILE),
        (int)TILE_WIDTH, (int)TILE_WIDTH, (unsigned)HiddenPageOffs);
    DrawSprite(239, 64,
        (char far *)(TileSprites+TILE_SIZE*CLEAN_RED_TILE),
        (int)TILE_WIDTH, (int)TILE_WIDTH, (unsigned)HiddenPageOffs);
    PrepareText(29,8,  "FIONA", GREY_TEXT);
    PrepareText(29,9, "SQUIDGY", GREY_TEXT);
    PrepareText(53,8,  "      ROLAND", GREY_TEXT);
    PrepareText(53,9, "     SQUIDGY", GREY_TEXT);

    DrawSprite(62, 96,
        (char far *)(TileSprites+TILE_SIZE*FAST_TOKEN_TILE),
        (int)TILE_WIDTH, (int)TILE_WIDTH, (unsigned)HiddenPageOffs);
    DrawSprite(239, 96,
        (char far *)(TileSprites+TILE_SIZE*SLOW_TOKEN_TILE),
        (int)TILE_WIDTH, (int)TILE_WIDTH, (unsigned)HiddenPageOffs);
    PrepareText(29,12, "SPEED UP", GREY_TEXT);
    PrepareText(29,13, "SYMBOL", GREY_TEXT);
    PrepareText(53,12, "   SLOW DOWN", GREY_TEXT);
    PrepareText(53,13, "      SYMBOL", GREY_TEXT);

    DrawSprite(62, 128,
        (char far *)(TileSprites+TILE_SIZE*VANISH_TOKEN_TILE),
        (int)TILE_WIDTH, (int)TILE_WIDTH, (unsigned)HiddenPageOffs);
    DrawSprite(239, 128,
        (char far *)(TileSprites+TILE_SIZE*HEALTH_TOKEN_TILE),
        (int)TILE_WIDTH, (int)TILE_WIDTH, (unsigned)HiddenPageOffs);
    PrepareText(29,16, "INVISIBILITY", GREY_TEXT);
    PrepareText(29,17, "SYMBOL", GREY_TEXT);
    PrepareText(53,16, "     HEALING", GREY_TEXT);
    PrepareText(53,17, "      SYMBOL", GREY_TEXT);

    DrawSprite(62, 160,
        (char far *)(TileSprites+TILE_SIZE*(GUN_TOKEN_TILES+1)),
        (int)TILE_WIDTH, (int)TILE_WIDTH, (unsigned)HiddenPageOffs);
    DrawSprite(239, 160,
        (char far *)(TileSprites+TILE_SIZE*(GUN2_TOKEN_TILES+2)),
        (int)TILE_WIDTH, (int)TILE_WIDTH, (unsigned)HiddenPageOffs);
    PrepareText(29,20, "FIRE", GREY_TEXT);
    PrepareText(29,21, "BALLS", GREY_TEXT);
    PrepareText(53,20, "      PLASMA", GREY_TEXT);
    PrepareText(53,21, "       BALLS", GREY_TEXT);

    DrawSprite(62, 192,
        (char far *)(TileSprites+TILE_SIZE*(LASER_TOKEN_TILES+1)),
        (int)TILE_WIDTH, (int)TILE_WIDTH, (unsigned)HiddenPageOffs);
    DrawSprite(239, 192,
        (char far *)(TileSprites+TILE_SIZE*SHIELD_TOKEN_TILE),
        (int)TILE_WIDTH, (int)TILE_WIDTH, (unsigned)HiddenPageOffs);
    PrepareText(29,24, "LASER", GREY_TEXT);
    PrepareText(29,25, "BOLTS", GREY_TEXT);
    PrepareText(53,24, "     ORBITAL", GREY_TEXT);
    PrepareText(53,25, "     SHIELDS", GREY_TEXT);

    DrawHelices();

    NextFrame();

  /* Redraw all but the text on the other pages  */
  /* (The print buffer will handle the text now) */
    for ( i = 0 ; i < 2 ; i++ ) {
        DrawSprite( (SCR_WIDTH-LOGO_WIDTH)/2, 6, (char far *)Logo,
            LOGO_WIDTH, LOGO_HEIGHT, (unsigned)HiddenPageOffs);

        DrawSprite(62, 64,
            (char far *)(TileSprites+TILE_SIZE*CLEAN_BLUE_TILE),
            (int)TILE_WIDTH, (int)TILE_WIDTH, (unsigned)HiddenPageOffs);
        DrawSprite(239, 64,
            (char far *)(TileSprites+TILE_SIZE*CLEAN_RED_TILE),
            (int)TILE_WIDTH, (int)TILE_WIDTH, (unsigned)HiddenPageOffs);

        DrawSprite(62, 96,
            (char far *)(TileSprites+TILE_SIZE*FAST_TOKEN_TILE),
            (int)TILE_WIDTH, (int)TILE_WIDTH, (unsigned)HiddenPageOffs);
        DrawSprite(239, 96,
            (char far *)(TileSprites+TILE_SIZE*SLOW_TOKEN_TILE),
            (int)TILE_WIDTH, (int)TILE_WIDTH, (unsigned)HiddenPageOffs);

        DrawSprite(62, 128,
            (char far *)(TileSprites+TILE_SIZE*VANISH_TOKEN_TILE),
            (int)TILE_WIDTH, (int)TILE_WIDTH, (unsigned)HiddenPageOffs);
        DrawSprite(239, 128,
            (char far *)(TileSprites+TILE_SIZE*HEALTH_TOKEN_TILE),
            (int)TILE_WIDTH, (int)TILE_WIDTH, (unsigned)HiddenPageOffs);

        DrawSprite(62, 160,
            (char far *)(TileSprites+TILE_SIZE*(GUN_TOKEN_TILES+1)),
            (int)TILE_WIDTH, (int)TILE_WIDTH, (unsigned)HiddenPageOffs);
        DrawSprite(239, 160,
            (char far *)(TileSprites+TILE_SIZE*(GUN2_TOKEN_TILES+2)),
            (int)TILE_WIDTH, (int)TILE_WIDTH, (unsigned)HiddenPageOffs);

        DrawSprite(62, 192,
            (char far *)(TileSprites+TILE_SIZE*(LASER_TOKEN_TILES+1)),
            (int)TILE_WIDTH, (int)TILE_WIDTH, (unsigned)HiddenPageOffs);
        DrawSprite(239, 192,
            (char far *)(TileSprites+TILE_SIZE*SHIELD_TOKEN_TILE),
            (int)TILE_WIDTH, (int)TILE_WIDTH, (unsigned)HiddenPageOffs);

        DrawHelices();

        NextFrame();
    }

  /* Wait until key pressed or until bored */
    t = VsyncIntTicks;
    while ( KeyPressed == 0 && ( VsyncIntTicks < t+650 || IntroSwitch ) ) {
        DrawHelices();
        NextFrame();
    }

    for ( i = 0 ; i < PRINT_BUFFER_ENTRIES ; i++ ) PrintBuffer[i].Times = 0;

  /* Clear all three screen pages leaving only the game logo */
    for ( i = 0 ; i < 3 ; i++ ) {
        UndrawSprite( 62, 64, 193, 144, (unsigned)HiddenPageOffs);
        UndrawSprite( 20, 0, 5, 240, (unsigned)HiddenPageOffs);
        UndrawSprite( 295, 0, 5, 240, (unsigned)HiddenPageOffs);
        NextFrame();
    }
}



/* Draw the two animated helix patterns down the sides of the intro page */

void DrawHelices() {
    char far * dest;
    char * srce;
    int plane;
    int i, j, k;
    static int frame = 0;

    frame = (frame + 1) & 3;

  /* Position lights for this frame */
    for ( i = 0 ; i < 3 ; i++ ) {
        j = HelixLights[frame][i];
        if ( j >= 0) HelixPattern[j] = RED_LIGHT_COL;
        j = HelixLights[frame][i+3];
        if ( j >= 0) HelixPattern[j] = BLUE_LIGHT_COL;
    }

  /* Draw first helix */
    dest = MK_FP(0xA000, HiddenPageOffs+5);
    srce = HelixPattern;
    plane = 1;
    for ( k = 0 ; k < 5 ; k++ ) {
        outport(0x03C4, 2 + 256*plane);
        for ( i = 0 ; i < 20 ; i++ ) {
            for ( j = 0 ; j < 12 ; j++ ) {
                *dest = *srce++;
                dest += 80;
            }
            srce -= 12;
        }
        dest -= 80*240;
        srce += 12;
        plane = plane * 2;
        if ( plane == 16 ) {
            plane = 1;
            dest++;
        }
    }

  /* Draw second helix */
    dest = MK_FP(0xA000, HiddenPageOffs+73);
    srce = HelixPattern+48;
    plane = 8;
    for ( k = 0 ; k < 5 ; k++ ) {
        outport(0x03C4, 2 + 256*plane);
        for ( i = 0 ; i < 20 ; i++ ) {
            for ( j = 0 ; j < 12 ; j++ ) {
                *dest = *srce++;
                dest += 80;
            }
            srce -= 12;
        }
        dest -= 80*240;
        srce -= 12;
        plane = plane * 2;
        if ( plane == 16 ) {
            plane = 1;
            dest++;
        }
    }

  /* Remove lights from helix */
    for ( i = 0 ; i < 3 ; i++ ) {
        j = HelixLights[frame][i];
        if ( j >= 0) HelixPattern[j] = RED_HELIX_COL;
        j = HelixLights[frame][i+3];
        if ( j >= 0) HelixPattern[j] = BLUE_HELIX_COL;
    }
}



/* Hacked-together music system for Adlib soundcard                     */
/*                                                                      */
/* The music is stored in the form of an ascii script                   */
/* The code is very rough - I don't understand much about the soundcard */
/* The bad music I just have to blame on lack of talent                 */


/* Read header information from music script (up to the note data)           */
/* An instrument is defined by a letter (A,B,...) followed by 18 numbers     */
/* A single number sets the relative speed of the music                      */
/* The data for the channels are aligned in columns - instrument selection   */
/*  (A,B,...) at the top, then volume (0-9), then notes (where the range     */
/*  of octaves are notated using (eg for C) -c, -C, c, C, +c, +C increasing) */
/* Lines beginning with colons (:) are ignored                               */

void ReadInstruments() {
    int i;
    char far * p;
    int col;
    int val;
    int letter;

    if ( !SoundOn || !MusicOn ) return;

    ActiveChannels = MUSIC_CHANNELS;
    MusicPointer = SkipBlankLines(MusicPointer);

  /* Read instruments */
    for ( i = 0 ; i < MAX_INSTRUMENTS ; i++ ) {
        p = MusicPointer;
        while ( *p == ' ' ) p++;
        letter = *p++;
        if ( letter != 'A'+i ) break;
        GetInstrument(Instruments+i, p);
        MusicPointer = SkipBlankLines( NextLine(p) );
    }
    TotalInstruments = i;
    if ( i == 0 ) Error("Bad instrument definitions");
    if ( i == MAX_INSTRUMENTS ) Error("Too many instruments");

  /* Read tempo */
    p = MusicPointer;
    while ( *p == ' ' ) p++;
    val = *p - '0';
    if ( val < 2 || val > 9 ) Error("Invalid tempo definition");
    FramesPerBeat = val;

    MusicPointer = NextLine(MusicPointer);
    MusicPointer = SkipBlankLines(MusicPointer);

  /* Read channels */
    col = 0;
    for ( i = 0 ; i < MUSIC_CHANNELS ; i++ ) {
        while ( MusicPointer[col] == ' ' ) col++;

        if ( MusicPointer[col] == 0 ) Error("Premature end of file");
        if ( MusicPointer[col] == '\n' ) {
            ActiveChannels = i;
            if ( i == 0 ) Error("Bad instrument definition");
            break;
        }

        val = MusicPointer[col] - 'A';
        if ( val >= 0 && val < TotalInstruments ) {
            Channel[i].Instrument = val;
            Channel[i].DataAlign = col;
        } else {
            Error("Bad instrument definition");
        }

        col++;
    }

    MusicPointer = NextLine(MusicPointer);
    MusicPointer = SkipBlankLines(MusicPointer);

  /* Read volume */
    for ( i = 0 ; i < ActiveChannels ; i++ ) {
        p = FindColumn(MusicPointer, Channel[i].DataAlign);
        if ( p == NULL ) Error("Channel volume undefined\n");
        val = *p - '0';
        if ( val < 0 || val > 9 ) Error("Bad volume selection");
        Channel[i].Volume = val;
    }

    MusicPointer = NextLine(MusicPointer);
    MusicPointer = SkipBlankLines(MusicPointer);

  /* Set up sound card */
    SBSetReg(0x01, 32);
    for ( i = 0 ; i < ActiveChannels ; i++ ) {
        PrepareInstrument(i);
    }

}



/* Read 18 integer values at text pointed to by <p> into structure */
/*  pointed to by <instr>                                          */

void GetInstrument(InstrumentData * instr, char far * p) {
    int i, j;
    int * ins = (int *) instr;
    int val;

    for ( i = 0 ; i < 18 ; i++ ) {
        while ( *p == ' ' ) p++;
        val = GetNumber(p);
        if ( val < 0 ) Error("Bad instrument definition");
        *ins++ = val;
        while ( *p >= '0' && *p <= '9'  ) p++;
    }
}



/* Set the Adlib channel <i> to play music using the selected 'instrument' */

void PrepareInstrument(int i) {
    int i1, i2;
    InstrumentData * instr;
    unsigned char byte;

    if ( !SoundOn || !MusicOn ) return;

    i1 = ChannelTable[i][0];
    i2 = ChannelTable[i][1];

    instr = Instruments;
    instr += Channel[i].Instrument;

    Channel[i].Freq = 0;
    Channel[i].Octave = 0;
    SBSetReg(0xB0+i, 0);

    byte = (((*instr).Trem1 & 1) << 7) + (((*instr).Vib1 & 1) << 6)
           + ((*instr).Harmonic1 & 15);
    SBSetReg(0x20+i1, byte);
    byte = (((*instr).Trem2 & 1) << 7) + (((*instr).Vib2 & 1) << 6)
           + ((*instr).Harmonic2 & 15);
    SBSetReg(0x20+i2, byte);

    byte = ((*instr).Volume1 & 63) + 128+64;
    SBSetReg(0x40+i1, byte);
    byte = 5*(9 - Channel[i].Volume) + 128+64;
    SBSetReg(0x40+i2, byte);

    byte = (((*instr).Attack1 & 15) << 4) + ((*instr).Decay1 & 15);
    SBSetReg(0x60+i1, byte);
    byte = (((*instr).Attack2 & 15) << 4) + ((*instr).Decay2 & 15);
    SBSetReg(0x60+i2, byte);

    byte = (((*instr).Sustain1 & 15) << 4) + ((*instr).Release1 & 15);
    SBSetReg(0x80+i1, byte);
    byte = (((*instr).Sustain2 & 15) << 4) + ((*instr).Release2 & 15);
    SBSetReg(0x80+i2, byte);

    byte = ((*instr).Wave1 & 3);
    SBSetReg(0xE0+i1, byte);
    byte = ((*instr).Wave2 & 3);
    SBSetReg(0xE0+i2, byte);

    byte = (((*instr).Feedback & 7) << 1);
    SBSetReg(0xC0+i, byte);

}



/* Play the music script, one line at a time                        */
/* All channels are 'keyed-off' one frame after they are 'keyed-on' */
/* Each beat of the music lasts several frames of the video update  */

void PlayInstruments() {
    int octave;
    int freq;
    int val;
    int i;
    char far * p;
    unsigned char byte1, byte2;

    static int timer=-1;

    if ( !SoundOn || !MusicOn ) return;

    if ( ++timer >= FramesPerBeat ) timer = 0;
    if ( timer == 1 ) for ( i = 0 ; i < ActiveChannels ; i++ ) MuteChannel(i);
    if ( timer != 0 ) return;

    if ( *MusicPointer == 0 ) {
        MusicOn = 0;
        for ( i = 0 ; i < ActiveChannels ; i++ ) MuteChannel(i);
        return;
    }

    for ( i = 0 ; i < ActiveChannels ; i++ ) {

        p = FindColumn(MusicPointer, Channel[i].DataAlign);
        if ( p != NULL && *p != ' ' ) {

            octave = 3;
            if ( *p == '+' ) {
                octave += 2;
                p++;
            } else if ( *p == '-' ) {
                octave -= 2;
                p++;
            }
            val = *p;
            if ( val >= 'A' && val <= 'G' ) {
                val += 'a' - 'A';
                octave += 1;
            }
            if ( val < 'a' || val > 'g' ) Error("Invalid symbol for note\n");
            val -= 'a';
            p++;
            if ( *p == '#' ) freq = SharpFreqTable[val];
            else if ( *p == 'b' ) freq = FlatFreqTable[val];
            else freq = NaturalFreqTable[val];

            Channel[i].Freq = freq;
            Channel[i].Octave = octave;

            byte1 = freq & 255;
            byte2 = 32 + (octave << 2) + ((freq >> 8) & 3);
            SBSetReg(0xA0+i, byte1);
            SBSetReg(0xB0+i, byte2);
        }

    }

    MusicPointer = NextLine(MusicPointer);
    MusicPointer = SkipCommentLines(MusicPointer);

}



/* 'Key-off' music channel <i> */

void MuteChannel(int i) {
    unsigned char byte;
    byte = (Channel[i].Octave << 2) + ((Channel[i].Freq >> 8) & 3);
    SBSetReg(0xB0+i, byte);
}



/* Turn off channel <i> completely */

void KillChannel(int i) {
    unsigned char byte;
    int i1, i2;

    i1 = ChannelTable[i][0];
    i2 = ChannelTable[i][1];

    byte = 128+64+63;
    SBSetReg(0x40+i1, byte);
    SBSetReg(0x40+i2, byte);

    SBSetReg(0xA0+i, 0);
    SBSetReg(0xB0+i, 0);

    SBSetReg(0x80+i1, 15);
    SBSetReg(0x80+i2, 15);
}



/* Text processing for music script - return pointer having skipped */
/*  any blank lines or comment lines                                */

char far * SkipBlankLines(char far * p) {
    char far * q;

    for (;;) {

        while ( *p=='\n' || *p=='\r' || *p=='\f' ) p++;

        p = SkipCommentLines(p);

        if ( *p == 0 ) return p;

        q = p;
        while ( *q == ' ' || *q == '\t' ) q++;
        if ( *q==0 || *q=='\n' || *q=='\r' || *q=='\f' ) p = q;
        else return p;
    }
}



/* Text processing for music script - return pointer having skipped */
/*  any comment lines (lines beginning with a colon: )              */

char far * SkipCommentLines(char far * p) {
    for (;;) {
        if ( *p == ':' ) {
            while ( *p != '\n' ) {
                if ( *p == 0 ) return p;
                p++;
            }
            p++;
        }
        else return p;
    }
}



/* Text processing for music script - return pointer at start of */
/*  next line of text                                            */

char far * NextLine(char far * p) {
    while ( *p != '\n' ) p++;
    return p+1;
}



/* Text processing for music script - return pointer to column requested */
/*  in current line of text, or NULL if line too short                   */

char far * FindColumn(char far * p, int column) {
    int i;
    for ( i = 0 ; i < column ; i++ ) {
        if ( *p == '\n' || *p == 0 ) return NULL;
        p++;
    }
    return p;
}



/* Text processing for music script - return integer value translated */
/*  from text at <p>, or -1 if not a number                           */

int GetNumber(char far * p) {
    int val = 0;

    if ( *p < '0' || *p > '9' ) return -1;
    while ( *p >= '0' && *p <= '9' ) {
        val = val * 10;
        val += *p - '0';
        p++;
    }
    return val;
}



