
/*
 *  SQ_GAME.C
 *
 *  Simon Hern     (Wolfson College, Cambridge, CB3 9BB, England)
 *  Summer 1995    (Email: sdh20@cam.ac.uk)
 *
 *  Code for "Squidgy Wars" game: game-playing routines
 *
 */


/* Squidgy Header */
#include "squidgy.h"



/* EXTERNAL FUNCTIONS */

/* SQ_MAIN.C */

extern int AbortPressed();

extern int TileType(int x, int y);
extern int TileTypeNum(int t);
extern void SwitchMaps();

extern void PrepareSoundss();
extern void MakeSound(int num);
extern void SoundEffects();

/* SQ_CODE.ASM */

extern void MapToViewBuffer(int LeftEdge, int TopEdge, char far * buff);
extern void ViewBufferToScr(char far * buff, unsigned scr_addr);
extern void DrawObject(char far * view, int x, int y, int tile);

extern void InstallKeyboard();
extern void RemoveKeyboard();
extern void ReadJoystick(int stick, int * x, int * y, int * b1, int * b2);

extern void DrawSprite(int x, int y, char far * bmap,
                       int width, int height, unsigned offset);

/* XLib */

extern void x_page_flip(WORD x, WORD y);
extern void x_rect_fill(WORD StartX, WORD StartY, WORD EndX, WORD EndY,
                        WORD PageBase, WORD Colour);



/* EXTERNAL VARIABLES */

/* SQ_MAIN.C */

extern char * TileSprites;

/* .ASM */

extern char Map[MAP_HEIGHT][MAP_WIDTH];
extern char far MainPalette[256*3];
extern char KeyTable[256];
extern int KeyPressed;

/* XLib */

extern WORD HiddenPageOffs;
extern long VsyncIntTicks;



/* FUNCTION DECLARATIONS */

int TileCollision(int x1, int y1, int x2, int y2);
int TileUnseen(int x, int y);
void FindUnseenTile(int * xp, int * yp);
void SetupCritters();
void SetupTokens();

void DrawMap(int plyr);
void BufferToScreen(int plyr);
void WipeBuffer(int plyr);
void DrawCritters(int plyr);

void UpdateControls(int plyr);
void CritterMotion(int plyr);
void MoveCritters();
void SingleStepCritter(int plyr);

void LaunchZaps(int plyr);
void ZapHitsSquidgy(int plyr);
void MoveZaps();
void ZapsHitWalls();
void DrawZaps();
void HeatSeekZaps();

void SetBlast(int x, int y, int type);
void DrawBlasts();

void MoveTokens();
void DrawTokens();
void CollectTokens(int plyr);
void PowersFade(int plyr);

void GenerateShields(int plyr);
void DrawShields();
void ShieldHitsSquidgy(int plyr);
void ShieldsHitShields();
void ShieldsHitZaps();

void PlayGame();
int GameLoop();

void UndrawSprite(int x, int y, int width, int height, unsigned offset);
void DisplayLives();
void DisplayGuns();
void DisplayHits();
void DisplayBars();



/* GLOBAL VARIABLES */

PlayerStruct Player[2];  /* Player details; 0=Player 1, 1=Player 2 */
ZapStruct Zap[TOTAL_ZAPS];  /* Details of fire/plasma balls in flight */
BlastStruct Blast[TOTAL_BLASTS];  /* Details of explosions */
TokenStruct Token[TOTAL_TOKENS];  /* Details of tokens in game */
ShieldStruct Shield[2*SHIELDS_PER_PLAYER];  /* Positions of shield units */

int GunTokenType;  /* Positive means fire balls; counts down until next time */
                   /*  plasma balls appear - 0 means plasma balls are here   */
                   /* Negative means laser bolts                             */
int GameType;  /* 0=fire ball game, 1=laser game */



/* THE CODE */

/* Function running the game routines                                 */
/* Initialises player states, counts lives, and keeps the screen tidy */
/* GameLoop() is called to do all the important stuff                 */

void PlayGame() {
    int exit_code;
    int i;
    int game_order;
    long t;

    Player[0].Lives = SQUIDGY_LIVES;
    Player[1].Lives = SQUIDGY_LIVES;

  /* Random order of GameTypes - each bit is the GameType for one round */
  /* Three rounds are accounted for, at least two are type 1            */
    game_order = random(4) + 3;
    if ( game_order == 4 ) game_order = 7;

  /* Reset sound effects */
    PrepareSounds();

    do {

      /* Choose game type for this level */
        if ( ( Player[0].Lives == 3 && Player[1].Lives == 3 )
             || ( Player[0].Lives == 1 && Player[1].Lives == 1 ) ) {
            GameType = 0;
        } else {
            GameType = game_order & 1;
            game_order = game_order >> 1;
        }

      /* Random choice of map */
        if ( random(2) == 0 ) SwitchMaps();
        ExpandRawTiles();

      /* Starting positions */
        SetupCritters();
        SetupTokens();

      /* Reset the display area (triple buffered video, so have to do */
      /* this three times - once for each screen buffer)              */
        WipeBuffer(0);
        WipeBuffer(1);
        for ( i = 0 ; i < 3 ; i++ ) {
          /* Clear viewing windows */
            BufferToScreen(0);
            BufferToScreen(1);
          /* Erase gun icons */
            DisplayGuns();
          /* Erase hit counters */
            UndrawSprite(88, 219, 10*SQUIDGY_HITS, 9, HiddenPageOffs);
            UndrawSprite(216, 219, 10*SQUIDGY_HITS, 9, HiddenPageOffs);
          /* Display changes and go on to next buffer */
            x_page_flip(0,0);
          /* Clear sound effects queue */
            SoundEffects();
        }

      /* Play one round of the game */
        exit_code = GameLoop();

      /* Reset sound effects */
        PrepareSounds();

      /* If game ended because abort was pressed, give up */
        if ( exit_code == RET_ESCAPE ) return;

      /* Freeze frame for a couple of seconds */
        t = VsyncIntTicks;
        while ( VsyncIntTicks < t+DEATH_HOLD ) if ( AbortPressed() ) return;

    } while ( Player[0].Lives && Player[1].Lives );

  /* Game over - freeze frame for a couple more seconds */
    t = VsyncIntTicks;
    while ( VsyncIntTicks < t+DEATH_HOLD ) if ( AbortPressed() ) return;

}



/* Play one round of the game - until a player dies or abort is pressed */

int GameLoop() {
    int update_lives = 3;  /* Because of triple buffering, changes to the */
    int update_gun = 3;    /* display must be made three times over       */
    int gun1, gun2;  /* Check if the 'gun-held' icon needs to be changed */

    while ( Player[0].Dead < DEATH_DELAY && Player[1].Dead < DEATH_DELAY ) {
        /* Game runs for several frames after one of the players has died */

        if ( AbortPressed() ) return RET_ESCAPE;

        gun1 = (Player[0].Gun!=0) + 2*Player[0].GunType;
        gun2 = (Player[1].Gun!=0) + 2*Player[1].GunType;

        PowersFade(0);  /* Time runs down on gun/tokens held */
        PowersFade(1);
        MoveTokens();  /* Tokens reappear or move around */
        HeatSeekZaps();  /* Plasma balls change direction seeking target */
        UpdateControls(0);  /* Read keyboard/joystick */
        UpdateControls(1);
        CritterMotion(0);  /* Players choose directions and request moves */
        CritterMotion(1);

        MoveCritters();  /* Players are moved */
        GenerateShields(0);  /* Position defensive shields */
        GenerateShields(1);
        ShieldsHitZaps();  /* Shields get between squidgies and missiles */
        ZapHitsSquidgy(0);  /* Check if fireballs hit players */
        ZapHitsSquidgy(1);
        MoveZaps();  /* Move fire balls */
        LaunchZaps(0);  /* Fire fire balls */
        LaunchZaps(1);
        ShieldsHitZaps();  /* Shields get between squidgies and missiles */
        ZapHitsSquidgy(0);  /* Check again if fireballs hit players */
        ZapHitsSquidgy(1);
        ZapsHitWalls();  /* Fire balls explode against walls */
        ShieldsHitShields();  /* Shields collide */
        ShieldHitsSquidgy(0);  /* Check if player runs into shields */
        ShieldHitsSquidgy(1);
        CollectTokens(0);  /* Players run into tokens */
        CollectTokens(1);

        DrawMap(0);  /* Floor/wall tiles are drawn in viewing buffers */
        DrawMap(1);
        DrawTokens();  /* Tokens are drawn in viewing buffers */
        DrawCritters(0);  /* Squidgies are drawn in viewing buffers */
        DrawCritters(1);
        DrawZaps();  /* Fire balls are drawn in viewing buffers */
        DrawShields();  /* Shield units are drawn in viewing buffer */
        DrawBlasts();  /* Explosions are drawn in viewing buffers */
        BufferToScreen(0);  /* Finally, viewing buffers are copied to */
        BufferToScreen(1);  /* viewing windows on screen              */

      /* Players die when they run out of hits                    */
      /* The game plays on for a few frames with only one squidgy */
        if ( Player[0].Dead ) Player[0].Dead++;
        if ( Player[1].Dead ) Player[1].Dead++;
        if ( ! Player[0].Dead && Player[0].Hits <= 0 ) {
            Player[0].Lives--;
            Player[0].Dead = 1;
            update_lives = 3;
        }
        if ( ! Player[1].Dead && Player[1].Hits <= 0 ) {
            Player[1].Lives--;
            Player[1].Dead = 1;
            update_lives = 3;
        }

      /* If a player has got a gun, lost a gun, or changed a gun */
      /* the gun icon in the player's display must be updated    */
        if ( gun1 != (Player[0].Gun!=0) + 2*Player[0].GunType
              || gun2 != (Player[1].Gun!=0) + 2*Player[1].GunType ) {
            update_gun = 3;
        }

      /* Update hit counters, gun-time-remaining bars, */
      /* life counters and gun-held icons              */
        DisplayHits();
        DisplayBars();
        if ( update_lives ) {
            DisplayLives();
            update_lives--;
        }
        if ( update_gun ) {
            DisplayGuns();
            update_gun--;
        }

      /* Display this frame */
        x_page_flip(0, 0);

      /* Update sound effects */
        if ( Player[0].Healing || Player[1].Healing ) MakeSound(WHIRR);
        else MakeSound(NO_WHIRR);
        SoundEffects();
    }
    return RET_GAMEOVER;
}



/* Returns true if tile at position (x,y) in map is not within the */
/*  viewing window of either of the players                        */

int TileUnseen(int x, int y) {
    int i;
    int dx, dy;

    for ( i = 0 ; i < 2 ; i++ ) {
        dx = x*TILE_WIDTH - Player[i].XPos;
        dy = y*TILE_WIDTH - Player[i].YPos;
        if ( dx > MAP_WIDTH*TILE_WIDTH/2 ) dx -= MAP_WIDTH*TILE_WIDTH;
        if ( dx < -MAP_WIDTH*TILE_WIDTH/2 ) dx += MAP_WIDTH*TILE_WIDTH;
        if ( dy > MAP_HEIGHT*TILE_WIDTH/2 ) dy -= MAP_HEIGHT*TILE_WIDTH;
        if ( dy < -MAP_HEIGHT*TILE_WIDTH/2 ) dy += MAP_HEIGHT*TILE_WIDTH;
        if ( dx < 5*TILE_WIDTH && dx >= - 5*TILE_WIDTH
             && dy < 5*TILE_WIDTH && dy >= - 5*TILE_WIDTH ) return 0;
    }
    return 1;
}



/* Gives x and y position of a random floor (not wall) tile in the map */
/*  which is not visible on either players viewing window              */
/* Tile chosen is not already occupied by a token                      */

void FindUnseenTile(int * xp, int * yp) {
    int done;
    int x, y;
    int i;

    do {
        done = 0;
        x = random(MAP_WIDTH);
        y = random(MAP_HEIGHT);
        if ( TileType(x,y) == FLOOR && TileUnseen(x, y) ) done = 1;
        for ( i = 0 ; i < TOTAL_TOKENS ; i++ ) {
            if ( Token[i].Type == UNUSED_TOKEN ) continue;
            if ( x == Token[i].XTile && y == Token[i].YTile ) done = 0;
        }
    } while ( ! done );

    *xp = x;
    *yp = y;
}



/* Checks whether two tile-size sprites at positions (x1,y1) */
/*  and (x2,y2) overlap                                      */

int TileCollision(int x1, int y1, int x2, int y2) {
    int dx = x1 - x2;
    int dy = y1 - y2;
    if ( dx > MAP_WIDTH*TILE_WIDTH/2 ) dx -= MAP_WIDTH*TILE_WIDTH;
    if ( dx < -MAP_WIDTH*TILE_WIDTH/2 ) dx += MAP_WIDTH*TILE_WIDTH;
    if ( dy > MAP_HEIGHT*TILE_WIDTH/2 ) dy -= MAP_HEIGHT*TILE_WIDTH;
    if ( dy < -MAP_HEIGHT*TILE_WIDTH/2 ) dy += MAP_HEIGHT*TILE_WIDTH;
    if ( dy > -TILE_WIDTH && dy < TILE_WIDTH
        && dx > -TILE_WIDTH && dx < TILE_WIDTH ) return 1;
    return 0;
}



/* Initialises players, fire balls, explosions, etc */

void SetupCritters() {
    int i, j;
    int k, l;

  /* Random starting positions, player 2 well away from player 1 */

    do {
        i = random(MAP_WIDTH);
        j = random(MAP_HEIGHT);
    } while ( TileType(i, j) == WALL );

    do {
        k = ModWidth(i+MAP_WIDTH/4+random(MAP_WIDTH/2));
        l = ModHeight(j+MAP_HEIGHT/4+random(MAP_HEIGHT/2));
    } while ( TileType(k, l) == WALL );

    Player[0].XPos = i*TILE_WIDTH;
    Player[0].YPos = j*TILE_WIDTH;
    Player[1].XPos = k*TILE_WIDTH;
    Player[1].YPos = l*TILE_WIDTH;

  /* Reset Player structures */

    Player[0].Speed = 4;
    Player[1].Speed = 4;

    Player[0].Direc = 2*random(4);
    Player[1].Direc = 2*random(4);

    Player[0].Frame = 2;
    Player[1].Frame = 2;

    Player[0].Tile = BLUE_CRITTER_TILES + Player[0].Direc + 1;
    Player[1].Tile = BLUE_CRITTER_TILES + Player[1].Direc + 1;

    Player[0].Gun = 0;
    Player[1].Gun = 0;
    Player[0].GunWait = 0;
    Player[1].GunWait = 0;

    Player[0].Shields = 0;
    Player[1].Shields = 0;
    Player[0].ShieldFrame = 0;
    Player[1].ShieldFrame = 0;

    Player[0].FastPower = 0;
    Player[1].FastPower = 0;
    Player[0].SlowPower = 0;
    Player[1].SlowPower = 0;
    Player[0].VanishPower = 0;
    Player[1].VanishPower = 0;
    Player[0].Healing = 0;
    Player[1].Healing = 0;

    Player[0].Hits = SQUIDGY_HITS;
    Player[1].Hits = SQUIDGY_HITS;

    Player[0].Dead = 0;
    Player[1].Dead = 0;

  /* Reset fire balls and explosions */

    for ( i = 0 ; i < TOTAL_ZAPS ; i++ ) Zap[i].Active = 0;
    for ( i = 0 ; i < TOTAL_BLASTS ; i++ ) Blast[i].Active = 0;
}



/* Choose random starting places for the tokens */
/*  and initialise the Token structures         */

void SetupTokens() {
    int x, y;
    int i;

    for ( i = 0 ; i < TOTAL_TOKENS ; i++ ) Token[i].Type = UNUSED_TOKEN;

    if ( GameType == 0 ) {

      /* Standard game - fire balls et al */
        Token[0].Type = GUN_TOKEN;
        Token[1].Type = FAST_TOKEN;
        Token[2].Type = SLOW_TOKEN;
        Token[3].Type = VANISH_TOKEN;
        Token[4].Type = HEALTH_TOKEN;
        Token[5].Type = SLOW_TOKEN;
        GunTokenType = HEATSEEK_REAPPEAR0 + random(HEATSEEK_REAPPEAR2);

    } else if ( GameType == 1 ) {

      /* Alternative game - lasers & shields */
        Token[0].Type = GUN_TOKEN;
        Token[1].Type = FAST_TOKEN;
        Token[2].Type = SHIELD_TOKEN;
        Token[3].Type = SHIELD_TOKEN;
        Token[4].Type = SLOW_TOKEN;
        Token[5].Type = SLOW_TOKEN;
        GunTokenType = -1;

    }

    for ( i = 0 ; i < TOTAL_TOKENS ; i++ ) {
        switch ( Token[i].Type ) {

          case UNUSED_TOKEN:
            break;

          case GUN_TOKEN:
            Token[i].Active = 1;
            Token[i].Delay = GUN_MOVE;
            FindUnseenTile(&x, &y);
            Token[i].XTile = x;
            Token[i].YTile = y;
            Token[i].Tile = GUN_TOKEN_TILES;
            Token[i].Frame = 0;
            break;

          case FAST_TOKEN:
            Token[i].Active = 1;
            Token[i].Delay = TOKEN_MOVE;
            FindUnseenTile(&x, &y);
            Token[i].XTile = x;
            Token[i].YTile = y;
            Token[i].Tile = FAST_TOKEN_TILE;
            break;

          case SLOW_TOKEN:
            Token[i].Active = 1;
            Token[i].Delay = TOKEN_MOVE;
            FindUnseenTile(&x, &y);
            Token[i].XTile = x;
            Token[i].YTile = y;
            Token[i].Tile = SLOW_TOKEN_TILE;
            break;

          case VANISH_TOKEN:
            Token[i].Active = 1;
            Token[i].Delay = TOKEN_MOVE;
            FindUnseenTile(&x, &y);
            Token[i].XTile = x;
            Token[i].YTile = y;
            Token[i].Tile = VANISH_TOKEN_TILE;
            break;

          case HEALTH_TOKEN:
            Token[i].Active = 0;
            Token[i].Delay = HEALTH_REAPPEAR/5;
            FindUnseenTile(&x, &y);
            Token[i].XTile = x;
            Token[i].YTile = y;
            Token[i].Tile = HEALTH_TOKEN_TILE;
            break;

          case SHIELD_TOKEN:
            Token[i].Active = 1;
            Token[i].Delay = TOKEN_MOVE;
            FindUnseenTile(&x, &y);
            Token[i].XTile = x;
            Token[i].YTile = y;
            Token[i].Tile = SHIELD_TOKEN_TILE;
            break;

        }
    }

}



/* Fill in viewing buffer for player <plyr>                  */
/* Wall and floor tiles are drawn according to map locations */
/* (plyr=0 : Player 1, plyr=1 : Player 2)                    */
/* (Calls assembler routine)                                 */

void DrawMap(int plyr) {
    MapToViewBuffer(ModWidth(Player[plyr].XPos / TILE_WIDTH - 4),
                    ModHeight(Player[plyr].YPos / TILE_WIDTH - 4),
                    Player[plyr].ViewBuffer);
}



/* Copy the relevant area of the viewing buffer to player <plyr>'s */
/*  viewing window on the screen                                   */
/* (plyr=0 : Player 1, plyr=1 : Player 2)                          */

void BufferToScreen(int plyr) {
    char far * addr;
    unsigned window;
    addr = Player[plyr].ViewBuffer;
    addr += VIEW_BUFF_WIDTH * (Player[plyr].YPos % TILE_WIDTH);
    addr += Player[plyr].XPos % TILE_WIDTH;
    if ( plyr == 0 ) window = SCR_ADDR_VIEW_1; else window = SCR_ADDR_VIEW_2;
    ViewBufferToScr(addr,(unsigned) HiddenPageOffs + window);
}



/* Blank player <plyr>'s viewing buffer   */
/* (plyr=0 : Player 1, plyr=1 : Player 2) */

void WipeBuffer(int plyr) {
    int i;
    char far * p = Player[plyr].ViewBuffer;
    for ( i = 0 ; i < VIEW_BUFF_SIZE ; i++ ) *(p++) = 0;
}



/* Draw both players' squidgies in player <plyr>'s viewing buffer */
/* Takes account of invisibility power of squidgies               */
/* (plyr=0 : Player 1, plyr=1 : Player 2)                         */

void DrawCritters(int plyr) {
    char far * p;
    int i;
    int dx, dy;

    p = Player[plyr].ViewBuffer
                + VIEW_BUFF_WIDTH*(Player[plyr].YPos % TILE_WIDTH)
                + (Player[plyr].XPos % TILE_WIDTH);

    i = plyr ^ 1;
    dx = Player[i].XPos - Player[plyr].XPos;
    if ( dx > MAP_WIDTH*TILE_WIDTH/2 ) dx -= MAP_WIDTH*TILE_WIDTH;
    if ( dx < -MAP_WIDTH*TILE_WIDTH/2 ) dx += MAP_WIDTH*TILE_WIDTH;
    dy = Player[i].YPos - Player[plyr].YPos;
    if ( dy > MAP_HEIGHT*TILE_WIDTH/2 ) dy -= MAP_HEIGHT*TILE_WIDTH;
    if ( dy < -MAP_HEIGHT*TILE_WIDTH/2 ) dy += MAP_HEIGHT*TILE_WIDTH;

    if ( ! Player[i].VanishPower || TileTypeNum(Player[i].Tile) == SKELEBONE ) {
        DrawObject(p, dx + 4*TILE_WIDTH, dy + 4*TILE_WIDTH, Player[i].Tile);
    }

    if ( ! Player[plyr].VanishPower
            || TileTypeNum(Player[plyr].Tile) == SKELEBONE ) {
        DrawObject(p, 4*TILE_WIDTH, 4*TILE_WIDTH, Player[plyr].Tile);
    }
}



/* Set Up/Down/Left/Right/Fire flags of player <plyr> */
/*  according to what controls are being operated     */
/* (plyr=0 : Player 1, plyr=1 : Player 2)             */

void UpdateControls(int plyr) {
    int x, y, b1, b2;

    switch (Player[plyr].ControlDevice) {

      case KEYBOARD:
        Player[plyr].Up = KeyTable[Player[plyr].KeyUp];
        Player[plyr].Down = KeyTable[Player[plyr].KeyDown];
        Player[plyr].Left = KeyTable[Player[plyr].KeyLeft];
        Player[plyr].Right = KeyTable[Player[plyr].KeyRight];
        Player[plyr].Fire = KeyTable[Player[plyr].KeyFire];
        break;

      case JOYSTICK1:
        ReadJoystick(1, &x, &y, &b1, &b2);
        Player[plyr].Right = ( x > Player[plyr].JoyXUpper );
        Player[plyr].Left = ( x < Player[plyr].JoyXLower );
        Player[plyr].Down = ( y > Player[plyr].JoyYUpper );
        Player[plyr].Up = ( y < Player[plyr].JoyYLower );
        Player[plyr].Fire = ( (b1+b2) != 0 );
        break;

      case JOYSTICK2:
        ReadJoystick(2, &x, &y, &b1, &b2);
        Player[plyr].Right = ( x > Player[plyr].JoyXUpper );
        Player[plyr].Left = ( x < Player[plyr].JoyXLower );
        Player[plyr].Down = ( y > Player[plyr].JoyYUpper );
        Player[plyr].Up = ( y < Player[plyr].JoyYLower );
        Player[plyr].Fire = ( (b1+b2) != 0 );
        break;

      default:
        break;
    }

  /* Laser gun is auto-fire */
    if ( GameType == 1) Player[plyr].Fire = 1;
}



/* Update player <plyr>'s direction (Direc), animation frame (frame),      */
/*  sprite number (Tile), and requested movement this frame (XMove, YMove) */
/* (plyr=0 : Player 1, plyr=1 : Player 2)                                  */

void CritterMotion(int plyr) {
    int direc;
    int dir_table[9] = { 7, 0, 1,
                         6,-1, 2,
                         5, 4, 3 };
    int d;

    if ( Player[plyr].Dead ) return;

    Player[plyr].XMove = Player[plyr].Right - Player[plyr].Left;
    Player[plyr].YMove = Player[plyr].Down - Player[plyr].Up;

    direc = 3*(Player[plyr].YMove+1) + (Player[plyr].XMove+1);
    direc = dir_table[direc];

    Player[plyr].XMove *= Player[plyr].Speed;
    Player[plyr].YMove *= Player[plyr].Speed;

    if ( direc != -1 ) Player[plyr].Frame = (Player[plyr].Frame + 1) % 3;
    else Player[plyr].Frame = 2;

    if ( direc != -1 ) Player[plyr].Direc = direc;

    Player[plyr].Tile = BLUE_CRITTER_TILES*(plyr==0)
                        + RED_CRITTER_TILES*(plyr==1)
                        + (EVIL_CRITTER_TILES - BLUE_CRITTER_TILES)
                            *(Player[plyr].Gun > 0)
                        + Player[plyr].Direc*2 + Player[plyr].Frame/2;
}



/* Move both squidgies by the amounts given in (XMove, YMove), but */
/*  don't let them walk through each other or through walls        */
/* Squidgies are moved and collision-checked one pixel at a time   */

void MoveCritters() {
    int i;
    int m0, m1;

  /* Player who's moving furthest gets priority */
    m0 = abs(Player[0].XMove);
    if ( m0 < abs(Player[0].YMove) ) m0 = abs(Player[0].YMove);
    m1 = abs(Player[1].XMove);
    if ( m1 < abs(Player[1].YMove) ) m1 = abs(Player[1].YMove);

  /* Player 1 moves further, so gets first moves */
    while ( m0 > m1 ) {
        SingleStepCritter(0);
        m0--;
    }

  /* Player 2 moves further, so gets first moves */
    while ( m1 > m0 ) {
        SingleStepCritter(1);
        m1--;
    }

  /* Players both have the same distance to go so take it in turns */
    for ( i = 0 ; i < m0 ; i++ ) {
        SingleStepCritter(0);
        SingleStepCritter(1);
    }
}



/* Horrible piece of code that moves a player by at most one pixel step, */
/*  reducing the step counters Player.XMove and Player.YMove             */
/* Players cannot move into each other or into walls                     */
/* Collisions with walls are only checked when the player first crosses  */
/*  a tile grid line                                                     */
/* Player's position wraps around the edges of the map                   */
/* (plyr=0 : Player 1, plyr=1 : Player 2)                                */

void SingleStepCritter(int plyr) {
    int x, y;
    int xh, xl, yh, yl;
    int other;

    other = plyr^1;
    x = Player[plyr].XPos;
    y = Player[plyr].YPos;

  /* (x, y) is the player's position in pixels                 */
  /* (xh, yh) is the player's position in tiles                */
  /* (xl, yl) is the player's position in pixels within a tile */
    xh = x / TILE_WIDTH;
    xl = x % TILE_WIDTH;
    yh = y / TILE_WIDTH;
    yl = y % TILE_WIDTH;

  /* Player wants to move one step to the right */
    if ( Player[plyr].XMove > 0 ) {
        if ( xl == 0 ) {
            if ( TileType(ModWidth(xh+1), yh) == FLOOR
                && ( yl == 0 || TileType(ModWidth(xh+1), ModHeight(yh+1))
                                     == FLOOR ) ) {
                if ( ! TileCollision(x+1, y, Player[other].XPos,
                                             Player[other].YPos) ) {
                    xl++;
                    x++;
                }
            }
        } else {
            if ( ++x == MAP_WIDTH*TILE_WIDTH ) x = 0;
            if ( ! TileCollision(x, y, Player[other].XPos,
                                       Player[other].YPos) ) {
                if ( ++xl == TILE_WIDTH ) {
                    xl = 0;
                    if ( ++xh == MAP_WIDTH ) xh = 0;
                }
            } else if ( --x == -1 ) x = MAP_WIDTH*TILE_WIDTH - 1;
        }
        Player[plyr].XMove--;
    }

  /* Player wants to move one step to the left */
    if ( Player[plyr].XMove < 0 ) {
        if ( xl == 0 ) {
            if ( TileType(ModWidth(xh-1), yh) == FLOOR
                 && ( yl == 0 || TileType(ModWidth(xh-1), ModHeight(yh+1))
                                                == FLOOR ) ) {
                if ( --x == -1 ) x = MAP_WIDTH*TILE_WIDTH - 1;
                if ( ! TileCollision(x, y, Player[other].XPos,
                                           Player[other].YPos) ) {
                    xl = TILE_WIDTH - 1;
                    if ( --xh == -1 ) xh = MAP_WIDTH - 1;
                } else if ( ++x == MAP_WIDTH*TILE_WIDTH ) x = 0;
            }
        } else {
            if ( ! TileCollision(x-1, y, Player[other].XPos,
                                         Player[other].YPos) ) {
                x--;
                xl--;
            }
        }
        Player[plyr].XMove++;
    }

  /* Player wants to move one step down */
    if ( Player[plyr].YMove > 0 ) {
        if ( yl == 0 ) {
            if ( TileType(xh, ModHeight(yh+1)) == FLOOR
                && ( xl == 0 || TileType(ModWidth(xh+1), ModHeight(yh+1))
                                     == FLOOR ) ) {
                if ( ! TileCollision(x, y+1, Player[other].XPos,
                                             Player[other].YPos) ) {
                    yl++;
                    y++;
                }
            }
        } else {
            if ( ++y == MAP_HEIGHT*TILE_WIDTH ) y = 0;
            if ( ! TileCollision(x, y, Player[other].XPos,
                                       Player[other].YPos) ) {
                if ( ++yl == TILE_WIDTH ) {
                    yl = 0;
                    if ( ++yh == MAP_HEIGHT ) {
                        yh = 0;
                    }
                }
            } else if ( --y == -1 ) y = MAP_HEIGHT*TILE_WIDTH - 1;
        }
        Player[plyr].YMove--;
    }

  /* Player wants to move one step up */
    if ( Player[plyr].YMove < 0 ) {
        if ( yl == 0 ) {
            if ( TileType(xh, ModHeight(yh-1)) == FLOOR
                 && ( xl == 0 || TileType(ModWidth(xh+1), ModHeight(yh-1))
                                                == FLOOR ) ) {
                if ( --y == -1 ) y = MAP_HEIGHT*TILE_WIDTH - 1;
                if ( ! TileCollision(x, y, Player[other].XPos,
                                           Player[other].YPos) ) {
                    yl = TILE_WIDTH - 1;
                    if ( --yh == -1 ) yh = MAP_HEIGHT - 1;
                } else if ( ++y == MAP_HEIGHT*TILE_WIDTH ) y = 0;
            }
        } else {
            if ( ! TileCollision(x, y-1, Player[other].XPos,
                                         Player[other].YPos) ) {
                y--;
                yl--;
            }
        }
        Player[plyr].YMove++;
    }

  /* Update position */
    Player[plyr].XPos = x;
    Player[plyr].YPos = y;
}



/* If player <plyr> has a loaded gun and is pressing fire, a fire ball */
/*  is launched just in front of the squidgy's nose                    */
/* The DontKillOwner flag ensures that the player firing the missile   */
/*  isn't immediately registered as having collided with a fire ball   */
/* The time-till-next-firing counter is also updated                   */
/* (plyr=0 : Player 1, plyr=1 : Player 2)                              */

void LaunchZaps(int plyr) {
    int i;
    int x, y;
    int zap_start[8][2] = { {0,-11}, {+8,-8}, {+11,0}, {+8,+8},
                            {0,+11}, {-8,+8}, {-11,0}, {-8,-8} };
                          /* Starting position of launched zap */

    if ( Player[plyr].Dead ) return;

    if ( Player[plyr].GunWait > 0 ) Player[plyr].GunWait--;

    if ( Player[plyr].Gun && Player[plyr].Fire && ! Player[plyr].GunWait ) {
        for ( i = 0 ; i < TOTAL_ZAPS && Zap[i].Active ; i++ );
        if ( i != TOTAL_ZAPS ) {

            Zap[i].Active = 1;

            if ( Player[plyr].GunType == LASER ) {
                Player[plyr].GunWait = ZAP2_DELAY;
            } else {
                Player[plyr].GunWait = ZAP_DELAY;
            }

            x = Player[plyr].XPos + zap_start[Player[plyr].Direc][0]
                + (TILE_WIDTH-ZAP_WIDTH)/2;
            y = Player[plyr].YPos + zap_start[Player[plyr].Direc][1]
                + (TILE_WIDTH-ZAP_WIDTH)/2;
            if ( x >= MAP_WIDTH*TILE_WIDTH ) x -= MAP_WIDTH*TILE_WIDTH;
            if ( y >= MAP_HEIGHT*TILE_WIDTH ) y -= MAP_HEIGHT*TILE_WIDTH;
            if ( x < 0 ) x += MAP_WIDTH*TILE_WIDTH;
            if ( y < 0 ) y += MAP_HEIGHT*TILE_WIDTH;
            Zap[i].XPos = x;
            Zap[i].YPos = y;

            Zap[i].Direc = 2*Player[plyr].Direc;
            Zap[i].Owner = plyr;
            Zap[i].DontKillOwner = 4;
            Zap[i].Type = Player[plyr].GunType;

            if ( Player[plyr].GunType == LASER ) MakeSound(FIZZ);
            else MakeSound(THUMP);

        }
    }
}



/* Check to see if player <plyr> is hit by a fire ball       */
/* If so an explosion is triggered and the player is injured */
/* (plyr=0 : Player 1, plyr=1 : Player 2)                    */

void ZapHitsSquidgy(int plyr) {
    int i;
    int x, y;
    int dx, dy;
    int t;

    for ( i = 0 ; i < TOTAL_ZAPS ; i++ ) {
        x = Player[plyr].XPos;
        y = Player[plyr].YPos;

        if ( Zap[i].Active && ( Zap[i].DontKillOwner == 0
                              || Zap[i].Owner != plyr ) ) {
            dx = x - Zap[i].XPos;
            dy = y - Zap[i].YPos;
            if ( dx > MAP_WIDTH*TILE_WIDTH/2 ) dx -= MAP_WIDTH*TILE_WIDTH;
            if ( dx < -MAP_WIDTH*TILE_WIDTH/2 ) dx += MAP_WIDTH*TILE_WIDTH;
            if ( dy > MAP_HEIGHT*TILE_WIDTH/2 ) dy -= MAP_HEIGHT*TILE_WIDTH;
            if ( dy < -MAP_HEIGHT*TILE_WIDTH/2 ) dy += MAP_HEIGHT*TILE_WIDTH;

            if ( dy > -TILE_WIDTH && dy < ZAP_WIDTH
                        && dx > -TILE_WIDTH && dx < ZAP_WIDTH ) {
                SetBlast(Zap[i].XPos - (TILE_WIDTH-ZAP_WIDTH)/2,
                         Zap[i].YPos - (TILE_WIDTH-ZAP_WIDTH)/2,
                         Zap[i].Type);
                Zap[i].Active = 0;
                if ( Player[plyr].Hits-- > 0 ) MakeSound(SQUEAK);
                else Player[plyr].Hits = 0;
                t = Player[plyr].Tile;
                if ( TileTypeNum(t) == CRITTER ) {
                    t = ( ( t - BLUE_CRITTER_TILES ) % 16 ) / 2;
                    t += SKELEBONE_TILES;
                    Player[plyr].Tile = t;
                }
                if ( TileTypeNum(t) == EVIL ) {
                    t = ( ( t - EVIL_CRITTER_TILES ) % 16 ) / 2;
                    t += SKELEBONE_TILES;
                    Player[plyr].Tile = t;
                }

            }
        }

    }
}



/* All fire balls are moved in the direction they are pointing        */
/* Fire balls can move in 16 directions (0=up, 4=right, 8=down, etc)  */
/* (DontKillOwner flags are updated here)                             */

void MoveZaps() {
    int i;
    int x, y;

    int zap_vels[16][2]
        = { {0,-ZAP_SPEED},                  {+3,-(ZAP_SPEED-1)},
            {+(ZAP_SPEED-1),-(ZAP_SPEED-1)}, {+(ZAP_SPEED-1),-3},
            {+ZAP_SPEED,0},                  {+(ZAP_SPEED-1),+3},
            {+(ZAP_SPEED-1),+(ZAP_SPEED-1)}, {+3,+(ZAP_SPEED-1)},
            {0,+ZAP_SPEED},                  {-3,+(ZAP_SPEED-1)},
            {-(ZAP_SPEED-1),+(ZAP_SPEED-1)}, {-(ZAP_SPEED-1),+3},
            {-ZAP_SPEED,0},                  {-(ZAP_SPEED-1),-3},
            {-(ZAP_SPEED-1),-(ZAP_SPEED-1)}, {-3,-(ZAP_SPEED-1)} };

    int laser_vels[16][2]
        = { {0,-ZAP2_SPEED},                   {+3,-(ZAP2_SPEED-2)},
            {+(ZAP2_SPEED-2),-(ZAP2_SPEED-2)}, {+(ZAP2_SPEED-2),-3},
            {+ZAP2_SPEED,0},                   {+(ZAP2_SPEED-2),+3},
            {+(ZAP2_SPEED-2),+(ZAP2_SPEED-2)}, {+3,+(ZAP2_SPEED-2)},
            {0,+ZAP2_SPEED},                   {-3,+(ZAP2_SPEED-2)},
            {-(ZAP2_SPEED-2),+(ZAP2_SPEED-2)}, {-(ZAP2_SPEED-2),+3},
            {-ZAP2_SPEED,0},                   {-(ZAP2_SPEED-2),-3},
            {-(ZAP2_SPEED-2),-(ZAP2_SPEED-2)}, {-3,-(ZAP2_SPEED-2)} };


    for ( i = 0 ; i < TOTAL_ZAPS ; i++ ) {
        if ( Zap[i].Active ) {
            if ( Zap[i].Type == LASER ) {
                x = Zap[i].XPos + laser_vels[Zap[i].Direc][0];
                y = Zap[i].YPos + laser_vels[Zap[i].Direc][1];
            } else {
                x = Zap[i].XPos + zap_vels[Zap[i].Direc][0];
                y = Zap[i].YPos + zap_vels[Zap[i].Direc][1];
            }

            if ( x >= MAP_WIDTH*TILE_WIDTH ) x -= MAP_WIDTH*TILE_WIDTH;
            if ( y >= MAP_HEIGHT*TILE_WIDTH ) y -= MAP_HEIGHT*TILE_WIDTH;
            if ( x < 0 ) x += MAP_WIDTH*TILE_WIDTH;
            if ( y < 0 ) y += MAP_HEIGHT*TILE_WIDTH;
            Zap[i].XPos = x;
            Zap[i].YPos = y;

            if ( Zap[i].DontKillOwner > 0 ) Zap[i].DontKillOwner--;
        }
    }
}



/* Plasma balls can lock-onto targets and change direction to pursue them */
/* A lock-on requires that the target be in range and (roughly) in front  */
/*  of the plasma ball                                                    */
/* The plasma ball can change direction by at most one click per round    */

void HeatSeekZaps() {
    int i;
    int targ;  /* Player heat-seeker is looking for */
    int dx, dy;  /* Relative position of target player */
    int ang;  /* Angular position (0-15) of target player */
    int temp;

    for ( i = 0 ; i < TOTAL_ZAPS ; i++ ) {
        if ( ! Zap[i].Active || ( Zap[i].Type != PLASMA
                               && Zap[i].Type != PLASMA_LOCKED ) ) continue;

        targ = Zap[i].Owner ^ 1;
        dx = Player[targ].XPos + TILE_WIDTH/2 - Zap[i].XPos - ZAP_WIDTH/2;
        dy = Player[targ].YPos + TILE_WIDTH/2 - Zap[i].YPos - ZAP_WIDTH/2;
        if ( dx > MAP_WIDTH*TILE_WIDTH/2 ) dx -= MAP_WIDTH*TILE_WIDTH;
        if ( dx < -MAP_WIDTH*TILE_WIDTH/2 ) dx += MAP_WIDTH*TILE_WIDTH;
        if ( dy > MAP_HEIGHT*TILE_WIDTH/2 ) dy -= MAP_HEIGHT*TILE_WIDTH;
        if ( dy < -MAP_HEIGHT*TILE_WIDTH/2 ) dy += MAP_HEIGHT*TILE_WIDTH;
        if ( dx == 0 && dy == 0 ) continue;

      /* Check target is in range */
        if ( abs(dx) < HEATSEEK_RANGE && abs(dy) < HEATSEEK_RANGE ) {

          /* Quadrant of target: 0-4=top-right, 4-8=bottom-right  */
          /*                     8-12=bottom-left, 12-16=top-left */
            ang = 0;
            if ( dx < 0 ) {
                dx = -dx;
                dy = -dy;
                ang += 8;
            }
            if ( dy >= 0 ) {
                temp = dy;
                dy = -dx;
                dx = temp;
                ang += 4;
            }

          /* Position of target within quadrant, based on top-right quadrant */
          /* Approximate inverse tan calculation is used to find angle       */
            if ( dy == 0 ) temp = 1000;
            else temp = (10*dx)/(-dy);
            if ( temp < 2 ) ang += 0;
            else if ( temp < 7 ) ang += 1;
            else if ( temp < 15 ) ang += 2;
            else if ( temp <= 50 ) ang += 3;
            else ang += 4;
            ang = ang & 15;

          /* Direction (-7 to +7) relative to direction of plasma ball */
            ang = (ang - Zap[i].Direc) & 15;
            if ( ang >= 8 ) ang -= 16;

            if ( Zap[i].Type == PLASMA ) {
                if ( abs(ang) < 3 ) Zap[i].Type = PLASMA_LOCKED;
            } else {
                if ( ang > 0 ) Zap[i].Direc = ( Zap[i].Direc + 1 ) & 15;
                if ( ang < 0 ) Zap[i].Direc = ( Zap[i].Direc - 1 ) & 15;
            }

        }

    }
}



/* Set explosions where any fire balls collide with walls */

void ZapsHitWalls() {
    int i;
    int xh, xl, yh, yl;

    for ( i = 0 ; i < TOTAL_ZAPS ; i++ ) {
        if ( Zap[i].Active ) {
            xh = Zap[i].XPos / TILE_WIDTH;
            xl = Zap[i].XPos % TILE_WIDTH;
            yh = Zap[i].YPos / TILE_WIDTH;
            yl = Zap[i].YPos % TILE_WIDTH;

            if ( TileType(xh, yh) == WALL ) {
                SetBlast(Zap[i].XPos - (TILE_WIDTH-ZAP_WIDTH)/2,
                         Zap[i].YPos - (TILE_WIDTH-ZAP_WIDTH)/2,
                         Zap[i].Type);
                Zap[i].Active = 0;
            }
            if ( xl > TILE_WIDTH-ZAP_WIDTH
                      && TileType(ModWidth(xh+1), yh) == WALL ) {
                SetBlast(Zap[i].XPos - (TILE_WIDTH-ZAP_WIDTH)/2,
                         Zap[i].YPos - (TILE_WIDTH-ZAP_WIDTH)/2,
                         Zap[i].Type);
                Zap[i].Active = 0;
            }
            if ( yl > TILE_WIDTH-ZAP_WIDTH
                      && TileType(xh, ModHeight(yh+1)) == WALL ) {
                SetBlast(Zap[i].XPos - (TILE_WIDTH-ZAP_WIDTH)/2,
                         Zap[i].YPos - (TILE_WIDTH-ZAP_WIDTH)/2,
                         Zap[i].Type);
                Zap[i].Active = 0;
            }
            if ( xl > TILE_WIDTH-ZAP_WIDTH && yl > TILE_WIDTH-ZAP_WIDTH
                    && TileType(ModWidth(xh+1), ModHeight(yh+1)) == WALL ) {
                SetBlast(Zap[i].XPos - (TILE_WIDTH-ZAP_WIDTH)/2,
                         Zap[i].YPos - (TILE_WIDTH-ZAP_WIDTH)/2,
                         Zap[i].Type);
                Zap[i].Active = 0;
            }
        }
    }
}



/* Draw active fire/plasma balls in both players' viewing buffers */

void DrawZaps() {
    char far * p;
    int i, j;
    int dx, dy;
    int t;

    for ( j = 0 ; j < 2 ; j++ ) {
        p = Player[j].ViewBuffer
                + VIEW_BUFF_WIDTH*(Player[j].YPos % TILE_WIDTH)
                + (Player[j].XPos % TILE_WIDTH);

        for ( i = 0 ; i < TOTAL_ZAPS ; i++ ) {
            if ( Zap[i].Active ) {
                dx = Zap[i].XPos - Player[j].XPos;
                if ( dx > MAP_WIDTH*TILE_WIDTH/2 )
                    dx -= MAP_WIDTH*TILE_WIDTH;
                if ( dx < -MAP_WIDTH*TILE_WIDTH/2 )
                    dx += MAP_WIDTH*TILE_WIDTH;
                dy = Zap[i].YPos - Player[j].YPos;
                if ( dy > MAP_HEIGHT*TILE_WIDTH/2 )
                    dy -= MAP_HEIGHT*TILE_WIDTH;
                if ( dy < -MAP_HEIGHT*TILE_WIDTH/2 )
                    dy += MAP_HEIGHT*TILE_WIDTH;
                t = ZAP_TILE
                    + (ZAP2_TILE - ZAP_TILE)*(Zap[i].Type == PLASMA_LOCKED)
                    + (LASER_TILE - ZAP_TILE + Zap[i].Direc/2)
                                       *(Zap[i].Type == LASER);
                DrawObject(p, dx + 4*TILE_WIDTH, dy + 4*TILE_WIDTH, t);
            }
        }
    }
}



/* Create an explosion at position (x, y)              */
/* <type> is the class of weapon causing the explosion */

void SetBlast(int x, int y, int type) {
    int i;

    if ( x >= MAP_WIDTH*TILE_WIDTH ) x -= MAP_WIDTH*TILE_WIDTH;
    if ( y >= MAP_HEIGHT*TILE_WIDTH ) y -= MAP_HEIGHT*TILE_WIDTH;
    if ( x < 0 ) x += MAP_WIDTH*TILE_WIDTH;
    if ( y < 0 ) y += MAP_HEIGHT*TILE_WIDTH;

    for ( i = 0 ; i < TOTAL_BLASTS && Blast[i].Active ; i++ ) ;
    if ( i == TOTAL_BLASTS ) return;
    Blast[i].Active = 1;
    Blast[i].XPos = x;
    Blast[i].YPos = y;
    Blast[i].Type = type;
}



/* Draw explosions in both players' viewing windows         */
/* Then reset the explosions - they only last for one frame */

void DrawBlasts() {
    char far * p;
    int i, j;
    int dx, dy;
    int t;

    for ( j = 0 ; j < 2 ; j++ ) {
        p = Player[j].ViewBuffer
                + VIEW_BUFF_WIDTH*(Player[j].YPos % TILE_WIDTH)
                + (Player[j].XPos % TILE_WIDTH);

        for ( i = 0 ; i < TOTAL_BLASTS ; i++ ) {
            if ( Blast[i].Active ) {
                dx = Blast[i].XPos - Player[j].XPos;
                if ( dx > MAP_WIDTH*TILE_WIDTH/2 )
                    dx -= MAP_WIDTH*TILE_WIDTH;
                if ( dx < -MAP_WIDTH*TILE_WIDTH/2 )
                    dx += MAP_WIDTH*TILE_WIDTH;
                dy = Blast[i].YPos - Player[j].YPos;
                if ( dy > MAP_HEIGHT*TILE_WIDTH/2 )
                    dy -= MAP_HEIGHT*TILE_WIDTH;
                if ( dy < -MAP_HEIGHT*TILE_WIDTH/2 )
                    dy += MAP_HEIGHT*TILE_WIDTH;
                t = BLAST_TILE
                   + (BLAST2_TILE - BLAST_TILE)*(Blast[i].Type==PLASMA_LOCKED)
                   + (LASER_BLAST_TILE - BLAST_TILE)*(Blast[i].Type==LASER);
                DrawObject(p, dx+4*TILE_WIDTH, dy+4*TILE_WIDTH, t);
            }
        }
    }

    for ( i = 0 ; i < TOTAL_BLASTS ; i++ ) Blast[i].Active = 0;
}



/* If a token (gun/speed/slow/etc) is not currently in play, count the  */
/*  delay until it reappears                                            */
/* If it is in play, count the delay until it relocates itself - it can */
/*  only relocate when it isn't visible to either player                */
/* The gun token is made to animate                                     */
/* The healing token is made to pulsate when in use                     */

void MoveTokens() {
    int i;

    for ( i = 0 ; i < TOTAL_TOKENS ; i++ ) {
        switch ( Token[i].Type) {

          case UNUSED_TOKEN:
            break;

          case GUN_TOKEN:
            Token[i].Frame = ( Token[i].Frame + 1 ) % NUM_GUN_TOKEN_TILES;
            Token[i].Tile = GUN_TOKEN_TILES + Token[i].Frame
                       + (GUN2_TOKEN_TILES-GUN_TOKEN_TILES)*(GunTokenType==0)
                       + (LASER_TOKEN_TILES-GUN_TOKEN_TILES)*(GunTokenType<0);

            if ( Token[i].Active ) {

                if ( --Token[i].Delay == 0 ) {
                    if ( TileUnseen(Token[i].XTile, Token[i].YTile) ) {
                        FindUnseenTile(&(Token[i].XTile), &(Token[i].YTile));
                        Token[i].Delay = GUN_MOVE;
                    } else {
                        Token[i].Delay = GUN_MOVE/5;
                    }
                }

            } else {

                if ( --Token[i].Delay == 0 ) {
                    FindUnseenTile(&(Token[i].XTile), &(Token[i].YTile));
                    Token[i].Delay = GUN_MOVE;
                    Token[i].Active = 1;
                }

            }
            break;

          case HEALTH_TOKEN:
            if ( Player[0].Healing || Player[1].Healing ) {
                switch(Token[i].Tile) {
                  case HEALTH_TOKEN_TILE:
                    Token[i].Tile = VANISH_TOKEN_TILE;
                    break;
                  case VANISH_TOKEN_TILE:
                    Token[i].Tile = FAST_TOKEN_TILE;
                    break;
                  case FAST_TOKEN_TILE:
                    Token[i].Tile = SLOW_TOKEN_TILE;
                    break;
                  case SLOW_TOKEN_TILE:
                    Token[i].Tile = HEALTH_TOKEN_TILE;
                    break;
                }
            } else {
                Token[i].Tile = HEALTH_TOKEN_TILE;
            }
          case FAST_TOKEN:
          case SLOW_TOKEN:
          case VANISH_TOKEN:
          case SHIELD_TOKEN:
            if ( Token[i].Active ) {

                if ( --Token[i].Delay == 0 ) {
                    if ( TileUnseen(Token[i].XTile, Token[i].YTile) ) {
                        FindUnseenTile(&(Token[i].XTile), &(Token[i].YTile));
                        Token[i].Delay = TOKEN_MOVE;
                    } else {
                        Token[i].Delay = TOKEN_MOVE/3;
                    }
                }

            } else {

                if ( --Token[i].Delay == 0 ) {
                    FindUnseenTile(&(Token[i].XTile), &(Token[i].YTile));
                    Token[i].Delay = TOKEN_MOVE;
                    Token[i].Active = 1;
                }

            }
            break;

        }
    }
}



/* All active tokens are drawn in both players' viewing buffers */

void DrawTokens() {
    char far * p;
    int i, j;
    int dx, dy;

    for ( j = 0 ; j < 2 ; j++ ) {
        p = Player[j].ViewBuffer
                + VIEW_BUFF_WIDTH*(Player[j].YPos % TILE_WIDTH)
                + (Player[j].XPos % TILE_WIDTH);

        for ( i = 0 ; i < TOTAL_TOKENS ; i++ ) {
            if ( Token[i].Active && Token[i].Type != UNUSED_TOKEN ) {
                dx = Token[i].XTile*TILE_WIDTH - Player[j].XPos;
                if ( dx > MAP_WIDTH*TILE_WIDTH/2 )
                    dx -= MAP_WIDTH*TILE_WIDTH;
                if ( dx < -MAP_WIDTH*TILE_WIDTH/2 )
                    dx += MAP_WIDTH*TILE_WIDTH;
                dy = Token[i].YTile*TILE_WIDTH - Player[j].YPos;
                if ( dy > MAP_HEIGHT*TILE_WIDTH/2 )
                    dy -= MAP_HEIGHT*TILE_WIDTH;
                if ( dy < -MAP_HEIGHT*TILE_WIDTH/2 )
                    dy += MAP_HEIGHT*TILE_WIDTH;
                DrawObject(p, dx+4*TILE_WIDTH, dy+4*TILE_WIDTH, Token[i].Tile);
            }
        }
    }
}



/* Check to see if player <plyr> is on a token, and act accordingly */
/* When a token is collected there is a countdown till it reappears */
/* Only one player can have a gun at any one time - the player with */
/*  the gun is made to move slower                                  */
/* A player can only have one speed/slow active at a time; a speed  */
/*  token cancels out a slow token and vice versa                   */
/* Player starts to heal if standing over a healing token           */
/* (plyr=0 : Player 1, plyr=1 : Player 2)                           */

void CollectTokens(int plyr) {
    int i;

    for ( i = 0 ; i < TOTAL_TOKENS ; i++ ) {
        switch ( Token[i].Type ) {

          case UNUSED_TOKEN:
            break;

          case GUN_TOKEN:
            if ( Token[i].Active
                && TileCollision(Player[plyr].XPos, Player[plyr].YPos,
                    Token[i].XTile*TILE_WIDTH, Token[i].YTile*TILE_WIDTH) ) {

                Token[i].Active = 0;
                Token[i].Delay = GUN_REAPPEAR;

                if ( Player[plyr].Gun == 0 ) {
                    Player[plyr].Speed--;
                    Player[plyr].GunWait = 0;
                }
                if ( Player[plyr^1].Gun ) {
                    Player[plyr^1].Gun = 0;
                    Player[plyr^1].Speed++;
                }

                if ( GunTokenType >= 0 ) {

                  /* Fire/plasma balls */
                    if ( (GunTokenType==0) || ( (Player[plyr].Gun!=0)
                                       && (Player[plyr].GunType==PLASMA) ) ) {
                        Player[plyr].GunType = PLASMA;
                    } else {
                        Player[plyr].GunType = FIRE;
                    }
                    Player[plyr].Gun = GUN_HELDFOR;
                    if ( --GunTokenType < 0 )
                        GunTokenType = HEATSEEK_REAPPEAR1
                                      + random(HEATSEEK_REAPPEAR2);

                } else {

                  /* Laser bolts */
                    Player[plyr].GunType = LASER;
                    Player[plyr].Gun = LASER_HELDFOR;

                }
                MakeSound(BIG_CHIME);
            }
            break;

          case FAST_TOKEN:
            if ( Token[i].Active
                 && TileCollision(Player[plyr].XPos, Player[plyr].YPos,
                      Token[i].XTile*TILE_WIDTH, Token[i].YTile*TILE_WIDTH) ) {
                Token[i].Active = 0;
                Token[i].Delay = TOKEN_REAPPEAR;
                if ( Player[plyr].SlowPower ) {
                    Player[plyr].SlowPower = 0;
                    Player[plyr].Speed++;
                } else {
                    if ( Player[plyr].FastPower == 0 ) {
                        Player[plyr].Speed++;
                    }
                    Player[plyr].FastPower = TOKEN_HELDFOR;
                }
                MakeSound(CHIME);
            }
            break;

          case SLOW_TOKEN:
            if ( Token[i].Active
                 && TileCollision(Player[plyr].XPos, Player[plyr].YPos,
                      Token[i].XTile*TILE_WIDTH, Token[i].YTile*TILE_WIDTH) ) {
                Token[i].Active = 0;
                Token[i].Delay = TOKEN_REAPPEAR;
                if ( Player[plyr].FastPower ) {
                    Player[plyr].FastPower = 0;
                    Player[plyr].Speed--;
                } else {
                    if ( Player[plyr].SlowPower == 0 ) {
                        Player[plyr].Speed--;
                    }
                    Player[plyr].SlowPower = TOKEN_HELDFOR;
                }
            }
            break;

          case VANISH_TOKEN:
            if ( Token[i].Active
                 && TileCollision(Player[plyr].XPos, Player[plyr].YPos,
                      Token[i].XTile*TILE_WIDTH, Token[i].YTile*TILE_WIDTH) ) {
                Token[i].Active = 0;
                Token[i].Delay = TOKEN_REAPPEAR;
                Player[plyr].VanishPower = TOKEN_HELDFOR;
                MakeSound(CHIME);
            }
            break;

          case HEALTH_TOKEN:
            if ( Token[i].Active && !Player[plyr].Dead
                    && TileCollision(Player[plyr].XPos, Player[plyr].YPos,
                          Token[i].XTile*TILE_WIDTH, Token[i].YTile*TILE_WIDTH)
                    && Player[plyr].Hits < SQUIDGY_HITS ) {
                if ( ++Player[plyr].Healing == HEALING_TIME ) {
                    Token[i].Active = 0;
                    Token[i].Delay = HEALTH_REAPPEAR;
                    Player[plyr].Healing = 0;
                    Player[plyr].Hits++;
                    MakeSound(CHIME);
                }
            } else {
                Player[plyr].Healing = 0;
            }
            break;

          case SHIELD_TOKEN:
            if ( Token[i].Active
                 && TileCollision(Player[plyr].XPos, Player[plyr].YPos,
                      Token[i].XTile*TILE_WIDTH, Token[i].YTile*TILE_WIDTH) ) {
                Token[i].Active = 0;
                Token[i].Delay = TOKEN_REAPPEAR;
                Player[plyr].ShieldPower = SHIELD_HELDFOR;
                if ( Player[plyr].Shields < SHIELDS_PER_PLAYER ) {
                    Player[plyr].Shields++;
                }
                MakeSound(CHIME);
            }
            break;

        }
    }
}



/* The times remaining for player <plyr> special powers (gun,speed,etc) */
/*  are decreased and eventually the powers are made to vanish          */
/* (plyr=0 : Player 1, plyr=1 : Player 2)                               */

void PowersFade(int plyr) {

    if ( Player[plyr].Gun > 0 ) {
        if ( --Player[plyr].Gun == 0 ) {
            Player[plyr].Speed++;
            Player[plyr].GunType = FIRE;
        }
    }

    if ( Player[plyr].FastPower > 0 ) {
        if ( --Player[plyr].FastPower == 0 ) {
            Player[plyr].Speed--;
        }
    }

    if ( Player[plyr].SlowPower > 0 ) {
        if ( --Player[plyr].SlowPower == 0 ) {
            Player[plyr].Speed++;
        }
    }

    if ( Player[plyr].VanishPower > 0 ) {
        Player[plyr].VanishPower--;
    }

    if ( Player[plyr].Shields && Player[plyr].ShieldPower > 0 ) {
        if ( --Player[plyr].ShieldPower == 0 ) {
            if ( --Player[plyr].Shields > 0 ) {
                Player[plyr].ShieldPower = SHIELD_HELDFOR;
            }
        }
    }

}



/* Position the shield units around player <plyr> for this frame  */
/* Their positions depend on the number of shields the player has */
/*  and on the progress of the player's ShieldFrame variable      */

void GenerateShields(int plyr) {
    int i;
    int phase;
    int x, y;

    int shield_pos[12][2] = { {0, -12}, {6, -10},  {10, -6},
                              {12, 0},  {10, 6},   {6, 10},
                              {0, 12},  {-6, 10},  {-10, 6},
                              {-12,0},  {-10, -6}, {-6, -10} };

    for ( i = 0 ; i < SHIELDS_PER_PLAYER ; i++ ) {
        Shield[i+plyr*SHIELDS_PER_PLAYER].Active = 0;
    }

    if ( Player[plyr].Dead ) return;

    if ( Player[plyr].Shields <= 0 ) return;

    phase = ( Player[plyr].ShieldFrame + 6/Player[plyr].Shields ) % 12;
    for ( i = 0 ; i < Player[plyr].Shields ; i++ ) {

        x = Player[plyr].XPos + shield_pos[phase][0] + 5;
        y = Player[plyr].YPos + shield_pos[phase][1] + 5;
        if ( x >= MAP_WIDTH*TILE_WIDTH ) x -= MAP_WIDTH*TILE_WIDTH;
        if ( y >= MAP_HEIGHT*TILE_WIDTH ) y -= MAP_HEIGHT*TILE_WIDTH;
        if ( x < 0 ) x += MAP_WIDTH*TILE_WIDTH;
        if ( y < 0 ) y += MAP_HEIGHT*TILE_WIDTH;

        Shield[i+plyr*SHIELDS_PER_PLAYER].Active = 1;
        Shield[i+plyr*SHIELDS_PER_PLAYER].XPos = x;
        Shield[i+plyr*SHIELDS_PER_PLAYER].YPos = y;
        Shield[i+plyr*SHIELDS_PER_PLAYER].Owner = plyr;

        phase = ( phase + 12/Player[plyr].Shields ) % 12;
    }

    Player[plyr].ShieldFrame = (Player[plyr].ShieldFrame+1) % 12;
}



/* Draw the shield units for both players */

void DrawShields() {
    char far * p;
    int i, j;
    int dx, dy;
    int t;

    for ( j = 0 ; j < 2 ; j++ ) {
        p = Player[j].ViewBuffer
                + VIEW_BUFF_WIDTH*(Player[j].YPos % TILE_WIDTH)
                + (Player[j].XPos % TILE_WIDTH);

        for ( i = 0 ; i < 2*SHIELDS_PER_PLAYER ; i++ ) {
            if ( Shield[i].Active ) {
                dx = Shield[i].XPos - Player[j].XPos;
                if ( dx > MAP_WIDTH*TILE_WIDTH/2 )
                    dx -= MAP_WIDTH*TILE_WIDTH;
                if ( dx < -MAP_WIDTH*TILE_WIDTH/2 )
                    dx += MAP_WIDTH*TILE_WIDTH;
                dy = Shield[i].YPos - Player[j].YPos;
                if ( dy > MAP_HEIGHT*TILE_WIDTH/2 )
                    dy -= MAP_HEIGHT*TILE_WIDTH;
                if ( dy < -MAP_HEIGHT*TILE_WIDTH/2 )
                    dy += MAP_HEIGHT*TILE_WIDTH;
                t = SHIELD_TILE;
                DrawObject(p, dx + 4*TILE_WIDTH, dy + 4*TILE_WIDTH, t);
            }
        }
    }
}



/* Check to see if player <plyr> runs into a shield unit      */
/* If so an explosion is triggered and the player is injured  */
/* Only the shields belonging to the other player are checked */
/* (plyr=0 : Player 1, plyr=1 : Player 2)                     */

void ShieldHitsSquidgy(int plyr) {
    int i;
    int x, y;
    int dx, dy;
    int t;

    x = Player[plyr].XPos;
    y = Player[plyr].YPos;

    for ( i = 0 ; i < 2*SHIELDS_PER_PLAYER ; i++ ) {

        if ( Shield[i].Active && Shield[i].Owner != plyr ) {

            dx = x - Shield[i].XPos;
            dy = y - Shield[i].YPos;
            if ( dx > MAP_WIDTH*TILE_WIDTH/2 ) dx -= MAP_WIDTH*TILE_WIDTH;
            if ( dx < -MAP_WIDTH*TILE_WIDTH/2 ) dx += MAP_WIDTH*TILE_WIDTH;
            if ( dy > MAP_HEIGHT*TILE_WIDTH/2 ) dy -= MAP_HEIGHT*TILE_WIDTH;
            if ( dy < -MAP_HEIGHT*TILE_WIDTH/2 ) dy += MAP_HEIGHT*TILE_WIDTH;

            if ( dy > -TILE_WIDTH && dy < ZAP_WIDTH
                        && dx > -TILE_WIDTH && dx < ZAP_WIDTH ) {
                SetBlast(Shield[i].XPos - (TILE_WIDTH-ZAP_WIDTH)/2,
                         Shield[i].YPos - (TILE_WIDTH-ZAP_WIDTH)/2, FIRE);

                Shield[i].Active = 0;
                if ( --Player[plyr^1].Shields > 0 ) {
                    Player[plyr^1].ShieldPower = SHIELD_HELDFOR;
                }

                if ( Player[plyr].Hits-- > 0 ) MakeSound(SQUEAK);
                else Player[plyr].Hits = 0;
                t = Player[plyr].Tile;
                if ( TileTypeNum(t) == CRITTER ) {
                    t = ( ( t - BLUE_CRITTER_TILES ) % 16 ) / 2;
                    t += SKELEBONE_TILES;
                    Player[plyr].Tile = t;
                }
                if ( TileTypeNum(t) == EVIL ) {
                    t = ( ( t - EVIL_CRITTER_TILES ) % 16 ) / 2;
                    t += SKELEBONE_TILES;
                    Player[plyr].Tile = t;
                }

            }
        }

    }
}



/* Destroy shields that collide with shields of the other player */

void ShieldsHitShields() {
    int i, j;
    int x, y;
    int x2, y2;
    int dx, dy;

    for ( i = 0 ; i < 2*SHIELDS_PER_PLAYER ; i++ ) {
        if ( Shield[i].Active && Shield[i].Owner == 0 ) {

            x = Shield[i].XPos;
            y = Shield[i].YPos;

            for ( j = 0 ; j < 2*SHIELDS_PER_PLAYER ; j++ ) {
                if ( Shield[j].Active && Shield[j].Owner == 1 ) {

                  /* Shield[i] is owned by player 1 */
                  /* Shield[j] is owned by player 2 */

                    dx = x - Shield[j].XPos;
                    dy = y - Shield[j].YPos;
                    if ( dx > MAP_WIDTH*TILE_WIDTH/2 )
                                dx -= MAP_WIDTH*TILE_WIDTH;
                    if ( dx < -MAP_WIDTH*TILE_WIDTH/2 )
                                dx += MAP_WIDTH*TILE_WIDTH;
                    if ( dy > MAP_HEIGHT*TILE_WIDTH/2 )
                                dy -= MAP_HEIGHT*TILE_WIDTH;
                    if ( dy < -MAP_HEIGHT*TILE_WIDTH/2 )
                                dy += MAP_HEIGHT*TILE_WIDTH;

                    if ( dy >= -ZAP_WIDTH && dy <= ZAP_WIDTH
                                && dx >= -ZAP_WIDTH && dx <= ZAP_WIDTH ) {

                        x2 = Shield[j].XPos + dx/2 - (TILE_WIDTH-ZAP_WIDTH)/2;
                        y2 = Shield[j].YPos + dy/2 - (TILE_WIDTH-ZAP_WIDTH)/2;
                        if ( x2 < 0 ) x2 += MAP_WIDTH*TILE_WIDTH;
                        if ( y2 < 0 ) y2 += MAP_HEIGHT*TILE_WIDTH;
                        if ( x2 >= MAP_WIDTH*TILE_WIDTH )
                                    x2 -= MAP_WIDTH*TILE_WIDTH;
                        if ( y2 >= MAP_HEIGHT*TILE_WIDTH )
                                    y2 -= MAP_HEIGHT*TILE_WIDTH;
                        SetBlast(x2, y2, FIRE);

                        Shield[i].Active = 0;
                        if ( --Player[0].Shields > 0 ) {
                                Player[0].ShieldPower = SHIELD_HELDFOR;
                        }
                        Shield[j].Active = 0;
                        if ( --Player[1].Shields > 0 ) {
                                Player[1].ShieldPower = SHIELD_HELDFOR;
                        }

                        break;
                    }

                }
            }
        }
    }
}



/* Destroy zaps that collide with shields of the other player */

void ShieldsHitZaps() {
    int i, j;
    int x, y;
    int x2, y2;
    int dx, dy;

    for ( i = 0 ; i < 2*SHIELDS_PER_PLAYER ; i++ ) {
        if ( Shield[i].Active ) {

            x = Shield[i].XPos;
            y = Shield[i].YPos;

            for ( j = 0 ; j < TOTAL_ZAPS ; j++ ) {
                if ( Zap[j].Active && Zap[j].Owner != Shield[i].Owner ) {

                  /* Shield[i] and Zap[j] have different owners */

                    dx = x - Zap[j].XPos;
                    dy = y - Zap[j].YPos;
                    if ( dx > MAP_WIDTH*TILE_WIDTH/2 )
                                dx -= MAP_WIDTH*TILE_WIDTH;
                    if ( dx < -MAP_WIDTH*TILE_WIDTH/2 )
                                dx += MAP_WIDTH*TILE_WIDTH;
                    if ( dy > MAP_HEIGHT*TILE_WIDTH/2 )
                                dy -= MAP_HEIGHT*TILE_WIDTH;
                    if ( dy < -MAP_HEIGHT*TILE_WIDTH/2 )
                                dy += MAP_HEIGHT*TILE_WIDTH;

                    if ( dy >= -ZAP_WIDTH && dy <= ZAP_WIDTH
                                && dx >= -ZAP_WIDTH && dx <= ZAP_WIDTH ) {

                        x2 = Zap[j].XPos + dx/2 - (TILE_WIDTH-ZAP_WIDTH)/2;
                        y2 = Zap[j].YPos + dy/2 - (TILE_WIDTH-ZAP_WIDTH)/2;
                        if ( x2 < 0 ) x2 += MAP_WIDTH*TILE_WIDTH;
                        if ( y2 < 0 ) y2 += MAP_HEIGHT*TILE_WIDTH;
                        if ( x2 >= MAP_WIDTH*TILE_WIDTH )
                                    x2 -= MAP_WIDTH*TILE_WIDTH;
                        if ( y2 >= MAP_HEIGHT*TILE_WIDTH )
                                    y2 -= MAP_HEIGHT*TILE_WIDTH;
                        SetBlast(x2, y2, FIRE);

                        Shield[i].Active = 0;
                        if ( --Player[Shield[i].Owner].Shields > 0 ) {
                                Player[Shield[i].Owner].ShieldPower
                                                        = SHIELD_HELDFOR;
                        }
                        Zap[j].Active = 0;

                        break;
                    }

                }
            }
        }
    }
}



/* Erase an area <width> by <height> from position (x, y) of the */
/*  virtaul screen at address <offset>                           */

void UndrawSprite(int x, int y, int width, int height, unsigned offset) {
    x_rect_fill(x, y, x+width, y+height, offset, 0);
}



/* Draw the life (and death) counters for both players */

void DisplayLives() {
    char far * alive;
    char far * dead;
    int i;
    int across;

    dead = (char far *) ( TileSprites + SKELEBONE_TILES*TILE_SIZE );

    alive = (char far *) ( TileSprites + CLEAN_BLUE_TILE*TILE_SIZE );
    across = 16;
    for ( i = 0 ; i < SQUIDGY_LIVES ; i++ ) {
        if ( i == Player[0].Lives ) alive = dead;
        DrawSprite(across, 200, alive, TILE_WIDTH, TILE_WIDTH, HiddenPageOffs);
        across += 17;
    }

    alive = (char far *) ( TileSprites + CLEAN_RED_TILE*TILE_SIZE );
    across = 288;
    for ( i = 0 ; i < SQUIDGY_LIVES ; i++ ) {
        if ( i == Player[1].Lives ) alive = dead;
        DrawSprite(across, 200, alive, TILE_WIDTH, TILE_WIDTH, HiddenPageOffs);
        across -= 17;
    }
}



/* Draw (or else erase) the animated gun-held icons for both players */

void DisplayGuns() {
    int t;
    int gun;

    gun = 0;
    while ( Token[gun].Type != GUN_TOKEN ) gun++;

    t = GUN_TOKEN_TILES + Token[gun].Frame
        + (GUN2_TOKEN_TILES - GUN_TOKEN_TILES)*(Player[0].GunType == PLASMA)
        + (LASER_TOKEN_TILES - GUN_TOKEN_TILES)*(Player[0].GunType == LASER);

    if ( Player[0].Gun ) {
        DrawSprite(88, 203, (char far *) TileSprites + TILE_SIZE*t,
                    TILE_WIDTH, TILE_WIDTH, HiddenPageOffs);
    } else {
        UndrawSprite(88, 203, TILE_WIDTH, TILE_WIDTH, HiddenPageOffs);
    }

    t = GUN_TOKEN_TILES + Token[gun].Frame
        + (GUN2_TOKEN_TILES - GUN_TOKEN_TILES)*(Player[1].GunType == PLASMA)
        + (LASER_TOKEN_TILES - GUN_TOKEN_TILES)*(Player[1].GunType == LASER);

    if ( Player[1].Gun ) {
        DrawSprite(216, 203, (char far *) TileSprites + TILE_SIZE*t,
                    TILE_WIDTH, TILE_WIDTH, HiddenPageOffs);
    } else {
        UndrawSprite(216, 203, TILE_WIDTH, TILE_WIDTH, HiddenPageOffs);
    }

}



/* Display the heart-shaped hit counters for bothe players */

void DisplayHits() {
    char far * heart;
    int across;
    int i;

    heart = (char far *) ( TileSprites + HEART_TILE*TILE_SIZE );

    UndrawSprite(21, 219, 10*SQUIDGY_HITS, 9, HiddenPageOffs);
    UndrawSprite(260, 219, 10*SQUIDGY_HITS, 9, HiddenPageOffs);

    across = 21;
    for ( i = 0 ; i < Player[0].Hits ; i++ ) {
        DrawSprite(across, 219, heart, TILE_WIDTH, 9, HiddenPageOffs);
        across += 10;
    }

    across = 260 + 10*(SQUIDGY_HITS - Player[1].Hits);
    for ( i = 0 ; i < Player[1].Hits ; i++ ) {
        DrawSprite(across, 219, heart, TILE_WIDTH, 9, HiddenPageOffs);
        across += 10;
    }

}



/* Draw bars showing the time remaining on the gun for both players */

void DisplayBars() {
    int i;
    int len;
    char far * p;
    int s;
    char c;
    int heldfor;

    c = BAR_COLOUR
        + (BAR2_COLOUR - BAR_COLOUR)*(Player[0].GunType==PLASMA)
        + (BAR3_COLOUR - BAR_COLOUR)*(Player[0].GunType==LASER);
    p = MK_FP( (unsigned)0xA000, (unsigned)(HiddenPageOffs + 224L*80 + 22) );
    if ( Player[0].GunType == LASER ) heldfor = LASER_HELDFOR;
    else heldfor = GUN_HELDFOR;
    len = (16*Player[0].Gun)/heldfor;
    outport(0x03C4, 0x0F02);
    for ( i = 0 ; i < len/4 ; i++ ) *(p+i) = *(p+i+80) = c;
    for ( i = len/4 ; i < 4 ; i++ ) *(p+i) = *(p+i+80) = 0;
    i = len%4;
    s = 1*(i > 0) + 2*(i > 1) + 4*(i > 2);
    outport(0x03C4, 2 + 256*s);
    i = len/4;
    *(p+i) = *(p+i+80) = c;

    c = BAR_COLOUR
        + (BAR2_COLOUR - BAR_COLOUR)*(Player[1].GunType==PLASMA)
        + (BAR3_COLOUR - BAR_COLOUR)*(Player[1].GunType==LASER);
    p = MK_FP( (unsigned)0xA000, (unsigned)(HiddenPageOffs + 224L*80 + 54) );
    if ( Player[1].GunType == LASER ) heldfor = LASER_HELDFOR;
    else heldfor = GUN_HELDFOR;
    len = (16*Player[1].Gun)/heldfor;
    outport(0x03C4, 0x0F02);
    for ( i = 0 ; i < len/4 ; i++ ) *(p+i) = *(p+i+80) = c;
    for ( i = len/4 ; i < 4 ; i++ ) *(p+i) = *(p+i+80) = 0;
    i = len%4;
    s = 1*(i > 0) + 2*(i > 1) + 4*(i > 2);
    outport(0x03C4, 2 + 256*s);
    i = len/4;
    *(p+i) = *(p+i+80) = c;

}



