
/*
 *  SQ_MAIN.C
 *
 *  Simon Hern     (Wolfson College, Cambridge, CB3 9BB, England)
 *  Summer 1995    (Email: sdh20@cam.ac.uk)
 *
 *  Code for "Squidgy Wars" game: main routines
 *
 */


/* Squidgy Header */
#include "squidgy.h"



/* Default keys for player 1 (Q,A,W,E,Z) */
#define KEY_P1_UP 16
#define KEY_P1_DOWN 30
#define KEY_P1_LEFT 17
#define KEY_P1_RIGHT 18
#define KEY_P1_FIRE 44
/* Default keys for player 2 (7,4,8,9,1 on number pad) */
#define KEY_P2_UP 71
#define KEY_P2_DOWN 75
#define KEY_P2_LEFT 72
#define KEY_P2_RIGHT 73
#define KEY_P2_FIRE 79



/* EXTERNAL VARIABLES */

/* SQ_GAME.C */

extern PlayerStruct Player[2];

/* .ASM */

extern char far MainPalette[256*3];
extern char far RawTiles[TILE_SIZE];
extern char far RawTokens[TILE_SIZE];
extern char far RawCritters[TILE_SIZE];
extern char far Logo[LOGO_WIDTH*LOGO_HEIGHT];
extern char Map[MAP_HEIGHT][MAP_WIDTH];
extern char Map2[MAP_HEIGHT][MAP_WIDTH];
extern char KeyTable[256];
extern int KeyPressed;

/* XLib */

extern long VsyncIntTicks;
extern WORD HiddenPageOffs;
extern WORD WaitingPageOffs;
extern WORD VisiblePageOffs;



/* EXTERNAL FUNCTIONS */

/* SQ_GAME.C */

extern void PlayGame();

/* SQ_FRONT.C */

extern int Titles();

/* SQ_CODE.ASM */

extern void MapToViewBuffer(int LeftEdge, int TopEdge, char far * buff);
extern void ViewBufferToScr(char far * buff, unsigned scr_addr);
extern void DrawObject(char far * view, int x, int y, int tile);

extern void InstallKeyboard();
extern void RemoveKeyboard();
extern void ReadJoystick(int stick, int * x, int * y, int * b1, int * b2);

extern void SBSetReg(int reg, int val);
extern int SBReadStatus();

extern void DrawSprite(int x, int y, char far * bmap,
                       int width, int height, unsigned offset);

/* XLib */

extern WORD x_set_mode(WORD mode, WORD width);
extern void x_text_mode();
extern void x_install_vsync_handler(int skip);
extern void x_remove_vsync_handler();
extern WORD x_set_tripplebuffer(WORD height);
extern void x_page_flip(WORD x, WORD y);
extern void x_put_pal_raw(BYTE far * pal, WORD num, WORD first);



/* FUNCTION DECLARATIONS */

void Terminate();
int ControlBreak();
void Error(char * comment);
int AbortPressed();

void AllocateMemory();
void FreeMemory();

void InstallHandlers();
int DetectJoysticks();
void DefaultControls();

int TileType(int x, int y);
int TileTypeNum(int t);
void TranslateMap();
void SwitchMaps();

void CopyTile(char far * source, int dest);
void AddTiles(int target, int source);
void MaskCopyTile(int srce, int dest);
void FlipTile_tb(int srce);
void FlipTile_lr(int srce);
void FlipTile_tlbr(int srce);
void FlipTile_trbl(int srce);

void ExpandRawTiles();
void ExpandRawTokens();
void ExpandRawCritters();
void TamperWithLogo();

int SBDetect();
void SBReset();
void PrepareSounds();
void MakeSound(int num);
void SoundEffects();



/* GLOBAL VARIABLES */

int Joysticks;  /* Bit 0 = Joystick A present, Bit 1 = Joystick B present */
int SoundOn;  /* Adlib compatible card detected */
int SoundStatus[TOTAL_SOUNDS];  /* Status bytes for the sound effects */
                                /* - see SoundTypes enum              */

char * TileSprites;  /* Address of expanded (tile-shaped) sprites */
char * alloc_TileSprites;  /* Address of first byte of memory allocated for */
                 /* TileSprites - TileSprites itself begins on an even word */

int MusicSwitch;  /* Command-line switch - no music */
int SoundSwitch;  /* Command-line switch - no sound */
int IntroSwitch;  /* Command-line switch - hold intro page */

char * ErrorMessage = NULL;  /* Message displayed on exit by Error() */

char * ExitMessage = "\n"  /* Shameless grovelling for money */

    " SQUIDGY WARS  (Version date " VERSION_DATE ")\n"
    " by Simon Hern and Roland Higginson\n"
    "\n"
    "  This program and its source code are Public Domain (they may be\n"
    " freely distributed provided they are not changed in any way).\n"
    "  But anyone who: a) believes religiously in the idea of Shareware,\n"
    "                  b) thinks this game is the best program ever written,\n"
    "               or c) has more money than they know what to do with\n"
    " is welcome to send charitable donations towards the \"Writing Computer\n"
    " Games Really Is More Fun Than Doing Proper Work\" fund to:\n"
    "\n"
    "     Simon Hern, Wolfson College, Cambridge, CB3 9BB, England.\n"
    "                    (Email: sdh20@cam.ac.uk)\n";

char * OptionsMessage = "\n"  /* Details of command-line options */

    " Command-line options:   /m - Don\'t play music\n"
    "                         /s - Turn off all sound\n"
    "                         /i - Hold introduction screen until key pressed\n";





/* THE CODE */

/* Main function. Initialises program and runs the game */

void main( int argc, char * argv[] ) {
    int exit_code;
    int i;
    char * p;
    long t;

  /* Process command-line arguments */
    MusicSwitch = SoundSwitch = IntroSwitch = 0;
    for ( i = 1 ; i < argc ; i++ ) {
        p = argv[i];
        if ( ( *p != '-' && *p != '/' ) || strlen(p) != 2 ) {
            Printf(OptionsMessage);
            exit(1);
        }
        switch ( *(p+1) ) {
          case 'm':
          case 'M':
            MusicSwitch = 1;
            break;
          case 's':
          case 'S':
            SoundSwitch = 1;
            break;
          case 'i':
          case 'I':
            IntroSwitch = 1;
            break;
          default:
            Printf(OptionsMessage);
            exit(1);
        }
    }

  /* Joystick and soundcard */
    Joysticks = DetectJoysticks();
    DefaultControls();
    SoundOn = SBDetect();
    if ( SoundSwitch ) SoundOn = 0;
    SBReset();

    AllocateMemory();
    randomize();

  /* Decode both game maps, initially in simple ascii form in MAP.ASM */
    TranslateMap();
    SwitchMaps();
    TranslateMap();

  /* Decompress/fiddle with the graphics data */
    TamperWithLogo();
    ExpandRawTokens();
    ExpandRawCritters();

  /* Video and keyboard */
    InstallHandlers();

    do {

        exit_code = Titles();  /* Present title menus */

        if ( exit_code == RET_GAME ) PlayGame();  /* Run the game */

    } while ( exit_code != RET_ESCAPE );

    FreeMemory();
    SBReset();
}



/* Function called automatically when exitting game           */
/* Removes graphics and keyboard handlers and prints messages */

void Terminate() {
    x_remove_vsync_handler();
    x_text_mode();
    RemoveKeyboard();
    if ( ErrorMessage != NULL ) printf("Error:  %s\n", ErrorMessage);
    else printf("%s\n%s", ExitMessage, OptionsMessage);
}



/* Function called automatically when control-break pressed              */
/* Terminates program. (Never needed - keyboard handler deals with this) */

int ControlBreak() {
    Terminate();
    return 0;
}



/* Sets up 320x240x256 screen mode, triple buffered at 15 frames/second */
/* Installs keyboard handler and writes palette                         */

void InstallHandlers() {

  /* XLib vsynced trip(p)le buffering */
  /* Screen updates 15 times a second */
    x_set_mode(X_MODE_320x240, SCR_WIDTH);
    x_install_vsync_handler(4);
    x_set_tripplebuffer(SCR_HEIGHT);

  /* Last one out, please turn off Mode X */
    atexit(Terminate);
    ctrlbrk(ControlBreak);

  /* Palette */
    x_put_pal_raw((BYTE far *)MainPalette, (WORD)256, (WORD)0);

  /* Keyboard handler */
    InstallKeyboard();
}



/* Returns bit 0 set if joystick 1 present, bit 1 set if joystick 2 present */

int DetectJoysticks() {
    int x, y, b1, b2;
    int j;

    j = 0;
    ReadJoystick(1, &x, &y, &b1, &b2);
    if ( x != 0 && y != 0 ) j += 1;
    ReadJoystick(2, &x, &y, &b1, &b2);
    if ( x != 0 && y != 0 ) j += 2;

    return j;
}



/* Keyboard control selected by default with predetermined keys */

void DefaultControls() {
    Player[0].ControlDevice = KEYBOARD;
    Player[0].KeyUp = KEY_P1_UP;
    Player[0].KeyDown = KEY_P1_DOWN;
    Player[0].KeyLeft = KEY_P1_LEFT;
    Player[0].KeyRight = KEY_P1_RIGHT;
    Player[0].KeyFire = KEY_P1_FIRE;

    Player[1].ControlDevice = KEYBOARD;
    Player[1].KeyUp = KEY_P2_UP;
    Player[1].KeyDown = KEY_P2_DOWN;
    Player[1].KeyLeft = KEY_P2_LEFT;
    Player[1].KeyRight = KEY_P2_RIGHT;
    Player[1].KeyFire = KEY_P2_FIRE;
}




/* Quit program, displaying error message <comment> */

void Error(char * comment) {
    ErrorMessage = comment;
    exit(1);
}



/* Returns true if keyboard handler detects abort key combination */
/* (either Escape, Control-C, or Ctrl-Alt-Del)                    */

int AbortPressed() {
    if ( KeyTable[KEY_ABORT_A1]
        || ( KeyTable[KEY_ABORT_B1] && KeyTable[KEY_ABORT_B2] )
        || ( KeyTable[KEY_ABORT_C1] && KeyTable[KEY_ABORT_C2]
                          && KeyTable[KEY_ABORT_C3] ) ) return 1;
    return 0;
}



/* Allocate memory space for the view buffers for each player       */
/*  and for the TileSprites array                                   */
/* The view buffers are always aligned so that they start at page   */
/*  boundaries (addresses mod 16), and the tiles so that they       */
/*  start at even addresses                                         */
/* (The '_alloc' variables keep track of the actual area allocated) */

void AllocateMemory() {
    unsigned seg, off;
    char far * mem_al;
    char * m_al;
    int i;

  /* ViewBuffer: far, aligned to segment boundary */
    for ( i = 0 ; i < 2 ; i++ ) {
        mem_al = (char far *) farmalloc(VIEW_BUFF_SIZE + 16);
        if ( mem_al == NULL ) Error("Insufficient memory for ViewBuffer array");
        seg = FP_SEG(mem_al) + (FP_OFF(mem_al) >> 4) + 1;
        Player[i].ViewBuffer = (char far *) MK_FP(seg, 0);
        Player[i].alloc_ViewBuffer = mem_al;
    }

  /* TileSprites: near, aligned to word boundary */
    m_al = (char *) malloc(TILE_SIZE*TOTAL_TILES + 1);
    if ( m_al == NULL ) Error("Insufficient memory for TileSprites array");
    TileSprites = (char *) ( m_al + ((unsigned)m_al & 1) );
    alloc_TileSprites = m_al;
}



/* Free the memory allocated by AllocateMemory() */

void FreeMemory() {
    farfree(Player[0].alloc_ViewBuffer);
    farfree(Player[1].alloc_ViewBuffer);
    free(alloc_TileSprites);
}



/* Given a tile coordinate (x,y) within the Map array, this returns  */
/*  the type of tile there - one of the values in the TileTypes enum */

int TileType(int x, int y) {
    char t = Map[ModHeight(y)][ModWidth(x)];
    if ( t >= FLOOR_TILES && t < FLOOR_TILES + NUM_FLOOR_TILES )
        return FLOOR;
    if ( t >= WALL_TILES && t < WALL_TILES + NUM_WALL_TILES )
        return WALL;
    if ( t >= BLUE_CRITTER_TILES && t < BLUE_CRITTER_TILES + 32 )
        return CRITTER;
    if ( t >= ZAP_TILE && t <= CLEAN_RED_TILE )
        return TOKEN;
    if ( t >= SKELEBONE_TILES && t < SKELEBONE_TILES + 8 )
        return SKELEBONE;
    if ( t >= EVIL_CRITTER_TILES && t < EVIL_CRITTER_TILES + 32 )
        return EVIL;
    return OTHER;
}



/* Given the number code of a tile, this returns a description of it */
/*  - one of the values in the TileTypes enum                        */

int TileTypeNum(int t) {
    if ( t >= FLOOR_TILES && t < FLOOR_TILES + NUM_FLOOR_TILES )
        return FLOOR;
    if ( t >= WALL_TILES && t < WALL_TILES + NUM_WALL_TILES )
        return WALL;
    if ( t >= BLUE_CRITTER_TILES && t < BLUE_CRITTER_TILES + 32 )
        return CRITTER;
    if ( t >= ZAP_TILE && t <= CLEAN_RED_TILE )
        return TOKEN;
    if ( t >= SKELEBONE_TILES && t < SKELEBONE_TILES + 8 )
        return SKELEBONE;
    if ( t >= EVIL_CRITTER_TILES && t < EVIL_CRITTER_TILES + 32 )
        return EVIL;
    return OTHER;
}



/* Scans the Map array (see MAP.ASM), replacing ascii '#' symbols with  */
/*  the code for wall tiles and ascii ' ' symbols with floor tiles      */
/* Adjacent wall sections are then made to join up, and floor tiles are */
/*  given shadows where they are next to walls                          */

void TranslateMap() {
    int i, j;
    int c;

  /* Translate from easy-to-read format to tile codes */

    for ( i = 0 ; i < MAP_HEIGHT ; i++ ) {
        for ( j = 0 ; j < MAP_WIDTH ; j++ ) {
            switch(Map[i][j]) {

              case ' ':
                Map[i][j] = FLOOR_TILES;
                break;

              case '#':
                Map[i][j] = WALL_TILES;
                break;

              default:
                Error("Unrecognized symbol in Map");

            }
        }
    }

  /* Change each tile to fit adjacent tiles */

    /* FLOOR_TILES = standard tile, no shadows                   */
    /*         + 1 = shadow from wall above                      */
    /*         + 2 = shadow from wall to left                    */
    /*         + 4 = shadow from wall above and to left          */
    /* etc, with FLOOR_TILES+7 looking the same as FLOOR_TILES+3 */

    /* WALL_TILES = solitary wall segment */
    /*        + 1 = wall joins to wall on right */
    /*        + 2 = wall joins to wall above    */
    /*        + 4 = wall joins to wall on left  */
    /*        + 8 = wall joins to wall below    */
    /* etc                                      */

    for ( i = 0 ; i < MAP_HEIGHT ; i++ ) {
        for ( j = 0 ; j < MAP_WIDTH ; j++ ) {
            if ( TileType(j, i) == FLOOR ) {
                c = FLOOR_TILES;
                c += 1*( TileType(j, ModHeight(i-1)) == WALL );
                c += 2*( TileType(ModWidth(j-1), i) == WALL );
                c += 4*( TileType(ModWidth(j-1), ModHeight(i-1)) == WALL );
                if ( c == FLOOR_TILES+7 ) c = FLOOR_TILES+3;
                Map[i][j] = c;
            } else if ( TileType(j, i) == WALL ) {
                c = WALL_TILES;
                c += 1*( TileType(ModWidth(j+1), i) == WALL );
                c += 2*( TileType(j, ModHeight(i-1)) == WALL );
                c += 4*( TileType(ModWidth(j-1), i) == WALL );
                c += 8*( TileType(j, ModHeight(i+1)) == WALL );
                Map[i][j] = c;
            }
        }
    }
}



/* Two map arrays are used in the game, Map and Map2                    */
/* Due to lack of programming foresight it turns out to be easier to    */
/*  physically swap the arrays than to redirect a pointer when swapping */
/*  between the maps                                                    */
/* So that's what happens here                                          */

void SwitchMaps() {
    int i, j;
    char c;

    for ( i = 0 ; i < MAP_HEIGHT ; i++ ) {
        for ( j = 0 ; j < MAP_WIDTH ; j++ ) {
            c = Map[i][j];
            Map[i][j] = Map2[i][j];
            Map2[i][j] = c;
        }
    }
}




/* Copy the sprite pointed to by <srce> from among the 'raw' collection */
/*  of sprites to position <dest> in the 'expanded' TileSprites array   */

void CopyTile(char far * srce, int dest) {
    int i;
    char * dp = TileSprites + dest*TILE_SIZE;
    for ( i = 0 ; i < TILE_SIZE ; i++ ) *dp++ = *srce++;
}




/* Add the sprite in tile <srce> to the sprite in tile <trgt> */
/* (This is used to give the shadow effect on floor tiles)    */

void AddTiles(int trgt, int srce) {
    int i;
    char * tp = TileSprites + trgt*TILE_SIZE;
    char * sp = TileSprites + srce*TILE_SIZE;
    for ( i = 0 ; i < TILE_SIZE ; i++ ) *tp++ += *sp++;
}



/* Copy the sprite in tile <srce> to the sprite in tile <trgt> leaving     */
/*  unchanged those pixels in the 'mask' region (source pixel has value 0) */
/* (This is used in the 'squidgies for walls' mode)                        */

void MaskCopyTile(int srce, int dest) {
    int i;
    char * sp = TileSprites + srce*TILE_SIZE;
    char * dp = TileSprites + dest*TILE_SIZE;
    char c;

    for ( i = 0 ; i < TILE_SIZE ; i++ ) {
        c = *sp++;
        if ( c != 0 ) *dp = c;
        dp++;
    }
}



/* Flip tile <srce>, top to bottom */

void FlipTile_tb(int srce) {
    char c;
    char * a = TileSprites + TILE_SIZE*srce;
    char * b = a + TILE_SIZE - TILE_WIDTH;
    int i, j;

    for ( i = 0 ; i < TILE_WIDTH/2 ; i++ ) {
        for ( j = 0 ; j < TILE_WIDTH ; j++ ) {
            c = *a;
            *a = *b;
            *b = c;
            a++;
            b++;
        }
        b -= 2*TILE_WIDTH;
    }
}



/* Flip tile <srce>, left to right */

void FlipTile_lr(int srce) {
    char c;
    char * a = TileSprites + TILE_SIZE*srce;
    char * b = a + TILE_WIDTH - 1;
    int i, j;

    for ( i = 0 ; i < TILE_WIDTH/2 ; i++ ) {
        for ( j = 0 ; j < TILE_WIDTH ; j++ ) {
            c = *a;
            *a = *b;
            *b = c;
            a += TILE_WIDTH;
            b += TILE_WIDTH;
        }
        a += 1 - TILE_SIZE;
        b -= TILE_SIZE + 1;
    }
}



/* Flip tile <srce>, top-left to bottom-right */

void FlipTile_tlbr(int srce) {
    char c;
    char * a;
    char * b;
    int i, j;

    for ( i = 0 ; i < TILE_WIDTH-1 ; i++ ) {
        a = TileSprites + TILE_SIZE*srce + i;
        b = TileSprites + TILE_SIZE*srce + TILE_SIZE-1 - TILE_WIDTH*i;
        for ( j = 0 ; j < TILE_WIDTH-1-i ; j++ ) {
            c = *a;
            *a = *b;
            *b = c;
            a += TILE_WIDTH;
            b -= 1;
        }
    }
}



/* Flip tile <srce>, top-right to bottom-left */

void FlipTile_trbl(int srce) {
    char c;
    char * a;
    char * b;
    int i, j;

    for ( i = 0 ; i < TILE_WIDTH-1 ; i++ ) {
        a = TileSprites + TILE_SIZE*srce + TILE_WIDTH-1 - i;
        b = TileSprites + TILE_SIZE*srce + TILE_SIZE - TILE_WIDTH*(i+1);
        for ( j = 0 ; j < TILE_WIDTH-1-i ; j++ ) {
            c = *a;
            *a = *b;
            *b = c;
            a += TILE_WIDTH;
            b += 1;
        }
    }
}



/* Copies floor and wall tiles from the RawTiles array to TileSprites */
/* There are three different floor textures and three different walls */
/*  - which to use is chosen at random                                */
/* Shadow effects are included in the 'expanded' floor tiles          */
/*                                                                    */
/* The sprites in RawTiles are arranged in the order:                 */
/*  3 different floor tiles                                           */
/*  6 shadow overlays to add to floors                                */
/*  3 different sets of 16 wall tiles                                 */

void ExpandRawTiles() {
    int i;
    char far * s;

    static int floor;
    static int walls;

  /* Choose floor and wall, different from last time */
    do i = random(3);
    while ( i == floor );
    floor = i;
    do i = random(3);
    while ( i == walls );
    walls = i;
    if ( Player[0].Lives == 2 && Player[1].Lives == 2 ) {
        if ( random(3) == 0 ) walls = -1;
    }

    if ( walls >= 0 ) {  /* Normal walls */

      /* Copy floor tiles */
        s = RawTiles + floor*TILE_SIZE;
        CopyTile(s, 0);

        s = RawTiles + 3*TILE_SIZE;
        for ( i = 1 ; i <= 6 ; i++ ) {
            CopyTile(s, i);
            s += TILE_SIZE;
        }

      /* And include shadows */
        for ( i = 1 ; i < 7 ; i++ ) {
            AddTiles(i, 0);
        }

      /* Copy wall tiles */
        s = RawTiles + 9*TILE_SIZE + walls*16*TILE_SIZE;
        for ( i = 7 ; i <= 22 ; i++ ) {
            CopyTile(s, i);
            s += TILE_SIZE;
        }

    } else {  /* Squidgies instead of walls */

        s = RawTiles + floor*TILE_SIZE;
        for ( i = 0 ; i <= 6 ; i++ ) {
            CopyTile(s,i);
        }
        for ( i = 7 ; i <= 22 ; i++ ) {
            CopyTile(s,i);
            MaskCopyTile(BLUE_CRITTER_TILES + 2*(i-7) + 1, i);
        }

    }
}



/* Copies pentagram, gun, etc tiles from the RawTokens array to TileSprites */
/*                                                                          */
/* The sprites in RawTokens appear in the same order as in TileSprites      */

void ExpandRawTokens() {
    int i;
    char far * s = RawTokens;

    for ( i = 0 ; i < 31 ; i++ ) {
        CopyTile(s, ZAP_TILE+i);
        s += TILE_SIZE;
    }
}



/* Copies squidgy tiles from the RawCritters array to TileSprites */
/* The squidgies are rotated, flipped, etc so that a tiles exist  */
/*  for each of the eight directions                              */
/*                                                                */
/* The sprites in RawCritters are arranged in the order:          */
/*  Blue squidgy: up frame 1, up frame 2,                         */
/*                up-right frame 1, up-right frame 2              */
/*  Red squidgy: as above (4 frames)                              */
/*  Blue squidgy with evil eyes: as above (4 frames)              */
/*  Red squidgy with evil eyes: as above (4 frames)               */
/*  Skelebones: pointing up, pointing up-right                    */

void ExpandRawCritters() {
    int i;
    char far * s = RawCritters;
    int t;

    for ( i = 0 ; i < 4 ; i++ ) CopyTile(s, BLUE_CRITTER_TILES+4*i);
    s += TILE_SIZE;
    for ( i = 0 ; i < 4 ; i++ ) CopyTile(s, BLUE_CRITTER_TILES+4*i+1);
    s += TILE_SIZE;
    for ( i = 0 ; i < 4 ; i++ ) CopyTile(s, BLUE_CRITTER_TILES+4*i+2);
    s += TILE_SIZE;
    for ( i = 0 ; i < 4 ; i++ ) CopyTile(s, BLUE_CRITTER_TILES+4*i+3);
    s += TILE_SIZE;

    for ( i = 0 ; i < 4 ; i++ ) CopyTile(s, RED_CRITTER_TILES+4*i);
    s += TILE_SIZE;
    for ( i = 0 ; i < 4 ; i++ ) CopyTile(s, RED_CRITTER_TILES+4*i+1);
    s += TILE_SIZE;
    for ( i = 0 ; i < 4 ; i++ ) CopyTile(s, RED_CRITTER_TILES+4*i+2);
    s += TILE_SIZE;
    for ( i = 0 ; i < 4 ; i++ ) CopyTile(s, RED_CRITTER_TILES+4*i+3);
    s += TILE_SIZE;

    for ( i = 0 ; i < 4 ; i++ ) CopyTile(s, EVIL_CRITTER_TILES+4*i);
    s += TILE_SIZE;
    for ( i = 0 ; i < 4 ; i++ ) CopyTile(s, EVIL_CRITTER_TILES+4*i+1);
    s += TILE_SIZE;
    for ( i = 0 ; i < 4 ; i++ ) CopyTile(s, EVIL_CRITTER_TILES+4*i+2);
    s += TILE_SIZE;
    for ( i = 0 ; i < 4 ; i++ ) CopyTile(s, EVIL_CRITTER_TILES+4*i+3);
    s += TILE_SIZE;

    for ( i = 0 ; i < 4 ; i++ ) CopyTile(s, EVIL_CRITTER_TILES+4*i+16);
    s += TILE_SIZE;
    for ( i = 0 ; i < 4 ; i++ ) CopyTile(s, EVIL_CRITTER_TILES+4*i+17);
    s += TILE_SIZE;
    for ( i = 0 ; i < 4 ; i++ ) CopyTile(s, EVIL_CRITTER_TILES+4*i+18);
    s += TILE_SIZE;
    for ( i = 0 ; i < 4 ; i++ ) CopyTile(s, EVIL_CRITTER_TILES+4*i+19);
    s += TILE_SIZE;

    for ( i = 0 ; i < 4 ; i++ ) CopyTile(s, SKELEBONE_TILES+2*i);
    s += TILE_SIZE;
    for ( i = 0 ; i < 4 ; i++ ) CopyTile(s, SKELEBONE_TILES+2*i+1);

    t = BLUE_CRITTER_TILES;
    for ( i = 0 ; i < 4 ; i++ ) {
        FlipTile_tlbr(t + 4);
        FlipTile_tlbr(t + 5);
        FlipTile_tb(t + 6);
        FlipTile_tb(t + 7);
        FlipTile_tb(t + 8);
        FlipTile_tb(t + 9);
        FlipTile_trbl(t + 10);
        FlipTile_trbl(t + 11);
        FlipTile_trbl(t + 12);
        FlipTile_trbl(t + 13);
        FlipTile_lr(t + 14);
        FlipTile_lr(t + 15);
        t += 16;
    }

    t = SKELEBONE_TILES;
    FlipTile_tlbr(t + 2);
    FlipTile_tb(t + 3);
    FlipTile_tb(t + 4);
    FlipTile_trbl(t + 5);
    FlipTile_trbl(t + 6);
    FlipTile_lr(t + 7);
}



/* Adjusts the colour offsets of the 'squidgy wars' logo sprite */
/*  and fills in appropriate values in the palette              */

void TamperWithLogo() {
    int i;
    char far * p;
    unsigned char c;

    p = (char far *) Logo;
    for ( i = 0 ; i < LOGO_WIDTH*LOGO_HEIGHT ; i++ ) {
        if ( (c = *p) > 15 ) *p = c - LOGO_PALETTE_WRONG + LOGO_PALETTE_RIGHT;
        p++;
    }

    p = (char far *) (MainPalette + 3*LOGO_PALETTE_RIGHT);
    for ( i = 0 ; i < 16 ; i++ ) {
        *p++ = ( LOGO_COL_RED * (15-i) ) / 15;
        *p++ = ( LOGO_COL_GREEN * (15-i) ) / 15;
        *p++ = ( LOGO_COL_BLUE * (15-i) ) / 15;
    }
}



/* Check to see if an Adlib or Soundblaster card is present */

int SBDetect() {
    int stat1, stat2;

    SBSetReg(0x01, 0x00);
    SBSetReg(0x04, 0x60);
    SBSetReg(0x04, 0x80);
    stat1 = SBReadStatus();
    SBSetReg(0x02, 0xFF);
    SBSetReg(0x04, 0x21);
    delay(1);
    stat2 = SBReadStatus();
    SBSetReg(0x04, 0x60);
    SBSetReg(0x04, 0x80);
    if ( (stat1 & 0xE0) == 0x00 && (stat2 & 0xE0) == 0xC0 ) return 1;
    return 0;

}



/* Resets the sound card (if present) */

void SBReset() {
    int i;

    if ( ! SoundOn ) return;

    for ( i = 0xB0 ; i <= 0xB8 ; i++ ) SBSetReg(i, 0x00);
    for ( i = 0x80 ; i <= 0x95 ; i++ ) SBSetReg(i, 0x0F);

    for ( i = 0x01 ; i <= 0x04 ; i++ ) SBSetReg(i, 0x00);
    SBSetReg(0x08, 0x00);
    for ( i = 0x20 ; i <= 0x35 ; i++ ) SBSetReg(i, 0x00);
    for ( i = 0x40 ; i <= 0x55 ; i++ ) SBSetReg(i, 0x00);
    for ( i = 0x60 ; i <= 0x75 ; i++ ) SBSetReg(i, 0x00);
    for ( i = 0x80 ; i <= 0x95 ; i++ ) SBSetReg(i, 0x00);
    for ( i = 0xA0 ; i <= 0xA8 ; i++ ) SBSetReg(i, 0x00);
    for ( i = 0xB0 ; i <= 0xB8 ; i++ ) SBSetReg(i, 0x00);
    SBSetReg(0xBD, 0x00);
    for ( i = 0xC0 ; i <= 0xC8 ; i++ ) SBSetReg(i, 0x00);
    for ( i = 0xE0 ; i <= 0xF5 ; i++ ) SBSetReg(i, 0x00);
}



/* Prepares the sound card to produce our five fantastic ;) sound effects */
/* The SoundStatus array is also prepared                                 */

void PrepareSounds() {
    int i;
    if ( ! SoundOn ) return;

    SBSetReg(0x01, 0x20);

    for ( i = 0xB0 ; i <= 0xB8 ; i++ ) SBSetReg(i, 0x00);
    for ( i = 0x80 ; i <= 0x95 ; i++ ) SBSetReg(i, 0x0F);

    SBSetReg(0xB0, 0x06);   /* Thump */
    SBSetReg(0x20, 0x01);   /* Channel 1 */
    SBSetReg(0x40, 0x10);
    SBSetReg(0x60, 0x77);
    SBSetReg(0x80, 0xFF);
    SBSetReg(0xE0, 0x00);
    SBSetReg(0x23, 0x01);
    SBSetReg(0x43, 0x00);
    SBSetReg(0x63, 0x86);
    SBSetReg(0x83, 0x17);
    SBSetReg(0xE3, 0x00);
    SBSetReg(0xA0, 0xAE);
    SBSetReg(0xC0, 0x0E);

    SBSetReg(0xB1, 0x0E);   /* Chime */
    SBSetReg(0x21, 0x00);   /* Channel 2 */
    SBSetReg(0x41, 0x18);
    SBSetReg(0x61, 0xF5);
    SBSetReg(0x81, 0xEA);
    SBSetReg(0xE1, 0x00);
    SBSetReg(0x24, 0x00);
    SBSetReg(0x44, 0x0A);
    SBSetReg(0x64, 0xF5);
    SBSetReg(0x84, 0xF5);
    SBSetReg(0xE4, 0x02);
    SBSetReg(0xA1, 0x00);
    SBSetReg(0xC1, 0x00);

    SBSetReg(0xB2, 0x15);   /* Big Chime */
    SBSetReg(0x22, 0x01);   /* Channel 3 */
    SBSetReg(0x42, 0x3F);
    SBSetReg(0x62, 0xFF);
    SBSetReg(0x82, 0x77);
    SBSetReg(0xE2, 0x00);
    SBSetReg(0x25, 0x41);
    SBSetReg(0x45, 0x0A);
    SBSetReg(0x65, 0xF6);
    SBSetReg(0x85, 0x93);
    SBSetReg(0xE5, 0x02);
    SBSetReg(0xA2, 0x6B);
    SBSetReg(0xC2, 0x00);

    SBSetReg(0xB3, 0x0D);   /* Whirr */
    SBSetReg(0x28, 0xC1);   /* Channel 4 */
    SBSetReg(0x48, 0x00);
    SBSetReg(0x68, 0xF0);
    SBSetReg(0x88, 0x77);
    SBSetReg(0xE8, 0x00);
    SBSetReg(0x2B, 0xC1);
    SBSetReg(0x4B, 0x19);
    SBSetReg(0x6B, 0xF0);
    SBSetReg(0x8B, 0x23);
    SBSetReg(0xEB, 0x00);
    SBSetReg(0xA3, 0xE5);
    SBSetReg(0xC3, 0x00);

    SBSetReg(0xB4, 0x19);   /* Squeak */
    SBSetReg(0x29, 0x02);   /* Channel 5 */
    SBSetReg(0x49, 0x00);
    SBSetReg(0x69, 0xA6);
    SBSetReg(0x89, 0xFF);
    SBSetReg(0xE9, 0x00);
    SBSetReg(0x2C, 0x0A);
    SBSetReg(0x4C, 0x05);
    SBSetReg(0x6C, 0x80);
    SBSetReg(0x8C, 0xFF);
    SBSetReg(0xEC, 0x00);
    SBSetReg(0xA4, 0x98);
    SBSetReg(0xC4, 0x00);

    SBSetReg(0xB5, 0x0A);   /* Fizz */
    SBSetReg(0x2A, 0x01);   /* Channel 6 */
    SBSetReg(0x4A, 0x0A);
    SBSetReg(0x6A, 0x77);
    SBSetReg(0x8A, 0xFF);
    SBSetReg(0xEA, 0x00);
    SBSetReg(0x2D, 0x01);
    SBSetReg(0x4D, 0x0A);
    SBSetReg(0x6D, 0x86);
    SBSetReg(0x8D, 0x17);
    SBSetReg(0xED, 0x03);
    SBSetReg(0xA5, 0xAE);
    SBSetReg(0xC5, 0x0E);

    for ( i = 0 ; i < TOTAL_SOUNDS ; i++ ) SoundStatus[i] = 0;
}



/* Given a sound number (see SoundTypes enum), sets the SoundStatus value */
/*  to indicate that the sound should be produced                         */

void MakeSound(int num) {

    switch ( num ) {
      case WHIRR:
        if ( SoundStatus[WHIRR] == 0 ) SoundStatus[WHIRR] = -1;
        break;
      case NO_WHIRR:
        if ( SoundStatus[WHIRR] == 2 ) SoundStatus[WHIRR] = 1;
        break;
      default:
        SoundStatus[num] = -1;
        break;
    }
}



/* Operates sound effects, depending on the values in SoundStatus       */
/* This routine is called once a frame, updating SoundStatus as it goes */

void SoundEffects() {
    int i;

    if ( ! SoundOn ) return;

    for ( i = 0 ; i < TOTAL_SOUNDS ; i++ ) {
        if ( SoundStatus[i] != 0 ) {
            switch ( i ) {

              case THUMP:
                if ( SoundStatus[i] == -1 ) {
                    SoundStatus[i] = 3;
                    SBSetReg(0xB0, 0x06);
                    SBSetReg(0xB0, 0x06 | 0x20);
                }
                if ( SoundStatus[i] == 1 ) {
                    SBSetReg(0xB0, 0x06);
                }
                SoundStatus[i]--;
                break;

              case FIZZ:
                if ( SoundStatus[i] == -1 ) {
                    SoundStatus[i] = 3;
                    SBSetReg(0xB5, 0x0A);
                    SBSetReg(0xB5, 0x0A | 0x20);
                }
                if ( SoundStatus[i] == 1 ) {
                    SBSetReg(0xB5, 0x0A);
                }
                SoundStatus[i]--;
                break;

              case CHIME:
                if ( SoundStatus[i] == -1 ) {
                    SoundStatus[i] = 2;
                    SBSetReg(0xB1, 0x0E);
                    SBSetReg(0xB1, 0x0E | 0x20);
                }
                SoundStatus[i]--;
                if ( SoundStatus[i] == 0 ) {
                    SBSetReg(0xB1, 0x0E);
                }
                break;

              case BIG_CHIME:
                if ( SoundStatus[i] == -1 ) {
                    SoundStatus[i] = 4;
                    SBSetReg(0xB2, 0x15);
                    SBSetReg(0xA2, 0x6B);
                    SBSetReg(0xB2, 0x15 | 0x20);
                }
                if ( SoundStatus[i] == 3 ) {
                    SBSetReg(0xB2, 0x16);
                    SBSetReg(0xA2, 0x20);
                    SBSetReg(0xB2, 0x16 | 0x20);
                }
                if ( SoundStatus[i] == 2 ) {
                    SBSetReg(0xB2, 0x15);
                    SBSetReg(0xA2, 0x6B);
                    SBSetReg(0xB2, 0x19 | 0x20);
                }
                if ( SoundStatus[i] == 1 ) {
                    SBSetReg(0xB2, 0x19);
                }
                SoundStatus[i]--;
                break;

              case WHIRR:
                if ( SoundStatus[i] == -1 ) {
                    SoundStatus[i] = 2;
                    SBSetReg(0xB3, 0x0D);
                    SBSetReg(0xB3, 0x0D | 0x20);
                }
                if ( SoundStatus[i] == 1 ) {
                    SoundStatus[i] = 0;
                    SBSetReg(0xB3, 0x0D);
                }
                break;

              case SQUEAK:
                if ( SoundStatus[i] == -1 ) {
                    SoundStatus[i] = 2;
                    SBSetReg(0xB4, 0x19);
                    SBSetReg(0xB4, 0x19 | 0x20);
                }
                SoundStatus[i]--;
                if ( SoundStatus[i] == 0 ) {
                    SBSetReg(0xB4, 0x19);
                }
                break;

              default:
                break;

            }
        }
    }
}



