
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <assert.h>
#include <mach/sa/stdarg.h> /*XXX*/
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>

#include "gzip.h"

volatile char (*vidbuf)[80][2];

/* Total size of the original gzip file before being split,
   and size of each individual file (except the last, which is shorter).
   Used only to display the percent-of-installation-complete bar.
   XXX It is annoying to have to set this here... */
#define GZIP_FILE_SIZE	16256325

/* Total size of install.exe.  XXX another horrible kludge. */
#define INSTALL_SIZE	133120

/* Name of file to try to create in order to add an IW icon
   to the Windows 95 start menu.  */
#define STARTMENUFILE	"C:\\WINDOWS\\STARTM~1\\PROGRAMS\\IW.PIF"

/* Color scheme to be used */
#define BGFILL_COLOR	color(WHITE, BLUE)
#define BG_COLOR	color(WHITE, BLUE)
#define TITLE_COLOR	lcolor(BLUE, CYAN)
#define MAIN_COLOR	color(WHITE, GREEN)
#define REQUEST_COLOR	color(WHITE, BLUE)
#define STRING_COLOR	color(YELLOW, GREEN)
#define OPTION_COLOR	color(WHITE, BLUE)
#define SELECT_COLOR	color(WHITE, CYAN)
#define ERROR_COLOR	color(YELLOW, RED)
#define PAGER_COLOR	lcolor(BLACK, CYAN)

#define COLOR_BLACK	0
#define COLOR_BLUE	1
#define COLOR_GREEN	2
#define COLOR_CYAN	3
#define COLOR_RED	4
#define COLOR_PURPLE	5
#define COLOR_BROWN	6
#define COLOR_WHITE	7
#define COLOR_YELLOW	14

#define color(fg, bg)	(COLOR_##fg | 0x08 | (COLOR_##bg << 4))
#define lcolor(fg, bg)	(COLOR_##fg | (COLOR_##bg << 4))

#define clear_screen()	\
	asm volatile("int $0x10" : : "a" (0x0003) : "eax")
#define move_cursor(x, y)	\
	asm volatile("int $0x10" : : "a" (0x0200), "b" (0), "d" ((y << 8) | x))
#define pokechar(x, y, c, a)	\
	(vidbuf[y][x][0] = (c), vidbuf[y][x][1] = (a))
#define pokeattr(x, y, a)	\
	(vidbuf[y][x][1] = (a))

static inline int bios_get_key()
{
	int keycode;
	asm volatile("int $0x16" : "=a" (keycode) : "a" (0x0000));
	return keycode & 0xffff;
}


void poke_string(int x, int y, char *str, char attr)
{
	while (*str)
	{
		pokechar(x, y, *str, attr);
		str++; x++;
	}
}

void center_string(int x, int y, char *str, char attr)
{
	poke_string(x - (strlen(str) >> 1), y, str, attr);
}

void fill_rect(int x, int y, int w, int h, char c, char attr)
{
	int i, j;

	for (i = 0; i < w; i++)
		for (j = 0; j < h; j++)
			pokechar(x+i, y+j, c, attr);
}

void shade_rect(int x, int y, int w, int h)
{
	int i, j;

	for (i = 0; i < w; i++)
		for (j = 0; j < h; j++)
			pokeattr(x+i, y+j, 0x07);
}

void line_rect(int x, int y, int w, int h, char attr)
{
	int i;

	for (i = 1; i < w-1; i++)
	{
		pokechar(x+i, y, 0xc4, attr);
		pokechar(x+i, y+h-1, 0xc4, attr);
	}
	for (i = 1; i < h-1; i++)
	{
		pokechar(x, y+i, 0xb3, attr);
		pokechar(x+w-1, y+i, 0xb3, attr);
	}
	pokechar(x, y, 0xda, attr);
	pokechar(x+w-1, y, 0xbf, attr);
	pokechar(x, y+h-1, 0xc0, attr);
	pokechar(x+w-1, y+h-1, 0xd9, attr);
}

void dline_rect(int x, int y, int w, int h, char attr)
{
	int i;

	for (i = 1; i < w-1; i++)
	{
		pokechar(x+i, y, 0xcd, attr);
		pokechar(x+i, y+h-1, 0xcd, attr);
	}
	for (i = 1; i < h-1; i++)
	{
		pokechar(x, y+i, 0xba, attr);
		pokechar(x+w-1, y+i, 0xba, attr);
	}
	pokechar(x, y, 0xc9, attr);
	pokechar(x+w-1, y, 0xbb, attr);
	pokechar(x, y+h-1, 0xc8, attr);
	pokechar(x+w-1, y+h-1, 0xbc, attr);
}

void line_box(int x, int y, int w, int h, char attr)
{
	shade_rect(x+2, y+1, w, h);
	fill_rect(x, y, w, h, ' ', attr);
	line_rect(x, y, w, h, attr);
}

void dline_box(int x, int y, int w, int h, char attr)
{
	shade_rect(x+2, y+1, w, h);
	fill_rect(x, y, w, h, ' ', attr);
	dline_rect(x, y, w, h, attr);
}

struct screensavebuf
{
	char buf[25][80][2];
};

void save_screen(struct screensavebuf *buf)
{
	memcpy(buf->buf, (void*)vidbuf, sizeof(struct screensavebuf));
}

void restore_screen(struct screensavebuf *buf)
{
	memcpy((void*)vidbuf, buf->buf, sizeof(struct screensavebuf));
}

void ui_exit(int rc)
{
	fill_rect(0, 0, 80, 25, ' ', color(WHITE, BLACK));
	move_cursor(0, 0);
	if (rc)
		abort_gzip();
	else
		exit(0);
}

#define PATH_LEN 40

void key_request(char *title, char attr)
{
	struct screensavebuf sbuf;
	int done = 0;

	save_screen(&sbuf);

	line_box(20, 11, 40, 4, attr);
	center_string(40, 12, title, attr);
	center_string(40, 13, "Press ENTER to continue.", attr);

	do
	{
		int i, c;

		c = bios_get_key();
		c &= 0xff;
		switch (c)
		{
		case 27:
			ui_exit(1);
			break;
		case '\r':
		case '\n':
			done = 1;
			break;
		}
	}
	while (!done);

	restore_screen(&sbuf);
}

void error_box(const char *fmt, ...)
{
	va_list vl;
	char str[38+1];

	va_start(vl, fmt);
	vsnprintf(str, sizeof(str), fmt, vl);
	va_end(vl);

	key_request(str, ERROR_COLOR);
}

void string_request(char *title, char *buf)
{
	struct screensavebuf sbuf;
	int done = 0;
	int pos = strlen(buf);

	save_screen(&sbuf);

	line_box(15, 10, 50, 6, REQUEST_COLOR);
	center_string(40, 12, title, REQUEST_COLOR);

	do
	{
		int i, c;

		/* Refresh the string field. */
		for (i = 0; i < PATH_LEN; i++)
		{
			if (buf[i])
				pokechar(20+i, 13, buf[i], STRING_COLOR);
			else for (; i < PATH_LEN; i++)
				pokechar(20+i, 13, ' ', STRING_COLOR);
		}

		/* Update the cursor position. */
		move_cursor(20+pos, 13);

		c = bios_get_key();
		switch (c)
		{
		case 'K' << 8:
			if (pos > 0) pos--;
			break;
		case 'M' << 8:
			if (pos < strlen(buf)) pos++;
			break;
		case 'G' << 8:
			pos = 0;
			break;
		case 'O' << 8:
			pos = strlen(buf);
			break;
		case 'S' << 8:
			if (pos < strlen(buf))
				strcpy(&buf[pos], &buf[pos+1]);
			break;
		default:
			c &= 0xff;
			switch (c)
			{
			case 27:
				ui_exit(1);
				break;
			case '\r':
			case '\n':
				done = 1;
				break;
			case 8:
				if (pos > 0)
				{
					strcpy(&buf[pos-1], &buf[pos]);
					pos--;
				}
				break;
			default:
				if ((c > ' ') && (c < 127) && (pos < PATH_LEN))
				{
					if (pos == strlen(buf))
						buf[pos+1] = 0;
					buf[pos++] = c;
				}
				break;
			}
		}
	}
	while (!done);

	move_cursor(0, 24);
	restore_screen(&sbuf);
}

int option_request(char *title, int sel, ...)
{
	struct screensavebuf sbuf;
	int done = 0;
	int i, items, toprow;
	char *item;
	va_list vl;

	/* Count the number of items to select from. */
	va_start(vl, sel);
	for (items = 0; va_arg(vl, char*); items++);
	va_end(vl);

	assert(sel >= 0);
	assert(sel < items);

	save_screen(&sbuf);

	toprow = 11-items/2;
	line_box(20, toprow, 40, 6+items, REQUEST_COLOR);
	center_string(40, toprow+2, title, REQUEST_COLOR);

	do
	{
		int i, c;

		/* Refresh the item display. */
		va_start(vl, sel);
		for (i = 0; i < items; i++)
		{
			int attr = i == sel ? SELECT_COLOR : OPTION_COLOR;
			fill_rect(22, toprow+4+i, 36, 1, ' ', attr);
			center_string(40, toprow+4+i, va_arg(vl, char*), attr);
		}
		va_end(vl);

		c = bios_get_key();
		switch (c)
		{
		case 'H' << 8:
			if (sel > 0) sel--;
			break;
		case 'P' << 8:
			if (sel < items-1) sel++;
			break;
		case 'G' << 8:
		case 'I' << 8:
			sel = 0;
			break;
		case 'O' << 8:
		case 'Q' << 8:
			sel = items-1;
			break;
		default:
			c &= 0xff;
			switch (c)
			{
			case 27:
				sel = -1;
			case '\r':
			case '\n':
				done = 1;
				break;
			}
		}
	}
	while (!done);

	restore_screen(&sbuf);

	return sel;
}

void blank_screen()
{
	clear_screen();

	fill_rect(0, 0, 80, 25, 0xb0, BGFILL_COLOR);
	poke_string(0, 24, "ESC = Exit", BG_COLOR);
}

void draw_title_box()
{
	dline_box(15, 1, 50, 7, TITLE_COLOR);
	center_string(40, 3, "Inner Worlds", TITLE_COLOR);
	center_string(40, 4, "Copyright 1996 Sleepless Software",
			TITLE_COLOR);
	center_string(40, 5, "All Rights Reserved", TITLE_COLOR);
}

#define METER_FULL 40
void draw_meter(int y, int full)
{
	int i;

	center_string(21, y, "0%", MAIN_COLOR);
	center_string(30, y, "25%", MAIN_COLOR);
	center_string(40, y, "50%", MAIN_COLOR);
	center_string(50, y, "75%", MAIN_COLOR);
	center_string(60, y, "100%", MAIN_COLOR);
	for (i = 0; i <= METER_FULL; i++)
	{
		char attr = i < full ? color(WHITE, RED) : MAIN_COLOR;
		char c1 = ' ', c2 = 0xc4;
		if ((i % 10) == 0)
		{
			c1 = 0xb3;
			c2 = (i == 0) ? 0xc0 : (i == METER_FULL) ? 0xd9 : 0xc1;
		}
		pokechar(20+i, y+1, c1, attr);
		pokechar(20+i, y+2, c1, attr);
		pokechar(20+i, y+3, c2, MAIN_COLOR);
	}
}

char *progpath;
char progdir[PATH_LEN+2];
char srcdir[PATH_LEN+2];
char destdir[PATH_LEN+2];

void addslash(char *dir)
{
	int lastc;

	/* Make sure we can add a filename to it. */
	if (dir[0] == 0)
		return;
	lastc = dir[strlen(dir)-1];
	if ((lastc != ':') && (lastc != '\\'))
	{
		strcat(dir, "\\");
	}
}

void remslash(char *dir)
{
	int lastc;

	if (dir[0] == 0)
		return;
	if (dir[strlen(dir)-1] == '\\')
		dir[strlen(dir)-1] = 0;
}

int disknum;
unsigned disk_start_bytes;
unsigned disk_total_bytes;

void open_next_input()
{
	char srcname[PATH_LEN+20];
	struct stat st;

	disk_start_bytes = bytes_in;

	/* Build the source file name. */
	strcpy(srcname, srcdir);
	strcat(srcname, "iw.gz#");
	srcname[strlen(srcname)-1] = '0' + disknum;

	/* Open the input file. */
	while ((ifd = open(srcname, O_RDONLY)) < 0)
	{
		char str[40];
		sprintf(str, "Please insert installation disk %d",
			disknum);
		key_request(str, REQUEST_COLOR);
	}

	/* Find its size, for updating the progress bar. */
	if ((fstat(ifd, &st) < 0) || (st.st_size <= 0))
		read_error();
	disk_total_bytes = disk_start_bytes + st.st_size;

	disknum++;
}

void update_progress_bars()
{
	draw_meter(10, (bytes_in - disk_start_bytes) * (METER_FULL+1) /
			(disk_total_bytes - disk_start_bytes));

	draw_meter(16, bytes_in * (METER_FULL+1) / GZIP_FILE_SIZE);
}

extern int remove_ofname;

void write_txtfile(char *name, char *text)
{
	char cr_text[strlen(text)*2];
	char *s, *d;

	/* Write out the text file in DOS CR/LF format. */
	for (s = text, d = cr_text; *s; *d++ = *s++)
		if (*s == '\n')
			*d++ = '\r';

	strcpy(ofname, destdir);
	strcat(ofname, name);
	if ((ofd = open(ofname, O_WRONLY | O_CREAT | O_TRUNC, 0666)) < 0)
	{
		error_box("Unable to create '%s': %s", ofname, strerror(errno));
		ui_exit(1);
	}
	if (write(ofd, cr_text, d - cr_text) != (d - cr_text))
		write_error();
	close(ofd);
}

#include "readme.h"
#include "order.h"

void do_install()
{
	char destname[PATH_LEN+20];

	/* Open the first input file. */
	disknum = 1;
	disk_start_bytes = 0;
	if (ifd < 0)
		open_next_input();

	/* Open the output file. */
	strcpy(ofname, destdir);
	strcat(ofname, "iw.exe");
	if ((ofd = open(ofname, O_WRONLY | O_CREAT | O_TRUNC, 0777)) < 0)
	{
		error_box("Unable to create '%s': %s", ofname, strerror(errno));
		ui_exit(1);
	}
	remove_ofname = 1;

	method = get_method(ifd);
	if (method < 0) {
		/* error message already emitted */
		ui_exit(exit_code);
	}

	if (unzip(ifd, ofd) != OK)
		ui_exit(exit_code);

	close(ofd);

	/* Write out the text files. */
	write_txtfile("README.TXT", readme);
	write_txtfile("ORDER.TXT", order);

	key_request("Installation successful!", REQUEST_COLOR);
}

int copy_file(char *src, char *dest, size_t max_size)
{
	int ifd, ofd, actual;
	char buf[4096];

	while ((ifd = open(src, O_RDONLY)) < 0)
	{
		key_request("Please insert installation disk 1", REQUEST_COLOR);
	}
	if ((ofd = open(dest, O_WRONLY | O_CREAT | O_TRUNC, 0777)) < 0)
	{
		return -1;
	}

	while (1)
	{
		actual = read(ifd, buf, 4096);
		if (actual < 0)
			read_error();
		if (actual > max_size)
			actual = max_size;
		if (actual == 0)
			break;
		if (write(ofd, buf, actual) != actual)
			write_error();
		max_size -= actual;
	}

	close(ifd);
	close(ofd);

	return 0;
}

void copy_installer_to(char *destname)
{
	char dest[strlen(destdir)+strlen(destname)+1];

	strcpy(dest, destdir);
	strcat(dest, destname);

	if (copy_file(progpath, dest, INSTALL_SIZE))
	{
		error_box("Unable to create '%s': %s", ofname, strerror(errno));
		ui_exit(1);
	}
}

void copy_win95_stuff()
{
	extern char _binary_iw_pif_start[], _binary_iw_pif_end[];
	extern char _binary_iw_icl_start[], _binary_iw_icl_end[];
	int ofd, destdirlen = strlen(destdir);
	int pifsize = _binary_iw_pif_end - _binary_iw_pif_start;
	int iclsize = _binary_iw_icl_end - _binary_iw_icl_start;
	char destname[destdirlen+20];

	/* Substitute the proper dest directory in various places. */
	strcpy(_binary_iw_pif_start+0x24, destdir);
	strcat(_binary_iw_pif_start+0x24, "IW.EXE");
	strcpy(_binary_iw_pif_start+0x65, destdir);
	if (destdir[destdirlen-1] == '\\')
		_binary_iw_pif_start[0x65+destdirlen-1] = 0;
	strcpy(_binary_iw_pif_start+0x273, destdir);
	strcat(_binary_iw_pif_start+0x273, "IW.ICL");

	/* Try to create a PIF file in the appropriate directory.
	   If they're not using Win95, or it's not on the C drive,
	   they just won't get an automatic start menu item.  */
	if ((ofd = open(STARTMENUFILE,
			O_WRONLY | O_CREAT | O_TRUNC, 0777)) >= 0)
	{
		/* Write out the PIF file. */
		if (write(ofd, _binary_iw_pif_start, pifsize) != pifsize)
			write_error();
		close(ofd);
	}

	/* However, in any case, always create a PIF file in the destdir.
	   Since it will look like the only "real" icon in the directory,
	   hopefully people will be clued in that this is what to click.  */
	strcpy(destname, destdir);
	strcat(destname, "IW.PIF");
	if ((ofd = open(destname, O_WRONLY | O_CREAT | O_TRUNC, 0666)) >= 0)
	{
		/* Write out the PIF file. */
		if (write(ofd, _binary_iw_pif_start, pifsize) != pifsize)
			write_error();
		close(ofd);
	}

	/* Write the icon library file into the destdir. */
	strcpy(destname, destdir);
	strcat(destname, "IW.ICL");
	if ((ofd = open(destname, O_WRONLY | O_CREAT | O_TRUNC, 0666)) >= 0)
	{
		if (write(ofd, _binary_iw_icl_start, iclsize) != iclsize)
			write_error();
		close(ofd);
	}
}

void install()
{
	blank_screen();
	dline_box(5, 4, 70, 19, MAIN_COLOR);
	draw_title_box();

#if 0
{
	int i;

	for (i = 0; i < 16; i++)
		pokechar(i, 10, '*', i);
	for (i = 0; i < 16; i++)
		pokechar(i, 11, '*', i << 4);
	for (i = 0; i < 128; i++)
		pokechar(i%32, 12+i/32, i+128, 0x07);
}
#endif

	draw_meter(10, 0);
	center_string(40, 14, "Percent of Current Disk Completed", MAIN_COLOR);

	draw_meter(16, 0);
	center_string(40, 20, "Percent of Installation Completed", MAIN_COLOR);

	/* Default source directory is the program directory. */
	strcpy(srcdir, progdir);

	/* Check for an unzip gz file. */
	if ((ifd = open("iw.gz", O_RDONLY)) >= 0)
	{
		struct stat st;

		/* Find its size, for updating the progress bar. */
		if ((fstat(ifd, &st) < 0) || (st.st_size <= 0))
			read_error();
		disk_total_bytes = st.st_size;
	}
	else
	{
		/* Ask for the source pathname */
		string_request("Directory name to install from:", srcdir);
		addslash(srcdir);
	}

	/* Ask for the destination pathname */
	{
		strcpy(destdir, "C:\\IW");
		string_request("Directory to install into:", destdir);
		addslash(destdir);
	}

	/* Create the destination directory if necessary. */
	{
		char *slash = destdir;
		while (slash = strchr(slash, '\\'))
		{
			if ((slash > destdir) && (slash[-1] != ':'))
			{
				*slash = 0;
				mkdir(destdir, 0777);
				*slash = '\\';
			}
			slash++;
		}
	}

	copy_installer_to("SETUP.EXE");
	copy_installer_to("README.EXE");

	copy_win95_stuff();

	do_install();

	if (destdir[strlen(destdir)-1] == '\\')
		destdir[strlen(destdir)-1] = 0;
	chdir(destdir);	
}

void page_txtfile(char *title, char *text)
{
	struct screensavebuf sbuf;
	char *first_line = text;
	char *last_line;
	int done = 0;

	void up(int n)
	{
		while ((n-- > 0) && (first_line > text)) {
			do {
				first_line--;
			} while ((first_line > text) &&
				 (first_line[-1] != '\n'));
		}
	}

	void down(int n)
	{
		while ((n-- > 0) && (*last_line)) {
			while (*first_line++ != '\n');
			while (*last_line && *last_line++ != '\n');
		}
	}

	save_screen(&sbuf);

	dline_box(2, 0, 76, 24, PAGER_COLOR);
	center_string(40, 0, title, PAGER_COLOR);

	do
	{
		int x, y, c;
		char *p;

		/* Refresh the pager display. */
		last_line = first_line; x = y = 0;
		while (*last_line && y < 22)
		{
			switch (*last_line)
			{
			case '\n':
				for (; x < 74; x++)
					pokechar(3+x, 1+y, ' ', PAGER_COLOR);
				x = 0;
				y++;
				break;
			case '\t':
				do {
					pokechar(3+x, 1+y, ' ', PAGER_COLOR);
					x++;
				} while (x & 7);
				break;
			default:
				pokechar(3+x, 1+y, *last_line, PAGER_COLOR);
				x++;
				break;
			}
			last_line++;
		}

		c = bios_get_key();
		switch (c)
		{
		case 'H' << 8:
			up(1);
			break;
		case 'P' << 8:
			down(1);
			break;
		case 'G' << 8:
			up(100000);
			break;
		case 'O' << 8:
			down(100000);
			break;
		case 'I' << 8:
			up(20);
			break;
		case 'Q' << 8:
			down(20);
			break;
		default:
			c &= 0xff;
			switch (c)
			{
			case 27:
				done = 1;
				break;
			case '\r':
			case '\n':
				down(1);
				break;
			case ' ':
				down(20);
				break;
			}
		}
	}
	while (!done);

	restore_screen(&sbuf);
}

static int irq2opt[] = {2, 0, 0, 1, 1, 2, 2, 3, 3, 3, 4, 5, 6, 6, 6, 7};
static int opt2irq[] = {2, 3, 5, 7, 10, 11, 12, 15};

static int dma2opt[] = {0, 1, 1, 2, 2, 3, 4, 5};
static int opt2dma[] = {0, 1, 3, 5, 6, 7};

static int opt2mix[] = {8000, 11000, 16000, 22000};

#define message(foo) /* nothing */
#define WARNING(foo) /* nothing */
#define AUTODETECT_CARD 0

#define INSTALLER
#include "../meg/gen/config.h"
#include "../meg/gen/config.cc"

void sound_config()
{
	int sel;

	/* Card type */
	sel = option_request("Select Sound Card Type",
				configrec.sound_card_type, 
				"No sound card",
				"SoundBlaster",
				"SoundBlaster 2.0",
				"SoundBlaster 2.01+",
				"SoundBlaster Pro",
				"SoundBlaster 16",
				"Gravis UltraSound",
				0);
	if (sel < 0)
		return;
	configrec.sound_card_type = sel;
	if (configrec.sound_card_type == 0)
		return;

	/* I/O address */
	if (configrec.sound_card_addr < 0x210)
		configrec.sound_card_addr = 0x220;
	sel = option_request("Select I/O Address",
				configrec.sound_card_addr/0x10-0x21,
				"0x210", "0x220", "0x230", "0x240",
				"0x250", "0x260", 0);
	if (sel < 0)
		return;
	configrec.sound_card_addr = 0x210+sel*0x10;

	/* Interrupt */
	sel = option_request("Select Interrupt (IRQ)",
				irq2opt[configrec.sound_card_irq],
				"2", "3", "5", "7", "10", "11", "12", "15", 0);
	if (sel < 0)
		return;
	configrec.sound_card_irq = opt2irq[sel];

	/* 8-bit DMA channel */
	if (configrec.sound_card_type <= 5)
	{
		if (configrec.sound_card_dma8 < 0)
			configrec.sound_card_dma8 = 1;
		sel = option_request("Select DMA channel for 8-bit Sound",
					dma2opt[configrec.sound_card_dma8],
					"DMA channel 0",
					"DMA channel 1",
					"DMA channel 3",
					0);
		if (sel < 0)
			return;
		configrec.sound_card_dma8 = opt2dma[sel];
	}

	/* 16-bit DMA channel */
	if (configrec.sound_card_type >= 5)
	{
		if (configrec.sound_card_dma16 < 0)
			configrec.sound_card_dma16 = 5;
		sel = option_request("Select DMA channel for 16-bit Sound",
					dma2opt[configrec.sound_card_dma16],
					"DMA channel 0",
					"DMA channel 1",
					"DMA channel 3",
					"DMA channel 5",
					"DMA channel 6",
					"DMA channel 7",
					0);
		if (sel < 0)
			return;
		configrec.sound_card_dma16 = opt2dma[sel];
	}

	/* Sound quality */
	if (configrec.sound_card_type >= 5)
	{
		sel = option_request("Select Sound Quality",
					configrec.sound_16bit,
					"8-bit sound        ",
					"16-bit sound (best)",
					0);
		if (sel < 0)
			return;
		configrec.sound_16bit = sel;
	}
	else
		configrec.sound_16bit = 0;

	for (sel = 0; (configrec.sound_mixrate > opt2mix[sel]) && (sel < 3); sel++);
	sel = option_request("Select Mixing Rate", sel,
				"8000  (low quality)   ",
				"11000 (medium quality)",
				"16000 (high quality)  ",
				"22000 (best quality)  ",
				0);
	if (sel < 0)
		return;
	configrec.sound_mixrate = opt2mix[sel];
}

void find_config()
{
	char *str;

	/* First check for an already existing config file. */
	load_config();

	/* If there was none or it didn't specify a sound card,
	   try to determine one from the environment. */
	if (configrec.sound_card_type != 0)
		return;

	/* First check for appropriate environment variables
	   and use them as default settings. */
	str = getenv("BLASTER");
	if (str)
	{
		configrec.sound_card_type = 1;
		while (*str)
		{
			switch (*str)
			{
			case 'A':
			case 'a':
				sscanf(++str, "%x", &configrec.sound_card_addr);
				while (*str > ' ') str++;
				break;
			case 'I':
			case 'i':
				sscanf(++str, "%d", &configrec.sound_card_irq);
				while (*str > ' ') str++;
				break;
			case 'D':
			case 'd':
				sscanf(++str, "%d", &configrec.sound_card_dma8);
				while (*str > ' ') str++;
				break;
			case 'H':
			case 'h':
				sscanf(++str, "%d", &configrec.sound_card_dma16);
				while (*str > ' ') str++;
				configrec.sound_card_type = 5;
				break;
			default:
				str++;
			}
		}
	}
	str = getenv("ULTRASND");
	if (str)
	{
		configrec.sound_card_type = 6;
		sscanf(str, "%x,%d,%*d,%d,%*d",
			&configrec.sound_card_addr,
			&configrec.sound_card_dma16,
			&configrec.sound_card_irq);
	}
}

void page_readme()
{
	page_txtfile(" General Information (README.TXT) ", readme);
}

void setup_menu()
{
	int sel = 0;

	blank_screen();
	draw_title_box();

	find_config();

	while (1)
	{
		sel = option_request("Inner Worlds Setup", sel,
					"Information (README.TXT)",
					"Ordering    (ORDER.TXT) ",
					"Sound card configuration",
					"Exit                    ", 0);
		switch (sel)
		{
		case 0:
			page_readme();
			break;
		case 1:
			page_txtfile(" Ordering Information ", order);
			break;
		case 2:
			sound_config();
			break;
		case 3:
		case -1:
			save_config();
			return;
		}
	}
}

int main(int argc, char **argv)
{
	static char pname[PATH_LEN+1];
	int proglen;
	int devmemfd;
	caddr_t fbaddr;
	char *ra, *rb;
	int i;

	/* Save the full path to our own program. */
	progpath = argv[0];

	/* Compute the base program name without directory path or extension. */
	strncpy(progname = pname, basename(argv[0]), PATH_LEN);
	proglen = strlen(progname);
	if (proglen > 4 && (strcasecmp(progname+proglen-4, ".exe") == 0)) {
		progname[proglen-4] = '\0';
	}

	/* Find the directory name containing this program. */
	ra = strrchr(progpath, ':');
	rb = strrchr(progpath, '\\');
	if (rb && rb > ra) ra = rb;
	if (ra)
	{
		int pathlen = ra-progpath+1;
		strncpy(progdir, progpath, pathlen);
		progdir[pathlen] = 0;
	}
	else
	{
		strcpy(progdir, "A:\\");
		/* XXX adjust for current drive */
	}
	addslash(progdir);

	if (mlockall(MCL_CURRENT | MCL_FUTURE) < 0)
		error("can't lock down program memory");

	devmemfd = open("/dev/mem", O_RDWR);
	if (devmemfd < 0)
		error("can't open /dev/mem");
	fbaddr = mmap(0, 4096, PROT_READ|PROT_WRITE, MAP_FILE|MAP_SHARED,
			devmemfd, 0xb8000);
	if (fbaddr == (caddr_t)-1)
		error("can't map video buffer");
	vidbuf = (void*)fbaddr;

	if (strcasecmp(progname, "setup") == 0) {
		setup_menu();
	} else if (strcasecmp(progname, "readme") == 0) {
		blank_screen();
		page_readme();
	} else {
		install();
		setup_menu();
	}

	clear_screen();

	/* Tell the user how to start IW. */
	{
	char *iwdir = destdir[0] ? destdir : progdir;
	remslash(iwdir);
	printf("From DOS, type IW to run Inner Worlds.\r\n");
	printf("From Windows 95, select the wolf-head icon in %s.\r\n",
		(open(STARTMENUFILE, O_RDONLY) >= 0)
		? "Start/Programs" : iwdir);
	}

	return 0;
}

