
/* CD-ROM Manual Viewer
   by Timothy Cain     April 1994

   compiled with Microsoft C 8.0, using large model:
   compiled with Borland C 3.1, using large model:
      cl /AL view.c
      bcc /ml view.c

    $Header: /ManualDOS/VIEW.C 2     5/15/95 12:39p Patel $
*/

#include <conio.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* -------------------- USER CONFIGURATIONS ----------------- */

 // one of these two should be set to 1
#define READ_LBM   0   // 640x960x16 LBM's
#define READ_PCX   1   // 640x960x16 PCX's

#define FIRST_PAGE   1   // number of the first page of the manual

#define TITLE   "CD-ROM On-line Manual"


/* -------------------- private definitions ----------------- */

#define PLANE0   0x01
#define PLANE1   0x02
#define PLANE2   0x04
#define PLANE3   0x08

#define set_bit_plane(plane)   outp(0x03c4,2), outp(0x03c5,plane)

#define write_line(screen,data,plane,n)   set_bit_plane(plane), \
                                          _fmemcpy(screen,data,n)

#define MY_SIZE   5000
#define my_fopen(name,type)   (my_ptr=MY_SIZE,(my_fp = jfopen(name,type)))
#define my_fclose()   fclose(my_fp)

#define DISPLAY_START    20
#define DISPLAY_END      (480-PANEL_LENGTH)
#define DISPLAY_LENGTH   (DISPLAY_END-DISPLAY_START)

#define MIN_LINE      0
#define MAX_LINE      (Image_Height-DISPLAY_LENGTH)
#define LINE_LENGTH   16

#define I_NOTHING        -1
#define I_LINE_UP         0   // see also button_x for the x-coord of the
#define I_LINE_DOWN       1   // corresponding control panel button
#define I_LINE_FIRST      2
#define I_LINE_LAST       3
#define I_PAGE_UP         4
#define I_PAGE_DOWN       5
#define I_PAGE_FIRST      6
#define I_PAGE_LAST       7
#define I_QUIT            12

#define UP_ARROW        72+256
#define DOWN_ARROW      80+256
#define LEFT_ARROW      75+256
#define RIGHT_ARROW     77+256
#define RETURN          13
#define ESC             27
#define TAB             9
#define PAGE_UP         73+256
#define PAGE_DOWN       81+256
#define HOME            71+256
#define END             79+256
#define SPACE_BAR       32

#define F1              59+256
#define F2              60+256
#define F3              61+256
#define F4              62+256
#define F5              63+256
#define F6              64+256
#define F7              65+256
#define F8              66+256
#define F9              67+256
#define F10             68+256
#define F11             133+256
#define F12             134+256

#define PANEL_LENGTH   64
#define PANEL_SIZE     20480
#define MAX_BUTTONS    13
int button_x[MAX_BUTTONS] = {
    51, 55, 47, 59,             // line buttons
    30, 34, 26, 38,             // page buttons
    9, 13, 5, 17,               // chapter buttons
    73                          // quit button
};
#define BUTTON_Y          20
#define BUTTON_LENGTH     32
#define BUTTON_BUF_SIZE   10240

#define	LAST_PAGE	(Num_Pages+FIRST_PAGE-1)
#define IMAGE_BANK_SIZE 64000l

/* -------------------- private functions ------------------- */

int main(int argc, char *argv[]);
void alloc_page_mem(void);
void init(void);
void init_mode(void);
void grey_palette(void);
void set_palette(unsigned char far * pal);
void reset_mode(void);
void fetch_font(void);
void fetch_lbm(char *name);
void decode_lbm(unsigned char far * scr_ptr, unsigned int len);
void fetch_pcx(char *name);
void decode_pcx(unsigned char far * scr_ptr, unsigned int len);
unsigned char my_getc(void);
void view_manual(void);
void recalc_buttons(int *dim, int curr_line, int curr_page);
void fetch_page(int num);
void display_page(int start_line, int page);
int get_input(void);
void load_panel(void);
void display_panel(void);
void display_bar(void);
void press_button(int b);
void unpress_button(int b);
void dim_button(int b);
void draw_str(char *str, int column, int line);
long GetTime(void);
int new_tick(void);

int init_mouse(void);
void show_mouse(void);
void hide_mouse(void);
void get_mouse_info(void);

void read_inits(char *prg, char *data);
FILE *jfopen(char *name, char *mode);
int jstrncmp(char *a, char *b);
int blank_line(char *s);

/* -------------------- global variables -------------------- */

unsigned char *scr_data[32];

unsigned char far palette[64 * 3];

unsigned char far panel_buf[PANEL_SIZE];

unsigned char far button_buf[BUTTON_BUF_SIZE];

FILE *my_fp;
int my_ptr;
unsigned char my_data[MY_SIZE];

unsigned char far *scr_ptr = (unsigned char far *) 0xA0000000;
unsigned char far font_buf[256 * 16];
unsigned char far *font_ptr = font_buf;

int have_mouse, mouse_x, mouse_y, mouse_buttons;

char Data_Path[256];		// drive/dir of where data files are.
char File_Name[256];		// handy global name of last file opened.
char *Page_Names[256];		// name of each page.
int Num_Pages = 0, Image_Height = 0;
long Image_Mem, Image_Num_Banks;

/* ---------------------------------------------------------- */

int main(int argc, char *argv[])
{
	if(argc != 2)
	{
		printf("usage: %s data_dir\n", argv[0]);
		exit(1);
	}
	read_inits(argv[0], argv[1]);

    alloc_page_mem();
    init();
    view_manual();

    return (0);
}

void read_inits(char *prg, char *data)
{
	int i, len, page;
	FILE *fp;

	// find trailing slash.
	for(i = len = 0; prg[i]; ++i)
	{
		if(prg[i] == '\\')
			len = i;
	}
	// create path.
	sprintf(Data_Path, "%.*s\\%s\\", len, prg, data);

	// read args from settings file.
	page = 0;
	for(i = 0; i != 256; ++i) Page_Names[i] = "";
	fp = jfopen("settings.ini", "rt");
	while(fgets(File_Name, 256, fp) != 0)
	{
		// read variable settings from file.
		if(jstrncmp(File_Name, "NUM_PAGES") == 0)
			sscanf(File_Name, "NUM_PAGES = %d", &Num_Pages);
		else if(jstrncmp(File_Name, "IMAGE_HEIGHT") == 0)
			sscanf(File_Name, "IMAGE_HEIGHT = %d", &Image_Height);
		else if(File_Name[0] == '#' || blank_line(File_Name))	// comment/blank line
			;
		// anything else is a page name.
		else
		{
			File_Name[strlen(File_Name) - 1] = 0;
			if(page < Num_Pages)
				Page_Names[page++] = strdup(File_Name);
			else
			{
				printf("More page names than pages\n");
				exit(1);
			}
		}
	}
	fclose(fp);
	Image_Mem = 320l * Image_Height;
	Image_Num_Banks = (Image_Mem + IMAGE_BANK_SIZE - 1) / IMAGE_BANK_SIZE;
}

void alloc_page_mem(void)
{
    int i;
    long mem_needed = Image_Mem;

    for (i = 0; i != Image_Num_Banks; ++i)
    {
        long l;

        l = mem_needed <= IMAGE_BANK_SIZE ? mem_needed : IMAGE_BANK_SIZE;

        scr_data[i] = malloc(l);
        if (scr_data[i] == NULL)
        {
            printf("Insufficient conventional memory for manual pages.\n");
            exit(1);
        }
        mem_needed -= l;
    }
}

void init()
{

    init_mode();
    fetch_font();
//   grey_palette();
    load_panel();
    set_palette(palette);
    display_panel();
    display_bar();
    have_mouse = init_mouse();
    show_mouse();
    atexit(reset_mode);
}

void init_mode()
{
	_asm {
		mov ax, 0012h
		int 10h                // set video mode to 12h
		 mov dx, 03ceh
		 mov al, 5
		 out dx, al
		 inc dx
		 mov al, 0
		 out dx, al             // set Write Mode 0
		 mov dx, 03ceh
		 mov al, 08h
		 out dx, al
		 inc dx
		 mov al, 0FFh
         out dx, al             // set Bit Mask register to all bits
    }
}

/* set the 16 color palette to greyscale.
   Note the order of the EGA palette registers:
    EGA_BLACK		 =  0,
    EGA_BLUE		 =  1,
    EGA_GREEN		 =  2,
    EGA_CYAN		 =  3,
    EGA_RED		 =  4,
    EGA_MAGENTA 	 =  5,
    EGA_BROWN		 =  20,
    EGA_LIGHTGRAY	 =  7,
    EGA_DARKGRAY	 =  56,
    EGA_LIGHTBLUE	 =  57,
    EGA_LIGHTGREEN	 =  58,
    EGA_LIGHTCYAN	 =  59,
    EGA_LIGHTRED	 =  60,
    EGA_LIGHTMAGENTA	 =  61,
    EGA_YELLOW		 =  62,
    EGA_WHITE		 =  63
*/

void grey_palette()
{
    int i;
    unsigned char pal[64 * 3], c = 0;

    for (i = 0; i < 8; i++, c += 4)
    {
        pal[i * 3 + 0] = c;
        pal[i * 3 + 1] = c;
        pal[i * 3 + 2] = c;
    }
    pal[20 * 3 + 0] = 24;
    pal[20 * 3 + 1] = 24;
    pal[20 * 3 + 2] = 24;
    for (i = 56; i < 64; i++, c += 4)
    {
        pal[i * 3 + 0] = c;
        pal[i * 3 + 1] = c;
        pal[i * 3 + 2] = c;
    }
    set_palette(pal);
}

void set_palette(unsigned char far * pal)
{
	_asm {
		mov ax, 1012h
		mov bx, 0               // start at palette register 0
		mov cx, 64              // and write 64 colors
		les dx,[pal]            // that are in pal
		int 10h
    }
}

void reset_mode()
{
	_asm {
		mov ax, 0003h
		int 10h
    }
}

void fetch_font()
{
    FILE *fp;

	sprintf(File_Name, "%sfont.fon", Data_Path);
    if ((fp = fopen(File_Name, "rb")) == NULL)
        font_ptr = (unsigned char far *) *((long far *) 0x0000010C);    // ROM font
    else
    {
        fread(font_ptr, 256 * 16, 1, fp);
        fclose(fp);
    }
}

#if READ_LBM

void fetch_lbm(char *name)
{
    int i, y, t[4] = {0, 0, 0, 0};

    if (my_fopen(name, "rb") == NULL)
        return;
    do
    {
        t[0] = t[1];
        t[1] = t[2];
        t[2] = t[3];
        t[3] = my_getc();
    } while ((t[0] != 'B') || (t[1] != 'O') || (t[2] != 'D') || (t[3] != 'Y'));
    for (y = 0; y < 4; y++)
        my_getc();

    for (i = y = 0; i != Image_Num_Banks; y += 200, ++i)
    {
        int j;

        j = min(200, Image_Height - y);
        decode_lbm(scr_data[i], j);
    }
    my_fclose();
}

#endif

/* decode len lines of RLE data into the buffer scr_ptr */

void decode_lbm(unsigned char far * scr_ptr, unsigned int len)
{
    unsigned int count = 0;
    unsigned char i, size, c;

    len *= 320;                 // 4 bit planes per line, 80 bytes per plane
    while (count < len)
    {
        size = my_getc();
        /* if size is positive then (size+1) uncompressed bytes follow */
        /* if size is negative then repeat the subsequent byte (100h-size)+1 times */
        if (size < 0x80)
        {                       /* size is positive */
            for (i = 0; (i <= size) && (count < len); i++)
            {
                scr_ptr[count++] = my_getc();
            }
        }
        else
        {
            c = my_getc();
            size = (unsigned char) (0x100 - size);
            for (i = 0; (i <= size) && (count < len); i++)
                scr_ptr[count++] = c;
        }
    }
}

#if READ_PCX

void fetch_pcx(char *name)
{
    int i, y;

    if (my_fopen(name, "rb") == NULL)
        return;
    for (y = 0; y < 128; y++)   // skip header
        my_getc();
    for (i = y = 0; i != Image_Num_Banks; y += 200, ++i)
    {
        int j;

        j = min(200, Image_Height - y);
        decode_pcx(scr_data[i], j);
    }
    my_fclose();
}

/* decode len lines of RLE data into the buffer scr_ptr */

void decode_pcx(unsigned char far * scr_ptr, unsigned int len)
{
    unsigned int count = 0;
    unsigned char i, size, c;

    len *= 320;                 // 4 bit planes per line, 80 bytes per plane
    while (count < len)
    {
        size = my_getc();
        /* if size <= 192 next byte is uncompressed */
        /* if size > 192 then repeat the subsequent byte (size-192) times */
        if (size <= 192)
            scr_ptr[count++] = size;
        else
        {
            c = (unsigned char) (size - 192);
            size = my_getc();
            for (i = 0; (i < c) && (count < len); i++)
                scr_ptr[count++] = size;
        }
    }
}

#endif

unsigned char my_getc()
{
    if (my_ptr == MY_SIZE)
    {
        my_ptr = 0;
        fread(my_data, MY_SIZE, 1, my_fp);
    }

    return (my_data[my_ptr++]);
}

void view_manual()
{
	int i, b, curr_line = MIN_LINE, curr_page = FIRST_PAGE;
	int dim[MAX_BUTTONS];

    for (i = 0; i < MAX_BUTTONS; i++)
        dim[i] = 0;
    fetch_page(curr_page);
    display_page(curr_line, curr_page);
    draw_str(TITLE, 1, 2);
    recalc_buttons(dim, curr_line, curr_page);
    for (;;)
    {
        i = get_input();
        if (i == I_NOTHING)
            continue;
        if (dim[i])
            continue;
        b = i;
        press_button(b);
        switch (i)
        {
            case I_QUIT:
            return;
            case I_LINE_UP:
            curr_line -= LINE_LENGTH;
            if (curr_line < MIN_LINE)
                curr_line = MIN_LINE;
            display_page(curr_line, curr_page);
            break;
            case I_LINE_DOWN:
            curr_line += LINE_LENGTH;
            if (curr_line > MAX_LINE)
                curr_line = MAX_LINE;
			display_page(curr_line, curr_page);
            break;
            case I_LINE_FIRST:
            curr_line -= DISPLAY_LENGTH;
            if (curr_line < MIN_LINE)
                curr_line = MIN_LINE;
			display_page(curr_line, curr_page);
            break;
            case I_LINE_LAST:
            curr_line += DISPLAY_LENGTH;
            if (curr_line > MAX_LINE)
                curr_line = MAX_LINE;
			display_page(curr_line, curr_page);
            break;
            case I_PAGE_UP:
            if (curr_page == FIRST_PAGE)
                break;
            curr_page--;
            fetch_page(curr_page);
            curr_line = MIN_LINE;
			display_page(curr_line, curr_page);
            break;
            case I_PAGE_DOWN:
            if (curr_page == LAST_PAGE)
                break;
            curr_page++;
            fetch_page(curr_page);
            curr_line = MIN_LINE;
			display_page(curr_line, curr_page);
            break;
            case I_PAGE_FIRST:
            if (curr_page == FIRST_PAGE)
                break;
            curr_page = FIRST_PAGE;
            fetch_page(curr_page);
            curr_line = MIN_LINE;
			display_page(curr_line, curr_page);
            break;
            case I_PAGE_LAST:
            if (curr_page == LAST_PAGE)
                break;
            curr_page = LAST_PAGE;
            fetch_page(curr_page);
            curr_line = MIN_LINE;
			display_page(curr_line, curr_page);
            break;
        }
        unpress_button(b);
		recalc_buttons(dim, curr_line, curr_page);
    }
}

void recalc_buttons(int *dim, int curr_line, int curr_page)
{
    int i, new_dim[MAX_BUTTONS];

    for (i = 0; i < MAX_BUTTONS; i++)
        new_dim[i] = 0;
    if (curr_line == MIN_LINE)
    {
        new_dim[I_LINE_UP] = 1;
        new_dim[I_LINE_FIRST] = 1;
    }
    if (curr_line == MAX_LINE)
    {
        new_dim[I_LINE_DOWN] = 1;
        new_dim[I_LINE_LAST] = 1;
    }
    if (curr_page == FIRST_PAGE)
    {
        new_dim[I_PAGE_UP] = 1;
        new_dim[I_PAGE_FIRST] = 1;
    }
    if (curr_page == LAST_PAGE)
    {
        new_dim[I_PAGE_DOWN] = 1;
        new_dim[I_PAGE_LAST] = 1;
    }
    for (i = 0; i < MAX_BUTTONS; i++)
        if (new_dim[i] != dim[i])
        {
            if (new_dim[i])
                dim_button(i);
            else
                unpress_button(i);
            dim[i] = new_dim[i];
        }
}

void fetch_page(int num)
{
    char str[40];

#if READ_LBM

    sprintf(str, "%03d.lbm", num);
    fetch_lbm(str);
#endif
#if READ_PCX
    sprintf(str, "%03d.pcx", num);
    fetch_pcx(str);
#endif
}

void display_page(int start_line, int page)
{
    int bank = -1, line, bank_switch;
    unsigned char far *d_ptr, *s_ptr = scr_ptr + (DISPLAY_START * 80);
    char str[80];

    sprintf(str, "Page %s", Page_Names[page - FIRST_PAGE]);
    draw_str(str, 79 - strlen(str), 2);
    hide_mouse();
    bank = start_line / 200;
    bank_switch = start_line % 200;
    d_ptr = scr_data[bank] + (start_line % 200) * 320;
    for (line = start_line; line < start_line + DISPLAY_LENGTH; line++, s_ptr += 80, bank_switch++)
    {
        if (bank_switch == 200)
        {
            bank = line / 200;
            bank_switch = 0;
            d_ptr = scr_data[bank];
        }
        write_line(s_ptr, d_ptr, PLANE0, 80);
        d_ptr += 80;
        write_line(s_ptr, d_ptr, PLANE1, 80);
        d_ptr += 80;
        write_line(s_ptr, d_ptr, PLANE2, 80);
        d_ptr += 80;
        write_line(s_ptr, d_ptr, PLANE3, 80);
        d_ptr += 80;
    }
    show_mouse();
}

int get_input()
{
    int i, retval = I_NOTHING;

    if (kbhit())
    {
        i = getch();
        if ((i == 0) || (i == 0xE0))
            i = getch() + 256;
        switch (i)
        {
            case F6:
            case UP_ARROW:
            case 'u':
            retval = I_LINE_UP;
            break;
            case F7:
            case DOWN_ARROW:
            case RETURN:
            case 'd':
            retval = I_LINE_DOWN;
            break;
            case ESC:
            case 'q':
            retval = I_QUIT;
            break;
            case F5:
            case PAGE_UP:
            retval = I_LINE_FIRST;
            break;
            case F8:
            case PAGE_DOWN:
            retval = I_LINE_LAST;
            break;
            case LEFT_ARROW:
            case F2:
            case 'p':
            retval = I_PAGE_UP;
            break;
            case RIGHT_ARROW:
            case F3:
            case SPACE_BAR:
            case 'n':
            retval = I_PAGE_DOWN;
            break;
            case F1:
            case HOME:
            case 'f':
            retval = I_PAGE_FIRST;
            break;
            case F4:
            case END:
            case 'l':
            retval = I_PAGE_LAST;
            break;
        }
    }
    else if (new_tick())
    {
        get_mouse_info();
        if ((mouse_buttons) && (mouse_y >= DISPLAY_END + BUTTON_Y) &&
            (mouse_y <= DISPLAY_END + BUTTON_Y + 32))
        {
            for (i = 0; i < MAX_BUTTONS; i++)
                if ((mouse_x >= (button_x[i] << 3)) && (mouse_x <= (button_x[i] << 3) + 32))
                    break;
            if (i < MAX_BUTTONS)
                retval = i;
        }
    }

    return (retval);
}

void load_panel()
{
    int y, t[4] = {0, 0, 0, 0};

    if (my_fopen("control.lbm", "rb") == NULL)
        return;
    do
    {
        t[0] = t[1];
        t[1] = t[2];
        t[2] = t[3];
        t[3] = my_getc();
    } while ((t[0] != 'C') || (t[1] != 'M') || (t[2] != 'A') || (t[3] != 'P'));
    for (y = 0; y < 4; y++)
        my_getc();
    for (y = 0; y < 16 * 3; y++)
        palette[y] = (unsigned char) (my_getc() >> 2);
    palette[20 * 3 + 0] = palette[6 * 3 + 0];
    palette[20 * 3 + 1] = palette[6 * 3 + 1];
    palette[20 * 3 + 2] = palette[6 * 3 + 2];
    for (y = 56 * 3; y < 64 * 3; y++)
        palette[y] = palette[y - 48 * 3];
    do
    {
        t[0] = t[1];
        t[1] = t[2];
        t[2] = t[3];
        t[3] = my_getc();
    } while ((t[0] != 'B') || (t[1] != 'O') || (t[2] != 'D') || (t[3] != 'Y'));
    for (y = 0; y < 4; y++)
        my_getc();
    decode_lbm(panel_buf, PANEL_LENGTH);
    decode_lbm(button_buf, BUTTON_LENGTH);
    my_fclose();
}

void display_panel()
{
    int y;
    unsigned char far *d_ptr = panel_buf, *s_ptr = scr_ptr + (DISPLAY_END * 80L);

    for (y = 0; y < PANEL_LENGTH; y++, s_ptr += 80)
    {
        write_line(s_ptr, d_ptr, PLANE0, 80);
        d_ptr += 80;
        write_line(s_ptr, d_ptr, PLANE1, 80);
        d_ptr += 80;
        write_line(s_ptr, d_ptr, PLANE2, 80);
        d_ptr += 80;
        write_line(s_ptr, d_ptr, PLANE3, 80);
        d_ptr += 80;
    }
}

void display_bar()
{
    int y;
    unsigned char d_ptr[80], *s_ptr = scr_ptr;

    for (y = 0; y < 80; y++)
        d_ptr[y] = 0xFF;
    for (y = 0; y < DISPLAY_START; y++, s_ptr += 80)
    {
        write_line(s_ptr, d_ptr, PLANE0, 80);
        write_line(s_ptr, d_ptr, PLANE1, 80);
        write_line(s_ptr, d_ptr, PLANE2, 80);
        write_line(s_ptr, d_ptr, PLANE3, 80);
    }
}

void press_button(int b)
{
    int i;
    unsigned char far *d_ptr = button_buf + button_x[b], *s_ptr = scr_ptr + (DISPLAY_END + BUTTON_Y) * 80L + button_x[b];

    hide_mouse();
    for (i = 0; i < BUTTON_LENGTH; i++, s_ptr += 80)
    {
        write_line(s_ptr, d_ptr, PLANE0, 4);
        d_ptr += 80;
        write_line(s_ptr, d_ptr, PLANE1, 4);
        d_ptr += 80;
        write_line(s_ptr, d_ptr, PLANE2, 4);
        d_ptr += 80;
        write_line(s_ptr, d_ptr, PLANE3, 4);
        d_ptr += 80;
    }
    show_mouse();
}

void unpress_button(int b)
{
    int i;
    unsigned char far *d_ptr = panel_buf + BUTTON_Y * 320 + button_x[b], *s_ptr = scr_ptr + (DISPLAY_END + BUTTON_Y) * 80L + button_x[b];

    hide_mouse();
    for (i = 0; i < BUTTON_LENGTH; i++, s_ptr += 80)
    {
        write_line(s_ptr, d_ptr, PLANE0, 4);
        d_ptr += 80;
        write_line(s_ptr, d_ptr, PLANE1, 4);
        d_ptr += 80;
        write_line(s_ptr, d_ptr, PLANE2, 4);
        d_ptr += 80;
        write_line(s_ptr, d_ptr, PLANE3, 4);
        d_ptr += 80;
    }
    show_mouse();
}

void dim_button(int b)
{
    int i;
	unsigned char far *d_ptr = panel_buf + BUTTON_Y * 320 + button_x[b],
					  *s_ptr = scr_ptr + (DISPLAY_END + BUTTON_Y) * 80L + button_x[b],
					  white[4] = {0xFF, 0xFF, 0xFF, 0xFF};

    hide_mouse();
    for (i = 0; i < BUTTON_LENGTH; i++, s_ptr += 80)
    {
        write_line(s_ptr, white, PLANE0, 4);
        d_ptr += 80;
        write_line(s_ptr, d_ptr, PLANE1, 4);
        d_ptr += 80;
        write_line(s_ptr, d_ptr, PLANE2, 4);
        d_ptr += 80;
        write_line(s_ptr, white, PLANE3, 4);
        d_ptr += 80;
    }
    show_mouse();
}

void draw_str(char *str, int column, int line)
{
    int i, j, k;
    unsigned char far *s_ptr;
    unsigned char ch;

    hide_mouse();
    s_ptr = scr_ptr + line * 80L + column;
    for (i = 0; str[i] != 0; i++, s_ptr++)
    {
        k = str[i];
        k <<= 4;
        for (j = 0; j < 16; j++, s_ptr += 80, k++)
        {
            ch = (unsigned char) ~(font_ptr[k]);
//         write_line(s_ptr,font_ptr+k,PLANE0,1);
//         write_line(s_ptr,font_ptr+k,PLANE1,1);
//         write_line(s_ptr,font_ptr+k,PLANE2,1);
            write_line(s_ptr, &ch, PLANE3, 1);
        }
        s_ptr -= 1280;
    }
    show_mouse();
}

long GetTime()
{
    long temp;

	_asm {
		xor ah, ah
		int 1ah
		mov word ptr[temp], dx
		mov word ptr[temp + 2], cx
    }

     return (temp);
}

int new_tick()
{
    static long last;
    long temp;

    temp = GetTime();
    if ((last == temp) || (last == temp - 1))   // wait for 2 ticks
        return (0);
    last = temp;
    return (1);
}

// The following mouse routines are a bare-bones Microsoft mouse
// handler. Mea culpa.

int init_mouse()
{
    int retval;

	_asm {
		mov ax, 0
		int 33h
		mov[retval], ax
    }

     return (retval);
}

void show_mouse()
{
    if (have_mouse == -1)
		_asm {
			mov ax, 1
			int 33h
		}
}

void hide_mouse()
{
	if (have_mouse == -1)
		_asm {
			mov ax, 2
			int 33h
		}
}

void get_mouse_info()
{
	int x, y, b;

	if (have_mouse == -1)
	{
		_asm {
			mov ax, 3
			int 33h
			mov[b], bx
			mov[x], cx
			mov[y], dx
		}
		mouse_x = x;
		mouse_y = y;
		mouse_buttons = b;
	}
}

FILE *jfopen(char *name, char *mode)
{
	FILE *fp;

	sprintf(File_Name, "%s%s", Data_Path, name);
	if((fp = fopen(File_Name, mode)) == 0)
	{
		perror(File_Name);
		exit(1);
	}
	return fp;
}

int jstrncmp(char *a, char *b)
{
	while(*a && *b && (*a == *b))
		++a, ++b;

	if(!*a || !*b) return 0;
	else return *a - *b;
}

int blank_line(char *s)
{
	for( ; *s; ++s)
	{
		if(!isspace(*s))
			return 0;
	}
	return 1;
}

