; Function to move sprite data to screen location masking out zero pixles
; Mode X (320x240, 256 colors). Uses Linear bitmaps, copying one plane at
; a time. First two bytes must be Ydim and Xdim of sprite.
; Xdim must be multiple of 4
; PageOffset is a modulo 4 location anywhere in video ram
;  C near-callable as:
;    void CopyMSpriteToDisplay(char* SourcePtr, unsigned int PageOffset,
;                              unsigned int Yloc, unsigned int Xloc)

SC_INDEX equ    03c4h   ;Sequence Controller Index register port
MAP_MASK equ    02h     ;index in SC of Map Mask register
SCREEN_SEG equ  0a000h  ;segment of display memory in mode X
LAST_LINE equ   304     ;bottom of video page - lines below this are ignored

parms   struc
        dw      2 dup (?) ;pushed BP and return address
SourcePtr    dw ?       ; pointer in DS to start of bitmap in which
                        ; source resides
PageOffset   dw ?       ; base offset in display memory of video page
Yloc         dw ?       ; Y offset (0-303) in video page of
                        ; sprite upper left corner
Xloc         dw ?       ; X offset (0-383) in video page of
                        ; sprite upper left corner
parms   ends


; Local Variables

Ydim        equ -1   ;local storage for distance from end of
                              ; one source scan line to start of next
Xdim        equ -3   ;local storage for distance from end of
                              ; one dest scan line to start of next
ScanLine    equ -5  ;local storage for distance from end of sprite line
                             ; to start of next scanline
NumLines    equ -7   ; local storage for number of y lines before clipping
STACK_FRAME_SIZE equ 8


        .model  small
        .code
        public  _CopyMSpriteToDisplay
_CopyMSpriteToDisplay proc    near
        push    bp      ;preserve caller's stack frame
        mov     bp,sp   ;point to local stack frame
        sub     sp,STACK_FRAME_SIZE ;allocate space for local vars
        push    si      ;preserve caller's register variables
        push    di

        cld
        mov     ax,SCREEN_SEG     ;point ES to display memory
        mov     es,ax


        mov     dx,SC_INDEX     ;point to SC Index register
        mov     al,MAP_MASK
        out     dx,al           ;point SC Index reg to the Map Mask
        inc     dx              ;point DX to SC Data reg

        mov    cx,[bp+SourcePtr] ;offset of first source rect pixel
        mov    si,cx             ; in DS
        lodsb
        mov    [bp+Ydim], al
        mov    [bp+NumLines] ,0  ; clear extra lines counter

                             ; BOTTOM CLIPPING
        xor    ah, ah            ; clear ah
        add    ax, [bp+Yloc]     ; add y dim to y location to check for clip
        cmp    ax, LAST_LINE     ; see if sprite overruns page
        jle    NoClip            ; jump over clipping if less than page end
        sub    ax, LAST_LINE     ; get overrun amount in al
        mov    ah,[bp+Ydim]      ; get ydim in ah
        sub    ah, al            ; subtract overrun from ydim
        mov    [bp+Ydim], ah     ; store this value
        mov    [bp+NumLines],al  ; store the number of missed lines

NoClip:
        lodsb
        shr    al, 2             ; divide x -dim by 4
        mov    [bp+Xdim], al     ; get and store sprite dimentions
        mov    ah, 96
        sub    ah, al
        mov    al, ah
        xor    ah, ah
        mov   [bp+ScanLine], ax    ; store distance to next scanline

        mov    ax, [bp+PageOffset]
        mov    bx, [bp+Yloc]
        shl    bx, 5
        add    ax, bx           ; YLoc * 32 added to PageOffset
        shl    bx, 1
        add    ax, bx           ; Yloc * 64 added to PageOffset

        mov    bx, [bp+Xloc]
        mov    cl, bl
        shr    bx, 2            ; Divide Xloc by 2  to get video mem address
        add    ax, bx           ; add to PageOffset
        mov    [bp+PageOffset], ax    ; store dest address

        and    cl, 011b         ; find dest offset modulo 4
        mov    al, 1
        shl    al, cl           ; set mask to first plane
        mov    bh, al
        xor    cx, cx
        push   cx
CopyPlane:
       mov     di,[bp+PageOffset] ;offset of first dest pixel
                                ; in display memory
       mov     bl, [bp+Ydim]    ; number of rows to copy
       mov     al, bh
       out     dx,al            ; set the plane for this pixel
CopyScanLine:
       mov     cl, [bp+Xdim]    ; number of bytes in scan line to copy
CopyLoop:
       lodsb                    ; load the next pixel
       or      al, al
       jz      BlankPixel       ; Do not copy if zero
       stosb                    ; copy byte to display
       jmp     CopiedByte
BlankPixel:
       inc     di               ; increment dest if not copied
CopiedByte:
       dec     cl
       jnz     CopyLoop         ; keep going until end of scanline
       add     di, [bp+ScanLine]; increment destination to next line
       dec     bl
       jnz    CopyScanLine
       rol     bh,1            ; set mask for next pixel's plane
       cmp     bh, 16
       jne     NoBoundry
       mov     bh, 1
       add     [bp+PageOffset], 1  ; inc dest due to plane crossing
NoBoundry:
       mov     al, [bp+NumLines]
       cmp     al, 0               ; see if any lines were skipped
       je      NoLinesSkipped
SkipLoop:
       mov     cl, [bp+Xdim]    ; number of bytes in scan line to copy
SkipScanLine:
       inc     si
       dec     cl
       jnz     SkipScanLine     ; move source pointer over scan line
       dec     al
       jnz     SkipLoop         ; move source pointer over skipped rows

NoLinesSkipped:
       pop     cx
       inc     cx
       push    cx
       cmp     cx, 4
       jl      CopyPlane

       pop     ax
       pop     di      ;restore caller's register variables
       pop     si
       mov     sp,bp   ;discard storage for local variables
       pop     bp      ;restore caller's stack frame
       ret
_CopyMSpriteToDisplay endp
	end
