/*
  Program:      Mine Mayhem
  File:         board.cc
  Date:         July, 1996 to February, 1997

  Everything to do with the board.

  Freeware, Copyright 1996-7 Jason Hood.

  You are free to use this code, or a portion thereof, as long as an
  appropriate acknowledgement is made.
*/

#include "board.h"
#include "options.h"
#include "custom.h"
#include "times.h"
#include "faces.h"
#include "squares.h"
#include <stdio.h>

#include <stdlib.h>

#define ID_FACE 1000

DEFINE_RESPONSE_TABLE(FaceButton, BasicButton)
  E_RBUTTONDOWN
END_RESPONSE_TABLE


// Change the icon and display it as well.
void FaceButton::Display(int Icon)
{
  ChangeIcon(Icon);
  Paint();
  RefreshWindow();
}


DEFINE_RESPONSE_TABLE(BoardWindow, Window)
  E_BUTTONUP(ID_FACE, cmNewGame)
  E_BUTTONR(ID_FACE, cmPause)
  E_LBUTTONDOWN
  E_RBUTTONDOWN
  E_MBUTTONDOWN
  E_LBUTTONUP
  E_RBUTTONUP
  E_MBUTTONUP
  E_TIMER
  E_KEYCODE(0x70, 0x19, cmPause);       // P
END_RESPONSE_TABLE

#define  Value(x, y) grid[(y)][(x)].value
#define Status(x, y) grid[(y)][(x)].status


/* ----------------------   BoardWindow Constructor   -----------------------

  Create the board window, set up the grid array with the maximum size and
  start a new game at level Level.

*/

BoardWindow::BoardWindow(int Level) :
  Window(NULL, NULL, (ws.GetDeskWidth() - (FACEICONW + 4*2 + FWIDTH*2))/2,
                     SysFont->height + FWIDTH*2 + 4 + 4,
                     (ws.GetDeskWidth() + (FACEICONW + 4*2 + FWIDTH*2))/2 - 1,
                     SysFont->height + FWIDTH*2 + 4 + 4 +
                      4 + DIGITH + 4 + FACEICONH + FWIDTH*2 - 1,
         WA_VISABLE | WA_SAVEAREA | WA_BORDER),
  level(0), isHiLite(FALSE), isPaused(FALSE)
{
  Face = new FaceButton(this, ID_FACE, FWIDTH+4, FWIDTH+4+DIGITH+4, FACESMILEY);
  grid = new Sq*[MaxGridY];
  for (int j = 0; j < MaxGridY; j++)
    grid[j] = new Sq[MaxGridX];
  NewGame(Level);
}


BoardWindow::~BoardWindow()
{
  delete Face;
  for (int j = MaxGridY-1; j >= 0; j--)
    delete[] grid[j];
  delete[] grid;
}


/* -------------------------   Mine- & SpaceCount   -------------------------

  Display the number of mines marked and the number of spaces to uncover.

*/

inline void BoardWindow::MineCount()
{
  Display(mines, 4, 4+DIGITH+4);
}

inline void BoardWindow::SpaceCount()
{
  Display(spaces, spacesx, 4+DIGITH+4);
}


/* -------------------------------   NewGame   ------------------------------

  Start a new game with level Level. If Level is zero then play the same level
  that was played before. If Level is invalid then play Beginner.

*/

void BoardWindow::NewGame(int Level)
{
  ws.StopTimer(this);
  if (isPaused)
  {
    isPaused = FALSE;
    DeleteBuffer(gridsave);
  }
  Face->Display(FACESMILEY);
  switch (Level)
  {
    case 0: if (level) break; else Level = BEGINNER;    // Fall through
    case BEGINNER:     layout = Beginner;     break;
    case INTERMEDIATE: layout = Intermediate; break;
    case EXPERT:       layout = Expert;       break;
    case CUSTOM: if (level == 0 || CustomDlg().Run())
                 {
                   layout = Custom;
                        if (layout == Beginner)     Level = BEGINNER;
                   else if (layout == Intermediate) Level = INTERMEDIATE;
                   else if (layout == Expert)       Level = EXPERT;
                 }
                 else Level = level;
                 break;
    default:
      if (Level < 2 || Level > CUSTOM)
      {
        Level = BEGINNER; layout = Beginner;
      }
      else
      {
        if (level)
        { // Calculate the dimensions to satisfy the difficulty
          int area;
          do
          {
            Random.width  = random()%(MaxGridX-MinGridX+1) + MinGridX;
            Random.height = random()%(MaxGridY-MinGridY+1) + MinGridY;
            area = Random.width * Random.height;
          }
          while (area % Level != 0);
          Random.mines = area / Level;
        }
        layout = Random;
      }
  }
  mines = mynes = layout.mines;
  spaces = layout.width*layout.height - mines;
  isStarted = isFinished = FALSE;
  secs = mils = 0;
  if (Level != 0 && (level != Level || Level == CUSTOM || Level < BEGINNER))
  { // The grid size has been changed so redraw the board window
    level = Level;
    Resize(layout.width, layout.height);
  }
  for (int y = 0; y < layout.height; y++)
  {
    for (int x = 0; x < layout.width; x++)
    {
      Value(x, y) = 0;
      Status(x, y) = 0;
      DisplaySquare(x, y, UNOPENED, FALSE);
    }
  }
  for (int j = 0; j < layout.mines; j++) PlaceMine();
  MineCount(); SpaceCount();
  DisplayTime(TRUE);
  RefreshWindow();
}


// Start a new game from the face button.
void BoardWindow::cmNewGame()
{
  NewGame(level);
}


// Set the board's size and position according to the dimensions.
void BoardWindow::Resize(int width, int height)
{
  //            grid         frame   margin outline
  int wid = width*SQUAREW + FWIDTH*2 + 4*2 + 2,
  //         <-------------menu----------->  gap
      ypos = SysFont->height + FWIDTH*2 + 4 + 4;

  Face->ShiftX((wid-FACEICONW)/2 - Face->x1);   // Re-centre the face button
  Window::Resize((ws.GetDeskWidth()-wid)/2, ypos, (ws.GetDeskWidth()+wid)/2-1,
  //   margin time  margin  button   margin outline   grid        frame
  ypos + 4 + DIGITH + 4 + FACEICONH + 4*2 + 2 + height*SQUAREH + FWIDTH*2 - 1);
}


// Draw the outline around the grid and display the dimensions and difficulty.
// Calculate some display offsets.
void BoardWindow::PaintWindow(int x1, int y1, int x2, int y2)
{
  Window::PaintWindow(x1, y1, x2, y2);

  ViewBuffer *buf = GetSubBuffer(x1, y1, x2, y2);

  // Top-left co-ordinate of the grid itself
  ox = cx1 + 4 + 1; oy = cy1 + 4 + DIGITH + 4 + FACEICONH + 4 + 1;
  spacesx = cx2 - 5 - 3*DIGITW;         // Spaces left
  timex = Face->x2 - 3*DIGITW - 1;      // Time taken, above the face button
                                        //  which is already centred
  Line(buf, ox-1, oy-1, cx2-4, oy-1, 235);  // Outline around the grid
  Line(buf, ox-1, oy,   ox-1, cy2-4, 235);  // 235 is white
  Line(buf, cx2-4, oy, cx2-4, cy2-4, 149);  // 149 is grey
  Line(buf, ox, cy2-4, cx2-4, cy2-4, 149);

  char num[8];
  // Display the dimensions in the top-left corner
  sprintf(num, "%d x %d", layout.width, layout.height);
  WriteText(buf, FWIDTH+4, FWIDTH+4, 0, SysFont, num);
  // and the difficulty in the top-right
  sprintf(num, "%.3g", layout.diff());
  WriteText(buf, cx2-4-StringWidth(num, SysFont), FWIDTH+4,
            0, SysFont, num);

  DeleteBuffer(buf);
}


/* -----------------------------   PlaceMine   ------------------------------

  Pick a random location to place a mine on the grid and increase the count
  around it. If the game is CROSS, leave the four squares at the top-left
  and bottom-right corners clear.

*/

void BoardWindow::PlaceMine()
{
  int x, y;
  do
  {
    x = random()%layout.width;
    y = random()%layout.height;
  }
  while (Value(x, y) == 9 || (Game == CROSS &&
         ((x < 2 && y < 2) || (x > layout.width-3 && y > layout.height-3))));
  Value(x, y) = 9;
  Count(x, y, 1);
}


/* ------------------------------   MoveMine   ------------------------------

  If there's a mine at (x, y) move it somewhere else, update the count and
  return TRUE. If there's no mine, return FALSE.
  The display is not updated if squares have already been opened.
  There's also a very small chance that (x, y) could be rechosen.

*/

BOOL BoardWindow::MoveMine(int x, int y)
{
  if (Value(x, y) != 9) return FALSE;
  Count(x, y, -1);
  Value(x, y) = Count(x, y, MINE) - 1;
  PlaceMine();
  return TRUE;
}


// When the game changes from CLEAR to CROSS, need to ensure that the top and
// bottom corners are free from mines.
void BoardWindow::AdjustGame()
{
  BOOL moving = TRUE;
  while (moving)
  {
    moving = (MoveMine(0, 0) || MoveMine(1, 0) ||
              MoveMine(0, 1) || MoveMine(1, 1) ||
              MoveMine(layout.width-2, layout.height-2) ||
              MoveMine(layout.width-1, layout.height-2) ||
              MoveMine(layout.width-2, layout.height-1) ||
              MoveMine(layout.width-1, layout.height-1));
  }
}


// When the Wrap-around option is changed, the edges need to be recounted.
void BoardWindow::AdjustWrap()
{
  for (int x = 0; x < layout.width; x++)
  {
    if (Value(x, 0) != 9) Value(x, 0) = Count(x, 0, MINE);
    if (Value(x, layout.height-1) != 9)
      Value(x, layout.height-1) = Count(x, layout.height-1, MINE);
  }
  for (int y = 1; y < layout.height-1; y++)
  {
    if (Value(0, y) != 9) Value(0, y) = Count(0, y, MINE);
    if (Value(layout.width-1, y) != 9)
      Value(layout.width-1, y) = Count(layout.width-1, y, MINE);
  }
}


// Ensure the correct row for wrap-around
inline int Around::row(int r)
{
  return (r < 0) ? r+h : (r >= h) ? r-h : r;
}

// Ensure the correct column for wrap-around
inline int Around::col(int c)
{
  return (c < 0) ? c+w : (c >= w) ? c-w : c;
}

inline int Around::x() { return rx; }
inline int Around::y() { return ry; }


// Set up a loop for the three-by-three square that (cx, cy) is the centre,
// taking into account the edges. Start at the bottom-right corner (see next).
Around::Around(int cx, int cy, int width, int height) :
  w(width), h(height), x1(cx-1), y1(cy-1), x2(cx+1), y2(cy+1)
{
  if (!isWrap || Game == CROSS)
  {
    x1 >?= 0; y1 >?= 0;
    x2 <?= w-1; y2 <?= h-1;
  }
  ax = x2; ay = y2;
  rx = col(ax); ry = row(ay);
}


// Determine the next co-ordinate in the loop, where the order is:
// 9 8 7        9 is (x1, y1), set by the c'tor
// 6 5 4        5 is (cx, cy)
// 3 2 1        1 is (x2, y2)
// The reason for this is to help minimise opening squares in the cross game.
// If the loop is finished return FALSE, otherwise TRUE.
BOOL Around::next()
{
  if (--ax < x1)
  {
    if (--ay < y1) return FALSE;
    ry = row(ay);
    ax = x2;
  }
  rx = col(ax);
  return TRUE;
}

// Return TRUE if the loop is finished, FALSE otherwise.
inline BOOL Around::done()
{
  return (ay < y1);
}


#define around(x, y, loop) \
for (Around loop(x, y, layout.width, layout.height); !loop.done(); loop.next())


/* --------------------------------   Open   --------------------------------

  Open the square (x, y) and test for game over. If automatic opening is on
  then clear around it. If automatic marking is on then test all around the
  squares that are around it.
  The square is assumed to be unopened and unmarked (both mine and question).
  The space count display is not updated.

*/

void BoardWindow::Open(int x, int y)
{
  if (isFinished) return;
  Status(x, y) = 1;
  if (Value(x, y) == 9)
  {
    DisplaySquare(x, y, DEADMINE);
    --mines;
    MineCount();
  }
  else
  {
    DisplaySquare(x, y, Value(x, y));
    --spaces;
  }
  //unsigned rest = GetTicks() + 500; while (GetTicks() < rest);
  if (GameOver(x, y)) return;
  if (isAutoOpen) ClearAround(x, y, FALSE);
  if (isAutoMark && !isFinished)
  {
    around (x, y, loop)
    {
      if (Status(loop.x(), loop.y()) != 1) continue;
      int count = 0;
      around (loop.x(), loop.y(), loop2)
      {
        if (Status(loop2.x(), loop2.y()) == 0 ||
            Status(loop2.x(), loop2.y()) == 2) count++;
      }
      if (Value(loop.x(), loop.y()) == count)
      {
        around (loop.x(), loop.y(), loop2)
        {
          if (Status(loop2.x(), loop2.y()) == 0) Mark(loop2.x(), loop2.y());
          if (isFinished) return;
        }
      }
    }
  }
}


/* -------------------------------   Clear   --------------------------------

  Clear around the selected square. This is used to automatically open squares
  that have no mines around them, and to clear around a square that has all
  its mines marked. For the clear game, a queue is used to keep track of
  squares to be opened. This creates a spiral effect which looks better than a
  recursive call. Doing this for the cross game could open unnecessary squares,
  so a straight recursive call is used to clear around, but a custom sequence
  is used to clear the blanks.
  The space count display is updated.

*/

void BoardWindow::Clear(int x, int y)
{
  if (isFinished) return;
  if (Game == CLEAR)
  {
    struct OpenQueue
    {
      int x, y;
      OpenQueue* next;
      OpenQueue() : next(NULL) {}
    } *head = NULL, *tail = NULL;

    for (;;)
    {
      around (x, y, loop)
      {
        if (Status(loop.x(), loop.y()) == 0)
        {
          Open(loop.x(), loop.y());
          if (isFinished) break;
          if (Value(loop.x(), loop.y()) == 0)
          {
            if (head == NULL) head = tail = new OpenQueue;
            else tail = tail->next = new OpenQueue;
            tail->x = loop.x(); tail->y = loop.y();
          }
        }
      }
      if (isFinished)
      {
        for (OpenQueue* temp = head; head != NULL; temp = head)
        {
          head = head->next;
          delete temp;
        }
        break;
      }
      if (head == NULL) break;
      x = head->x;
      y = head->y;
      OpenQueue* temp = head->next;
      delete head;
      head = temp;
    }
  }
  else                                  // The cross game
  {
    if (Value(x, y) != 0)               // Clear around
    {
      around (x, y, loop)
      {
        if (Status(loop.x(), loop.y()) == 0)
        {
          Open(loop.x(), loop.y()); if (isFinished) return;
          if (Value(loop.x(), loop.y()) == 0)
          {
            Clear(loop.x(), loop.y()); if (isFinished) return;
          }
        }
      }
    }
    else                                // Clear blanks
    {
      // This uses the ordering:
      // 8 6 4
      // 7   2
      // 5 3 1
      // I could put this order into the Around::next() function, allowing
      // the straight recursive call. However, I think it would overly
      // complicate next() and since these are already written...
      if (x < layout.width-1 && y < layout.height-1 && Status(x+1, y+1) == 0)
      {
        Open(x+1, y+1); if (isFinished) return;
        if (Value(x+1, y+1) == 0)
        {
          Clear(x+1, y+1); if (isFinished) return;
        }
      }
      if (x < layout.width-1 && Status(x+1, y) == 0)
      {
        Open(x+1, y); if (isFinished) return;
        if (Value(x+1, y) == 0)
        {
          Clear(x+1, y); if (isFinished) return;
        }
      }
      if (y < layout.height-1 && Status(x, y+1) == 0)
      {
        Open(x, y+1); if (isFinished) return;
        if (Value(x, y+1) == 0)
        {
          Clear(x, y+1); if (isFinished) return;
        }
      }
      if (x < layout.width-1 && y > 0 && Status(x+1, y-1) == 0)
      {
        Open(x+1, y-1); if (isFinished) return;
        if (Value(x+1, y-1) == 0)
        {
          Clear(x+1, y-1); if (isFinished) return;
        }
      }
      if (x > 0 && y < layout.height-1 && Status(x-1, y+1) == 0)
      {
        Open(x-1, y+1); if (isFinished) return;
        if (Value(x-1, y+1) == 0)
        {
          Clear(x-1, y+1); if (isFinished) return;
        }
      }
      if (x > 0 && Status(x-1, y) == 0)
      {
        Open(x-1, y); if (isFinished) return;
        if (Value(x-1, y) == 0)
        {
          Clear(x-1, y); if (isFinished) return;
        }
      }
      if (y > 0 && Status(x, y-1) == 0)
      {
        Open(x, y-1); if (isFinished) return;
        if (Value(x, y-1) == 0)
        {
          Clear(x, y-1); if (isFinished) return;
        }
      }
      if (x > 0 && y > 0 && Status(x-1, y-1) == 0)
      {
        Open(x-1, y-1); if (isFinished) return;
        if (Value(x-1, y-1) == 0)
        {
          Clear(x-1, y-1); if (isFinished) return;
        }
      }
    }
  }
  SpaceCount();
}


/* --------------------------------   Mark   --------------------------------

  Mark an unopened square with a mine; change a marked square to an unsure
  square (if the Marks options is set) or remove a mark. Update the mine count
  and test for game over (for Finish & ALL_MARKED).
  If automatic opening is on then unmarking is disabled, and test the
  squares around the one being marked.

*/

void BoardWindow::Mark(int x, int y)
{
  if (!isAutoOpen || Status(x, y) == 0)
  {
    switch (Status(x, y))
    {
      case 1: return;
      case 0: if (mines == 0) return;
              Status(x, y) = 2; mines--;
              mynes -= (Value(x, y) == 9);
              DisplaySquare(x, y, MARKED);
              break;
      case 2: mines++;
              mynes += (Value(x, y) == 9);
              if (isMarks)
              {
                Status(x, y) = 3;
                DisplaySquare(x, y, UNSURE);
                break;
              }
              // else fall through
      case 3: Status(x, y) = 0;
              DisplaySquare(x, y, UNOPENED);
    }
    MineCount();
    if (GameOver(x, y)) return;
  }
  if (isAutoOpen && Status(x, y) == 2)
  {
    around (x, y, loop)
    {
      ClearAround(loop.x(), loop.y(), FALSE);
      if (isFinished) break;
    }
  }
}


/* ----------------------------   ClearAround   -----------------------------

  If the number of marked mines around (x, y) equals the value of (x, y) then
  open all the surrounding squares. If it doesn't then highlight the unopened
  squares (provided HiLite is TRUE). If more mines need to be marked, display
  mines, else display opened squares.

*/

void BoardWindow::ClearAround(int x, int y, BOOL HiLite)
{
  if (Status(x, y) != 1) return;
  int c = Count(x, y, MARKED);
  if (Value(x, y) == c) Clear(x, y);
  else if (HiLite)
  {
    HighLight(x, y, (Value(x, y) > c ? MINE : OPENED));
    hx = x; hy = y;
  }
}


/* -----------------------------   HighLight   ------------------------------

  Highlight the squares that could have a mine for clear-around. The highlight
  is removed when the mouse button is released.

*/

void BoardWindow::HighLight(int x, int y, int type)
{
  around (x, y, loop)
  {
    if (Status(loop.x(), loop.y()) == 0)
      DisplaySquare(loop.x(), loop.y(), type);
  }

  isHiLite = (type != UNOPENED);
}

void BoardWindow::LButtonUp(int, int, int)
{
  if (isHiLite && ClearButton == CLRLEFT) HighLight(hx, hy, UNOPENED);
}

void BoardWindow::RButtonUp(int, int, int)
{
  if (isHiLite && ClearButton == CLRRIGHT) HighLight(hx, hy, UNOPENED);
}

void BoardWindow::MButtonUp(int, int, int)
{
  if (isHiLite && ClearButton == CLRMIDDLE) HighLight(hx, hy, UNOPENED);
}


/* ------------------------------   GameOver   ------------------------------

  This function tests for game over. If it isn't, it exists immediately. If
  the game was lost then all the mines are displayed. If the game was won then
  all the mines are marked and all the squares are opened.

*/

BOOL BoardWindow::GameOver(int x, int y)
{
  if (isFinished) return TRUE;
  if (Game == CLEAR)
  {
    if (!((Value(x, y) == 9 && Status(x, y) == 1) ||
         (spaces == 0 && Finish & ALL_OPENED) ||
         (mynes == 0 && Finish & ALL_MARKED) ||
         (spaces == 0 && mynes == 0 && Finish == 0))) return FALSE;
  }
  else
  {
    if (!((Value(x, y) == 9 && Status(x, y) == 1) ||
         Status(layout.width-1, layout.height-1) == 1)) return FALSE;
  }

  mils = GetTicks() - startmils;
  ws.StopTimer(this);
  isFinished = TRUE;
  secs = mils / 1000; mils %= 1000;
  DisplayTime(TRUE);

  if (Value(x, y) == 9 && Status(x, y) == 1)            // Died
  {
    Face->Display(FACEDEAD);
    for (int j = 0; j < layout.height; j++)
    {
      for (int k = 0; k < layout.width; k++)
      {
        if (Value(k, j) == 9)
        {
          if (Status(k, j) == 0 || Status(k, j) == 3)
            DisplaySquare(k, j, MINE);
        }
        else if (Status(k, j) == 2)
        {
          DisplaySquare(k, j, WRONGMINE);
          mines++; MineCount();
        }
      }
    }
  }
  else                                                  // Completed
  {
    if (Game == CLEAR)
    {
      spaces = mines = 0;
    }
    else
    {
      spaces = (layout.width*layout.height-layout.mines) - spaces;
      mines = layout.mines - mines;
    }
    MineCount(); SpaceCount();
    Face->Display(FACECOOL);
    for (int j = 0; j < layout.height; j++)
    {
      for (int k = 0; k < layout.width; k++)
      {
        if (Game == CLEAR)
        {
          if (Status(k, j) == 0 || Status(k, j) == 3)
            DisplaySquare(k, j, (Value(k, j) == 9 ? MARKED : Value(k, j)));
        }
        else if (Status(k, j) == 1) DisplaySquare(k, j, OPENED);
      }
    }
    times.Check(level, secs*1000+mils, spaces, mines);
  }
  return TRUE;
}


/* -------------------------------   Count   --------------------------------

  Count around the square (x, y) and return a value depending on what:
  MINE:   the number of mines;
  MARKED: the number of marked squares (not questions);
   1:     increase the value of the surrounding squares (returns 0);
  -1:     decrease the value of the surrounding squares (returns 0).

*/

int BoardWindow::Count(int x, int y, int what)
{
  int count = 0;
  around (x, y, loop)
  {
    switch (what)
    {
      case MINE:   if (Value( loop.x(), loop.y()) == 9) count++; break;
      case MARKED: if (Status(loop.x(), loop.y()) == 2) count++; break;
      case -1: case 1:
        if (Value(loop.x(), loop.y()) != 9) Value(loop.x(), loop.y()) += what;
    }
  }
  return count;
}


/* -------------------------------   Timer   --------------------------------

  The timer routine, called every quarter-second. It reads the system time and
  updates the display if the seconds have changed.

*/

void BoardWindow::Timer()
{
  if (!isPaused)
  {
    mils = GetTicks() - startmils;
    secs = mils / 1000;
    if (secs != oldsecs)
    {
      oldsecs = secs;
      DisplayTime();
    }
  }
}


/* ----------------------------   DisplayTime   -----------------------------

  Display the time. If full is TRUE then it displays the milliseconds, other-
  wise only the seconds are displayed.

*/

void BoardWindow::DisplayTime(BOOL full)
{
  Display(secs, timex, 4);
  if (full)
    Display(mils, timex+3*DIGITW, 4+DIGITH-SMALLDIGITH, SmallDigit, TRUE);
}


/* ------------------------------   Display   -------------------------------

  Display the number num at (x, y) (relative to the board window's client
  area).  It uses the bmp[] buffers for each digit (0 to 9, 10 for minus, 11
  for blank, all the same dimensions).  If zeros is TRUE the number is padded
  with zeros, otherwise blanks.  num is assumed to be three or less characters
  (ie.  -99 .. 999). (It can be four, in which case only the first three are
  printed; more than that will probably crash.)

*/

void BoardWindow::Display(int num, int x, int y, ViewBuffer* bmp[], BOOL zeros)
{
  char str[5];
  int n, x1, j,
      wid = bmp[0]->width, hyt = bmp[0]->height;
  x += cx1; y += cy1;

  sprintf(str, (zeros) ? "%03d" : "%3d", num);
  for (x1 = x, j = 0; j < 3; j++, x1 += wid)
  {
    if (str[j] == ' ') n = BLANK;
    else if (str[j] == '-') n = MINUS;
    else n = str[j] - '0';
    BitBlt(wnd, x1, y, bmp[n], 0, 0, wid-1, hyt-1);
  }
  RefreshWindow(absx+x, absy+y, absx+x1-1, absy+y+hyt-1);
}


/* ---------------------------   DisplaySquare   ----------------------------

  Display the square (x, y) on the board. If refresh is FALSE then the square
  is not refreshed onto the screen.

*/

void BoardWindow::DisplaySquare(int x, int y, int square, BOOL refresh)
{
  int sx = ox + x*SQUAREW, sy = oy + y*SQUAREH;
  BitBlt(wnd, sx, sy, Square[square], 0, 0, SQUAREW-1, SQUAREH-1);
  if (refresh)
    RefreshWindow(absx+sx, absy+sy, absx+sx+SQUAREW-1, absy+sy+SQUAREH-1);
}


/* ------------------------------   Convert   -------------------------------

  Convert the arguments from an absolute screen co-ordinate to a grid co-
  ordinate.  Returns FALSE if the co-ordinate is not on the grid, otherwise
  TRUE. FALSE is also returned if the square is not next to an open square in
  the CROSS game.

*/

BOOL BoardWindow::Convert(int& x, int& y)
{
  x -= absx+ox; y -= absy+oy;
  if (x < 0 || x >= layout.width*SQUAREW ||
      y < 0 || y >= layout.height*SQUAREH) return FALSE;
  x /= SQUAREW; y /= SQUAREH;

  if (Game == CROSS && GameinProgress())
  {
    BOOL found = FALSE;
    around (x, y, loop)
    {
      if (Status(loop.x(), loop.y()) == 1)
      {
        found = TRUE; break;
      }
    }
    return found;
  }

  return TRUE;
}


/* ----------------------------   LButtonDown   -----------------------------

  This is called when the left mouse button is pressed on the board window. If
  the game is over, the cursor is outside the grid, or the square has already
  been pressed, the routine exits immediately.  If it is the first press, it
  checks the starting condition and acts appropriately.  The square is then
  opened. If it is a blank, then it clears squares around it. If clear around
  is set to the left button it will do that and then exit.

*/

void BoardWindow::LButtonDown(int x, int y, int)
{
  if (isPaused) return;
  if (isFinished) return;
  if (!Convert(x, y)) return;

  if (ClearButton == CLRLEFT) ClearAround(x, y);
  if (Status(x, y) != 0) return;

  if (!isStarted)
  {
    isStarted = TRUE;
    if (Game == CROSS) x = y = 0;
    else switch (Start)
    {
      case NO_REPOS: break;
      case ONE_REPOS: MoveMine(x, y); break;
      case AUTO:
      {
        int bx = -1, by = -1, dist,
            closest = MaxGridX*MaxGridX + MaxGridY*MaxGridY;
        for (int j = 0; j < layout.width; j++)
        {
          for (int k = 0; k < layout.height; k++)
          {
            if (Value(j, k) == 0)
            {
              dist = (j-x)*(j-x) + (k-y)*(k-y);
              if (dist < closest)
              {
                closest = dist;
                bx = j; by = k;
              }
            }
          }
        }
        if (bx == -1) MoveMine(x, y);
        else { x = bx; y = by; }
      }
    }
    oldsecs = 0;
    startmils = GetTicks();
    ws.StartTimer(this, 250);
  }

  Open(x, y);
  if (Value(x, y) == 0) { if (!isFinished) Clear(x, y); }
  else SpaceCount();
}


/* ----------------------------   RButtonDown   -----------------------------

  This is called when the right mouse button is pressed on the board window.
  If the cursor is not on the grid it exits immediately. If the game is over,
  then a new game is started with the same dimensions. Otherwise it tests the
  status of the square and acts appropriately.

*/

void BoardWindow::RButtonDown(int x, int y, int)
{
  if (isPaused) return;
  if (isFinished)
  {
    NewGame();
    return;
  }
  int temp = Game;
  if (isAutoOpen) Game = CROSS;         // Bit of a hack, but what the heck
  BOOL valid = Convert(x, y);
  Game = temp;
  if (!valid) return;
  if (ClearButton == CLRRIGHT) ClearAround(x, y);
  if (isAutoMark || ((Game == CROSS || isAutoOpen) && !isStarted)) return;
  Mark(x, y);
}


/* ----------------------------   MButtonDown   -----------------------------

  This is called when the middle mouse button is pressed on the board window.
  If the ClearButton is not middle then this function has no effect, otherwise
  it validates the press and clears around the square (x, y).
*/

void BoardWindow::MButtonDown(int x, int y, int)
{
  if (ClearButton != CLRMIDDLE) return;
  if (isPaused) return;
  if (!Convert(x, y)) return;
  ClearAround(x, y);
}


/* ------------------------------   cmPause   -------------------------------

  Pause the game. Display a sleepy face and blank the board. Call again to
  unpause.

*/

void BoardWindow::cmPause(int)
{
  if (!GameinProgress()) return;

  int wid =  layout.width * SQUAREW,
      hyt = layout.height * SQUAREH;

  if (isPaused)
  {
    Face->Display(FACESMILEY);
    isPaused = FALSE;
    BitBlt(wnd, ox, oy, gridsave, 0, 0, wid-1, hyt-1);
    DeleteBuffer(gridsave);
    RefreshWindow(absx+ox, absy+oy, absx+ox+wid-1, absy+oy+hyt-1);
    startmils += (GetTicks() - pausemils);
  }
  else
  {
    pausemils = GetTicks();
    Face->Display(FACESLEEPY);
    isPaused = TRUE;
    gridsave = NewBuffer(wid, hyt);
    BitBlt(gridsave, 0, 0, wnd, ox, oy, ox+wid-1, oy+hyt-1);
    for (int y = 0; y < layout.height; y++)
      for (int x = 0; x < layout.width; x++)
        DisplaySquare(x, y, OPENED, FALSE);
    RefreshWindow(absx+ox, absy+oy, absx+ox+wid-1, absy+oy+hyt-1);
  }
}

