/*
 -> MYCONFIG.CC
 -> MyConfigFile class for reading and writing configuration files.

 Jason Hood, 29 November, 1996.
 4 February, 1997 - Added GetItem and GetInt.
*/

#include "myconfig.h"
#include <string.h>
#include <stdio.h>

#define isaspace(c) ((c) == ' ' || (c) == '\t')


/* --------------------------------   Open   --------------------------------

  Read filename into memory, using a linked list of lines. If it cannot be
  opened return FALSE. If a file is already opened, then close it.

  It is assumed there is enough memory to hold the file.

*/

BOOL MyConfigFile::Open(const char* filename)
{
  if (cfgfile) Close();

  cfgfile = strdup(filename);           // Remember the filename for writing
  Head = Last = new LineList;
  First = LastItem = NULL;

  FILE* file = fopen(filename, "rt");
  if (!file) return FALSE;

  char buffer[1025];                    // 1K line length should be plenty
  LineList* ptr = Head;

  while (fgets(buffer, sizeof(buffer), file) != NULL)
  {
    *(strchr(buffer, '\n')) = 0;        // Don't include the newline
    ptr->line = strdup(buffer);
    ptr->next = new LineList;
    for (ptr->Name = ptr->line; isaspace(*ptr->Name); ptr->Name++);
    if (strchr("#%;\\/'\"|$@!:\n", *ptr->Name) == NULL)
    {                                   // Maintain the list of items
      if (!First) First = ptr;
      else LastItem->item = ptr;
      LastItem = ptr;
      LastItem->item = ptr->next;
      // Find the end of the name
      ptr->Item = strchr(ptr->Name, '=');
      for (ptr->pos = ptr->Item-1; isaspace(*ptr->pos); --ptr->pos);
      ptr->ch = *(++ptr->pos); *ptr->pos = 0;
      // Find the start of the item
      for (ptr->Item++; isaspace(*ptr->Item); ptr->Item++);
    }
    Last = ptr = ptr->next;
  }
  fclose(file);
  return TRUE;
}


/* --------------------------------   Close   -------------------------------

  Write the list of lines to the file. Returns FALSE if no file has been
  opened.

*/

BOOL MyConfigFile::Close()
{
  if (!cfgfile) return FALSE;

  FILE* file = fopen(cfgfile, "wt");
  free(cfgfile); cfgfile = NULL;

  LineList* ptr = Head;

  while (Head->next)
  {
    if (Head->pos) *Head->pos = Head->ch;  // Replace the overwritten character
    fputs(Head->line, file); fputc('\n', file);
    Head = Head->next;
    delete ptr;
    ptr = Head;
  }
  fclose(file);
  return TRUE;
}


/* -------------------------------   GetItem   ------------------------------

  Return the item associated with "name" (case insensitive). Leading spaces in
  the item are ignored, trailing spaces are not. If "name" does not exist (or
  no file is opened) return "defalt". If "name" exists more than once, the
  FIRST one will be returned.

*/

char* MyConfigFile::GetItem(const char* name, char* defalt)
{
  if (!cfgfile || !First) return defalt;

  LineList* ptr = First;

  while (ptr->line && stricmp(ptr->Name, name)) ptr = ptr->item;
  if (!ptr->line) return defalt;

  return ptr->Item;
}


/* -------------------------------   GetInt   -------------------------------

  As for GetItem, but returns an integer.

*/

int MyConfigFile::GetInt(const char* name, int defalt)
{
  char* item = GetItem(name, "");
  return (*item == 0) ? defalt : atoi(item);
}


/* ------------------------------   WriteItem   -----------------------------

  Add "name = item" if it doesn't exist, otherwise preserve (leading) spacing
  and overwrite the old value. Eg:

  "     name    =   olditem   "

  becomes

  "     name    =   item"

  If "name" is duplicated, only the FIRST one will be modified.

  Returns FALSE if no file is open.

*/

BOOL MyConfigFile::WriteItem(const char* name, const char* item)
{
  if (!cfgfile) return FALSE;

  if (!First) First = Last;             // There are no items in the file at all

  LineList* ptr = First;

  while (ptr->line && stricmp(ptr->Name, name)) ptr = ptr->item;
  if (!ptr->line)                       // The name doesn't exist
  {
    Last->line = (char*)malloc(strlen(name) + 3 + strlen(item) + 1);
    sprintf(Last->line, "%s = %s", name, item);
    Last->Name = Last->line;
    Last->pos = Last->line + strlen(name);
    Last->ch = ' '; *Last->pos = 0;
    Last->Item = Last->pos + 3;
    if (LastItem) LastItem->item = Last;
    LastItem = Last;
    LastItem->item = Last = Last->next = new LineList;
  }
  else
  {
    if (strlen(ptr->Item) < strlen(item))
    { // Allocate a longer line
      ptr->line = (char*)realloc((void*)ptr->line,
                   ptr->Item - ptr->line + strlen(item) + 1);
      // Need to re-find everything
      for (ptr->Name = ptr->line; isaspace(*ptr->Name); ptr->Name++);
      ptr->pos = strchr(ptr->Name, '\0');
      for (ptr->Item = ptr->pos+1; isaspace(*ptr->Item) || *ptr->Item == '=';
           ptr->Item++);
    }
    strcpy(ptr->Item, item);
  }
  return TRUE;
}

