/*
  Program:      Mine Mayhem
  File:         times.cc
  Date:         July, 1996 to February, 1997

  Everything to do with record times.

  Freeware, Copyright 1996-7 Jason Hood.

  You are free to use this code, or a portion thereof, as long as an
  appropriate acknowledgement is made.
*/

#include <time.h>
#include <stdio.h>
#include "times.h"
#include "board.h"
#include "options.h"


// Constructor for an individual entry in a times table, excluding the time
// itself.
Table::Entry::Entry() : start(0), finish(0), wrap(FALSE)
{
  strcpy(name, "Still waiting");
  strcpy(date, "??-???-????");
}


// Constructor for an entire times table.
Table::Table() : games(0), timetotal(0), spacetotal(0), minetotal(0)
{
  for (unsigned j = 0; j < Entries; j++)
  {
    fastest[j].time = 999999;           // ie. 999.999
    fastest[j].spaces = fastest[j].mines = 0;
    slowest[j].time = 0;
    slowest[j].spaces = slowest[j].mines = 999;
  }
}


// Update the number of games and total time for this table. If time is a
// record, then ask for the name and insert it at the appropriate position.
// Return TRUE for a record time, FALSE otherwise.
BOOL Table::Enter(unsigned time, unsigned spaces, unsigned mines)
{
  int isFast = -1, isSlow = FALSE;

  ++games;
  timetotal += time;
  spacetotal += spaces;
  minetotal += mines;

  for (unsigned j = 0; j < Entries; j++)
  {
    if (time <= fastest[j].time)        // Same time is regarded as better
    {                                   //  to reflect the later date
      shift(j, fastest);
      Stats(time, spaces, mines, j, fastest);
      isFast = j;
      break;
    }
  }
  for (unsigned j = 0; j < Entries; j++)
  {
    if (time >= slowest[j].time)        // Same time is regarded as worse
    {                                   //  to reflect the later date
      shift(j, slowest);
      if (isFast != -1) slowest[j] = fastest[isFast];
      else Stats(time, spaces, mines, j, slowest);
      isSlow = TRUE;
      break;
    }
  }
  return (isFast != -1 || isSlow);
}


// Make room for the new entry.
void Table::shift(unsigned from, Entry which[])
{
  // which[from+1..Entries-1] = which[from..Entries-2]
  for (unsigned j = Entries-1; j > from; j--)
  {
    which[j] = which[j-1];
  }
}


// Place the values in the entry.
void Table::Stats(unsigned time, unsigned spaces, unsigned mines,
                  unsigned position, Entry which[])
{
  GetDate(which[position]);
  AskName(position, which);
  which[position].time = time;
  which[position].start = Start;
  which[position].finish = Finish;
  which[position].wrap = isWrap;
  which[position].spaces = spaces;
  which[position].mines = mines;
}


// Place the date in the entry as "dd-Mmm-yyyy" (eg. 15-Feb-1997).
void Table::GetDate(Entry& entry)
{
  time_t now;
  time(&now);
  strftime(entry.date, DateLen, "%e-%b-%Y", localtime(&now));
}


// Show the player's position in which table and ask for the name.
void Table::AskName(unsigned position, Entry which[])
{
  static char *place[] = { "", " second", " third", " fourth", " fifth" },
              *fast = "Wow! That is the%s fastest time!",
              *slow = "Unfortunately, that is the%s slowest time.";

  char buffer[80], *message;

  message = (which == fastest) ? fast : slow;
  sprintf(buffer, message, place[position]);
  strcpy(which[position].name, EnterNameDlg(buffer).Run());
}


CustomPopupRes CustomPopup;


Times::Times()
{
  head = prevCustom = new CustomTimes;
  head->next = NULL;
}


Times::~Times()
{
  for (CustomTimes* list = head; head; )
  {
    head = head->next;
    delete list;
  }
}


// Given the level, type of game and automatic options, return a pointer to
// the appropriate table. If level is CUSTOM and there is no matching layout
// then create a new one.
Table* Times::FindTable(int level, int game, BOOL automark, BOOL autoopen)
{
  Table* table;

  switch (level)
  {
    case BEGINNER:     table = Beginner;     break;
    case INTERMEDIATE: table = Intermediate; break;
    case EXPERT:       table = Expert;       break;
    default: if (Custom == prevCustom->layout) table = prevCustom->times;
             else
             {
               CustomTimes* list = head;
               for (; list->next; list = list->next)
               {
                 if (list->layout == Custom)
                 {
                   prevCustom = list;
                   table = prevCustom->times;
                   break;
                 }
               }
               if (list->next == NULL)
               {
                 list->layout = Custom;
                 table = list->times;
                 prevCustom = list;
                 list = list->next = new CustomTimes;
                 list->next = NULL;
                 CustomPopup.Insert(Custom);
               }
             }
             break;
  }
  int which = (game!=0)*3 + (automark!=0)*2 + (autoopen!=0);
  return &table[which];
}


// Determine if time is a record for level, provided it's not a random game,
// or it's not automatic opening AND marking.
BOOL Times::Check(int level, unsigned time, unsigned spaces, unsigned mines)
{
  if (level < BEGINNER || (isAutoMark && isAutoOpen)) return FALSE;

  Table* table = FindTable(level);
  BOOL rc = table->Enter(time, spaces, mines);
  if (rc) TimesWindow(*table).Run();
  return rc;
}


// Display the times for level.
void Times::Display(int level)
{
  BOOL mark = isAutoMark, open = isAutoOpen;
  Table* table =  FindTable(level);
  if (isAutoMark && isAutoOpen) isAutoMark = isAutoOpen = FALSE;
  TimesWindow(*table).Run();
  isAutoMark = mark; isAutoOpen = open;
}


// Write all the times to disk.
// Returns FALSE if the file could not be created, TRUE otherwise.
BOOL Times::Save()
{
  FILE* file = fopen("mine.tym", "wb");
  if (!file) return FALSE;

  fwrite(Beginner, sizeof(Beginner), 1, file);
  fwrite(Intermediate, sizeof(Intermediate), 1, file);
  fwrite(Expert, sizeof(Expert), 1, file);

  for (CustomTimes* list = head; list->next; list = list->next)
  {
    fwrite(&list->layout, sizeof(list->layout), 1, file);
    fwrite(list->times, sizeof(list->times), 1, file);
  }

  fclose(file);
  return TRUE;
}


// Read all the times from disk and create the custom popup menu.
// Returns FALSE if the file could not be opened, TRUE otherwise.
BOOL Times::Load()
{
  FILE* file = fopen("mine.tym", "rb");
  if (!file) return FALSE;

  fread(Beginner, sizeof(Beginner), 1, file);
  fread(Intermediate, sizeof(Intermediate), 1, file);
  fread(Expert, sizeof(Expert), 1, file);

  CustomTimes* list = head;
  for (;;)
  {
    fread(&list->layout, sizeof(list->layout), 1, file);
    if (feof(file)) break;
    fread(list->times, sizeof(list->times), 1, file);
    CustomPopup.Insert(list->layout);
    list = list->next = new CustomTimes;
  }
  list->next = NULL;

  fclose(file);
  return TRUE;
}


#define ID_AUTOMARK 30001
#define ID_AUTOOPEN 30002

#define ID_GAMECLEAR 30003
#define ID_GAMECROSS 30004
#define ID_LEVELBEGINNER 30005
#define ID_LEVELINTERMEDIATE 30006
#define ID_LEVELEXPERT 30007
#define ID_LEVELCUSTOM 30008

DEFINE_RESPONSE_TABLE(TimesWindow, TabWin)
  E_BUTTONUP(ID_OK,           cmOK)
  E_BOXCHECKED(ID_AUTOMARK,   CheckAutoMark)
  E_BOXCHECKED(ID_AUTOOPEN,   CheckAutoOpen)
  E_BOXUNCHECKED(ID_AUTOMARK, cmTable)
  E_BOXUNCHECKED(ID_AUTOOPEN, cmTable)

  E_BOXCHECKED(ID_LEVELBEGINNER,       CheckLevelBeginner)
  E_BOXCHECKED(ID_LEVELINTERMEDIATE,   CheckLevelIntermediate)
  E_BOXCHECKED(ID_LEVELEXPERT,         CheckLevelExpert)
  E_BOXCHECKED(ID_LEVELCUSTOM,         CheckLevelCustom)
  E_BOXCHECKED(ID_GAMECLEAR,           CheckGameClear)
  E_BOXCHECKED(ID_GAMECROSS,           CheckGameCross)
  E_BOXUNCHECKED(ID_LEVELBEGINNER,     UnCheckLevelBeginner)
  E_BOXUNCHECKED(ID_LEVELINTERMEDIATE, UnCheckLevelIntermediate)
  E_BOXUNCHECKED(ID_LEVELEXPERT,       UnCheckLevelExpert)
  E_BOXUNCHECKED(ID_LEVELCUSTOM,       UnCheckLevelCustom)
  E_BOXUNCHECKED(ID_GAMECLEAR,         UnCheckGameClear)
  E_BOXUNCHECKED(ID_GAMECROSS,         UnCheckGameCross)

END_RESPONSE_TABLE


#define TIMES_WIDTH  (64*FixedFont->width)
#define TIMES_HEIGHT (27*FixedFont->height)

// Create the times' display and display table.
TimesWindow::TimesWindow(const Table& table) :
  TabWin(NULL, NULL, (ws.GetDeskWidth()-TIMES_WIDTH)/2,
                     (ws.GetDeskHeight()-TIMES_HEIGHT)/2,
                     (ws.GetDeskWidth()+TIMES_WIDTH)/2-1,
                     (ws.GetDeskHeight()+TIMES_HEIGHT)/2-1,
         WA_VISABLE | WA_BORDER | WA_SAVEAREA),
  CustomSave(Custom), first(TRUE)
{
  const int fw = FixedFont->width,
            fh = FixedFont->height;

  SysFont = FixedFont;
  int oyf = FWIDTH+2*fh+fh/2, oys = oyf + 5*(fh+2) + 4*fh,
      wid = 60*fw;
  for (unsigned j = 0; j < Entries; j++)
  {
    fast[j] = new StaticText(this, "", 2*fw, oyf, wid);
    oyf += fh+2;
    slow[j] = new StaticText(this, "", 2*fw, oys, wid);
    oys += fh+2;
  }
  status = new StaticText(this, "", 2*fw, oyf+fh+fh/2, wid);

  wid = 12*fw; //wid = 15*fw;
  int hyt = 2*(fh+2),
      ox = (TIMES_WIDTH/2-wid)/2, oy = TIMES_HEIGHT-5*fh-fh/2, oy1 = oy;
  //game = new RadioButtonContainer(this, ox, oy, ox+wid-1, oy+hyt-1);
  //game->AddButton("C&lear", CLEAR);
  //game->AddButton("C&ross", CROSS);
  //game->SelectButton(Game);
  ox += 3; oy += 3;
  gameclear = new CheckBox(this, ID_GAMECLEAR, "C&lear", ox, oy, wid);
  oy += fh;
  gamecross = new CheckBox(this, ID_GAMECROSS, "C&ross", ox, oy, wid);
  (Game == CLEAR) ? gameclear->Check() : gamecross->Check();

  // ox += 3; oy += hyt-1;
  oy += fh;
  automark = new CheckBox(this,ID_AUTOMARK, "Auto&Mark", ox,oy,wid, isAutoMark);
  oy += fh;
  autoopen = new CheckBox(this,ID_AUTOOPEN, "Auto&Open", ox,oy,wid, isAutoOpen);

  ox += TIMES_WIDTH/2-1; oy = oy1;
  hyt = 4*(fh+2);
  //level = new RadioButtonContainer(this, ox, oy, ox+wid-1, oy+hyt-1,
  //                                 RADIO_TYPE_DIP);
  //level->AddButton("&Beginner", BEGINNER);
  //level->AddButton("&Intermediate", INTERMEDIATE);
  //level->AddButton("&Expert", EXPERT);
  //level->AddButton("&Custom", CUSTOM);
  //level->SelectButton((Board->GetLevel() < BEGINNER) ? CUSTOM : Board->GetLevel());
  ox += 3; oy += 3;
  levelbeginner = new CheckBox(this, ID_LEVELBEGINNER, "&Beginner", ox, oy, wid);
  oy += fh;
  levelintermediate = new CheckBox(this, ID_LEVELINTERMEDIATE, "&Intermediate", ox, oy, wid);
  oy += fh;
  levelexpert = new CheckBox(this, ID_LEVELEXPERT, "&Expert", ox, oy, wid);
  oy += fh;
  levelcustom = new CheckBox(this, ID_LEVELCUSTOM, "&Custom", ox, oy, wid);
  switch (Board->GetLevel())
  {
    case BEGINNER:     levelbeginner->Check();     break;
    case INTERMEDIATE: levelintermediate->Check(); break;
    case EXPERT:       levelexpert->Check();       break;
    default:           levelcustom->Check();
  }

  wid = StringWidth("OK", FixedFont)*2;
  ox = (TIMES_WIDTH-wid)/2; oy = TIMES_HEIGHT-fh*4;
  ok = new TextButton(this, ID_OK, ox, oy,
                      ox+wid-1, oy+SysFontBold->height+3, "O&K");

  //AddTabStop(game);
  AddTabStop(gameclear);
  AddTabStop(gamecross);
  AddTabStop(automark);
  AddTabStop(autoopen);
  //AddTabStop(level);
  AddTabStop(levelbeginner);
  AddTabStop(levelintermediate);
  AddTabStop(levelexpert);
  AddTabStop(levelcustom);
  AddTabStop(ok);

  Paint(); RefreshWindow();
  Print(table);
  SysFont = SysFont1;
}


TimesWindow::~TimesWindow()
{
  for (unsigned j = 0; j < Entries; j++)
  {
    delete fast[j]; delete slow[j];
  }
  delete status;
  //delete game; delete level;
  delete gameclear; delete gamecross;
  delete levelbeginner; delete levelintermediate;
  delete levelexpert; delete levelcustom;
  delete automark; delete autoopen;
  delete ok;
  Custom = CustomSave;
}

extern BYTE HighLightColour;

// Draw the frames and display the titles for the fastest and slowest times,
// and draw the boxes around the selections.
void TimesWindow::PaintWindow(int x1, int y1, int x2, int y2)
{
  Window::PaintWindow(x1, y1, x2, y2);

  ViewBuffer* buf = GetSubBuffer(x1, y1, x2, y2);
  const int fw = FixedFont->width,
            fh = FixedFont->height;

  int oy = FWIDTH+fh/2,
      hyt = 2*fh + 5*(fh+2) + fh/2;
  Frame(buf, fw, oy, TIMES_WIDTH-fw-1, oy+hyt-1);
  oy += hyt + 3*fh;
  InvFrame(buf, fw, oy, TIMES_WIDTH-fw-1, oy+hyt-1);
  int len = StringWidth("FASTEST TIMES", FixedFont);
  WriteText(buf, (TIMES_WIDTH-len)/2, FWIDTH+fh,
                 HighLightColour, FixedFont, "FASTEST TIMES");
  len = StringWidth("SLOWEST TIMES", FixedFont);
  WriteText(buf, (TIMES_WIDTH-len)/2, oy+hyt-fh-fh/2,
                 HighLightColour, FixedFont, "SLOWEST TIMES");

  oy = TIMES_HEIGHT-5*fh-fh/2-1;
  hyt = 4*(fh+2)+1;
  int wid = 15*fw,
      ox = (TIMES_WIDTH/2-12*fw)/2-1,
      ey = oy + hyt-2;
  HDip(buf, ox, ox+wid-2, oy);
  VDip(buf, ox, oy+1, ey);
  HDip(buf, ox, ox+wid-2, ey);
  VDip(buf, ox+wid-2, oy, ey);

  ox += 3 + TIMES_WIDTH/2-1;
  HDip(buf, ox, ox+wid-2, oy);
  VDip(buf, ox, oy+1, ey);
  HDip(buf, ox, ox+wid-2, ey);
  VDip(buf, ox+wid-2, oy, ey);

  DeleteBuffer(buf);
}


// Update the window with the values from table.
void TimesWindow::Print(const Table& table)
{
  char buffer[80],
       *start[] =  { "None", "One ", "Auto" },
       *finish[] = { "Both", "Open", "Mark", "Itha" },
       *wrap[] = { "", "Wrap" },
       *format, *stat;

  if (/*game->GetSelectedButton() == CROSS*/gamecross->isChecked())
  {
    format = "%3u.%03u  %-20s  %s  %4d  %4d";
    stat =   "%7.3f  Average               %-11s  %4d  %4d";
  }
  else
  {
    format = "%3u.%03u  %-20s  %s  %s  %s  %s";
    stat =   "%7.3f  Average               %d cleared";
  }

  for (unsigned j = 0; j < Entries; j++)
  {
    unsigned k = Entries-1 - j;
    if (gameclear->isChecked())
    {
      sprintf(buffer, format, table.fastest[j].time / 1000,
                              table.fastest[j].time % 1000,
                              table.fastest[j].name,
                              table.fastest[j].date,
                              start[table.fastest[j].start],
                              finish[table.fastest[j].finish],
                              wrap[table.fastest[j].wrap]);
    }
    else
    {
      sprintf(buffer, format, table.fastest[j].time / 1000,
                              table.fastest[j].time % 1000,
                              table.fastest[j].name,
                              table.fastest[j].date,
                              table.fastest[j].spaces,
                              table.fastest[j].mines);
    }
    fast[j]->SetText(buffer);
    if (gameclear->isChecked())
    {
      sprintf(buffer, format, table.slowest[k].time / 1000,
                              table.slowest[k].time % 1000,
                              table.slowest[k].name,
                              table.slowest[k].date,
                              start[table.slowest[k].start],
                              finish[table.slowest[k].finish],
                              wrap[table.slowest[k].wrap]);
    }
    else
    {
      sprintf(buffer, format, table.slowest[k].time / 1000,
                              table.slowest[k].time % 1000,
                              table.slowest[k].name,
                              table.slowest[k].date,
                              table.slowest[k].spaces,
                              table.slowest[k].mines);
    }
    slow[j]->SetText(buffer);
  }
  if (table.games == 0) *buffer = 0;
  else if (gamecross->isChecked())
  {
    char temp[13];
    sprintf(temp, "%d crossed", table.games);
    sprintf(buffer, stat, float(table.timetotal)/table.games/1000.0, temp,
            table.spacetotal/table.games, table.minetotal/table.games);
  }
  else sprintf(buffer, stat, float(table.timetotal)/table.games/1000.0,
               table.games);
  status->SetText(buffer);
}


void TimesWindow::Run()
{
  SysFont = FixedFont;
  SetFocus(ok);
  CatchMouse();
  CatchKeys();
  ws.RunEvents();
  ws.ResetEvents();
  ReleaseMouse();
  ReleaseKeys();
  SysFont = SysFont1;
}


// Since there's no automatic opening AND marking, uncheck the other one if
// it's checked.
void TimesWindow::CheckAutoMark()
{
  if (autoopen->isChecked()) autoopen->UnCheck();
  cmTable();
}

void TimesWindow::CheckAutoOpen()
{
  if (automark->isChecked()) automark->UnCheck();
  cmTable();
}


// Simulate radio buttons with checkboxes.
void TimesWindow::CheckGameClear()
{
  if (gamecross->isChecked()) gamecross->UnCheck();
  cmTable();
}

void TimesWindow::CheckGameCross()
{
  if (gameclear->isChecked()) gameclear->UnCheck();
  cmTable();
}

void TimesWindow::UnCheckGameClear()
{
  if (!gamecross->isChecked()) gameclear->Check();
}

void TimesWindow::UnCheckGameCross()
{
  if (!gameclear->isChecked()) gamecross->Check();
}

void TimesWindow::CheckLevelBeginner()
{
  if (first) first = FALSE;
  if (levelintermediate->isChecked()) levelintermediate->UnCheck();
  else if (levelexpert->isChecked()) levelexpert->UnCheck();
  else if (levelcustom->isChecked()) levelcustom->UnCheck();
  cmTable();
}

void TimesWindow::CheckLevelIntermediate()
{
  if (first) first = FALSE;
  if (levelbeginner->isChecked()) levelbeginner->UnCheck();
  else if (levelexpert->isChecked()) levelexpert->UnCheck();
  else if (levelcustom->isChecked()) levelcustom->UnCheck();
  cmTable();
}

void TimesWindow::CheckLevelExpert()
{
  if (first) first = FALSE;
  if (levelbeginner->isChecked()) levelbeginner->UnCheck();
  else if (levelintermediate->isChecked()) levelintermediate->UnCheck();
  else if (levelcustom->isChecked()) levelcustom->UnCheck();
  cmTable();
}


// If there's no custom games then prevent this from being selected,
// otherwise pop-up the menu underneath the "Custom".
void TimesWindow::CheckLevelCustom()
{
  if (CustomPopup.number == 0)
  {
    levelcustom->UnCheck();
    return;
  }
  if (first)
  {
    first = FALSE;
    return;
  }
  int curlevel;
  if (levelbeginner->isChecked())
  {
    levelbeginner->UnCheck(); curlevel = BEGINNER;
  }
  else if (levelintermediate->isChecked())
  {
    levelintermediate->UnCheck(); curlevel = INTERMEDIATE;
  }
  else if (levelexpert->isChecked())
  {
    levelexpert->UnCheck(); curlevel = EXPERT;
  }
  else curlevel = CUSTOM;

  int ox = absx + TIMES_WIDTH*3/4-6*FixedFont->width + 3 + 20,
      oy = absy + TIMES_HEIGHT-5*FixedFont->height-FixedFont->height/2 +
                  4*(FixedFont->height+2) - 2;
  if (CustomPopupMenu(NULL, ox, oy).Run()) cmTable();
  else
  {
    switch (curlevel)
    {
      case BEGINNER:     levelbeginner->Check();     break;
      case INTERMEDIATE: levelintermediate->Check(); break;
      case EXPERT:       levelexpert->Check();       break;
    }
  }
  CatchKeys();                          // It seems PopupMenu releases keys
}


void TimesWindow::UnCheckLevelBeginner()
{
  if (!levelintermediate->isChecked() && !levelexpert->isChecked() &&
      !levelcustom->isChecked()) levelbeginner->Check();
}

void TimesWindow::UnCheckLevelIntermediate()
{
  if (!levelbeginner->isChecked() && !levelexpert->isChecked() &&
      !levelcustom->isChecked()) levelintermediate->Check();
}

void TimesWindow::UnCheckLevelExpert()
{
  if (!levelintermediate->isChecked() && !levelbeginner->isChecked() &&
      !levelcustom->isChecked()) levelexpert->Check();
}

void TimesWindow::UnCheckLevelCustom()
{
  if (CustomPopup.number == 0) return;
  if (!levelintermediate->isChecked() && !levelexpert->isChecked() &&
      !levelbeginner->isChecked()) levelcustom->Check();
}


// Find the table corresponding to the selections and display it.
void TimesWindow::cmTable()
{
  int level, game;
  if (levelbeginner->isChecked()) level = BEGINNER;
  else if (levelintermediate->isChecked()) level = INTERMEDIATE;
  else if (levelexpert->isChecked()) level = EXPERT;
  else level = CUSTOM;
  game = (gameclear->isChecked()) ? CLEAR : CROSS;

  Table* table = times.FindTable(level,//->GetSelectedButton(),
                                 game,//->GetSelectedButton(),
                                 automark->isChecked(),
                                 autoopen->isChecked());
  Print(*table);
}


#define NAME_WIDTH  (50*SysFont1->width)
#define NAME_HEIGHT (12*SysFont1->height)

#define NAME_ID 30000

DEFINE_RESPONSE_TABLE(EnterNameDlg, Window)
  E_KEYCODE(0x0d, 0x1c, KeyEnter)
  E_BUTTONUP(ID_OK, cmOK)
END_RESPONSE_TABLE

char EnterNameDlg::lastname[NameLen] = { 0 };


// Create the enter name dialog box, where message indicates whether it was a
// fast or slow time. The name is entered using the fixed font.
EnterNameDlg::EnterNameDlg(const char* message) :
  Window(NULL, "Congratulations", (ws.GetDeskWidth()-NAME_WIDTH)/2,
                                  (ws.GetDeskHeight()-NAME_HEIGHT)/2,
                                  (ws.GetDeskWidth()+NAME_WIDTH)/2-1,
                                  (ws.GetDeskHeight()+NAME_HEIGHT)/2-1,
         WA_VISABLE | WA_CAPTION | WA_BORDER | WA_SAVEAREA)
{
  int len = StringWidth(message, SysFont);
  msg = new StaticText(this, message,
                       (NAME_WIDTH-len)/2, SysFont->height*3, len);
  len = NameLen*FixedFont->width;
  SysFont = FixedFont;
  name = new EditText(this, NAME_ID, "", NameLen,
                      (NAME_WIDTH-len)/2, SysFont1->height*6, len);
  SysFont = SysFont1;
  len = StringWidth("OK", SysFont)*2;
  int ox = (NAME_WIDTH-len)/2, oy = cy2-SysFont->height*2-4;
  ok = new TextButton(this, ID_OK, ox,oy, ox+len-1, oy+SysFont->height+3, "OK");
  SetFocus(name);
  name->SetText(lastname);
}


EnterNameDlg::~EnterNameDlg()
{
  delete msg;
  delete name;
  delete ok;
}


// Write the prompt onto the window.
void EnterNameDlg::PaintWindow(int x1, int y1, int x2, int y2)
{
  Window::PaintWindow(x1, y1, x2, y2);

  ViewBuffer* buf = GetSubBuffer(x1, y1, x2, y2);
  WriteText(buf, (NAME_WIDTH-NameLen*FixedFont->width)/2, SysFont->height*5,
                 HighLightColour, SysFontItallic, "Please enter your name:");
  DeleteBuffer(buf);
}


char* EnterNameDlg::Run()
{
  Paint();
  SysFont = FixedFont;
  name->Paint();
  RefreshWindow();
  CatchMouse();
  CatchKeys();
  ws.RunEvents();
  ws.ResetEvents();
  ReleaseMouse();
  ReleaseKeys();
  SysFont = SysFont1;
  strcpy(lastname, name->GetText());
  return (*lastname == 0) ? "Anonymous" : lastname;
}

