DECLARE FUNCTION BytesRequired& (Filename$)
DECLARE SUB Delay (Repetitions%)
DECLARE SUB DetectSettings (BasePort%, IRQ%, LoDMA%, HiDMA%, CardType%)
DECLARE SUB DriversLoaded (SBMIDI%, SBSIM%)
DECLARE SUB GetBass (LeftChannel%, RightChannel%)
DECLARE SUB GetMaster (LeftChannel%, RightChannel%)
DECLARE SUB GetMIDI (LeftChannel%, RightChannel%)
DECLARE SUB GetTreble (LeftChannel%, RightChannel%)
DECLARE SUB GetVoice (LeftChannel%, RightChannel%)
DECLARE SUB LoadAndPlayLarge (Filename$)
DECLARE SUB LoadAndPlayMIDI (Filename$, MIDISegment%, MIDIOffset%)
DECLARE SUB LoadLarge (Filename$)
DECLARE SUB LoadMIDI (Filename$, MIDISegment%, MIDIOffset%)
DECLARE FUNCTION MIDIError$ ()
DECLARE FUNCTION MixerChip$ ()
DECLARE SUB PauseMIDI ()
DECLARE SUB ResumeMIDI ()
DECLARE SUB PlayLarge ()
DECLARE SUB PlayMIDI (MIDISegment%, MIDIOffset%)
DECLARE SUB SetBass (LeftChannel%, RightChannel%)
DECLARE SUB SetCard (CardType%)
DECLARE SUB SetMaster (LeftChannel%, RightChannel%)
DECLARE SUB SetMIDI (LeftChannel%, RightChannel%)
DECLARE SUB SetTreble (LeftChannel%, RightChannel%)
DECLARE SUB SetVoice (LeftChannel%, RightChannel%)
DECLARE FUNCTION SoundCard$ (CardType%)
DECLARE SUB StopMIDI ()
DECLARE FUNCTION TimeMIDI! ()


'INTERNAL ROUTINES (CALLED BY QMIDI)

DECLARE FUNCTION InternalBitRead% (Variable%, BitNum%)
DECLARE SUB InternalBitSet (Variable%, BitNum%, OnOff%)
DECLARE SUB InternalBitToggle (Variable%, BitNum%)
DECLARE SUB InternalGetIntVector (IntNum%, Segment%, Offset%)
DECLARE SUB InternalGetVol (LeftChannel%, RightChannel%, Index%)
DECLARE SUB InternalSetVol (LeftChannel%, RightChannel%, Index%)
DECLARE SUB InternalWriteMixer (Index%, Value%)
DECLARE FUNCTION InternalReadMixer% (Index%)

 TYPE KeyIndexType
  KeyMVal AS INTEGER
  KeyDelay AS INTEGER
  KeyVal AS INTEGER
 END TYPE

COMMON SHARED MIDI.PLAYTIME AS SINGLE, MIDI.ERROR AS INTEGER
COMMON SHARED MIDI.LOADED AS INTEGER, SBMIDI.INTERRUPT AS INTEGER
COMMON SHARED SBSIM.INTERRUPT AS INTEGER, PAUSED AS SINGLE
COMMON SHARED MIXER.CHIP AS INTEGER, SB.BASEPORT AS INTEGER, SB.IRQ AS INTEGER
COMMON SHARED SB.LODMA AS INTEGER, SB.HIDMA AS INTEGER, SB.CARDTYPE AS INTEGER
COMMON SHARED BIT.STORAGE() AS INTEGER, SENSITIVE AS INTEGER
COMMON SHARED REVERSE.STEREO AS INTEGER, SOUND.DISABLED AS INTEGER
  COMMON SHARED PlaySndFlag%, WavFlag%
  COMMON SHARED BasePort%, LenPort%, Channel%, IRQ%, HaveBlast%, WavRep%, TheWavLen&
  COMMON SHARED TFWav() AS STRING * 15000
  COMMON SHARED SoundEvent%
 COMMON SHARED Mouse$
 COMMON SHARED LineCount%, Score&, Level%, Goal%, CurGoal%
 COMMON SHARED SeeNextFlag%, LineBuf%
 COMMON SHARED MusFile$, MusLen%, Song%, MIDIFlag%, SongFlag%
COMMON SHARED TetMask() AS SINGLE, TetFont() AS SINGLE, TfStg() AS STRING
COMMON SHARED Keys() AS KeyIndexType
COMMON SHARED FadeFlag%, MIDIPlaying%, SLevel%, DriverFlag%
'$DYNAMIC
DIM SHARED BIT.STORAGE(0 TO 7) AS INTEGER

'BytesRequired - Returns the amount of memory needed to store a file.
FUNCTION BytesRequired& (Filename$)
'Open the file.
FF% = FREEFILE
OPEN Filename$ FOR BINARY AS #FF%
'Store the length of the file.
FileLen& = LOF(FF%)
'Close the file.
CLOSE FF%
'If the length of the file is 0, assume it does not exist and delete it.
IF FileLen& = 0 THEN
    KILL Filename$
    MIDI.ERROR = 1
    EXIT FUNCTION
END IF
'Return the length of the file as the number of bytes required.
BytesRequired& = FileLen&
MIDI.ERROR = 0
END FUNCTION

REM $STATIC
'A machine-independent delay
SUB Delay (Repetitions%)
FOR I% = 1 TO Repetitions%
    WAIT &H3DA, 8, 8
    WAIT &H3DA, 8
NEXT I%
END SUB

REM $DYNAMIC
'DetectSettings - Attempt to detect Sound Blaster settings
SUB DetectSettings (BasePort%, IRQ%, LoDMA%, HiDMA%, CardType%)

BasePort% = 0
IRQ% = 0
LoDMA% = 0
HiDMA% = 0
CardType% = 0

'Read the BLASTER environment variable
Settings$ = ENVIRON$("BLASTER")

'Attempt to extract the base port, High DMA, Low DMA, IRQ, and card type
'from the BLASTER enironment variable.
FOR I% = 1 TO LEN(Settings$) - 1
    'If the type of sound card was found, get it and exit the loop.
    SELECT CASE UCASE$(MID$(Settings$, I%, 1))
        'If the card type was found...
        CASE "T"
            CardType% = VAL(MID$(Settings$, I% + 1, 1))
            'If the base port address was found...
        CASE "A"
            BasePort% = VAL("&H" + LTRIM$(STR$(VAL(MID$(Settings$, I% + 1, 3)))))
        'If the IRQ was found...
        CASE "I"
            IRQ% = VAL(MID$(Settings$, I% + 1, 2))
        'If the low DMA channel was found...
        CASE "D"
            LoDMA% = VAL(MID$(Settings$, I% + 1, 1))
        'If the high DMA channel was found...
        CASE "H"
            HiDMA% = VAL(MID$(Settings$, I% + 1, 1))
    END SELECT
NEXT I%

'If the card type wasn't found in the BLASTER variable, try to figure
'out the type using another method.

IF CardType% = 0 THEN
    'Examine the card's DMA channel.
    SELECT CASE LoDMA%
        'If the DMA is 210h or 230h, the card is an SB1.0/1.5.
        CASE &H210, &H230
            CardType% = 1
        'If the DMA is 250h or 260h, the card is either an SB2CD or a
        'Sound Blaster 16.  It could also be a Sound Blaster 1.0/1.5,
        'but it probably isn't.  Actually, it's also unlikely that the card
        'is an SB16, but I check for it anyway, because there's an easy way
        'to tell if it is - the High DMA channel will be greater than
        '0.
        '
        'On the other hand, there's no way that I know of to
        'distinguish an SB 1.0 from an SB 2.0, except by looking at the
        'BLASTER environment variable.  And since this code is executing
        'that method obviously failed.
        CASE &H250, &H260
            'Examining the High DMA channel will narrow it down.
            'If the High DMA is greater than 0, the card is an SB16.
            IF HiDMA% THEN
                CardType% = 6
            'Otherwise, define the card as a Sound Blaster 2.0.
            ELSE
                CardType% = 3
            END IF
        'If the DMA channel is any other value....
        CASE ELSE
            'Check the High DMA channel.  If it's a non-zero value,
            'we've got an SB16.
            IF HiDMA% THEN
                CardType% = 6
            'Otherwise....
            ELSE
                'If sensitive error checking is on, define the card as
                'a Sound Blaster 1.0/1.5.
                IF SENSITIVE THEN
                    CardType% = 1
                'Otherwise, assume it's a Sound Blaster Pro.
                ELSE
                    CardType% = 4
                END IF
            END IF
    END SELECT
END IF

'Determine the sound card's mixer chip
SELECT CASE CardType%
    'If the card could not be detected....
    CASE 0
        MIDI.ERROR = 7
        'If sensitive error checking is on, disable mixer operations
        IF SENSITIVE THEN
            MIXER.CHIP = 0
        'Otherwise, assume the default mixer chip.
        ELSE
            MIXER.CHIP = 2
        END IF
    'If the card is a Sound Blaster 1.0/1.5 or equivalent....
    CASE 1
        'Return an error.
        MIDI.ERROR = 6
        'If sensitive error checking is on, disable mixer operations and
        'exit.
        IF SENSITIVE THEN
            MIXER.CHIP = 0
            EXIT SUB
        'Otherwise, set the earliest mixer chip and continue.
        ELSE
            MIXER.CHIP = 1
        END IF
    'If the card is a Sound Blaster 2.0/2.5 or equivalent....
    CASE 3
        'There are two different kinds of SB 2.0 cards: the regular SB2,
        'and the SB2CD.  The SB2CD has a mixer chip (the CT1335), whereas
        'the SB 2.0 does not.  The way to tell them apart is that the
        'Sound Blaster 2.0 uses Base Ports 220h and 240h, and the SB2CD
        'uses ports 250h and 260h.
        '
        'Assume the sound card is an SB2CD for now...
        MIXER.CHIP = 1
        'If the card is defined as an SB 2.0, not an SB 2.0 CD, and
        'sensitive error checking is on, disable mixer operations.
        IF (BasePort% = &H220 OR BasePort% = &H240) AND SENSITIVE <> 0 THEN
            MIXER.CHIP = 0
        END IF
        MIDI.ERROR = 0
    'If the card is a Sound Blaster Pro, assume chip CT1345
    CASE 2, 4, 5
        MIXER.CHIP = 2
        MIDI.ERROR = 0
    'If the card is a Sound Blaster 16 or later, assume chip CT1745
    CASE IS >= 6
        MIXER.CHIP = 3
        MIDI.ERROR = 0
END SELECT
END SUB

'DriversLoaded - Attempt to detect if sound drivers are loaded
SUB DriversLoaded (SBMIDI%, SBSIM%) STATIC
'Open the data file.
FF% = FREEFILE
OPEN "DRIVERS.DAT" FOR BINARY AS #FF%
'If the file is empty, return an error
FileSize& = LOF(FF%)
IF FileSize& = 0 THEN
    CLOSE FF%
    KILL "DRIVERS.DAT"
    MIDI.ERROR = 1
    EXIT SUB
'If the file is not exactly 1 kilobyte in size, return an error
ELSEIF FileSize& <> 1024 THEN
    CLOSE FF%
    MIDI.ERROR = 9
    EXIT SUB
END IF
'Load the driver data.
REDIM DRIVERDATA$(1 TO 5)
FOR I% = 1 TO 4
    DRIVERDATA$(I%) = INPUT$(256, #FF%)
NEXT I%
CLOSE #FF%

'Check the interrupt handlers for int 80h-E1h to see if they contain
'program code from either SBSIM or SBMIDI.
SBMIDI% = 0
SBSIM% = 0
FOR I% = &H80 TO &HE1
    'Get the address of the interrupt handler.
    InternalGetIntVector I%, Segment%, Offset%
    'If the address is null, then the interrupt is not in use, and can be
    'skipped.
    IF Segment% = 0 AND Offset% = 0 THEN GOTO Skip:
    'The following code checks for the drivers by looking for the text
    '"SBMIDI" and "SBSIM" at certain locations in the driver code.  It
    'sounds simplistic, but it's very accurate.  If it doesn't work,
    'a different method is used.
    IF SBMIDI% = 0 THEN
        NewSegment% = CVI(MKI$(CVL(MKI$(Segment%) + CHR$(0) + CHR$(0)) - &H11&))
        DEF SEG = NewSegment%
        TEMP$ = ""
        FOR J% = 1 TO 6
            TEMP$ = TEMP$ + CHR$(PEEK(271 + J%))
        NEXT
        IF TEMP$ = "SBMIDI" THEN SBMIDI% = I%
    END IF
    IF SBSIM% = 0 THEN
        NewSegment% = CVI(MKI$(CVL(MKI$(Segment%) + CHR$(0) + CHR$(0)) - &H1&))
        DEF SEG = NewSegment%
        TEMP$ = ""
        FOR J% = 1 TO 5
            TEMP$ = TEMP$ + CHR$(PEEK(274 + J%))
        NEXT
        IF TEMP$ = "SBSIM" THEN SBSIM% = I%
    END IF
  
    'This is the second detection method.  It's more complex than the first
    'method, but not really any more accurate.  In fact, it's probably
    'less accurate.  It's kind of a last ditch effort in case the first
    'method fails.

    'Point to the segment of the interrupt handler.
    DEF SEG = Segment%
    'Read 256 bytes of the driver code.
    DRIVERDATA$(5) = ""
    FOR J% = 0 TO 255
        Byte% = PEEK(Offset% + J%)
        DRIVERDATA$(5) = DRIVERDATA$(5) + CHR$(Byte%)
    NEXT J%
    'Check to see if the code matches any of the previously saved data.
    FOR J% = 1 TO 4
        MATCH% = 1
        FOR K% = 0 TO 255
            IF MID$(DRIVERDATA$(J%), K% + 1, 1) <> MID$(DRIVERDATA$(5), K% + 1, 1) THEN
                SELECT CASE K%
                    CASE IS = 14, 15, 113, 114, 235, 236
                    CASE ELSE
                        MATCH% = 0
                        EXIT FOR
                END SELECT
            END IF
        NEXT K%
        IF MATCH% THEN
            IF J% = 1 THEN SBSIM% = I%
            IF J% <> 1 THEN SBMIDI% = I%
        END IF
        IF SBSIM% <> 0 AND SBMIDI% <> 0 THEN EXIT FOR
    NEXT J%
    IF SBSIM% <> 0 AND SBMIDI% <> 0 THEN EXIT FOR
Skip:
NEXT I%
MIDI.ERROR = 0
END SUB

'GetBass - Returns the current Bass level
SUB GetBass (LeftChannel%, RightChannel%)
SELECT CASE MIXER.CHIP
    'If no mixer chip is present, return an error
    CASE 0
        MIDI.ERROR = 6
        EXIT SUB
    'If the mixer chip is CT1335 or CT1345, return an error, becuase
    'these chips do not support Treble and Bass control.
    CASE 1, 2
        MIDI.ERROR = 8
        EXIT SUB
    'If the mixer chip is CT1745....
    CASE 3
        'Get the bass settings.
        LBass% = InternalReadMixer%(&H46)
        RBass% = InternalReadMixer%(&H47)
        LeftChannel% = 0
        RighChannel% = 0
        'Extract the settings
        FOR I% = 4 TO 7
            BIT.STORAGE(I%) = InternalBitRead%(LBass%, I%)
            IF BIT.STORAGE(I%) THEN InternalBitToggle LeftChannel%, I% - 4
            BIT.STORAGE(I%) = InternalBitRead%(RBass%, I%)
            IF BIT.STORAGE(I%) THEN InternalBitToggle RightChannel%, I% - 4
        NEXT I%
        'Convert the 4-bit settings to 5-bit values.
        LeftChannel% = LeftChannel% * 2
        IF LeftChannel% = 30 THEN LeftChannel% = 31
        RightChannel% = RightChannel% * 2
        IF RightChannel% = 30 THEN RightChannel% = 31
        MIDI.ERROR = 0
END SELECT
END SUB

'GetMaster - Returns the current Master volume level
SUB GetMaster (LeftChannel%, RightChannel%)
SELECT CASE MIXER.CHIP
    'If no mixer chip is present, return an error
    CASE 0: MIDI.ERROR = 6: EXIT SUB
    'If the mixer chip is CT1335....
    CASE 1
        InternalGetVol LeftChannel%, RightChannel%, 2
        MIDI.ERROR = 0
    'If the mixer chip is CT1345....
    CASE 2
        InternalGetVol LeftChannel%, RightChannel%, &H22
        MIDI.ERROR = 0
    'If the mixer chip is CT1745....
    CASE 3
        InternalGetVol LeftChannel%, RightChannel%, &H30
        MIDI.ERROR = 0
END SELECT
END SUB

'GetMIDI - Returns the current MIDI volume level
SUB GetMIDI (LeftChannel%, RightChannel%)
SELECT CASE MIXER.CHIP
    'If no mixer chip is present, return an error
    CASE 0: MIDI.ERROR = 6: EXIT SUB
    'If the mixer chip is CT1335....
    CASE 1
        InternalGetVol LeftChannel%, RightChannel%, 6
        MIDI.ERROR = 0
    'If the mixer chip is CT1345....
    CASE 2
        InternalGetVol LeftChannel%, RightChannel%, &H26
        MIDI.ERROR = 0
    'If the mixer chip is CT1745....
    CASE 3
        InternalGetVol LeftChannel%, RightChannel%, &H34
        MIDI.ERROR = 0
END SELECT
END SUB

'GetTreble - Returns the current Treble level
SUB GetTreble (LeftChannel%, RightChannel%)
SELECT CASE MIXER.CHIP
    'If no mixer chip is present, return an error
    CASE 0
        MIDI.ERROR = 6
        EXIT SUB
    'If the mixer chip is CT1335 or CT1345, return an error, becuase
    'these chips do not support Treble and Bass control.
    CASE 1, 2
        MIDI.ERROR = 8
        EXIT SUB
    'If the mixer chip is CT1745....
    CASE 3
        'Get the treble settings.
        LTreble% = InternalReadMixer%(&H44)
        RTreble% = InternalReadMixer%(&H45)
        LeftChannel% = 0
        RighChannel% = 0
        'Extract the settings
        FOR I% = 4 TO 7
            BIT.STORAGE(I%) = InternalBitRead%(LTreble%, I%)
            IF BIT.STORAGE(I%) THEN InternalBitToggle LeftChannel%, I% - 4
            BIT.STORAGE(I%) = InternalBitRead%(RTreble%, I%)
            IF BIT.STORAGE(I%) THEN InternalBitToggle RightChannel%, I% - 4
        NEXT I%
        'Convert the 4-bit settings to 5-bit values.
        LeftChannel% = LeftChannel% * 2
        IF LeftChannel% = 30 THEN LeftChannel% = 31
        RightChannel% = RightChannel% * 2
        IF RightChannel% = 30 THEN RightChannel% = 31
        MIDI.ERROR = 0
END SELECT
END SUB

'GetVoice - Returns the current Voice volume level
SUB GetVoice (LeftChannel%, RightChannel%)
SELECT CASE MIXER.CHIP
    'If no mixer chip is present, return an error
    CASE 0: MIDI.ERROR = 6: EXIT SUB
    'If the mixer chip is CT1335....
    CASE 1
        'Reset the Left Channel variable
        LeftChannel% = 0
        'Read the byte containing the Voice volume.
        MasterVolume% = InternalReadMixer%(&HA)
        'Extract the actual volume setting from the byte.
        FOR I% = 1 TO 2
            BIT.STORAGE(I%) = InternalBitRead%(MasterVolume%, I%)
            IF BIT.STORAGE(I%) = 1 THEN
                InternalBitToggle LeftChannel%, I% - 1
            END IF
        NEXT I%
        'Convert the 2-bit value returned to a 5-bit value.
        LeftChannel% = LeftChannel% * 8
        IF LeftChannel% = 24 THEN LeftChannel% = 31
        'The CT1335 mixer chip only allows monaural volume control, so copy
        'the value of the Left Channel volume into the Right Channel variable.
        RightChannel% = LeftChannel%
    'If the mixer chip is CT1345....
    CASE 2
        InternalGetVol LeftChannel%, RightChannel%, &H4
    'If the mixer chip is CT1745....
    CASE 3
        InternalGetVol LeftChannel%, RightChannel%, &H32
END SELECT
END SUB

REM $STATIC
SUB InitMIDI
 'INIT-----Music-----
   DriversLoaded SBMIDI.INTERRUPT, SBSIM.INTERRUPT
    IF SBMIDI.INTERRUPT = 0 THEN
      PRINT "Error! The SBMIDI interrupt was not found!"
       DO
        PRINT "Warning: Leaving music enabled could cause unstable results!"
        INPUT "Disabled Music? Y/N "; InpMIDI$
         PRINT
         InpMIDI$ = MID$(UCASE$(LTRIM$(RTRIM$(InpMIDI$))), 1, 1)
       LOOP UNTIL InpMIDI$ = "Y" OR InpMIDI$ = "N"
        IF ImpMIDI$ = "Y" THEN MIDIFlag% = 1: DriverFlag% = 1 ELSE MIDIFlag% = 0: DriverFlag% = 0
     IF MIDIFlag% <> 0 THEN SBMIDI.INTERRUPT = &H80
    ELSE
     PRINT "SBMIDI interrupt found at: &h"; HEX$(SBMIDI.INTERRUPT)
     DriverFlag% = 1
     MIDIFlag% = 1
    END IF
 
    IF SBSIM.INTERRUPT = 0 THEN
      PRINT "Error! The SBSIM interrupt was not found!"
       DO
        PRINT "Warning: Leaving music enabled could cause unstable results!"
        INPUT "Disabled Music? Y/N "; InpSIM$
         PRINT
         InpSIM$ = MID$(UCASE$(LTRIM$(RTRIM$(InpSIM$))), 1, 1)
       LOOP UNTIL InpSIM$ = "Y" OR InpSIM$ = "N"
        IF ImpSIM$ = "Y" THEN MIDIFlag% = 1: DriverFlag% = 1 ELSE MIDIFlag% = 0: DriverFlag% = 0
      SBSIM.INTERRUPT = &H81
     ELSE
      PRINT "SBSIM interrupt found at: &h"; HEX$(SBSIM.INTERRUPT)
      DriverFlag% = 1
      MIDIFlag% = 1
     END IF

 IF MIDIFlag% <> 0 THEN
   DetectSettings SB.BASEPORT, SB.IRQ, SB.LODMA, SB.HIDMA, SB.CARDTYPE
   IF SB.CARDTYPE = 0 THEN
     SetCard 2
     PRINT "Soundcard not detect. Setting it to "; SoundCard$(6)
    ELSE
     PRINT SoundCard$(SB.CARDTYPE); " detected."
    END IF
   IF SB.BASEPORT = 0 THEN
     PRINT "BasePort not found, setting to default of &h220": SB.BASEPORT = &H220
    ELSE
     PRINT "BasePort found at: "; STR$(SB.BASEPORT)
    END IF
   IF SB.IRQ = 0 THEN
     PRINT "IRQ not detected, setting to default of 5": SB.IRQ = 5
    ELSE
     PRINT "IRQ detected to be: "; STR$(SB.IRQ)
    END IF
   IF SB.LODMA = 0 THEN
     PRINT "Low DMA not detected, setting to default of 1": SB.LODMA = 1
    ELSE
     PRINT "Low DMA detected to be: "; STR$(SB.LODMA)
    END IF
   IF SB.HIDMA = 0 AND SB.CARDTYPE = 6 THEN
     PRINT "High DMA not found, setting to default of 5": SB.HIDMA = 5
    ELSE
     IF SB.HIDMA = 0 THEN PRINT "High DMA not detected."
     IF SB.HIDMA <> 0 THEN PRINT "High DMA detected to be: "; STR$(SB.HIDMA)
    END IF
   PRINT "Mixer chip: "; MixerChip$
 END IF
 'INIT-----End Music-----
END SUB

REM $DYNAMIC
FUNCTION InternalBitRead% (Variable%, BitNum%)
VarSegment% = VARSEG(Variable%)
Offset% = VARPTR(Variable%)
DEF SEG = VarSegment%
InternalBitRead% = -((PEEK(Offset% + BitNum% \ 8) AND 2 ^ (BitNum% MOD 8)) > 0)
DEF SEG
END FUNCTION

SUB InternalBitSet (Variable%, BitNum%, OnOff%)
VarSegment% = VARSEG(Variable%)
Offset% = VARPTR(Variable%)
DEF SEG = VarSegment%
IF OnOff% THEN
    POKE Offset% + BitNum% \ 8, PEEK(Offset% + BitNum% \ 8) OR 2 ^ (BitNum% MOD 8)
ELSE
    POKE Offset% + BitNum% \ 8, PEEK(Offset% + BitNum% \ 8) AND 255 - 2 ^ (BitNum% MOD 8)
END IF
DEF SEG
END SUB

SUB InternalBitToggle (Variable%, BitNum%)
VarSegment% = VARSEG(Variable%)
Offset% = VARPTR(Variable%)
DEF SEG = VarSegment%
POKE Offset% + BitNum% \ 8, PEEK(Offset% + BitNum% \ 8) XOR 2 ^ (BitNum% MOD 8)
DEF SEG
END SUB

SUB InternalGetIntVector (IntNum%, Segment%, Offset%) STATIC
'If the code hasn't been loaded already, do it now.
IF GetIntVCodeLoaded% = 0 THEN
    asm$ = asm$ + CHR$(&H55)
    asm$ = asm$ + CHR$(&H89) + CHR$(&HE5)
    asm$ = asm$ + CHR$(&H8B) + CHR$(&H5E) + CHR$(&HA)
    asm$ = asm$ + CHR$(&H8A) + CHR$(&H7)
    asm$ = asm$ + CHR$(&HB4) + CHR$(&H35)
    asm$ = asm$ + CHR$(&HCD) + CHR$(&H21)
    asm$ = asm$ + CHR$(&H8C) + CHR$(&HC1)
    asm$ = asm$ + CHR$(&H89) + CHR$(&HDA)
    asm$ = asm$ + CHR$(&H8B) + CHR$(&H5E) + CHR$(&H8)
    asm$ = asm$ + CHR$(&H89) + CHR$(&HF)
    asm$ = asm$ + CHR$(&H8B) + CHR$(&H5E) + CHR$(&H6)
    asm$ = asm$ + CHR$(&H89) + CHR$(&H17)
    asm$ = asm$ + CHR$(&H5D)
    asm$ = asm$ + CHR$(&HCB)
    asm$ = asm$ + CHR$(&H34) + CHR$(&H0)
    asm$ = asm$ + CHR$(&H60)
    asm$ = asm$ + CHR$(&H23) + CHR$(&H0)
    GetIntVCodeLoaded% = 1
END IF
'Execute the code
DEF SEG = VARSEG(asm$)
CALL Absolute(IntNum%, Segment%, Offset%, SADD(asm$))
END SUB

SUB InternalGetVol (LeftChannel%, RightChannel%, Index%)
SELECT CASE MIXER.CHIP
    'If no mixer chip is present, exit
    CASE 0: EXIT SUB
    'If the mixer chip is CT1335....
    CASE 1
        'Reset the Left Channel variable
        LeftChannel% = 0
        'Read the byte containing the Volume.
        MasterVolume% = InternalReadMixer%(Index%)
        'Extract the actual volume setting from the byte.
        FOR I% = 1 TO 3
            BIT.STORAGE(I%) = InternalBitRead%(MasterVolume%, I%)
            IF BIT.STORAGE(I%) = 1 THEN
                InternalBitToggle LeftChannel%, I% - 1
            END IF
        NEXT I%
        'Convert the 3-bit value returned to a 5-bit value.
        LeftChannel% = LeftChannel% * 4
        IF LeftChannel% = 28 THEN LeftChannel% = 31
        'The CT1335 mixer chip only allows monaural volume control, so copy
        'the value of the Left Channel volume into the Right Channel variable.
        RightChannel% = LeftChannel%
    'If the mixer chip is CT1345....
    CASE 2
        'Reset the Left Channel variable
        LeftChannel% = 0
        'Reset the Right Channel variable
        RightChannel% = 0
        'Read the byte containing the Volume.
        MasterVolume% = InternalReadMixer%(Index%)
        'Extract the volume settings from the byte.
        FOR I% = 1 TO 3
            BIT.STORAGE(I%) = InternalBitRead%(MasterVolume%, I%)
            IF BIT.STORAGE(I%) THEN
                InternalBitToggle RightChannel%, I% - 1
            END IF
            BIT.STORAGE(I% + 4) = InternalBitRead%(MasterVolume%, I% + 4)
            IF BIT.STORAGE(I% + 4) THEN
                InternalBitToggle LeftChannel%, I% - 1
            END IF
        NEXT I%
        'Convert the 3-bit values returned to 5-bit values.
        LeftChannel% = LeftChannel% * 4
        IF LeftChannel% = 28 THEN LeftChannel% = 31
        RightChannel% = RightChannel% * 4
        IF LeftChannel% = 28 THEN LeftChannel% = 31
    'If the mixer chip is CT1745....
    CASE 3
        'Reset the Left Channel variable
        LeftChannel% = 0
        'Reset the Right Channel variable
        RightChannel% = 0
        'Read the byte containing the Left Volume.
        LMasterVolume% = InternalReadMixer%(Index%)
        'Read the byte containing the Right Volume.
        RMasterVolume% = InternalReadMixer%(Index% + 1)
        'Extract the left channel volume settings from the byte.
        FOR I% = 3 TO 7
            BIT.STORAGE(I%) = InternalBitRead%(LMasterVolume%, I%)
            IF BIT.STORAGE(I%) THEN
                InternalBitToggle LeftChannel%, I% - 3
            END IF
        NEXT I%
        'Extract the right channel volume settings from the byte.
        FOR I% = 3 TO 7
            BIT.STORAGE(I%) = InternalBitRead%(RMasterVolume%, I%)
            IF BIT.STORAGE(I%) THEN
                InternalBitToggle RightChannel%, I% - 3
            END IF
        NEXT I%
END SELECT
END SUB

FUNCTION InternalReadMixer% (Index%)
OUT SB.BASEPORT + 4, Index%
InternalReadMixer% = INP(SB.BASEPORT + 5)
END FUNCTION

SUB InternalSetVol (LeftChannel%, RightChannel%, Index%)
SELECT CASE MIXER.CHIP
    'If no mixer chip is present, exit
    CASE 0: EXIT SUB
    'If the mixer chip is CT1335....
    CASE 1
        'If the left channel volume is not in the proper range, don't
        'change any settings.  NOTE: Since cards with the CT1335 mixer chip
        'only support monaural sound, the right channel volume setting is
        'ignored.
        IF LeftChannel% > -1 AND LeftChannel% < 32 THEN
            'Convert the 5-bit Left Channel value to a 3-bit value.
            LeftChannel% = LeftChannel% \ 4
            IF LeftChannel% > 7 THEN LeftChannel% = 7
            'Get the current volume setting.
            Volume% = InternalReadMixer%(2)
            'Extract the volume setting.
            FOR I% = 0 TO 2
                BIT.STORAGE(I%) = InternalBitRead%(LeftChannel%, I%)
                IF BIT.STORAGE(I%) THEN
                    InternalBitSet Volume%, I% + 1, 1
                END IF
            NEXT I%
            'Write the new volume setting
            InternalWriteMixer Index%, Volume%
        END IF
    'If the mixer chip is CT1345....
    CASE 2
        'Get the current volume setting.
        Volume% = InternalReadMixer%(Index%)
        'Convert the 5-bit Left Channel value to a 3-bit value.
        LeftChannel% = LeftChannel% \ 4
        IF LeftChannel% > 7 THEN LeftChannel% = 7
        'Convert the 5-bit Right Channel value to a 3-bit value.
        RightChannel% = RightChannel% \ 4
        IF RightChannel% > 7 THEN RightChannel% = 7
        'If Reverse Stereo is enabled, swap the variables.
        IF REVERSE.STEREO THEN SWAP LeftChannel%, RightChannel%
        'Modify the volume settings as necessary.
        FOR I% = 0 TO 2
            'First, the right channel....
            BIT.STORAGE(I%) = InternalBitRead%(RightChannel%, I%)
            IF RightChannel% > -1 AND RightChannel% < 32 THEN
                IF BIT.STORAGE(I%) THEN BitVal% = 1 ELSE BitVal% = 0
                InternalBitSet Volume%, I% + 1, BitVal%
            END IF
            'Now, the left channel....
            BIT.STORAGE(I%) = InternalBitRead%(LeftChannel%, I%)
            IF LeftChannel% > -1 AND LeftChannel% < 32 THEN
                IF BIT.STORAGE(I%) THEN BitVal% = 1 ELSE BitVal% = 0
                InternalBitSet Volume%, I% + 5, BitVal%
            END IF
        NEXT I%
        'Write the new volume settings
        InternalWriteMixer Index%, Volume%
    'If the mixer chip is CT1745....
    CASE 3
        'Get the current left channel volume setting.
        LVolume% = InternalReadMixer%(Index%)
        'Get the current right channel volume setting.
        RVolume% = InternalReadMixer%(Index% + 1)
        'If Reverse Stereo is enabled, swap the variables.
        IF REVERSE.STEREO THEN SWAP LeftChannel%, RightChannel%
        'Modify the volume settings as necessary.
        FOR I% = 0 TO 4
            'First, the right channel....
            BIT.STORAGE(I%) = InternalBitRead%(RightChannel%, I%)
            IF RightChannel% > -1 AND RightChannel% < 32 THEN
                IF BIT.STORAGE(I%) THEN BitVal% = 1 ELSE BitVal% = 0
                InternalBitSet RVolume%, I% + 3, BitVal%
            END IF
            'Now, the left channel....
            BIT.STORAGE(I%) = InternalBitRead%(LeftChannel%, I%)
            IF LeftChannel% > -1 AND LeftChannel% < 32 THEN
                IF BIT.STORAGE(I%) THEN BitVal% = 1 ELSE BitVal% = 0
                InternalBitSet LVolume%, I% + 3, BitVal%
            END IF
        NEXT I%
        'Write the new volume settings
        InternalWriteMixer Index%, LVolume%
        InternalWriteMixer Index% + 1, RVolume%
END SELECT

END SUB

SUB InternalWriteMixer (Index%, Value%)
OUT SB.BASEPORT + 4, Index%
OUT SB.BASEPORT + 5, Value%
END SUB

SUB LoadAndPlayLarge (Filename$) STATIC
IF SBSIM.INTERRUPT < &H80 AND SENSITIVE <> 0 THEN MIDI.ERROR = 5: EXIT SUB
'See if an extension was supplied, and if not, add one.
IF INSTR(Filename$, ".") = 0 THEN Filename$ = Filename$ + ".MID"
'Open the file
FF% = FREEFILE
OPEN Filename$ FOR BINARY AS #FF%
'If the file contains no data, return an error
FileLen& = LOF(FF%)
CLOSE #FF%
IF FileLen& = 0 THEN
    KILL Filename$
    MIDI.ERROR = 1
    EXIT SUB
END IF
'The sound driver needs an ASCIIZ string (a string that ends with
'character 0) for a file name.
IF RIGHT$(Filename$, 1) <> CHR$(0) THEN Filename$ = Filename$ + CHR$(0)
'Initialize the MIDI driver and load the file in memory
IF Lasm$ = "" THEN
        Lasm$ = Lasm$ + CHR$(&H55)
        Lasm$ = Lasm$ + CHR$(&H89) + CHR$(&HE5)
        Lasm$ = Lasm$ + CHR$(&H8B) + CHR$(&H5E) + CHR$(&H8)
        Lasm$ = Lasm$ + CHR$(&H8B) + CHR$(&H7)
        Lasm$ = Lasm$ + CHR$(&H8B) + CHR$(&H5E) + CHR$(&H6)
        Lasm$ = Lasm$ + CHR$(&H8B) + CHR$(&H17)
        Lasm$ = Lasm$ + CHR$(&HBB) + CHR$(&H0) + CHR$(&H5)
        Lasm$ = Lasm$ + CHR$(&HCD) + CHR$(SBSIM.INTERRUPT)
        Lasm$ = Lasm$ + CHR$(&H5D)
        Lasm$ = Lasm$ + CHR$(&HCB)
END IF
DEF SEG = VARSEG(Lasm$)
Offset% = SADD(Lasm$)
CALL Absolute(SADD(Filename$), VARSEG(Filename$), Offset%)

'Start the music!!
IF Pasm$ = "" THEN
        Pasm$ = Pasm$ + CHR$(&HBB) + CHR$(&H1) + CHR$(&H5)
        Pasm$ = Pasm$ + CHR$(&HCD) + CHR$(SBSIM.INTERRUPT)
        Pasm$ = Pasm$ + CHR$(&HCB)
END IF
IF SOUND.DISABLED = 0 THEN
    DEF SEG = VARSEG(Pasm$)
    Offset% = SADD(Pasm$)
    CALL Absolute(Offset%)
    MIDI.PLAYTIME = TIMER
END IF
MIDI.LOADED = 1
MIDI.ERROR = 0
END SUB

'LoadAndPlayMIDI - Loads a MIDI file into memory and plays it
SUB LoadAndPlayMIDI (Filename$, MIDISegment%, MIDIOffset%) STATIC
IF SBMIDI.INTERRUPT < &H80 AND SENSITIVE <> 0 THEN MIDI.ERROR = 4: EXIT SUB
'See if an extension was supplied, and if not, add one.
IF INSTR(Filename$, ".") = 0 THEN Filename$ = Filename$ + ".MID"
'Open the file
FF% = FREEFILE
OPEN Filename$ FOR BINARY AS #FF%
FileLen& = LOF(FF%)
CLOSE #FF%
'If the file is empty, delete it and exit now.
IF FileLen& = 0 THEN KILL Filename$: MIDI.ERROR = 1: EXIT SUB
'If the file is too large, exit now.
IF FileLen& > 65536 THEN MIDI.ERROR = 2: EXIT SUB
'Make the filename an ASCIIZ string.
Filename$ = Filename$ + CHR$(0)
'Check if the MIDI loading code has already been loaded;
'if not, do it now.
IF asm1$ = "" THEN
        asm1$ = asm1$ + CHR$(&H1E)
        asm1$ = asm1$ + CHR$(&H55)
        asm1$ = asm1$ + CHR$(&H89) + CHR$(&HE5)
        asm1$ = asm1$ + CHR$(&HB8) + CHR$(&H0) + CHR$(&H3D)
        asm1$ = asm1$ + CHR$(&H8B) + CHR$(&H5E) + CHR$(&HE)
        asm1$ = asm1$ + CHR$(&H8B) + CHR$(&H17)
        asm1$ = asm1$ + CHR$(&H8B) + CHR$(&H5E) + CHR$(&H10)
        asm1$ = asm1$ + CHR$(&H8E) + CHR$(&H1F)
        asm1$ = asm1$ + CHR$(&HCD) + CHR$(&H21)
        asm1$ = asm1$ + CHR$(&H89) + CHR$(&HC6)
        asm1$ = asm1$ + CHR$(&HB4) + CHR$(&H3F)
        asm1$ = asm1$ + CHR$(&H8B) + CHR$(&H5E) + CHR$(&H8)
        asm1$ = asm1$ + CHR$(&H8B) + CHR$(&HF)
        asm1$ = asm1$ + CHR$(&H8B) + CHR$(&H5E) + CHR$(&HA)
        asm1$ = asm1$ + CHR$(&H8B) + CHR$(&H17)
        asm1$ = asm1$ + CHR$(&H8B) + CHR$(&H5E) + CHR$(&HC)
        asm1$ = asm1$ + CHR$(&H8E) + CHR$(&H1F)
        asm1$ = asm1$ + CHR$(&H89) + CHR$(&HF3)
        asm1$ = asm1$ + CHR$(&HCD) + CHR$(&H21)
        asm1$ = asm1$ + CHR$(&HB4) + CHR$(&H3E)
        asm1$ = asm1$ + CHR$(&HCD) + CHR$(&H21)
        asm1$ = asm1$ + CHR$(&H5D)
        asm1$ = asm1$ + CHR$(&H1F)
        asm1$ = asm1$ + CHR$(&HCA) + CHR$(&HA) + CHR$(&H0)
END IF
'Call the assembly language routine.
DEF SEG = VARSEG(asm1$)
CALL Absolute(VARSEG(Filename$), SADD(Filename$), MIDISegment%, MIDIOffset%, &HFFFF, SADD(asm1$))
'Check to see if the MIDI playing code has previously been loaded.
'If not, load it now.
IF asm2$ = "" THEN
        'Load the machine codes into a string.
        asm2$ = asm2$ + CHR$(&H55)
        asm2$ = asm2$ + CHR$(&H89) + CHR$(&HE5)
        asm2$ = asm2$ + CHR$(&H8B) + CHR$(&H5E) + CHR$(&H8)
        asm2$ = asm2$ + CHR$(&H8B) + CHR$(&H17)
        asm2$ = asm2$ + CHR$(&H8B) + CHR$(&H5E) + CHR$(&H6)
        asm2$ = asm2$ + CHR$(&H8B) + CHR$(&H7)
        asm2$ = asm2$ + CHR$(&HBB) + CHR$(&H4) + CHR$(&H0)
        asm2$ = asm2$ + CHR$(&HCD) + CHR$(SBMIDI.INTERRUPT)
        asm2$ = asm2$ + CHR$(&HBB) + CHR$(&H5) + CHR$(&H0)
        asm2$ = asm2$ + CHR$(&HCD) + CHR$(SBMIDI.INTERRUPT)
        asm2$ = asm2$ + CHR$(&H5D)
        asm2$ = asm2$ + CHR$(&HCA) + CHR$(&H4) + CHR$(&H0)
END IF
IF SOUND.DISABLED = 0 THEN
    'Call the machine language routine to play the music.
    DEF SEG = VARSEG(asm2$)
    CALL Absolute(MIDISegment%, MIDIOffset%, SADD(asm2$))
    'Start the MIDI timer.
    MIDI.PLAYTIME = TIMER
END IF
MIDI.ERROR = 0
END SUB

SUB LoadLarge (Filename$) STATIC
IF SBSIM.INTERRUPT < &H80 AND SENSITIVE <> 0 THEN MIDI.ERROR = 5: EXIT SUB
'See if an extension was supplied, and if not, add one.
IF INSTR(Filename$, ".") = 0 THEN Filename$ = Filename$ + ".MID"
'Open the file
FF% = FREEFILE
OPEN Filename$ FOR BINARY AS #FF%
'If the file contains no data, return an error
FileLen& = LOF(FF%)
CLOSE #FF%
IF FileLen& = 0 THEN
    KILL Filename$
    MIDI.ERROR = 1
    EXIT SUB
END IF
'The sound driver needs an ASCIIZ string (a string that ends with
'character 0) for a file name.
IF RIGHT$(Filename$, 1) <> CHR$(0) THEN Filename$ = Filename$ + CHR$(0)
'Initialize the MIDI driver and load the file in memory
IF asm$ = "" THEN
        asm$ = asm$ + CHR$(&H55)
        asm$ = asm$ + CHR$(&H89) + CHR$(&HE5)
        asm$ = asm$ + CHR$(&H8B) + CHR$(&H5E) + CHR$(&H8)
        asm$ = asm$ + CHR$(&H8B) + CHR$(&H7)
        asm$ = asm$ + CHR$(&H8B) + CHR$(&H5E) + CHR$(&H6)
        asm$ = asm$ + CHR$(&H8B) + CHR$(&H17)
        asm$ = asm$ + CHR$(&HBB) + CHR$(&H0) + CHR$(&H5)
        asm$ = asm$ + CHR$(&HCD) + CHR$(SBSIM.INTERRUPT)
        asm$ = asm$ + CHR$(&H5D)
        asm$ = asm$ + CHR$(&HCB)
END IF
DEF SEG = VARSEG(asm$)
Offset% = SADD(asm$)
CALL Absolute(SADD(Filename$), VARSEG(Filename$), Offset%)
MIDI.LOADED = 1
MIDI.ERROR = 0
END SUB

'LoadMIDI - loads a MIDI file into memory
SUB LoadMIDI (Filename$, MIDISegment%, MIDIOffset%) STATIC
IF SBMIDI.INTERRUPT < &H80 AND SENSITIVE <> 0 THEN MIDI.ERROR = 4: EXIT SUB
'See if an extension was supplied, and if not, add one.
IF INSTR(Filename$, ".") = 0 THEN Filename$ = Filename$ + ".MID"
'Open the file
FF% = FREEFILE
OPEN Filename$ FOR BINARY AS #FF%
FileLen& = LOF(FF%)
CLOSE #FF%
'If the file is empty, delete it and exit now.
IF FileLen& = 0 THEN KILL Filename$: MIDI.ERROR = 1: EXIT SUB
'If the file is too large, exit now.
IF FileLen& > 65536 THEN MIDI.ERROR = 2: EXIT SUB
'Make the filename an ASCIIZ string.
Filename$ = Filename$ + CHR$(0)
'Check if the assembly language code has already been loaded;
'if not, do it now.
IF ask$ = "" THEN
        asm$ = asm$ + CHR$(&H1E)
        asm$ = asm$ + CHR$(&H55)
        asm$ = asm$ + CHR$(&H89) + CHR$(&HE5)
        asm$ = asm$ + CHR$(&HB8) + CHR$(&H0) + CHR$(&H3D)
        asm$ = asm$ + CHR$(&H8B) + CHR$(&H5E) + CHR$(&HE)
        asm$ = asm$ + CHR$(&H8B) + CHR$(&H17)
        asm$ = asm$ + CHR$(&H8B) + CHR$(&H5E) + CHR$(&H10)
        asm$ = asm$ + CHR$(&H8E) + CHR$(&H1F)
        asm$ = asm$ + CHR$(&HCD) + CHR$(&H21)
        asm$ = asm$ + CHR$(&H89) + CHR$(&HC6)
        asm$ = asm$ + CHR$(&HB4) + CHR$(&H3F)
        asm$ = asm$ + CHR$(&H8B) + CHR$(&H5E) + CHR$(&H8)
        asm$ = asm$ + CHR$(&H8B) + CHR$(&HF)
        asm$ = asm$ + CHR$(&H8B) + CHR$(&H5E) + CHR$(&HA)
        asm$ = asm$ + CHR$(&H8B) + CHR$(&H17)
        asm$ = asm$ + CHR$(&H8B) + CHR$(&H5E) + CHR$(&HC)
        asm$ = asm$ + CHR$(&H8E) + CHR$(&H1F)
        asm$ = asm$ + CHR$(&H89) + CHR$(&HF3)
        asm$ = asm$ + CHR$(&HCD) + CHR$(&H21)
        asm$ = asm$ + CHR$(&HB4) + CHR$(&H3E)
        asm$ = asm$ + CHR$(&HCD) + CHR$(&H21)
        asm$ = asm$ + CHR$(&H5D)
        asm$ = asm$ + CHR$(&H1F)
        asm$ = asm$ + CHR$(&HCA) + CHR$(&HA) + CHR$(&H0)
END IF
'Call the assembly language routine.
DEF SEG = VARSEG(asm$)
CALL Absolute(VARSEG(Filename$), SADD(Filename$), MIDISegment%, MIDIOffset%, &HFFFF, SADD(asm$))
MIDI.ERROR = 0
END SUB

'MIDIError - Translates a QMIDI error code into text
FUNCTION MIDIError$
SELECT CASE MIDI.ERROR
        CASE 0: MIDIError$ = "NO ERROR"
        CASE 1: MIDIError$ = "FILE CONTAINS NO DATA"
        CASE 2: MIDIError$ = "FILE IS TOO LARGE"
        CASE 3: MIDIError$ = "NO MIDI FILE PLAYING"
        CASE 4: MIDIError$ = "INVALID SBMIDI INTERRUPT"
        CASE 5: MIDIError$ = "INVALID SBSIM INTERRUPT"
        CASE 6: MIDIError$ = "NO MIXER CHIP"
        CASE 7: MIDIError$ = "COULD NOT DETECT SOUND CARD"
        CASE 8: MIDIError$ = "FEATURE UNAVAILABLE"
        CASE 9: MIDIError$ = "FILE IS CORRUPT"
        CASE 10: MIDIError$ = "INVALID SOUND CARD TYPE"
        CASE ELSE: MIDIError$ = "UNKNOWN ERROR"
END SELECT
END FUNCTION

'MixerChip - Returns name of mixer chip used by sound card
FUNCTION MixerChip$
SELECT CASE MIXER.CHIP
    CASE 0: MixerChip$ = "No Mixer Chip Detected"
    CASE 1: MixerChip$ = "CT1335"
    CASE 2: MixerChip$ = "CT1345"
    CASE 3: MixerChip$ = "CT1745"
    CASE ELSE: MixerChip$ = "Unknown"
END SELECT
END FUNCTION

'PauseMIDI - Pauses a MIDI file that is currently playing
SUB PauseMIDI STATIC
IF SBSIM.INTERRUPT < &H80 AND SENSITIVE <> 0 THEN MIDI.ERROR = 5: EXIT SUB
'If no MIDI file is playing, exit now
IF MIDI.PLAYTIME = 0 THEN
    MIDI.ERROR = 3
    EXIT SUB
END IF
'Pause the music.
IF asm$ = "" THEN
        asm$ = asm$ + CHR$(&HBB) + CHR$(&H3) + CHR$(&H5)
        asm$ = asm$ + CHR$(&HCD) + CHR$(SBSIM.INTERRUPT)
        asm$ = asm$ + CHR$(&HCB)
END IF
DEF SEG = VARSEG(asm$)
Offset% = SADD(asm$)
CALL Absolute(Offset%)
'Save the number of seconds the MIDI file has been playing.
PAUSED = TimeMIDI!
'If it hasn't been playing long enough for TimeMIDI! to return
'a value greater than 0, change PAUSED to a tiny positive value.
IF PAUSED = 0! THEN PAUSED = .00001
'Indicate that the file has stopped playing.
MIDI.PLAYTIME = 0
MIDI.ERROR = 0
END SUB

SUB PlayLarge STATIC
IF SBSIM.INTERRUPT < &H80 AND SENSITIVE <> 0 THEN MIDI.ERROR = 5: EXIT SUB
'If no MIDI file is loaded, exit now
IF MIDI.LOADED = 0 THEN EXIT SUB
'Start the music!!
IF asm$ = "" THEN
        asm$ = asm$ + CHR$(&HBB) + CHR$(&H1) + CHR$(&H5)
        asm$ = asm$ + CHR$(&HCD) + CHR$(SBSIM.INTERRUPT)
        asm$ = asm$ + CHR$(&HCB)
END IF
IF SOUND.DISABLED = 0 THEN
    DEF SEG = VARSEG(asm$)
    Offset% = SADD(asm$)
    CALL Absolute(Offset%)
    MIDI.PLAYTIME = TIMER
END IF
END SUB

'PlayMIDI - Begins playing a MIDI file in the background.
SUB PlayMIDI (MIDISegment%, MIDIOffset%) STATIC
IF SBMIDI.INTERRUPT < &H80 AND SENSITIVE <> 0 THEN MIDI.ERROR = 4: EXIT SUB
'Check to see if the MIDI playing code has previously been loaded.
'If not, load it now.
IF asm$ = "" THEN
        'Load the machine codes into a string.
        asm$ = asm$ + CHR$(&H55)
        asm$ = asm$ + CHR$(&H89) + CHR$(&HE5)
        asm$ = asm$ + CHR$(&H8B) + CHR$(&H5E) + CHR$(&H8)
        asm$ = asm$ + CHR$(&H8B) + CHR$(&H17)
        asm$ = asm$ + CHR$(&H8B) + CHR$(&H5E) + CHR$(&H6)
        asm$ = asm$ + CHR$(&H8B) + CHR$(&H7)
        asm$ = asm$ + CHR$(&HBB) + CHR$(&H4) + CHR$(&H0)
        asm$ = asm$ + CHR$(&HCD) + CHR$(&H80)
        asm$ = asm$ + CHR$(&HBB) + CHR$(&H5) + CHR$(&H0)
        asm$ = asm$ + CHR$(&HCD) + CHR$(&H80)
        asm$ = asm$ + CHR$(&H5D)
        asm$ = asm$ + CHR$(&HCA) + CHR$(&H4) + CHR$(&H0)
END IF
IF SOUND.DISABLED = 0 THEN
    'Call the machine language routine to play the music.
    DEF SEG = VARSEG(asm$)
    Offset% = SADD(asm$)
    CALL Absolute(MIDISegment%, MIDIOffset%, Offset%)
    'Start the MIDI timer.
    MIDI.PLAYTIME = TIMER
END IF
MIDI.ERROR = 0
END SUB

'ResumeMIDI - Starts playing a MIDI file after it has been paused
SUB ResumeMIDI STATIC
IF SBSIM.INTERRUPT < &H80 AND SENSITIVE <> 0 THEN MIDI.ERROR = 5: EXIT SUB
'If no MIDI file is paused, exit now
IF PAUSED = 0! THEN EXIT SUB
'Resume playing.
IF asm$ = "" THEN
        asm$ = ""
        asm$ = asm$ + CHR$(&HBB) + CHR$(&H4) + CHR$(&H5)
        asm$ = asm$ + CHR$(&HCD) + CHR$(SBSIM.INTERRUPT)
        asm$ = asm$ + CHR$(&HCB)
END IF
DEF SEG = VARSEG(asm$)
Offset% = SADD(asm$)
CALL Absolute(Offset%)
MIDI.PLAYTIME = TIMER - PAUSED
PAUSED = 0!
MIDI.ERROR = 0
END SUB

'SetBass - Set the Bass level
SUB SetBass (LeftChannel%, RightChannel%)
LC% = LeftChannel%
RC% = RightChannel%
SELECT CASE MIXER.CHIP
    'If no mixer chip is present, return an error
    CASE 0
        MIDI.ERROR = 6
        EXIT SUB
    'If the mixer chip is CT1335 or CT1345, return an error, becuase
    'these chips do not support Treble and Bass control.
    CASE 1, 2
        MIDI.ERROR = 8
        EXIT SUB
    'If the mixer chip is CT1745....
    CASE 3
        'Get the current bass values.
        LBass% = InternalReadMixer%(&H46)
        RBass% = InternalReadMixer%(&H47)
        'Convert the 5-bit settings to 4-bit values.
        LC% = LC% \ 2
        IF LC% > 15 THEN LC% = 15
        RC% = RC% \ 2
        IF RC% > 15 THEN RC% = 15
        'Place the settings into new variables.
        FOR I% = 0 TO 3
            BIT.STORAGE(I%) = InternalBitRead%(LC%, I%)
            IF BIT.STORAGE(I%) THEN BitVal% = 1 ELSE BitVal% = 0
            InternalBitSet LBass%, I% + 4, BitVal%
            BIT.STORAGE(I%) = InternalBitRead%(RC%, I%)
            IF BIT.STORAGE(I%) THEN BitVal% = 1 ELSE BitVal% = 0
            InternalBitSet RBass%, I% + 4, BitVal%
        NEXT I%
        'Write the new settings.
        InternalWriteMixer &H46, LBass%
        InternalWriteMixer &H47, RBass%
        MIDI.ERROR = 0
END SELECT
END SUB

REM $STATIC
'SetCard - Properly Sets the Sound Card Type
SUB SetCard (CardType%)
'If the card type is invalid, return an error
IF CardType% < 1 OR CardType% > 8 THEN
    MIDI.ERROR = 10
    EXIT SUB
END IF
'Otherwise, set the new card type.
SB.CARDTYPE = CardType%
'Then, set the mixer chip accordingly.
SELECT CASE CardType%
    'If the card is a Sound Blaster 1.0/1.5 or equivalent....
    CASE 1
        'Return an error.
        MIDI.ERROR = 6
        'If sensitive error checking is on, disable mixer operations and
        'exit.
        IF SENSITIVE THEN
            MIXER.CHIP = 0
            EXIT SUB
        'Otherwise, set the earliest mixer chip and continue.
        ELSE
            MIXER.CHIP = 1
        END IF
    'If the card is a Sound Blaster 2.0/2.5 or equivalent....
    CASE 3
        'There are two different kinds of SB 2.0 cards: the regular SB2,
        'and the SB2CD.  The SB2CD has a mixer chip (the CT1335), whereas
        'the SB 2.0 does not.  The way to tell them apart is that the
        'Sound Blaster 2.0 uses Base Ports 220h and 240h, and the SB2CD
        'uses ports 250h and 260h.
        '
        'Assume the sound card is an SB2CD for now...
        MIXER.CHIP = 1
        'If the card is defined as an SB 2.0, not an SB 2.0 CD, and
        'sensitive error checking is on, disable mixer operations.
        IF (BasePort% = &H220 OR BasePort% = &H240) AND SENSITIVE <> 0 THEN
            MIXER.CHIP = 0
        END IF
        MIDI.ERROR = 0
    'If the card is a Sound Blaster Pro, assume chip CT1345
    CASE 2, 4, 5
        MIXER.CHIP = 2
        MIDI.ERROR = 0
    'If the card is a Sound Blaster 16 or later, assume chip CT1745
    CASE ELSE
        MIXER.CHIP = 3
        MIDI.ERROR = 0
END SELECT
END SUB

REM $DYNAMIC
'SetMaster - Changes the Master volume level
SUB SetMaster (LeftChannel%, RightChannel%)
SELECT CASE MIXER.CHIP
    'If no mixer chip is present, return an error
    CASE 0: MIDI.ERROR = 6: EXIT SUB
    'If the mixer chip is CT1335....
    CASE 1
        LC% = LeftChannel%: RC% = RightChannel%
        InternalSetVol LC%, RC%, 2
        MIDI.ERROR = 0
    'If the mixer chip is CT1345....
    CASE 2
        LC% = LeftChannel%: RC% = RightChannel%
        InternalSetVol LC%, RC%, &H22
        MIDI.ERROR = 0
    'If the mixer chip is CT1745....
    CASE 3
        LC% = LeftChannel%: RC% = RightChannel%
        InternalSetVol LC%, RC%, &H30
        MIDI.ERROR = 0
END SELECT
END SUB

SUB SetMIDI (LeftChannel%, RightChannel%)
SELECT CASE MIXER.CHIP
    'If no mixer chip is present, return an error
    CASE 0: MIDI.ERROR = 6: EXIT SUB
    'If the mixer chip is CT1335....
    CASE 1
        LC% = LeftChannel%: RC% = RightChannel%
        InternalSetVol LC%, RC%, 6
        MIDI.ERROR = 0
    'If the mixer chip is CT1345....
    CASE 2
        LC% = LeftChannel%: RC% = RightChannel%
        InternalSetVol LC%, RC%, &H26
        MIDI.ERROR = 0
    'If the mixer chip is CT1745....
    CASE 3
        LC% = LeftChannel%: RC% = RightChannel%
        InternalSetVol LC%, RC%, &H34
        MIDI.ERROR = 0
END SELECT
END SUB

'SetTreble - Set the Treble level
SUB SetTreble (LeftChannel%, RightChannel%)
LC% = LeftChannel%
RC% = RightChannel%
SELECT CASE MIXER.CHIP
    'If no mixer chip is present, return an error
    CASE 0
        MIDI.ERROR = 6
        EXIT SUB
    'If the mixer chip is CT1335 or CT1345, return an error, becuase
    'these chips do not support Treble and Bass control.
    CASE 1, 2
        MIDI.ERROR = 8
        EXIT SUB
    'If the mixer chip is CT1745....
    CASE 3
        'Get the current treble values.
        LTreble% = InternalReadMixer%(&H44)
        RTreble% = InternalReadMixer%(&H45)
        'Convert the 5-bit settings to 4-bit values.
        LC% = LC% \ 2
        IF LC% > 15 THEN LC% = 15
        RC% = RC% \ 2
        IF RC% > 15 THEN RC% = 15
        'Place the settings into new variables.
        FOR I% = 0 TO 3
            BIT.STORAGE(I%) = InternalBitRead%(LC%, I%)
            IF BIT.STORAGE(I%) THEN BitVal% = 1 ELSE BitVal% = 0
            InternalBitSet LTreble%, I% + 4, BitVal%
            BIT.STORAGE(I%) = InternalBitRead%(RC%, I%)
            IF BIT.STORAGE(I%) THEN BitVal% = 1 ELSE BitVal% = 0
            InternalBitSet RTreble%, I% + 4, BitVal%
        NEXT I%
        'Write the new settings.
        InternalWriteMixer &H44, LTreble%
        InternalWriteMixer &H45, RTreble%
        MIDI.ERROR = 0
END SELECT
END SUB

SUB SetVoice (LeftChannel%, RightChannel%)
LC% = LeftChannel%
RC% = RightChannel%
SELECT CASE MIXER.CHIP
    'If no mixer chip is present, return an error
    CASE 0: MIDI.ERROR = 6: EXIT SUB
    'If the mixer chip is CT1335....
    CASE 1
        'If the left channel volume is not in the proper range, don't
        'change any settings.  NOTE: Since cards with the CT1335 mixer chip
        'only support monaural sound, the right channel volume setting is
        'ignored.
        IF LC% > -1 AND LC% < 32 THEN
            'Convert the 5-bit Left Channel value to a 2-bit value.
            LC% = LC% \ 8
            IF LC% > 3 THEN LC% = 3
            'Get the current volume setting.
            Volume% = InternalReadMixer%(&HA)
            'Extract the volume setting.
            FOR I% = 0 TO 1
                BIT.STORAGE(I%) = InternalBitRead%(LC%, I%)
                IF BIT.STORAGE(I%) THEN BitVal% = 1 ELSE BitVal% = 0
                InternalBitSet Volume%, I% + 1, 0
            NEXT I%
            'Write the new volume setting
            InternalWriteMixer &HA, Volume%
        END IF
    'If the mixer chip is CT1345....
    CASE 2
        InternalSetVol LC%, RC%, &H4
    'If the mixer chip is CT1745....
    CASE 3
        InternalSetVol LC%, RC%, &H32
END SELECT
END SUB

'SoundCard - Translates card type into text
FUNCTION SoundCard$ (CardType%)
SELECT CASE CardType%
    CASE 1: SoundCard$ = "Sound Blaster 1.0/1.5"
    CASE 2: SoundCard$ = "Sound Blaster Pro"
    CASE 3: SoundCard$ = "Sound Blaster 2.0/2.0CD"
    CASE 4: SoundCard$ = "Sound Blaster Pro 2"
    CASE 5: SoundCard$ = "Sound Blaster Pro 2 (Microchannel Version)"
    CASE 6: SoundCard$ = "Sound Blaster 16/16 ASP/AWE 32"
    CASE 7, 8: SoundCard$ = "Unknown (Probably SB32/AWE64)"
    CASE ELSE: SoundCard$ = "Unknown"
END SELECT
END FUNCTION

'StopMIDI - Stops playing MIDI file
SUB StopMIDI STATIC
IF SBMIDI.INTERRUPT < &H80 AND SENSITIVE <> 0 THEN MIDI.ERROR = 4: EXIT SUB
'Stop the music!!
IF asm$ = "" THEN
    asm$ = asm$ + CHR$(&HBB) + CHR$(&H4) + CHR$(&H0)
    asm$ = asm$ + CHR$(&HCD) + CHR$(SBMIDI.INTERRUPT)
    asm$ = asm$ + CHR$(&HCB)
END IF
IF MIDI.PLAYTIME THEN
    DEF SEG = VARSEG(asm$)
    Offset% = SADD(asm$)
    CALL Absolute(Offset%)
    MIDI.ERROR = 0
ELSE
    MIDI.ERROR = 3
END IF
'No MIDI file is playing, so reset the timer
MIDI.PLAYTIME = 0
END SUB

FUNCTION TimeMIDI!
'If a MIDI file is paused, lock the current playing time
IF PAUSED > 0! THEN
    TimeMIDI! = PAUSED
    MIDI.ERROR = 0
'If a MIDI file is playing, carry out the timing routine
ELSEIF MIDI.PLAYTIME THEN
    'Get the current time
    CurrentTime! = TIMER
    'If midnight has come since the MIDI file started playing, change
    'CurrentTime! accordingly
    IF CurrentTime! - MIDI.PLAYTIME < 0 THEN
        CurrentTime! = 86400 + CurrentTime!
    END IF
    'Get the final result
    TimeMIDI! = CurrentTime! - MIDI.PLAYTIME
    MIDI.ERROR = 0
ELSE
    MIDI.ERROR = 3
END IF
END FUNCTION

