// SHAMLib

// SCR_MODE.H
// For Setting and ReSetting Screen Modes

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <math.h>

#define VIDEO_INT           0x10
#define SET_MODE            0x00
#define MODE_13H			0x13
#define MODE_3H		        0x03

#define MISC_OUTPUT         0x03c2    /* VGA misc. output register */
#define SC_INDEX            0x03c4    /* VGA sequence controller */
#define SC_DATA             0x03c5
#define PALETTE_INDEX       0x03c8    /* VGA digital-to-analog converter */
#define PALETTE_DATA        0x03c9
#define CRTC_INDEX          0x03d4    /* VGA CRT controller */

#define MAP_MASK            0x02      /* Sequence controller registers */
#define MEMORY_MODE         0x04

#define H_TOTAL             0x00      /* CRT controller registers */
#define H_DISPLAY_END       0x01
#define H_BLANK_START       0x02
#define H_BLANK_END         0x03
#define H_RETRACE_START     0x04
#define H_RETRACE_END       0x05
#define V_TOTAL             0x06
#define OVERFLOW            0x07
#define MAX_SCAN_LINE       0x09
#define V_RETRACE_START     0x10
#define V_RETRACE_END       0x11
#define V_DISPLAY_END       0x12
#define OFFSET              0x13
#define UNDERLINE_LOCATION  0x14
#define V_BLANK_START       0x15
#define V_BLANK_END         0x16
#define MODE_CONTROL        0x17

#define NUM_COLORS          256       /* number of colors in mode 0x13 */

/* macro to write a word to a port */
#define word_out(port,register,value) \
  outpw(port,(((word)value<<8) + register))

typedef unsigned char  byte;
typedef unsigned short word;
typedef unsigned long  dword;

byte *VGART=(byte *)0xA0000000L;        /* this points to video memory. */
word screen_width, screen_height;

void set_mode(byte mode);
void initgraph(int width=320, int height=240);
void endgraph(void);

// SET_MODE
// Sets screen mode to 'mode'
void set_mode(byte mode)
{
  union REGS regs;

  regs.h.ah = SET_MODE;
  regs.h.al = mode;
  int86(VIDEO_INT, &regs, &regs);
}

// INITGRAPH
// Sets screen mode to mode 13h
void initgraph(int width, int height)
{
 set_mode(MODE_13H);

  word i;
  dword *ptr=(dword *)VGART;

  /* turn off chain-4 mode */
  word_out(SC_INDEX, MEMORY_MODE,0x06);

  /* set map mask to all 4 planes for screen clearing */
  word_out(SC_INDEX, MAP_MASK, 0xff);

  /* clear all 256K of memory */
  for(i=0;i<0x4000;i++)
	*ptr++ = 0;

  /* turn off long mode */
  word_out(CRTC_INDEX, UNDERLINE_LOCATION, 0x00);

  /* turn on byte mode */
  word_out(CRTC_INDEX, MODE_CONTROL, 0xe3);


  screen_width=320;
  screen_height=200;

  if (width==360)
  {
	/* turn off write protect */
	word_out(CRTC_INDEX, V_RETRACE_END, 0x2c);

	outp(MISC_OUTPUT, 0xe7);
	word_out(CRTC_INDEX, H_TOTAL, 0x6b);
	word_out(CRTC_INDEX, H_DISPLAY_END, 0x59);
	word_out(CRTC_INDEX, H_BLANK_START, 0x5a);
	word_out(CRTC_INDEX, H_BLANK_END, 0x8e);
	word_out(CRTC_INDEX, H_RETRACE_START, 0x5e);
	word_out(CRTC_INDEX, H_RETRACE_END, 0x8a);
	word_out(CRTC_INDEX, OFFSET, 0x2d);

	/* set vertical retrace back to normal */
	word_out(CRTC_INDEX, V_RETRACE_END, 0x8e);

	screen_width=360;
  }
  else
  {
	outp(MISC_OUTPUT, 0xe3);
  }

  if (height==240 || height==480)
  {
	/* turn off write protect */
	word_out(CRTC_INDEX, V_RETRACE_END, 0x2c);

	word_out(CRTC_INDEX, V_TOTAL, 0x0d);
	word_out(CRTC_INDEX, OVERFLOW, 0x3e);
	word_out(CRTC_INDEX, V_RETRACE_START, 0xea);
	word_out(CRTC_INDEX, V_RETRACE_END, 0xac);
	word_out(CRTC_INDEX, V_DISPLAY_END, 0xdf);
	word_out(CRTC_INDEX, V_BLANK_START, 0xe7);
	word_out(CRTC_INDEX, V_BLANK_END, 0x06);
	screen_height=height;
  }

  if (height==400 || height==480)
  {
	word_out(CRTC_INDEX, MAX_SCAN_LINE, 0x40);
	screen_height=height;
  }
}

// ENDGRAPH
// Sets screen mode back to mode 3h (system text mode)
void endgraph(void)
{
 set_mode(MODE_3H);
}