; ---------------------------------------------------------------------------
; DIRECTQB.ASM
;
; DirectQB: a game programming library for QuickBasic 4.5
;                                 (but it could be easily adapted to your
;                                  own programming language!)
;
; version 1.2 - October, the 27th 1998
;
;     by Angelo Mottola - Enhanced Creations 1998
;
; ===================
; News in version 1.2
; ===================
;
; For a complete list of the new features, read the file DIRECTQB.DOC provided
; with this new DirectQB distribuition.
;
; Comments, hints, bomb-mails or anything else are to be sent to:
;
; Angelo Mottola
; angelillo@bigfoot.com
;
; I cannot check my mail everyday; this means that you must wait for a reply,
; so don't get angry if you don't get it soon. Anyway I'll try to answer to
; all your questions!
; 
; Thanks again for using DirectQB!!
;
; ***************************************************************************

; Let's use a medium memory model...
.MODEL medium,basic

; ...and 386 instructions!
.386

; *** Macros ***

; ---------------------------------------------------------------------------
; WAITWRITEDSP macro
; purpose:
;   Waits the DSP until it's ready to receive data
; calling:
;   DX  Write wait port (BaseAddr+0Ch)
; changes:
;   AL
; ---------------------------------------------------------------------------
WAITWRITEDSP MACRO
LOCAL WaitWrite,EndWriteDSP
  PUSH CX
  XOR CX,CX
WaitWrite:
  DEC CX
  JZ EndWriteDSP
  IN AL,DX
  OR AL,AL
  JS WaitWrite
EndWriteDSP:
  POP CX
ENDM

; ---------------------------------------------------------------------------
; WAITREADDSP macro
; purpose:
;   Waits the DSP until it's ready to send data
; calling:
;   DX  Read wait port (BaseAddr+0Eh)
; changes:
;   AL
; ---------------------------------------------------------------------------
WAITREADDSP MACRO
LOCAL WaitRead,EndReadDSP
  PUSH CX
  XOR CX,CX
WaitRead:
  DEC CX
  JZ EndReadDSP
  IN AL,DX
  OR AL,AL
  JNS WaitRead
EndReadDSP:
  POP CX
ENDM

; ---------------------------------------------------------------------------
; SETUPDMA macro
; purpose:
;   Sets up the DMA chip for the transfer
; ---------------------------------------------------------------------------
SETUPDMA MACRO
LOCAL NoOverRide
  MOV SI,OFFSET SoundBuffer
  MOV CX,255
  MOV AX,@DATA
  ROL AX,4
  MOV BL,AL
  AND BL,00Fh
  AND AL,0F0h
  ADD SI,AX
  ADC BL,0
  NEG SI
  CMP SI,CX
  JA NoOverRide
  NEG SI
  ADD SI,CX
  INC SI
  INC BL
  NEG SI
NoOverRide:
  NEG SI
  MOV AL,DMA
  ADD AL,4
  OUT 0Ah,AL
  OUT 0Ch,AL
  MOV AL,01011000b    ; DMA mode: Single/Autoinit/Read
  ADD AL,DMA
  OUT 0Bh,AL
  MOV AL,BL
  MOV DX,DMA_Page
  OUT DX,AL
  MOV AX,SI
  MOV DX,DMA_Base
  OUT DX,AL
  MOV AL,AH
  OUT DX,AL
  MOV AL,CL
  MOV DX,DMA_Count
  OUT DX,AL
  MOV AL,CH
  OUT DX,AL
  MOV AL,DMA
  OUT 0Ah,AL
  MOV DX,BaseAddr
  ADD DX,0Ch
  WAITWRITEDSP
  MOV AL,040h
  OUT DX,AL
  WAITWRITEDSP
  MOV AL,SamplingRate
  OUT DX,AL
  WAITWRITEDSP
  MOV AL,048h
  OUT DX,AL
  MOV CX,255
  WAITWRITEDSP
  MOV AL,CL
  OUT DX,AL
  WAITWRITEDSP
  MOV AL,CH
  OUT DX,AL
  WAITWRITEDSP
  MOV AL,01Ch
  OUT DX,AL
ENDM

; ---------------------------------------------------------------------------
; MIXVOICES macro
; purpose:
;   Mixes sounds data into the second sound buffer and updates voices
; ---------------------------------------------------------------------------
MIXVOICES MACRO
LOCAL MixSample,UpdateVoices,MixIt,EndMix,MapPage,ClipSound
LOCAL UpdateVoiceLoop,CheckVoiceLoop,NextVoice,LoopVoice
  CLD
  CLI
  MOV AX,@DATA
  MOV ES,AX
  MOV DI,OFFSET MixBuffer
  MOV CX,128
  MOV EAX,0       ; Writes complete silence into the sound buffer
  REP STOSD
  CMP Playing,0   ; Avoid mixing if no sounds are playing
  JE EndMix
  XOR CX,CX
MixSample:
  MOV SI,CX
  SHL SI,1
  INC CX
  CMP CX,Channels
  JG UpdateVoices
  CMP VoiceSound[SI],0
  JE MixSample
  PUSH CX
  MOV BX,VoiceSound[SI]
  DEC BX
  SHL BX,1
  MOV CX,SoundLen[BX]
  SHL BX,1
  ADD BX,SoundPage
  MOV DX,EMShdl
  XOR AX,AX       ; Let's clear AX
MapPage2:
  MOV AH,44h      ; LIM/EMS function 44h, starting with physical page 0
  INT 67h         ; Map the page!
  INC BX          ; Next logical page
  INC AL          ; Next physical page
  CMP AL,4        ; Map all the four physical pages
  JNE MapPage2
  MOV AX,EMSseg
  MOV DI,OFFSET MixBuffer
  MOV BX,VoicePos[SI]
  MOV DX,VoiceVol[SI]
  MOV SI,BX
  MOV BX,@DATA
  MOV ES,BX
  MOV DS,AX
  MOV CX,256
MixIt:
  XOR AX,AX
  LODSB
  MOV BX,AX
  SHL BX,5
  ADD BX,DX
  ADD BX,OFFSET VolTable
  MOV AL,ES:[BX]
  SUB AX,080h
  ADD ES:[DI],AX
  ADD DI,2
  LOOP MixIt
  POP CX
  MOV AX,@DATA
  MOV DS,AX
  JMP MixSample
UpdateVoices:
  MOV CX,Channels
UpdateVoiceLoop:
  MOV SI,CX
  DEC SI
  SHL SI,1
  MOV DI,VoiceSound[SI]
  CMP DI,0
  JE NextVoice
  CMP VoicePos[SI],0FF00h   ; Has the sound reached the 64K length limit?
  JE CheckVoiceLoop
  DEC DI
  SHL DI,1
  MOV BX,SoundLen[DI]
  CMP VoicePos[SI],BX
  JG CheckVoiceLoop
  ADD VoicePos[SI],256
  LOOP UpdateVoiceLoop
  JMP EndMix
CheckVoiceLoop:
  CMP VoiceLoop[SI],0
  JNE LoopVoice
  MOV VoiceSound[SI],0
  DEC Playing
  LOOP UpdateVoiceLoop
  JMP EndMix
LoopVoice:
  MOV VoicePos[SI],0
NextVoice:
  LOOP UpdateVoiceLoop
EndMix:
  MOV AX,@DATA
  MOV DS,AX
  MOV ES,AX
  MOV SI,OFFSET MixBuffer
  MOV DI,OFFSET SoundBuffer2
  MOV CX,256
CopySample:
  LODSW
  ADD AX,080h
  CMP AX,255
  JLE NoUclip
  MOV AX,255
NoUclip:
  CMP AX,0
  JGE NoLclip
  MOV AX,0
NoLclip:
  STOSB
  LOOP CopySample
  STI
ENDM

; ---------------------------------------------------------------------------
; EPIXEL macro
; purpose:
;   Draws a pixel for the DQBellipse routine
; calling:
;   DX  x
;   BX  y
; ---------------------------------------------------------------------------
EPIXEL MACRO
LOCAL SkipcPixel
  CMP DX,ClipX1
  JL SkipcPixel
  CMP DX,ClipX2
  JG SkipcPixel
  CMP BX,ClipY1
  JL SkipcPixel
  CMP BX,ClipY2
  JG SkipcPixel
  MOV DI,BX
  SHL BX,8
  SHL DI,6
  ADD DI,BX
  ADD DI,DX
  STOSB
SkipcPixel:
ENDM


; 384 bytes of stack are enough for us!
.STACK 180h

; *** Data ***
.DATA

EMMid         DB  'EMMXXXX0'      ; EMM id string
EMSseg        DW  ?               ; EMS pageframe segment
EMShdl        DW  ?               ; EMS handle currently being used
EMSpage       DW  ?               ; Current EMS page
EMSallocated  DW  ?               ; Amount of allocated EMS memory
PutMode       DW  0               ; Sprite put mode (0:transparent, 1:solid)
deltax        DW  ?               ; \
deltay        DW  ?               ;  |
d             DW  ?               ;  |
dinc1         DW  ?               ;  |
dinc2         DW  ?               ;   - Bresenham's line algorithm variables
xinc1         DW  ?               ;  |
xinc2         DW  ?               ;  |
yinc1         DW  ?               ;  |
yinc2         DW  ?               ; /
ClipX1        DW  0               ; Left clipping border
ClipX2        DW  319             ; Right clipping border
ClipY1        DW  0               ; Upper clipping border
ClipY2        DW  199             ; Lower clipping border
BMapActive    DB  0               ; Is the blender map active?
BMapSeg       DW  ?               ; Blender map segment
FontSeg       DW  0FFA6h          ; Default font segment (BIOS font)
FontOff       DW  0000Eh          ; Default font offset (BIOS font)
TextMode      DB  0               ; Text print mode (0:transparent, 1:solid)
TextBackCol   DB  0               ; Background text color for solid text mode
FontBuffer    DB  2048 dup (?)    ; Internal font buffer
Buffer        DB  ?               ; File I/O byte buffer
BSVheader     DB  0FDh,000h,0A0h,000h      ; BSAVE file header (for a 320x200
              DB  000h,000h,0FAh           ; pixels image) - 7 bytes
BMPheader     DB  042h,04Dh,036h,0FEh,000h,000h,000h,000h   ; BMP file header
              DB  000h,000h,036h,004h,000h,000h,028h,000h   ; (for a 320x200
              DB  000h,000h,040h,001h,000h,000h,0C8h,000h   ; pixels image,
              DB  000h,000h,001h,000h,008h,000h,000h,000h   ; uncompressed)
              DB  000h,000h,000h,0FAh,000h,000h,0CEh,00Eh   ; - 54 bytes
              DB  000h,000h,0C4h,00Eh,000h,000h,000h,000h
              DB  000h,000h,000h,000h,000h,000h
PCXheader     DB  00Ah,005h,001h,008h,000h,000h,000h,000h   ; PCX file header
              DB  03Fh,001h,0C7h,000h,040h,001h,0C8h,000h   ; (for a 320x200
              DB  000h,000h,000h,0D8h,098h,038h,078h,074h   ; pixels image,
              DB  004h,070h,06Ch,004h,0ECh,0ACh,04Ch,0F8h   ; encoded using
              DB  0C4h,080h,040h,024h,024h,024h,028h,014h   ; PCX version 3.0
              DB  0F8h,0BCh,068h,0D4h,090h,09Ch,03Ch,024h   ; specifications)
              DB  024h,074h,070h,008h,078h,074h,008h,07Ch   ; - 128 bytes
              DB  078h,008h,034h,030h,004h,0F0h,0C4h,088h
              DB  000h,001h,040h,001h,000h,000h,000h,000h
              DB  000h,000h,000h,000h,000h,000h,000h,000h
              DB  000h,000h,000h,000h,000h,000h,000h,000h
              DB  000h,000h,000h,000h,000h,000h,000h,000h
              DB  000h,000h,000h,000h,000h,000h,000h,000h
              DB  000h,000h,000h,000h,000h,000h,000h,000h
              DB  000h,000h,000h,000h,000h,000h,000h,000h
              DB  000h,000h,000h,000h,000h,000h,000h,000h
KeyActive     DB  0               ; Is the keyboard handler already active?
OldInt9seg    DW  ?               ; Original int 9h segment
OldInt9off    DW  ?               ; Original int 9h offset
KeyStatus     DB  128 dup(0)      ; Status of each key (0:released, 1:pressed)
JoyDetected   DB  2 dup(0)        ; Have the joysticks been detected?
JoySens       DW  40              ; Default joystick sensibility
JoyX          DW  2 dup(0)        ; Joysticks last x position
JoyY          DW  2 dup(0)        ; Joysticks last y position
JoyCX         DW  2 dup(0)        ; Joysticks center x position
JoyCY         DW  2 dup(0)        ; Joysticks center y position
JoyButA       DB  2 dup(0)        ; Joysticks A button
JoyButB       DB  2 dup(0)        ; Joysticks B button
JoyMask       DB  00000001b       ; Joystick 1 x coordinate mask
              DB  00000010b       ; Joystick 1 y coordinate mask
              DB  00010000b       ; Joystick 1 button A mask
              DB  00100000b       ; Joystick 1 button B mask
              DB  00000100b       ; Joystick 2 x coordinate mask
              DB  00001000b       ; Joystick 2 y coordinate mask
              DB  01000000b       ; Joystick 2 button A mask
              DB  10000000b       ; Joystick 2 button B mask
MouseDetected DB  0               ; Has a mouse been detected?
MouseOn       DB  0               ; Is the mouse cursor visible?
MouseX        DW  0               ; Current mouse x coordinate
MouseY        DW  0               ; Current mouse y coordinate
MouseBut      DB  0               ; Current mouse buttons status
MouseShape    DB  0FFh,03Fh,0FFh,01Fh,0FFh,00Fh,0FFh,007h   ; Default mouse
              DB  0FFh,003h,0FFh,001h,0FFh,000h,07Fh,000h   ; cursor shape
              DB  03Fh,000h,07Fh,000h,0FFh,00Fh,0FFh,0BFh   ; data
              DB  0FFh,0FFh,0FFh,0FFh,0FFh,0FFh,0FFh,0FFh
              DB  000h,000h,000h,040h,000h,060h,000h,070h
              DB  000h,078h,000h,07Ch,000h,07Eh,000h,07Fh
              DB  080h,07Fh,000h,070h,000h,040h,000h,000h
              DB  000h,000h,000h,000h,000h,000h,000h,000h
SoundActive   DB  0               ; Has the SB custom IRQ been installed?
NumSounds     DB  0               ; Number of allocated sounds
SoundLen      DW  256 dup(0)      ; Sounds length (256 sounds allowed)
SoundPage     DW  ?               ; Starting EMS sound page
BaseAddr      DW  0220h           ; SB base address
IRQ           DB  7               ; SB IRQ
DMA           DB  1               ; SB DMA channel
OldIRQseg     DW  ?               ; Old IRQ interrupt segment
OldIRQoff     DW  ?               ; Old IRQ interrupt offset
PicANDmask    DB  ?               ; AND pic mask to clear IRQ
PicORmask     DB  ?               ; OR pic mask to set IRQ
DMA_Page      DW  ?               ; DMA page register
DMA_Base      DW  ?               ; DMA base address
DMA_Count     DW  ?               ; DMA count register
SamplingRate  DB  165             ; Sampling rate (default = 11000 Hz)
Playing       DB  0               ; Number of sounds currently playing
PauseSound    DB  0               ; Sound pause flag
Channels      DW  8               ; Number of channels used for sample mixing
VoiceSound    DW  32 dup(0)       ; Sounds played by each of the 32 voices
VoicePos      DW  32 dup(0)       ; Sound position of each voice
VoiceLoop     DW  32 dup(0)       ; Loop voice flag (0:play once, 1:loop)
VoiceVol      DW  32 dup(31)      ; Volume of each voice
MixBuffer     DW  256 dup(0)      ; Mixing sound buffer
SoundBuffer   DB  256 dup(?)      ; Output sound buffer
SoundBuffer2  DB  256 dup(?)      ; Second sound buffer for double buffering
StartX        DW  200 dup(?)      ; Start x value for each scanline (DQBgpoly)
EndX          DW  200 dup(?)      ; End x value for each scanline (DQBgpoly)
StartCol      DW  200 dup(?)      ; Start color for each scanline (DQBgpoly)
EndCol        DW  200 dup(?)      ; End color for each scanline (DQBgpoly)
DTA           DB  44 dup(?)       ; Disk transfer area for directory scan
INCLUDE TABLES.INC                ; Pre-built tables (volume/sqr/sin)

; Code begins here!
.CODE


; ***************************************************************************
; Internal library procedures
; ***************************************************************************


; ---------------------------------------------------------------------------
; MapEMS internal function
; purpose:
;   Maps the 4 physical pages of the pageframe beginning from the given
;   logical one, and stores the new page index into EMSpage
; calling:
;   DX  Logical page to begin mapping with
; changes:
;   AX,BX,DX
; ---------------------------------------------------------------------------
MapEMS PROC
  MOV EMSpage,DX  ; Stores the new current page index
  MOV BX,DX
  MOV DX,EMShdl
  XOR AX,AX       ; Let's clear AX
MapPage:
  MOV AH,44h      ; LIM/EMS function 44h, starting with physical page 0
  INT 67h         ; Map the page!
  INC BX          ; Next logical page
  INC AL          ; Next physical page
  CMP AL,4        ; Map all the four physical pages
  JNE MapPage
  RET
MapEMS ENDP

; ---------------------------------------------------------------------------
; GetLayerSeg internal function
; purpose:
;   Sets CX to the current layer segment; if the layer is video, returns
;   A000h, otherwise the EMS pageframe segment.
; calling:
;   BX  layer to retrieve segment of
; changes:
;   CX,BX,DX (,AX if MAPEMS is called)
; ---------------------------------------------------------------------------
GetLayerSeg PROC
  MOV CX,0A000h
  CMP BX,0        ; Is specified layer the video?
  JNE LayerOnEMS  ; No: layer is on EMS
  RET             ; Layer is video: returns screen segment (A000h)
LayerOnEMS:
  MOV CX,EMSseg   ; Returns the EMS pageframe segment
  DEC BX
  SHL BX,2        ; Translates the layer number into an EMS page index
  CMP BX,EMSpage  ; Do we need to change the actually mapped EMS page?
  JNE OnDifferentPage   ; Yes: map the new page
  RET
OnDifferentPage:
  MOV DX,BX
  CALL MapEMS          ; Calls the MAPEMS procedure to map the new page
  MOV CX,EMSseg   ; Returns the EMS pageframe segment (restored)
  RET
GetLayerSeg ENDP

; ---------------------------------------------------------------------------
; DrawPixel internal procedure
; purpose:
;   Draws a pixel for the DQBprint function
; ---------------------------------------------------------------------------
DrawPixel PROC
  CMP DX,ClipX1     ; Are we off the left layer border?
  JL SkipDrawPixel  ; Yes: skip it
  CMP DX,ClipX2     ; Are we off the right layer border?
  JG SkipDrawPixel  ; Yes: skip it
  TEST AH,BL        ; Is the pixel on?
  JZ SkipDrawPixel  ; No: skip it
  MOV ES:[DI],AL    ; Plots the pixel
  JMP ExitDrawPixel
SkipDrawPixel:
  CMP TextMode,0     ; Are we in solid text mode?
  JE ExitDrawPixel   ; No: jump away
  PUSH AX
  MOV AL,TextBackCol ; Gets the background text color
  MOV ES:[DI],AL     ; Draws the background pixel
  POP AX
ExitDrawPixel:
  INC DX            ; Next pixel
  INC DI
  RET
DrawPixel ENDP

; ---------------------------------------------------------------------------
; WriteBuffer internal procedure
; purpose:
;   Writes the buffer to file
; ---------------------------------------------------------------------------
WriteBuffer PROC
  PUSH DS
  PUSH AX
  PUSH CX
  PUSH DX
  MOV DX,OFFSET Buffer
  MOV AX,@DATA
  MOV DS,AX
  MOV CX,1
  MOV AH,40h
  INT 21h
  POP DX
  POP CX
  POP AX
  POP DS
  RET
WriteBuffer ENDP

; ---------------------------------------------------------------------------
; ReadBuffer internal procedure
; purpose:
;   Reads data from file and places it into the buffer
; ---------------------------------------------------------------------------
ReadBuffer PROC
  PUSH DS
  PUSH AX
  PUSH CX
  PUSH DX
  MOV DX,OFFSET Buffer
  MOV AX,@DATA
  MOV DS,AX
  MOV CX,1
  MOV AH,3Fh
  INT 21h
  POP DX
  POP CX
  POP AX
  POP DS
  RET
ReadBuffer ENDP

; ---------------------------------------------------------------------------
; ResetJoy internal procedure
; purpose:
;   Detects and calibrates both joystick 1 and 2.
; ---------------------------------------------------------------------------
ResetJoy PROC
  CLI
  MOV DX,201h
  OUT DX,AL
  IN AL,DX
  TEST AL,JoyMask[2]  ; We test also the buttons status to simulate the
  SETZ JoyButA[0]     ; exact time required to accomplish the DQBpollJoy
  TEST AL,JoyMask[3]  ; function
  SETZ JoyButB[0]
  MOV BL,JoyMask[0]
  ADD BL,JoyMask[1]
  XOR CX,CX
  OUT DX,AL
cJoyLoop1:
  IN AL,DX
  TEST AL,JoyMask[0]
  JZ cSkipXsave1
  MOV JoyCX[0],CX
cSkipXsave1:
  TEST AL,JoyMask[1]
  JZ cSkipYsave1
  MOV JoyCY[0],CX
cSkipYsave1:
  INC CX
  CMP CX,65535
  JE JoyDone1
  TEST AL,BL
  JNZ cJoyLoop1
  CMP CX,65535
  SETNZ JoyDetected[0]
JoyDone1:
  OUT DX,AL
  IN AL,DX
  TEST AL,JoyMask[6]
  SETZ JoyButA[1]
  TEST AL,JoyMask[7]
  SETZ JoyButB[1]
  MOV BL,JoyMask[4]
  ADD BL,JoyMask[5]
  XOR CX,CX
  OUT DX,AL
cJoyLoop2:
  IN AL,DX
  TEST AL,JoyMask[4]
  JZ cSkipXsave2
  MOV JoyCX[2],CX
cSkipXsave2:
  TEST AL,JoyMask[5]
  JZ cSkipYsave2
  MOV JoyCY[2],CX
cSkipYsave2:
  INC CX
  CMP CX,65535
  JE JoyDone2
  TEST AL,BL
  JNZ cJoyLoop2
  CMP CX,65535
  SETNZ JoyDetected[1]
JoyDone2:
  STI
  RET
ResetJoy ENDP


; ***************************************************************************
; GENERAL PURPOSE ROUTINES
; ***************************************************************************


; ---------------------------------------------------------------------------
; DQBinit FUNCTION
; purpose:
;   Initializes the library, by allocating the needed amount of EMS memory
;   to store layers (if requested). Also checks if a 386 CPU is available on
;   the system, calibrates the joysticks (where available) and sets up the
;   mouse. On success returns 0, otherwise an error code.
;   Always call this procedure before calling any of the library functions!
; declaration:
;   DECLARE FUNCTION DQBinit(BYVAL NumLayers,BYVAL NumSounds)
; ---------------------------------------------------------------------------
PUBLIC DQBinit
DQBinit PROC
  ; Stack layout:
  ;
  ; 08  NumLayers
  ; 06  NumSounds
  ; 04  Basic segment
  ; 02  Basic offset
  ; 00  BP
  PUSH BP
  MOV BP,SP
  PUSHF
  MOV CX,7000h    ; Set bits 12..15
  PUSH CX
  POPF
  PUSHF           ; Push them again...
  POP AX          ; ...and get them back
  AND AX,7000h    ; Mask bits 12..15
  SETNZ AL
  XOR AH,AH       ; Zero it
  POPF
  CMP AX,0
  JE Error_NoCPU
  XOR AX,AX
  MOV EMSallocated,AX
  MOV DX,[BP+8]
  SHL DX,2
  MOV SoundPage,DX
  MOV DX,[BP+6]
  MOV NumSounds,DL
  MOV DX,[BP+8]   ; Let's find out the total amount of EMS needed
  ADD DX,[BP+6]
  SHL DX,2        ; EMS needed=(NumLayers+NumSounds)*4 pages
  CMP DX,0
  JE MemDone      ; no EMS memory is needed, so skip memory allocation
  MOV ES,AX
  MOV BX,19Eh     ; Int 67h address
  MOV AX,ES:[BX]  ; Gets the EMM segment from the Interrupts Vector Table
  MOV ES,AX       ; ES now is the segment were the EMM driver is loaded
  MOV CL,8        ; Let's examine the 8 characters of the EMM id
  MOV SI,10       ; ES:SI now points to the supposed EMM id
  XOR BX,BX
CmpLoop:
  MOV AL,ES:[SI]
  CMP AL,DS:[EMMid+BX]    ; Compares each character
  JNE Error_NoEMS         ; If one of them is not the supposed one, fails
  INC SI          ; \
  INC BX          ;  |- Process next character
  DEC CL          ; /
  JNZ CmpLoop
  MOV AH,42h
  INT 67h         ; Let's check if enough EMS is available
  MOV DX,[BP+8]   ; Let's find out again the total amount of EMS needed
  ADD DX,[BP+6]
  SHL DX,2        ; EMS needed=(NumLayers+NumSounds)*4 pages
  CMP BX,DX       ; BX contains the number of available pages
  JL Error_NotEnoughEMS     ; not enough memory!
  MOV EMSallocated,DX       ; Stores amount of allocated EMS pages
  MOV AH,43h
  MOV BX,DX
  INT 67h         ; Allocates memory
  MOV EMShdl,DX   ; Stores the EMS handle
  MOV AH,41h
  INT 67h         ; Gets the EMS pageframe...
  MOV EMSseg,BX   ; ...and stores it
  MOV ES,BX
  MOV BX,[BP+6]
  CMP BX,0
  JE CheckLayers
  MOV EAX,080808080h
  ADD BX,[BP+8]
InitSounds:
  MOV DX,BX
  PUSH AX
  PUSH BX
  PUSH DX
  CALL MapEMS
  POP DX
  POP BX
  POP AX
  XOR DI,DI
  MOV CX,16384
  REP STOSD
  DEC BX
  JNZ InitSounds
CheckLayers:
  MOV BX,[BP+8]
  CMP BX,0
  JE MemDone
  XOR EAX,EAX
InitLayers:
  MOV DX,BX
  PUSH AX
  PUSH BX
  PUSH DX
  CALL MapEMS
  POP DX
  POP BX
  POP AX
  XOR DI,DI
  MOV CX,16384
  REP STOSD
  DEC BX
  JNZ InitLayers
  XOR DX,DX
  CALL MapEMS     ; Maps the pageframe starting from logical page 0
  XOR AX,AX       ; EMS memory initialized!
  JMP MemDone
Error_NoCPU:
  MOV AX,1        ; Error 1: No 386 CPU detected!
  JMP EndInit
Error_NoEMS:
  MOV AX,2        ; Error 2: No EMS memory manager detected!
  JMP EndInit
Error_NotEnoughEMS:
  MOV AX,3        ; Error 3: Not enough free EMS memory!
  JMP EndInit
MemDone:
  CALL ResetJoy
  XOR AX,AX
  INT 33h
  CMP AX,0
  JE MouseDone
  MOV CX,31
  MOV AX,SEG MouseISR
  MOV ES,AX
  MOV DX,OFFSET MouseISR
  MOV AX,0Ch
  INT 33h
  MOV MouseDetected,1
  JMP MouseDone
MouseISR:
  PUSH DS
  MOV AX,@DATA
  MOV DS,AX
  MOV MouseX,CX
  MOV MouseY,DX
  MOV MouseBut,BL
  POP DS
  RET
MouseDone:
  XOR AX,AX
EndInit:
  POP BP
  RET 4
DQBinit ENDP

; ---------------------------------------------------------------------------
; DQBemsseg FUNCTION
; purpose:
;   Returns the EMS pageframe segment
; declaration:
;   DECLARE FUNCTION DQBemsseg()
; ---------------------------------------------------------------------------
PUBLIC DQBemsseg
DQBemsseg PROC
  MOV AX,EMSseg   ; AX (function return value) holds the EMS pageframe
  RET
DQBemsseg ENDP

; ---------------------------------------------------------------------------
; DQBclose SUB
; purpose:
;   Deallocated previously allocated EMS memory (if it was allocated), and
;   does other closing stuff needed.
;   ALWAYS CALL THIS SUB BEFORE ENDING YOUR PROGRAMS!
; declaration:
;   DECLARE SUB DQBclose()
; ---------------------------------------------------------------------------
PUBLIC DQBclose
DQBclose PROC
  CMP KeyActive,0
  JE NoRemoveKey
  PUSH DS
  MOV DX,OldInt9off
  MOV AX,OldInt9seg
  MOV DS,AX
  MOV AX,2509h
  INT 21h
  POP DS
  MOV KeyActive,0
NoRemoveKey:
  CMP SoundActive,0
  JE NoRemoveSB
  MOV DX,BaseAddr
  ADD DX,06h
  MOV AL,1
  OUT DX,AL
  IN AL,DX
  IN AL,DX
  IN AL,DX
  IN AL,DX
  XOR AL,AL
  OUT DX,AL
  ADD DX,08h
  WAITREADDSP
  SUB DX,4
  IN AL,DX          ; SB reset
  IN AL,021h
  OR AL,PicORmask
  OUT 021h,AL
  XOR AX,AX
  MOV ES,AX
  MOV AL,IRQ
  ADD AL,8
  MOV SI,AX
  SHL SI,2
  CLI
  MOV AX,OldIRQoff
  MOV ES:[SI],AX
  MOV AX,OldIRQseg
  MOV ES:[SI+2],AX  ; Old IRQ restored
  STI
  MOV SoundActive,0
NoRemoveSB:
  CMP EMSallocated,0    ; Did we allocate EMS memory?
  JE NoDealloc      ; No EMS memory was allocated at startup!
  MOV AH,45h        ; Function 45h: release allocated EMS memory
  MOV DX,EMShdl
  INT 67h
NoDealloc:
  CMP BMapActive,0
  JE NoFreeBMap
  MOV AH,49h
  MOV ES,BMapSeg
  INT 21h
NoFreeBMap:
  RET
DQBclose ENDP

; ---------------------------------------------------------------------------
; DQBsetBit FUNCTION
; purpose:
;   Sets a specified bit on a given integer value
; declaration:
;   DECLARE FUNCTION DQBsetBit(BYVAL Value,BYVAL Bit)
; ---------------------------------------------------------------------------
PUBLIC DQBsetBit
DQBsetBit PROC
  PUSH BP
  MOV BP,SP
  MOV AX,[BP+8]
  MOV CX,[BP+6]
  MOV BX,1
  CMP CX,0
  JE SkipShift
FindBit:
  SHL BX,1
  LOOP FindBit
SkipShift:
  TEST AX,BX
  JNZ EndSetBit
  ADD AX,BX
EndSetBit:
  POP BP
  RET 4
DQBsetBit ENDP

; ---------------------------------------------------------------------------
; DQBresetBit FUNCTION
; purpose:
;   Resets a specified bit on a given integer value
; declaration:
;   DECLARE FUNCTION DQBresetBit(BYVAL Value,BYVAL Bit)
; ---------------------------------------------------------------------------
PUBLIC DQBresetBit
DQBresetBit PROC
  PUSH BP
  MOV BP,SP
  MOV AX,[BP+8]
  MOV CX,[BP+6]
  MOV BX,1
  CMP CX,0
  JE SkipShift1
FindBit1:
  SHL BX,1
  LOOP FindBit1
SkipShift1:
  MOV DX,0FFFFh
  SUB DX,BX
  AND AX,DX
  POP BP
  RET 4
DQBresetBit ENDP

; ---------------------------------------------------------------------------
; DQBreadBit FUNCTION
; purpose:
;   Returns true if a specified bit is set into a given integer value
; declaration:
;   DECLARE FUNCTION DQBreadBit(BYVAL Value,BYVAL Bit)
; ---------------------------------------------------------------------------
PUBLIC DQBreadBit
DQBreadBit PROC
  PUSH BP
  MOV BP,SP
  MOV AX,[BP+8]
  MOV CX,[BP+6]
  MOV BX,1
  CMP CX,0
  JE SkipShift1
FindBit1:
  SHL BX,1
  LOOP FindBit1
SkipShift1:
  TEST AX,BX
  SETNZ AL
  XOR AH,AH
  NEG AX
  POP BP
  RET 4
DQBreadBit ENDP

; ---------------------------------------------------------------------------
; DQBtoggleBit FUNCTION
; purpose:
;   Toggles a given bit into a specified integer value
; declaration:
;   DECLARE FUNCTION DQBtoggleBit(BYVAL Value,BYVAL Bit)
; ---------------------------------------------------------------------------
PUBLIC DQBtoggleBit
DQBtoggleBit PROC
  PUSH BP
  MOV BP,SP
  MOV AX,[BP+8]
  MOV CX,[BP+6]
  MOV BX,1
  CMP CX,0
  JE SkipShift2
FindBit2:
  SHL BX,1
  LOOP FindBit2
SkipShift2:
  TEST AX,BX
  JNZ TurnOff
  ADD AX,BX
  JMP EndToggleBit
TurnOff:
  SUB AX,BX
EndToggleBit:
  POP BP
  RET 4
DQBtoggleBit ENDP

; ---------------------------------------------------------------------------
; DQBshiftLeft FUNCTION
; purpose:
;   Shifts specified integer value a given number of bits left
; declaration:
;   DECLARE FUNCTION DQBshiftLeft(BYVAL Value,BYVAL nBits)
; ---------------------------------------------------------------------------
PUBLIC DQBshiftLeft
DQBshiftLeft PROC
  PUSH BP
  MOV BP,SP
  MOV AX,[BP+8]
  MOV CX,[BP+6]
  CMP CX,0
  JE SkipShift3
ShiftLeft:
  SHL AX,1
  LOOP ShiftLeft
SkipShift3:
  POP BP
  RET 4
DQBshiftLeft ENDP

; ---------------------------------------------------------------------------
; DQBshiftRight FUNCTION
; purpose:
;   Shifts specified integer value a given number of bits right
; declaration:
;   DECLARE FUNCTION DQBshiftRight(BYVAL Value,BYVAL nBits)
; ---------------------------------------------------------------------------
PUBLIC DQBshiftRight
DQBshiftRight PROC
  PUSH BP
  MOV BP,SP
  MOV AX,[BP+8]
  MOV CX,[BP+6]
  CMP CX,0
  JE SkipShift4
ShiftRight:
  SHR AX,1
  LOOP ShiftRight
SkipShift4:
  POP BP
  RET 4
DQBshiftRight ENDP

; ---------------------------------------------------------------------------
; DQBdir$ FUNCTION
; purpose:
;   Scans directory for specified file(s), and returns it. If a file spec is
;   given, the routine performs a findfirst, otherwise a findnext.
; declaration:
;   DECLARE SUB xDQBdir(BYVAL MaskSeg,BYVAL MaskOff,BYVAL Attrib,BYVAL FileSeg,
;                       BYVAL FileOff)
;   DECLARE FUNCTION DQBdir$(Mask AS STRING,Attrib AS INTEGER)
; ---------------------------------------------------------------------------
PUBLIC xDQBdir
xDQBdir PROC
  ; Stack layout:
  ;
  ; 16  MaskSeg
  ; 14  MaskOff
  ; 12  Attrib
  ; 10  FileSeg
  ; 08  FileOff
  ; 06  Basic return segment
  ; 04  Basic return offset
  ; 02  DS
  ; 00  BP
  PUSH DS
  PUSH BP
  MOV BP,SP
  MOV AX,@DATA
  MOV DS,AX
  MOV AX,01A00h
  MOV DX,OFFSET DTA
  INT 21h
  MOV AX,[BP+16]
  MOV ES,AX
  MOV DI,[BP+14]
  XOR BL,BL
  CMP ES:[DI],BL
  JNE DirFindFirst
  MOV AX,04F00h
  JMP GetEntry
DirFindFirst:
  MOV DS,AX
  MOV DX,DI
  MOV CX,[BP+12]
  MOV AX,04E00h
GetEntry:
  INT 21h
  JC DirNotFound
  MOV AX,@DATA
  MOV DS,AX
  MOV AX,[BP+10]
  MOV ES,AX
  MOV DI,[BP+8]
  MOV SI,OFFSET DTA
  MOV AL,[SI+21]
  MOV BX,[BP+12]
  AND AL,BL
  JZ DirNotFound
  ADD SI,30
  MOV CX,13
  REP MOVSB
  POP BP
  POP DS
  RET 10
DirNotFound:
  MOV AX,[BP+10]
  MOV ES,AX
  MOV DI,[BP+8]
  XOR AL,AL
  STOSB
  POP BP
  POP DS
  RET 10
xDQBdir ENDP

; ---------------------------------------------------------------------------
; DQBdrive$ FUNCTION
; purpose:
;   Returns the letter of the current drive
; declaration:
;   DECLARE FUNCTION xDQBdrive()
;   DECLARE FUNCTION DQBdrive$()
; ---------------------------------------------------------------------------
PUBLIC xDQBdrive
xDQBdrive PROC
  PUSH BP
  MOV BP,SP
  MOV AX,01900h
  INT 21h
  XOR AH,AH
  ADD AL,65
  POP BP
  RET
xDQBdrive ENDP

; ---------------------------------------------------------------------------
; DQBpath$ FUNCTION
; purpose:
;   Returns the current drive path
; declaration:
;   DECLARE SUB xDQBpath(BYVAL PathSeg,BYVAL PathOff)
;   DECLARE FUNCTION DQBpath$()
; ---------------------------------------------------------------------------
PUBLIC xDQBpath
xDQBpath PROC
  PUSH DS
  PUSH BP
  MOV BP,SP
  MOV AX,01900h
  INT 21h
  INC AL
  XOR DH,DH
  MOV DL,AL
  MOV AX,[BP+10]
  MOV DS,AX
  MOV SI,[BP+8]
  MOV AH,047h
  INT 21h
  POP BP
  POP DS
  RET 4
xDQBpath ENDP


; ---------------------------------------------------------------------------
; DQBnumDrives FUNCTION
; purpose:
;   Returns the total number of drives
; declaration:
;   DECLARE FUNCTION DQBnumDrives()
; ---------------------------------------------------------------------------
PUBLIC DQBnumDrives
DQBnumDrives PROC
  MOV AX,01900h
  INT 21h
  MOV DL,AL
  MOV AX,0E00h
  XOR DH,DH
  INT 21h
  MOV DI,AX
  XOR CX,CX
CheckLogical:
  MOV AX,0440Eh
  MOV BX,CX
  INC BX
  INT 21h
  JC EndNumDrives
  INC CX
  CMP CX,DI
  JL CheckLogical
EndNumDrives:
  MOV AX,CX
  RET
DQBnumDrives ENDP

; ---------------------------------------------------------------------------
; DQBsetDrive SUB
; purpose:
;   Changes the current drive to the new specified one
; declaration:
;   DECLARE SUB xDQBsetDrive()
;   DECLARE SUB DQBsetDrive()
; ---------------------------------------------------------------------------
PUBLIC xDQBsetDrive
xDQBsetDrive PROC
  PUSH BP
  MOV BP,SP
  MOV AX,0E00h
  MOV DX,[BP+6]
  INT 21h
  POP BP
  RET
xDQBsetDrive ENDP


; ***************************************************************************
; GRAPHICAL ROUTINES
; ***************************************************************************


; ---------------------------------------------------------------------------
; DQBcopyLayer SUB
; purpose:
;   Copies a layer onto another one.
; declaration:
;   DECLARE SUB DQBcopyLayer(BYVAL SourceLayer,BYVAL DestLayer)
; ---------------------------------------------------------------------------
PUBLIC DQBcopyLayer
DQBcopyLayer PROC
  ; Stack layout:
  ;
  ; 10  SourceLayer
  ; 08  DestLayer
  ; 06  Basic segment
  ; 04  Basic offset
  ; 02  DS
  ; 00  BP
  PUSH DS
  PUSH BP
  MOV BP,SP
  MOV BX,[BP+10]  ; BX holds the source layer number
  MOV DX,[BP+8]   ; DX holds the destination layer number
  CMP BX,DX       ; Are we processing the same layers?
  JE EndCopy      ; Yes: exits function
  CMP BX,0
  JE OneIsVideo   ; Is source layer on video?
  CMP DX,0
  JE OneIsVideo   ; Is destination layer on video?
  MOV DX,EMShdl   ; Both source and destination layer are on EMS
  MOV AX,EMSseg
  MOV DS,AX       ; \
  MOV ES,AX       ; |- Both DS and ES points to the EMS pageframe segment
  DEC BX
  SHL BX,2
  MOV AX,4400h
  INT 67h
  MOV AX,4401h
  INC BX
  INT 67h
  MOV BX,[BP+8]
  DEC BX
  SHL BX,2
  MOV AX,4402h
  INT 67h
  MOV AX,4403h    ; At the end the first two physical pages are mapped to the    
  INC BX          ; first half of the source layer, and the third and fourth
  INT 67h         ; pages points to the first half of the destination layer
  MOV SI,0
  MOV DI,8000h
  MOV CX,2000h    ; Copy the DWORDs!
  REP MOVSD
  MOV AX,4400h
  MOV BX,[BP+10]
  DEC BX
  SHL BX,2
  ADD BX,2
  INT 67h
  MOV AX,4401h
  INC BX
  INT 67h
  MOV BX,[BP+8]
  DEC BX
  SHL BX,2
  ADD BX,2
  MOV AX,4402h
  INT 67h
  MOV AX,4403h
  INC BX
  INT 67h
  MOV SI,0
  MOV DI,8000h
  MOV CX,2000h
  REP MOVSD
  MOV AX,@DATA
  MOV DS,AX
  MOV DX,EMSpage
  CALL MapEMS
  JMP EndCopy
OneIsVideo:
  CALL GetLayerSeg
  MOV DI,CX       ; Saves source segment into DI
  MOV BX,[BP+8]
  CALL GetLayerSeg
  MOV ES,CX       ; Sets destination segment...
  MOV DS,DI       ; ...and source segment
  MOV DI,0        ; Both source and destination offsets start as 0
  MOV SI,0
  MOV CX,16000    ; Move 16000 DWORDs
  REP MOVSD
EndCopy:
  POP BP
  POP DS
  RET 4
DQBcopyLayer ENDP

; ---------------------------------------------------------------------------
; DQBcopyTransLayer SUB
; purpose:
;   Copies a layer onto another one. Pixels with color 0 on source layer are
;   skipped, allowing a transparency effect; this can be useful to create
;   parallax scrolling effects.
; declaration:
;   DECLARE SUB DQBcopyTransLayer(BYVAL SourceLayer,BYVAL DestLayer)
; ---------------------------------------------------------------------------
PUBLIC DQBcopyTransLayer
DQBcopyTransLayer PROC
  ; Stack layout:
  ;
  ; 10  SourceLayer
  ; 08  DestLayer
  ; 06  Basic segment
  ; 04  Basic offset
  ; 02  DS
  ; 00  BP
  PUSH DS
  PUSH BP
  MOV BP,SP
  MOV BX,[BP+10]  ; BX holds the source layer number
  MOV DX,[BP+8]   ; DX holds the destination layer number
  CMP BX,DX       ; Are we processing the same layers?
  JE EndCopy1     ; Yes: exits function
  CMP BX,0
  JE OneIsVideo1  ; Is source layer on video?
  CMP DX,0
  JE OneIsVideo1  ; Is destination layer on video?
  MOV DX,EMShdl   ; Both source and destination layer are on EMS
  MOV AX,EMSseg
  MOV DS,AX       ; \
  MOV ES,AX       ; |- Both DS and ES points to the EMS pageframe segment
  DEC BX
  SHL BX,2
  MOV AX,4400h
  INT 67h
  MOV AX,4401h
  INC BX
  INT 67h
  MOV BX,[BP+8]
  DEC BX
  SHL BX,2
  MOV AX,4402h
  INT 67h
  MOV AX,4403h    ; At the end the first two physical pages are mapped to the    
  INC BX          ; first half of the source layer, and the third and fourth
  INT 67h         ; pages points to the first half of the destination layer
  MOV SI,0
  MOV DI,8000h
  MOV CX,4000h    ; Copies half the screen
CopyTrans1:
  LODSW
  CMP AX,0
  JNE CopyPixel1
  ADD DI,2
  LOOP CopyTrans1
  JMP EndCopyTrans1
CopyPixel1:
  CMP AL,0
  JNE CopyLowPixel1
  XCHG AL,AH
  INC DI
  STOSB
  LOOP CopyTrans1
  JMP EndCopyTrans1
CopyLowPixel1:
  CMP AH,0
  JNE CopyHighPixel1
  STOSB
  INC DI
  LOOP CopyTrans1
  JMP EndCopyTrans1
CopyHighPixel1:
  STOSW
  LOOP CopyTrans1
EndCopyTrans1:
  MOV AX,4400h
  MOV BX,[BP+10]
  DEC BX
  SHL BX,2
  ADD BX,2
  INT 67h
  MOV AX,4401h
  INC BX
  INT 67h
  MOV BX,[BP+8]
  DEC BX
  SHL BX,2
  ADD BX,2
  MOV AX,4402h
  INT 67h
  MOV AX,4403h
  INC BX
  INT 67h
  MOV SI,0
  MOV DI,8000h
  MOV CX,4000h    ; Copies the second half of the screen
CopyTrans2:
  LODSW
  CMP AX,0
  JNE CopyPixel2
  ADD DI,2
  LOOP CopyTrans2
  JMP EndCopyTrans2
CopyPixel2:
  CMP AL,0
  JNE CopyLowPixel2
  XCHG AL,AH
  INC DI
  STOSB
  LOOP CopyTrans2
  JMP EndCopyTrans2
CopyLowPixel2:
  CMP AH,0
  JNE CopyHighPixel2
  STOSB
  INC DI
  LOOP CopyTrans2
  JMP EndCopyTrans2
CopyHighPixel2:
  STOSW
  LOOP CopyTrans2
EndCopyTrans2:
  MOV AX,@DATA
  MOV DS,AX
  MOV DX,EMSpage
  CALL MapEMS
  JMP EndCopy1
OneIsVideo1:
  CALL GetLayerSeg
  MOV DI,CX       ; Saves source segment into DI
  MOV BX,[BP+8]
  CALL GetLayerSeg
  MOV ES,CX       ; Sets destination segment...
  MOV DS,DI       ; ...and source segment
  MOV DI,0        ; Both source and destination offsets start as 0
  MOV SI,0
  MOV CX,32000    ; Copies 64000 bytes
CopyTrans3:
  LODSW
  CMP AX,0
  JNE CopyPixel3
  ADD DI,2
  LOOP CopyTrans3
  JMP EndCopy1
CopyPixel3:
  CMP AL,0
  JNE CopyLowPixel3
  XCHG AL,AH
  INC DI
  STOSB
  LOOP CopyTrans3
  JMP EndCopy1
CopyLowPixel3:
  CMP AH,0
  JNE CopyHighPixel3
  STOSB
  INC DI
  LOOP CopyTrans3
  JMP EndCopy1
CopyHighPixel3:
  STOSW
  LOOP CopyTrans3
EndCopy1:
  POP BP
  POP DS
  RET 4
DQBcopyTransLayer ENDP

; ---------------------------------------------------------------------------
; DQBclearLayer SUB
; purpose:
;   Clears the content of a given layer to black (color 0 used)
; declaration:
;   DECLARE SUB DQBclearLayer(BYVAL Layer)
; ---------------------------------------------------------------------------
PUBLIC DQBclearLayer
DQBclearLayer PROC
  ; Stack layout:
  ;
  ; 06  Layer
  ; 04  Basic return segment
  ; 02  Basic return offset
  ; 00  BP
  PUSH BP
  MOV BP,SP
  MOV BX,[BP+6]
  CALL GetLayerSeg
  MOV ES,CX
  MOV CX,16384
  XOR EAX,EAX
  XOR DI,DI
  REP STOSD
  POP BP
  RET 2
DQBclearLayer ENDP

; ---------------------------------------------------------------------------
; DQBpset SUB
; purpose:
;   Writes a pixel on the given segment, assuming an offset of 0 for the first
;   pixel on the upper-left corner of the screen.
; declaration:
;   DECLARE SUB DQBpset(BYVAL Layer,BYVAL x,BYVAL y,BYVAL Col)
; ---------------------------------------------------------------------------
PUBLIC DQBpset
DQBpset PROC
  ; Stack layout:
  ; 
  ; 12  Layer
  ; 10  x
  ; 08  y
  ; 06  Color
  ; 04  Basic segment
  ; 02  Basic offset
  ; 00  BP
  PUSH BP
  MOV BP,SP
  MOV AX,[BP+10]
  CMP AX,ClipX1
  JL ExitPset
  CMP AX,ClipX2
  JG ExitPset
  MOV AX,[BP+8]
  CMP AX,ClipY1
  JL ExitPset
  CMP AX,ClipY2
  JG ExitPset
  MOV BX,[BP+12]    ; Gets the output layer
  CALL GetLayerSeg
  MOV ES,CX         ; Sets the appropriate segment
  MOV DI,[BP+8]    ; Calculates the offset
  MOV BX,DI
  SHL DI,8
  SHL BX,6
  ADD DI,BX
  ADD DI,[BP+10]
  MOV AL,[BP+6]     ; Gets the pixel color...
  STOSB
ExitPset:
  POP BP
  RET 8
DQBpset ENDP

; ---------------------------------------------------------------------------
; DQBpoint FUNCTION
; purpose:
;   Returns the color of the pixel at the given coordinates and on the given
;   layer.
; declaration:
;   FUNCTION DQBpoint(BYVAL Layer,BYVAL x,BYVAL y)
; ---------------------------------------------------------------------------
PUBLIC DQBpoint
DQBpoint PROC
  ; Stack layout:
  ; 
  ; 10  Layer
  ; 08  x
  ; 06  y
  ; 04  Basic segment
  ; 02  Basic offset
  ; 00  BP
  PUSH BP
  MOV BP,SP
  MOV BX,[BP+10]    ; Gets the output layer
  CALL GetLayerSeg
  MOV ES,CX         ; Sets the appropriate segment
  MOV SI,[BP+6]     ; Calculates the offset
  MOV BX,SI
  SHL SI,8
  SHL BX,6
  ADD SI,BX
  ADD SI,[BP+8]
  XOR AH,AH
  MOV AL,ES:[SI]
  POP BP
  RET 6
DQBpoint ENDP

; ---------------------------------------------------------------------------
; DQBline SUB
; purpose:
;   Draw a line from x1,y1 to x2,y2 on specified layer with the given color,
;   using the Bresenham's line algorithm.
; declaration:
;   DECLARE SUB DQBline(BYVAL Layer,BYVAL x1,BYVAL y1,BYVAL x2,BYVAL y2,
;                       BYVAL Col)
; ---------------------------------------------------------------------------
PUBLIC DQBline
DQBline PROC
  ; Stack layout:
  ;
  ; 16  Layer
  ; 14  x1
  ; 12  y1
  ; 10  x2
  ; 08  y2
  ; 06  Col
  ; 04  Basic return segment
  ; 02  Basic return offset
  ; 00  BP
  PUSH BP
  MOV BP,SP
  MOV BX,[BP+16]
  CALL GetLayerSeg
  MOV ES,CX
  MOV AX,[BP+14]
  MOV BX,[BP+10]
  CMP AX,BX
  JG Line1
  SUB BX,AX
  MOV deltax,BX
  JMP Line2
Line1:
  SUB AX,BX
  MOV deltax,AX
Line2:
  MOV AX,[BP+12]
  MOV BX,[BP+8]
  CMP AX,BX
  JG Line3
  SUB BX,AX
  MOV deltay,BX
  JMP Line4
Line3:
  SUB AX,BX
  MOV deltay,AX
Line4:
  MOV AX,deltax
  CMP AX,deltay
  JL Line5
  MOV CX,deltax
  INC CX
  MOV BX,deltay
  SHL BX,1
  MOV dinc1,BX
  SUB BX,AX
  MOV d,BX
  MOV BX,deltay
  SUB BX,AX
  SHL BX,1
  MOV dinc2,BX
  MOV xinc1,1
  MOV xinc2,1
  MOV yinc1,0
  MOV yinc2,1
  JMP Line6
Line5:
  MOV CX,deltay
  INC CX
  MOV BX,deltax
  SHL BX,1
  MOV dinc1,BX
  SUB BX,CX
  MOV d,BX
  SUB AX,CX
  INC AX
  SHL AX,1
  MOV dinc2,AX
  MOV xinc1,0
  MOV xinc2,1
  MOV yinc1,1
  MOV yinc2,1
Line6:
  MOV AX,[BP+14]
  CMP AX,[BP+10]
  JLE Line7
  MOV DX,xinc1
  MOV AX,-1
  MUL DX
  MOV xinc1,AX
  MOV DX,xinc2
  MOV AX,-1
  MUL DX
  MOV xinc2,AX
Line7:
  MOV AX,[BP+12]
  CMP AX,[BP+8]
  JLE Line8
  MOV DX,yinc1
  MOV AX,-1
  MUL DX
  MOV yinc1,AX
  MOV DX,yinc2
  MOV AX,-1
  MUL DX
  MOV yinc2,AX
Line8:
  MOV DX,[BP+14]    ; DX holds the current x coordinate
  MOV BX,[BP+12]    ; BX holds the current y coordinate
  MOV AX,[BP+6]     ; AL holds the color
LineLoop:
  CMP DX,ClipX1
  JL SkipLinePixel
  CMP DX,ClipX2
  JG SkipLinePixel
  CMP BX,ClipY1
  JL SkipLinePixel
  CMP BX,ClipY2
  JG SkipLinePixel
  MOV DI,BX
  MOV SI,BX
  SHL DI,8
  SHL SI,6
  ADD DI,SI
  ADD DI,DX
  STOSB
SkipLinePixel:
  CMP d,0
  JL Line9
  MOV SI,d
  ADD SI,dinc2
  MOV d,SI
  ADD DX,xinc2
  ADD BX,yinc2
  JMP Line10
Line9:
  MOV SI,d
  ADD SI,dinc1
  MOV d,SI
  ADD DX,xinc1
  ADD BX,yinc1
Line10:
  LOOP LineLoop
  POP BP
  RET 12
DQBline ENDP

; ---------------------------------------------------------------------------
; DQBellipse SUB
; purpose:
;   Draws an ellipse on specified layer with center in (x,y), and rx,ry as
;   the horizontal and vertical radiuses, with col color. The function works
;   for any radiuses ranging 0-255, otherwise you'll get unwanted results.
;   DQBellipse is affected by the clipping box.
; declaration:
;   DECLARE SUB DQBellipse(BYVAL Layer,BYVAL x,BYVAL y,BYVAL rx,BYVAL ry,
;                          BYVAL col)
; ---------------------------------------------------------------------------
PUBLIC DQBellipse
DQBellipse PROC
  ; Stack layout:
  ;
  ; 28  Layer
  ; 26  x
  ; 24  y
  ; 22  rx
  ; 20  ry
  ; 18  Col
  ; 16  Basic return segment
  ; 14  Basic return offset
  ; 12  BP
  ; 10  x
  ; 08  y
  ; 06  xe
  ; 04  ye
  ; 02  e
  ; 00  c
  PUSH BP
  MOV BP,SP
  MOV BX,[BP+16]
  CALL GetLayerSeg
  MOV ES,CX
  PUSH AX           ; Preserve stack space for internal variables
  PUSH AX
  PUSH AX
  PUSH AX
  PUSH AX
  PUSH AX
  MOV BP,SP
  MOV AX,[BP+22]
  CMP AX,0
  JNE NotVline
  MOV AX,[BP+18]
  MOV CX,[BP+20]
  NEG CX
eVline:
  MOV DX,[BP+26]
  MOV BX,[BP+24]
  ADD BX,CX
  EPIXEL
  INC CX
  CMP CX,[BP+20]
  JLE eVline
  JMP EndEllipse
NotVline:
  MOV AX,[BP+20]
  CMP AX,0
  JNE NotHline
  MOV AX,[BP+18]
  MOV CX,[BP+22]
  NEG CX
eHline:
  MOV DX,[BP+26]
  ADD DX,CX
  MOV BX,[BP+24]
  EPIXEL
  INC CX
  CMP CX,[BP+22]
  JLE eHline
  JMP EndEllipse
NotHline:
  CMP AX,[BP+22]
  JG OtherAxis
  XOR AX,AX
  MOV [BP+10],AX
  MOV [BP+6],AX
  MOV AX,[BP+20]
  MOV [BP+8],AX
  MOV AX,[BP+22]
  MOV BX,AX
  MUL BX
  MOV [BP+4],AX
  PUSH AX
  SHR AX,1
  NEG AX
  MOV [BP+2],AX
  POP AX
  XOR DX,DX
  MOV BX,[BP+20]
  DIV BX
  MOV [BP],AX
  MOV AX,[BP+18]
Circle1:
  MOV CX,[BP+2]
  CMP CX,0
  JG Circle3
Circle2:
  MOV DX,[BP+26]
  ADD DX,[BP+10]
  MOV BX,[BP+24]
  ADD BX,[BP+8]
  EPIXEL
  MOV DX,[BP+26]
  SUB DX,[BP+10]
  MOV BX,[BP+24]
  ADD BX,[BP+8]
  EPIXEL
  MOV DX,[BP+26]
  ADD DX,[BP+10]
  MOV BX,[BP+24]
  SUB BX,[BP+8]
  EPIXEL
  MOV DX,[BP+26]
  SUB DX,[BP+10]
  MOV BX,[BP+24]
  SUB BX,[BP+8]
  EPIXEL
  MOV BX,[BP+10]
  INC BX
  MOV [BP+10],BX
  MOV BX,[BP+6]
  ADD BX,[BP+20]
  MOV [BP+6],BX
  MOV BX,[BP+2]
  ADD BX,[BP+6]
  MOV [BP+2],BX
  CMP BX,0
  JLE Circle2
  JMP Circle4
Circle3:
  MOV DX,[BP+26]
  ADD DX,[BP+10]
  MOV BX,[BP+24]
  ADD BX,[BP+8]
  EPIXEL
  MOV DX,[BP+26]
  SUB DX,[BP+10]
  MOV BX,[BP+24]
  ADD BX,[BP+8]
  EPIXEL
  MOV DX,[BP+26]
  ADD DX,[BP+10]
  MOV BX,[BP+24]
  SUB BX,[BP+8]
  EPIXEL
  MOV DX,[BP+26]
  SUB DX,[BP+10]
  MOV BX,[BP+24]
  SUB BX,[BP+8]
  EPIXEL
Circle4:
  MOV BX,[BP+4]
  SUB BX,[BP]
  MOV [BP+4],BX
  MOV BX,[BP+2]
  SUB BX,[BP+4]
  MOV [BP+2],BX
  MOV BX,[BP+8]
  DEC BX
  MOV [BP+8],BX
  CMP BX,0
  JNZ Circle1
  MOV DX,[BP+26]
  ADD DX,[BP+10]
  MOV BX,[BP+24]
  EPIXEL
  MOV DX,[BP+26]
  SUB DX,[BP+10]
  MOV BX,[BP+24]
  EPIXEL
  JMP EndEllipse
OtherAxis:
  XOR AX,AX
  MOV [BP+10],AX
  MOV [BP+6],AX
  MOV AX,[BP+22]
  MOV [BP+8],AX
  MOV AX,[BP+20]
  MOV BX,AX
  MUL BX
  MOV [BP+4],AX
  PUSH AX
  SHR AX,1
  NEG AX
  MOV [BP+2],AX
  POP AX
  XOR DX,DX
  MOV BX,[BP+22]
  DIV BX
  MOV [BP],AX
  MOV AX,[BP+18]
Circle5:
  MOV CX,[BP+2]
  CMP CX,0
  JG Circle7
Circle6:
  MOV DX,[BP+26]
  ADD DX,[BP+8]
  MOV BX,[BP+24]
  ADD BX,[BP+10]
  EPIXEL
  MOV DX,[BP+26]
  SUB DX,[BP+8]
  MOV BX,[BP+24]
  ADD BX,[BP+10]
  EPIXEL
  MOV DX,[BP+26]
  ADD DX,[BP+8]
  MOV BX,[BP+24]
  SUB BX,[BP+10]
  EPIXEL
  MOV DX,[BP+26]
  SUB DX,[BP+8]
  MOV BX,[BP+24]
  SUB BX,[BP+10]
  EPIXEL
  MOV BX,[BP+10]
  INC BX
  MOV [BP+10],BX
  MOV BX,[BP+6]
  ADD BX,[BP+22]
  MOV [BP+6],BX
  MOV BX,[BP+2]
  ADD BX,[BP+6]
  MOV [BP+2],BX
  CMP BX,0
  JLE Circle6
  JMP Circle8
Circle7:
  MOV DX,[BP+26]
  ADD DX,[BP+8]
  MOV BX,[BP+24]
  ADD BX,[BP+10]
  EPIXEL
  MOV DX,[BP+26]
  SUB DX,[BP+8]
  MOV BX,[BP+24]
  ADD BX,[BP+10]
  EPIXEL
  MOV DX,[BP+26]
  ADD DX,[BP+8]
  MOV BX,[BP+24]
  SUB BX,[BP+10]
  EPIXEL
  MOV DX,[BP+26]
  SUB DX,[BP+8]
  MOV BX,[BP+24]
  SUB BX,[BP+10]
  EPIXEL
Circle8:
  MOV BX,[BP+4]
  SUB BX,[BP]
  MOV [BP+4],BX
  MOV BX,[BP+2]
  SUB BX,[BP+4]
  MOV [BP+2],BX
  MOV BX,[BP+8]
  DEC BX
  MOV [BP+8],BX
  CMP BX,0
  JNZ Circle5
  MOV DX,[BP+26]
  MOV BX,[BP+24]
  ADD BX,[BP+10]
  EPIXEL
  MOV DX,[BP+26]
  MOV BX,[BP+24]
  SUB BX,[BP+10]
  EPIXEL
EndEllipse:
  POP AX
  POP AX
  POP AX
  POP AX
  POP AX
  POP AX
  POP BP
  RET 12
DQBellipse ENDP

; ---------------------------------------------------------------------------
; DQBbox SUB
; purpose:
;   Draws an empty box with x1,y1 and x2,y2 as the opposite corners with a
;   given color. Please note that must be x1<x2 and y1<y2; no checks are
;   done, and clipping is not supported to speed things up.
; declaration:
;   DECLARE SUB DQBbox(BYVAL Layer,BYVAL x1,BYVAL y1,BYVAL x2,BYVAL y2,
;                       BYVAL Col)
; ---------------------------------------------------------------------------
PUBLIC DQBbox
DQBbox PROC
  ; Stack layout:
  ;
  ; 16  Layer
  ; 14  x1
  ; 12  y1
  ; 10  x2
  ; 08  y2
  ; 06  Col
  ; 04  Basic return segment
  ; 02  Basic return offset
  ; 00  BP
  PUSH BP
  MOV BP,SP
  MOV BX,[BP+16]
  CALL GetLayerSeg
  MOV ES,CX
  MOV DI,[BP+12]
  MOV BX,DI
  SHL DI,8
  SHL BX,6
  ADD DI,BX
  ADD DI,[BP+14]
  MOV CX,[BP+10]
  SUB CX,[BP+14]
  INC CX
  MOV AL,[BP+6]
  MOV AH,AL
  MOV DX,CX
  PUSH CX
  CMP CX,2
  JL BoxByteTest1
  SHR CX,1
  REP STOSW
BoxByteTest1:
  TEST DX,1
  JZ DrawBorders
  STOSB
DrawBorders:
  POP DX
  SUB DX,2
  MOV DI,[BP+12]
  INC DI
  MOV BX,DI
  SHL DI,8
  SHL BX,6
  ADD DI,BX
  ADD DI,[BP+14]
  MOV CX,[BP+8]
  SUB CX,[BP+12]
  DEC CX
  PUSH CX
  CMP CX,0
  JLE LowerBorder
BorderLoop:
  STOSB
  ADD DI,DX
  STOSB
  ADD DI,318
  SUB DI,DX
  LOOP BorderLoop
LowerBorder:
  POP CX
  CMP CX,0
  JL EndBox
  MOV CX,[BP+10]
  SUB CX,[BP+14]
  INC CX
  MOV DX,CX
  CMP CX,2
  JL BoxByteTest2
  SHR CX,1
  REP STOSW
BoxByteTest2:
  TEST DX,1
  JZ EndBox
  STOSB
EndBox:
  POP BP
  RET 12
DQBbox ENDP

; ---------------------------------------------------------------------------
; DQBboxf SUB
; purpose:
;   Draws an full box with x1,y1 and x2,y2 as the opposite corners with a
;   given color. Please note that must be x1<x2 and y1<y2; no checks are
;   done and clipping is not supported to speed things up.
; declaration:
;   DECLARE SUB DQBboxf(BYVAL Layer,BYVAL x1,BYVAL y1,BYVAL x2,BYVAL y2,
;                       BYVAL Col)
; ---------------------------------------------------------------------------
PUBLIC DQBboxf
DQBboxf PROC
  ; Stack layout:
  ;
  ; 16  Layer
  ; 14  x1
  ; 12  y1
  ; 10  x2
  ; 08  y2
  ; 06  Col
  ; 04  Basic return segment
  ; 02  Basic return offset
  ; 00  BP
  PUSH BP
  MOV BP,SP
  MOV BX,[BP+16]
  CALL GetLayerSeg
  MOV ES,CX
  MOV DI,[BP+12]
  MOV BX,DI
  SHL DI,8
  SHL BX,6
  ADD DI,BX
  ADD DI,[BP+14]
  MOV CX,[BP+10]
  SUB CX,[BP+14]
  INC CX
  MOV DX,[BP+8]
  SUB DX,[BP+12]
  INC DX
  MOV AL,[BP+6]
  MOV AH,AL
BoxfLoop:
  PUSH CX
  SHR CX,1
  CMP CX,0
  JE BoxfByteTest
  REP STOSW
BoxfByteTest:
  POP CX
  TEST CX,1
  JZ NextBoxfLine
  STOSB
NextBoxfLine:
  ADD DI,320
  SUB DI,CX
  DEC DX
  JNZ BoxfLoop
  POP BP
  RET 12
DQBboxf ENDP

; ---------------------------------------------------------------------------
; DQBfilterBox SUB
; purpose:
;   Draws a translucent full box with x1,y1 and x2,y2 as the opposite corners
;   with a given color. Please note that must be x1<x2 and y1<y2; no checks
;   are done and clipping is not supported to speed things up.
; declaration:
;   DECLARE SUB DQBfilterBox(BYVAL Layer,BYVAL x1,BYVAL y1,BYVAL x2,BYVAL y2,
;                       BYVAL Col)
; ---------------------------------------------------------------------------
PUBLIC DQBfilterBox
DQBfilterBox PROC
  ; Stack layout:
  ;
  ; 18  Layer
  ; 16  x1
  ; 14  y1
  ; 12  x2
  ; 10  y2
  ; 08  Col
  ; 06  Basic return segment
  ; 04  Basic return offset
  ; 02  DS
  ; 00  BP
  PUSH DS
  PUSH BP
  MOV BP,SP
  CMP BMapActive,0
  JE EndFilterBox
  MOV BX,[BP+18]
  CALL GetLayerSeg
  MOV ES,CX
  MOV DI,[BP+14]
  MOV BX,DI
  SHL DI,8
  SHL BX,6
  ADD DI,BX
  ADD DI,[BP+16]
  MOV DX,[BP+10]
  SUB DX,[BP+14]
  INC DX
  MOV AX,BMapSeg
  MOV DS,AX
fBoxYloop:
  MOV CX,[BP+12]
  SUB CX,[BP+16]
  INC CX
  PUSH DX
fBoxXloop:
  MOV BL,ES:[DI]
  MOV AX,[BP+8]
  MOV BH,AL
  MOV AL,DS:[BX]
  STOSB
  LOOP fBoxXloop
  POP DX
  ADD DI,320
  SUB DI,[BP+12]
  ADD DI,[BP+16]
  DEC DI
  DEC DX
  JNZ fBoxYloop
EndFilterBox:
  POP BP
  POP DS
  RET 12
DQBfilterBox ENDP

; ---------------------------------------------------------------------------
; DQBscroll SUB
; purpose:
;   Scrolls specified layer with dx and dy increments. It's up to the
;   programmer to draw the layer area that should be replaced.
; declaration:
;   DECLARE SUB DQBscroll(BYVAL Layer,BYVAL dx,BYVAL dy)
; ---------------------------------------------------------------------------
PUBLIC DQBscroll
DQBscroll PROC
  ; Stack layout:
  ;
  ; 12    Layer
  ; 10    dx
  ; 08    dy
  ; 06    Basic return segment
  ; 04    Basic return offset
  ; 02    DS
  ; 00    BP
  PUSH DS
  PUSH BP
  MOV BP,SP
  MOV BX,[BP+12]
  CALL GetLayerSeg  ; \
  MOV DS,CX         ;  |- Sets both source and destination segments to the
  MOV ES,CX         ; /   specified layer segment
  MOV DX,[BP+08]
  MOV AX,DX
  SHL DX,8
  SHL AX,6
  ADD DX,AX
  ADD DX,[BP+10]    ; DX holds the number of bytes to shift
  CMP DX,0
  JGE Positive
  CLD
  MOV AX,-1
  MUL DX            ; Makes DX positive
  INC DX
  XOR SI,SI
  MOV DI,DX
  MOV CX,64000
  SUB CX,DX         ; BX holds the number of bytes to copy
  JMP Scroll
Positive:
  STD               ; Sets the direction flag
  MOV CX,64000
  SUB CX,DX
  MOV SI,CX
  INC CX            ; BX holds the number of bytes to copy
  MOV DI,64000
Scroll:
  REP MOVSB         ; Moves the bytes
  CLD               ; Clears the direction flag
  POP BP
  POP DS
  RET 6
DQBscroll ENDP

; ---------------------------------------------------------------------------
; DQBwait
; purpose:
;   Waits for video vertical retrace as many times as specified when called
; declaration:
;   DECLARE SUB DQBwait(BYVAL times)
; ---------------------------------------------------------------------------
PUBLIC DQBwait
DQBwait PROC
  PUSH BP
  MOV BP,SP
  MOV CX,[BP+6]
  MOV DX,3DAh       ; Vertical retrace port
Loop1:
  IN AL,DX
  AND AL,8
  JNZ Loop1
Loop2:
  IN AL,DX
  AND AL,8
  JZ Loop2
  LOOP Loop1
  POP BP
  RET 2
DQBwait ENDP

; ---------------------------------------------------------------------------
; DQBsetTransPut SUB
; purpose:
;   Sets the transparent sprite put mode. Next DQBput calls will draw
;   transparent sprites.
; declaration:
;   DECLARE SUB DQBsetTransPut()
; ---------------------------------------------------------------------------
PUBLIC DQBsetTransPut
DQBsetTransPut PROC
  MOV PutMode,0
  RET
DQBsetTransPut ENDP

; ---------------------------------------------------------------------------
; DQBsetSolidPut SUB
; purpose:
;   Sets the solid sprite put mode. Next DQBput calls will draw solid sprites.
; declaration:
;   DECLARE SUB DQBsetSolidPut()
; ---------------------------------------------------------------------------
PUBLIC DQBsetSolidPut
DQBsetSolidPut PROC
  MOV PutMode,1
  RET
DQBsetSolidPut ENDP

; ---------------------------------------------------------------------------
; DQBsize FUNCTION
; purpose:
;   Returns the amount of bytes needed to store the graphics contained into
;   a specified box. This value can be used to declare arrays and to use them
;   with the DQBget and DQBput routines.
; declaration:
;   DECLARE FUNCTION DQBsize(BYVAL x1,BYVAL y1,BYVAL x2,BYVAL y2)
; ---------------------------------------------------------------------------
PUBLIC DQBsize
DQBsize PROC
  ; Stack layout:
  ;
  ; 12  x1
  ; 10  y1
  ; 08  x2
  ; 06  y2
  ; 04  Basic return segment
  ; 02  Basic return offset
  ; 00  BP
  PUSH BP
  MOV BP,SP
  MOV AX,[BP+8]     ; Gets the x2 value into BX
  SUB AX,[BP+12]    ; Subtract x1 to x2...
  INC AX            ; ...and increase the result by 1
  MOV DX,[BP+6]     ; Gets the y2 value into DX
  SUB DX,[BP+10]    ; Subtract y1 to y2...
  INC DX            ; ...and increase the result by 1
  MUL DX            ; Let's find the area in pixels
  ADD AX,4          ; Add 4 bytes of sprite header
  POP BP
  RET 8
DQBsize ENDP

; ---------------------------------------------------------------------------
; DQBget SUB
; purpose:
;   Gets data for a sprite and puts the info into the given sprite array to
;   be used with DQBput. DQBget sprite format is different from the one used
;   by normal GET and PUT, so pay attention. It must be x1<x2 and y1<y2; the
;   routine is optimized if the sprite width is a multiple of 2 or 4 (better)
; declaration:
;   DECLARE SUB DQBget(BYVAL Layer,BYVAL x1,BYVAL y1,BYVAL x2,BYVAL y2,
;                       BYVAL BufferSeg,BYVAL BufferOff)
; ---------------------------------------------------------------------------
PUBLIC DQBget
DQBget PROC
  ; Stack layout (after startup):
  ;
  ; 20  (24)  Layer
  ; 18  (22)  x1
  ; 16  (20)  y1
  ; 14  (18)  x2
  ; 12  (16)  y2
  ; 10  (14)  Buffer segment
  ; 08  (12)  Buffer offset
  ; 06  (10)  Basic return segment
  ; 04  (08)  Basic return offset
  ; 02  (06)  DS
  ; 00  (04)  BP
  ;     (02)  Height
  ;     (00)  Width
  PUSH DS
  PUSH BP
  MOV BP,SP
  MOV AX,[BP+14]
  SUB AX,[BP+18]
  INC AX            ; AX holds the width
  MOV BX,[BP+12]
  SUB BX,[BP+16]
  INC BX            ; BX holds the height
  PUSH BX           ; Pushes the height onto the stack
  PUSH AX           ; Pushes the width onto the stack
  MOV BP,SP         ; Gets the new stack pointer
  MOV BX,[BP+24]
  CALL GetLayerSeg
  MOV DS,CX         ; Sets the source segment as the specified layer segment
  MOV SI,[BP+20]
  MOV BX,SI
  SHL SI,8
  SHL BX,6
  ADD SI,BX
  ADD SI,[BP+22]    ; SI now points to (x1,y1) onto the layer
  MOV BX,[BP+14]
  MOV ES,BX
  MOV DI,[BP+12]    ; ES:DI points to the buffer
  MOV DX,[BP+2]     ; DX holds the height of the sprite...
  MOV BX,[BP]       ; ...and BX its width
  SHL BX,3          ; Multiply width by 8 for GET/PUT compatibility
  MOV ES:[DI],BX    ; Stores the width and the height into the sprite buffer
  MOV ES:[DI+2],DX  ; for later use with DQBput
  ADD DI,4
  MOV BX,[BP]
  AND BX,3
  CMP BX,0
  JE DWordGetLoop   ; Width is multiple of 4: DWORDs can be used (really fast)
  MOV BX,[BP]
  AND BX,1
  CMP BX,0
  JE WordGetLoop    ; Width is multiple of 2: let's use WORDs (fast)
ByteGetLoop:
  MOV CX,[BP]       ; No optimization is possible: using bytes copy (slow)
  REP MOVSB         ; Copies "width" bytes
  SUB SI,[BP]
  ADD SI,320
  DEC DX
  JNZ ByteGetLoop
  JMP EndGet
WordGetLoop:
  MOV CX,[BP]
  SHR CX,1
  REP MOVSW         ; Copies "width\2" words
  SUB SI,[BP]
  ADD SI,320
  DEC DX
  JNZ WordGetLoop
  JMP EndGet
DWordGetLoop:
  MOV CX,[BP]
  SHR CX,2
  REP MOVSD         ; Copies "width\4" dwords
  SUB SI,[BP]
  ADD SI,320
  DEC DX
  JNZ DWordGetLoop
EndGet:
  POP BX
  POP BX
  POP BP
  POP DS
  RET 14
DQBget ENDP

; ---------------------------------------------------------------------------
; DQBsetClipBox SUB
; purpose:
;   Sets the actual clipping box used by the graphical functions.
; declaration:
;   DECLARE SUB DQBsetClipBox(BYVAL x1,BYVAL y1,BYVAL x2,BYVAL y2)
; ---------------------------------------------------------------------------
PUBLIC DQBsetClipBox
DQBsetClipBox PROC
  ; Stack layout
  ;
  ; 12  x1
  ; 10  y1
  ; 08  x2
  ; 06  y2
  ; 04  Basic return segment
  ; 02  Basic return offset
  ; 00  BP
  PUSH BP
  MOV BP,SP
  MOV AX,[BP+12]
  MOV BX,[BP+8]
  CMP BX,AX         ; Is x1>x2?
  JG CheckOther
  XCHG AX,BX        ; Yes: exchange them
CheckOther:
  MOV ClipX1,AX
  MOV ClipX2,BX
  MOV AX,[BP+10]
  MOV BX,[BP+6]
  CMP BX,AX         ; Is y1>y1?
  JG EndSetClipBox
  XCHG AX,BX        ; Yes: exchange them
EndSetClipBox:
  MOV ClipY1,AX
  MOV ClipY2,BX
  POP BP
  RET 8
DQBsetClipBox ENDP

; ---------------------------------------------------------------------------
; DQBput SUB
; purpose:
;   Draws a given sprite at specified coordinates on the given layer. If the
;   PutMode internal variable is 0, black pixels (color 0 assumed) are
;   skipped, allowing a transparent sprite drawing. The sprite automatically
;   clips when it reaches the actual clipping box borders; any x and y value
;   are accepted, as long as they're INTEGERs. The sprite can have been
;   stored into an array with DQBget as well as with GET, as the data format
;   is the same.
; declaration:
;   DECLARE SUB DQBput(BYVAL Layer,BYVAL x,BYVAL y,BYVAL BufferSeg,
;                       BYVAL BufferOff)
; ---------------------------------------------------------------------------
PUBLIC DQBput
DQBput PROC
  ; Stack layout
  ;
  ; 16  Layer
  ; 14  x
  ; 12  y
  ; 10  Buffer segment
  ; 08  Buffer offset
  ; 06  Basic return segment
  ; 04  Basic return offset
  ; 02  DS
  ; 00  BP
  PUSH DS
  PUSH BP
  MOV BP,SP
  MOV SI,[BP+08]    ; SI points to the buffer offset
  MOV BX,[BP+16]    ; Gets the layer number...
  CALL GetLayerSeg
  MOV ES,CX         ; ...and sets ES to its segment
  MOV DI,[BP+12]    ; DI contains the y value
  DEC DI
  MOV AX,PutMode    ; Puts the put mode and the clipping box coordinates
  PUSH AX           ; on the stack for later use
  MOV AX,ClipX1
  PUSH AX
  MOV AX,ClipX2
  PUSH AX
  MOV AX,ClipY1
  PUSH AX
  MOV AX,ClipY2
  PUSH AX
  MOV BX,[BP+10]
  MOV DS,BX         ; DS points to the buffer segment
  MOV BX,DI
  SHL DI,8
  SHL BX,6
  ADD DI,BX
  ADD DI,[BP+14]    ; DI points to (x,y-1) on specified layer
  LODSW
  SHR AX,3          ; Divide width by 8 for GET/PUT compatibility
  PUSH AX           ; Gets the sprite width and puts it onto the stack
  LODSW
  MOV AH,AL         ; Sets AH to the height of the sprite
  INC AH
  XOR AL,AL
  MOV BP,SP
  MOV BX,[BP+24]    ; BX holds the value y-1
  DEC BX
  ADD DI,[BP]
  CMP [BP+10],AL    ; Are we in solid put mode?
  JNE SolidYloop    ; Yes: jump to the solid put loop
Yloop:
  ADD SI,[BP]
  ADD DI,320        ; Adds 320 to DI
  INC BX            ; Next line
  DEC AH
  JZ EndPut         ; Last sprite line reached
  CMP BX,[BP+4]     ; Are we out of the upper clipping border?
  JL Yloop          ; Yes: increase the line counter
  CMP BX,[BP+2]     ; Are we out of the lower clipping border?
  JG EndPut         ; Yes: exit DQBput
  SUB DI,[BP]       ; Subtracts the sprite width to DI and SI
  SUB SI,[BP]
  MOV DX,[BP+26]    ; DX holds the current x value
  MOV CX,[BP]       ; Sets CX to the sprite width
Xloop:
  LODSB             ; Gets a pixel from the buffer
  CMP AL,0
  JE SkipPixel      ; Transparent pixel: skip it!
  CMP DX,[BP+8]
  JL SkipPixel      ; Out of left layer border: skip it!
  CMP DX,[BP+6]
  JG SkipPixel      ; Out of right layer border: skip it!
  STOSB             ; All checks done: writes pixel on the layer
  INC DX            ; Next pixel
  LOOP Xloop        ; Repeats sprite width times
  JMP Yloop         ; Next line
SkipPixel:
  INC DI            ; Increase DI without writing pixel
  INC DX            ; Next pixel
  LOOP Xloop        ; Repeats sprite width times
  JMP Yloop         ; Next line
SolidYloop:
  ADD SI,[BP]
  ADD DI,320        ; Adds 320 to DI
  INC BX            ; Next line
  DEC AH
  JZ EndPut         ; Last sprite line reached?
  CMP BX,[BP+4]     ; Are we out of the upper clipping border?
  JL SolidYloop     ; Yes: increase the line counter
  CMP BX,[BP+2]     ; Are we out of the lower clipping border?
  JG EndPut         ; Yes: exit DQBput
  SUB DI,[BP]       ; Subtracts the sprite width to DI and SI
  SUB SI,[BP]
  MOV DX,[BP+26]    ; DX holds the current x value
  MOV CX,[BP]       ; Sets CX to the sprite width
SolidXloop:
  LODSB             ; Gets a pixel from the buffer
  CMP DX,[BP+8]
  JL SkipSolidPixel    ; Out of left layer border: skip it!
  CMP DX,[BP+6]
  JG SkipSolidPixel    ; Out of right layer border: skip it!
  STOSB             ; All checks done: writes pixel on the layer
  INC DX            ; Next pixel
  LOOP SolidXloop   ; Repeats sprite width times
  JMP SolidYloop    ; Next line
SkipSolidPixel:
  INC DI            ; Increase DI without writing pixel
  INC DX            ; Next pixel
  LOOP SolidXloop   ; Repeats sprite width times
  JMP SolidYloop    ; Next line
EndPut:
  POP BX            ; Release stack memory
  POP BX
  POP BX
  POP BX
  POP BX
  POP BX
  POP BP
  POP DS
  RET 10
DQBput ENDP

; ---------------------------------------------------------------------------
; DQBsPut SUB
; purpose:
;   Draws a given sprite at specified coordinates on the given layer, scaled
;   to the specified new width and height. This sub works almost the same as
;   DQBput, including the support for clipping and transparency.
; declaration:
;   DECLARE SUB DQBsPut(BYVAL Layer,BYVAL x,BYVAL y,BYVAL BufferSeg,
;                       BYVAL BufferOff,BYVAL NewWidth,BYVAL NewHeight)
; ---------------------------------------------------------------------------
PUBLIC DQBsPut
DQBsPut PROC
  ; Stack layout
  ;
  ; 40  Layer
  ; 38  x
  ; 36  y
  ; 34  Buffer segment
  ; 32  Buffer offset
  ; 30  NewWidth
  ; 28  NewHeight
  ; 26  Basic return segment
  ; 24  Basic return offset
  ; 22  DS
  ; 20  BP
  ; 18  PutMode
  ; 16  ClipX1
  ; 14  ClipX2
  ; 12  ClipY1
  ; 10  ClipY2
  ; 08  xi
  ; 06  yi
  ; 04  yadd
  ; 02  original width
  ; 00  original height
  PUSH DS
  PUSH BP
  MOV AX,PutMode
  PUSH AX
  MOV AX,ClipX1
  PUSH AX
  MOV AX,ClipX2
  PUSH AX
  MOV AX,ClipY1
  PUSH AX
  MOV AX,ClipY2
  PUSH AX
  PUSH AX           ; keep stack space for xi
  PUSH AX           ; keep stack space for yi
  PUSH AX           ; keep stack space for yadd
  MOV BP,SP
  MOV BX,[BP+36]
  CALL GetLayerSeg
  MOV ES,CX
  MOV AX,[BP+30]
  MOV DS,AX
  MOV SI,[BP+28]
  MOV AX,[SI]
  SHR AX,3
  PUSH AX
  MOV AX,[SI+2]
  PUSH AX
  MOV BP,SP
  SHL AX,7          ; 9.7 fixed point math used to allow sprites of up to
  MOV BX,[BP+28]    ; 511x511 pixels
  XOR DX,DX
  DIV BX
  MOV [BP+6],AX     ; yi stored
  MOV AX,[BP+2]
  SHL AX,7          ; Fixed point math used (9.7)
  MOV BX,[BP+30]
  XOR DX,DX
  DIV BX
  MOV [BP+8],AX     ; xi stored
  XOR CX,CX
  MOV [BP+4],CX
  MOV SI,[BP+32]
  ADD SI,4          ; SI holds the actual sprite offset position
  MOV AX,[BP+18]
  CMP AX,0
  JNE SolidScaledYloop
ScaledYloop:
  MOV DI,[BP+36]
  ADD DI,CX
  CMP DI,[BP+12]
  JL SkipScaledLine
  CMP DI,[BP+10]
  JG SkipScaledLine
  MOV BX,DI
  SHL DI,8
  SHL BX,6
  ADD DI,BX
  ADD DI,[BP+38]
  PUSH CX
  XOR CX,CX
  XOR DX,DX
  XOR BX,BX
ScaledXloop:
  MOV AX,[BP+38]
  ADD AX,CX
  CMP AX,[BP+16]
  JL SkipScaledPixel
  CMP AX,[BP+14]
  JG SkipScaledPixel
  MOV AL,[SI+BX]
  CMP AL,0
  JE SkipScaledPixel
  STOSB
  ADD DX,[BP+8]
  MOV BX,DX
  SHR BX,7
  INC CX
  CMP CX,[BP+30]
  JL ScaledXloop
  POP CX
  JMP SkipScaledLine
SkipScaledPixel:
  INC DI
  ADD DX,[BP+8]
  MOV BX,DX
  SHR BX,7
  INC CX
  CMP CX,[BP+30]
  JL ScaledXloop
  POP CX
SkipScaledLine:
  MOV SI,[BP+32]
  ADD SI,4
  MOV DX,[BP+6]
  ADD [BP+4],DX
  MOV AX,[BP+4]
  SHR AX,7
  MOV BX,[BP+2]
  XOR DX,DX
  MUL BX
  ADD SI,AX
  INC CX
  CMP CX,[BP+28]
  JL ScaledYloop
  POP AX
  POP AX
  POP AX
  POP AX
  POP AX
  POP AX
  POP AX
  POP AX
  POP AX
  POP AX
  POP BP
  POP DS
  RET 14
SolidScaledYloop:
  MOV DI,[BP+36]
  ADD DI,CX
  CMP DI,[BP+12]
  JL SolidSkipScaledLine
  CMP DI,[BP+10]
  JG SolidSkipScaledLine
  MOV BX,DI
  SHL DI,8
  SHL BX,6
  ADD DI,BX
  ADD DI,[BP+38]
  PUSH CX
  XOR CX,CX
  XOR DX,DX
  XOR BX,BX
SolidScaledXloop:
  MOV AX,[BP+38]
  ADD AX,CX
  CMP AX,[BP+16]
  JL SolidSkipScaledPixel
  CMP AX,[BP+14]
  JG SolidSkipScaledPixel
  MOV AL,[SI+BX]
  STOSB
  ADD DX,[BP+8]
  MOV BX,DX
  SHR BX,7
  INC CX
  CMP CX,[BP+30]
  JL SolidScaledXloop
  POP CX
  JMP SolidSkipScaledLine
SolidSkipScaledPixel:
  INC DI
  ADD DX,[BP+8]
  MOV BX,DX
  SHR BX,7
  INC CX
  CMP CX,[BP+30]
  JL SolidScaledXloop
  POP CX
SolidSkipScaledLine:
  MOV SI,[BP+32]
  ADD SI,4
  MOV DX,[BP+6]
  ADD [BP+4],DX
  MOV AX,[BP+4]
  SHR AX,7
  MOV BX,[BP+2]
  XOR DX,DX
  MUL BX
  ADD SI,AX
  INC CX
  CMP CX,[BP+28]
  JL SolidScaledYloop
  POP AX
  POP AX
  POP AX
  POP AX
  POP AX
  POP AX
  POP AX
  POP AX
  POP AX
  POP AX
  POP BP
  POP DS
  RET 14
DQBsPut ENDP

; ---------------------------------------------------------------------------
; DQBrPut SUB
; purpose:
;   Draws a sprite rotated by a specified angle. The original sprite size
;   is kept, and rotated pixels that lies outside the sprite area will not be
;   drawn; this allows to use a pixel-perfect rotation, but it'll require
;   that you get your sprite into a larger area, in order to allow a rotation
;   without loosing any pixel.
; declaration:
;   DECLARE SUB DQBrPut(BYVAL Layer,BYVAL x,BYVAL y,BYVAL BufferSeg,
;                       BYVAL BufferOff,BYVAL Angle,BYVAL Zoom)
; ---------------------------------------------------------------------------
PUBLIC DQBrPut
DQBrPut PROC
  ; Stack layout
  ;
  ; 46  Layer
  ; 44  x
  ; 42  y
  ; 40  BufferSeg
  ; 38  BufferOff
  ; 36  Angle
  ; 34  Zoom
  ; 32  Basic return segment
  ; 30  Basic return offset
  ; 28  DS
  ; 26  BP
  ; 24  PutMode
  ; 22  ClipX1
  ; 20  ClipX2
  ; 18  ClipY1
  ; 16  ClipY2
  ; 14  Sprite width
  ; 12  Sprite height
  ; 10  Sprite width \ 2
  ; 08  Sprite height \ 2
  ; 04  -- Internal variables
  ; 00  /
  PUSH DS
  PUSH BP
  MOV AX,PutMode
  PUSH AX
  MOV AX,ClipX1
  PUSH AX
  MOV AX,ClipX2
  PUSH AX
  MOV AX,ClipY1
  PUSH AX
  MOV AX,ClipY2
  PUSH AX
  PUSH AX
  PUSH AX
  PUSH AX
  PUSH AX
  SUB SP,8
  MOV BP,SP
  MOV BX,[BP+46]
  CALL GetLayerSeg
  MOV ES,CX
  MOV DI,[BP+42]
  MOV SI,DI
  SHL DI,8
  SHL SI,6
  ADD DI,SI
  ADD DI,[BP+44]
  MOV AX,[BP+36]
  ADD AX,64
  XOR AH,AH
  MOV SI,AX
  SHL SI,1
  MOV AX,SinTable[SI]
  MOV BX,100
  IMUL BX
  SHL EDX,16
  MOV DX,AX
  MOV EAX,EDX
  CDQ
  XOR ESI,ESI
  MOV SI,[BP+34]
  IDIV ESI
  MOV [BP+4],EAX
  MOV AX,[BP+36]
  XOR AH,AH
  MOV SI,AX
  SHL SI,1
  MOV AX,SinTable[SI]
  IMUL BX
  SHL EDX,16
  MOV DX,AX
  MOV EAX,EDX
  CDQ
  XOR ESI,ESI
  MOV SI,[BP+34]
  IDIV ESI
  MOV [BP],EAX
  MOV AX,[BP+40]
  MOV DS,AX
  MOV SI,[BP+38]
  XOR EAX,EAX
  LODSW
  SHR AX,3
  MOV [BP+14],AX
  SHR AX,1
  MOV [BP+10],AX
  LODSW
  MOV [BP+12],AX
  SHR AX,1
  MOV [BP+8],AX
  MOV EDX,[BP]
  IMUL EDX
  MOV ESI,EAX
  NEG ESI
  XOR EAX,EAX
  MOV AX,256
  MOV DX,[BP+8]
  MUL DX
  ADD ESI,EAX
  XOR EAX,EAX
  MOV AX,[BP+10]
  MOV EDX,[BP+4]
  IMUL EDX
  SUB ESI,EAX
  XOR EAX,EAX
  MOV AX,[BP+10]
  MOV EDX,[BP]
  IMUL EDX
  MOV EBX,EAX
  NEG EBX
  XOR EAX,EAX
  MOV AX,256
  MOV DX,[BP+10]
  MUL DX
  ADD EBX,EAX
  XOR EAX,EAX
  MOV AX,[BP+8]
  MOV EDX,[BP+4]
  IMUL EDX
  ADD EAX,EBX
  XOR CX,CX
RotYloop:
  PUSH CX
  PUSH EAX
  PUSH ESI
  ADD DI,[BP+14]
  ADD CX,[BP+42]
  CMP CX,[BP+18]
  JL RotSkipLine
  CMP CX,[BP+16]
  JG RotSkipLine
  SUB DI,[BP+14]
  XOR CX,CX
RotXloop:
  PUSH CX
  ADD CX,[BP+44]
  CMP CX,[BP+22]
  JL RotSkip
  CMP CX,[BP+20]
  JG RotSkip
  MOV EBX,EAX
  SHR EBX,8
  MOV EDX,ESI
  SHR EDX,8
  CMP DX,0
  JL RotSkip
  CMP DX,[BP+12]
  JGE RotSkip
  CMP BX,0
  JL RotSkip
  CMP BX,[BP+14]
  JGE RotSkip
  PUSH EAX
  PUSH EDX
  MOV AX,[BP+14]
  MUL DX
  ADD BX,AX
  ADD BX,[BP+38]
  MOV BL,DS:[BX+4]
  MOV AX,[BP+24]
  CMP AX,0
  JNE SkipTest
  CMP BL,0
  JE RotSkip0
SkipTest:
  MOV ES:[DI],BL
RotSkip0:
  POP EDX
  POP EAX
RotSkip:
  POP CX
  ADD EAX,[BP]
  ADD ESI,[BP+4]
  INC DI
  INC CX
  CMP CX,[BP+14]
  JL RotXloop
RotSkipLine:
  ADD DI,320
  SUB DI,[BP+14]
  POP ESI
  POP EAX
  SUB EAX,[BP+4]
  ADD ESI,[BP]
  POP CX
  INC CX
  CMP CX,[BP+12]
  JL RotYloop
  ADD SP,8
  POP AX
  POP AX
  POP AX
  POP AX
  POP AX
  POP AX
  POP AX
  POP AX
  POP AX
  POP BP
  POP DS
  RET 14
DQBrPut ENDP

; ---------------------------------------------------------------------------
; DQBfPut SUB
; purpose:
;   Puts a sprite onto specified layer, using a fast algorithm. This routine
;   skips all clipping box checks, and does not allow transparency; it just
;   copy the entire sprite block at the given coordinates.
; declaration:
;   DECLARE SUB DQBfPut(BYVAL Layer,BYVAL x,BYVAL y,BYVAL BufferSeg,
;                       BYVAL BufferOff)
; ---------------------------------------------------------------------------
PUBLIC DQBfPut
DQBfPut PROC
  ; Stack layout
  ;
  ; 16  Layer
  ; 14  x
  ; 12  y
  ; 10  Buffer segment
  ; 08  Buffer offset
  ; 06  Basic return segment
  ; 04  Basic return offset
  ; 02  DS
  ; 00  BP
  PUSH DS
  PUSH BP
  MOV BP,SP
  MOV BX,[BP+16]
  CALL GetLayerSeg
  MOV ES,CX
  MOV DI,[BP+12]
  MOV AX,DI
  SHL AX,8
  SHL DI,6
  ADD DI,AX
  ADD DI,[BP+14]
  MOV AX,[BP+10]
  MOV DS,AX
  MOV SI,[BP+8]
  LODSW
  MOV DX,AX
  SHR DX,3          ; DX holds the sprite width
  MOV BX,320
  SUB BX,DX         ; BX holds the value "320-width" in pixels
  LODSW
  MOV CX,AX         ; CX holds the sprite height
  TEST DX,2
  JNZ FastWordCopy
  TEST DX,1
  JNZ BYTEloop
  SHR DX,2
DWORDloop:
  PUSH CX
  MOV CX,DX
  REP MOVSD
  POP CX
  ADD DI,BX
  LOOP DWORDloop
  POP BP
  POP DS
  RET 10
FastWordCopy:
  TEST DX,1
  JNZ BYTEloop
  SHR DX,1
WORDloop:
  PUSH CX
  MOV CX,DX
  REP MOVSW
  POP CX
  ADD DI,BX
  LOOP WORDloop
  POP BP
  POP DS
  RET 10
BYTEloop:
  PUSH CX
  MOV CX,DX
  REP MOVSB
  POP CX
  ADD DI,BX
  LOOP BYTEloop
  POP BP
  POP DS
  RET 10
DQBfPut ENDP

; ---------------------------------------------------------------------------
; DQBbPut SUB
; purpose:
;   Draws a sprite onto given layer, blending colors using the specified
;   blender map previously built by calling the DQBsetBlenderMap function.
;   This function supports clipping and transparency.
; declaration:
;   DECLARE SUB DQBbPut(BYVAL Layer,BYVAL x,BYVAL y,BYVAL BufferSeg,
;                       BYVAL BufferOff)
; ---------------------------------------------------------------------------
PUBLIC DQBbPut
DQBbPut PROC
  ; Stack layout
  ;
  ; 16  Layer
  ; 14  x
  ; 12  y
  ; 10  Buffer segment
  ; 08  Buffer offset
  ; 06  Basic return segment
  ; 04  Basic return offset
  ; 02  DS
  ; 00  BP
  PUSH DS
  PUSH BP
  MOV BP,SP
  CMP BMapActive,0
  JE EndBPut1
  MOV SI,[BP+08]    ; SI points to the buffer offset
  MOV BX,[BP+16]    ; Gets the layer number...
  CALL GetLayerSeg
  MOV ES,CX         ; ...and sets ES to its segment
  MOV DI,[BP+12]    ; DI contains the y value
  DEC DI
  MOV AX,PutMode    ; Puts the put mode and the clipping box coordinates
  PUSH AX           ; on the stack for later use
  MOV AX,ClipX1
  PUSH AX
  MOV AX,ClipX2
  PUSH AX
  MOV AX,ClipY1
  PUSH AX
  MOV AX,ClipY2
  PUSH AX
  MOV AX,BMapSeg
  MOV GS,AX
  MOV BX,[BP+10]
  MOV DS,BX         ; DS points to the buffer segment
  MOV BX,DI
  SHL DI,8
  SHL BX,6
  ADD DI,BX
  ADD DI,[BP+14]    ; DI points to (x,y-1) on specified layer
  LODSW
  SHR AX,3          ; Divide width by 8 for GET/PUT compatibility
  PUSH AX           ; Gets the sprite width and puts it onto the stack
  LODSW
  MOV AH,AL         ; Sets AH to the height of the sprite
  INC AH
  XOR AL,AL
  MOV BP,SP
  MOV BX,[BP+24]    ; BX holds the value y-1
  DEC BX
  ADD DI,[BP]
  CMP [BP+10],AL    ; Are we in solid put mode?
  JNE bSolidYloop   ; Yes: jump to the solid put loop
bYloop:
  ADD SI,[BP]
  ADD DI,320        ; Adds 320 to DI
  INC BX            ; Next line
  DEC AH
  JZ EndBPut         ; Last sprite line reached
  CMP BX,[BP+4]     ; Are we out of the upper clipping border?
  JL bYloop         ; Yes: increase the line counter
  CMP BX,[BP+2]     ; Are we out of the lower clipping border?
  JG EndBPut        ; Yes: exit DQBput
  SUB DI,[BP]       ; Subtracts the sprite width to DI and SI
  SUB SI,[BP]
  MOV DX,[BP+26]    ; DX holds the current x value
  MOV CX,[BP]       ; Sets CX to the sprite width
bXloop:
  LODSB             ; Gets a pixel from the buffer
  CMP AL,0
  JE bSkipPixel     ; Transparent pixel: skip it!
  CMP DX,[BP+8]
  JL bSkipPixel     ; Out of left layer border: skip it!
  CMP DX,[BP+6]
  JG bSkipPixel     ; Out of right layer border: skip it!
  PUSH BX
  MOV BL,ES:[DI]
  MOV BH,AL
  MOV AL,GS:[BX]
  POP BX
  STOSB             ; All checks done: writes pixel on the layer
  INC DX            ; Next pixel
  LOOP bXloop       ; Repeats sprite width times
  JMP bYloop        ; Next line
bSkipPixel:
  INC DI            ; Increase DI without writing pixel
  INC DX            ; Next pixel
  LOOP bXloop       ; Repeats sprite width times
  JMP bYloop        ; Next line
bSolidYloop:
  ADD SI,[BP]
  ADD DI,320        ; Adds 320 to DI
  INC BX            ; Next line
  DEC AH
  JZ EndBPut        ; Last sprite line reached?
  CMP BX,[BP+4]     ; Are we out of the upper clipping border?
  JL bSolidYloop    ; Yes: increase the line counter
  CMP BX,[BP+2]     ; Are we out of the lower clipping border?
  JG EndBPut        ; Yes: exit DQBput
  SUB DI,[BP]       ; Subtracts the sprite width to DI and SI
  SUB SI,[BP]
  MOV DX,[BP+26]    ; DX holds the current x value
  MOV CX,[BP]       ; Sets CX to the sprite width
bSolidXloop:
  LODSB             ; Gets a pixel from the buffer
  CMP DX,[BP+8]
  JL bSkipSolidPixel    ; Out of left layer border: skip it!
  CMP DX,[BP+6]
  JG bSkipSolidPixel    ; Out of right layer border: skip it!
  PUSH BX
  MOV BL,ES:[DI]
  MOV BH,AL
  MOV AL,GS:[BX]
  POP BX
  STOSB             ; All checks done: writes pixel on the layer
  INC DX            ; Next pixel
  LOOP bSolidXloop  ; Repeats sprite width times
  JMP bSolidYloop   ; Next line
bSkipSolidPixel:
  INC DI            ; Increase DI without writing pixel
  INC DX            ; Next pixel
  LOOP bSolidXloop  ; Repeats sprite width times
  JMP bSolidYloop   ; Next line
EndBPut:
  POP BX            ; Release stack memory
  POP BX
  POP BX
  POP BX
  POP BX
  POP BX
EndBPut1:
  POP BP
  POP DS
  RET 10
DQBbPut ENDP

; ---------------------------------------------------------------------------
; DQBcreateBMap FUNCTION
; purpose:
;   Allocates memory for the blender map
; declaration:
;   DECLARE FUNCTION xDQBcreateBMap()
;   DECLARE FUNCTION DQBcreateBMap()
; ---------------------------------------------------------------------------
PUBLIC xDQBcreateBMap
xDQBcreateBMap PROC
  CMP BMapActive,1
  JE Error_AlreadyUp
  MOV AH,48h
  MOV BX,4096
  INT 21h
  JC Error_AllocBMap
  MOV BMapSeg,AX
  MOV BMapActive,1
  XOR ECX,ECX
  XOR DI,DI
  MOV ES,AX
CreateBMap:
  MOV ES:[DI],CH
  INC ECX
  INC DI
  CMP ECX,65536
  JL CreateBMap
  XOR AX,AX
  RET
Error_AlreadyUp:
  MOV AX,2          ; Error 2: Blender map already created
  RET
Error_AllocBMap:
  MOV AX,1          ; Error 1: Not enough free conventional memory
  RET
xDQBcreateBMap ENDP

; ---------------------------------------------------------------------------
; DQBsetBMap SUB
; purpose:
;   Sets a color entry on the blender map (if active)
; declaration:
;   DECLARE SUB DQBsetBMap(BYVAL ForeCol,BYVAL BackCol,BYVAL NewCol)
; ---------------------------------------------------------------------------
PUBLIC DQBsetBMap
DQBsetBMap PROC
  ; Stack layout:
  ;
  ; 10  ForeCol
  ; 08  BackCol
  ; 06  NewCol
  ; 04  Basic return segment
  ; 02  Basic return offset
  ; 00  BP
  PUSH BP
  MOV BP,SP
  CMP BMapActive,0
  JE EndSetBMap
  MOV AX,BMapSeg
  MOV ES,AX
  MOV AX,[BP+10]
  MOV BX,[BP+8]
  MOV BH,AL
  MOV AL,[BP+6]
  MOV ES:[BX],AL
EndSetBMap:
  POP BP
  RET 6
DQBsetBMap ENDP

; ---------------------------------------------------------------------------
; DQBgetBMap FUNCTION
; purpose:
;   Returns a color entry on the blender map (if active)
; declaration:
;   DECLARE FUNCTION DQBgetBMap(BYVAL ForeCol,BYVAL BackCol)
; ---------------------------------------------------------------------------
PUBLIC DQBgetBMap
DQBgetBMap PROC
  ; Stack layout:
  ;
  ; 08  ForeCol
  ; 06  BackCol
  ; 04  Basic return segment
  ; 02  Basic return offset
  ; 00  BP
  PUSH BP
  MOV BP,SP
  MOV AX,-1
  CMP BMapActive,0
  JE EndGetBMap
  MOV AX,BMapSeg
  MOV ES,AX
  MOV AX,[BP+8]
  MOV BX,[BP+6]
  MOV BH,AL
  XOR AH,AH
  MOV AL,ES:[BX]
EndGetBMap:
  POP BP
  RET 4
DQBgetBMap ENDP

; ---------------------------------------------------------------------------
; DQBloadBMap FUNCTION
; purpose:
;   Loads the blender map from a specified file
; declaration:
;   DECLARE FUNCTION xDQBloadBMap(BYVAL FileSeg,BYVAL FileOff)
;   DECLARE FUNCTION DQBloadBMap(FileName AS STRING)
; ---------------------------------------------------------------------------
PUBLIC xDQBloadBMap
xDQBloadBMap PROC
  ; Stack layout:
  ;
  ; 10  FileSeg
  ; 08  FileOff
  ; 06  Basic return segment
  ; 04  Basic return offset
  ; 02  DS
  ; 00  BP
  PUSH DS
  PUSH BP
  MOV BP,SP
  CMP BMapActive,0
  JE Error_NoBMapLoad
  MOV AX,BMapSeg
  MOV ES,AX
  MOV AX,03D00h
  MOV BX,[BP+10]
  MOV DS,BX
  MOV DX,[BP+8]
  INT 21h
  JC Error_OpenBMapLoad
  MOV BX,AX
  MOV AX,ES
  MOV DS,AX
  MOV AH,03Fh
  XOR DX,DX
  MOV CX,08000h
  INT 21h
  JC Error_ReadBMap
  MOV AH,03Fh
  MOV DX,08000h
  MOV CX,08000h
  INT 21h
  JC Error_ReadBMap
  MOV AH,03Eh
  INT 21h
  XOR AX,AX
  JMP EndLoadBMap
Error_NoBMapLoad:
  MOV AH,03Eh
  INT 21h
  MOV AX,1          ; Error 1: Blender map not yet created
  JMP EndLoadBMap
Error_OpenBMapLoad:
  MOV AH,03Eh
  INT 21h
  MOV AX,2          ; Error 2: Unable to open file or file does not exist
  JMP EndLoadBMap
Error_ReadBMap:
  MOV AH,03Eh
  INT 21h
  MOV AX,3          ; Error 3: General file read error
EndLoadBMap:
  POP BP
  POP DS
  RET 4
xDQBloadBMap ENDP

; ---------------------------------------------------------------------------
; DQBsaveBMap FUNCTION
; purpose:
;   Saves the blender map into a specified file
; declaration:
;   DECLARE FUNCTION xDQBsaveBMap(BYVAL FileSeg,BYVAL FileOff)
;   DECLARE FUNCTION DQBsaveBMap(FileName AS STRING)
; ---------------------------------------------------------------------------
PUBLIC xDQBsaveBMap
xDQBsaveBMap PROC
  ; Stack layout:
  ;
  ; 10  FileSeg
  ; 08  FileOff
  ; 06  Basic return segment
  ; 04  Basic return offset
  ; 02  DS
  ; 00  BP
  PUSH DS
  PUSH BP
  MOV BP,SP
  CMP BMapActive,0
  JE Error_NoBMapSave
  MOV AX,BMapSeg
  MOV ES,AX
  MOV AH,03Ch
  MOV CX,020h
  MOV BX,[BP+10]
  MOV DS,BX
  MOV DX,[BP+8]
  INT 21h
  JC Error_OpenBMapSave
  MOV BX,AX
  MOV AX,ES
  MOV DS,AX
  MOV AH,040h
  XOR DX,DX
  MOV CX,08000h
  INT 21h
  JC Error_WriteBMap
  MOV AH,040h
  MOV DX,08000h
  MOV CX,08000h
  INT 21h
  JC Error_WriteBMap
  MOV AH,03Eh
  INT 21h
  XOR AX,AX
  JMP EndSaveBMap
Error_NoBMapSave:
  MOV AH,03Eh
  INT 21h
  MOV AX,1          ; Error 1: Blender map not yet created
  JMP EndSaveBMap
Error_OpenBMapSave:
  MOV AH,03Eh
  INT 21h
  MOV AX,2          ; Error 2: Unable to create file
  JMP EndSaveBMap
Error_WriteBMap:
  MOV AH,03Eh
  INT 21h
  MOV AX,3          ; Error 3: General file write error
EndSaveBMap:
  POP BP
  POP DS
  RET 4
xDQBsaveBMap ENDP

; ---------------------------------------------------------------------------
; DQBsetCol SUB
; purpose:
;   Sets the palette for a specified color
; declaration:
;   DECLARE SUB DQBsetCol(BYVAL ColorIndex,BYVAL r,BYVAL g,BYVAL b)
; ---------------------------------------------------------------------------
PUBLIC DQBsetCol
DQBsetCol PROC
  ; Stack layout:
  ;
  ; 12  Color index
  ; 10  r
  ; 08  g
  ; 06  b
  ; 04  Basic return segment
  ; 02  Basic return offset
  ; 00  BP
  PUSH BP
  MOV BP,SP
  MOV DX,3C8h       ; Port 3C8h: write palette
  MOV AX,[BP+12]
  OUT DX,AL         ; Sets the color
  MOV DX,3C9h
  MOV AL,[BP+10]
  OUT DX,AL         ; Outs the red value
  MOV AL,[BP+8]
  OUT DX,AL         ; Outs the green value
  MOV AL,[BP+6]
  OUT DX,AL         ; Outs the blue value
  POP BP
  RET 8
DQBsetCol ENDP

; ---------------------------------------------------------------------------
; DQBgetCol SUB
; purpose:
;   Gets the hues of a given color index
; declaration:
;   DECLARE SUB DQBgetCol(BYVAL ColorIndex,r,g,b)
; ---------------------------------------------------------------------------
PUBLIC DQBgetCol
DQBgetCol PROC
  ; Stack layout:
  ;
  ; 12  Color index
  ; 10  r (address)
  ; 08  g (address)
  ; 06  b (address)
  ; 04  Basic return segment
  ; 02  Basic return offset
  ; 00  BP
  PUSH BP
  MOV BP,SP
  MOV DX,3C7h       ; Port 3C7h: read palette
  MOV AX,[BP+12]
  OUT DX,AL         ; Sets the color
  MOV DX,3C9h
  XOR AX,AX
  IN AL,DX          ; Gets the red value...
  MOV BX,[BP+10]
  MOV [BX],AX       ; ...and stores it into the red variable
  IN AL,DX          ; Gets the green value...
  MOV BX,[BP+8]
  MOV [BX],AX       ; ...and stores it into the green variable
  IN AL,DX          ; Gets the blue value...
  MOV BX,[BP+6]
  MOV [BX],AX       ; ...and stores it into the blue variable
  POP BP
  RET 8
DQBgetCol ENDP

; ---------------------------------------------------------------------------
; DQBfindCol FUNCTION
; purpose:
;   Returns the color index that is nearest to the specified hues, by
;   searching into the current palette entries.
; declaration:
;   DECLARE FUNCTION DQBfindCol(BYVAL Red,BYVAL Green,BYVAL Blue)
; ---------------------------------------------------------------------------
PUBLIC DQBfindCol
DQBfindCol PROC
  ; Stack layout
  ;
  ; 16  Red
  ; 14  Green
  ; 12  Blue
  ; 10  Basic return segment
  ; 08  Basic return offset
  ; 06  BP
  ; 04  BestRed
  ; 02  BestGreen
  ; 00  BestBlue
  PUSH BP
  PUSH AX           ; \
  PUSH AX           ;  |- Keep space for best hues found
  PUSH AX           ; /
  MOV BP,SP
  MOV AX,64
  MOV [BP],AX
  MOV [BP+2],AX
  MOV [BP+4],AX
  MOV DX,03C7h
  XOR AX,AX
  OUT DX,AL         ; Starts with color 0
  XOR DI,DI         ; Color found
  XOR CX,CX
FindColLoop:
  MOV DX,03C9h
  IN AL,DX
  MOV BL,AL         ; BL holds the red hue of current color
  IN AL,DX
  MOV BH,AL         ; BH holds the green hue of current color
  IN AL,DX
  XOR AH,AH         ; AX holds the blue hue of current color (word)
  XOR DH,DH
  MOV DL,BL         ; DX holds the red hue of current color (word)
  SHR BX,8          ; BX holds the green hue of current color (word)
  SUB AX,[BP+12]
  TEST AX,08000h    ; Checks if blue difference is negative
  JZ BlueOk
  NEG AX            ; Makes it positive
BlueOk:
  SUB DX,[BP+16]
  TEST DX,08000h    ; Checks if red difference is negative
  JZ RedOk
  NEG DX            ; Makes it positive
RedOk:
  SUB BX,[BP+14]
  TEST BX,08000h    ; Checks if green difference is negative
  JZ GreenOk
  NEG BX            ; Makes it positive
GreenOk:
  CMP DX,[BP+4]
  JG TryAnother
  CMP BX,[BP+2]
  JG TryAnother
  CMP AX,[BP]
  JG TryAnother
  MOV [BP+4],DX
  MOV [BP+2],BX
  MOV [BP],AX
  MOV DI,CX
TryAnother:
  INC CX
  CMP CX,256
  JL FindColLoop
  MOV AX,DI
  POP BX
  POP BX
  POP BX
  POP BP
  RET 6
DQBfindCol ENDP

; ---------------------------------------------------------------------------
; DQBsetPal SUB
; purpose:
;   Sets the entire palette to the specified one, passed as a string*768,
;   containing in order the r,g and b data for each color. A hue is a byte and
;   ranges from 0 to 63
; declaration:
;   DECLARE SUB xDQBsetPal(BYVAL PalSeg,BYVAL PalOff)
;   DECLARE SUB DQBsetPal(Pal AS STRING)
; ---------------------------------------------------------------------------
PUBLIC xDQBsetPal
xDQBsetPal PROC
  ; Stack layout:
  ;
  ; 10  PalSeg
  ; 08  PalOff
  ; 06  Basic return segment
  ; 04  Basic return offset
  ; 02  DS
  ; 00  BP
  PUSH DS
  PUSH BP
  MOV BP,SP
  MOV AX,[BP+10]
  MOV DS,AX
  XOR AX,AX
  MOV DX,03C8h      ; Port 03C8h: write palette index selector
  OUT DX,AL         ; Starts with color 0
  MOV SI,[BP+8]     ; SI holds the address of "Pal"
  MOV CX,768        ; Repeats 768 times
  MOV DX,03C9h      ; Port 03C9h: palette entry
SetPalLoop:
  LODSB
  OUT DX,AL         ; Outs the value to the VGA port
  LOOP SetPalLoop
  POP BP
  POP DS
  RET 4
xDQBsetPal ENDP

; ---------------------------------------------------------------------------
; DQBgetPal SUB
; purpose:
;   Gets the entire palette and stores it into a given string of 768 chars,
;   ready to be used with DQBsetPal
; declaration:
;   DECLARE SUB xDQBgetPal(BYVAL PalSeg,BYVAL PalOff)
;   DECLARE SUB DQBgetPal(Pal AS STRING)
; ---------------------------------------------------------------------------
PUBLIC xDQBgetPal
xDQBgetPal PROC
  ; Stack layout:
  ;
  ; 08  PalSeg
  ; 06  PalOff
  ; 04  Basic return segment
  ; 02  Basic return offset
  ; 00  BP
  PUSH BP
  MOV BP,SP
  MOV AX,[BP+8]
  MOV ES,AX
  MOV DX,03C7h      ; Port 03C7h: read palette index selector
  XOR AX,AX
  OUT DX,AL         ; Starts with color 0
  MOV DI,[BP+6]     ; SI holds the address of "Pal"
  MOV CX,768        ; Repeats 768 times
  MOV DX,03C9h      ; Port 03C9h: palette entry
GetPalLoop:
  IN AL,DX          ; Gets a byte from the VGA port...
  STOSB             ; ...and writes it into the buffer
  LOOP GetPalLoop
  POP BP
  RET 4
xDQBgetPal ENDP

; ---------------------------------------------------------------------------
; DQBfadeIn SUB
; purpose:
;   Fades the current palette to a specified one
; declaration:
;   DECLARE SUB xDQBfadeIn(BYVAL PalSeg,BYVAL PalOff)
;   DECLARE SUB DQBfadeIn(Pal AS STRING)
; ---------------------------------------------------------------------------
PUBLIC xDQBfadeIn
xDQBfadeIn PROC
  ; Stack layout:
  ;
  ; 08  PalSeg
  ; 06  PalOff
  ; 04  Basic return segment
  ; 02  Basic return offset
  ; 00  BP
  PUSH BP
  MOV BP,SP
  MOV AX,[BP+8]
  MOV ES,AX
CheckAllCols:
  MOV DX,3DAh
WaitPal0:
  IN AL,DX
  AND AL,8
  JNZ WaitPal0
WaitPal:
  IN AL,DX
  AND AL,8
  JZ WaitPal
  XOR SI,SI         ; SI is 0 when all colors are ok
  XOR CX,CX         ; CX is the color counter
  MOV DI,[BP+6]
CheckCol:
  MOV DX,03C7h
  MOV AL,CL
  OUT DX,AL         ; Read data from color
  MOV DX,03C9h
  IN AL,DX
  MOV BL,AL         ; BL holds the red hue
  IN AL,DX
  MOV BH,AL         ; BH holds the green hue
  IN AL,DX
  MOV AH,AL         ; AH holds the blue hue
  MOV DX,03C8h
  MOV AL,CL
  OUT DX,AL         ; Write data to color
  MOV DX,03C9h
  MOV AL,ES:[DI]
  INC DI
  CMP BL,AL
  JG DecRed
  CMP BL,AL
  JL IncRed
  OUT DX,AL
  JMP CheckGreen
DecRed:
  INC SI
  MOV AL,BL
  DEC AL
  OUT DX,AL
  JMP CheckGreen
IncRed:
  INC SI
  MOV AL,BL
  INC AL
  OUT DX,AL
CheckGreen:
  MOV AL,ES:[DI]
  INC DI
  CMP BH,AL
  JG DecGreen
  CMP BH,AL
  JL IncGreen
  OUT DX,AL
  JMP CheckBlue
DecGreen:
  INC SI
  MOV AL,BH
  DEC AL
  OUT DX,AL
  JMP CheckBlue
IncGreen:
  INC SI
  MOV AL,BH
  INC AL
  OUT DX,AL
CheckBlue:
  MOV AL,ES:[DI]
  INC DI
  CMP AH,AL
  JG DecBlue
  CMP AH,AL
  JL IncBlue
  OUT DX,AL
  JMP NextCol
DecBlue:
  INC SI
  MOV AL,AH
  DEC AL
  OUT DX,AL
  JMP NextCol
IncBlue:
  INC SI
  MOV AL,AH
  INC AL
  OUT DX,AL
NextCol:
  INC CX
  CMP CX,256
  JL CheckCol
  CMP SI,0
  JNE CheckAllCols
  POP BP
  RET 4
xDQBfadeIn ENDP

; ---------------------------------------------------------------------------
; DQBfadeTo SUB
; purpose:
;   Fades all the colors of current palette to a specified color
; declaration:
;   DECLARE SUB DQBfadeIn(BYVAL Red,BYVAL Green,BYVAL Blue)
; ---------------------------------------------------------------------------
PUBLIC DQBfadeTo
DQBfadeTo PROC
  PUSH BP
  MOV BP,SP
CheckAllCols1:
  MOV DX,3DAh
WaitPal1:
  IN AL,DX
  AND AL,8
  JNZ WaitPal1
WaitPal2:
  IN AL,DX
  AND AL,8
  JZ WaitPal2
  XOR CX,CX
  XOR SI,SI
CheckCol1:
  MOV DX,03C7h
  MOV AL,CL
  OUT DX,AL
  MOV DX,03C9h
  IN AL,DX
  MOV BL,AL         ; BL holds the current red hue
  IN AL,DX
  MOV BH,AL         ; BH holds the current green hue
  IN AL,DX
  MOV AH,AL         ; AH holds the current blue hue
  MOV DX,03C8h
  MOV AL,CL
  OUT DX,AL
  MOV DX,03C9h
  MOV AL,[BP+10]
  CMP BL,AL
  JG DecRed1
  CMP BL,AL
  JL IncRed1
  OUT DX,AL
  JMP CheckGreen1
DecRed1:
  INC SI
  MOV AL,BL
  DEC AL
  OUT DX,AL
  JMP CheckGreen1
IncRed1:
  INC SI
  MOV AL,BL
  INC AL
  OUT DX,AL
CheckGreen1:
  MOV AL,[BP+8]
  CMP BH,AL
  JG DecGreen1
  CMP BH,AL
  JL IncGreen1
  OUT DX,AL
  JMP CheckBlue1
DecGreen1:
  INC SI
  MOV AL,BH
  DEC AL
  OUT DX,AL
  JMP CheckBlue1
IncGreen1:
  INC SI
  MOV AL,BH
  INC AL
  OUT DX,AL
CheckBlue1:
  MOV AL,[BP+6]
  CMP AH,AL
  JG DecBlue1
  CMP AH,AL
  JL IncBlue1
  OUT DX,AL
  JMP NextCol1
DecBlue1:
  INC SI
  MOV AL,AH
  DEC AL
  OUT DX,AL
  JMP NextCol1
IncBlue1:
  INC SI
  MOV AL,AH
  INC AL
  OUT DX,AL
NextCol1:
  INC CX
  CMP CX,256
  JL CheckCol1
  CMP SI,0
  JNE CheckAllCols1
  POP BP
  RET 6
DQBfadeTo ENDP

; ---------------------------------------------------------------------------
; DQBpalOff SUB
; purpose:
;   Turns all the colors in current palette to black.
; declaration:
;   DECLARE SUB DQBpalOff()
; ---------------------------------------------------------------------------
PUBLIC DQBpalOff
DQBpalOff PROC
  MOV DX,03C8h
  XOR AX,AX
  OUT DX,AL
  MOV DX,03C9h
  MOV CX,768
PalOff:
  OUT DX,AL
  LOOP PalOff
  RET
DQBpalOff ENDP

; ---------------------------------------------------------------------------
; DQBprint SUB
; purpose:
;   Prints a string onto specified layer, at the given coordinates and with
;   the given color and current font
; declaration:
;   DECLARE SUB xDQBprint(BYVAL Layer,BYVAL TextSeg,BYVAL TextOff,BYVAL x,
;                         BYVAL y,BYVAL Col)
;   DECLARE SUB DQBprint(Layer AS INTEGER,Text AS STRING,x AS INTEGER,y AS
;                         INTEGER,Col AS INTEGER)
; ---------------------------------------------------------------------------
PUBLIC xDQBprint
xDQBprint PROC
  ; Stack layout (after startup):
  ;
  ; 18  Layer
  ; 16  TextSeg
  ; 14  TextOff
  ; 12  x
  ; 10  y
  ; 08  Col
  ; 06  Basic return segment
  ; 04  Basic return offset
  ; 02  DS
  ; 00  BP
  PUSH DS
  PUSH BP
  MOV BP,SP
  MOV BX,[BP+18]
  CALL GetLayerSeg
  MOV ES,CX         ; ES is the layer segment
  MOV DX,[BP+12]    ; DX holds the x value
NextChar:
  MOV SI,DX
  SUB SI,[BP+12]
  SHR SI,3          ; Now SI contains the character number being processed
  ADD SI,[BP+14]
  MOV BX,[BP+16]
  MOV DS,BX
  MOV AL,[SI]       ; AL holds the current character ascii code
  CMP AL,0
  JE EndPrint       ; Character 0 ends the string
  MOV BX,@DATA
  MOV DS,BX
  MOV BX,FontSeg
  MOV SI,FontOff
  MOV DS,BX
  XOR BH,BH
  MOV BL,AL
  SHL BX,3          ; Let's find the offset of our character
  ADD SI,BX         ; DS:SI now points to the font data for our character
  MOV DI,[BP+10]
  MOV BX,DI
  SHL DI,8
  SHL BX,6
  ADD DI,BX         ; ES:DI points to character (x,y) on specified layer
  ADD DI,DX
  MOV BX,[BP+10]    ; BX holds the starting y value
  MOV CX,8          ; Repeats 8 times
LineLoop:
  PUSH DS
  MOV AX,@DATA
  MOV DS,AX
  CMP BX,ClipY1
  JL SkipLine       ; Out of upper layer border
  CMP BX,ClipY2
  JG SkipLine       ; Out of lower layer border
  POP DS
  MOV AH,[SI]       ; AH holds character line data
  MOV AL,[BP+8]     ; AL is now the color
  PUSH BX
  PUSH DS
  MOV BX,@DATA
  MOV DS,BX
  MOV BL,80h
  CALL DrawPixel
  SHR BL,1
  CALL DrawPixel
  SHR BL,1
  CALL DrawPixel
  SHR BL,1
  CALL DrawPixel
  SHR BL,1
  CALL DrawPixel
  SHR BL,1
  CALL DrawPixel
  SHR BL,1
  CALL DrawPixel
  SHR BL,1
  CALL DrawPixel
  POP DS
  POP BX
  JMP NextLine
SkipLine:
  POP DS
  ADD DX,8
  ADD DI,8
NextLine:
  INC SI
  SUB DX,8          ; Adjust current x position
  ADD DI,312        ; Place cursor at the left border of character
  INC BX            ; Next line
  LOOP LineLoop
  ADD DX,8          ; Next character
  JMP NextChar
EndPrint:
  POP BP
  POP DS
  RET 12
xDQBprint ENDP

; ---------------------------------------------------------------------------
; DQBsetBIOSfont SUB
; purpose:
;   Restores the old VGA BIOS font as the actual font
; declaration:
;   DECLARE SUB DQBsetBIOSfont()
; ---------------------------------------------------------------------------
PUBLIC DQBsetBIOSfont
DQBsetBIOSfont PROC
  MOV FontSeg,0FFA6h
  MOV FontOff,0Eh
  RET
DQBsetBIOSfont ENDP

; ---------------------------------------------------------------------------
; DQBsetFont SUB
; purpose:
;   Sets the actual font to a new one stored in memory
; declaration:
;   DECLARE SUB xDQBsetFont(BYVAL FontSeg,BYVAL FontOff)
;   DECLARE SUB DQBsetFont(Font AS STRING)
; ---------------------------------------------------------------------------
PUBLIC xDQBsetFont
xDQBsetFont PROC
  ; Stack layout:
  ;
  ; 10  FontSeg
  ; 08  FontOff
  ; 06  Basic return segment
  ; 04  Basic return offset
  ; 02  DS
  ; 00  BP
  PUSH DS
  PUSH BP
  MOV BP,SP
  MOV AX,@DATA
  MOV FontSeg,AX
  MOV ES,AX
  MOV DI,OFFSET FontBuffer
  MOV FontOff,DI
  MOV AX,[BP+10]
  MOV DS,AX
  MOV SI,[BP+8]
  MOV CX,512
  REP MOVSD
  POP BP
  POP DS
  RET 4
xDQBsetFont ENDP

; ---------------------------------------------------------------------------
; DQBsetTransText SUB
; purpose:
;   After calling this function, using DQBprint will draw transparent text
; declaration:
;   DECLARE SUB DQBsetTransText()
; ---------------------------------------------------------------------------
PUBLIC DQBsetTransText
DQBsetTransText PROC
  MOV TextMode,0
  RET
DQBsetTransText ENDP

; ---------------------------------------------------------------------------
; DQBsetSolidText SUB
; purpose:
;   After calling this function, using DQBprint will draw solid text
; declaration:
;   DECLARE SUB DQBsetSolidText()
; ---------------------------------------------------------------------------
PUBLIC DQBsetSolidText
DQBsetSolidText PROC
  MOV TextMode,1
  RET
DQBsetSolidText ENDP

; ---------------------------------------------------------------------------
; DQBsetBackTextCol SUB
; purpose:
;   Sets the color used to fill the background of text print in solid mode
; declaration:
;   DECLARE SUB DQBsetBackTextCol(BYVAL Col)
; ---------------------------------------------------------------------------
PUBLIC DQBsetTextBackCol
DQBsetTextBackCol PROC
  PUSH BP
  MOV BP,SP
  MOV AX,[BP+6]
  MOV TextBackCol,AL
  POP BP
  RET 2
DQBsetTextBackCol ENDP

; ---------------------------------------------------------------------------
; DQBloadLayer FUNCTION
; purpose:
;   Loads an image onto specified layer. It supports BSAVE, BMP and PCX image
;   formats, and it automatically detects which one is used by specified file.
;   When loading a BMP or a PCX, its palette is stored at specified address;
;   when loading BSAVEd images, the palette is ignored.
; declaration:
;   DECLARE FUNCTION xDQBloadLayer(BYVAL Layer,BYVAL FileSeg,BYVAL FileOff,
;                                 BYVAL PalSeg,BYVAL PalOff)
;   DECLARE FUNCTION DQBloadLayer(Layer AS INTEGER,FileName AS STRING,
;                                 Pal AS STRING)
; ---------------------------------------------------------------------------
PUBLIC xDQBloadLayer
xDQBloadLayer PROC
  ; Stack layout:
  ;
  ; 16  Layer
  ; 14  FileSeg
  ; 12  FileOff
  ; 10  PalSeg
  ; 08  PalOff
  ; 06  Basic return segment
  ; 04  Basic return offset
  ; 02  DS
  ; 00  BP
  PUSH DS
  PUSH BP
  MOV BP,SP
  MOV BX,[BP+16]
  CALL GetLayerSeg
  MOV ES,CX
  MOV AX,[BP+14]
  MOV DS,AX
  MOV DX,[BP+12]
  MOV AX,3D00h
  INT 21h
  JC Error_Open
  MOV BX,AX
  MOV AX,@DATA
  MOV DS,AX
  XOR AX,AX
  XOR DX,DX
  XOR SI,SI
  CLD
FindFormat:
  CALL ReadBuffer
  JC Error_Read
  MOV CL,Buffer
  CMP CL,BSVheader[SI]
  JNE NotBSV
  INC AH
NotBSV:
  CMP CL,BMPheader[SI]
  JNE NotBMP
  INC DL
NotBMP:
  CMP CL,PCXheader[SI]
  JNE NotPCX
  INC AL
NotPCX:
  INC SI
  CMP SI,7
  JL FindFormat
  CMP AH,7
  JE LoadBSV
  CMP DL,7
  JE LoadBMP
  CMP AL,7
  JE LoadPCX
  JMP Error_BadFile
LoadBMP:
  XOR SI,SI
  MOV CX,47         ; Skip remaining of BMP header
CheckBMPheader:
  CALL ReadBuffer
  JC Error_Read
  LOOP CheckBMPheader
  PUSH ES
  MOV AX,[BP+10]
  MOV ES,AX
  MOV DI,[BP+8]
  MOV CX,256
LoadPalette:
  CALL ReadBuffer
  JC Error_Read
  MOV AL,Buffer
  SHR AL,2
  MOV ES:[DI+2],AL
  CALL ReadBuffer
  JC Error_Read
  MOV AL,Buffer
  SHR AL,2
  MOV ES:[DI+1],AL
  CALL ReadBuffer
  JC Error_Read
  MOV AL,Buffer
  SHR AL,2
  MOV ES:[DI],AL
  CALL ReadBuffer
  JC Error_Read
  ADD DI,3
  LOOP LoadPalette
  POP ES
  MOV CX,200
LoadBMPloop:
  MOV DX,CX
  DEC DX
  MOV SI,DX
  SHL DX,8
  SHL SI,6
  ADD DX,SI
  PUSH CX
  MOV AX,ES
  MOV DS,AX
  MOV CX,320
  MOV AH,3Fh
  INT 21h
  JC Error_Read
  POP CX
  LOOP LoadBMPloop
  JMP LoadCloseFile
LoadPCX:
  XOR SI,SI
  MOV CX,121        ; Skip remaining of PCX header
CheckPCXheader:
  CALL ReadBuffer
  JC Error_Read
  LOOP CheckPCXheader
  XOR DI,DI
  XOR AX,AX
LoadPCXloop:
  CALL ReadBuffer
  JC Error_Read
  MOV AL,Buffer
  PUSH AX
  AND AL,0C0h
  CMP AL,0C0h
  POP AX
  JE DecodeChunk
  STOSB
  JMP NextChunk
DecodeChunk:
  SUB AL,0C0h
  PUSH AX
  CALL ReadBuffer
  JC Error_Read
  POP CX
  MOV AL,Buffer
  REP STOSB
NextChunk:
  CMP DI,0FA00h
  JB LoadPCXloop
  CALL ReadBuffer
  JC Error_Read
  MOV AX,[BP+10]
  MOV ES,AX
  MOV DI,[BP+8]
  MOV CX,768
LoadPalette1:
  CALL ReadBuffer
  JC Error_Read
  MOV AL,Buffer
  SHR AL,2
  STOSB
  LOOP LoadPalette1
  JMP LoadCloseFile
LoadBSV:
  MOV AX,ES
  MOV DS,AX
  MOV CX,64000
  XOR DX,DX
  MOV AH,3Fh
  INT 21h
  JC Error_Read
  JMP LoadCloseFile
Error_Open:
  MOV AX,1          ; Error 1: Cannot open file (does it exist?)
  JMP EndLoadLayer
Error_Read:
  MOV AH,3Eh
  INT 21h
  MOV AX,2          ; Error 2: Unable to read from file
  JMP EndLoadLayer
Error_BadFile:
  MOV AH,3Eh
  INT 21h
  MOV AX,3          ; Error 3: Bad file format
  JMP EndLoadLayer
LoadCloseFile:
  MOV AH,3Eh
  INT 21h
  XOR AX,AX
EndLoadLayer:
  POP BP
  POP DS
  RET 10
xDQBloadLayer ENDP

; ---------------------------------------------------------------------------
; DQBsaveLayer FUNCTION
; purpose:
;   Saves specified layer to a file. Two saving methods available: the first
;   saves only the layer into a BSAVE compatible binary file, the second
;   stores the layer as a PCX file, including specified palette. The function
;   returns 0 on success, otherwise an error code.
; declaration:
;   DECLARE FUNCTION xDQBsaveLayer(BYVAL Layer,BYVAL FileSeg,BYVAL FileOff,
;                                 BYVAL PalSeg,BYVAL PalOff,BYVAL Format)
;   DECLARE FUNCTION DQBsaveLayer(Layer AS INTEGER,FileName AS STRING,
;                                 Pal AS STRING,Format AS INTEGER)
; ---------------------------------------------------------------------------
PUBLIC xDQBsaveLayer
xDQBsaveLayer PROC
  ; Stack layout:
  ;
  ; 18  Layer
  ; 16  FileSeg
  ; 14  FileOff
  ; 12  PalSeg
  ; 10  PalOff
  ; 08  Format
  ; 06  Basic return segment
  ; 04  Basic return offset
  ; 02  DS
  ; 00  BP
  PUSH DS
  PUSH BP
  MOV BP,SP
  MOV BX,[BP+18]
  CALL GetLayerSeg
  MOV ES,CX
  MOV DX,[BP+14]    ; DS:DX points to the asciiz filename
  MOV AX,[BP+16]
  MOV DS,AX
  MOV AH,3Ch        ; Function 3Ch: create file
  MOV CX,20h        ; archive attribute
  INT 21h
  JC Error_Create
  MOV BX,AX
  MOV AX,[BP+8]
  CMP AX,0
  JE SaveBSV
  CMP AX,1
  JE SaveBMP
  MOV DX,OFFSET PCXheader
  MOV AX,@DATA
  MOV DS,AX
  MOV CX,128        ; Let's write the PCX header at the beginning of the file
  MOV AH,40h        ; Function 40h: write to file
  INT 21h
  JC Error_Write
  XOR DI,DI         ; Let's begin with the first layer pixel
  MOV CX,200        ; Scans 200 lines
SavePCXloop:
  XOR DX,DX         ; DX holds the pixel being processed
EncodeLine:
  XOR SI,SI         ; SI holds the number of consecutive equal pixels
  PUSH BX
FindEquals:
  MOV BX,DX
  ADD BX,SI
  CMP BX,320        ; Have we reached the end of line?
  JGE WriteData      ; Yes: write data
  CMP SI,63         ; A maximum of 63 consecutive pixels is allowed
  JGE WriteData
  MOV BX,DI
  ADD BX,SI
  MOV AL,ES:[BX]
  CMP AL,ES:[BX+1]  ; Let's check if we have two equal consecutive pixels
  JNE WriteData     ; No: write a single pixel
  INC SI            ; Yes: increase the consecutive pixels counter (SI)
  JMP FindEquals
WriteData:
  CMP SI,0          ; Are there any consecutive equal pixels?
  JE SinglePixel    ; No: write a single pixel
  ADD DX,SI
  ADD DI,SI
  MOV BX,SI
  OR BL,192
  MOV AH,BL
  POP BX
  MOV Buffer,AH
  CALL WriteBuffer
  JC Error_Write
  MOV Buffer,AL
  CALL WriteBuffer
  JC Error_Write
  JMP EncodeNext
SinglePixel:
  POP BX
  INC DX
  INC DI
  TEST AL,192
  JZ SinglePixel1
  MOV Buffer,193
  CALL WriteBuffer
  JC Error_Write
  JMP SinglePixel1
EncodeNext:
  CMP DX,320
  JL EncodeLine
  LOOP SavePCXloop
  JMP SavePalette
SinglePixel1:
  MOV Buffer,AL
  CALL WriteBuffer
  JC Error_Write
  JMP EncodeNext
SavePalette:
  MOV Buffer,12
  CALL WriteBuffer
  JC Error_Write
  MOV DX,[BP+12]
  MOV DS,DX
  MOV DX,@DATA
  MOV SI,[BP+10]
  MOV CX,768
WritePalette:
  LODSB
  SHL AL,2
  PUSH DS
  MOV DS,DX
  MOV Buffer,AL
  CALL WriteBuffer
  JC Error_Write
  POP DS
  DEC CX
  JNZ WritePalette
  JMP SaveCloseFile
SaveBMP:
  MOV CX,54
  MOV DX,OFFSET BMPheader
  MOV AX,@DATA
  MOV DS,AX
  MOV AH,40h        ; Saves the BMP header
  INT 21h
  JC Error_Write
  MOV SI,[BP+10]    ; DS:SI now points to the palette string
  MOV CX,256
WritePalette1:
  MOV AX,[BP+12]
  MOV DS,AX
  LODSB
  PUSH AX
  LODSB
  PUSH AX
  LODSB
  MOV DX,@DATA
  MOV DS,DX
  SHL AL,2
  MOV Buffer,AL
  CALL WriteBuffer
  JC Error_Write
  POP AX
  SHL AL,2
  MOV Buffer,AL
  CALL WriteBuffer
  JC Error_Write
  POP AX
  SHL AL,2
  MOV Buffer,AL
  CALL WriteBuffer
  JC Error_Write
  MOV Buffer,0
  CALL WriteBuffer
  JC Error_Write
  LOOP WritePalette1
  MOV CX,200
SaveBMPloop:
  MOV DX,CX
  DEC DX
  MOV SI,DX
  SHL DX,8
  SHL SI,6
  ADD DX,SI
  PUSH CX
  MOV AX,ES
  MOV DS,AX
  MOV AH,40h
  MOV CX,320
  INT 21h
  JC Error_Write
  POP CX
  LOOP SaveBMPloop
  JMP SaveCloseFile
SaveBSV:
  MOV CX,7
  MOV DX,OFFSET BSVheader
  MOV AX,@DATA
  MOV DS,AX
  MOV AH,40h        ; Saves the BSAVE header
  INT 21h
  JC Error_Write
  MOV CX,64000
  XOR DX,DX
  MOV AX,ES
  MOV DS,AX
  MOV AH,40h        ; Saves the entire layer
  INT 21h
  JC Error_Write
  JMP SaveCloseFile
Error_Create:
  MOV AX,1          ; Error 1: cannot create file
  JMP EndSaveLayer
Error_Write:
  MOV AH,3Eh
  INT 21h
  MOV AX,2          ; Error 2: cannot write to disk (disk full?)
  JMP EndSaveLayer
SaveCloseFile:
  MOV AH,3Eh        ; Function 3Eh: close file
  INT 21h
  XOR AX,AX
EndSaveLayer:
  POP BP
  POP DS
  RET 12
xDQBsaveLayer ENDP


; ***************************************************************************
; INPUT ROUTINES
; ***************************************************************************


; ---------------------------------------------------------------------------
; DQBinstallKeyboard SUB
; purpose:
;   Installs a keyboard interrupt handler that updates the status of each key
; declaration:
;   DECLARE SUB DQBinstallKeyboard()
; ---------------------------------------------------------------------------
PUBLIC DQBinstallKeyboard
DQBinstallKeyboard PROC
  CMP KeyActive,1
  JE EndKeyInstall
  PUSH DS
  MOV AX,3509h
  INT 21h
  POP DS
  MOV OldInt9seg,ES
  MOV OldInt9off,BX
  MOV AX,SEG KeyboardISR
  MOV DX,OFFSET KeyboardISR
  PUSH DS
  MOV DS,AX
  MOV AX,2509h
  INT 21h
  POP DS
  MOV KeyActive,1
  RET
KeyboardISR:
  PUSHF
  PUSHA
  PUSH DS
  MOV BL,1
  IN AL,60h
  TEST AL,80h
  JZ StoreStatus
  AND AL,7Fh
  MOV BL,0
StoreStatus:
  XOR AH,AH
  MOV SI,AX
  MOV CX,@DATA
  MOV DS,CX
  MOV KeyStatus[SI],BL
  IN AL,61h
  OR AL,82h
  OUT 61h,AL
  AND AL,7Fh
  OUT 61h,AL
  MOV AL,20h
  OUT 20h,AL
  POP DS
  POPA
  POPF
  IRET
EndKeyInstall:
  RET
DQBinstallKeyboard ENDP

; ---------------------------------------------------------------------------
; DQBremoveKeyboard SUB
; purpose:
;   Removes previously installed keyboard interrupt handler and restores
;   the old one
; declaration:
;   DECLARE SUB DQBremoveKeyboard()
; ---------------------------------------------------------------------------
PUBLIC DQBremoveKeyboard
DQBremoveKeyboard PROC
  CMP KeyActive,0
  JE EndKeyRemove
  PUSH DS
  MOV DX,OldInt9off
  MOV AX,OldInt9seg
  MOV DS,AX
  MOV AX,2509h
  INT 21h
  POP DS
  XOR CX,CX
EmptyKeyStatus:
  MOV SI,CX
  MOV KeyStatus[SI],0
  INC CX
  CMP CX,128
  JL EmptyKeyStatus
  MOV KeyActive,0
EndKeyRemove:
  RET
DQBremoveKeyboard ENDP

; ---------------------------------------------------------------------------
; DQBkey FUNCTION
; purpose:
;   Returns the current status of a given key
; declaration:
;   DECLARE FUNCTION DQBkey(BYVAL ScanCode)
; ---------------------------------------------------------------------------
PUBLIC DQBkey
DQBkey PROC
  ; Stack layout:
  ;
  ; 06  ScanCode
  ; 04  Basic return segment
  ; 02  Basic return offset
  ; 00 BP
  PUSH BP
  MOV BP,SP
  XOR AH,AH
  MOV SI,[BP+6]
  CMP KeyStatus[SI],1
  SETE AL
  NEG AX
  POP BP
  RET 2
DQBkey ENDP

; ---------------------------------------------------------------------------
; DQBreadKey FUNCTION
; purpose:
;   Wait for the user to press a key and returns its scancode. If the keyboard
;   interrupt handler has not been installed by calling DQBinstallKeyboard,
;   returns -1.
; declaration:
;   DECLARE FUNCTION DQBreadKey()
; ---------------------------------------------------------------------------
PUBLIC DQBreadKey
DQBreadKey PROC
  MOV AX,0FFFFh
  CMP KeyActive,0
  JE EndReadKey
  XOR AX,AX
  XOR SI,SI
ReadKeyLoop:
  CMP KeyStatus[SI],1
  JE ReadFound
  INC SI
  CMP SI,128
  JL ReadKeyLoop
  XOR SI,SI
  JMP ReadKeyLoop
ReadFound:
  CMP KeyStatus[SI],1
  JE ReadFound
  MOV AX,SI
EndReadKey:
  RET
DQBreadKey ENDP

; ---------------------------------------------------------------------------
; DQBwaitKey SUB
; purpose:
;   Wait for the user to press a specified key. If the keyboard interrupt
;   handler has not been installed by calling DQBinstallKeyboard, returns
;   without waiting.
; declaration:
;   DECLARE SUB DQBwaitKey(BYVAL ScanCode)
; ---------------------------------------------------------------------------
PUBLIC DQBwaitKey
DQBwaitKey PROC
  PUSH BP
  MOV BP,SP
  CMP KeyActive,0
  JE EndWaitKey
  MOV BX,[BP+6]
WaitKeyLoop:
  CMP KeyStatus[BX],1
  JE WaitFound
  JMP WaitKeyLoop
WaitFound:
  CMP KeyStatus[BX],1
  JE WaitFound
EndWaitKey:
  POP BP
  RET 2
DQBwaitKey ENDP

; ---------------------------------------------------------------------------
; DQBjoyDetected FUNCTION
; purpose:
;   Returns 1 if specified joystick has been detected, otherwise 0
; declaration:
;   DECLARE FUNCTION DQBjoyDetected(BYVAL JoyNum)
; ---------------------------------------------------------------------------
PUBLIC DQBjoyDetected
DQBjoyDetected PROC
  PUSH BP
  MOV BP,SP
  XOR AH,AH
  MOV SI,[BP+6]
  MOV AL,JoyDetected[SI]
  NEG AX
  POP BP
  RET 2
DQBjoyDetected ENDP

; ---------------------------------------------------------------------------
; DQBpollJoy SUB
; purpose:
;   Updates the internal joystick variables for later use
; declaration:
;   DECLARE SUB DQBpollJoy(BYVAL JoyNum)
; ---------------------------------------------------------------------------
PUBLIC DQBpollJoy
DQBpollJoy PROC
  PUSH BP
  MOV BP,SP
  CLI
  MOV DX,201h
  OUT DX,AL
  IN AL,DX
  MOV DI,[BP+6]
  MOV SI,DI
  SHL DI,2
  TEST AL,JoyMask[DI+2]
  SETZ JoyButA[SI]
  TEST AL,JoyMask[DI+3]
  SETZ JoyButB[SI]
  MOV BL,JoyMask[DI]
  ADD BL,JoyMask[DI+1]
  XOR CX,CX
  OUT DX,AL
JoyLoop:
  IN AL,DX
  TEST AL,JoyMask[DI]
  JZ SkipXsave
  MOV JoyX[SI],CX
SkipXsave:
  TEST AL,JoyMask[DI+1]
  JZ SkipYsave
  MOV JoyY[SI],CX
SkipYsave:
  INC CX
  CMP CX,65535
  JE EndPollJoy
  TEST AL,BL
  JNZ JoyLoop
EndPollJoy:
  STI
  POP BP
  RET 2
DQBpollJoy ENDP

; ---------------------------------------------------------------------------
; DQBjoyMove FUNCTION
; purpose:
;   Returns true if specified joystick points to the given direction
; declaration:
;   DECLARE FUNCTION DQBjoy(BYVAL JoyNum,BYVAL Direction)
; ---------------------------------------------------------------------------
PUBLIC DQBjoyMove
DQBjoyMove PROC
  PUSH BP
  MOV BP,SP
  MOV DI,[BP+8]
  MOV BX,[BP+6]
  XOR AH,AH
  CMP BX,0
  JE JoyUp
  CMP BX,1
  JE JoyDown
  CMP BX,2
  JE JoyLeft
  MOV DX,JoyX[DI]
  MOV CX,JoyCX[DI]
  ADD CX,JoySens
  CMP DX,CX
  SETGE AL
  NEG AX
  POP BP
  RET 4
JoyUp:
  MOV DX,JoyY[DI]
  MOV CX,JoyCY[DI]
  SUB CX,JoySens
  CMP DX,CX
  SETLE AL
  NEG AX
  POP BP
  RET 4
JoyDown:
  MOV DX,JoyY[DI]
  MOV CX,JoyCY[DI]
  ADD CX,JoySens
  CMP DX,CX
  SETGE AL
  NEG AX
  POP BP
  RET 4
JoyLeft:
  MOV DX,JoyX[DI]
  MOV CX,JoyCX[DI]
  SUB CX,JoySens
  CMP DX,CX
  SETLE AL
  NEG AX
  POP BP
  RET 4
DQBjoyMove ENDP

; ---------------------------------------------------------------------------
; DQBjoyFire FUNCTION
; purpose:
;   Returns true if specified fire button on specified joystick is pressed
; declaration:
;   DECLARE FUNCTION DQBjoyFire(BYVAL JoyNum,BYVAL Button)
; ---------------------------------------------------------------------------
PUBLIC DQBjoyFire
DQBjoyFire PROC
  PUSH BP
  MOV BP,SP
  MOV DI,[BP+8]
  MOV BX,[BP+6]
  XOR AH,AH
  CMP BX,0
  JNE ButtonB
  CMP JoyButA[DI],0
  SETNE AL
  NEG AX
  POP BP
  RET 4
ButtonB:
  CMP JoyButB[DI],0
  SETNE AL
  NEG AX
  POP BP
  RET 4
DQBjoyFire ENDP

; ---------------------------------------------------------------------------
; DQBresetJoy SUB
; purpose:
;   Calibrates both joysticks
; declaration:
;   DECLARE SUB DQBresetJoy()
; ---------------------------------------------------------------------------
PUBLIC DQBresetJoy
DQBresetJoy PROC
  CALL ResetJoy
  RET
DQBresetJoy ENDP

; ---------------------------------------------------------------------------
; DQBsetJoySens SUB
; purpose:
;   Allows to change the joysticks sensibility
; declaration:
;   DECLARE SUB DQBsetJoySens(BYVAL NewSens)
; ---------------------------------------------------------------------------
PUBLIC DQBsetJoySens
DQBsetJoySens PROC
  PUSH BP
  MOV BP,SP
  MOV BX,[BP+6]
  MOV JoySens,BX
  POP BP
  RET 2
DQBsetJoySens ENDP

; ---------------------------------------------------------------------------
; DQBmouseDetected FUNCTION
; purpose:
;   Returns true if mouse has been successfully initialized, otherwise 0
; declaration:
;   DECLARE FUNCTION DQBmouseDetected()
; ---------------------------------------------------------------------------
PUBLIC DQBmouseDetected
DQBmouseDetected PROC
  XOR AH,AH
  CMP MouseDetected,1
  SETE AL
  NEG AX
  RET
DQBmouseDetected ENDP

; ---------------------------------------------------------------------------
; DQBmouseX FUNCTION
; purpose:
;   Returns the current mouse x coordinate
; declaration:
;   DECLARE FUNCTION DQBmouseX()
; ---------------------------------------------------------------------------
PUBLIC DQBmouseX
DQBmouseX PROC
  MOV AX,MouseX
  SHR AX,1
  RET
DQBmouseX ENDP

; ---------------------------------------------------------------------------
; DQBmouseY FUNCTION
; purpose:
;   Returns the current mouse y coordinate
; declaration:
;   DECLARE FUNCTION DQBmouseY()
; ---------------------------------------------------------------------------
PUBLIC DQBmouseY
DQBmouseY PROC
  MOV AX,MouseY
  RET
DQBmouseY ENDP

; ---------------------------------------------------------------------------
; DQBmouseLB FUNCTION
; purpose:
;   Returns true if left mouse button is currently pressed
; declaration:
;   DECLARE FUNCTION DQBmouseLB()
; ---------------------------------------------------------------------------
PUBLIC DQBmouseLB
DQBmouseLB PROC
  XOR AH,AH
  TEST MouseBut,1
  SETNZ AL
  NEG AX
  RET
DQBmouseLB ENDP

; ---------------------------------------------------------------------------
; DQBmouseRB FUNCTION
; purpose:
;   Returns true if right mouse button is currently pressed
; declaration:
;   DECLARE FUNCTION DQBmouseRB()
; ---------------------------------------------------------------------------
PUBLIC DQBmouseRB
DQBmouseRB PROC
  XOR AH,AH
  TEST MouseBut,2
  SETNZ AL
  NEG AX
  RET
DQBmouseRB ENDP

; ---------------------------------------------------------------------------
; DQBmouseShow SUB
; purpose:
;   Shows the current mouse cursor on the screen
; declaration:
;   DECLARE SUB DQBmouseShow()
; ---------------------------------------------------------------------------
PUBLIC DQBmouseShow
DQBmouseShow PROC
  CMP MouseOn,1
  JE EndMouseShow
  MOV AX,1
  INT 33h
  MOV MouseOn,1
EndMouseShow:
  RET
DQBmouseShow ENDP

; ---------------------------------------------------------------------------
; DQBmouseHide SUB
; purpose:
;   Hides the mouse cursor
; declaration:
;   DECLARE SUB DQBmouseHide()
; ---------------------------------------------------------------------------
PUBLIC DQBmouseHide
DQBmouseHide PROC
  CMP MouseOn,0
  JE EndMouseHide
  MOV AX,2
  INT 33h
  MOV MouseOn,0
EndMouseHide:
  RET
DQBmouseHide ENDP

; ---------------------------------------------------------------------------
; DQBsetMouseRange SUB
; purpose:
;   Sets the mouse range box from x1,y1 to x2,y2
; declaration:
;   DECLARE SUB DQBsetMouseRange(BYVAL x1,BYVAL y1,BYVAL x2,BYVAL y2)
; ---------------------------------------------------------------------------
PUBLIC DQBsetMouseRange
DQBsetMouseRange PROC
  ; Stack layout:
  ;
  ; 12  x1
  ; 10  y1
  ; 08  x2
  ; 06  y2
  ; 04  Basic return segment
  ; 02  Basic return offset
  ; 00  BP
  PUSH BP
  MOV BP,SP
  CMP MouseOn,0
  JE SkipHide
  MOV AX,2
  INT 33h
SkipHide:
  MOV AX,7
  MOV CX,[BP+12]
  SHL CX,1
  MOV DX,[BP+8]
  SHL DX,1
  INT 33h
  MOV AX,8
  MOV CX,[BP+10]
  MOV DX,[BP+6]
  INT 33h
  CMP MouseOn,0
  JE SkipShow
  MOV AX,1
  INT 33h
SkipShow:
  POP BP
  RET 8
DQBsetMouseRange ENDP

; ---------------------------------------------------------------------------
; DQBsetMouseShape SUB
; purpose:
;   Sets a new mouse cursor shape
; declaration:
;   DECLARE SUB xDQBsetMouseShape(BYVAL hotX,BYVAL hotY,BYVAL ShapeSeg,
;                                 BYVAL ShapeOff)
;   DECLARE SUB DQBsetMouseShape(hotX AS INTEGER,hotY AS INTEGER,Shape AS
;                                 STRING)
; ---------------------------------------------------------------------------
PUBLIC xDQBsetMouseShape
xDQBsetMouseShape PROC
  ; Stack layout:
  ;
  ; 12  hotX
  ; 10  hotY
  ; 08  ShapeSeg
  ; 06  ShapeOff
  ; 04  Basic return segment
  ; 02  Basic return offset
  ; 00  BP
  PUSH BP
  MOV BP,SP
  CMP MouseOn,0
  JE SkipHide1
  MOV AX,2
  INT 33h
SkipHide1:
  MOV BX,[BP+12]
  SHL BX,1
  MOV CX,[BP+10]
  MOV AX,[BP+8]
  MOV ES,AX
  MOV DX,[BP+6]
  MOV AX,9
  INT 33h
  CMP MouseOn,0
  JE SkipShow1
  MOV AX,1
  INT 33h
SkipShow1:
  POP BP
  RET 8
xDQBsetMouseShape ENDP

; ---------------------------------------------------------------------------
; DQBresetMouse SUB
; purpose:
;   Sets the mouse range to (0,0)-(319,199) and sets the default cursor shape
; declaration:
;   DECLARE SUB DQBresetMouse()
; ---------------------------------------------------------------------------
PUBLIC DQBresetMouse
DQBresetMouse PROC
  CMP MouseOn,0
  JE SkipHide2
  MOV AX,2
  INT 33h
SkipHide2:
  MOV AX,7
  MOV CX,0
  MOV DX,639
  INT 33h
  MOV AX,8
  MOV CX,0
  MOV DX,199
  INT 33h
  MOV BX,0
  MOV CX,0
  MOV AX,@DATA
  MOV ES,AX
  MOV DX,OFFSET MouseShape
  MOV AX,9
  INT 33h
  CMP MouseOn,0
  JE SkipShow2
  MOV AX,1
  INT 33h
SkipShow2:
  RET
DQBresetMouse ENDP


; ***************************************************************************
; SOUND ROUTINES
; ***************************************************************************


; ---------------------------------------------------------------------------
; DQBinstallSB FUNCTION
; purpose:
;   Initialized the SB and starts the realtime mixing. Returns 0 on successful
;   otherwise an error code.
; declaration:
;   DECLARE FUNCTION DQBinstallSB(BYVAL Channels,BYVAL BaseAddr,BYVAL IRQ,
;                                 BYVAL DMA)
; ---------------------------------------------------------------------------
PUBLIC xDQBinstallSB
xDQBinstallSB PROC
  ; Stack layout:
  ;
  ; 12  Channels
  ; 10  BaseAddr
  ; 08  IRQ
  ; 06  DMA
  ; 04  Basic return segment
  ; 02  Basic return offset
  ; 00  BP
  PUSH BP
  MOV BP,SP
  CMP EMSallocated,0
  JG CheckSounds
  MOV AX,1          ; Error 1: No sounds were initialized
  POP BP
  RET 8
CheckSounds:
  CMP NumSounds,0
  JG InstallSB
  MOV AX,1          ; Error 1 again
  POP BP
  RET 8
InstallSB:
  CMP SoundActive,0
  JE NotInstalled
  XOR AX,AX
  POP BP
  RET 8
NotInstalled:
  MOV Playing,0
  MOV AX,[BP+10]
  MOV BaseAddr,AX
  MOV CX,[BP+8]
  MOV IRQ,CL
  MOV BL,1
  SHL BL,CL
  MOV PicORmask,BL
  XOR BL,0FFh
  MOV PicANDmask,BL
  MOV AX,[BP+6]
  MOV DMA,AL
  CMP AL,0
  JNE TestDMA1
  MOV DMA_Page,087h
  MOV DMA_Base,0
  MOV DMA_Count,1
  JMP DMAdone
TestDMA1:
  CMP AL,1
  JNE TestDMA2
  MOV DMA_Page,083h
  MOV DMA_Base,2
  MOV DMA_Count,3
  JMP DMAdone
TestDMA2:
  CMP AL,2
  JNE TestDMA3
  MOV DMA_Page,081h
  MOV DMA_Base,4
  MOV DMA_Count,5
  JMP DMAdone
TestDMA3:
  CMP AL,3
  JNE DMAerror
  MOV DMA_Page,082h
  MOV DMA_Base,6
  MOV DMA_Count,7
  JMP DMAdone
DMAerror:
  MOV AX,4          ; Error 4: only DMA channels 0-3 are supported
  POP BP
  RET 8
DMAdone:
  MOV DX,BaseAddr
  ADD DX,06h
  MOV AL,1
  OUT DX,AL
  IN AL,DX
  IN AL,DX
  IN AL,DX
  IN AL,DX
  XOR AL,AL
  OUT DX,AL
  ADD DX,08h
  WAITREADDSP
  SUB DX,4
  IN AL,DX
  CMP AL,0AAh       ; Has the DSP reset successfully?
  JE resetOk
  MOV AX,2          ; Error 2: DSP failed to reset
  POP BP
  RET 8
resetOk:            ; All ok: installs our new IRQ
  MOV DX,[BP+12]
  MOV Channels,DX
  MOV DX,BaseAddr
  ADD DX,0Ch
  WAITWRITEDSP
  MOV AL,0E1h       ; Get DSP version
  OUT DX,AL
  ADD DX,2
  WAITREADDSP
  SUB DX,4
  IN AL,DX
  MOV BH,AL
  ADD DX,4
  WAITREADDSP
  SUB DX,4
  IN AL,DX
  MOV BL,AL
  CMP BX,0200h
  JGE DSPok
  MOV AX,3          ; Error 3: Old soundblaster versions not supported
  POP BP
  RET 8
DSPok:
  ADD DX,2
  WAITWRITEDSP
  MOV AL,0D1h
  OUT DX,AL         ; Turns speakers on
  XOR AX,AX
  MOV ES,AX
  MOV AL,IRQ
  ADD AL,8
  MOV SI,AX
  SHL SI,2
  IN AL,021h
  AND AL,PicANDmask
  OUT 021h,AL       ; Disable IRQ
  MOV AX,ES:[SI]
  MOV OldIRQoff,AX
  MOV AX,OFFSET SBmix
  CLI
  MOV ES:[SI],AX
  MOV AX,ES:[SI+2]
  MOV OldIRQseg,AX
  MOV AX,SEG SBmix
  MOV ES:[SI+2],AX  ; We've changed the IRQ vector
  STI
  IN AL,021h
  AND AL,PicANDmask
  OUT 021h,AL       ; Enable IRQ
  SETUPDMA          ; Starts DMA transfers
  MOV SoundActive,1
  XOR AX,AX
  POP BP
  RET 8
SBmix:
  PUSHF
  PUSHA
  PUSH DS
  PUSH ES
  MOV AX,@DATA
  MOV DS,AX
  MOV ES,AX
  MOV DX,BaseAddr
  ADD DX,0Eh        ; Acknowledge SB
  IN AL,DX
  MOV SI,OFFSET SoundBuffer2      ; Double buffering to avoid "clicks"
  MOV DI,OFFSET SoundBuffer
  MOV CX,64
  REP MOVSD
  MOV AL,20h        ; Acknowledge interrupt
  OUT 20h,AL
  MOV AH,47h
  MOV DX,EMShdl
  INT 67h
  MIXVOICES         ; Mix into the second buffer while the first is playing
  MOV AH,48h
  MOV DX,EMShdl
  INT 67h
  POP ES
  POP DS
  POPA
  POPF
  IRET
xDQBinstallSB ENDP

; ---------------------------------------------------------------------------
; DQBsetSamplingRate SUB
; purpose:
;   Sets the sound sampling rate
; declaration:
;   DECLARE SUB xDQBsetSamplingRate(BYVAL Rate)
;   DECLARE SUB DQBsetSamplingRate(Rate)
; ---------------------------------------------------------------------------
PUBLIC xDQBsetSamplingRate
xDQBsetSamplingRate PROC
  PUSH BP
  MOV BP,SP
  MOV DX,BaseAddr
  ADD DX,0Ch
  WAITWRITEDSP
  MOV AL,0D0h
  OUT DX,AL
  WAITWRITEDSP
  MOV AL,040h
  OUT DX,AL
  WAITWRITEDSP
  MOV AX,[BP+6]
  OUT DX,AL
  MOV SamplingRate,AL
  WAITWRITEDSP
  MOV AL,0D4h
  OUT DX,AL
  POP BP
  RET 2
xDQBsetSamplingRate ENDP

; ---------------------------------------------------------------------------
; DQBloadSound FUNCTION
; purpose:
;   Loads a sound sample into a specified sound slot. Sounds must be 8 bit
;   mono and their sampling rate must not be greater than 22000 Hz.
; declaration:
;   DECLARE FUNCTION xDQBloadSound(BYVAL Slot,BYVAL FileSeg,BYVAL FileOff)
;   DECLARE FUNCTION DQBloadSound(Slot AS INTEGER,FileName AS STRING)
; ---------------------------------------------------------------------------
PUBLIC xDQBloadSound
xDQBloadSound PROC
  ; Stack layout
  ;
  ; 12  Slot
  ; 10  FileSeg
  ; 08  FileOff
  ; 06  Basic return segment
  ; 04  Basic return offset
  ; 02  DS
  ; 00  BP
  PUSH DS
  PUSH BP
  MOV BP,SP
  MOV AX,[BP+12]
  XOR BH,BH
  MOV BL,NumSounds
  CMP AX,BX
  JG Error_BadSlot
  MOV AX,[BP+10]
  MOV DS,AX
  MOV DX,[BP+8]
  MOV AX,03D00h
  INT 21h
  JC Error_OpenSound
  MOV BX,AX
  MOV AX,@DATA
  MOV DS,AX
  CALL ReadBuffer
  JC Error_ReadSound
  CMP Buffer,'R'
  JNE Error_BadSound
  CALL ReadBuffer
  JC Error_ReadSound
  CMP Buffer,'I'
  JNE Error_BadSound
  CALL ReadBuffer
  JC Error_ReadSound
  CMP Buffer,'F'
  JNE Error_BadSound
  CALL ReadBuffer
  JC Error_ReadSound
  CMP Buffer,'F'
  JNE Error_BadSound
  MOV CX,4
Skiphd1:
  CALL ReadBuffer
  JC Error_ReadSound
  LOOP Skiphd1
  CALL ReadBuffer
  JC Error_ReadSound
  CMP Buffer,'W'
  JNE Error_BadSound
  CALL ReadBuffer
  JC Error_ReadSound
  CMP Buffer,'A'
  JNE Error_BadSound
  CALL ReadBuffer
  JC Error_ReadSound
  CMP Buffer,'V'
  JNE Error_BadSound
  CALL ReadBuffer
  JC Error_ReadSound
  CMP Buffer,'E'
  JNE Error_BadSound
  MOV CX,10
Skiphd2:
  CALL ReadBuffer
  JC Error_ReadSound
  LOOP Skiphd2
  CALL ReadBuffer
  JC Error_ReadSound
  CMP Buffer,1
  JNE Error_NotSupported
  MOV CX,17
Skiphd3:
  CALL ReadBuffer
  JC Error_ReadSound
  LOOP Skiphd3
  CALL ReadBuffer
  JC Error_ReadSound
  MOV AL,Buffer
  CALL ReadBuffer
  JC Error_ReadSound
  MOV AH,Buffer
  CALL ReadBuffer
  JC Error_ReadSound
  MOV DL,Buffer
  CALL ReadBuffer
  JC Error_ReadSound
  MOV DH,Buffer
  CMP DX,0
  JNE Error_SoundTooBig
  MOV SI,[BP+12]
  DEC SI
  SHL SI,1
  DEC AX
  MOV SoundLen[SI],AX
  SHL SI,1
  ADD SI,SoundPage
  MOV DX,SI
  PUSH AX
  PUSH BX
  CALL MapEMS
  POP BX
  MOV AX,EMSseg
  MOV DS,AX
  XOR SI,SI
  MOV EAX,080808080h
  MOV CX,16384
ClearSound:
  MOV [SI],EAX
  ADD SI,4
  LOOP ClearSound
  POP CX
  XOR DX,DX
  MOV AH,03Fh
  INT 21h
  JC Error_ReadSound
  MOV AX,03E00h
  INT 21h
  XOR AX,AX
  JMP EndLoadSound
Error_OpenSound:
  MOV AX,1          ; Error 1: Unable to open specified sound file
  JMP EndLoadSound
Error_ReadSound:
  MOV AX,03E00h
  INT 21h
  MOV AX,2          ; Error 2: General sound file read error
  JMP EndLoadSound
Error_BadSound:
  MOV AX,03E00h
  INT 21h
  MOV AX,3          ; Error 3: Bad sound file format
  JMP EndLoadSound
Error_NotSupported:
  MOV AX,03E00h
  INT 21h
  MOV AX,4          ; Error 4: Sound format not yet supported
  JMP EndLoadSound
Error_SoundTooBig:
  MOV AX,03E00h
  INT 21h
  MOV AX,5          ; Error 5: Sound file too big
  JMP EndLoadSound
Error_BadSlot:
  MOV AX,03E00h
  INT 21h
  MOV AX,6          ; Error 6: Sound slot does not exist
EndLoadSound:
  POP BP
  POP DS
  RET 6
xDQBloadSound ENDP

; ---------------------------------------------------------------------------
; DQBloadRawSound FUNCTION
; purpose:
;   Loads a sound sample into a specified sound slot. Sounds must be 8 bit
;   mono and their sampling rate must not be greater than 22000 Hz, although
;   this function does not check for the format to be supported.
;   DQBloadRawSound also requires the offset where the sound data begins into
;   the specified file, plus the sound length in bytes; in this way you can
;   store several sounds into the same file, and load them all using this
;   function.
; declaration:
;   DECLARE FUNCTION xDQBloadRawSound(BYVAL Slot,BYVAL FileSeg,BYVAL FileOff,
;                                     BYVAL HiOff,BYVAL LoOff,BYVAL Length)
;   DECLARE FUNCTION DQBloadRawSound(Slot AS INTEGER,FileName AS STRING,
;                                     Offset AS LONG,Length AS LONG)
; ---------------------------------------------------------------------------
PUBLIC xDQBloadRawSound
xDQBloadRawSound PROC
  ; Stack layout
  ;
  ; 18  Slot
  ; 16  FileSeg
  ; 14  FileOff
  ; 12  HiOff
  ; 10  LoOff
  ; 08  Length
  ; 06  Basic return segment
  ; 04  Basic return offset
  ; 02  DS
  ; 00  BP
  PUSH DS
  PUSH BP
  MOV BP,SP
  MOV AX,[BP+18]
  XOR BH,BH
  MOV BL,NumSounds
  CMP AX,BX
  JG Error_BadSlot1
  MOV AX,[BP+16]
  MOV DS,AX
  MOV DX,[BP+14]
  MOV AX,03D00h
  INT 21h
  JC Error_OpenRaw
  MOV BX,AX
  MOV AX,04200h
  MOV CX,[BP+12]
  MOV DX,[BP+10]
  INT 21h
  JC Error_ReadRaw
  MOV AX,@DATA
  MOV DS,AX
  MOV SI,[BP+18]
  DEC SI
  SHL SI,1
  MOV AX,[BP+8]
  MOV SoundLen[SI],AX
  SHL SI,1
  ADD SI,SoundPage
  MOV DX,SI
  PUSH BX
  CALL MapEMS
  POP BX
  MOV AX,EMSseg
  MOV ES,AX
  MOV DS,AX
  XOR DI,DI
  MOV EAX,080808080h
  MOV CX,16384
  REP STOSD
  MOV CX,[BP+8]
  MOV AH,03Fh
  XOR DX,DX
  INT 21h
  JC Error_ReadRaw
  CMP AX,0
  JE Error_EOF
  MOV AX,03E00h
  INT 21h
  XOR AX,AX
  JMP EndLoadRaw
Error_OpenRaw:
  MOV AX,1          ; Error 1: Unable to open specified sound file
  JMP EndLoadRaw
Error_ReadRaw:
  MOV AX,03E00h
  INT 21h
  MOV AX,2          ; Error 2: General sound file read error
  JMP EndLoadRaw
Error_EOF:
  MOV AX,03E00h
  INT 21h
  MOV AX,3          ; Error 3: Cannot read past the end of file
  JMP EndLoadRaw
Error_BadSlot1:
  MOV AX,4          ; Error 4: Specified sound slot does not exist
EndLoadRaw:
  POP BP
  POP DS
  RET 12
xDQBloadRawSound ENDP

; ---------------------------------------------------------------------------
; DQBplaySound SUB
; purpose:
;   Plays a sound previously loaded in memory
; declaration:
;   DECLARE SUB DQBplaySound(BYVAL SoundNum,BYVAL Voice,BYVAL LoopFlag)
; ---------------------------------------------------------------------------
PUBLIC DQBplaySound
DQBplaySound PROC
  ; Stack layout
  ;
  ; 10  SoundNum
  ; 08  Voice
  ; 06  LoopFlag
  ; 04  Basic return segment
  ; 02  Basic return offset
  ; 00  BP
  PUSH BP
  MOV BP,SP
  CMP SoundActive,0
  JE EndPlaySound
  MOV AX,[BP+10]
  XOR AH,AH
  CMP AL,NumSounds
  JG EndPlaySound
  MOV SI,[BP+8]
  DEC SI
  SHL SI,1
  CMP VoiceSound[SI],0
  JNE StillPlaying
  INC Playing
StillPlaying:
  MOV VoiceSound[SI],AX
  MOV AX,[BP+6]
  MOV VoiceLoop[SI],AX
  MOV VoicePos[SI],0
EndPlaySound:
  POP BP
  RET 6
DQBplaySound ENDP

; ---------------------------------------------------------------------------
; DQBinUse FUNCTION
; purpose:
;   Returns true if a sound is currently being played on specified voice,
;   otherwise false.
; declaration:
;   DECLARE FUNCTION DQBinUse(BYVAL Voice)
; ---------------------------------------------------------------------------
PUBLIC DQBinUse
DQBinUse PROC
  PUSH BP
  MOV BP,SP
  MOV SI,[BP+6]
  DEC SI
  SHL SI,1
  XOR AX,AX
  CMP VoiceSound[SI],0
  SETNE AL
  NEG AX
  POP BP
  RET 2
DQBinUse ENDP

; ---------------------------------------------------------------------------
; DQBpauseSound SUB
; purpose:
;   Pauses the samples sound output
; declaration:
;   DECLARE SUB DQBpauseSound()
; ---------------------------------------------------------------------------
PUBLIC DQBpauseSound
DQBpauseSound PROC
  CMP PauseSound,1
  JE EndPauseSound
  MOV DX,BaseAddr
  ADD DX,0Ch
  WAITWRITEDSP
  MOV AL,0D0h
  OUT DX,AL
  MOV PauseSound,1
EndPauseSound:
  RET
DQBpauseSound ENDP

; ---------------------------------------------------------------------------
; DQBresumeSound SUB
; purpose:
;   Resumes the samples sound output
; declaration:
;   DECLARE SUB DQBresumeSound()
; ---------------------------------------------------------------------------
PUBLIC DQBresumeSound
DQBresumeSound PROC
  CMP PauseSound,0
  JE EndResumeSound
  MOV DX,BaseAddr
  ADD DX,0Ch
  WAITWRITEDSP
  MOV AL,0D4h
  OUT DX,AL
  MOV PauseSound,0
EndResumeSound:
  RET
DQBresumeSound ENDP

; ---------------------------------------------------------------------------
; DQBstopVoice SUB
; purpose:
;   Stops sound playing on specified voice
; declaration:
;   DECLARE SUB DQBstopVoice(BYVAL Voice)
; ---------------------------------------------------------------------------
PUBLIC DQBstopVoice
DQBstopVoice PROC
  PUSH BP
  MOV BP,SP
  MOV SI,[BP+6]
  DEC SI
  SHL SI,1
  CMP VoiceSound[SI],0
  JE EndStopVoice
  DEC Playing
  MOV VoiceSound[SI],0
EndStopVoice:
  POP BP
  RET 2
DQBstopVoice ENDP

; ---------------------------------------------------------------------------
; DQBsetVoiceVol SUB
; purpose:
;   Sets the sound output volume of a voice
; declaration:
;   DECLARE SUB DQBsetVoiceVol(BYVAL Voice,BYVAL NewVol)
; ---------------------------------------------------------------------------
PUBLIC DQBsetVoiceVol
DQBsetVoiceVol PROC
  PUSH BP
  MOV BP,SP
  MOV SI,[BP+8]
  DEC SI
  SHL SI,1
  MOV AX,[BP+6]
  AND AX,01Fh
  MOV VoiceVol[SI],AX
  POP BP
  RET 4
DQBsetVoiceVol ENDP

; ---------------------------------------------------------------------------
; DQBremoveSB SUB
; purpose:
;   Turns off SB output
; declaration:
;   DECLARE SUB DQBremoveSB()
; ---------------------------------------------------------------------------
PUBLIC DQBremoveSB
DQBremoveSB PROC
  CMP SoundActive,0
  JE EndRemoveSB
  MOV DX,BaseAddr
  ADD DX,06h
  MOV AL,1
  OUT DX,AL
  IN AL,DX
  IN AL,DX
  IN AL,DX
  IN AL,DX
  XOR AL,AL
  OUT DX,AL
  ADD DX,08h
  WAITREADDSP
  SUB DX,4
  IN AL,DX          ; SB reset
  IN AL,021h
  OR AL,PicORmask
  OUT 021h,AL
  XOR AX,AX
  MOV ES,AX
  MOV AL,IRQ
  ADD AL,8
  MOV SI,AX
  SHL SI,2
  CLI
  MOV AX,OldIRQoff
  MOV ES:[SI],AX
  MOV AX,OldIRQseg
  MOV ES:[SI+2],AX  ; Old IRQ restored
  STI
  MOV SoundActive,0
EndRemoveSB:
  RET
DQBremoveSB ENDP

; ---------------------------------------------------------------------------
; DQBsetVolume SUB
; purpose:
;   Sets the master volume for sound output, in the range 0-15
; declaration:
;   DECLARE SUB DQBsetVolume(BYVAL Volume)
; ---------------------------------------------------------------------------
PUBLIC DQBsetVolume
DQBsetVolume PROC
  PUSH BP
  MOV BP,SP
  MOV DX,BaseAddr
  ADD DX,4
  MOV AL,022h
  OUT DX,AL
  INC DX
  MOV AX,[BP+6]
  AND AL,0Fh
  MOV BL,AL
  SHL BL,4
  OR AL,BL
  OUT DX,AL
  POP BP
  RET 2
DQBsetVolume ENDP


; ***************************************************************************
; 3D ROUTINES
; ***************************************************************************


; ---------------------------------------------------------------------------
; DQBgtri SUB
; purpose:
;   Draws a gouraud shaded triangle with (x1,y1), (x2,y2) and (x3,y3) as
;   vertex, with their own colors c1, c2 and c3, on specified layer.
; declaration:
;   DECLARE SUB DQBgtri(BYVAL Layer,BYVAL x1,BYVAL y1,BYVAL c1,BYVAL x2,
;                         BYVAL y2,BYVAL c2,BYVAL x3,BYVAL y3,BYVAL c3)
; ---------------------------------------------------------------------------
PUBLIC DQBgtri
DQBgtri PROC
  ; Stack layout:
  ;           (test)
  ; 24  Layer
  ; 22  x1
  ; 20  y1
  ; 18  c1
  ; 16  x2
  ; 14  y2
  ; 12  c2
  ; 10  x3
  ; 08  y3
  ; 06  c3
  ; 04  Basic return segment
  ; 02  Basic return offset
  ; 00  BP
  PUSH BP
  MOV BP,SP
  MOV BX,[BP+24]
  CALL GetLayerSeg
  MOV ES,CX
  MOV AX,[BP+20]
  CMP AX,[BP+14]    ; Is y1>y2?
  JL Sorted12
  MOV BX,[BP+14]    ; Yes: swap y1 and y2
  MOV [BP+14],AX
  MOV [BP+20],BX
  MOV AX,[BP+22]    ; swap x1 and x2
  MOV BX,[BP+16]
  MOV [BP+16],AX
  MOV [BP+22],BX
  MOV AX,[BP+18]    ; swap c1 and c2
  MOV BX,[BP+12]
  MOV [BP+12],AX
  MOV [BP+18],BX
Sorted12:
  MOV AX,[BP+20]
  CMP AX,[BP+8]     ; Is y1>y3?
  JL Sorted13
  MOV BX,[BP+8]     ; Yes: swap y1 and y3
  MOV [BP+8],AX
  MOV [BP+20],BX
  MOV AX,[BP+22]    ; swap x1 and x3
  MOV BX,[BP+10]
  MOV [BP+10],AX
  MOV [BP+22],BX
  MOV AX,[BP+18]    ; swap c1 and c3
  MOV BX,[BP+6]
  MOV [BP+6],AX
  MOV [BP+18],BX
Sorted13:
  MOV AX,[BP+14]
  CMP AX,[BP+8]     ; Is y2>y3?
  JL Sorted23
  MOV BX,[BP+8]     ; Yes: swap y2 and y3
  MOV [BP+8],AX
  MOV [BP+14],BX
  MOV AX,[BP+16]    ; swap x2 and x3
  MOV BX,[BP+10]
  MOV [BP+10],AX
  MOV [BP+16],BX
  MOV AX,[BP+12]    ; swap c2 and c3
  MOV BX,[BP+6]
  MOV [BP+6],AX
  MOV [BP+12],BX
Sorted23:
  XOR SI,SI
MarkLine:
  MOV StartX[SI],0FFFFh
  INC SI
  CMP SI,200
  JL MarkLine

  MOV CX,[BP+14]
  SUB CX,[BP+20]    ; CX holds (y2-y1)
  JNZ Step12
  MOV AX,[BP+20]
  CMP AX,ClipY1     ; Out of clipping box: skip it
  JL Connect23
  CMP AX,ClipY2
  JG Connect23
  MOV AX,[BP+22]
  MOV SI,[BP+20]    ; SI holds the scanline number
  MOV StartX[SI],AX
  MOV AX,[BP+18]
  MOV StartCol[SI],AX
  JMP Connect23
Step12:
  MOV BX,[BP+18]    ; BX holds the current color value
  SHL BX,8          ; 8.8 fixed point math used
  MOV AX,[BP+16]
  SUB AX,[BP+22]
  SHL AX,7
  XOR DX,DX
  DIV CX           ; AX holds the m parameter
  MOV DX,[BP+22]    ; DX holds the current x value
  SHL DX,7          ; 9.7 fixed point math used
  PUSH AX
  PUSH DX
  MOV AX,[BP+12]
  SUB AX,[BP+18]
  SHL AX,8
  XOR DX,DX
  DIV CX
  MOV CX,AX         ; CX holds the color step
  POP DX
  POP AX
  MOV SI,[BP+20]
Draw12:
  CMP SI,ClipY1
  JL Next12
  CMP SI,ClipY2
  JG Next12
  PUSH DX
  SHR DX,7
  MOV StartX[SI],DX
  POP DX
  MOV StartCol[SI],BX
  SHR StartCol[SI],8
Next12:
  ADD DX,AX
  ADD BX,CX
  INC SI
  CMP SI,[BP+14]
  JLE Draw12

Connect23:
  MOV CX,[BP+8]
  SUB CX,[BP+14]    ; CX holds (y3-y2)
  JNZ Step23
  MOV AX,[BP+14]
  CMP AX,ClipY1     ; Out of clipping box: skip it
  JL Connect13
  CMP AX,ClipY2
  JG Connect13
  MOV AX,[BP+16]
  MOV SI,[BP+14]    ; SI holds the scanline number
  MOV StartX[SI],AX
  MOV AX,[BP+12]
  MOV StartCol[SI],AX
  JMP Connect13
Step23:
  MOV BX,[BP+12]    ; BX holds the current color value
  SHL BX,8          ; 8.8 fixed point math used
  MOV AX,[BP+10]
  SUB AX,[BP+16]
  SHL AX,7
  XOR DX,DX
  DIV CX           ; AX holds the m parameter
  MOV DX,[BP+16]    ; DX holds the current x value
  SHL DX,7          ; 9.7 fixed point math used
  PUSH AX
  PUSH DX
  MOV AX,[BP+6]
  SUB AX,[BP+12]
  SHL AX,8
  XOR DX,DX
  DIV CX
  MOV CX,AX         ; CX holds the color step
  POP DX
  POP AX
  MOV SI,[BP+14]
Draw23:
  CMP SI,ClipY1
  JL Next23
  CMP SI,ClipY2
  JG Next23
  PUSH DX
  SHR DX,7
  MOV StartX[SI],DX
  POP DX
  MOV StartCol[SI],BX
  SHR StartCol[SI],8
Next23:
  ADD DX,AX
  ADD BX,CX
  INC SI
  CMP SI,[BP+8]
  JLE Draw23

Connect13:
  MOV CX,[BP+8]
  SUB CX,[BP+20]    ; CX holds (y3-y1)
  JNZ Step13
  MOV AX,[BP+20]
  CMP AX,ClipY1     ; Out of clipping box: skip it
  JL DrawTri
  CMP AX,ClipY2
  JG DrawTri
  MOV AX,[BP+22]
  MOV SI,[BP+20]    ; SI holds the scanline number
  MOV EndX[SI],AX
  MOV AX,[BP+18]
  MOV EndCol[SI],AX
  JMP DrawTri
Step13:
  MOV BX,[BP+18]    ; BX holds the current color value
  SHL BX,8          ; 8.8 fixed point math used
  MOV AX,[BP+10]
  SUB AX,[BP+22]
  SHL AX,7
  XOR DX,DX
  DIV CX           ; AX holds the m parameter
  MOV DX,[BP+22]    ; DX holds the current x value
  SHL DX,7          ; 9.7 fixed point math used
  PUSH AX
  PUSH DX
  MOV AX,[BP+6]
  SUB AX,[BP+18]
  SHL AX,8
  XOR DX,DX
  DIV CX
  MOV CX,AX         ; CX holds the color step
  POP DX
  POP AX
  MOV SI,[BP+20]
Draw13:
  CMP SI,ClipY1
  JL Next13
  CMP SI,ClipY2
  JG Next13
  PUSH DX
  SHR DX,7
  MOV EndX[SI],DX
  POP DX
  MOV EndCol[SI],BX
  SHR EndCol[SI],8
Next13:
  ADD DX,AX
  ADD BX,CX
  INC SI
  CMP SI,[BP+8]
  JLE Draw13

DrawTri:
  MOV SI,ClipY1
  DEC SI
DrawTriLoop:
  MOV AX,0FFFFh
  INC SI
  CMP SI,ClipY2
  JG EndGTri
  CMP StartX[SI],AX
  JE DrawTriLoop
  MOV CX,EndX[SI]
  SUB CX,StartX[SI]
  INC CX
  MOV AX,EndCol[SI]
  SUB AX,StartCol[SI]
  SHL AX,8
  XOR DX,DX
  DIV CX
  MOV BX,AX
  MOV DX,StartCol[SI]
  SHL DX,8
  MOV DI,SI
  MOV CX,DI
  SHL DI,8
  SHL CX,6
  ADD DI,CX
  MOV CX,StartX[SI]
  DEC CX
  ADD DI,CX
  SUB DX,BX
DrawScanLine:
  INC CX
  INC DI
  ADD DX,BX
  CMP CX,EndX[SI]
  JG DrawTriLoop
  CMP CX,ClipX1
  JL DrawScanLine
  CMP CX,ClipX2
  JG DrawScanLine
  MOV ES:[DI],DH
  JMP DrawScanLine
EndGTri:
  POP BP
  RET 20
DQBgtri ENDP


END
