\TREK-N.XPL	AUG-17-92
\Arcade version of classic Star Trek game.
\Copyright 1992 Loren Blaney

\REVISIONS:
\JUN-03-92, Released to bulletin boards.
\JUN-07-92, Fixed blinking background on some computers, added more crystals.
\JUN-19-92, Seven-segment displays, bargraph reticle, planet and pinwheel
\ galaxy, fix spelling errors, fade all explosion fragments.
\JUL-25-92, Used optimized compiler, fast beep when energy is low, improved
\ red and yellow condition displays, command-line switch to start at any level,
\ fixed small bug in epilogue where hyphen broke word in wrong place. Flash
\ damage bargraphs and display GREEN condition at beginning of level. Klingons
\ shoot back. Two sound channels.
\AUG-17-92, Widen 7-segment displays, show yellow "ENERGY" and SHIELDS when low.
\ Display quadrant coordinates in traditional order: Y, X, automatically turn
\ sound off for slow processors, and improve explosion. Released as version 1.3.

include	\CXPL\CODESI;

ext	SETKBV, RESKBV, GETKEYS;		\Keyboard routines
ext	SETSTV, RESSTV, PLAYSND1, PLAYSND2;	\Sound routines

def	LF=$0A, FF=$0C, CR=$0D, SP=$20;
def	INTSIZE=2;		\Number of bytes per integer
\def	DEBUG= false;		\\Flag: Debug mode

def	SCRSIZE =80*25;		\Number of characters on the video screen
def	QUADWIDTH=80, QUADHEIGHT=13;	\Quadrant dimensions in character cells
def	GALWIDTH=QUADWIDTH*8, GALHEIGHT=QUADHEIGHT*8;
	\Note: GALWIDTH must be an even multiple of 16 because of XTALMAP, etc.
def	GAL16WIDTH =GALWIDTH /16;
def	ENERGYMAX=1600, SHIELDSMAX=1000;
def	GALMAPX=24, GALMAPY=16;	\Coords of galaxy map on screen (upper-left
				\ corner)
def	COMPX=60, COMPY=14;	\Location of compass on screen
def	STATX=19, STATY=14;	\Upper-right corner of print in status box
def	DAMAGEX=61, DAMAGEY=17;	\Upper-left corner of print in damage box
def	DAMAGEMAX=6;		\Maximum amount of damage an item can sustain
def \DAMAGE\ DAMENGINES, DAMPHASERS, DAMSENSORS, DAMVIEWER, DAMMAP, DAMCOMPASS;
def \CONDITION\	GREEN, YELLOW, RED;
def	SWEEPSIZE = 8888;	\Number of bytes in SWEEPSND

int	CPUREG,		\Address of copy of registers for soft interrupts, etc.
	PSPSEG,		\Segment of PSP (for sound switch)
	DATASEG,	\Our data segment (heap & stack)
	VIDMODE,	\Initial video mode to be restored upon exit
	VID7,		\Flag: VIDMODE = 7 (monochrome)
	CRTCSTAT,	\Port address of CRTC status register
	GRAPHICS,	\Flag: Display has character graphics capability
	VGA,		\Flag: VGA display (with fadeable color registers)
	EXPLOSION,	\Array: Explosion images
	KLNS,		\Initial number of Klingons for a level
	XTALS,		\Initial number of crystals for a level
	TRIBS,		\Initial number of tribbles for a level
	TRIBALL,	\Total tribbles for all levels played
	TRIBSAVE,	\Tribbles saved
	TRIBKILL,	\Tribbles killed by Klingons
	TRIBPHASER,	\Tribbles killed by Enterprise
	KLNKILL,	\Klingons killed by Enterprise
	TIME,		\Master timer, incremented 18.2 times per second
	ENERGY,		\Current energy level
	SHIELDS,	\Current shield level
	DAMAGE,		\Array: Damage for each item
	STARDATE,	\Current stardate
	LEVEL,		\Current playing level
	BASEX, BASEY,	\Location of starbase in the galaxy
	BASEOPEN,	\Flag: Starbase doors are open
	CONDITION,	\Green, yellow, red, etc.
	KEYS,		\Bit array of keys (which work like pushbuttons)
	SOUNDON,	\Flag: Sound is enabled
	CHEAT,		\Flag: Cheat mode enabled
	WON,		\Flag: Won level
	SHIPS,		\Number of Enterprise ships left
	LOGLINE,	\Line number on Captain's log
	II, JJ,		\Scratch for MAIN
	;
addr	KLNIMG,		\Image string for Klingon (used for graphics)
	ENTRIMG, ENTLIMG, ENTSIMG, \Enterprise images: left, right, and small
	PHASIMG2, PHASIMG1,	\Phaser images: "/" and "\
	BARIMG,		\Image for bar graphs
	NCCIMG,		\"NCC-1701" legend image
	PALREG,		\Array: Palette registers (and border color)
	COLOR0REG,	\Array: Initial values of all 256 VGA color registers
	COLORREG,	\Array: Color (DAC) registers: red, green, blue
	;
seg int	GALAXY;		\Big array containing background pattern of stars, etc.
seg addr SCREENSEG;	\Array for directly accessing the video screen for speed
int	SCRSEG;		\Address *16 of video screen memory
addr	VIEWPORT;	\Flag array: Indicates which bytes may be written to SCREENSEG
\Sound samples:
seg addr BOOMSND, CRASHSND, PHASERSND, OPENPODSND, BOBEEPSND, TRIBSND, SIRENSND,
	SQUEALSND, WARNSND, PINGSND, SWEEPSND;

\----------------------------------------------------------------------

proc	FATAL(STR);	\Fatal error handler
\WARNING: This does not restore keyboard and system timer interrupt handlers
addr	STR;
begin
SETVID(VIDMODE);
CURSOR(0, 0);
TEXT(0, STR);
exit;
end;	\FATAL



\proc	ERROR(STR, A, B, C);	\\Fatal error report (for debug)
\addr	STR;
\int	A, B, C;
\begin
\CURSOR(0, 0);
\TEXT(0, STR);
\INTOUT(0, A);   CHOUT(0, SP);
\INTOUT(0, B);   CHOUT(0, SP);
\INTOUT(0, C);	CHOUT(0, SP);
\loop;
\end;	\\ERROR

\----------------------------------------------------------------------

func	GETVID;		\Returns the current video mode
begin			\Requires: CPUREG:= GETREG;
CPUREG(0):= $0F00;	\AX function $0F
SOFTINT($10);		\Call BIOS routine
return CPUREG(0) & $FF;
end;	\GETVID



proc	SHOWCUR(FLAG);	\Turns flashing cursor on or off
int	FLAG;		\True = cursor on; False = cursor off
begin			\Requires: CPUREG:= GETREG;
CPUREG(0):= $0100;
CPUREG(2):= if FLAG then $0007 else $2000;
SOFTINT($10);		\Call BIOS routine
end;	\SHOWCUR



proc	WAITVB;		\Wait for the beginning of the next vertical blank
begin
while PIN(CRTCSTAT, 0) & $08 do;	\Wait for no vertical blank
repeat until PIN(CRTCSTAT, 0) & $08;	\Wait for vertical blank
end;	\WAITVB;



proc	SETBLKCOLOR;	\Set block of color registers
begin
CPUREG(0):= $1012;
CPUREG(1):= 0;
CPUREG(2):= $40;
CPUREG(3):= COLORREG;
CPUREG(11):= DATASEG;
SOFTINT($10);
end;	\SETBLKCOLOR



proc	FADE(DN, SPD);	\Fade the colors on the viewer up or down
int	DN, SPD;	\Flag: Fade down, Speed: 0=slow, 1=fast, 2=very fast
int	TBL, INT;


	proc	SET(I);		\Set intensity of the colors
	int	I;
	int	J, K, N, M, R, G, B;
	begin
	for J:= 0, 8 do
		begin
		N:= PALREG(TBL(J)) *3;
		for K:= N, N+2 do
			COLORREG(K):= (COLOR0REG(K) * I) >>6;
		end;
	WAITVB;
	SETBLKCOLOR;
	end;	\SET


begin	\FADE
\Table of palette registers that are fadeable (assigned to viewer):
TBL:= [$1, $3, $5, $6, $9, $B, $C, $D, $F];
\Thus, $0, $2, $4, $7, $8, $A, and $E are not fadeable (assigned to console).

if DN then
	for INT:= -$3F, 0 do [SET(-INT);   INT:= INT +SPD]
else	for INT:= 1, $40 do [SET(INT);   INT:= INT +SPD];
end;	\FADE



proc	CHANGECOLOR(A, B);	\Change color A to B
int	A, B;
int	I;
begin
A:= PALREG(A) *3;
B:= PALREG(B) *3;
for I:= 0, 2 do
	COLOR0REG(A+I):= COLOR0REG(B+I);
end;	\CHANGECOLOR

\----------------------------------------------------------------------

proc	DELAY(D);	\Delay D 18ths of a second (without messing up sound)
int	D;
int	I, J;
def	TIMER=$046C;	\least significant byte of system timer
begin
for I:= 1, D do
	begin
	J:= PEEK(0, TIMER);
	repeat until J # PEEK(0, TIMER);
	end;
end;	\DELAY



proc	PLAYSNDX(C, V, A);	\Play sound if enabled
int	C, V, A;
begin
if SOUNDON then
	case C of
	  1:	PLAYSND1(V, A);
	  2:	PLAYSND2(V, A)
	other;
end;	\PLAYSNDX



proc	REVSWEEP;	\Reverse the SWEEP sound sample
int	I, J, K;
begin
J:= SWEEPSIZE -2;	\Index to last sound byte before terminator
for I:= 0, SWEEPSIZE>>1-1 do
	begin
	K:= SWEEPSND(0, I);
	SWEEPSND(0, I):= SWEEPSND(0, J);
	SWEEPSND(0, J):= K;
	J:= J -1;
	end;
end;	\REVSWEEP

\----------------------------------------------------------------------

func	WRAPHEIGHT(Y);	\Wrap Y within the limits of the galaxy
int	Y;
begin
if Y >= GALHEIGHT then return Y -GALHEIGHT;
if Y < 0 then return Y +GALHEIGHT;
return Y;
end;	\WRAPHEIGHT



func	WRAPWIDTH(X);	\Wrap X within the limits of the galaxy
int	X;
begin
if X >= GALWIDTH then return X -GALWIDTH;
if X < 0 then return X +GALWIDTH;
return X;
end;	\WRAPWIDTH



func	CMPHEIGHT(A, B);	\Compare two Y coordinates in galaxy with wrap
int	A, B;
begin
if ABS(A-B) < GALHEIGHT>>1 then
	return A-B
else	return B-A;
end;	\CMPHEIGHT



func	CMPWIDTH(A, B);		\Compare two X coordinates in galaxy with wrap
int	A, B;
begin
if ABS(A-B) < GALWIDTH>>1 then
	return A-B
else	return B-A;
end;	\CMPWIDTH

\----------------------------------------------------------------------

proc	FLASHVIEW;	\Blinding flash on viewscreen
int	J;
begin
CURSOR(1, 1);
ATTRIB($0F);
for J:= 1, 78 do CHOUT(6, ^);
for J:= 2, 12 do
	BLIT(SCRSEG, 81*2, SCRSEG, (J*80+1)*2, 78*2);
BLIT(SCRSEG, 81*2, SCRSEG, (13*80+22)*2, (58-22+1)*2);
end;	\FLASHVIEW



proc	CLEARVIEW;	\Clear viewscreen
int	J;
begin
CURSOR(1, 1);
ATTRIB($0F);
for J:= 1, 78 do CHOUT(6, SP);
for J:= 2, 12 do
	BLIT(SCRSEG, 81*2, SCRSEG, (J*80+1)*2, 78*2);
BLIT(SCRSEG, 81*2, SCRSEG, (13*80+22)*2, (58-22+1)*2);
end;	\CLEARVIEW



proc	JUSTOUT(I);	\Output right-justified, 5-digit, positive integer
int	I;
int	J, K, F, D;
begin
K:= if GRAPHICS then $90 else $30;
D:= 1000;
F:= 0;
for J:= 1, 3 do
	begin
	I:= I /D;
	F:= F ! I;
	CHOUT(6, if F then I+K else SP);
	I:= REM(0);
	D:= D /10;
	end;
CHOUT(6, I+K);
end;	\JUSTOUT



proc	CTXT(X, Y, STR, ATB);
\This routine writes strings to the screen, but unlike RAWTEXT, attibutes
\ are defined for each character, SCREENSEG can be anywhere in memory, space
\ characters are transparent, the text is clipped to an arbitrarily shapped
\ viewport, and it doesn't wrap, and  is converted to a big star.
int	X, Y;		\Cursor position
addr	STR, ATB;	\Text and attribute strings
\Other inputs:
\	SCREENSEG,	Segment array cooresponding to physical video screen
\	VIEWPORT,	Flag array: Indicates which bytes may be written to SCREENSEG
int	CUR, I, J, CH;
begin
CUR:= Y*80 +X;			\Set cursor position
if CUR<0 ! CUR>=SCRSIZE then return;
for I:= 0, 32000 do
	begin
	CH:= STR(I);
	case CH of
	  SP:	;
	  $A0:	return		\Terminating space
	other	if VIEWPORT(CUR) then
		    begin
		    if CH = ^ then CH:= $0F;	\Handle big star
		    SCREENSEG(0, CUR+CUR):= CH;
		    J:= CUR+CUR+1;
		    SCREENSEG(0,J):= SCREENSEG(0,J)&$F0 ! ATB(I)&$0F;
		    end;
	CUR:= CUR +1;
	if REM(CUR/80) = 0 then return;		\Don't wrap
	end;
end;	\CTXT



proc	GRAPHRIGHT(AMT, MAX);	\Show AMT on bar graph (from left to right)
int	AMT, MAX;
int	N, I;
begin
N:= (AMT*16 + MAX >>1) /MAX;
for I:= 1, N do CHOUT(6, BARIMG);
ATTRIB(if VID7 then $70 else $80);
for I:= I, 16 do CHOUT(6, ^);
end;	\GRAPHRIGHT



proc	GRAPHDAMAGE(AMT);	\Show AMT on bar graph (from right to left)
int	AMT;
int	N, I, C;
begin
N:= DAMAGEMAX - AMT + 1;
C:= ^;
ATTRIB(if VID7 then $70 else $80);
for I:= 1, DAMAGEMAX do
	begin
	if I = N then
		begin
		C:= BARIMG;
		ATTRIB(if VID7 then $70 else $84);
		end;
	CHOUT(6, C);
	end;
end;	\GRAPHDAMAGE

\----------------------------------------------------------------------

proc	SHOWSTARDATE;	\Status display for stardate
int	I, J;
begin
ATTRIB(if VID7 then $07 else $0E);
CURSOR(STATX-5, STATY);
I:= STARDATE /10;
J:= REM(0);
JUSTOUT(I);
CHOUT(6, ^.);
CHOUT(6, J + (if GRAPHICS then $90 else $30));
end;	\SHOWSTARDATE



proc	SHOWCOND;	\Status display for condition
int	SEQ, NAME, C, X;
begin
SEQ:= [1, true];
if SEQ(0) # TIME then return;
SEQ(1):= ~SEQ(1);

C:= CONDITION;
if C = GREEN then
	case of ENERGY<ENERGYMAX/4, SHIELDS<=300: C:= YELLOW
	other;

case C of
  GREEN:  [SEQ(0):= TIME +9;  X:= 0];
  YELLOW: [SEQ(0):= TIME + (if ENERGY<200 ! SHIELDS<100 then 3 else 9);  X:= 1];
  RED:	  [SEQ(0):= TIME +3;  X:= 3]
other	X:= 0;

if C # GREEN then
    if SEQ(1) then
    	begin
	X:= if C = YELLOW then 2 else 4;	\Flashing
	end
    else if C = YELLOW then PLAYSNDX(2, 1, WARNSND(0));

NAME:=
  if VID7 then
       [[$7000+^ , $7000+^, $0700+^G, $0700+^R, $0700+^E, $0700+^E, $0700+^N, $7000+^],
	[$7000+^, $0700+^Y, $0700+^E, $0700+^L, $0700+^L, $0700+^O, $0700+^W, $7000+^],
	[$7000+^, $0000+^Y, $0000+^E, $0000+^L, $0000+^L, $0000+^O, $0000+^W, $7000+^],
	[$7000+^ , $7000+^, $0700+^ , $0700+^R, $0700+^E, $0700+^D, $0700+^ , $7000+^],
	[$7000+^ , $7000+^, $0000+^ , $0000+^R, $0000+^E, $0000+^D, $0000+^ , $7000+^]]
  else [[$8800+^ , $8200+^, $2000+^G, $2000+^R, $2000+^E, $2000+^E, $2000+^N, $8200+^],
	[$8E00+^, $E000+^Y, $E000+^E, $E000+^L, $E000+^L, $E000+^O, $E000+^W, $8E00+^],
	[$8000+^, $0E00+^Y, $0E00+^E, $0E00+^L, $0E00+^L, $0E00+^O, $0E00+^W, $8000+^],
	[$8800+^ , $8400+^, $4000+^ , $4000+^R, $4000+^E, $4000+^D, $4000+^ , $8400+^],
	[$8800+^ , $8000+^, $0400+^ , $0400+^R, $0400+^E, $0400+^D, $0400+^ , $8000+^]];

BLIT(DATASEG, NAME(X), SCRSEG, ((STATY+1)*80+STATX-6)*2, 8*2);
end;	\SHOWCOND



proc	SHOWENERGY;	\Status display for energy
int	SEQ;
begin
ATTRIB(if VID7 then $07 else $0E);
CURSOR(STATX-3, STATY+4);   JUSTOUT(ENERGY);
ATTRIB(if VID7 then $70 else $87);
CURSOR(STATX-15, STATY+5);   GRAPHRIGHT(ENERGY, ENERGYMAX);

SEQ:= [false];
if ENERGY < 200 then		\Show "ENERGY" in yellow
	begin
	if ~SEQ(0) then
		begin
		ATTRIB(if VID7 then $70 else $8E);
		CURSOR(STATX-15, STATY+4);
		TEXT(6, "ENERGY");
		SEQ(0):= true;
		end;
	end
else	begin			\Show "ENERGY" in black
	if SEQ(0) then
		begin
		ATTRIB(if VID7 then $70 else $80);
		CURSOR(STATX-15, STATY+4);
		TEXT(6, "ENERGY");
		SEQ(0):= false;
		end;
	end;
end;	\SHOWENERGY



proc	SHOWSHIELDS;	\Status display for shields
int	SEQ;
begin
ATTRIB(if VID7 then $07 else $0E);
CURSOR(STATX-3, STATY+6);   JUSTOUT(SHIELDS);
ATTRIB(if VID7 then $70 else $87);
CURSOR(STATX-15, STATY+7);   GRAPHRIGHT(SHIELDS, SHIELDSMAX);

SEQ:= [false];
if SHIELDS < 100 then		\Show "SHIELDS" in yellow
	begin
	if ~SEQ(0) then
		begin
		ATTRIB(if VID7 then $70 else $8E);
		CURSOR(STATX-15, STATY+6);
		TEXT(6, "SHIELDS");
		SEQ(0):= true;
		end;
	end
else	begin			\Show "SHIELDS" in black
	if SEQ(0) then
		begin
		ATTRIB(if VID7 then $70 else $80);
		CURSOR(STATX-15, STATY+6);
		TEXT(6, "SHIELDS");
		SEQ(0):= false;
		end;
	end;
end;	\SHOWSHIELDS



proc	SHOWLEVEL;	\Status display for level
begin
ATTRIB(if VID7 then $07 else $0E);
CURSOR(STATX-3, STATY+8);   JUSTOUT(LEVEL);
end;	\SHOWLEVEL



proc	SHOWSAVED;	\Status display for tribbles saved
begin
ATTRIB(if VID7 then $07 else $0E);
CURSOR(STATX-3, STATY+9);   JUSTOUT(TRIBSAVE);
end;	\SHOWSAVED



proc	SETGALMAP;	\Quickly set up galactic map (speed required for damage)
int	LINE, J;
begin
\GREEN "--- --- --- --- --- --- --- --- "
LINE:= [$022D, $022D, $022D, $0220,
	$022D, $022D, $022D, $0220,
	$022D, $022D, $022D, $0220,
	$022D, $022D, $022D, $0220,
	$022D, $022D, $022D, $0220,
	$022D, $022D, $022D, $0220,
	$022D, $022D, $022D, $0220,
	$022D, $022D, $022D, $0220];

for J:= GALMAPY, GALMAPY+7 do
	BLIT(DATASEG, LINE, SCRSEG, (J*80+GALMAPX+1)*2, 32*2);
end;	\SETGALMAP

\----------------------------------------------------------------------

proc	BURNENERGY(E);	\Reduce Enterprise energy by specified amount
int	E;
begin
ENERGY:= ENERGY -E;
if ENERGY < 0 then ENERGY:= 0;
end;	\BURNENERGY

\======================================================================

proc	PLAYLEVEL;	\Play a level
int	SCRX, SCRY,	\Coords of screen (includes the frame)
	ENTX, ENTY,	\Coords of left end of Enterprise in galaxy
	ENTSX, ENTSY,	\Screen coords of Enterprise
	ENTQX, ENTQY,	\Quadrant the Enterprise is in (0..7)
	ENTQX0, ENTQY0,	\Quadrant Enterprise was in
	KLNMAP,		\Array: Bit map of Klingons in galaxy
	KLNCNT,		\Array: Count of Klingons for each quadrant
	XTALMAP,	\Array: Bit map of crystals in galaxy
	XTALCNT,	\Array: Count of crystals for each quadrant
	TRIBMAP,	\Array: Bit map of tribbles in galaxy
	TRIBCNT,	\Array: Count of tribbles for each quadrant
	BITMAP,		\Table of bits 0 = left most
	CLUMPS,		\Number of clumps in the galaxy
	CLUMPTBL,	\Table of clumps: each entry contains: X, Y, size
	HEADRIGHT,	\Flag: Enterprise is moving to the right
	ENTDLY, KLNDLY, TIMEDLY, GASDLY, ENERDLY, \Control update rate
	I, J, K, L, X, Y,	\Tempories
	;



proc	HIT_ENT;	\Hit on Enterprise, shake things up
int	I;
begin
FLASHVIEW;
DELAY(2);
CLEARVIEW;

BLIT(SCRSEG, 0, SCRSEG, 24, SCRSIZE*2);
PLAYSNDX(1, 10, BOOMSND(0));
DELAY(4);
BLIT(SCRSEG, 24, SCRSEG, 0, SCRSIZE*2);
DELAY(4);

BLIT(SCRSEG, 0, SCRSEG, 150, SCRSIZE*2);
DELAY(4);
BLIT(SCRSEG, 150, SCRSEG, 0, SCRSIZE*2);

SHIELDS:= SHIELDS -300;
if SHIELDS < 0 then SHIELDS:= 0;
SHOWSHIELDS;

\Shake printer with reset
CPUREG(0):= $0100;
CPUREG(3):= $0000;
SOFTINT($17);

if ~CHEAT then
	begin		\Damage something at random
	I:= RAN(6);
	DAMAGE(I):= DAMAGE(I) + RAN(DAMAGEMAX) + 2;
	if DAMAGE(I) > DAMAGEMAX then DAMAGE(I):= DAMAGEMAX;

	CURSOR(DAMAGEX, DAMAGEY+I);
	GRAPHDAMAGE(DAMAGE(I));

	if I = DAMMAP then SETGALMAP;
	end;
PLAYSNDX(2, 10, CRASHSND(0));
end;	\HIT_ENT



proc	ENTDEAD;	\Fatal hit on Enterprise
int	I, J;
begin
for I:= 0, 6-1 do
	begin
	DAMAGE(I):= DAMAGEMAX;
	CURSOR(DAMAGEX, DAMAGEY+I);
	GRAPHDAMAGE(DAMAGE(I));
	end;

\Show damaged viewer and listen to explosion
for I:= 1, 18 do
	begin
	for J:= 1, 12 do
		BLIT(RAN(1000), 0, SCRSEG, (J*80+1)*2, (80-2)*2);
	BLIT(RAN(1000), 0, SCRSEG, (13*80+22)*2, (58-22+1)*2);
	DELAY(1);
	end;
CLEARVIEW;
DELAY(22);
end;	\ENTDEAD

\----------------------------------------------------------------------

proc	DOCOMPASS;	\Display the compass angle which is from the
			\ Enterprise's heading to the starbase
int	X,		\Current compass direction (degrees) (own-type variable)
	X0,		\Copy of X(0)
	X1,		\Target compass direction (degrees)
	DX, DY, V, I, K;
addr	NUMS;
begin
if DAMAGE(DAMCOMPASS)\#0\ then return;
X:= [0];

DX:= BASEX+3 -ENTX;
if DX >= GALWIDTH/2 then DX:= DX -GALWIDTH;	\Handle wrap around
if DX < -GALWIDTH/2 then DX:= DX +GALWIDTH;

DY:= BASEY+1 -ENTY;
if DY >= GALHEIGHT/2 then DY:= DY -GALHEIGHT;
if DY < -GALHEIGHT/2 then DY:= DY +GALHEIGHT;

X1:= if DX=0 & DY=0 then 0 else FIX(ATAN2(FLOAT(DY), FLOAT(DX)) *57.29577951);
if ~HEADRIGHT then X1:= X1 +180;
if X1 >= 360 then X1:= X1 -360
else if X1 < 0 then X1:= X1 +360;

\Move compass toward target direction with velocity V
X0:= X(0);
V:= X1 - X0;
if V > 180 then V:= V -360;
if V < -180 then V:= V +360;
case of
 V>60:	V:= 4;
 V<-60:	V:= -4;
 V>40:	V:= 3;
 V<-40:	V:= -3;
 V>20:	V:= 2;
 V<-20:	V:= -2;
 V>0:	V:= 1;
 V<0:	V:= -1
other;	\V=0
X0:= X0 +V;
if X0 >= 360 then X0:= X0 -360
else if X0 < 0 then X0:= X0 +360;

NUMS:= "       0      10      20      30      40      50      60      70      80      90     100     110     120     130     140     150     160     170     180     190     200     210     220     230     240     250     260     270     280     290     300     310     320     330     340     350       0       ";

K:= (COMPY *80 + COMPX) *2;		\Set cursor position
for I:= X0, X0+16 do
	begin
	SCREENSEG(0, K):= NUMS(I);
	K:= K +2;
	end;

X(0):= X0;
end;	\DOCOMPASS

\----------------------------------------------------------------------

proc	SHOWQUAD;	\Display the quadrant that the Enterprise is in
int	EX, EY, I, NAME;
begin
ATTRIB(if VID7 then $07 else $0E);
CURSOR(STATX-2, STATY+2);
I:= if GRAPHICS then $91 else $31;
CHOUT(6, ENTQY+I);   CHOUT(6, SP);   CHOUT(6, ENTQX+I);

\Display name of quadrant:		QUADRANT     8 6
\					Betelgeuse IV

NAME:=	["Deneb",	"Arcturus",	"Pollux",	"Capella",
	"Vega",		"Regulus",	"Procyon",	"Aldebaran",
	"Altair",	"Spica",	"Sirius",	"Betelgeuse",
	"Achernar",	"Antares",	"Canopus",	"Rigel"];

ATTRIB(if VID7 then $70 else $87);
CURSOR(STATX-15, STATY+3);   TEXT(6, "              ");
CURSOR(STATX-15, STATY+3);

EX:= ENTQX >>1;   EY:= ENTQY >>1;
TEXT(6, NAME(EY<<2 !EX));

CHOUT(6, SP);   CHOUT(6, ^I);			\I II III IV
case (ENTQX&$01) ! (ENTQY&$01) <<1 of
  0:	CHOUT(6, ^I);
  2:	[CHOUT(6, ^I);   CHOUT(6, ^I)];
  3:	CHOUT(6, ^V)
other;
end;	\SHOWQUAD

\----------------------------------------------------------------------

proc	DOGALMAP;	\Update map with long-range sensor scan
int	SEQ,		\Own-type variable to control rate of execution
	KBS,		\Klingons, Bases, Stars
	K,
	SCANTBL,
	SCANCNT,	\Sequentially scans quadrants for long-range scanner
	X, Y;
addr	SIZETBL;


	proc	SHOWSCAN(QX, QY);	\Show scan for quadrant
	int	QX, QY;		\Quadrant being scanned
	int	I, J;
	begin
	CURSOR(GALMAPX+1+QX*4, GALMAPY+QY);

	I:= KLNCNT(QX, QY);
	if I > 62 then I:= 62;		\62 = 10 +26 +26 +1 -1
	CHOUT(6, if I = 0 then SP else SIZETBL(I));

	J:= XTALCNT(QX, QY);
	if J > 62 then J:= 62;
	CHOUT(6, if J=0 & I=0 then SP else SIZETBL(J));

	J:= TRIBCNT(QX, QY);
	if J > 62 then J:= 62;
	CHOUT(6, SIZETBL(J));
	end;	\SHOWSCAN


begin	\DOGALMAP
SEQ:= [1, 0];
if SEQ(1) # LEVEL then	\make sure damage is repaired when starting new level
	[SEQ(1):= LEVEL;   SEQ(0):= TIME];
if SEQ(0) # TIME then return;

K:= DAMAGE(DAMSENSORS);
SEQ(0):= TIME + 5 + K *K *K;

SCANCNT:= [0];
SIZETBL:= "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz ";

if DAMAGE(DAMMAP)\#0\ then SETGALMAP
else	begin	\Turn off previously intensified quadrant (beware of Ent having
		\ moved out of previous quad, also HILITE intrinsic is slow)
	for Y:= GALMAPY, GALMAPY+7 do
		begin
		K:= (Y *80 + (GALMAPX+1)) *2 +1;
		for X:= 0, 7 do
			begin
			SCREENSEG(0, K):= $02;
			SCREENSEG(0, K+2):= $02;
			SCREENSEG(0, K+4):= $02;
			K:= K +8;
			end;
		end;
	end;

\Scan quadrant Enterprise is in and display inverse video numbers
ATTRIB(if VID7 then $70 else $20);
SHOWSCAN(ENTQX, ENTQY);

\Scan next quadrant and display intensified numbers
SCANCNT(0):= SCANCNT(0) -1;		\Scan clockwise
SCANTBL:= [0, 1, 1, 1, 0, -1, -1, -1];
ATTRIB($0A);
X:= (ENTQX + SCANTBL(SCANCNT(0) &7) ) &7;	\Wrap both table and map
Y:= (ENTQY + SCANTBL((SCANCNT(0)+2) &7) ) &7;
SHOWSCAN(X, Y);
end;	\DOGALMAP

\----------------------------------------------------------------------

proc	DELETEKLN(X, Y);	\Delete the Klingon at X, Y
int	X, Y;
int	XW, K;
begin
\if DEBUG then
\	if X<0 ! X>=GALWIDTH ! Y<0 ! Y>=GALHEIGHT then
\		ERROR("DELETEKLN OUT OF RANGE (X, Y, 0) ", X, Y, 0);
XW:= X /16;   K:= REM(0);
KLNMAP(XW, Y):= KLNMAP(XW, Y) & ~BITMAP(K);

X:= X /QUADWIDTH;   Y:= Y /QUADHEIGHT;
KLNCNT(X, Y):= KLNCNT(X, Y) -1;
end;	\DELETEKLN



proc	DELETEXTAL(X, Y);	\Delete the crystal at X, Y
int	X, Y;
int	XW, K;
begin
\if DEBUG then
\	if X<0 ! X>=GALWIDTH ! Y<0 ! Y>=GALHEIGHT then
\		ERROR("DELETEXTAL OUT OF RANGE (X, Y, 0) ", X, Y, 0);
XW:= X /16;   K:= REM(0);
XTALMAP(XW, Y):= XTALMAP(XW, Y) & ~BITMAP(K);

X:= X /QUADWIDTH;   Y:= Y /QUADHEIGHT;
XTALCNT(X, Y):= XTALCNT(X, Y) -1;
end;	\DELETEXTAL



proc	DELETETRIB(X, Y);	\Delete the tribble at X, Y
int	X, Y;
int	XW, K;
begin
\if DEBUG then
\	if X<0 ! X>=GALWIDTH ! Y<0 ! Y>=GALHEIGHT then
\		ERROR("DELETETRIB OUT OF RANGE (X, Y, 0) ", X, Y, 0);
XW:= X /16;   K:= REM(0);
TRIBMAP(XW, Y):= TRIBMAP(XW, Y) & ~BITMAP(K);

X:= X /QUADWIDTH;   Y:= Y /QUADHEIGHT;
TRIBCNT(X, Y):= TRIBCNT(X, Y) -1;
end;	\DELETETRIB



proc	DOPHASER;	\Phaser control
int	J, X, Y;
addr	S;
int	SEQ, SSEQ, ARGX, ARGY, ARGZ;	\OWN-TYPE VARIABLES


	proc	EXPLODE(I);	\Explode a Klingon
	int	I;		\frame of explosion
	int	X, Y, J;
	begin
	X:= WRAPWIDTH(ARGX(I)-SCRX-8);
	Y:= ARGY(I)-SCRY;
	J:= I + 20 *ARGZ(I);
	CTXT(X, WRAPHEIGHT(Y-1), EXPLOSION(J, 0, 0), EXPLOSION(J, 1, 0));
	CTXT(X, WRAPHEIGHT(Y), EXPLOSION(J, 0, 1), EXPLOSION(J, 1, 1));
	CTXT(X, WRAPHEIGHT(Y+1), EXPLOSION(J, 0, 2), EXPLOSION(J, 1, 2));
	end;	\EXPLODE



	func	HIT;	\Returns 'true' if something is hit
	\Inputs: X,Y = position of segment of phaser beam
	int	K, X0, X1, XW;
	begin
	for X0:= X-4, X do			\Klingons are 5 characters wide
		begin
		X1:= if X0<0 then X0 +GALWIDTH else X0;
		XW:= X1 /16;   K:= BITMAP(REM(0));
		if KLNMAP(XW, Y) & K then
			begin			\Hit Klingon
			DELETEKLN(X1, Y);
			KLNKILL:= KLNKILL +1;
			SEQ(0):= TIME +1;	\Wait for flash (box problem)
			ARGX(0):= X;   ARGY(0):= Y;
			ARGZ(0):= if X0 <= X-2 then 1 else 0;
			PLAYSNDX(1, 10, BOOMSND(0));
			return true;
			end;

		if XTALMAP(XW, Y) & K then
		    case X-X0 of 2, 1, 0:	\Xtals are 3 chars wide
			begin			\Hit crystal
			DELETEXTAL(X1, Y);
			return true;
			end
		    other;
		end;

	if TRIBMAP(XW, Y) & K then		\Tribbles are one char wide
		begin				\Hit poor little tribble
		PLAYSNDX(2, 10, SQUEALSND(0));
		DELETETRIB(X, Y);
		TRIBPHASER:= TRIBPHASER +1;
		return true;
		end;

	return false;
	end;	\HIT


begin	\DOPHASER
SEQ:=  [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
ARGX:= [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
ARGY:= [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
ARGZ:= [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
SSEQ:= [1];

\Movie sequence:
for J:= 0, 19 do
	begin
	if SEQ(J) = TIME then
		begin
		EXPLODE(J);
		SEQ(J+1):= TIME +2;	\Interleaves explosion patterns
		ARGX(J+1):= ARGX(J);  ARGY(J+1):= ARGY(J);  ARGZ(J+1):= ARGZ(J);
		end;
	end;
if SEQ(20) = TIME then EXPLODE(20);
if SEQ(0) = TIME then FLASHVIEW;

if SSEQ(0) = TIME then
    begin
    if (KEYS&$40)\#0\ & DAMAGE(DAMPHASERS)<5  then	\space bar = fire
	begin
	PLAYSNDX(1, 2, PHASERSND(0));
	BURNENERGY(20);

	X:= ENTX;   Y:= ENTY;
	loop	begin			\Bit:	3	2	1	0	
		case of			\Key:	right	left	down	up
		(KEYS & $0003) #0 & (KEYS & $000C) #0:
		    begin	\Moving diagonally
		    X:= WRAPWIDTH(X + (if KEYS & $0008 then 6 else -2));
		    S:= if (KEYS&8)#0 & (KEYS&2)#0 !
		    	(KEYS&8)=0 & (KEYS&2)=0 then PHASIMG1 else PHASIMG2;
		    for J:= 1, 5-DAMAGE(DAMPHASERS) do
			begin
			Y:= WRAPHEIGHT(Y + (if KEYS & $0002 then 1 else -1));
			CTXT(WRAPWIDTH(X-SCRX), WRAPHEIGHT(Y-SCRY), S, "<< ");
			if HIT then quit;
			X:= WRAPWIDTH(X + (if KEYS & $0008 then 2 else -2));
			end;
		    quit;	\Missed
		    end;

		KEYS & $0003:
		    begin	\Moving up or down
		    X:= WRAPWIDTH(X + (if HEADRIGHT then 3 else 2));
		    for J:= 1, 5-DAMAGE(DAMPHASERS) do
			begin
			Y:= WRAPHEIGHT(Y + (if KEYS & $0002 then 1 else -1));
			CTXT(WRAPWIDTH(X-SCRX), WRAPHEIGHT(Y-SCRY), " ", "< ");
			if HIT then quit;
			end;
		    quit;	\Missed
		    end

		other
		    begin	\Shoot left or right
		    if HEADRIGHT then X:= WRAPWIDTH(X +5);
		    for J:= 1, 17-DAMAGE(DAMPHASERS)*4 do
			begin
			X:= WRAPWIDTH(X + (if HEADRIGHT then 1 else -1));
			CTXT(WRAPWIDTH(X-SCRX), WRAPHEIGHT(Y-SCRY), " ", "< ");
			if HIT then quit;
			end;
		    quit;	\Missed
		    end;
		end;	\loop

	SSEQ(0):= TIME + 3;
	if DAMAGE(DAMPHASERS)\#0\ then
		SSEQ(0):= SSEQ(0) + RAN(DAMAGE(DAMPHASERS)*10);
	end
    else SSEQ(0):= TIME +1;
    end;
end;	\DOPHASER

\----------------------------------------------------------------------

proc	DOBASE;		\Handle starbase (this doesn't handle galaxy wrap)
int	I, ATB, ATB2, ATB3;
int	SEQ, ARG, GOING;	\Own-type variables
def	DLY=2;			\Controls speed that doors open
def	LP=$DD, RP=$DE;		\Left and right door panels
\Inputs: BASEX, BASEY
begin
SEQ:= [0, 0, 0, 0, 0, 0, 1];   ARG:= [0];   GOING:= [false];
\	[4][V][A]
\	 123454321 
\	[<][>]

ATB:= if VID7 then $0700 else if VGA then $D900 else $E800;
ATB2:= if VID7 then $7000 else if VGA then $9000 else $8000;

if TIME = SEQ(6) then
	begin				\Flash green and black
	ATB3:= if SEQ(6)&1 then
		(if VID7 then $7700 else (if VGA then $9A00 else $8A00))
		else ATB2;
	GALAXY(BASEY+2, BASEX+10):= ATB2+$1A;		\no flash
	GALAXY(BASEY+2, BASEX+2):= ATB2+$1B;
	I:= CMPWIDTH(ENTX, BASEX+3);
	case of
	  I<-1:	GALAXY(BASEY+2, BASEX+10):= ATB3+$1A;	\flash right
	  I>+1:	GALAXY(BASEY+2, BASEX+2):= ATB3+$1B	\flash left
	other;

	GALAXY(BASEY, BASEX+7):= ATB2+$19;		\no flash
	GALAXY(BASEY, BASEX+10):= ATB2+$18;
	I:= CMPHEIGHT(ENTY, BASEY+1);
	case of
	  I<0:	GALAXY(BASEY, BASEX+7):= ATB3+$19;	\flash down
	  I>0:	GALAXY(BASEY, BASEX+10):= ATB3+$18	\flash up
	other;

	SEQ(6):= TIME +5;		\Must be odd number
	end;

if ABS(ENTX-(BASEX+3))<12 then
    if ABS(ENTY-(BASEY+1))<4 & ~BASEOPEN & ~GOING(0) then
	begin				\Open the pod bay doors, Hal
	GOING(0):= true;
	PLAYSNDX(2, 10, OPENPODSND(0));
	SEQ(5):= TIME +39;		\Schedule opening
	end;

case TIME of
SEQ(5):	begin
	GALAXY(BASEY+1, BASEX+6):= ATB+SP;	\Open center
	SEQ(0):= TIME +DLY;		\Schedule next sections
	ARG(0):= 1;			\Pass an argument
	end;
SEQ(0):	begin				\Open left and right panels
	I:= ARG(0);
	GALAXY(BASEY+1, BASEX+6-I):= ATB+LP;
	GALAXY(BASEY+1, BASEX+6+I):= ATB+RP;
	SEQ(1):= TIME +DLY;		\Schedule next sections
	end;
SEQ(1):	begin				\Open some more
	I:= ARG(0);
	GALAXY(BASEY+1, BASEX+6-I):= ATB+SP;
	GALAXY(BASEY+1, BASEX+6+I):= ATB+SP;
	I:= I +1;
	if I <= 4 then 
		[SEQ(0):= TIME +DLY;   ARG(0):= I]
	else	[BASEOPEN:= true;   GOING(0):= false];
	end
other;

if BASEOPEN then
    if (ABS(ENTX-(BASEX+3))>12 ! ABS(ENTY-(BASEY+1))>4) & ~GOING(0) then
	begin				\Close 'em Hal
	GOING(0):= true;
	BASEOPEN:= false;
	SEQ(2):= TIME;			\Start closing immediately
	ARG(0):= 4;
	end;

case TIME of
SEQ(2):	begin
	I:= ARG(0);
	GALAXY(BASEY+1, BASEX+6-I):= ATB+LP;
	GALAXY(BASEY+1, BASEX+6+I):= ATB+RP;
	SEQ(3):= TIME +DLY;
	end;
SEQ(3):	begin
	I:= ARG(0);
	GALAXY(BASEY+1, BASEX+6-I):= ATB2;
	GALAXY(BASEY+1, BASEX+6+I):= ATB2;
	I:= I -1;
	if I >= 1 then 
		[SEQ(2):= TIME +DLY;   ARG(0):= I]
	else	SEQ(4):= TIME +DLY;
	end;
SEQ(4):	begin
	GALAXY(BASEY+1, BASEX+6):= ATB2;
	GOING(0):= false;
	end
other;
end;	\DOBASE



proc	DOCK;		\Docking sequence
int	X, Y, I;
def	DLY=2;			\Controls speed that doors close
def	LP=$DD, RP=$DE, FP=$DB;	\Left, right and full door panels
\Inputs: BASEX, BASEY, SCRX, SCRY
\	[4][V][A]
\	 123454321 
\	[<][>]
begin
X:= BASEX-SCRX;		Y:= BASEY-SCRY;
ATTRIB(if VID7 then $70 else if VGA then $90 else $80);
CURSOR(WRAPWIDTH(X+10), WRAPHEIGHT(Y+2));   CHOUT(6, $1A);	\no flash
CURSOR(WRAPWIDTH(X+2), WRAPHEIGHT(Y+2));   CHOUT(6, $1B);
CURSOR(WRAPWIDTH(X+7), WRAPHEIGHT(Y));   CHOUT(6, $19);
CURSOR(WRAPWIDTH(X+10), WRAPHEIGHT(Y));   CHOUT(6, $18);

X:= BASEX+6-SCRX;	Y:= WRAPHEIGHT(BASEY+1-SCRY);
ATTRIB(if VID7 then $07 else if VGA then $D9 else $E8);
for I:= -4, -1 do
	begin				\Close 'em Hal
	CURSOR(WRAPWIDTH(X+I), Y);   CHOUT(6, LP);
	CURSOR(WRAPWIDTH(X-I), Y);   CHOUT(6, RP);
	DELAY(DLY);

	CURSOR(WRAPWIDTH(X+I), Y);   CHOUT(6, FP);
	CURSOR(WRAPWIDTH(X-I), Y);   CHOUT(6, FP);
	DELAY(DLY);
	end;

CURSOR(WRAPWIDTH(X), Y);   CHOUT(6, FP);
DELAY(9);
end;	\DOCK

\----------------------------------------------------------------------

proc	KLNPHASER(X, Y, KEYS);	\Klingon phaser control
int	X, Y, KEYS;		\Coordinates of Klingon firing phasers
int	J;
addr	S;


	func	HIT;
	\Returns 'true' if something (other than a Klingon) is hit.
	\Inputs: X, Y = position of segment of phaser beam
	int	K, X0, X1, XW, I;
	begin
	for X0:= X-5, X do			\Enterprise is 6 characters wide
		begin
		X1:= if X0<0 then X0 +GALWIDTH else X0;
		if ENTX=X1 & ENTY=Y then
			begin			\Hit Enterprise
			PLAYSNDX(1, 9, CRASHSND(0));
			I:= (RAN(8)+1) <<1;
			BLIT(SCRSEG, 0, SCRSEG, I, SCRSIZE*2);
			DELAY(1);
			BLIT(SCRSEG, I, SCRSEG, 0, SCRSIZE*2);

			SHIELDS:= SHIELDS -19;
			if SHIELDS < 0 then SHIELDS:= 0;
			SHOWSHIELDS;
			return true;
			end;

		XW:= X1 /16;   K:= BITMAP(REM(0));
		if XTALMAP(XW, Y) & K then
		    case X-X0 of 2, 1, 0:	\Xtals are 3 chars wide
			begin			\Hit crystal
			DELETEXTAL(X1, Y);
			return true;
			end
		    other;
		end;

	if TRIBMAP(XW, Y) & K then		\Tribbles are one char wide
		begin				\Hit poor little tribble
		PLAYSNDX(2, 10, SQUEALSND(0));
		DELETETRIB(X, Y);
		TRIBKILL:= TRIBKILL +1;
		return true;
		end;

	return false;
	end;	\HIT


begin	\KLNPHASER
PLAYSNDX(1, 2, PHASERSND(0));
loop	begin
	case of
	(KEYS & $0003) #0 & (KEYS & $000C) #0:
	    begin	\Shoot diagonally
	    X:= WRAPWIDTH(X + (if KEYS & $0008 then 5 else -2));
	    S:= if (KEYS&8)#0 & (KEYS&2)#0 !
	    	(KEYS&8)=0 & (KEYS&2)=0 then PHASIMG1 else PHASIMG2;
	    for J:= 1, 5 do
		begin
		Y:= WRAPHEIGHT(Y + (if KEYS & $0002 then 1 else -1));
		if HIT then quit;
		CTXT(WRAPWIDTH(X-SCRX), WRAPHEIGHT(Y-SCRY), S, "33 ");
		X:= WRAPWIDTH(X + (if KEYS & $0008 then 2 else -2));
		end;
	    quit;	\Missed
	    end;

	KEYS & $0003:
	    begin	\Shoot up or down
	    X:= WRAPWIDTH(X+2);
	    for J:= 1, 5 do
		begin
		Y:= WRAPHEIGHT(Y + (if KEYS & $0002 then 1 else -1));
		if HIT then quit;
		CTXT(WRAPWIDTH(X-SCRX), WRAPHEIGHT(Y-SCRY), " ", "3 ");
		end;
	    quit;	\Missed
	    end

	other
	    begin	\Shoot left or right
	    if KEYS & 8 then X:= WRAPWIDTH(X +4);
	    for J:= 1, 17 do
		begin
		X:= WRAPWIDTH(X + (if KEYS & 8 then 1 else -1));
		if HIT then quit;
		CTXT(WRAPWIDTH(X-SCRX), WRAPHEIGHT(Y-SCRY), " ", "3 ");
		end;
	    quit;	\Missed
	    end;

	end;	\loop
end;	\KLNPHASER



proc	DOKLNS(XW, Y);	\Deal with Klingons
\Other inputs: ENTX, ENTY, SCRX, SCRY, KLNMAP, BITMAP, KLNCNT, KLNDLY
\(Klingons moving down or to the right get to move twice sometimes.)
int	XW, Y;		\coordinates of word in upper-left corner of viewer
int	X, K,		\X,Y & X1,Y1 are cell coordinates relative to galaxy
	X1, Y1,		\new position
	XW1, K1,	\new position for word form of X position
	XQ, YQ,
	L, X2,
	SEQ,
	LX, LY, DX, DY,
	J;
begin
SEQ:= [0, 0];
L:= KLNMAP(XW, Y);	\get 16 bits
K:= 0;
while L \#0\ do
	begin
	if L & $8000 then
		begin
		X:= XW*16 +K;
		X2:= WRAPWIDTH(X-SCRX);
		if X2>=1 & X2<=78 then
			begin			\Klingon is on screen
			CONDITION:= RED;
			if TIME -SEQ(0) >= 0 then \tricky wrap at 32K
				begin		  \ don't use TIME >= SEQ(0)
				PLAYSNDX(1, 10, SIRENSND(0));
				SEQ(0):= TIME +36;
				end;

			X1:= X;    Y1:= Y;	\Top secret Klingon tactics
			LY:= 10 -LEVEL;		\(hint: big box is easy)
			if LY < 1 then LY:= 1;
			LX:= LY *5;
			DX:= CMPWIDTH(X1, ENTX);
			DY:= CMPHEIGHT(Y1, ENTY);
			if ABS(DX)<=LX & ABS(DY)<=LY then [LX:= 0;   LY:= 0];
			if KLNDLY \#0\ then
				begin		\Move toward Enterprise
				if DX <-LX then X1:= X1 +1;
				if DX >LX then X1:= X1 -1;
				X1:= WRAPWIDTH(X1);
				end;
			KLNDLY:= KLNDLY +1;	\Each Klingon should have its
			if KLNDLY >= 5 then	\ own delay, but this makes them
				begin		\ fly in interesting formations
				KLNDLY:= 0;
				if DY <-LY then Y1:= Y1 +1;
				if DY >LY then Y1:= Y1 -1;
				Y1:= WRAPHEIGHT(Y1);
				end;

			\if DEBUG then
			\	if X1<0 ! X1>=GALWIDTH ! Y1<0 ! Y1>=GALHEIGHT then
			\		ERROR("DOKLNS OUT OF RANGE (X1, Y1, L) ", X1, Y1, L);

			XW1:= X1 /16;   K1:= REM(0);
			\Move Klingon unless it will be on top of another Klingon
			if (KLNMAP(XW1, Y1) & BITMAP(K1)) = 0 then
				begin
				DELETEKLN(X, Y);	\Remove from current posn
				KLNMAP(XW1, Y1):= KLNMAP(XW1, Y1) ! BITMAP(K1);
				XQ:= X1 /QUADWIDTH;   YQ:= Y1 /QUADHEIGHT;
				KLNCNT(XQ, YQ):= KLNCNT(XQ, YQ) +1;

				if XTALMAP(XW1, Y1) & BITMAP(K1) then
					DELETEXTAL(X1, Y1);
				if TRIBMAP(XW1, Y1) & BITMAP(K1) then
					begin
					PLAYSNDX(2, 10, SQUEALSND(0));
					DELETETRIB(X1, Y1);
					TRIBKILL:= TRIBKILL +1;
					end;
				end;

			CTXT(WRAPWIDTH(X1-SCRX), WRAPHEIGHT(Y1-SCRY), KLNIMG,
			    if VID7 then "77777 " else if VGA then "99999 "else "88888 ");

			if TIME -SEQ(1) >= 0 then	\(tricky wrap at 32K)
			    begin		\Ent is too hard to control if
			    DX:= DX /3;		\ it is hit continuously
			    if RAN(25 +DX*DX +DY*DY) < LEVEL then
				begin
				SEQ(1):= TIME +2;
				J:= 0;
				if DX > 0 then J:= J !4
				else if DX < 0 then [J:= J !8;  DX:= -DX];
				if DY >= DX then J:= J !1
				else if DY <= -DX then J:= J !2;
				KLNPHASER(X1, Y1, J);
				end;
			    end;

			\See if a Klingon rammed the Enterprise
			if Y1=ENTY & ABS(X1-ENTX)<=4 then
				begin
				HIT_ENT;
				DELETEKLN(X1, Y1);
				KLNKILL:= KLNKILL +1;
				end;
			end;
		end;
	L:= L <<1;
	K:= K +1;
	end;
end;	\DOKLNS

\----------------------------------------------------------------------

proc	DOXTALS(XW, Y);	\Deal with dilithium crystals
int	XW, Y;		\Coordinates of word containing one or more xtals
int	K, L, M, N, X1, Y1, SEQ;
addr	TBL, STR, ATB;
begin
SEQ:= [100];
TBL:= "-+-==== ";	\Table of centers
STR:= [$0011, $A010];	\"<=> " This is really a string
ATB:= RESERVE(3);	\Attributes corresponding to characters in string

K:= XW +Y;
M:= REM(ABS(TIME+K+K)/20);	\rotate center
N:= REM(M/10);
STR(1):= TBL(N);
ATB(1):= if N=1 then ^?	\glint
	else (if M>=10 then ^5 else^3);
if M>=10 then
	begin
	ATB(0):= if K&1 then ^3 else ^5;
	ATB(2):= if K&1 then ^5 else ^3;
	end
else	begin
	ATB(2):= if K&1 then ^3 else ^5;
	ATB(0):= if K&1 then ^5 else ^3;
	end;

if TIME -SEQ(0) >= 0 then	\tricky wrap at 32K
	begin			\ don't use TIME >= SEQ(0)
	PLAYSNDX(2, 1, PINGSND(0));
	SEQ(0):= TIME +50 +RAN(100);
	end;

L:= XTALMAP(XW, Y);	\get 16 bits
K:= 0;
while L \#0\ do
	begin
	if L & $8000 then
		begin
		X1:= XW*16 +K -SCRX;	\offset from left edge of screen
		if X1 < 0 then X1:= X1 +GALWIDTH;
		if X1 <= 78 then
		    begin
		    if Y=ENTY & X1>=ENTSX-2 & X1<ENTSX+6 then
			begin		\picked up by Enterprise
			DELETEXTAL(XW*16+K, Y);
			\3 xtals restores a Klingon hit (300), therefore
			\ each xtal is worth 100 shield units or, if
			\ shields are not low then 100 energy units.
			if SHIELDS*16/SHIELDSMAX <= ENERGY*16/ENERGYMAX then
				begin
				SHIELDS:= SHIELDS +100;
				if SHIELDS > SHIELDSMAX then SHIELDS:= SHIELDSMAX;
				SHOWSHIELDS;
				end
			else	begin
				ENERGY:= ENERGY +100;
				if ENERGY > ENERGYMAX then ENERGY:= ENERGYMAX;
				SHOWENERGY;
				end;
			PLAYSNDX(2, 6, BOBEEPSND(0));
			end
		    else CTXT(X1, WRAPHEIGHT(Y-SCRY), STR, ATB);
		    end;
		end;
	L:= L <<1;
	K:= K +1;
	end;
end;	\DOXTALS

\----------------------------------------------------------------------

proc	DOTRIBS(XW, Y);	\Deal with tribbles
int	XW, Y;		\Coordinates of word containing one or more tribbles
int	K, L, X1, Y1, ATB, CUR;
addr	TBL;
begin
TBL:= if VGA then "69=6669666969 " else "68>6668666868 ";
ATB:= TBL(RAN(13)) -^0;

L:= TRIBMAP(XW, Y);	\get 16 bits
K:= 0;			\bit position (0 = MSB)
while L \#0\ do
	begin
	if L & $8000 then
		begin
		X1:= XW*16 +K -SCRX;	\offset from left edge of screen
		if X1 < 0 then X1:= X1 +GALWIDTH;
		if X1 <= 78 then
		    begin
		    if Y=ENTY & X1>=ENTSX & X1<ENTSX+6 then
			begin		\picked up by Enterprise
			DELETETRIB(XW*16+K, Y);
			TRIBSAVE:= TRIBSAVE +1;
			SHOWSAVED;
			PLAYSNDX(2, 6, TRIBSND(0));
			end
		    else
			begin
			CUR:= Y -SCRY;		\Set cursor position
			if CUR < 0 then CUR:= CUR +GALHEIGHT;
			CUR:= CUR *80 +X1;
			if CUR<SCRSIZE & VIEWPORT(CUR) then
				begin
				SCREENSEG(0, CUR+CUR):= $0F;	\"*"
				CUR:= CUR+CUR+1;
				SCREENSEG(0,CUR):= SCREENSEG(0,CUR)&$F0 ! ATB;
				end;
			end;
		    end;
		end;
	L:= L <<1;
	K:= K +1;
	end;
end;	\DOTRIBS



proc	SCANVIEW;
\Scan what's in viewer for any Klingons, crystals or tribbles.
\Since space is mostly empty, this quickly scans 16 locations at once.
begin
CONDITION:= GREEN;		\assume green until shown otherwise
K:= WRAPWIDTH(SCRX+1) /16;	\left edge of viewer (for speed)
Y:= WRAPHEIGHT(SCRY +1); 	\X, Y = coordinate of 16-bit word
for J:= 1, 13 do		\height of viewer
	begin
	X:= K;
	for I:= 1, 6 do		\width of viewer in 16-bit words +1
		begin
		if TRIBMAP(X, Y) \#0\ then DOTRIBS(X, Y);
		if XTALMAP(X, Y) \#0\ then DOXTALS(X, Y);
		if KLNMAP(X, Y) \#0\ then DOKLNS(X, Y);
		X:= X +1;
		if X >= GAL16WIDTH then X:= 0;
		end;
	Y:= Y +1;
	if Y >= GALHEIGHT then Y:= 0;
	end;
end;	\SCANVIEW

\----------------------------------------------------------------------

proc	RANPT;		\Generate a random point X,Y in the galaxy
int	C, S;
\Inputs: CLUMPS, CLUMPTBL
\Outputs: X, Y
begin
C:= RAN(CLUMPS);

X:= CLUMPTBL(0, C);
Y:= CLUMPTBL(1, C);
S:= CLUMPTBL(2, C);

Y:= Y -S +RAN(S) +RAN(S);
while Y < 0 do Y:= Y +GALHEIGHT;
while Y >= GALHEIGHT do Y:= Y -GALHEIGHT;

S:= S *4;
X:= X -S +RAN(S) +RAN(S);
while X < 0 do X:= X +GALWIDTH;
while X >= GALWIDTH do X:= X -GALWIDTH;
end;	\RANPT



proc	INITCLUMPS(MIN, MAX);	\Initialize the clumps
int	MIN, MAX;
int	I, J;
begin
J:= MAX -MIN;
if J <=0 then J:= 1;
for I:= 0, CLUMPS-1 do
	begin
	CLUMPTBL(0, I):= RAN(GALWIDTH);
	CLUMPTBL(1, I):= RAN(GALHEIGHT);
	CLUMPTBL(2, I):= MIN +RAN(J) +1;
	end;
end;	\INITCLUMPS

\----------------------------------------------------------------------

begin	\PLAYLEVEL
\KLNMAP(GAL16WIDTH, GALHEIGHT);
KLNMAP:= RESERVE(GAL16WIDTH *INTSIZE);
for X:= 0, GAL16WIDTH -1 do
	begin
	KLNMAP(X):= RESERVE(GALHEIGHT *INTSIZE);
	for Y:= 0, GALHEIGHT-1 do KLNMAP(X, Y):= 0;
	end;

KLNCNT:= RESERVE(8 *INTSIZE);
for X:= 0, 7 do
	begin
	KLNCNT(X):= RESERVE(8 *INTSIZE);
	for Y:= 0, 7 do KLNCNT(X, Y):= 0;
	end;

\XTALMAP(GAL16WIDTH, GALHEIGHT);
XTALMAP:= RESERVE(GAL16WIDTH *INTSIZE);
for X:= 0, GAL16WIDTH -1 do
	begin
	XTALMAP(X):= RESERVE(GALHEIGHT *INTSIZE);
	for Y:= 0, GALHEIGHT-1 do XTALMAP(X, Y):= 0;
	end;

XTALCNT:= RESERVE(8 *INTSIZE);
for X:= 0, 7 do
	begin
	XTALCNT(X):= RESERVE(8 *INTSIZE);
	for Y:= 0, 7 do XTALCNT(X, Y):= 0;
	end;

\TRIBMAP(GAL16WIDTH, GALHEIGHT);
TRIBMAP:= RESERVE(GAL16WIDTH *INTSIZE);
for X:= 0, GAL16WIDTH -1 do
	begin
	TRIBMAP(X):= RESERVE(GALHEIGHT *INTSIZE);
	for Y:= 0, GALHEIGHT-1 do TRIBMAP(X, Y):= 0;
	end;

TRIBCNT:= RESERVE(8 *INTSIZE);
for X:= 0, 7 do
	begin
	TRIBCNT(X):= RESERVE(8 *INTSIZE);
	for Y:= 0, 7 do TRIBCNT(X, Y):= 0;
	end;

\Note: MSB is 0
BITMAP:=[$8000, $4000, $2000, $1000, $0800, $0400, $0200, $0100,
	 $0080, $0040, $0020, $0010, $0008, $0004, $0002, $0001];

CLUMPTBL:= RESERVE(3 *INTSIZE);
CLUMPS:= 100;		\Maximum number of clumps ever needed
CLUMPTBL(0):= RESERVE(CLUMPS *INTSIZE);
CLUMPTBL(1):= RESERVE(CLUMPS *INTSIZE);
CLUMPTBL(2):= RESERVE(CLUMPS *INTSIZE);


\Put Enterprise in quadrant furthest away from quadrant containing starbase
ENTQX:= (BASEX/QUADWIDTH + 4) & 7;	ENTQY:= (BASEY/QUADHEIGHT + 4) & 7;
if CHEAT then	\Next door if cheating
	[ENTQX:= (BASEX/QUADWIDTH + 1) & 7;	ENTQY:= BASEY/QUADHEIGHT];
ENTQX0:= ENTQX;				ENTQY0:= ENTQY;
SCRX:= ENTQX *QUADWIDTH - 1;		SCRY:= ENTQY *QUADHEIGHT - 1;
ENTSX:= 40 -3;				ENTSY:= 8;
ENTX:= SCRX +ENTSX;			ENTY:= SCRY +ENTSY;
HEADRIGHT:= true;

CLUMPS:= LEVEL*2;
I:= LEVEL;
J:= LEVEL *2;
INITCLUMPS(I, J);

for K:= 0, KLNS-1 do	\Generate Klingons for other quadrants
	begin
	L:= 20;
	repeat	RANPT;
		L:= L -1;
		X:= X & $FFF8;
		I:= X /16;   J:= REM(0); \Don't put Klingons on top of each other
	until (X/QUADWIDTH#ENTQX ! Y/QUADHEIGHT#ENTQY) &
		((KLNMAP(I, Y) & BITMAP(J)) =0) ! L=0;

	if L # 0 then
		begin
		KLNMAP(I, Y):= KLNMAP(I, Y) ! BITMAP(J);
		I:= X /QUADWIDTH;   J:= Y /QUADHEIGHT;
		KLNCNT(I, J):= KLNCNT(I, J) +1;
		end;
	end;

CLUMPS:= LEVEL *3 /2;
I:= 3;
J:= 4;
INITCLUMPS(I, J);

for K:= 0, XTALS-1 do	\Generate crystals
	begin
	L:= 20;
	repeat	RANPT;
		L:= L -1;
		X:= X & $FFFC;
		I:= X /16;   J:= REM(0);
	until (XTALMAP(I, Y) & BITMAP(J))=0 ! L=0;

	if L # 0 then
		begin
		XTALMAP(I, Y):= XTALMAP(I, Y) ! BITMAP(J);
		I:= X /QUADWIDTH;   J:= Y /QUADHEIGHT;
		XTALCNT(I, J):= XTALCNT(I, J) +1;
		end;
	end;

if LEVEL <= 5 then
	begin	\Put a tribble near the base to help beginners find it
	X:= WRAPWIDTH(BASEX -RAN(25));   Y:= WRAPHEIGHT(BASEY + RAN(7));
	I:= X /16;   J:= REM(0);
	TRIBMAP(I, Y):= TRIBMAP(I, Y) ! BITMAP(J);
	I:= X /QUADWIDTH;   J:= Y /QUADHEIGHT;
	TRIBCNT(I, J):= TRIBCNT(I, J) +1;
	TRIBALL:= TRIBALL +1;
	end;

CLUMPS:= LEVEL/2 +1;
I:= LEVEL/4 +1;
J:= I *2;
INITCLUMPS(I, J);

for K:= 0, TRIBS-1 do	\Generate tribbles
	begin
	L:= 20;
	repeat	RANPT;
		L:= L -1;
		I:= X /16;   J:= REM(0);
	until (TRIBMAP(I, Y) & BITMAP(J))=0 & (XTALMAP(I, Y) & BITMAP(J))=0 ! L=0;

	if L # 0 then
		begin
		TRIBMAP(I, Y):= TRIBMAP(I, Y) ! BITMAP(J);
		I:= X /QUADWIDTH;   J:= Y /QUADHEIGHT;
		TRIBCNT(I, J):= TRIBCNT(I, J) +1;
		TRIBALL:= TRIBALL +1;
		end;
	end;

\Fire up the Enterprise
PLAYSNDX(2, 2, WARNSND(0));
SHOWSTARDATE;

I:= if VID7 then	\Condition
	[$7000+^ , $7000+^, $0700+^G, $0700+^R, $0700+^E, $0700+^E, $0700+^N, $7000+^]
else	[$8800+^ , $8200+^, $2000+^G, $2000+^R, $2000+^E, $2000+^E, $2000+^N, $8200+^];
BLIT(DATASEG, I, SCRSEG, ((STATY+1)*80+STATX-6)*2, 8*2);

SHOWQUAD;	DELAY(9);

I:= FIX(FLOAT(ENERGY) /FLOAT(ENERGYMAX) *FLOAT(SWEEPSIZE-1)) >>4;
PLAYSNDX(2, 2, SWEEPSND(0)+I);
loop	begin
	ENERGY:= ENERGY +ENERGYMAX /16;
	if ENERGY >= ENERGYMAX then quit;
	SHOWENERGY;	DELAY(1);
	end;
ENERGY:= ENERGYMAX;
SHOWENERGY;	DELAY(2);

I:= FIX(FLOAT(SHIELDS) /FLOAT(SHIELDSMAX) *FLOAT(SWEEPSIZE-1)) >>4;
PLAYSNDX(2, 2, SWEEPSND(0)+I);
loop	begin
	SHIELDS:= SHIELDS +SHIELDSMAX /16;
	if SHIELDS >= SHIELDSMAX then quit;
	SHOWSHIELDS;	DELAY(1);
	end;
SHIELDS:= SHIELDSMAX;
SHOWSHIELDS;	DELAY(2);

PLAYSNDX(2, 2, WARNSND(0));
SHOWLEVEL;	SHOWSAVED;	DELAY(9);

SETGALMAP;	DELAY(9);

REVSWEEP;
for I:= 0, 6-1 do
	begin
	CURSOR(DAMAGEX, DAMAGEY+I);
	GRAPHDAMAGE(6);
	DELAY(3);

	J:= FIX(FLOAT(DAMAGEMAX-DAMAGE(I)) /FLOAT(DAMAGEMAX) *FLOAT(SWEEPSIZE-1)) >>4;
	PLAYSNDX(2, 2, SWEEPSND(0)+J);

	loop	begin
		DAMAGE(I):= DAMAGE(I) -1;
		if DAMAGE(I) <= 0 then quit;
		CURSOR(DAMAGEX, DAMAGEY+I);
		GRAPHDAMAGE(DAMAGE(I));
		DELAY(1);
		end;
	DAMAGE(I):= 0;
	CURSOR(DAMAGEX, DAMAGEY+I);
	GRAPHDAMAGE(DAMAGE(I));
	end;
DELAY(6);
REVSWEEP;
DELAY(3);

\Display remaining ships
ATTRIB(if VID7 then $70 else $87);
CURSOR(60, 24);
RAWTEXT(6, "           ");
CURSOR(60, 24);
for I:= 2, SHIPS do RAWTEXT(6, ENTSIMG);
DELAY(9);

ENTDLY:= 0;   KLNDLY:= 0;   TIMEDLY:= TIME;   GASDLY:= 0;   ENERDLY:= TIME;

WAITVB;
for J:= 1, 12 do			\Show what's on the viewer
	begin
	I:= SCRY +J;
	if I >= GALHEIGHT then I:= I -GALHEIGHT;
	BLIT(GALAXY(I), (SCRX+1)*2, SCRSEG, (J*80+1)*2, (80-2)*2);
	end;
\Draw Enterprise
CTXT(ENTSX, ENTSY, if HEADRIGHT then ENTRIMG else ENTLIMG,
	if VGA then ";;;;;; " else "777777 ");
I:= SCRY +13;
if I >= GALHEIGHT then I:= I -GALHEIGHT;
BLIT(GALAXY(I), (SCRX+22)*2, SCRSEG, (13*80+22)*2, (58-22+1)*2);

SCANVIEW;
if VGA then FADE(false, 0);
PLAYSNDX(2, 2, PINGSND(0));


\Main timing loop
loop	begin
	J:= ENTSY;			\Show what's on the viewer
	I:= SCRY +J;
	if I >= GALHEIGHT then I:= I -GALHEIGHT;
	WAITVB;
	if RAN(6) < DAMAGE(DAMVIEWER) then
	     BLIT(RAN(1000), 0, SCRSEG, (J*80+1)*2, (80-2)*2)
	else BLIT(GALAXY(I), (SCRX+1)*2, SCRSEG, (J*80+1)*2, (80-2)*2);

	\Draw Enterprise
	CTXT(ENTSX, ENTSY, if HEADRIGHT then ENTRIMG else ENTLIMG,
		if VGA then ";;;;;; " else "777777 ");

	for J:= 1, 12 do		\Show what's on the viewer
	    if J # ENTSY then
		begin
		I:= SCRY +J;
		if I >= GALHEIGHT then I:= I -GALHEIGHT;
		if RAN(6) < DAMAGE(DAMVIEWER) then
		     BLIT(RAN(1000), 0, SCRSEG, (J*80+1)*2, 156)
		else BLIT(GALAXY(I), (SCRX+1)*2, SCRSEG, (J*80+1)*2, 156);
		end;

	I:= SCRY +13;
	if I >= GALHEIGHT then I:= I -GALHEIGHT;
	if RAN(6) < DAMAGE(DAMVIEWER) then
	     BLIT(RAN(1000), 0, SCRSEG, (13*80+22)*2, (58-22+1)*2)
	else BLIT(GALAXY(I), (SCRX+22)*2, SCRSEG, (13*80+22)*2, (58-22+1)*2);

	SCANVIEW;

	if BASEOPEN then
	    if ENTY = BASEY+1 & ABS(ENTX-(BASEX+3)) <= 1 then
	    	begin
		DOCK;
		WON:= true;
		quit;
		end;

	\Bit:	6	5	4	3	2	1	0	
	\Key:	space	tab	escape	right	left	down	up
	I:= false;
	repeat	KEYS:= GETKEYS;
		if KEYS = $0030 then
			[CHEAT:= true;   SOUND(1, 7, 7000)];
		if KEYS & $0020 then I:= true;
	until (KEYS & $0020) = 0;

	if I then
		begin
		SOUNDON:= ~SOUNDON;
		if SOUNDON then SETSTV else RESSTV;
		end;

	if KEYS & $0080 then KEYS:= KEYS ! $0005;	\Keypad controls
	if KEYS & $0100 then KEYS:= KEYS ! $0009;
	if KEYS & $0200 then KEYS:= KEYS ! $0002;
	if KEYS & $0400 then KEYS:= KEYS ! $0006;
	if KEYS & $0800 then KEYS:= KEYS ! $000A;
	if KEYS & $1000 then KEYS:= KEYS ! $0040;	\Convert ctrl to space

	DOPHASER;		\(DOPHASER needs KEYS)
	DOBASE;
	DOCOMPASS;
	DOGALMAP;

	if KEYS & $10 then				\Auto-destruct
		begin
		HIT_ENT;
		ENTDEAD;
		quit;
		end;

	if DAMAGE(DAMENGINES)\#0\ then
		KEYS:= if DAMAGE(DAMENGINES) >= 4 then KEYS ! RAN($10)
			else KEYS & RAN($10);

	if KEYS & $03 then		\Up or down key is pressed
		begin
		ENTDLY:= ENTDLY +1;	\Move up & down less often than right & left
		if ENTDLY >= (if KEYS & $0C \diagonal\ then 4 else 3) then
			begin
			ENTDLY:= 0;
			if KEYS & 1 then ENTY:= ENTY -1;	\Up
			if KEYS & 2 then ENTY:= ENTY +1;	\Down
			if KEYS & 4 then		\This avoids the jitters
				[ENTX:= ENTX -3;   HEADRIGHT:= false];	\Left
			if KEYS & 8 then
				[ENTX:= ENTX +3;   HEADRIGHT:= true];	\Right
			end;
		end
	else	begin			\No up or down key is pressed
		if KEYS & 8 then
			[ENTX:= ENTX +1;   HEADRIGHT:= true];	\Right
		if KEYS & 4 then
			[ENTX:= ENTX -1;   HEADRIGHT:= false];	\Left
		end;

	if KEYS & $0F then		\Finger is on the gas button
		begin
		ENTX:= WRAPWIDTH(ENTX);
		ENTY:= WRAPHEIGHT(ENTY);
		ENTQX0:= ENTQX;   ENTQY0:= ENTQY;
		ENTQX:= ENTX /QUADWIDTH;   ENTQY:= ENTY /QUADHEIGHT;
		if ENTQX#ENTQX0 ! ENTQY#ENTQY0 then SHOWQUAD;

		GASDLY:= GASDLY +1;
		if GASDLY >= 4 then
			begin
			GASDLY:= 0;
			BURNENERGY(1);
			end;
		end;

	\If Enterprise is moving off of screen then move the screen.
	\ Note that Enterprise might have moved right or left more than one char
	if CMPWIDTH(ENTX, WRAPWIDTH(SCRX+30)) <0 then SCRX:= WRAPWIDTH(ENTX -30);
	if CMPWIDTH(ENTX, WRAPWIDTH(SCRX+50-6)) >0 then SCRX:= WRAPWIDTH(ENTX -50+6);
	if ENTY = WRAPHEIGHT(SCRY+5) then SCRY:= WRAPHEIGHT(SCRY -1);
	if ENTY = WRAPHEIGHT(SCRY+9) then SCRY:= WRAPHEIGHT(SCRY +1);

	ENTSX:= WRAPWIDTH(ENTX -SCRX);
	ENTSY:= WRAPHEIGHT(ENTY -SCRY);

	if TIME = ENERDLY  then	\Since energy can change constantly only
		begin		\ update the display about twice per second
		ENERDLY:= TIME +9;	\Must be odd increment
		if TIME & $0001 then BURNENERGY(1);	\Life support, etc.
		SHOWENERGY;
		end;

	if TIME = TIMEDLY then
		begin
		TIMEDLY:= TIME +109;	\One stardate per minute
		STARDATE:= STARDATE +1;
		SHOWSTARDATE;

		K:= 0;   I:= 0;		\Repair item with most damage
		for J:= 0, 6-1 do
			if DAMAGE(J) > K then [K:= DAMAGE(J);   I:= J];
		if DAMAGE(I) > 0 then
			begin
			DAMAGE(I):= DAMAGE(I) -1;
			CURSOR(DAMAGEX, DAMAGEY+I);
			GRAPHDAMAGE(DAMAGE(I));
			end;
		end;

	SHOWCOND;

	if not CHEAT then
		begin
		if SHIELDS <= 0 then
			begin
			WON:= false;
			ENTDEAD;
			quit;
			end;

		if ENERGY <= 0 then
			begin
			WON:= false;
			CURSOR(ENTSX-3, ENTSY);
			ATTRIB(if VGA then $0B else $07);
			TEXT(6, "(out of gas)");
			SHOWENERGY;
			DELAY(40);
			quit;
			end;
		end;

	TIME:= TIME +1;
	\Wait for system clock to tick (BIOS $1A causes distortion of sound)
	J:= PEEK(0, $046C);
	repeat until J # PEEK(0, $046C);
	end;

if ~VGA ! DAMAGE(DAMVIEWER)\#0\ then CLEARVIEW;
if VGA then FADE(true, 1);
end;	\PLAYLEVEL

\=========================== CAPTAIN'S LOG ============================

proc	OPENLOG;	\Begin Captain's Log, set LOGLINE to top of map
int	I, J;
begin
CURSOR(GALMAPX, GALMAPY+8);
RAWTEXT(1, " C A P T A I N S   L O G  ");

ATTRIB($02);
for J:= GALMAPY, GALMAPY+7 do		\Clear log screen
	begin
	CURSOR(GALMAPX+1, J);
	for I:= 0, 30 do CHOUT(6, SP);
	end;

SHOWCUR(true);
OPENI(0);   OPENI(1);
LOGLINE:= GALMAPY;		\Top line
CURSOR(GALMAPX+1, LOGLINE);
end;	\OPENLOG



proc	NEWLINE;	\Start new line
int	OFF, I;
begin
LOGLINE:= LOGLINE +1;
if LOGLINE > GALMAPY+7 then
	begin
	LOGLINE:= GALMAPY +7;
	OFF:= (GALMAPY*80 +GALMAPX);
	for I:= 1, 7 do
		BLIT(SCRSEG, (OFF+I*80)*2, SCRSEG, (OFF+(I-1)*80)*2, 32*2);
	CURSOR(GALMAPX+1, LOGLINE);
	for I:= 0, 30 do CHOUT(1, SP);
	end;
CURSOR(GALMAPX+1, LOGLINE);
end;	\NEWLINE



proc	SHOWLOG;	\Display text in device 8 on the Captain's Log
int	CH;
begin
OPENI(8);
loop	begin
	if CHKKEY then quit;
	repeat CH:= CHIN(8) until CH # LF;
	if CH = $A0 then quit;
	if CH = CR then NEWLINE else CHOUT(1, CH);
	if CH # SP then DELAY(2);
	end;	\loop
end;	\SHOWLOG



proc	WRAPLOG;	\Display text in device 8 on the Captain's Log
def	LINLEN= 31;	\Max characters per line for auto wrap
int	COL, HYPHEN, LEN, CH, J;
addr	WORD;
begin
WORD:= RESERVE(40);
OPENI(8);
COL:= 0;   HYPHEN:= false;
if ~CHKKEY then
   loop	begin
	LEN:= 0;	\Read in a word or syllable and get its length
	repeat CH:= CHIN(8) until CH # LF;
	if CH = $A0 then quit;
	loop	begin
		WORD(LEN):= CH;		\A word can be a CR or space
		LEN:= LEN +1;		\Length of word or syllable
		repeat CH:= CHIN(8) until CH # LF;
		if CH=SP ! CH=$A0 ! CH=CR then quit;
		if CH = ^` then [LEN:= LEN +1;   quit];   \Might need hyphen
		end;

	if LEN +COL +(if HYPHEN then 0 else 1) > LINLEN then
		begin		\Start next line
		if HYPHEN then CHOUT(1, ^-);
		NEWLINE;
		COL:= 0;
		end;
	if COL#0 & ~HYPHEN then
		begin
		CHOUT(1, SP);
		COL:= COL +1;
		end;
	HYPHEN:= CH = ^`;	\Set flag if this syllable might need a hyphen
	if HYPHEN then LEN:= LEN -1;

	for J:= 0, LEN-1 do	\Output word
		begin
		if WORD(J) = CR then [NEWLINE;   NEWLINE;   COL:= 0]
		else [CHOUT(1, WORD(J));   COL:= COL +1];
		DELAY(2);
		if CHKKEY then quit;
		end;

	if CH = $A0 then quit;
	end;	\loop
end;	\WRAPLOG



proc	CLOSELOG;
begin
repeat until CHKKEY;
OPENI(0);   OPENI(1);

SHOWCUR(false);

CURSOR(GALMAPX, GALMAPY+8);
RAWTEXT(1, " G A L A C T I C   M A P  ");
end;	\CLOSELOG

\----------------------------------------------------------------------

proc	EPILOGUE;	\Write epilogue out to device 8
\WARNING: Keep messages < 256 characters so they fit in device 8's buffer.
int	A;	\Tribbles abandoned
begin
OPENO(8);
TEXT(8, "       STARDATE: ");
INTOUT(8, STARDATE/10);   CHOUT(8, ^.);   INTOUT(8, REM(0));
TEXT(8, "

");
case of
TRIBSAVE >=TRIBALL/2 & TRIBSAVE >1:
	begin
	TEXT(8, "COM`MEN`DA`TIONS ARE DUE. ");
	INTOUT(8, TRIBSAVE);
	TEXT(8, " TRIB`BLES HAVE BEEN SAVED!");
	A:= TRIBALL -TRIBSAVE -TRIBPHASER -TRIBKILL;
	if A > 1 then
		begin
		TEXT(8, " REC`ORDS INDI`CATE HOW`EVER, THAT ");
		INTOUT(8, A);
		TEXT(8, " WERE LEFT BE`HIND");
		if TRIBKILL > 1 then
			begin
			TEXT(8, ", AND ");
			INTOUT(8, TRIBKILL);
			TEXT(8, " WERE MUR`DER`ED BY KLING`ONS.");
			if KLNKILL > 1 then
				begin
				TEXT(8, " IT IS ONLY FIT`TING THAT ");
				INTOUT(8, KLNKILL);
				TEXT(8, " OF THEIR SHIPS WERE DE`STROY`ED.");
				end;
			end
		else	CHOUT(8, ^.);
		end;
	end;

5*TRIBPHASER >TRIBSAVE & TRIBPHASER >1:
	begin
	TEXT(8, "THE HEART`LESS SLAUGH`TER OF ");
	INTOUT(8, TRIBPHASER);
	TEXT(8, " TRIB`BLES, NO DOUBT, WILL BE PER`MA`NENT`LY RE`CORD`ED BY STAR`FLEET.");
	if KLNKILL > 1 then
		begin
		TEXT(8, " AT LEAST THEY WILL ALSO RE`CORD THE FATE OF ");
		INTOUT(8, KLNKILL);
		TEXT(8, " KLING`ON WAR`SHIPS.");
		end;
	end

other	begin
	TEXT(8, "RE`GRET`TABLY, ");
	if TRIBSAVE = 0 then
		TEXT(8, "NO TRIB`BLES WERE RES`CUED.")
	else	begin
		if TRIBSAVE = 1 then
			TEXT(8, "ONLY 1 TRIB`BLE WAS RES`CUED")
		else	begin
			TEXT(8, "ONLY ");
			INTOUT(8, TRIBSAVE);
			TEXT(8, " TRIB`BLES WERE RES`CUED");
			end;
		if TRIBALL -TRIBSAVE > 1 then
			begin
			TEXT(8, ", LEAV`ING ");
			INTOUT(8, TRIBALL -TRIBSAVE);
			TEXT(8, " UN`AC`COUNT`ED FOR");
			end;
		CHOUT(8, ^.);
		end;

	if KLNKILL > 3 then
		begin
		TEXT(8, " THIS MAY BE PAR`TIAL`LY OFF`SET, HOW`EVER, BY THE DE`STRUC`TION OF ");
		INTOUT(8, KLNKILL);
		TEXT(8, " KLING`ON BAT`TLE CRUIS`ERS.");
		end;
	end;

CHOUT(8, $A0);
OPENLOG;
WRAPLOG;

OPENO(8);
TEXT(8, "

HIT SPACEBAR TO TERMINATE.");
CHOUT(8, $A0);
WRAPLOG;
CLOSELOG;
end;	\EPILOGUE



proc	INSTRUCTIONS;
begin
OPENO(8);
TEXT(8, "      STARDATE: 2300.0

THE ENTERPRISE IS ON A MISSION
TO RESCUE TRIBBLES FROM A BAND
OF MURDEROUS KLINGONS.

");   CHOUT(8, $A0);
OPENLOG;
SHOWLOG;

OPENO(8);
TEXT(8, "LONG-RANGE SENSORS FEED THE
SHIP'S COMPUTER, WHICH GENER-
ATES THE GALACTIC MAP.  EACH
QUADRANT SHOWS THREE DIGITS
INDICATING HOW MANY KLINGONS,
DILITHIUM CRYSTALS, AND TRIB-
BLES THERE ARE.

");   CHOUT(8, $A0);
SHOWLOG;

OPENO(8);
TEXT(8, "THE NAVIGATIONAL COMPASS
SHOWS THE HEADING TO THE
STARBASE.

CONSOLE CONTROLS
  ARROW KEYS:  ENGINE CONTROL
  SPACEBAR:    FIRE PHASERS
  ESC KEY:     AUTO-DESTRUCT


HIT SPACEBAR TO BEGIN MISSION.");
CHOUT(8, $A0);
SHOWLOG;
CLOSELOG;
end;	\INSTRUCTIONS

\======================================================================

proc	SETSTARS;	\Set up the stars
int	I, J, N, TBL, TBL2;
addr	STR;
begin
for J:= 0, GALHEIGHT-1 do
    for I:= 0, GALWIDTH-1 do
	GALAXY(J, I):= 0;

\Brightnesses
TBL:= if VGA then
	[$0B00, $0B00, $0900, $0900, $0900, $0900, $0900, $0F00]
else	[$0700, $0700, $0800, $0800, $0800, $0800, $0800, $0F00];

\Fadeable colors (and not green)
TBL2:=	[$0100, $0300, $0500, $0600, $0900, $0B00, $0C00, $0D00, $0F00];

for I:= 1, 5000 do
	begin
	N:= TBL(I&$0007);
	if RAN(100)<3 & ~VID7 then
		N:= TBL2(RAN(9));
	GALAXY(RAN(GALHEIGHT), RAN(GALWIDTH)):= N ! ^;
	end;

for I:= 1, 1400 do
	begin
	N:= TBL(I&$0007);
	if VID7 then N:= $0700
	else if RAN(100) <7 then
		N:= TBL2(RAN(9));
	GALAXY(RAN(GALHEIGHT), RAN(GALWIDTH)):= N ! ^.;

	N:= if VID7 then $0700 else if VGA then $0900 else $0800;
	if RAN(100) <7 then N:= if VID7 then $0F00 else TBL2(RAN(9));
	GALAXY(RAN(GALHEIGHT), RAN(GALWIDTH)):= N ! ^;
	end;

for I:= 1, 50 do
	begin
	N:= if VGA then $0B07 else $0707;
	if RAN(100) <4 then N:= $0C07;		\Red giant
	GALAXY(RAN(GALHEIGHT), RAN(GALWIDTH)):= N;
	end;

for I:= 1, 1 do
	begin
	N:= (if VGA then $0B00 else $0700) ! ^:;
	GALAXY(RAN(GALHEIGHT), RAN(GALWIDTH)):= N;
	end;

if GRAPHICS & ~VID7 then
	begin					\Pinwheel galaxy
	I:= GALWIDTH/4+6;	J:= GALHEIGHT/4*3+9;
	STR:= " ";		for N:= 0, 4 do GALAXY(J, I+N):= $0900 !STR(N);
	STR:= " ";		for N:= 0, 3 do GALAXY(J+1, I+N):= $0900 !STR(N);
	STR:= " ";		for N:= 0, 2 do GALAXY(J+2, I+N):= $0900 !STR(N);
						\Planet
	I:= GALWIDTH/4*3-4;	J:= GALHEIGHT/4-6;
	STR:= " ";	for N:= 0, 7 do GALAXY(J, I+N):= $0B00 !STR(N);
	STR:= "      ";	for N:= 0, 7 do GALAXY(J+1, I+N):= $0B00 !STR(N);
	STR:= "       ";	for N:= 0, 7 do GALAXY(J+2, I+N):= $0B00 !STR(N);
	STR:= "         ";	for N:= 2, 6 do GALAXY(J+3, I+N):= $0B00 !STR(N);
	end;
end;	\SETSTARS



proc	SETGRID;	\Draw grid corners for quadrants
int	ATB, I, J, K, II, JJ;
begin
ATB:= if VID7 then $0700 else if VGA then $0900 else $0800;
for J:= 0, 7 do
	begin
	JJ:= J *QUADHEIGHT;
	for I:= 0, 7 do
		begin
		II:= I *QUADWIDTH;
		GALAXY(JJ, II):= ATB !$C5;			\
		GALAXY(JJ+1, II):= ATB !$B3;			\
		GALAXY(JJ+QUADHEIGHT-1, II):= ATB !$B3;		\
		for K:= 0, 2 do
			begin
			GALAXY(JJ, II+1+K):= ATB !$C4;		 \
			GALAXY(JJ, II+QUADWIDTH-3+K):= ATB !$C4; \
			end;
		end;
	end;
end;	\SETGRID



proc	SETBASE;	\Set up the starbase at BASEX, BASEY
int	ATB, I, J, L, CH;
addr	K;
begin
ATB:= if VID7 then $7000 else if VGA then $9000 else $8000;
K:=
"[#][V][A]
           
[<][>] ";

for J:= 0, 2 do
	begin
	for I:= 0, 12 do
		begin
		case K(I) of
		  ^V:	CH:= $19\$1F\;
		  ^A:	CH:= $18\$1E\;
		  ^<:	CH:= $1B;
		  ^>:	CH:= $1A
		other	CH:= K(I);

		GALAXY(BASEY+J, BASEX+I):= ATB ! CH;
		end;
	K:= K + 13 +2;
	end;

\Show next level
L:= LEVEL +1;
if L < 10 then
	GALAXY(BASEY, BASEX+2):= ATB ! L ! ^0
else	begin
	I:= L /10;
	L:= REM(0);
	GALAXY(BASEY, BASEX+2):= ATB ! I ! ^0;
	GALAXY(BASEY, BASEX+3):= ATB ! L ! ^0;
	GALAXY(BASEY, BASEX+4):= ATB ! ^];
	end;

BASEOPEN:= false;
end;	\SETBASE



proc	SETGRAPH;	\Set up character graphics
begin
CPUREG(0):= $1130;	\Is there a settable font (EGA/VGA/MCGA)?
CPUREG(1):= $0300;	\Height of characters is >= 14 <=20 if so
CPUREG(2):= $0000;
SOFTINT($10);
GRAPHICS:= CPUREG(2)>=14 & CPUREG(2)<=20;

\Redefine fonts for block characters (these handle the 9th bit problem)
KLNIMG:= "=Ϳ ";
ENTRIMG:= "_-- ";
ENTLIMG:= "--_ ";
ENTSIMG:= "_+   ";
PHASIMG1:= "\ ";
PHASIMG2:= "/ ";
BARIMG:= ^;
NCCIMG:= "NCC-1701 ";

if GRAPHICS then
	begin		\Redefine fonts
	KLNIMG:= " ";
	ENTRIMG:= " ";
	ENTLIMG:= " ";
	ENTSIMG:= "   ";
	PHASIMG1:= " ";
	PHASIMG2:= " ";
	BARIMG:= ^;
	NCCIMG:= "  ";

	CPUREG(0):= $1100;		\Klingon
	CPUREG(1):= $0E00;
	CPUREG(2):= $0005;
	CPUREG(3):= $00CE;	\TOP				BOTTOM
	CPUREG(6):= [	$0000, $0000, $0000, $7F1F, $C0E0, $4040, $2020,
			$0000, $0000, $F707, $FDF8, $001E, $0000, $0000,
			$0000, $7C10, $FFFF, $FFFE, $38FE, $0000, $0000,
			$0000, $0000, $DEC0, $7F3F, $00F0, $0000, $0000,
			$0000, $0000, $0000, $FCF0, $060E, $0404, $0808];
	CPUREG(11):= DATASEG;
	SOFTINT($10);

	CPUREG(0):= $1100;		\Right Enterprise
	CPUREG(1):= $0E00;
	CPUREG(2):= $0006;
	CPUREG(3):= $00D3;
	CPUREG(6):= [	$0000, $FF00, $007F, $0000, $0000, $0000, $0000,
			$0000, $FF00, $00FF, $0000, $3F00, $031F, $0000,
			$0000, $FF00, $00FF, $6060, $FF60, $FFFF, $0000,
			$0000, $8000, $3F00, $381C, $FC70, $FCFE, $0000,
			$0000, $0E04, $FF7F, $0E7F, $0000, $0000, $0000,
			$0000, $0000, $FFC0, $00C0, $0000, $0000, $0000];
	CPUREG(11):= DATASEG;
	SOFTINT($10);

	CPUREG(0):= $1100;		\Left Enterprise
	CPUREG(1):= $0E00;
	CPUREG(2):= $0002;
	CPUREG(3):= $00C6;
	CPUREG(6):= [	$0000, $0000, $FF03, $0003, $0000, $0000, $0000,
			$0000, $7020, $FFFE, $70FE, $0000, $0000, $0000];
	CPUREG(11):= DATASEG;
	SOFTINT($10);

	CPUREG(0):= $1100;
	CPUREG(1):= $0E00;
	CPUREG(2):= $0003;
	CPUREG(3):= $00CA;
	CPUREG(6):= [	$0000, $0100, $FC00, $1C38, $3F0E, $3F7F, $0000,
			$0000, $FF00, $00FF, $0606, $FF06, $FFFF, $0000,
			$0000, $FF00, $00FF, $0000, $FC00, $C0F8, $0000];
	CPUREG(11):= DATASEG;
	SOFTINT($10);

	CPUREG(0):= $1100;
	CPUREG(1):= $0E00;
	CPUREG(2):= $0001;
	CPUREG(3):= $00E0;
	CPUREG(6):= [	$0000, $FF00, $00FE, $0000, $0000, $0000, $0000];
	CPUREG(11):= DATASEG;
	SOFTINT($10);

	CPUREG(0):= $1100;
	CPUREG(1):= $0E00;
	CPUREG(2):= $0007;
	CPUREG(3):= $0080;		\Little Enterprise
	CPUREG(6):= [	$0000, $0000, $00FF, $0700, $0003, $0000, $0000,
			$0000, $0000, $43F8, $FC46, $00FE, $0000, $0000,
			$0000, $0000, $FF38, $0038, $0000, $0000, $0000,
					\Phaser 1 \
			$60C0, $1830, $060C, $0103, $0000, $0000, $0000,
			$0000, $0000, $0000, $C000, $3060, $0C18, $0306,
					\Phaser 2 /
			$0000, $0000, $0000, $0301, $0C06, $3018, $C060,
			$0603, $180C, $6030, $00C0, $0000, $0000, $0000];
	CPUREG(11):= DATASEG;
	SOFTINT($10);

	CPUREG(0):= $1100;
	CPUREG(1):= $0E00;
	CPUREG(2):= $0001;
	CPUREG(3):= $00C1;		\Bar graph reticule  
	CPUREG(6):= if VID7 then
			[$0000, $0000, $0101, $0101, $0000, $0000, $0000]
		else 	[$0000, $FFFF, $FEFE, $FEFE, $FFFF, $0000, $0000];
	CPUREG(11):= DATASEG;
	SOFTINT($10);

	CPUREG(0):= $1100;
	CPUREG(1):= $0E00;
	CPUREG(2):= $0001;
	CPUREG(3):= $00C3;		\Bar graph  
	CPUREG(6):=	[$0000, $FFFF, $FEFE, $FEFE, $FFFF, $0000, $0000];
	CPUREG(11):= DATASEG;
	SOFTINT($10);

	CPUREG(0):= $1100;
	CPUREG(1):= $0E00;
	CPUREG(2):= $0007;
	CPUREG(3):= $0088;		\NCC-1701
	CPUREG(6):= [	 $0000, $0000, $0000, $0000, $0C08, $090A, $0008,
			 $0000, $0000, $0000, $0000, $918E, $9190, $008E,
			 $0000, $0000, $0000, $0000, $221C, $2220, $001C,
			 $0000, $0000, $0000, $0000, $0001, $0078, $0001,
			 $0000, $0000, $0000, $0000, $818F, $8482, $00C4,
			 $0000, $0000, $0000, $0000, $221C, $2222, $001C,
			 $0000, $0000, $0000, $0000, $2060, $2020, $0070];
	CPUREG(11):= DATASEG;
	SOFTINT($10);

	CPUREG(0):= $1100;
	CPUREG(1):= $0E00;
	CPUREG(2):= $000A;
	CPUREG(3):= $0090;		\7-segment digits
	CPUREG(6):= [	$0000, $423C, $4242, $0042, $4242, $4242, $003C,
			$0000, $0808, $0808, $0008, $0808, $0808, $0008,
			$0000, $023C, $0202, $3C02, $4040, $4040, $003C,
			$0000, $023C, $0202, $3C02, $0202, $0202, $003C,
			$0000, $4242, $4242, $3C42, $0202, $0202, $0002,
			$0000, $403C, $4040, $3C40, $0202, $0202, $003C,
			$0000, $403C, $4040, $3C40, $4242, $4242, $003C,
			$0000, $023C, $0202, $0002, $0202, $0202, $0002,
			$0000, $423C, $4242, $3C42, $4242, $4242, $003C,
			$0000, $423C, $4242, $3C42, $0202, $0202, $003C];
	CPUREG(11):= DATASEG;
	SOFTINT($10);

	CPUREG(0):= $1100;		\Stars
	CPUREG(1):= $0E00;
	CPUREG(2):= $0002;
	CPUREG(3):= $00F9;
	CPUREG(6):= [	$0000, $0001, $0000, $0000, $0000, $0000, $0000,
			$0000, $0000, $0000, $0400, $0000, $0000, $0000];
	CPUREG(11):= DATASEG;
	SOFTINT($10);

	CPUREG(0):= $1100;		\Up and down arrows
	CPUREG(1):= $0E00;
	CPUREG(2):= $0002;
	CPUREG(3):= $0018;
	CPUREG(6):= [	$0000, $1800, $7E3C, $1818, $1818, $0000, $0000,
			$0000, $1800, $1818, $7E18, $183C, $0000, $0000];
	CPUREG(11):= DATASEG;
	SOFTINT($10);

	CPUREG(0):= $1100;
	CPUREG(1):= $0E00;
	CPUREG(2):= $000B;
	CPUREG(3):= $00E1;		\Pinwheel galaxy
	CPUREG(6):= [	$0000, $0000, $0000, $0200, $0000, $0200, $0104, \E1
			$0000, $0000, $0001, $0010, $0000, $0080, $0401,
			$802C, $0009, $0000, $2000, $4000, $8040, $2148,
			$0000, $00E0, $0020, $0000, $2002, $0080, $0000,
			$0000, $0000, $0000, $4000, $0040, $00A0, $70A0,

			$0008, $0000, $0B00, $0020, $4408, $0210, $3080, \E6
			$3840, $43DC, $4787, $1E1F, $1C0E, $3F1E, $2B19,
			$0A07, $0801, $4309, $C082, $1080, $B882, $4010,
			$8000, $0100, $0181, $0104, $010E, $8020, $8000,

			$2008, $9004, $0700, $0000, $0202, $0000, $0000, \EA
			$0400, $0000, $0000, $0400, $70E2, $0048, $0000];
	CPUREG(11):= DATASEG;
	SOFTINT($10);

	CPUREG(0):= $1100;
	CPUREG(1):= $0E00;
	CPUREG(2):= $0001;
	CPUREG(3):= $00ED;		\Pinwheel galaxy (cont.)
	CPUREG(6):= [	$0000, $0100, $0800, $4010, $0000, $0000, $0000];
	CPUREG(11):= DATASEG;
	SOFTINT($10);

	CPUREG(0):= $1100;
	CPUREG(1):= $0E00;
	CPUREG(2):= $000D;
	CPUREG(3):= $00A0;		\Planet
	CPUREG(6):= [	$0000, $0000, $0000, $0000, $0100, $0603, $0C0E, \A0
			$0000, $0000, $0F02, $111B, $DBF0, $1A2A, $2492,
			$0300, $FF16, $F013, $8E80, $0140, $0650, $5101,
			$A414, $A83F, $2002, $8842, $3100, $0201, $0068,
			$B6FD, $08ED, $00BD, $8031, $0682, $0000, $0000,
			$D800, $28FE, $08A2, $2004, $0005, $0000, $0000,
			$0000, $4000, $0220, $2029, $0082, $0000, $0000,
			$0000, $0000, $0000, $8000, $2000, $0800, $0400,

			$390F, $3530, $5A34, $CA64, $88DC, $9470, $F9E8, \A8
			$8148, $0001, $8010, $0122, $8200, $0422, $6008,
			$0200, $0008, $0048, $0010, $0000, $0000, $0000,

			$2CE6, $3128, $2806, $0A0C, $0204, $0000, $0000, \AB
			$0000, $4000, $0000, $8080, $8000, $0040, $0000];
	CPUREG(11):= DATASEG;
	SOFTINT($10);

	CPUREG(0):= $1103;		\In case it's MCGA
	CPUREG(1):= $0000;
	SOFTINT($10);
	end;
end;	\SETGRAPH



proc	SETEXPLOSION;		\Convert explosion images
int	F,	\Frame
	L,	\Line
	I;	\Scratch
addr	S;	\String
begin
EXPLOSION:=[[
       [" ",			\FRAME
	"      Ϳ ",		\0
	" "],			\Text
       [" ",
	"      88888 ",
	" "]],			\Attributes

      [[" ",
	"      +Ϳ ",		\1
	" "],
       [" ",
	"      ?8888 ",
	" "]],

      [[" ",
	"      *Ϳ ",		\2
	" "],
       [" ",
	"      ?7888 ",
	" "]],

      [[" ",
	"      Ϳ ",		\3
	" "],
       [" ",
	"      ?7888 ",
	" "]],

      [["     . , ",
	"      >Ϳ ",		\4
	"     ' ` "],
       ["     ? ? ",
	"      ?7888 ",
	"     ? ? "]],

      [["     ., / ",
	"    -|Ϳ ",		\5
	"    /'|` "],
       ["     >? ? ",
	"    ????788 ",
	"    ?>?> "]],

      [["    \.| / ",
	"   --=;|*Ϳ ",		\6
	"    /'| \ "],
       ["    ?>? ? ",
	"   >?>7?788 ",
	"    ?7> ? "]],

      [["    \ |,. ",
	"   -= , *Ϳ ",		\7
	"    /  `, "],
       ["    ? ??> ",
	"   >? > ?78 ",
	"    7  <? "]],

      [["    . |,. ",
	"   -  , *> ",		\8
	"    /  ` "],
       ["    > ><< ",
	"   >  ? ?78 ",
	"    ?  <? "]],

      [["   .    ,. ",
	"   - ,  *>, ",	\9
	"    '   `  "],
       ["   >    << ",
	"   ? >?  77< ",
	"    >   > < "]],

      [["        ; ",
	"     .  >;, ",	\10
	"   '     ` "],
       ["   <     7 ",
	"     ?7  7>< ",
	"   <     < "]],

      [["        , . ",
	"     .   | ; ",	\11
	"  '       `' "],
       ["  <      ? > ",
	"    < ?   7 ? ",
	"  <       >> "]],

      [["         .. ",
	"     .   > : ",	\12
	"  '       ` "],
       ["  <       ?< ",
	"    < <   7 ? ",
	"  <       7< "]],

      [["          . . ",
	"     .    |  . ",	\13
	" '          "],
       [" <         ? < ",
	"   ?  <    7  < ",
	" <         < "]],

      [["          . . ",
	"          |  . ",	\14
	"           "],
       [" <         7 < ",
	"   <       8  < ",
	" <         < "]],

      [["           .  . ",
	"           ;   . ",	\15
	"            "],
       ["<           7  < ",
	"  <         8   < ",
	"<           < "]],

      [["            . ",
	"           ;   . ",	\16
	" "],
       ["            < ",
	"  <         ?   < ",
	"< "]],

      [[" ",
	"           ,   . ",	\17
	" "],
       [" ",
	"  <         <   < ",
	"< "]],

      [[" ",
	"           ,    . ",	\18
	" "],
       [" ",
	"  <         <    < ",
	" "]],

      [[" ",
	"            .     . ",	\19
	" "],
       [" ",
	"            <     < ",
	" "]],

      [[" ",
	"            . ",	\20
	" "],
       [" ",
	"            < ",
	" "]],


      [[" ",			\2nd explosion pattern
	"      Ϳ ",		\0
	" "],			\Text
       [" ",
	"      88888 ",
	" "]],			\Attributes

      [[" ",
	"      + ",		\1
	" "],
       [" ",
	"      888?8 ",
	" "]],

      [[" ",
	"      * ",		\2
	" "],
       [" ",
	"      888?7 ",
	" "]],

      [[" ",
	"      鎿 ",		\3
	" "],
       [" ",
	"      888?7 ",
	" "]],

      [["        . , ",
	"      > ",		\4
	"        ' ` "],
       ["        ? ? ",
	"      ?78?7 ",
	"        ? ? "]],

      [["        `  / ",
	"      |  ",	\5
	"       /'|` "],
       ["        ?  ? ",
	"      87?? 7 ",
	"       ?>?? "]],

      [["       \`| / ",
	"      =>;=  ",	\6
	"       /'| \ "],
       ["       ?>? ? ",
	"      87??? 7 ",
	"       ?>? ? "]],

      [["       \ |,. ",
	"      * , =  ",	\7
	"       /  `, "],
       ["       ? ??> ",
	"      7? ? ? 7 ",
	"       ?  ?> "]],

      [["       ` |,. ",
	"      <* ,  -  ",	\8
	"       /  `, "],
       ["       > ?<> ",
	"      7? ?  ? 7 ",
	"       ?  7> "]],

      [["         ' ' ",
	"       ., . - ",	\9
	"       ,  .  "],
       ["         ? > ",
	"     7 ? ?? ? 7 ",
	"       ?  7  "]],

      [["        ;   . ",
	"      =  .  .. ",	\10
	"      .    . "],
       ["        >   > ",
	"    7  ?  ?  ?7 ",
	"      >    > "]],

      [["       ,    . ",
	"    -  -  .  .. ",	\11
	"      .    . "],
       ["       >    > ",
	"    ?  7  ?  ?7 ",
	"      <    > "]],

      [["       .    . ",
	"    .  -     . ",	\12
	"     .     . "],
       ["       >    < ",
	"    ?  7     7 ",
	"     <     > "]],

      [["      . ",
	"   .   ,     .  ",	\13
	"     .     . "],
       ["      > ",
	"   7   7     7 ",
	"     <     > "]],

      [["      . ",
	"   .   ,     .  ",	\14
	"     .     . "],
       ["      > ",
	"   7   7     7 ",
	"     <     > "]],

      [["      . ",
	"  .    ,      . ",	\15
	"     .      . "],
       ["      < ",
	"  7    7      7 ",
	"     <      < "]],

      [["      . ",
	"  .    ,      . ",	\16
	"     . "],
       ["      < ",
	"  <    7      < ",
	"     < "]],

      [["      . ",
	"       . ",		\17
	"     . "],
       ["      < ",
	"       7 ",
	"     < "]],

      [["      . ",
	"       . ",		\18
	"     . "],
       ["      < ",
	"       < ",
	"     < "]],

      [[" ",
	"       . ",		\19
	" "],
       [" ",
	"       < ",
	" "]],

      [[" ",
	"       . ",		\20
	" "],
       [" ",
	"       < ",
	" "]]];

if GRAPHICS then	\Convert to graphic characters
	begin
	for F:= 0, 41 do
	    for L:= 0, 2 do
		begin
		S:= EXPLOSION(F, 0, L);
		loop for I:= 0, 80 do
			begin
			case S(I) of
			  ^:	S(I):= ^;
			  ^=:	S(I):= ^;
			  ^:	S(I):= ^;
			  ^:	S(I):= ^;
			  ^:	S(I):= ^;
			 $A0:	quit
			other;
			end;
		end;
	end;

if VGA then		\Convert to fadeable colors
	begin
	for F:= 0, 41 do
	    for L:= 0, 2 do
		begin
		S:= EXPLOSION(F, 1, L);
		loop for I:= 0, 80 do
			begin
			case S(I) of
			  ^4:	S(I):= ^<;
			  ^7:	S(I):= ^;;
			  ^8:	S(I):= ^9;
			  ^>:	S(I):= ^=;
			  $A0:	quit
			other;
			end;
		end;
	end;
end;	\SETEXPLOSION



proc	SETSCREEN;	\Raise the curtain...
int	I, J, N, L, CH, SX, SY;
addr	A, K;
begin
K:=
"ͻ
                                                                              
                                                                              
                  
                   
                         
                           
                      
                                  
                                                                              
                     THENTHITERATION                    
                                                                              
                              byLorenBlaney                             
 Ŀ                                     ¿ 
ͳ STARDATE       ݳͳ                 ͼ
   CONDITION       ݳ Ŀ   
   QUADRANT       ݳ                                 Ŀ  
                                                         ENGINES    
   ENERGY         ݳ                                     PHASERS    
                                         SENSORS    
   SHIELDS        ݳ                                   VIEWER     
                                         MAP        
   LEVEL          ݳ                                     COMPASS    
   TRIBBLES       ݳ                                    D A M A G E   
   S T A T U S  C A P T A I N S   L O G                      
";

A:=
"88888888888888888888888888888888888888888888888888888888888888888888888888888888
8                                                                              8
8                                                                              8
8     :FFFFFFF:FFFFFFF :FFFFF :FFFFFF    :FFFFFFF:FFFFFF :FFFFFFF:FFF:FFF      8
8     :FFF::: :::FFF: :FFF:FFF:FFF:FFF   :::FFF: :FFF:FFF:FFF::: :FFFFFF       8
8     :FFFFFFF  :FFF  :FFF:FFF:FFF:FFF     :FFF  :FFF:FFF:FFFFFF :FFFFF        8
8     :::::FFF  :FFF  :FFFFFFF:FFFFFF      :FFF  :FFFFFF :FFF::  :FFFFFF       8
8     :FFFFFFF  :FFF  :FFF:FFF:FFF:FFF     :FFF  :FFF:FFF:FFFFFFF:FFF:FFF      8
8     :::::::   :::   ::: ::: ::: :::      :::   ::: ::: ::::::: ::: :::       8
8                                                                              8
8                      FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF                     8
8                                                                              8
8                               77777777777777777                              8
8 ********************                                     9999999999999999999 8
88************      **8888888888888888888888888888888888888999999999999999999988
  *************     **8***********************************89999999999999999999  
  ***************   **8*222222222222222222222222222222222*8*******************  
  ********************8*222222222222222222222222222222222*8*******************  
  **************    **8*222222222222222222222222222222222*8*******************  
  ********************8*222222222222222222222222222222222*8*******************  
  **************    **8*222222222222222222222222222222222*8*******************  
  ********************8*222222222222222222222222222222222*8*******************  
  **************    **8*222222222222222222222222222222222*8*******************  
  **************    **8*222222222222222222222222222222222*8*******************  
  ********************8***********************************8*******************  
";

J:= 24*82 +70;
for I:= 0, 7 do K(I+J):= NCCIMG(I);

\Set initial screen position to quadrant that Enterprise is in which is
\ on the far side of the galaxy from the starbase
SX:= (BASEX/QUADWIDTH + 4) & 7;		SY:= (BASEY/QUADHEIGHT + 4) & 7;
SX:= SX*QUADWIDTH-1;			SY:= SY*QUADHEIGHT-1;

CHOUT(0, FF);		\Make sure all the screen is clear (lower-right corner)

L:= 25*(80+2);		\Point to end of text string+1 (80 chars +CR +LF)
for J:= -24, 0 do
	begin
	\Draw stars in quadrant (SX, SY)
	if -J>=1 & -J<=12 then
		BLIT(GALAXY(SY-J), (SX+1)*2, SCRSEG, (-J*80+1)*2, (80-2)*2);
	if -J=13 then
		BLIT(GALAXY(SY+13), (SX+22)*2, SCRSEG, (13*80+22)*2, (58-22+1)*2);

	L:= L -82;			\Point to start of line
	N:= -J *80 *2;			\Point to corresponding point on screen
	for I:= 0, 79 do
		begin
		CH:= K(L+I);
		if CH # SP then			\Spaces are transparent so 
			SCREENSEG(0, N):= CH;	\ background is not overwritten
		case A(L+I) of
		  ^2:	SCREENSEG(0, N+1):= $02;
		  ^7:	SCREENSEG(0, N+1):= if VGA then $0B else $07;
		  ^8:	SCREENSEG(0, N+1):= if VID7 then $07 else $08;
		  ^9:	SCREENSEG(0, N+1):= if VID7 then $07 else $08;
		  ^F:	SCREENSEG(0, N+1):= if VID7 then $0F else $0F;
		  ^::	SCREENSEG(0, N+1):= if VID7 then $00 else $01;
		  ^*:	SCREENSEG(0, N+1):= if VID7 then $70 else $80
		other;
		N:= N +2;
		end;
	DELAY(2);
	end;

\Set up VIEWPORT mask to match the shape of the view screen (used by CTXT)
for I:= 0, SCRSIZE-1 do VIEWPORT(I):= false;
for J:= 1, 12 do
	for I:= J*80+1, J*80+78 do VIEWPORT(I):= true;
for I:= 13*80+22, 13*80+58 do VIEWPORT(I):= true;
end;	\SETSCREEN



proc	SETUP;		\Make a galaxy and other initialization
int	I, J;
begin
I:= RAN(0);	\make repeatable random sequence for recognizable star patterns
SETSTARS;
SETGRID;

\for I:= 0, LEVEL do	\\make repeatable random sequence for current level
\	J:= RAN(2);
J:= RAN(-1);

\Keep the base away from the edges of the galaxy because wrap isn't handled.
\ Also, don't put base in left column of quadrants for same reason.
SETBASE;

\Make an extra copy of the quadrants of the left side to deal with wrap around.
for J:= 0, GALHEIGHT-1 do
    for I:= 0, QUADWIDTH-1 do
	GALAXY(J, I+GALWIDTH):= GALAXY(J, I);
end;	\SETUP

\----------------------------------------------------------------------

func	LOADSND(NAME, LEN);	\Load sound file into memory
addr	NAME;
int	LEN;
seg addr SND;
int	I, J,
	HAND;	\Input file handle
begin
SND:= RESERVE(2);
TRAP(false);
SND(0):= MALLOC((LEN+15)/16);
if GETERR then FATAL("OOPS, NOT ENOUGH FREE MEMORY. TRY REMOVING SOME TSR'S.");

HAND:= FOPEN(NAME, 0);
if GETERR then FATAL("NOPE, ^".SND^" FILE NOT IN CURRENT DIRECTORY.");
\if DEBUG then TRAP(true);
FSET(HAND, ^I);
OPENI(3);

I:= 0;
repeat	begin
	SND(0, I):= CHIN(3);
	I:= I +1;
	end;
until I = LEN;

FCLOSE(HAND);
return SND(0);
end;	\LOADSND



proc	GETSWITCH;	\Set up switches from command line
\Outputs: SOUNDON, LEVEL
int	I, CH;
addr	CMDTAIL;
begin
\Get command tail from PSP
CMDTAIL:= RESERVE($80);
BLIT(PSPSEG, $80, DATASEG, CMDTAIL, $80);

for I:= 1, CMDTAIL(0) do
	begin
	CH:= CMDTAIL(I);
	if CH = ^/ then
		begin
		CH:= CMDTAIL(I+1);
		if CH=^S ! CH=^s then SOUNDON:= false
		else if CH>=^0 & CH<=^9 then
			begin
			LEVEL:= CH -^0;
			CH:= CMDTAIL(I+2);
			if CH>=^0 & CH<=^9 then LEVEL:= LEVEL *10 + CH-^0;
			end;
		end;
	end;
end;	\GETSWITCH

\======================================================================

begin	\MAIN
CPUREG:= GETREG;
PSPSEG:= CPUREG(11);
DATASEG:= CPUREG(12);

BOOMSND:= RESERVE(2);
CRASHSND:= RESERVE(2);
PHASERSND:= RESERVE(2);
OPENPODSND:= RESERVE(2);
BOBEEPSND:= RESERVE(2);
TRIBSND:= RESERVE(2);
SIRENSND:= RESERVE(2);
SQUEALSND:= RESERVE(2);
WARNSND:= RESERVE(2);
PINGSND:= RESERVE(2);
SWEEPSND:= RESERVE(2);

DAMAGE:= RESERVE(6 *INTSIZE);

VIEWPORT:= RESERVE(SCRSIZE);
GALAXY:= RESERVE(GALHEIGHT *2);
TRAP(false);
for JJ:= 0, GALHEIGHT-1 do
	begin
	GALAXY(JJ):= MALLOC( ((GALWIDTH+QUADWIDTH)*2+15) /16);
	if GETERR then FATAL("NOPE, NOT ENOUGH FREE MEMORY.");
	end;
\if DEBUG then TRAP(true);

VIDMODE:= GETVID;
VID7:= VIDMODE = 7;	\Monochrome?
CRTCSTAT:= $3BA;
if ~VID7 then CRTCSTAT:= $3DA;

SCREENSEG:= RESERVE(2);
SCRSEG:= if VID7 \MDA or Herc\ then $B000 else $B800;
SCREENSEG(0):= SCRSEG;

CPUREG(0):= $1201;	\Set 350 scan lines and 8-dot wide chars if VGA
CPUREG(1):= $0030;
SOFTINT($10);
SETVID(VIDMODE);

SHOWCUR(false);		\get rid of the flashing cursor

\Do color stuff for fades:
PALREG:= RESERVE($11);		\Get a copy of the palette registers
PALREG(0):= 123;
CPUREG(0):= $1009;		\AX function $10, subfunction $09
CPUREG(11):= DATASEG;		\ES:DX points to 17-byte array
CPUREG(3):= PALREG;
SOFTINT($10);			\Call BIOS routine
VGA:= PALREG(0)#123 & ~VID7;

COLOR0REG:= RESERVE($40*3);	\Get a copy of the initial color registers
CPUREG(0):= $1017;
CPUREG(1):= 0;
CPUREG(2):= $40;
CPUREG(3):= COLOR0REG;
CPUREG(11):= DATASEG;
SOFTINT($10);

CHANGECOLOR($9, $8);	\Change light blue to gray
CHANGECOLOR($B, $7);	\Change light cyan to white
CHANGECOLOR($D, $E);	\Change light magenta to yellow

COLORREG:= RESERVE($40*3);	\Set up copy of color registers for fades
for II:= 0, $40*3-1 do COLORREG(II):= COLOR0REG(II);
SETBLKCOLOR;


SETGRAPH;
SETEXPLOSION;

II:= EQUIP;		\Default switch values
SOUNDON:= II(3) > 200;	\Sound defaults to off on slow computers
LEVEL:= 3;
GETSWITCH;
if LEVEL = 0 then LEVEL:= 1;

\Don't start Enterprise near edge to minimize confusion
BASEX:= GALWIDTH-QUADWIDTH-RAN(QUADWIDTH);
BASEY:= GALHEIGHT-QUADHEIGHT-RAN(QUADHEIGHT);
SETUP;	\(Inputs LEVEL)

\Warning: Some BIOS's reset the blink bit when loading a font.
if ~VID7 then
	begin		\Redefine blink bit in attribute to be intensity
	if ~VGA then	\Assume it is CGA
		begin
		JJ:= PEEK(0, $463) +$304;	\port address of mode set register
		II:= PEEK(0, $465) & $DF;	\clear blink enable bit
		POUT(II, JJ, 0);
		POKE(0, $465, II);		\save updated value in BIOS data area
		end
	else	begin	\BIOS can handle it if it's EGA or VGA
		CPUREG(0):= $1003;
		CPUREG(1):= $0000;
		SOFTINT($10);
		end;
	end;

SETSCREEN;		\Show starting screen while sound files load

BOOMSND(0):= LOADSND("BOOM.SND", 26000);
CRASHSND(0):= LOADSND("CRASH.SND", 9407);
PHASERSND(0):= LOADSND("PHASER.SND", 4800);
OPENPODSND(0):= LOADSND("OPENPOD.SND", 16000);
BOBEEPSND(0):= LOADSND("BOBEEP.SND", 11520);
TRIBSND(0):= LOADSND("TRIB.SND", 6520);
SIRENSND(0):= LOADSND("SIREN.SND", 9216);
SQUEALSND(0):= LOADSND("SQUEAL.SND", 7680);
WARNSND(0):= LOADSND("WARN.SND", 500);
PINGSND(0):= LOADSND("PING.SND", 7200);
SWEEPSND(0):= LOADSND("SWEEP.SND", SWEEPSIZE);

INSTRUCTIONS;
SETKBV;			\Set up keyboard stuff after instructions have been read

SETGALMAP;   DELAY(9);	\Clear captain's log to acknowledge keystroke
if VGA then FADE(true, 0) else CLEARVIEW;
DELAY(9);

if SOUNDON then SETSTV;

TRIBALL:= 0;   TRIBSAVE:= 0;   TRIBKILL:= 0;   TRIBPHASER:= 0;
KLNKILL:= 0;

SHIPS:= 3;
STARDATE:= 23000;	\(This should agree with value in instructions)
TIME:= 1;
CHEAT:= false;
ENERGY:= 0;   SHIELDS:= 0;
for II:= 0, 6-1 do DAMAGE(II):= 0;

loop	begin
	KLNS:= 4 <<LEVEL;
	if KLNS>512 ! LEVEL>7 then KLNS:= 512;

	XTALS:= LEVEL *LEVEL *2;
	if XTALS > 512 then XTALS:= 512;

	TRIBS:= 1 <<LEVEL;
	if TRIBS>512 ! LEVEL>9 then TRIBS:= 512;

	PLAYLEVEL;		\main routine to play game
	if KEYS & $10 then quit;	\Esc
	if WON then
		LEVEL:= LEVEL +1
	else	begin
		SHIPS:= SHIPS -1;
		if SHIPS <= 0 then [DELAY(18);   quit];
		LEVEL:= LEVEL -1;
		end;
	BASEX:= RAN(GALWIDTH-24-QUADWIDTH)+10+QUADWIDTH;
	BASEY:= RAN(GALHEIGHT-8)+4;
	SETUP;
	end;

if SOUNDON then RESSTV;
RESKBV;
CLEARVIEW;
EPILOGUE;

if VIDMODE # 7 then
	begin
	CPUREG(0):= $1202;	\Set 400 scan lines if VGA
	CPUREG(1):= $0030;
	SOFTINT($10);
	end;
SETVID(VIDMODE);	\Clear attributes etc.
SOUND(1, 1, 50);	\Clear sound (otherwise PROCOMM doesn't sound right)
end;	\MAIN
