{ MUSICV.PAS : Music Vision memory game

  Title   : MUSICV
  Language: Borland Pascal v7.0 + Turbo Vision v2.0
  Version : 1.4
  Date    : Feb 27, 2000
  Author  : J.R. Ferguson, Amsterdam, The Netherlands
  Usage   : MS-DOS real mode program
  Download: http://hello.to/ferguson
  E-mail  : j.r.ferguson@iname.com

  This program and its source may be copied and used freely without charge,
  but only for  non-commercial purposes.  In no way  the author can be held
  responsible to  any damage or loss of data  that may be caused by the use
  of this software.
}

PROGRAM MUSICV;

uses Objects, App, Menus, Drivers, Views, Dialogs, MsgBox, Crt, Dos;

{$I OBJTYPE.INC}

const
  C_MinKeyCount    =  3;
  C_DflKeyCount    =  8;
  C_MaxKeyCount    = 10;

  C_DflFixedStart  = true;
  C_DflPlayTunes   = true;

  C_MinSpeed       =  1;
  C_DflSpeed       = 12;
  C_MaxSpeed       = 23;

  C_ProgIdent      = 'MUSICV v1.4';
  C_ProgTitle      = 'Music Vision';
  C_Copyright      = '(c) 1996-2000, J.R. Ferguson';
  C_Email          = 'j.r.ferguson@iname.com';
  C_URL            = 'http://hello.to/ferguson';
  C_SettingFileName= 'MUSICV.SET';

  cm_MainMenu      = cmMenu;
  cm_GameStart     = 100;
  cm_GameOptions   = 101;
  cm_GameExit      = cmQuit;
  cm_HelpExplain   = 110;
  cm_HelpAbout     = 111;
  cm_DflSettings   = 120;
  cm_KeyBase       = 200;

type
  PTitleStr     = ^TTitleStr;
  P_FramedText  = ^T_FramedText;
  P_Frame       = ^T_Frame;
  P_Note        = ^T_Note;
  P_Melody      = ^T_Melody;
  P_Key         = ^T_Key;
  P_KeyBoard    = ^T_KeyBoard;
  P_PromptText  = ^T_PromptText;
  P_StatusText  = ^T_StatusText;
  P_Settings    = ^T_Settings;
  P_GameWindow  = ^T_GameWindow;
  P_MessageDlg  = ^T_MessageDlg;
  P_EditDialog  = ^T_EditDialog;
  P_SettingsBuf = ^T_SettingsBuf;
  P_SettingsDlg = ^T_SettingsDlg;
  P_MenuBar     = ^T_MenuBar;
  P_StatusLine  = ^T_StatusLine;
  P_Application = ^T_Application;

  T_Tone        = 1..C_MaxKeyCount;
  T_Duration    = Word;
  T_KeyLabel    = String[4];

  T_FramedText  = Object(TStaticText)
    Constructor Init(var V_Bounds: TRect; V_Title: TTitleStr);
  end;

  T_Frame       = Object(T_FramedText)
    Constructor Init(var V_Bounds: TRect);
  end;

  T_Note        = Object(TObject)
    Tone        : T_Tone;
    Duration    : T_Duration;
    Constructor Init(V_Tone: T_Tone; V_Duration: T_Duration);
  end;

  T_Melody      = Object(TCollection) { of P_Note }
    Constructor Init;
    procedure   Reset;
    procedure   AddNote(V_Tone: T_Tone; V_Duration: T_Duration);
  end;

  T_Key         = Object(TButton)
    Tone        : T_Tone;
    Constructor Init(var V_Bounds: TRect; V_Tone: T_Tone);
  end;

  T_KeyBoard    = Object(TGroup)
    Keys        : PCollection;
    KeyStatus   : integer;
    Constructor Init(var V_Bounds: TRect; V_KeyCount: T_Tone);
    Destructor  Done; virtual;
    function    FindKey(V_Tone: T_Tone): P_Key;
    procedure   Reset;
    procedure   ActivateKeys;
    procedure   DeactivateKeys;
    function    KeysActive: boolean;
  end;

  T_PromptText  = Object(TParamText)
    ParmRec     : PTitleStr;
    Prompt      : TTitleStr;
    Constructor Init(V_Bounds: TRect);
    procedure   Display(V_Prompt: TTitleStr);
    procedure   DisplayOff;
  end;

  T_StatusText  = Object(TParamText)
    ParmRec     : PTitleStr;
    StatusStr   : TTitleStr;
    Constructor Init(V_Bounds: TRect);
    procedure   Display(V_Count: integer);
    procedure   DisplayOff;
  end;

  T_Settings    = Object(TObject)
    KeyCount    : integer;
    FixedStart  : boolean;
    PlayTunes   : boolean;
    Speed       : integer;
    Constructor Init(V_KeyCount   : integer;
                     V_FixedStart : boolean;
                     V_PlayTunes  : boolean;
                     V_Speed      : integer);
    Constructor Load (var V_Stream: TStream);
    procedure   Store(var V_Stream: TStream);
  end;

  T_GameWindow  = Object(TDialog)
    Settings    : P_Settings;
    PromptText  : P_PromptText;
    StatusText  : P_StatusText;
    KeyBoard    : P_KeyBoard;
    Melody      : P_Melody;
    ActiveGame  : boolean;
    KeyStatus   : integer;
    NoteCount   : integer;
    Constructor Init(var V_Bounds: TRect; V_Title: TTitleStr);
    Destructor  Done; virtual;
    procedure   GameInit;
    procedure   GameTerm;
    procedure   GameReset;
    procedure   GameReInit;
    procedure   HandleEvent(var V_Event: TEvent); virtual;
    procedure   ClearPendingEvents;
    procedure   ActivateKeys;
    procedure   DeactivateKeys;
    function    KeysActive: boolean;
    procedure   DoKey(V_Tone: T_Tone);
    procedure   DoGameBegin;
    procedure   DoGameSettings;
    procedure   NewMelody;
    procedure   NewNote;
    procedure   Wait(V_Duration: T_Duration);
    procedure   PlayNote(V_Tone: T_Tone; V_Duration: T_Duration);
    procedure   PlayMelody(V_Melody: P_Melody);
    procedure   StartTune;
    procedure   EndTune;
    procedure   DoExitGame;
  end;

  T_MessageDlg  = Object(TDialog)
    Constructor Init(V_Title: TTitleStr; V_MsgRows: integer; V_Msg: String);
  end;


  T_EditDialog  = Object(TDialog)
    IOBuffer    : Pointer;
    GameWindow  : P_GameWindow;
    Constructor Init(V_Cols, V_Rows: integer; V_Title: TTitleStr;
                     V_GameWindow: P_GameWindow);
    Destructor  Done;                virtual;
    function    Valid(V_Command: Word): boolean; virtual;
    procedure   InitBuffer;          virtual;
    procedure   InitControls;        virtual;
    procedure   ImportData;          virtual;
    function    ExportData: boolean; virtual;
  end;

  T_SettingsBuf = record
                    IO_KeyCount   : Word;
                    IO_FixedStart : Word;
                    IO_PlayTunes  : Word;
                  end;

  T_SettingsDlg = Object(T_EditDialog)
    ScrollBar   : PScrollbar;
    Settings    : P_Settings;
    Constructor Init(V_GameWindow: P_GameWindow);
    Destructor  Done;                virtual;
    procedure   InitBuffer;          virtual;
    procedure   InitControls;        virtual;
    procedure   ImportData;          virtual;
    function    ExportData: boolean; virtual;
    function    Valid(V_Command: Word): boolean; virtual;
    procedure   HandleEvent(var V_Event: TEvent); virtual;
    procedure   DoReset;
  end;

  T_MenuBar     = Object(TMenuBar)
    procedure   Draw; virtual;
  end;

  T_StatusLine  = Object(TStatusLine)
    procedure   Draw; virtual;
  end;

  T_Application = Object(TApplication)
    GameWindow  : P_GameWindow;
    Constructor Init;
    Destructor  Done;           virtual;
    procedure   InitMenuBar;    virtual;
    procedure   InitStatusLine; virtual;
    procedure   HandleEvent(var V_Event: TEvent); virtual;
    procedure   DoHelpExplain;
    procedure   DoHelpAbout;
  end;

const
  Hz        : array[T_Tone] of Word =
    (523,587,659,698,784,880,988,1047,1175,1319);

  R_Settings: TStreamRec = (
    ObjType : OT_MUSICV_Settings;
    VmtLink : Ofs(TypeOf(T_Settings)^);
    Load    : @T_Settings.Load;
    Store   : @T_Settings.Store);

var
  GV_HomeDir: PathStr;


{ --- General --- }

function KeyLabel(V_Tone: T_Tone): T_KeyLabel;
var s: String[2];
begin Str(V_Tone:2,s); KeyLabel:= s[1]+'~'+s[2]+'~'; end;

procedure StreamRegistration;
begin
  RegisterType(R_Settings);
end;

function GetHomeDir: PathStr;
var ExePath: PathStr; Dir: DirStr; Name: NameStr; Ext: ExtStr;
begin
  ExePath:= FExpand(ParamStr(0));
  FSplit(ExePath,Dir,Name,Ext);
  GetHomeDir:= Dir;
end;

procedure ReadSettings(var V_Settings: P_Settings);
var DosStream: TDosStream;
begin
  DosStream.Init(GV_HomeDir+C_SettingFileName,stOpenRead);
  V_Settings:= P_Settings(DosStream.Get);
  DosStream.Done;
  if DosStream.Status <> StOK then
    New(V_Settings,Init(C_DflKeyCount,C_DflFixedStart,C_DflPlayTunes,C_DflSpeed));
end;

function SaveSettings(V_Settings: P_Settings): boolean;
var DosStream: TDosStream;
begin
  DosStream.Init(GV_HomeDir+C_SettingFileName,stCreate);
  DosStream.Put(V_Settings);
  DosStream.Done;
  if DosStream.Status <> StOK then begin
    MessageBox(#3'Unable to save settings'#13#13 +
               #3+GV_HomeDir+C_SettingFileName,nil,
    mfError or mfOKButton);
    SaveSettings:= false;
  end
  else SaveSettings:= true;
end;


{ --- T_FramedText --- }

Constructor T_FramedText.Init(var V_Bounds: TRect; V_Title: TTitleStr);
begin
  Inherited Init(V_Bounds,V_Title);
  Options:= Options or ofFramed;
end;


{ --- T_Frame --- }

Constructor T_Frame.Init(var V_Bounds: TRect);
begin Inherited Init(V_Bounds,''); end;


{ --- T_Note --- }

Constructor T_Note.Init(V_Tone: T_Tone; V_Duration: T_Duration);
begin
  Inherited Init;
  Tone:= V_Tone;
  Duration:= V_Duration;
end;


{ --- T_Melody --- }

Constructor T_Melody.Init;
begin Inherited Init(20,20); end;

procedure   T_Melody.Reset;
begin FreeAll; end;

procedure   T_Melody.AddNote(V_Tone: T_Tone; V_Duration: T_Duration);
begin Insert(New(P_Note,Init(V_Tone,V_Duration))); end;


{ --- T_Key --- }

Constructor T_Key.Init(var V_Bounds: TRect; V_Tone: T_Tone);
begin
  Inherited Init(V_Bounds,KeyLabel(V_Tone),cm_KeyBase+V_Tone,bfNormal);
  Options:= Options and not ofSelectable;
  Tone:= V_Tone;
end;


{ --- T_KeyBoard --- }

Constructor T_KeyBoard.Init(var V_Bounds: TRect; V_KeyCount: T_Tone);
var i: T_Tone; p: P_Key; R0,R: TRect; dx: integer;
begin
  Inherited Init(V_Bounds);
  KeyStatus:= 0;
  New(Keys,Init(V_KeyCount,0));

  GetExtent(R); Insert(New(PStaticText,Init(R,'')));

  GetExtent(R0); dx:= R0.B.X div V_KeyCount;
  R0.Grow(-((R0.B.X - (V_KeyCount * dx)) div 2),0);

  R.Copy(R0);
  Dec(R.B.Y,3); R.B.X:= R.A.X + dx;
  for i:= 1 to V_KeyCount do begin
    New(p,Init(R,i));
    Keys^.Insert(p);
    Insert(p);
    R.Move(dx,0);
  end;

  R.Copy(R0); R.A.Y:= R.B.Y-2;
  Insert(New(PButton,Init(R,'Start game',cm_GameStart,bfDefault)));
  SelectNext(false);
end;

Destructor T_KeyBoard.Done;
begin
  Keys^.DeleteAll; Dispose(Keys,Done);
  Inherited Done;
end;

function   T_KeyBoard.FindKey(V_Tone: T_Tone): P_Key;
begin
  if V_Tone > Keys^.Count then FindKey:= nil
  else FindKey:= Keys^.At(V_Tone-1);
end;

procedure  T_KeyBoard.Reset;
begin KeyStatus:= 0; end;

procedure   T_KeyBoard.ActivateKeys;
begin
  if KeyStatus = 1 then begin
    KeyStatus:= 0;
  end
  else if KeyStatus > 0 then Dec(KeyStatus);
end;

procedure   T_KeyBoard.DeactivateKeys;
begin
  if KeyStatus = 0 then begin
    KeyStatus:= 1;
  end
  else if KeyStatus < MaxInt then Inc(KeyStatus);
end;

function    T_KeyBoard.KeysActive: boolean;
begin KeysActive:= KeyStatus = 0; end;


{ --- T_PromptText --- }

Constructor T_PromptText.Init(V_Bounds: TRect);
begin
  Inherited Init(V_Bounds,#3'%s',1);
  Prompt:=''; ParmRec:= @Prompt; SetData(ParmRec);
end;

procedure   T_PromptText.Display(V_Prompt: TTitleStr);
begin Prompt:= V_Prompt; Draw; end;

procedure   T_PromptText.DisplayOff;
begin Display(''); end;


{ --- T_StatusText --- }

Constructor T_StatusText.Init(V_Bounds: TRect);
begin
  Inherited Init(V_Bounds,#3'%s',1);
  StatusStr:= ''; ParmRec:= @StatusStr; SetData(ParmRec);
end;

procedure   T_StatusText.Display(V_Count: integer);
var SCnt: string[3];
begin
  Str(V_Count:3,SCnt); StatusStr:= 'Keys:'+SCnt;
  Draw;
end;

procedure   T_StatusText.DisplayOff;
begin StatusStr:= ''; Draw; end;


{ --- T_Settings --- }

Constructor T_Settings.Init(V_KeyCount      : integer;
                              V_FixedStart : boolean;
                              V_PlayTunes  : boolean;
                              V_Speed      : integer);
begin
  KeyCount   := V_KeyCount;
  FixedStart := V_FixedStart;
  PlayTunes  := V_PlayTunes;
  Speed      := V_Speed;
end;

Constructor T_Settings.Load (var V_Stream: TStream);
begin with V_Stream do begin
  Read(KeyCount  ,SizeOf(KeyCount  ));
  Read(FixedStart,SizeOf(FixedStart));
  Read(PlayTunes ,SizeOf(PlayTunes ));
  Read(Speed     ,SizeOf(Speed     ));
end; end;

procedure   T_Settings.Store(var V_Stream: TStream);
begin with V_Stream do begin
  Write(KeyCount  ,SizeOf(KeyCount  ));
  Write(FixedStart,SizeOf(FixedStart));
  Write(PlayTunes ,SizeOf(PlayTunes ));
  Write(Speed     ,SizeOf(Speed     ));
end; end;


{ --- T_GameWindow --- }

Constructor T_GameWindow.Init(var V_Bounds: TRect; V_Title: TTitleStr);
var R: TRect;
begin {T_GameWindow.init}
  Inherited Init(V_Bounds,V_Title);
  Flags       := Flags and not (wfMove or wfClose);
  Palette     := wpBlueWindow;
  ReadSettings(Settings);
  GetExtent(R); R.A.Y:= 2; R.B.Y:= 3; R.Grow(-4,0);
  New(PromptText,Init(R)); Insert(PromptText);
  R.Move(0,1);
  New(StatusText,Init(R)); Insert(StatusText);
  New(Melody,Init);
  GameInit;
end;

Destructor  T_GameWindow.Done;
begin
  PromptText^.DisplayOff;
  StatusText^.DisplayOff;
  DeactivateKeys;
  Dispose(Melody,Done);
  EndTune;
  GameTerm;
  SaveSettings(Settings);
  Dispose(Settings,Done);
  Inherited Done;
end;

procedure   T_GameWindow.GameInit;
var R: TRect;
begin
  GameReset;
  GetExtent(R); R.Grow(-4,-1); Inc(R.A.Y,4);
  New(KeyBoard,Init(R,Settings^.KeyCount)); Insert(KeyBoard);
end;

procedure   T_GameWindow.GameTerm;
begin
  Delete(KeyBoard);
  Dispose(KeyBoard,Done);
end;

procedure   T_GameWindow.GameReSet;
begin
  PromptText^.Display('');
  StatusText^.DisplayOff;
  ActiveGame:= false;
  NoteCount := 0;
  KeyBoard^.Reset;
  Melody^.Reset;
end;

procedure   T_GameWindow.GameReInit;
begin
  DeactivateKeys;
  GameTerm;
  GameInit;
  ActivateKeys;
end;

procedure   T_GameWindow.ActivateKeys;
begin
  ClearPendingEvents;
  KeyBoard^.ActivateKeys;
end;

procedure   T_GameWindow.DeactivateKeys;
begin KeyBoard^.DeactivateKeys; end;

function    T_GameWindow.KeysActive: boolean;
begin KeysActive:= KeyBoard^.KeysActive; end;

procedure   T_GameWindow.HandleEvent(var V_Event: TEvent);
begin with V_Event do begin
  Inherited HandleEvent(V_Event);
  case What of
    evCommand: case Command of
      cm_GameStart   : begin DoGameBegin   ; ClearEvent(V_Event); end;
      cm_GameOptions : begin DoGameSettings; ClearEvent(V_Event); end;
    end;
  end;
  if KeysActive then case What of
    evCommand: if (Command >= cm_KeyBase + 1                 ) and
                  (Command <= cm_KeyBase + Settings^.KeyCount)
               then begin
                 DoKey(Command - cm_KeyBase);
                 ClearEvent(V_Event);
               end;
  end;
end; end;

procedure   T_GameWindow.ClearPendingEvents;
var Event: TEvent;
begin while EventAvail do GetEvent(Event); end;

procedure   T_GameWindow.DoKey(V_Tone: T_Tone);
begin
  PlayNote(V_Tone,3);
  if ActiveGame then begin
    Inc(NoteCount);
    if (NoteCount <= Melody^.Count) then begin
      if P_Note(Melody^.At(NoteCount-1))^.Tone = V_Tone then begin
        if NoteCount = Melody^.Count then begin {all right}
          DeactivateKeys;
          Wait(3);
          NewNote;
          ActivateKeys;
        end;
      end
      else begin {wrong note}
        DeactivateKeys;
        DoExitGame;
        ActivateKeys;
      end;
    end;
  end;
end;

procedure   T_GameWindow.DoGameBegin;
begin NewMelody; ActiveGame:= true; end;

procedure   T_GameWindow.DoGameSettings;
begin
  if Application^.ExecuteDialog(New(P_SettingsDlg,Init(@Self)),nil) = cmOK
    then GameReInit
    else GameReset;
end;

procedure   T_GameWindow.NewNote;
var i,n: integer;
begin with Settings^ do begin
  DeactivateKeys;
  if FixedStart then Melody^.AddNote(1+Random(KeyCount),3)
  else begin
    n:= Melody^.Count + 1; Melody^.Reset;
    for i:= 1 to n do Melody^.AddNote(1+Random(KeyCount),3);
  end;
  PromptText^.Display('Listen carefully ...');
  StatusText^.Display(Melody^.Count);
  NoteCount:= 0;
  PlayMelody(Melody);
  PromptText^.Display('Your turn ...');
  ActivateKeys;
end; end;

procedure   T_GameWindow.NewMelody;
begin
  DeactivateKeys;
  Melody^.Reset;
  NewNote;
  ActivateKeys;
end;


procedure   T_GameWindow.Wait(V_Duration: T_Duration);
var t0,t: LongInt;
const C_cSecPerDay = 24 * 60 * 60 * 100;
  function ClockTime: LongInt;
  var u,m,s,c: Word;
  begin
    GetTime(u,m,s,c);
    ClockTime:= ((LongInt(u)*60+LongInt(m))*60+LongInt(s))*100+LongInt(c);
  end;
begin {Wait}
  t0:= ClockTime;
  repeat
    t:= ClockTime; if t<t0 then Inc(t,C_cSecPerDay);
  until (t-t0) > (2+C_MaxSpeed-Settings^.Speed) * V_Duration;
end;


procedure   T_GameWindow.PlayNote(V_Tone: T_Tone; V_Duration: T_Duration);
var Key: P_Key;
begin
  Key:= KeyBoard^.FindKey(V_Tone);
  if Key <> nil then Key^.DrawState(true);
  Sound(Hz[V_Tone]); Wait(V_Duration); NoSound;
  if Key <> nil then Key^.DrawState(false);
end;

procedure   T_GameWindow.PlayMelody(V_Melody: P_Melody);
  procedure Play(V_Note: Pointer); far;
  begin with P_Note(V_Note)^ do PlayNote(Tone,Duration); end;
begin
  V_Melody^.ForEach(@Play);
end;

procedure   T_GameWindow.StartTune;
var Tone: T_Tone; OldSpeed: integer;
begin with Settings^ do if PlayTunes then begin
  DeactivateKeys;
  OldSpeed:= Speed; Speed:= C_DflSpeed;
  PlayNote(1,2);
  for Tone:= 2 to 7 do PlayNote(Tone,1);
  PlayNote(8,3);
  Speed:= OldSpeed;
  ActivateKeys;
end; end;

procedure   T_GameWindow.EndTune;
var Tone: T_Tone; OldSpeed: integer;
begin with Settings^ do if PlayTunes then begin
  DeactivateKeys;
  OldSpeed:= Speed; Speed:= C_DflSpeed;
  PlayNote(8,2);
  for Tone:= 7 downto 2 do PlayNote(Tone,1);
  PlayNote(1,3);
  Speed:= OldSpeed;
  ActivateKeys;
end; end;

procedure   T_GameWindow.DoExitGame;
var s: String;
begin
  ActiveGame:= false; PromptText^.Display('');
  if Melody^.Count = 1 then
    s:= 'That''s not the right note.'
  else begin
    Str(Melody^.Count-1,s);
    s:= 'You had '+s+' notes right.';
  end;
  Application^.ExecuteDialog(New(P_MessageDlg,Init('Game over',1,#3+s)),nil);
  StatusText^.DisplayOff;
end;



{ --- T_MessageDlg --- }


Constructor T_MessageDlg.Init(V_Title: TTitleStr;
                              V_MsgRows: integer; V_Msg: String);
var R: TRect; X: integer;
begin
  DeskTop^.GetExtent(R); X:= (R.B.X-R.A.X) div 2;
  R.Assign(X-20,07,X+20,07 + V_MsgRows + 6);
  Inherited Init(R,V_Title);

  R.Assign(02,02,38,02 + V_MsgRows);
  Insert(New(PStaticText,Init(R, V_Msg)));

  R.Assign(15,02 + V_MsgRows + 1, 25,02 + V_MsgRows + 3);
  Insert(New(PButton,Init(R,'O~K~',cmOK,bfDefault)));
end;


{ --- T_EditDialog --- }

Constructor T_EditDialog.Init(V_Cols, V_Rows: integer; V_Title: TTitleStr;
                              V_GameWindow: P_GameWindow);
var R: TRect;
begin
  R.Assign(0,0,V_Cols,V_Rows);
  Inherited Init(R,V_Title);
  GameWindow:= V_GameWindow;
  Options:= Options or ofCentered;
  InitBuffer;
  InitControls;
  if IOBuffer <> nil then ImportData;
end;

Destructor  T_EditDialog.Done;
begin
  if IOBuffer <> nil then begin
    Dispose(IOBuffer);
    IOBuffer:= nil;
  end;
  Inherited Done;
end;

function    T_EditDialog.Valid(V_Command: Word): boolean;
var IsOK: boolean;
begin
  if V_Command = cmCancel then IsOK:= true else IsOK:= ExportData;
  Valid:= IsOK and Inherited Valid(V_Command);
end;

procedure T_EditDialog.InitBuffer;          begin IOBuffer:= nil; end;
procedure T_EditDialog.InitControls;        begin end;
procedure T_EditDialog.ImportData;          begin end;
function  T_EditDialog.ExportData: boolean; begin ExportData:= true; end;


{ --- T_SettingsDlg --- }

Constructor T_SettingsDlg.Init(V_GameWindow: P_GameWindow);
begin
  Inherited Init(42,15,'Options',V_GameWindow);
end;

Destructor  T_SettingsDlg.Done;
begin
  Dispose(Settings,Done);
  Inherited Done;
end;

procedure   T_SettingsDlg.InitBuffer;
begin
  IOBuffer  := new(P_SettingsBuf);
  New(Settings,Init(
    GameWindow^.Settings^.KeyCount  ,
    GameWindow^.Settings^.FixedStart,
    GameWindow^.Settings^.PlayTunes ,
    GameWindow^.Settings^.Speed     ));
end;

procedure   T_SettingsDlg.InitControls;
var R: TRect; StringList: PSItem; i: integer; p: PView;
begin
  R.Assign(03,02,11,13); Insert(New(P_FramedText,Init(R,'Number of keys')));
  StringList:= nil;
  for i:= C_MaxKeyCount downto C_MinKeyCount do begin
    StringList:= NewSItem(KeyLabel(i),StringList);
  end;
  R.Assign(03,05,11,13); Insert(New(PRadioButtons,Init(R,StringList)));

  R.Assign(14,02,39,04); Insert(New(P_Frame,Init(R)));
  R.Assign(14,02,39,03); Insert(New(PCheckBoxes,Init(R,NewSItem('~F~ixed melody start',nil))));
  R.Assign(14,03,39,04); Insert(New(PCheckBoxes,Init(R,NewSItem('Play ~t~unes',nil))));

  R.Assign(14,05,39,07); Insert(New(P_Frame,Init(R)));
  R.Assign(14,05,39,06); Insert(New(PLabel,Init(R,'slow      Speed    fast',ScrollBar)));
  R.Assign(14,06,39,07); ScrollBar:= New(PScrollBar,Init(R));
  with ScrollBar^ do begin
    SetParams(C_DflSpeed,C_MinSpeed,C_MaxSpeed,4,1);
    Options:= Options or ofSelectable or ofFirstClick;
  end;
  Insert(ScrollBar);

  R.Assign(27,08,40,10); Insert(New(PButton,Init(R,'~D~efault',cm_DflSettings,bfNormal )));
  R.Assign(27,10,40,12); Insert(New(PButton,Init(R,'~C~ancel' ,cmCancel      ,bfNormal )));
  R.Assign(27,12,40,14); Insert(New(PButton,Init(R,'O~K~'     ,cmOK          ,bfDefault)));

  SelectNext(false);
end;

procedure   T_SettingsDlg.ImportData;
begin with Settings^, P_SettingsBuf(IOBuffer)^ do begin
  IO_KeyCount:= KeyCount - C_MinKeyCount;
  if FixedStart then IO_FixedStart:= 1 else IO_FixedStart:= 0;
  if PlayTunes  then IO_PlayTunes := 1 else IO_PlayTunes := 0;
  SetData(IOBuffer^);
  ScrollBar^.SetValue(Speed);
  Redraw;
end; end;

function    T_SettingsDlg.ExportData: boolean;
begin with Settings^, P_SettingsBuf(IOBuffer)^ do begin
  GetData(IOBuffer^);
  KeyCount  := C_MinKeyCount + IO_KeyCount;
  FixedStart:= IO_FixedStart = 1;
  PlayTunes := IO_PlayTunes  = 1;
  Speed     := ScrollBar^.value;
  ExportData:= true;
end; end;

function    T_SettingsDlg.Valid(V_Command: Word): boolean;
begin
  if Inherited Valid(V_Command) then begin
    if V_Command <> cmCancel then begin
      GameWindow^.Settings^.KeyCount  := Settings^.KeyCount  ;
      GameWindow^.Settings^.FixedStart:= Settings^.FixedStart;
      GameWindow^.Settings^.PlayTunes := Settings^.PlayTunes ;
      GameWindow^.Settings^.Speed     := Settings^.Speed     ;
    end;
    Valid:= true;
  end
  else Valid:= false;
end;

procedure   T_SettingsDlg.HandleEvent(var V_Event: TEvent);
begin with V_Event do begin
  Inherited HandleEvent(V_Event);
  case What of
    evCommand: case Command of
      cm_DflSettings: begin DoReset; ClearEvent(V_Event); end;
    end;
  end;
end; end;

procedure   T_SettingsDlg.DoReset;
begin
  Dispose(Settings,Done);
  New(Settings,Init(C_DflKeyCount,C_DflFixedStart,C_DflPlayTunes,C_DflSpeed));
  ImportData;
end;


{ --- T_MenuBar --- }

procedure   T_MenuBar.Draw;
var R: TRect;
begin
  Inherited Draw;
  GetExtent(R);
end;


{ --- T_StatusLine methods --- }

procedure   T_StatusLine.Draw;
var R: TRect;
begin
  Inherited Draw;
  GetExtent(R);
  WriteStr(R.B.X-13, R.A.Y, C_ProgIdent, 2);
end;


{ --- T_Application --- }

Constructor T_Application.Init;
var R: TRect;
begin
  Inherited Init;
  DeskTop^.GetExtent(R);
  New(GameWindow,Init(R,C_ProgTitle)); InsertWindow(GameWindow);
  GameWindow^.StartTune;
end;

Destructor T_Application.Done;
begin
  Dispose(GameWindow,Done);
  Inherited Done;
end;

procedure   T_Application.InitMenuBar;
const hc0=hcNoContext; kb0=kbNoKey;
var R: TRect;
begin
  GetExtent(R); R.B.Y:= R.A.Y + 1;
  MenuBar:= New(P_MenuBar,Init(R,NewMenu(
    NewSubMenu('~G~ame', hc0,NewMenu(
      NewItem('~S~tart game','F2'   ,kbF2  ,cm_GameStart ,hc0,
      NewItem('~O~ptions'   ,'F3'   ,kbF3  ,cm_GameOptions,hc0,
      NewLine(
      NewItem('E~x~it'      ,'Alt+X',kbAltX,cm_GameExit ,hc0,
    nil))))),
    NewSubMenu('~H~elp', hc0,NewMenu(
      NewItem('~E~xplain'   ,'F1'   ,kbF1  ,cm_HelpExplain, hc0,
      NewItem('~A~bout'     ,''     ,kb0   ,cm_HelpAbout  , hc0,
    nil))),
  nil)))));
end;


procedure   T_Application.InitStatusLine;
var R: TRect;
begin
  GetExtent(R); R.A.Y:= R.B.Y-1;
  StatusLine:= New(P_StatusLine,Init(R,NewStatusDef($0000,$FFFF,
    NewStatusKey('~F1~ Help'      ,kbF1   ,cm_HelpExplain,
    NewStatusKey('~F2~ Start game',kbF2   ,cm_GameStart,
    NewStatusKey('~F3~ Options'   ,kbF3   ,cm_GameOptions,
    NewStatusKey('~F10~ Menu'     ,kbF10  ,cm_MainMenu,
    NewStatusKey('~Alt+X~ Exit'   ,kbAltX ,cm_GameExit,
    StdStatusKeys(nil)))))),
  nil)));
end;


procedure   T_Application.HandleEvent(var V_Event: TEvent);
begin with V_Event do begin
  Inherited HandleEvent(V_Event);
  case What of
    evCommand: case Command of
      cm_HelpExplain : begin DoHelpExplain ; ClearEvent(V_Event); end;
      cm_HelpAbout   : begin DoHelpAbout   ; ClearEvent(V_Event); end;
    end;
  end;
end; end;

procedure   T_Application.DoHelpExplain;
begin
  ExecuteDialog(New(P_MessageDlg,Init('Help',3,
    'Chose "Start game". Listen well, then play the same melody, '+
    'using the mouse or the numerical keys.')),nil);
end;

procedure   T_Application.DoHelpAbout;
begin
  ExecuteDialog(New(P_MessageDlg,Init('About this game',6,
    #3+C_ProgIdent+#13 +
    #3'Borland Pascal + Turbo Vision'#13#13 +
    #3+C_Copyright+#13 +
    #3+C_Email+#13 +
    #3+C_URL)),nil);
end;


{ --- Main program --- }

begin
  Randomize;
  GV_HomeDir:= GetHomeDir;
  StreamRegistration;
  Application:= New(P_Application,Init);
  Application^.Run;
  Dispose(Application,Done);
end.
