/*
 * MyMan - a public-domain text-terminal video game (like the yellow fellow)
 * by "bsittler" <bsittler@iname.com>
 */

#include <ctype.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <curses.h>

#ifdef __CYGWIN32__
#ifndef USE_GETOPT_H
/* Cygnus GNU-Win32 doesn't declare optarg in <unistd.h> as of B19 */
#define USE_GETOPT_H 1
#endif
#endif

#ifdef __sun__
#ifdef __unix__
/* SunOS sux */
extern char *optarg;
extern int optind, opterr;
#endif
#endif

#ifdef __PDCURSES__
#ifndef USE_PALETTE
/* PDCurses doesn't implement init_color as of release 2.2 */
#define USE_PALETTE 0
#endif
#endif

#ifndef USE_GETOPT_H
#define USE_GETOPT_H 0
#endif

#if USE_GETOPT_H
#include <getopt.h>
#endif

#ifndef USE_KEYPAD
#ifdef KEY_LEFT
#define USE_KEYPAD 1
#else
#define USE_KEYPAD 0
#endif
#endif

#ifndef USE_ACS
#ifdef ACS_ULCORNER
#define USE_ACS 1
#else
#define USE_ACS 0
#endif
#endif

#ifndef SWAPDOTS
#define SWAPDOTS 0
#endif

#ifndef USE_ATTR
#ifdef A_BOLD
#define USE_ATTR 1
#else
#define USE_ATTR 0
#ifndef USE_COLOR
#define USE_COLOR 0
#endif
#endif
#endif

#ifndef SOUND
#define SOUND 0
#endif

#ifndef USE_COLOR
#ifdef COLOR_BLACK
#define USE_COLOR 1
#else
#define USE_COLOR 0
#endif
#endif

#ifndef COLORIZE
#define COLORIZE 1
#endif

#ifndef USE_PALETTE
#define USE_PALETTE USE_COLOR
#endif

#ifndef USE_RAW
#define USE_RAW 0
#endif

#ifndef TILEFILE
#define TILEFILE "tile5x2.txt"
#endif
#ifdef BUILTIN_TILE
#undef TILEFILE
#define TILEFILE 0
#define tile_storage extern
tile_storage char *builtin_tilefile;
#else
#define tile_storage static
#define builtin_tilefile TILEFILE
#endif
#ifndef TILE_W
#define TILE_W 5
#endif
#ifndef TILE_H
#define TILE_H 2
#endif

tile_storage char tile[256 * TILE_H * TILE_W];
tile_storage int tile_used[256];
#if USE_COLOR
tile_storage int tile_color[256];
#endif

#ifndef STILEFILE
#define STILEFILE "stile7x3.txt"
#endif
#ifdef BUILTIN_STILE
#undef STILEFILE
#define STILEFILE 0
#define stile_storage extern
tile_storage char *builtin_stilefile;
#else
#define stile_storage static
#define builtin_stilefile STILEFILE
#endif
#ifndef STILE_W
#define STILE_W 7
#endif
#ifndef STILE_H
#define STILE_H 3
#endif

#ifndef VISIBLE_EYES
#define VISIBLE_EYES 1
#endif

stile_storage char stile[256 * STILE_H * STILE_W];
stile_storage int stile_used[256];
#if USE_COLOR
stile_storage int stile_color[256];
#endif

#if USE_ACS
chtype char_trans[256];

void init_trans(int swapdots) {
  int i;

  for (i = 0; i < 256; i ++)
    if (isprint(i))
      char_trans[i] = i;
    else {
      char_trans[i] = '?';
#if USE_ATTR
#ifdef A_STANDOUT
      char_trans[8] |= A_STANDOUT;
#endif
#endif
    }
  char_trans[201] = 
#ifdef ACS_BDDB
     ACS_BDDB;
#endif
  char_trans[218] =
#ifdef ACS_BSSB
    ACS_BSSB;
#else
#ifdef ACS_ULCORNER
    ACS_ULCORNER;
#else
    '+';
#endif
#endif
  char_trans[200] =
#ifdef ACS_DDBB
     ACS_DDBB;
#endif
  char_trans[192] =
#ifdef ACS_SSBB
     ACS_SSBB;
#else
#ifdef ACS_LLCORNER
    ACS_LLCORNER;
#else
    '+';
#endif
#endif
  char_trans[187] =
#ifdef ACS_BBDD
     ACS_BBDD;
#endif
  char_trans[191] =
#ifdef ACS_BBSS
     ACS_BBSS;
#else
#ifdef ACS_URCORNER
    ACS_URCORNER;
#else
    '+';
#endif
#endif
  char_trans[188] =
#ifdef ACS_DBBD
     ACS_DBBD;
#endif
  char_trans[217] =
#ifdef ACS_SBBS
     ACS_SBBS;
#else
#ifdef ACS_LRCORNER
    ACS_LRCORNER;
#else
    '+';
#endif
#endif
  char_trans[185] =
#ifdef ACS_DBDD
     ACS_DBDD;
#endif
  char_trans[181] =
#ifdef ACS_SBSD
    ACS_SBSD;
#endif
  char_trans[182] =
#ifdef ACS_DBDS
     ACS_DBDS;
#endif
  char_trans[180] =
#ifdef ACS_SBSS
     ACS_SBSS;
#else
#ifdef ACS_RTEE
    ACS_RTEE;
#else
    '+';
#endif
#endif
  char_trans[204] =
#ifdef ACS_DDDB
     ACS_DDDB;
#endif
  char_trans[198] =
#ifdef ACS_SDSB
    ACS_SDSB;
#endif
  char_trans[199] =
#ifdef ACS_DSDB
     ACS_DSDB;
#endif
  char_trans[195] =
#ifdef ACS_SSSB
     ACS_SSSB;
#else
#ifdef ACS_LTEE
    ACS_LTEE;
#else
    '+';
#endif
#endif
  char_trans[202] =
#ifdef ACS_DDBD
     ACS_DDBD;
#endif
  char_trans[207] =
#ifdef ACS_SDBD
     ACS_SDBD;
#endif
  char_trans[208] =
#ifdef ACS_DSBS
    ACS_DSBS;
#endif
  char_trans[193] =
#ifdef ACS_SSBS
     ACS_SSBS;
#else
#ifdef ACS_BTEE
    ACS_BTEE;
#else
    '+';
#endif
#endif
  char_trans[203] =
#ifdef ACS_BDDD
     ACS_BDDD;
#endif
  char_trans[209] =
#ifdef ACS_BDSD
     ACS_BDSD;
#endif
  char_trans[210] =
#ifdef ACS_BSDS
    ACS_BSDS;
#endif
  char_trans[194] =
#ifdef ACS_BSSS
     ACS_BSSS;
#else
#ifdef ACS_TTEE
    ACS_TTEE;
#else
    '+';
#endif
#endif
  char_trans[213] = 
#ifdef ACS_BDSS
     ACS_BDSS;
#else
#ifdef ACS_BDSB
     ACS_BDSB;
#else
     char_trans[194];
#endif
#endif
  char_trans[214] = 
#ifdef ACS_SSDB
     ACS_SSDB;
#else
#ifdef ACS_BSDB
     ACS_BSDB;
#else
     char_trans[195];
#endif
#endif
  char_trans[212] =
#ifdef ACS_SDBS
     ACS_SDBS;
#else
#ifdef ACS_SDBB
     ACS_SDBB;
#else
     char_trans[193];
#endif
#endif
  char_trans[211] =
#ifdef ACS_DSSB
     ACS_DSSB;
#else
#ifdef ACS_DSBB
     ACS_DSBB;
#else
     char_trans[195];
#endif
#endif
  char_trans[184] =
#ifdef ACS_BSSD
     ACS_BSSD;
#else
#ifdef ACS_BBSD
     ACS_BBSD;
#else
     char_trans[194];
#endif
#endif
  char_trans[183] =
#ifdef ACS_SBDS
     ACS_SBDS;
#else
#ifdef ACS_BBDS
     ACS_BBDS;
#else
     char_trans[180];
#endif
#endif
  char_trans[190] =
#ifdef ACS_SSBD
     ACS_SSBD;
#else
#ifdef ACS_SBBD
     ACS_SBBD;
#else
     char_trans[193];
#endif
#endif
  char_trans[189] =
#ifdef ACS_DBSS
     ACS_DBSS;
#else
#ifdef ACS_DBBS
     ACS_DBBS;
#else
     char_trans[180];
#endif
#endif
  char_trans[205] =
#ifdef ACS_BDBD
    ACS_BDBD;
#endif
  char_trans[196] =
#ifdef ACS_BSBS
    ACS_BSBS;
#else
#ifdef ACS_HLINE
    ACS_HLINE;
#else
    '-';
#endif
#endif
  char_trans[186] =
#ifdef ACS_DBDB
    ACS_DBDB;
#endif
  char_trans[179] =
#ifdef ACS_SBSB
    ACS_SBSB;
#else
#ifdef ACS_VLINE
    ACS_VLINE;
#else
    '|';
#endif
#endif
  char_trans[206] =
#ifdef ACS_DDDD
    ACS_DDDD;
#endif
  char_trans[215] =
#ifdef ACS_SDSD
    ACS_SDSD;
#endif
  char_trans[216] =
#ifdef ACS_DSDS
    ACS_DSDS;
#endif
  char_trans[197] =
#ifdef ACS_SSSS
    ACS_SSSS;
#else
#ifdef ACS_PLUS
    ACS_PLUS;
#endif
#endif
  char_trans[4] =
#ifdef ACS_DIAMOND
    ACS_DIAMOND;
#else
    '+';
#endif
  char_trans[248] =
#ifdef ACS_DEGREE
    ACS_DEGREE;
#else
    '\'';
#endif
  char_trans[241] =
#ifdef  ACS_PLMINUS
    ACS_PLMINUS;
#else
    '#';
#endif
  char_trans[7] =
#ifdef ACS_BBBB
    ACS_BBBB;
#endif
  char_trans[8] =
  char_trans[9] =
  char_trans[254] =
#ifdef ACS_BULLET
    ACS_BULLET;
#else
    'o';
#endif
#if USE_ATTR
#ifdef A_REVERSE
  char_trans[8] |= A_REVERSE;
#else
#ifdef A_STANDOUT
  char_trans[8] |= A_STANDOUT;
#endif
#endif
#endif
  char_trans[25] =
  char_trans[31] =
#ifdef ACS_DARROW
    ACS_DARROW;
#else
    'v';
#endif
  char_trans[24] =
  char_trans[30] =
#ifdef ACS_UARROW
    ACS_UARROW;
#else
    '^';
#endif
  char_trans[15] =
#ifdef ACS_LANTERN
    ACS_LANTERN;
#endif
  char_trans[176] =
#ifdef ACS_BOARD
    ACS_BOARD;
#endif
  char_trans[177] =
  char_trans[178] =
#ifdef ACS_CKBOARD
    ACS_CKBOARD;
#endif
  char_trans[10] =
  char_trans[219] =
#ifdef ACS_BLOCK
    ACS_BLOCK;
#else
    '#';
#endif
  char_trans[27] =
  char_trans[17] =
#ifdef ACS_LARROW
    ACS_LARROW;
#endif
  char_trans[174] =
  char_trans[243] =
#ifdef ACS_LEQUAL
    ACS_LEQUAL;
#else
    '<';
#endif
  char_trans[26] =
  char_trans[16] =
#ifdef ACS_RARROW
    ACS_RARROW;
#endif
  char_trans[175] =
  char_trans[242] =
#ifdef ACS_GEQUAL
    ACS_GEQUAL;
#else
    '>';
#endif
  char_trans[227] =
#ifdef ACS_PI
    ACS_PI;
#else
    '*';
#endif
  char_trans[156] =
#ifdef ACS_STERLING
    ACS_STERLING;
#else
    'f';
#endif
  char_trans[0] = ' ';
  char_trans[240] = '=';
  char_trans[247] = '=';
  if (swapdots) {
    char_trans[249] =
    char_trans[250] = char_trans[254];
    char_trans[254] = 'o';
  } else {
    char_trans[249] =
    char_trans[250] = '.';
  }
  char_trans[255] = ' ';
}

#define CHAR_TRANS(i) (char_trans[(i) & 0xff])

#else

#define CHAR_TRANS(i) (i)

#endif

#if USE_RAW
int my_addch(chtype c) {
  unsigned char b;

  if (c)
    c ^=
      (b = c & 255);
  else {
    b = '0';
#if USE_ATTR
#ifdef A_STANDOUT
    c = A_STANDOUT;
#endif
#endif
  }
  attrset(c);
  return addnstr(&b, 1);
}
#else
#define my_addch addch
#endif

#ifndef MAZEFILE
#define MAZEFILE "maze.txt"
#endif
#ifdef BUILTIN_MAZE
#undef MAZEFILE
#define MAZEFILE 0
#define maze_storage extern
tile_storage char *builtin_mazefile;
#else
#define maze_storage static
#define builtin_mazefile MAZEFILE
#endif
#ifndef MAZE_W
#define MAZE_W 28
#endif
#ifndef MAZE_H
#define MAZE_H 31
#endif
#ifndef MSGLEN
#define MSGLEN 10
#endif
#ifndef DOTS
#define DOTS 244
#endif
#ifndef DOTS_FRUIT1
#define DOTS_FRUIT1 70
#endif
#ifndef DOTS_FRUIT2
#define DOTS_FRUIT2 140
#endif

maze_storage unsigned char maze[MAZE_H][MAZE_W + 1];
static unsigned char blank_maze[MAZE_H][MAZE_W + 1];
unsigned char msgbuf[MSGLEN], msgbuf2[MSGLEN];

#define SPRITES 16

unsigned char sprite[SPRITES];
static unsigned char sprite_frame[SPRITES];
int sprite_x[SPRITES], sprite_y[SPRITES];
static int sprite_used[SPRITES];
static int sprite_timer[SPRITES];
#if USE_COLOR
static int sprite_color[SPRITES];
#endif

#ifndef GHOSTS
#define GHOSTS 4
#endif

int ghost_dir[GHOSTS], ghost_mem[GHOSTS], ghost_man[GHOSTS], ghost_timer[GHOSTS];
static unsigned char home_dir[GHOSTS][MAZE_H][MAZE_W + 1];

#define WHOSE_HOME_DIR(r,c) (home_dir[PINKY][(r)][(c)] ? PINKY \
			: home_dir[INKY][(r)][(c)] ? INKY \
			: home_dir[CLYDE][(r)][(c)] ? CLYDE \
			: BLINKY)

#define HOME_DIR(r,c) (home_dir[WHOSE_HOME_DIR((r),(c))][(r)][(c)])

/* sprite numbers */
#define INKY 0
#define BLINKY 1
#define PINKY 2
#define CLYDE 3
#define MEAN(ghost) ((ghost) + 4)
#define MYMAN 8
#define BLUE(ghost) ((ghost) + 9)
#define FRUIT 13
#define GHOST_SCORE 14
#define FRUIT_SCORE 15

#define STILE_FRUIT 0
#define STILE_MEAN 8
#define STILE_EYES 10
#define STILE_BLUE 14
#define STILE_MYMAN 16
#define STILE_FRUIT_SCORE 41
#define STILE_200 49
#define STILE_400 50
#define STILE_800 51
#define STILE_1600 52
#define STILE_WHITE 53

#define XTILE(x) ((x) / TILE_W)
#define YTILE(y) ((y) / TILE_H)

#define PIX_W ((MAZE_W + 1) * TILE_W)
#define PIX_H (MAZE_H * TILE_H)

#define XPIXWRAP(x) (((x) + PIX_W) % PIX_W)
#define YPIXWRAP(y) (((y) + PIX_H) % PIX_H)

#define XWRAP(x) (((x) + MAZE_W + 1) % (MAZE_W + 1))
#define YWRAP(y) (((y) + MAZE_H) % MAZE_H)

#define ISDOT(c) (((c) == 249) || ((c) == '.'))
#define ISPELLET(c) (((c) == 254) || ((c) == 'o'))
#define ISOPEN(c) (((c) == ' ') || ISDOT(c) || ISPELLET(c) || \
		   ((c) == '!') || (((c) >= 'A') && ((c) <= 'Z')))
#define ISDOOR(c) (((c) == '=') || (c == ':') || \
                   ((c) == 240) || (c == 255))

#define NOTTOP(y) (y >= (TILE_H / 2))
#define NOTBOTTOM(y) (y <= (TILE_H / 2))
#define NOTLEFT(x) (x >= (TILE_W / 2))
#define NOTRIGHT(x) (x <= (TILE_W / 2))

#define UP 1
#define LEFT 2
#define DOWN 3
#define RIGHT 4

#define DIRWRAP(dir) (((dir) + 3) % 4 + 1)

#define XDIR(dir) (((dir) == RIGHT) - ((dir) == LEFT))
#define YDIR(dir) (((dir) == DOWN) - ((dir) == UP))

#define YLEAVING(dir, y) ( -(((dir) == UP) && ! NOTTOP(y)) + \
			   (((dir) == DOWN) && ! NOTBOTTOM(y)))
#define XLEAVING(dir, x) ( -(((dir) == LEFT) && ! NOTLEFT(x)) + \
			   (((dir) == RIGHT) && ! NOTRIGHT(x)))

#ifndef CMYMAN
#define CMYMAN (MAZE_W * 0.5)
#endif
#define XMYMAN (CMYMAN * TILE_W)

#ifndef RMYMAN
#define RMYMAN 23.5
#endif
#define YMYMAN (RMYMAN * TILE_H)

#ifndef CFRUIT
#define CFRUIT CMYMAN
#endif
#define XFRUIT (CFRUIT * TILE_W)

#ifndef RFRUIT
#define RFRUIT 17.5
#endif
#define YFRUIT (RFRUIT * TILE_H)

#ifndef RGHOST
#define RGHOST (RFRUIT - 3.0)
#endif
#define YGHOST (RGHOST * TILE_H)

#ifndef COGHOST
#define COGHOST 2
#endif

#ifndef ROGHOST
#define ROGHOST 3
#endif

#ifndef RTOP
#define RTOP (RFRUIT - 5.0)
#endif
#define YTOP (RTOP * TILE_H)

#ifndef CMSG
#define CMSG 9
#endif

#ifndef RMSG
#define RMSG 17
#endif

#ifndef CMSG2
#define CMSG2 9
#endif

#ifndef RMSG2
#define RMSG2 11
#endif

#ifndef PLAYER1
#define PLAYER1  "PLAYER ONE"
#endif

#ifndef PLAYER2
#define PLAYER2  "PLAYER TWO"
#endif

#define PLAYER(n) \
     (((n) == 1) \
      ? PLAYER1 \
      : PLAYER2)

#ifndef READY
#define READY    "  READY!  "
#endif

#ifndef GAMEOVER
#define GAMEOVER "GAME  OVER"
#endif

#ifndef PAUSE
#define PAUSE " - PAUSED - "
#endif

#define BONUS(n) \
     (((n) < 4) \
      ? ((n) - 1) \
      : (((n) < 15) \
	 ? ((n) + 1) / 2 \
	 : 7))

int bonus_score[8] = {100, 300, 500, 700, 1000, 2000, 3000, 5000};

#ifndef DELAY
#define DELAY 25000
#endif

#ifndef SPEEDUP
#define SPEEDUP 1000
#endif

#define TWOSECS    (delay ? (((2000000 / delay) > 2) ? (2000000 / delay) : 2000000) : 80)
#define ONESEC     (TWOSECS / 2)
#define FRUITLIFE  (TWOSECS * 15 / 4)
#define DEATHSHIFT 3
#define DEATHDELAY (1 << (DEATHSHIFT + 2))
#define MEMDELAY   (3 * TWOSECS)

int readfont(char *fontfile, int w, int h, unsigned char *font, int *used
#if USE_COLOR
             , int *color
#endif
             ) {
  FILE *infile;
  int c, i, j, k;
  int rw, rh;
  char X;

  for (i = 0; i < 256; i ++) {
    used[i] = 0;
#if USE_COLOR
    color[i] = 0;
#endif
  }
  if (! (infile = fopen(fontfile, "rb"))) {
    perror(fontfile);
    return 1;
  }
  if ((fscanf(infile, "%d%c%d", &rw, &X, &rh) != 3) ||
      (tolower(X) != 'x')) {
    fprintf(stderr, "%s: can't find a dimension specification WxH\n",
	    fontfile);
    return 1;
  }
  if ((rw != w) || (rh != h)) {
    fprintf(stderr, "%s: dimension specification %dx%d does't match %dx%d\n",
	    fontfile, rw, rh, w, h);
    return 1;
  }
  while (! feof(infile)) {
    if (fscanf(infile, "%x", &i) != 1) {
      if (! feof(infile)) {
	fprintf(stderr, "%s: can't find an index\n",
		fontfile);
	return 1;
      }
      continue;
    }
    if ((i < 0) || (i > 255)) {
      fprintf(stderr, "%s: invalid index %0.2X ignored\n", fontfile, i);
      continue;
    }
    if (used[i])
      fprintf(stderr, "%s: duplicate definition for %0.2X\n", fontfile, i);
    used[i] = 1;
#if USE_COLOR
    if ((c = fgetc(infile)) == '~') {
      if (fscanf(infile, "%x", &c) != 1) {
        if (! feof(infile)) {
	  fprintf(stderr, "%s: can't find a color for index %0.2X\n",
		  fontfile, i);
          return 1;
        }
        continue;
      }
      if ((c < 0) || (c > 15))
        fprintf(stderr, "%s: invalid color %0.2X ignored\n", fontfile, c);
      else
        color[i] = c;
    } else
      ungetc(c, infile);
#endif
    for (j = 0; j < h; j ++)
      for (k = 0; k < w; k ++)
	font[(i * h + j) * w + k] = ' ';
    for (j = 0; j < h; j ++) {
      while ((c = fgetc(infile)) != ':')
	if (c == EOF) {
	  if (! feof(infile))
	    perror(fontfile);
	  else
	    fprintf(stderr, "%s: premature EOF in index %0.2X\n",
		    fontfile, i);
	  return 1;
	}
      if ((c = fgetc(infile)) == EOF)
	if (! feof(infile)) {
	  perror(fontfile);
	  return 1;
	}
      for (k = 0;
	   (k < w) &&
	     (c != '\v') && (c != '\f') &&
	     (c != '\n') && (c != '\r') && ! feof(infile);
	   k ++) {
	font[(i * h + j) * w + k] = c;
	if ((c = fgetc(infile)) == EOF) {
	  if (feof(infile))
	    continue;
	  perror(fontfile);
	  return 1;
	}
      }
      while ((c != '\v') && (c != '\f') &&
	     (c != '\n') && (c != '\r') && ! feof(infile)) {
	if (c == EOF) {
	  if (feof(infile))
	    continue;
	  perror(fontfile);
	  return 1;
	}
	c = fgetc(infile);
      }
    }
  }
  fclose(infile);
  return 0;
}

#define collide(s1, s2) ((XTILE(sprite_x[s1] - STILE_W / 2) == XTILE(sprite_x[s2] - STILE_W / 2)) \
			 && (YTILE(sprite_y[s1] - STILE_H / 2) == YTILE(sprite_y[s2] - STILE_H / 2)))

#if USE_COLOR

chtype pen[16];

#define trans_color(i) \
((i == 0) ? COLOR_BLACK \
 : (i == 1) ? COLOR_BLUE \
 : (i == 2) ? COLOR_GREEN \
 : (i == 3) ? COLOR_CYAN \
 : (i == 4) ? COLOR_RED \
 : (i == 5) ? COLOR_MAGENTA \
 : (i == 6) ? COLOR_YELLOW \
 : COLOR_WHITE)

#if USE_ATTR
#ifndef PEN_BRIGHT
#ifdef A_BOLD
#define PEN_BRIGHT A_BOLD
#else
#define PEN_BRIGHT 0
#endif
#endif
#ifndef PEN_DIM
#ifdef A_DIM
#define PEN_DIM A_DIM
#else
#define PEN_DIM 0
#endif
#endif
#endif

#if USE_PALETTE
/* this palette is totally untested */
short pen_pal[16][3] = {
/*{ red,green, blue} */
  {   0,    0,    0}, /* 0: black */
  {   0,    0,  867}, /* 1: blue */
  {   0,  718,    0}, /* 2: green (peach stem) */
  {   0,  867,  867}, /* 3: cyan */
  { 867,  589,  277}, /* 4: brown (apple/cherry stem) */
  {1000,  128,  589}, /* 5: magenta (mypal trim?) */
  {1000,  718,  277}, /* 6: salmon (clyde, peach, PUSH START BUTTON) */
  {1000,  718,  589}, /* 7: light yellow (dot, blue face, BONUS MYMAN FOR x Pts) */
  { 400,  400,  400}, /* 8: grey */
  { 128,  128,  867}, /* 9: light blue (wall, blue ghost) */
  { 589, 1000,    0}, /* A: light green (mypal?, super dot?) */
  {   0, 1000,  867}, /* B: light cyan (inky, key top, 1 PLAYER ONLY, ghost scores) */
  {1000,    0,    0}, /* C: light red (blinky, apple/cherry, GAME  OVER) */
  {1000,  718,  867}, /* D: pink (pinky, door, NEW MAN - X, fruit scores) */
  {1000, 1000,    0}, /* E: yellow (myman, ship) */
  { 867,  867,  867}, /* F: light grey (text, eye, apple/cherry shine, key, bell) */
};
static short old_pal[16][3];
#endif

static short old_pair[16][2];

void destroy_pen(void) {
  int i;

#if USE_PALETTE
  if (can_change_color() && (COLORS >= 16) && (COLOR_PAIRS >= 16)) {
    for (i = 0; i < 16; i ++)
      init_color(i, old_pal[i][0], old_pal[i][1], old_pal[i][2]);
  }
#endif
  for (i = 0; i < 16; i ++)
    if (i < COLOR_PAIRS)
      init_pair(i, old_pair[i][0], old_pair[i][1]);
}

void init_pen(void) {
  int i;

  for (i = 0; i < 16; i ++)
    if (i < COLOR_PAIRS)
      pair_content(i, old_pair[i], old_pair[i] + 1);
#if USE_PALETTE
  if (can_change_color() && (COLORS >= 16) && (COLOR_PAIRS >= 16)) {
    for (i = 0; i < 16; i ++)
      color_content(i, old_pal[i], old_pal[i] + 1, old_pal[i] + 2);
    for (i = 0; i < 16; i ++) {
      init_color(i,
		 pen_pal[i][0],
		 pen_pal[i][1],
		 pen_pal[i][2]);
      init_pair(i, i, 0);
      pen[i] = COLOR_PAIR(i);
    }
  } else
#endif
    {
      for (i = 0; i < 8; i ++) {
	if (i && (i < COLOR_PAIRS) && 
	    (init_pair(i, trans_color(i), COLOR_BLACK) != ERR)) {
	  pen[i] = COLOR_PAIR(i);
	  pen[8 + i] = COLOR_PAIR(i) | PEN_BRIGHT;
	} else {
	  pen[i] = COLOR_PAIR(0);
	  pen[8 + i] = pen[i] | ((i == 8) ? PEN_DIM : PEN_BRIGHT);
	}
      }
    }
}

#endif

void escape(unsigned char *s, int len) {
  int c;
  int i;

  for (i = 0; i < len; i ++)
    if (isprint((c = s[i]))) {
      if ((c == '"') || (c == '\\'))
	putchar('\\');
      putchar(c);
    } else {
      if (((i + 1 == len) || ! isdigit(s[i + 1])) && (c == '\0'))
	printf("\\0");
      else
	printf("\\%3.3o", c);
    }
}

void writefont(char *file, char *prefix, int w, int h, unsigned char *font, int *used
#if USE_COLOR
	      , int *color
#endif
             ) {
  int c, i, j;

  printf("char *builtin_%sfile = \"", prefix);
  escape(file, strlen(file));
  printf("\";\n");
  printf("unsigned char %s[256 * %d * %d] = {", prefix, h, w);
  for (c = 0; c < 256; c ++) {
    printf("\n/* 0x%2.2X */", c);
    for (i = 0; i < h; i ++) {
      printf("\n ");
      for (j = 0; j < w; j ++) {
	printf(" ");
	printf("0x%2.2X", font[(c * h + i) * w + j]);
	if (((c + 1) < 256) || ((i + 1) < h) || ((j + 1) < w))
	  printf(",");
      }
    }
  }
  printf("};\n");
  printf("int %s_used[256] = {\n", prefix);
  for (c = 0; c < 256; c ++) {
    if (c && ! (c & 3))
      printf(",\n");
    else if (c)
      printf(", ");
    printf("/* 0x%2.2X */ %d", c, used[c]);
  }
  printf("};\n");
#if USE_COLOR
  printf("int %s_color[256] = {\n", prefix);
  for (c = 0; c < 256; c ++) {
    if (c && ! (c & 3))
      printf(",\n");
    else if (c)
      printf(", ");
    printf("/* 0x%2.2X */ 0x%X", c, color[c]);
  }
  printf("};\n");
#endif
}

int main(int argc, char **argv) {
  FILE *infile;
  int i, j, k, kk = ERR, c;
  int dir;
  int debug = 0;
  int nogame = 0;
  int sound = SOUND;
  int
    dump_maze = 0,
    dump_stile = 0,
    dump_tile = 0;
  char *tilefile = TILEFILE;
  char *stilefile = STILEFILE;
  char *mazefile = MAZEFILE;
  int
    level = 0,
    cycles = 0,
    score = 0,
    dots = 0,
    points = 0,
    lives = 3,
    earned = 0,
    dying = 0,
    dead = 0, deadpan = 0;
  int
    oldplayer = 0,
    player = 1;
  unsigned long winning = 1;
#if DELAY
  unsigned long delay = DELAY;
  unsigned long uli;
#endif
  unsigned long
    pellet_timer = 0,
    pellet_time = 7 * ONESEC;
  int pause_timer = 0;
  int munched = MYMAN;
#if USE_COLOR
  int colorize_p = 0;
  int colorize = 0;
#endif
#if USE_ATTR
  int underline = 0;
#endif
#if USE_ACS
  int swapdots = SWAPDOTS;
#endif

  for (i = 0; i < SPRITES; i ++) {
    sprite_used[i] = 0;
    sprite_frame[i] = '\0';
#if USE_COLOR
    sprite_color[i] = 0x7;
#endif
  }
#if USE_COLOR
  for (i = 0; i < 256; i ++) {
#ifndef BUILTIN_TILE
    tile_color[i] = 0x7;
#endif
#ifndef BUILTIN_STILE
    stile_color[i] = 0x7;
#endif
  }
#endif
  while ((i = getopt(argc, argv, "bcd:hm:nopqs:t:uMST")) != -1)
    switch (i) {
    case 'b':
      sound = 1;
      break;
    case 'q':
      sound = 0;
      break;
    case 'd':
      if (sscanf(optarg, "%lu", &uli) != 1) {
	fprintf(stderr,
		"%s: argument to -d must be an unsigned long integer.\n",
		*argv);
      }
#if DELAY
      delay = uli;
#else
      if (uli) {
	fprintf(stderr,
		"%s: compile with -DDELAY=1 to enable the -d option.\n",
		*argv);
	exit(1);
      }
#endif
      break;
    case 'h':
      printf("Usage: %s [options]\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n%s\n",
	     *argv,
	     "-h \tdisplay this help and exit",
             "-b \tenable sounds",
             "-q \tdisable sounds",
#if USE_COLOR
	     "-c \tenable color support",
#else
	     "-c \tenable color support (must recompile first)",
#endif
	     "-n \tdisable color support",
#if USE_ACS
	     "-o \tuse 'o' for power pellets and ACS_BULLET for dots",
#else
	     "-o \tuse 'o' for power pellets and ACS_BULLET for dots (must recompile first)",
#endif
#if USE_ACS
	     "-p \tuse ACS_BULLET for power pellets and '.' for dots",
#else
	     "-p \tuse ACS_BULLET for power pellets and '.' for dots (must recompile first)",
#endif
#if DELAY
	     "-d NUM \tdelay NUM microseconds/refresh",
#else
	     "-d NUM \tdelay NUM microseconds/refresh (must recompile first)",
#endif
#if USE_ATTR
	     "-u \tuse the underline attribute for maze walls",
#else
	     "-u \tuse the underline attribute for maze walls (must recompile first)",
#endif
	     "-m FILE \tuse the maze in FILE",
	     "-s FILE \tuse sprites from FILE",
	     "-t FILE \tuse tiles from FILE",
	     "-M \twrite the maze to stdout in C format and exit",
	     "-S \twrite sprites to stdout in C format and exit",
	     "-T \twrite tiles to stdout in C format and exit");
      printf("Defaults:%s%s%s%s -d %lu -m \"",
	     sound ? " -b" : " -q",
#if USE_COLOR
	     colorize_p ? (colorize ? " -c" : " -n") : "",
#else
	     " -n",
#endif
#if USE_ACS
	     swapdots ? " -o" : " -p",
#else
	     "",
#endif
#if USE_ATTR
	     underline ? " -u" : "",
#else
	     "",
#endif
#if DELAY
	     delay ? delay :
#endif
	     0);
      if (mazefile)
	escape(mazefile, strlen(mazefile));
      else {
	printf("(");
	escape(builtin_mazefile, strlen(builtin_mazefile));
	printf(")");
      }
      printf("\" -s \"");
      if (stilefile)
	escape(stilefile, strlen(stilefile));
      else {
	printf("(");
	escape(builtin_stilefile, strlen(builtin_stilefile));
	printf(")");
      }
      printf("\" -t \"");
      if (tilefile)
	escape(tilefile, strlen(tilefile));
      else {
	printf("(");
	escape(builtin_tilefile, strlen(builtin_tilefile));
	printf(")");
      }
      printf("\"\n");
      exit(0);
      break;
    case 'u':
#if USE_ATTR
      underline = 1;
#else
      fprintf(stderr,
	      "%s: compile with -DUSE_ATTR=1 to enable the -u option.\n",
	      *argv);
      exit(1);
#endif
      break;
    case 'M':
      dump_maze = 1;
      nogame = 1;
      break;
    case 'S':
      dump_stile = 1;
      nogame = 1;
      break;
    case 'T':
      dump_tile = 1;
      nogame = 1;
      break;
    case 'm':
#ifdef BUILTIN_MAZE
      if ((*optarg == '(') &&
	  (strlen(optarg) == (strlen(builtin_mazefile) + 2)) &&
	  (optarg[strlen(optarg) - 1] == ')') &&
	  ! strncmp(optarg + 1, builtin_mazefile, strlen(builtin_mazefile))) {
	mazefile = 0;
	break;
      }
#endif
      mazefile = optarg;
      break;
    case 'n':
#if USE_COLOR
      colorize_p = 1;
      colorize = 0;
#endif
      break;
    case 'o':
#if USE_ACS
      swapdots = 1;
#else
      fprintf(stderr,
	      "%s: compile with -DUSE_ACS=1 to enable output character translation.\n",
	      *argv);
      exit(1);
#endif
      break;
    case 'p':
#if USE_ACS
      swapdots = 0;
#else
      fprintf(stderr,
	      "%s: compile with -DUSE_ACS=1 to enable output character translation.\n",
	      *argv);
      exit(1);
#endif
      break;
    case 'c':
#if USE_COLOR
      colorize_p = 1;
      colorize = 1;
#else
      fprintf(stderr,
	      "%s: compile with -DUSE_COLOR=1 to enable color support.\n",
	      *argv);
      exit(1);
#endif
      break;
    case 't':
#ifdef BUILTIN_TILE
      if ((*optarg == '(') &&
	  (strlen(optarg) == (strlen(builtin_tilefile) + 2)) &&
	  (optarg[strlen(optarg) - 1] == ')') &&
	  ! strncmp(optarg + 1, builtin_tilefile, strlen(builtin_tilefile))) {
	tilefile = 0;
	break;
      }
#endif
      tilefile = optarg;
      break;
    case 's':
#ifdef BUILTIN_STILE
      if ((*optarg == '(') &&
	  (strlen(optarg) == (strlen(builtin_stilefile) + 2)) &&
	  (optarg[strlen(optarg) - 1] == ')') &&
	  ! strncmp(optarg + 1, builtin_stilefile, strlen(builtin_stilefile))) {
	stilefile = 0;
	break;
      }
#endif
      stilefile = optarg;
      break;
    default:
      fprintf(stderr, "Usage: %s [-h] [options]\n", *argv);
      exit(2);
    }
  if (optind < argc) {
    fprintf(stderr, "Usage: %s [-h] [options]\n", *argv);
    exit(2);
  }
  if ((tilefile && readfont(tilefile, TILE_W, TILE_H, tile, tile_used
#if USE_COLOR
			    , tile_color
#endif
      )) ||
      (stilefile && readfont(stilefile, STILE_W, STILE_H, stile, stile_used
#if USE_COLOR
			     , stile_color
#endif
			     )))
    return 1;
  if (mazefile) {
    if (! (infile = fopen(mazefile, "rb"))) {
      perror(mazefile);
      return 1;
    }
    {
      int rn, rw, rh;
      char X;
      
      if ((fscanf(infile, "%d%d%c%d", &rn, &rw, &X, &rh) != 4) ||
	  (tolower(X) != 'x') || (fgetc(infile) != '\n'))  {
	fprintf(stderr, "%s: can't find a dimension specification N WxH\n",
		mazefile);
	return 1;
      }      
      if ((rw != MAZE_W) || (rh != MAZE_H) || (rn < 1)) {
	fprintf(stderr, "%s: dimension specification %d %dx%d does't match %d %dx%d\n",
		mazefile, rn, rw, rh, 1, MAZE_W, MAZE_H);
	return 1;
      }
    }
    for (i = 0; i < MAZE_H; i ++) {
      for (j = 0; j < MAZE_W; j ++)
	if ((c = fgetc(infile)) == EOF) {
	  if (feof(infile))
	    fprintf(stderr, "%s: premature EOF\n", mazefile);
	  else
	    perror(mazefile);
	  return 1;
	} else if ((c == '\r') || (c == '\n') || (c == '\f') || (c == '\v'))
	  j --;
	else
	  maze[i][j] = c;
      maze[i][MAZE_W] = c;
    }
    fclose(infile);
  }
  memcpy((void *)blank_maze,
	 (void *)maze,
	 (MAZE_W + 1) * MAZE_H * sizeof(unsigned char));
  sprite[FRUIT] = STILE_FRUIT;
  sprite[FRUIT_SCORE] = STILE_FRUIT_SCORE;
  sprite_frame[FRUIT] =
    sprite_frame[FRUIT_SCORE] = BONUS(level);
  sprite_x[FRUIT] =
    sprite_x[FRUIT_SCORE] = XFRUIT;
  sprite_y[FRUIT] =
    sprite_y[FRUIT_SCORE] = YFRUIT;

  sprite[GHOST_SCORE] = STILE_200;
  sprite_frame[GHOST_SCORE] = 0;

  sprite[MYMAN] = STILE_MYMAN + 4;
  dir = LEFT;
  sprite_frame[MYMAN] = 0;
  sprite_x[MYMAN] = XMYMAN;
  sprite_y[MYMAN] = YMYMAN;
  sprite_used[MYMAN] = 0;

#if USE_COLOR
  sprite_color[MYMAN] = 0xE;
#endif

  for (i = 0; i < GHOSTS; i++) {
    int mean, blue;
    
    mean = MEAN(i);
    blue = BLUE(i);
    sprite[i] = STILE_EYES;
    sprite[mean] = STILE_MEAN;
    sprite[blue] = STILE_BLUE;
    sprite_used[i] =
    sprite_used[mean] =
    sprite_used[blue] = 0;
    sprite_frame[i] =
    sprite_frame[mean] =
    sprite_frame[blue] = 0;
    ghost_mem[i] = 0;
    ghost_timer[i] = TWOSECS;
    ghost_man[i] = 0;
#if USE_COLOR
    sprite_color[i] = 0xF;
    sprite_color[blue] = 0x9;
    sprite_color[mean] = 0xA;
#endif
  }

#if USE_COLOR
  sprite_color[MEAN(INKY)] = 0xB;
  sprite_color[MEAN(BLINKY)] = 0xC;
  sprite_color[MEAN(PINKY)] = 0xD;
  sprite_color[MEAN(CLYDE)] = 0x6;
#endif
  
  if (dump_maze) {
    printf("char *builtin_mazefile = \"");
    escape(mazefile ? mazefile : builtin_mazefile,
	   strlen(mazefile ? mazefile : builtin_mazefile));
    printf("\";\n");
    printf("unsigned char maze[%d][%d + 1] = {\n", MAZE_H, MAZE_W);
    for (i = 0; i < MAZE_H; i ++) {
      printf("  \"");
      escape(maze[i], MAZE_W + 1);
      printf("\"%s\n",
	     ((i + 1) < MAZE_H) ? "," : "};");
    }
  }
  if (dump_stile)
    writefont(stilefile ? stilefile : builtin_stilefile,
	      "stile", STILE_W, STILE_H, stile, stile_used
#if USE_COLOR
	     , stile_color
#endif
	     );
  if (dump_tile)
    writefont(tilefile ? tilefile : builtin_tilefile,
	      "tile", TILE_W, TILE_H, tile, tile_used
#if USE_COLOR
	     , tile_color
#endif
	     );
  if (nogame)
    exit(0);

  initscr();
  clear();
  cbreak();
  noecho();
  nonl();
  nodelay(stdscr, TRUE);
  intrflush(stdscr, FALSE);
#if USE_ATTR
  attrset(0);
  curs_set(0);
#endif
#if USE_KEYPAD
  keypad(stdscr, TRUE);
#endif
  idlok(stdscr, TRUE);
  leaveok(stdscr, TRUE);
#if USE_ACS
  init_trans(swapdots);
#endif
#if USE_COLOR
#if COLORIZE
  if (! colorize_p) {
    colorize = has_colors();
    colorize_p = 1;
  }
#endif
  start_color();
  if (colorize)
    init_pen();
#endif
  while ((lives || dead || dying || pause_timer) &&
	 ((k = getch()) != 'q') && (k != 'Q'))
  {
    int s, x1, y1, line, col, xtile, ytile, xoff, yoff;
    int roff, coff;

#if DELAY
    if (delay)
      usleep(delay);
#endif
    if ((k == 'r') || (k == 'R')) {
      clearok(curscr, TRUE);
      wrefresh(stdscr);
      continue;
    }
#if USE_COLOR
    if ((k == 'c') || (k == 'C')) {
      if ((colorize = ! colorize))
	init_pen();
      else
	destroy_pen();
      attrset(0);
      clear();
      clearok(curscr, TRUE);
      continue;
    }
#endif
#if USE_ATTR
    if ((k == 'u') || (k == 'U')) {
      underline = ! underline;
      clearok(curscr, TRUE);
      continue;
    }
#endif
    if ((k == 's') || (k == 'S')) {
      sound = ! sound;
      continue;
    }
    if ((k == 'p') || (k == 'P') || (k == 27)) {
      standout();
      mvprintw(LINES / 2, (COLS - strlen(PAUSE)) / 2, PAUSE);
      standend();
      wrefresh(stdscr);
      while (getch() == ERR) {
#if DELAY
	if (delay)
	  usleep(delay);
#endif
      }
      continue;
    }
    if (k == ERR)
      k = kk;
    else
      kk = k;
    xoff = sprite_x[MYMAN] % TILE_W;
    yoff = sprite_y[MYMAN] % TILE_H;
    xtile = XTILE(sprite_x[MYMAN]);
    ytile = YTILE(sprite_y[MYMAN]);
    if (pause_timer && ! -- pause_timer) {
      sprite_used[GHOST_SCORE] = 0;
      sprite_frame[GHOST_SCORE] ++;
      if (sprite_used[MYMAN]) {
	memcpy(maze[RMSG] + CMSG, msgbuf, MSGLEN);
	if (sound && lives)
	  beep();
      } else {
	sprite_used[MYMAN] = 1;
	sprite_used[munched] = 1;
      }
    }
    if (winning || dying) {
      int reset = 0;
      static int init = 0;

      if (dying && ! -- dying) {
	reset = 1;
	-- lives;
      } else if ((! pause_timer) && winning && ! -- winning) {
	memcpy((void *)maze,
	       (void *)blank_maze,
	       (MAZE_W + 1) * MAZE_H * sizeof(unsigned char));
	dots = 0;
	pellet_timer = 0;
	reset = 1;
	++ level;
	sprite_frame[FRUIT] =
	  sprite_frame[FRUIT_SCORE] = BONUS(level);
	if (pellet_time > ONESEC)
	  pellet_time -= ONESEC;
	else
	  pellet_time = 0;
#if DELAY
	if (delay > SPEEDUP)
	  delay -= SPEEDUP;
#endif
      }
      if ((! init) && (winning || reset)) {
	init = 1;
	sprite_used[FRUIT] =
	  sprite_used[FRUIT_SCORE] =
	  sprite_used[GHOST_SCORE] = 0;
	for (i = 0; i < GHOSTS; i++) {
	  int mean, blue;

	  mean = MEAN(i);
	  blue = BLUE(i);
	  sprite_used[i] =
	  sprite_used[mean] =
	  sprite_used[blue] = 0;
	}
	sprite_frame[MYMAN] = 0;
      }
      if (reset) {
	memset((void *)home_dir, 0, sizeof(home_dir));
	for (i = 0; i < GHOSTS; i++) {
	  int mean, blue;

	  mean = MEAN(i);
	  blue = BLUE(i);
	  sprite_used[i] =
	  sprite_used[mean] =
	  sprite_used[blue] = 0;
	  sprite_frame[i] = (ghost_dir[i] = i + 1) - 1;
	  ghost_mem[i] = 0;
	  ghost_timer[i] = TWOSECS;
	  ghost_man[i] = 0;
	  sprite_x[mean] =
	    (sprite_x[i] = XMYMAN + COGHOST * TILE_W * ((i == CLYDE) - (i == INKY)));
	  sprite_y[mean] =
	    (sprite_y[i] = YGHOST - ROGHOST * TILE_H * (i == BLINKY));
	}
	kk = ERR;
	init = 0;
	cycles = 0;
	sprite_used[MYMAN] = 0;
	sprite_frame[MYMAN] = 0;
	sprite_x[MYMAN] = XMYMAN;
	sprite_y[MYMAN] = YMYMAN;
	dead =
	  deadpan = YMYMAN - YGHOST;
	if (dead < 2)
	  dead = 2;
	memcpy(msgbuf, maze[RMSG] + CMSG, MSGLEN);
	memcpy(maze[RMSG] + CMSG, lives ? READY : GAMEOVER, MSGLEN);
	memcpy(msgbuf2, maze[RMSG2] + CMSG2, MSGLEN);
	if (player != oldplayer) {
	  oldplayer = player;
	  pause_timer = ONESEC;
	  munched = MYMAN;
	  memcpy(maze[RMSG2] + CMSG2, PLAYER(player), MSGLEN);
	}
      }
      if (dying)
	sprite_frame[MYMAN] = ((DEATHDELAY - dying) >> DEATHSHIFT) % 4;
    } else if (dead && ! pause_timer) {
      if (deadpan)
	deadpan --;
      if (! -- dead) {
	for (i = 0; i < GHOSTS; i ++) {
	  int mean, blue;

	  mean = MEAN(i);
	  blue = BLUE(i);
	  sprite_used[i] = lives ? VISIBLE_EYES : 0;
	  sprite_used[mean] = lives ? 1 : 0;
	}
	sprite_used[MYMAN] = 1;
	pause_timer = ONESEC;
	munched = MYMAN;
	memcpy(maze[RMSG2] + CMSG2, msgbuf2, MSGLEN);
	sprite[MYMAN] = STILE_MYMAN + 4;
	dir = LEFT;
#if USE_COLOR
	if (! colorize)
#endif
#if USE_ATTR
	  if (underline)
	    clearok(curscr, TRUE);
#else
	;
#endif
      }
    } else {
      if (((score >= 10000) && ! earned) ||
	  ((score >= 50000) && (earned == 1))) {
	earned ++;
	lives ++;
      }
      if (ISPELLET((c = maze[ytile][xtile])) || ISDOT(c)) {
	maze[ytile][xtile] = ' ';
	sprite_frame[MYMAN] = 0;
	score += 10 + 40 * ISPELLET(c);
	if (ISPELLET(c)) {
	  points = 200;
	  sprite_frame[GHOST_SCORE] = 0;
	  pellet_timer = pellet_time + 1;
	  for (s = 0; s < GHOSTS; s++) {
	    int mean, blue;

	    blue = BLUE(s);
	    mean = MEAN(s);
	    sprite_frame[blue] = 0;
	    if (sprite_used[mean] || sprite_used[blue]) {
	      if (sprite_used[mean]) {
		sprite_x[blue] = sprite_x[s];
		sprite_y[blue] = sprite_y[s];
	      }
	      sprite_used[s] = 0;
	      sprite_used[mean] = 0;
	      sprite_used[blue] = 1;
	      sprite[blue] = STILE_BLUE;
	      ghost_dir[s] = DIRWRAP(ghost_dir[s] + 2);
	    }
	  }
	}
	if ((++ dots == DOTS_FRUIT1) || (dots == DOTS_FRUIT2)) {
	  sprite_used[FRUIT] = 1;
	  sprite_timer[FRUIT] = FRUITLIFE;
	  if (sound)
	    beep();
	} else if (dots == DOTS) {
	  munched = MYMAN;
	  winning = TWOSECS;
	  if (sound)
	    beep();
	}
      } else if (sprite_used[FRUIT] &&
		 collide(MYMAN, FRUIT)) {
	sprite_used[FRUIT] = 0;
	sprite_used[FRUIT_SCORE] = 1;
	sprite_timer[FRUIT_SCORE] = TWOSECS;
	score += bonus_score[sprite_frame[FRUIT]];
	sprite_frame[MYMAN] = 0;
	if (sound)
	  beep();
      }
      if ((k == 'w') || (k == 'W')) {
	dots = DOTS - 1;
	kk = ERR;
      } else if ((k == 'd') || (k == 'D')) {
	debug = ! debug;
	kk = ERR;
      } else if (((k == 'h') || (k == 'H') || (k == '4')
#if USE_KEYPAD
#ifdef KEY_LEFT
		  || (k == KEY_LEFT)
#endif
#endif
		  )
	  && ISOPEN(maze[ytile][XWRAP(xtile - NOTRIGHT(xoff))])) {
	dir = LEFT;
	sprite[MYMAN] = STILE_MYMAN + 4;
      } else if (((k == 'l') || (k == 'L') || (k == '6')
#if USE_KEYPAD
#ifdef KEY_RIGHT
		  || (k == KEY_RIGHT)
#endif
#endif
		  )
	       && ISOPEN(maze[ytile][XWRAP(xtile + NOTLEFT(xoff))])) {
	dir = RIGHT;
	sprite[MYMAN] = STILE_MYMAN + 12;
      } else if (((k == 'k') || (k == 'K') || (k == '8')
#if USE_KEYPAD
#ifdef KEY_UP
		  || (k == KEY_UP)
#endif
#endif
		  )
	       && ISOPEN(maze[YWRAP(ytile - NOTBOTTOM(yoff))][xtile])) {
	dir = UP;
	sprite[MYMAN] = STILE_MYMAN;
      } else if (((k == 'j') || (k == 'J') || (k == '2')
#if USE_KEYPAD
#ifdef KEY_DOWN
		  || (k == KEY_DOWN)
#endif
#endif
		  )
		 && ISOPEN(maze[YWRAP(ytile + NOTTOP(yoff))][xtile])) {
	dir = DOWN;
	sprite[MYMAN] = STILE_MYMAN + 16;
      }
      if (! pause_timer) {
	if (XDIR(dir))
	  sprite_y[MYMAN] = (1 + 2 * ytile) * TILE_H / 2;
	else if (dir)
	  sprite_x[MYMAN] = (1 + 2 * xtile) * TILE_W / 2;
	if (ISOPEN(maze
		   [YWRAP(ytile + YLEAVING(dir, yoff + YDIR(dir)))]
		   [XWRAP(xtile + XLEAVING(dir, xoff + XDIR(dir)))])) {
	  sprite_x[MYMAN] = XPIXWRAP(sprite_x[MYMAN] + XDIR(dir));
	  sprite_y[MYMAN] = YPIXWRAP(sprite_y[MYMAN] + YDIR(dir) * (cycles & 1));
	  sprite_frame[MYMAN] = (sprite_frame[MYMAN] + 1) % 4;
	}
      }
      for (s = 0; s < GHOSTS; s ++) {
	int blue, mean;
	
	blue = BLUE(s);
	mean = MEAN(s);
	if (! pause_timer)
	  sprite_frame[blue] =
	    (sprite_frame[mean] = (cycles & 4) ? 1 : 0);
	if (dying || winning) {
	  sprite_used[s] =
	    (sprite_used[blue] =
	     (sprite_used[mean] = 0));
	} else if (sprite_used[mean] && collide(mean, MYMAN) && ! pause_timer) {
	  if (sound)
	    beep();
	  dying = DEATHDELAY;
	  sprite[MYMAN] = STILE_MYMAN + 8;
	  sprite_frame[MYMAN] = 0;
	  pause_timer = ONESEC;
	  munched = MYMAN;
	  s = -1;
	  continue;
	} else if (sprite_used[blue] && collide(blue, MYMAN) && ! pause_timer) {
	  if (sound)
	    beep();
	  score += points;
	  points *= 2;
	  pause_timer = ONESEC;
	  sprite_used[MYMAN] = 0;
	  munched = s;
	  sprite_used[GHOST_SCORE] = 1;
	  sprite_x[GHOST_SCORE] = sprite_x[blue];
	  sprite_y[GHOST_SCORE] = sprite_y[blue];
	  sprite_x[s] = sprite_x[blue];
	  sprite_y[s] = sprite_y[blue];
	  sprite_used[blue] = 0;
	} else {
	  int dir0, dir1, dir2, o0, o1, o2, x, y, i1, j1, i2, j2;

	  dir1 = ghost_dir[s];
	  dir0 = DIRWRAP(dir1 + (s != PINKY) - (s == PINKY));
	  dir2 = DIRWRAP(dir1 - (s != PINKY) + (s == PINKY));
	  if (sprite_used[blue] && ! pause_timer) {
	    /* runnin' scared */
	    if (cycles & 1) {
	      x = sprite_x[blue];
	      y = sprite_y[blue];
	      i1 = XTILE(x);
	      j1 = YTILE(y);
	      i2 = XTILE(sprite_x[MYMAN]);
	      j2 = YTILE(sprite_y[MYMAN]);
	      if ((i1 == i2) && (j1 <= j2)) {
		for (j = j1; j <= j2; j ++)
		  if (! ISOPEN(maze[j][i1]))
		    break;
		if (j > j2) {
		  ghost_mem[s] = dir;
		  ghost_timer[s] = MEMDELAY;
		}
	      } else if ((i1 == i2) && (j1 >= j2)) {
		for (j = j2; j <= j1; j ++)
		  if (! ISOPEN(maze[j][i1]))
		    break;
		if (j > j1) {
		  ghost_mem[s] = dir;
		  ghost_timer[s] = MEMDELAY;
		}
	      } else if ((j1 == j2) && (i1 <= i2)) {
		for (i = i1; i <= i2; i ++)
		  if (! ISOPEN(maze[j1][i]))
		    break;
		if (i > i2) {
		  ghost_mem[s] = dir;
		  ghost_timer[s] = MEMDELAY;
		}
	      } else if ((j1 == j2) && (i1 >= i2)) {
		for (i = i2; i <= i1; i ++)
		  if (! ISOPEN(maze[j1][i]))
		    break;
		if (i > i1) {
		  ghost_mem[s] = dir;
		  ghost_timer[s] = MEMDELAY;
		}
	      } 
	      o0 = ISOPEN(maze[YWRAP(j1 + YDIR(dir0))][XWRAP(i1 + XDIR(dir0))]);
	      o1 = ISOPEN(maze[YWRAP(j1 + YDIR(dir1))][XWRAP(i1 + XDIR(dir1))]);
	      o2 = ISOPEN(maze[YWRAP(j1 + YDIR(dir2))][XWRAP(i1 + XDIR(dir2))]);
	      if (((TILE_W / 2 == x % TILE_W) && XDIR(dir1))
		  || ((TILE_H / 2 == y % TILE_H) && YDIR(dir1))) {
		if (o0 || o2) {
		  if (! ghost_mem[s]) {
		    if (o2)
		      dir1 = dir2;
		    else if (! o1)
		      dir1 = dir0;
		  } else {
		    if (o0 && (dir0 != ghost_mem[s]))
		      dir1 = dir0;
		    else if (o2 && (dir2 != ghost_mem[s]))
		      dir1 = dir2;
		    else if (o1 && (dir1 != ghost_mem[s]))
			;
		    else if (o0)
		      dir1 = dir0;
		    else
		      dir1 = dir2;
		  }
		} else if (! o1)
		  dir1 = DIRWRAP(dir1 + 2);
	      }
	      if (XDIR(dir1))
		y = (1 + 2 * j1) * TILE_H / 2;
	      else if (dir1)
		x = (1 + 2 * i1) * TILE_W / 2;
	      sprite_x[blue] = XPIXWRAP(x + XDIR(dir1));
	      if (cycles & 2)
		sprite_y[blue] = YPIXWRAP(y + YDIR(dir1));
	      if (! home_dir[s][YTILE(sprite_y[blue])][XTILE(sprite_x[blue])])
		home_dir[s][YTILE(sprite_y[blue])][XTILE(sprite_x[blue])] =
		  DIRWRAP(dir1 + 2);
	    }
	  } else if (sprite_used[mean] && ! pause_timer) {
	    /* out huntin' */
	    int door, d0, d1, d2;

	    x = sprite_x[mean];
	    y = sprite_y[mean];
	    i1 = XTILE(x);
	    j1 = YTILE(y);
	    i2 = XTILE(sprite_x[MYMAN]);
	    j2 = YTILE(sprite_y[MYMAN]);
	    if ((i1 == i2) && (j1 <= j2)) {
	      for (j = j1; j <= j2; j ++)
		if (! ISOPEN(maze[j][i1]))
		  break;
	      if (j > j2) {
		ghost_mem[s] = dir;
		ghost_timer[s] = MEMDELAY;
	      }
	    } else if ((i1 == i2) && (j1 >= j2)) {
	      for (j = j2; j <= j1; j ++)
		if (! ISOPEN(maze[j][i1]))
		  break;
	      if (j > j1) {
		ghost_mem[s] = dir;
		ghost_timer[s] = MEMDELAY;
	      }
	    } else if ((j1 == j2) && (i1 <= i2)) {
	      for (i = i1; i <= i2; i ++)
		if (! ISOPEN(maze[j1][i]))
		  break;
	      if (i > i2) {
		ghost_mem[s] = dir;
		ghost_timer[s] = MEMDELAY;
	      }
	    } else if ((j1 == j2) && (i1 >= i2)) {
	      for (i = i2; i <= i1; i ++)
		if (! ISOPEN(maze[j1][i]))
		  break;
	      if (i > i1) {
		ghost_mem[s] = dir;
		ghost_timer[s] = MEMDELAY;
	      }
	    } 
	    o0 = ISOPEN(maze[YWRAP(j1 + YDIR(dir0))]
			[XWRAP(i1 + XDIR(dir0))]);
	    o1 = ISOPEN(maze[YWRAP(j1 + YDIR(dir1))]
			[XWRAP(i1 + XDIR(dir1))]);
	    o2 = ISOPEN(maze[YWRAP(j1 + YDIR(dir2))][XWRAP(i1 + XDIR(dir2))]);
	    door = ISDOOR(maze[YWRAP(j1 - 1)][i1]);
	    d0 = (dir0 == UP) && door;
	    d1 = (dir1 == UP) && door;
	    d2 = (dir2 == UP) && door;
	    if (((TILE_W / 2 == x % TILE_W) && XDIR(dir1))
		|| ((TILE_H / 2 == y % TILE_H) && YDIR(dir1))) {
	      if (d2 || (o2 && (dir2 == ghost_mem[s])))
		dir1 = dir2;
	      else if (d1 || (o1 && (dir1 == ghost_mem[s])))
		  ;
	      else if (d0 || (o0 && (dir0 == ghost_mem[s])))
		dir1 = dir0;
	      else if (o1)
		  ;
	      else if (o2)
		dir1 = dir2;
	      else if (o0)
		dir1 = dir0;
	      else
		dir1 = DIRWRAP(dir1 + 2);
	    }
	    if (XDIR(dir1))
	      y = (1 + 2 * j1) * TILE_H / 2;
	    else if (dir1)
	      x = (1 + 2 * i1) * TILE_W / 2;
	    sprite_x[mean] = (sprite_x[s] = XPIXWRAP(x + XDIR(dir1)));
	    if (cycles & 1)
	      sprite_y[mean] = (sprite_y[s] = YPIXWRAP(y + YDIR(dir1)));
	    if (! home_dir[s][YTILE(sprite_y[mean])][XTILE(sprite_x[mean])])
	      home_dir[s][YTILE(sprite_y[mean])][XTILE(sprite_x[mean])] =
		DIRWRAP(dir1 + 2);
	  } else if (sprite_used[s] && ! pause_timer) {
	    int dx, dy;
	    unsigned char d, d1;

	    /* goin' home */
	    ghost_timer[s] = MEMDELAY;
	    x = sprite_x[s];
	    y = sprite_y[s];
	    i1 = XTILE(x);
	    j1 = YTILE(y);
	    dx = (XMYMAN - x) / TILE_W;
	    dy = ((dx ? YTOP : YGHOST) - y) / TILE_H;
	    if ((d = HOME_DIR(j1, i1)) &&
		(! ((d1 = DIRWRAP(HOME_DIR(YWRAP(j1 + YDIR(d)), XWRAP(i1 + XDIR(d))) + 2)) &&
		    (d1 == d))))
	      ghost_mem[s] = d;
	    else {
	      if (dx * dx > dy * dy) {
		if (dx > 0)
		  ghost_mem[s] = RIGHT;
		else
		  ghost_mem[s] = LEFT;
	      } else {
		if (dy < 0)
		  ghost_mem[s] = UP;
		else
		  ghost_mem[s] = DOWN;
	      }
	    }
	    if (! (dx || dy)) {
	      sprite_used[s] = lives ? VISIBLE_EYES : 0;
	      sprite_used[mean] = 1;
	      sprite_x[mean] = x;
	      sprite_y[mean] = y;
	      sprite_frame[s] = (ghost_dir[s] = s + 1) - 1;
	      ghost_mem[s] = 0;
	      continue;
	    }
	    o0 = ISOPEN(maze[YWRAP(j1 + YDIR(dir0))][XWRAP(i1 + XDIR(dir0))])
	      || ISDOOR(maze[YWRAP(j1 + YDIR(dir0))][XWRAP(i1 + XDIR(dir0))]);
	    o1 = ISOPEN(maze[YWRAP(j1 + YDIR(dir1))][XWRAP(i1 + XDIR(dir1))])
	      || ISDOOR(maze[YWRAP(j1 + YDIR(dir1))][XWRAP(i1 + XDIR(dir1))]);
	    o2 = ISOPEN(maze[YWRAP(j1 + YDIR(dir2))][XWRAP(i1 + XDIR(dir2))])
	      || ISDOOR(maze[YWRAP(j1 + YDIR(dir2))][XWRAP(i1 + XDIR(dir2))]);
	    if (o2 && (dir2 == ghost_mem[s]))
	      dir1 = dir2;
	    else if (o1 && (dir1 == ghost_mem[s]))
		;
	    else if (o0 && (dir0 == ghost_mem[s]))
	      dir1 = dir0;
	    else if (o1)
		;
	    else if (o2)
	      dir1 = dir2;
	    else if (o0)
	      dir1 = dir0;
	    else
	      dir1 = DIRWRAP(dir1 + 2);
	    sprite_x[mean] = (sprite_x[s] = XPIXWRAP(x + XDIR(dir1)));
	    sprite_x[s] = XPIXWRAP(x + XDIR(dir1));
	    sprite_y[s] = YPIXWRAP(y + YDIR(dir1));
	    if (! home_dir[s][YTILE(sprite_y[s])][XTILE(sprite_x[s])])
	      home_dir[s][YTILE(sprite_y[s])][XTILE(sprite_x[s])] =
		DIRWRAP(dir1 + 2);
	  }
	  if ((! pause_timer) && ghost_timer[s] && ! -- ghost_timer[s]) {
	    ghost_timer[s] = MEMDELAY;
	    if (ghost_mem[s]) {
	      ghost_mem[s] = 0;
	    } else
	      dir1 = DIRWRAP(ghost_dir[s] + 2);
	  }
	  ghost_dir[s] = (sprite_frame[s] = dir1 - 1) + 1;
	  if (sprite_used[mean] && collide(mean, MYMAN) && ! pause_timer) {
	    if (sound)
	      beep();
	    dying = DEATHDELAY;
	    sprite[MYMAN] = STILE_MYMAN + 8;
	    sprite_frame[MYMAN] = 0;
	    pause_timer = ONESEC;
	    munched = MYMAN;
	    s = -1;
	    continue;
	  } else if (sprite_used[blue] && collide(blue, MYMAN) && ! pause_timer) {
	    if (sound)
	      beep();
	    score += points;
	    points *= 2;
	    pause_timer = ONESEC;
	    sprite_used[MYMAN] = 0;
	    munched = s;
	    sprite_used[GHOST_SCORE] = 1;
	    sprite_x[GHOST_SCORE] = sprite_x[blue];
	    sprite_y[GHOST_SCORE] = sprite_y[blue];
	    sprite_x[s] = sprite_x[blue];
	    sprite_y[s] = sprite_y[blue];
	    sprite_used[blue] = 0;
	  }
	}
      }
      if (! pause_timer)
	for (s = 0; s < SPRITES; s ++) {
	  if (sprite_used[s] && sprite_timer[s])
	    if (! -- sprite_timer[s])
	      sprite_used[s] = 0;
	}
    }
    erase();
    x1 = sprite_x[MYMAN] - COLS / 2;
    y1 = sprite_y[MYMAN] - LINES / 2 - deadpan;
    if (x1 + COLS > MAZE_W * TILE_W)
      x1 = MAZE_W * TILE_W - COLS;
    if (y1 + LINES - 1 > MAZE_H * TILE_H)
      y1 = MAZE_H * TILE_H - (LINES - 1);
    if (x1 < 0)
      x1 = 0;
    if (y1 < 0)
      y1 = 0;
    roff = 0;
    coff = 0;
    if (TILE_H * MAZE_H < LINES)
      roff = (LINES - 1 - TILE_H * MAZE_H) / 2;
    if (TILE_W * MAZE_W < COLS)
      coff = (COLS - TILE_W * MAZE_W) / 2;
    for (line = 0; (line < LINES) && (line < TILE_H * MAZE_H); line++) {
      move(line + roff, coff);
      for (col = 0; (col < COLS) && (col < TILE_W * MAZE_W); col++) {
	int s;
        int a;
	
        a = 0;
	c = 0;
	xtile = XTILE((i = col + x1));
	ytile = YTILE((j = line + y1));
	for (s = 0; s < SPRITES; s ++) {
	  int t, x, y;
	  
	  t = sprite[s] + sprite_frame[s];
	  if (debug
	      && sprite_used[s]
	      && ((x = sprite_x[s]) == i)
	      && ((y = sprite_y[s]) == j)) {
#if USE_COLOR
	    if (colorize) {
	      if (! (a = stile_color[t]))
		a = sprite_color[s];
	      a = pen[a];
	    } else
#endif
#if USE_ATTR
#ifdef A_BOLD
	      if (s == MYMAN)
		a |= A_BOLD;
#endif
#endif
	    ;
	    if (s < GHOSTS)
	      c = (".^<v>")[ghost_mem[s]];
	    else
	      c = '.';
	    break;
	  } else if (sprite_used[s]
		     && stile_used[t]
		     && ((x = sprite_x[s] - STILE_W / 2) <= i)
		     && ((xoff = i - x) < STILE_W)
		     && ((y = sprite_y[s] - STILE_H / 2) <= j)
		     && ((yoff = j - y) < STILE_H)
		     && (c = stile[(t * STILE_H + yoff) * STILE_W + xoff])) {
#if USE_COLOR
	    if (colorize) {
	      if (! (a = stile_color[t]))
		a = sprite_color[s];
	      a = pen[a];
	    } else
#endif
#if USE_ATTR
#ifdef A_BOLD
	      if (s == MYMAN)
		a |= A_BOLD;
#endif
#endif
	    ;
	    break;
          }
	}
	if ((! c) && (ytile < MAZE_H) && (xtile < MAZE_W)) {
	  if (tile_used[(c = maze[ytile][xtile])]) {
#if USE_COLOR
	    if (colorize) {
	      a = tile_color[c];
	      a = pen[a];
	    } else
#endif
#if USE_ATTR
	      {
#ifdef A_BOLD
		if (ISPELLET(c))
		  a |= A_BOLD;
#endif
#ifdef A_UNDERLINE
		if (underline)
		  if (! (ISDOT(c) || ISPELLET(c) || ISOPEN(c) || ISDOOR(c)))
		    a |= A_UNDERLINE;
#endif
	      }
#endif
	    ;
	    if (debug) {
	      int s = WHOSE_HOME_DIR(ytile, xtile);
	      unsigned char d = home_dir[s][ytile][xtile];

	      c = (d == UP) ? '^'
		: (d == DOWN) ? 'v'
		: (d == LEFT) ? '<'
		: (d == RIGHT) ? '>'
		: ISDOT(c) ? ','
		: ISPELLET(c) ? ';'
		: ISOPEN(c) ? ' '
		: ISDOOR(c) ? 'X'
		: '@';
#if USE_COLOR
	      if (colorize && d) {
		a = stile_color[sprite[MEAN(s)] + sprite_frame[MEAN(s)]];
		if (! a)
		  a = sprite_color[MEAN(s)];
		a = pen[a];
	      }
#endif
	    } else {
              if ((ISPELLET(c) && (cycles & 8) && (! dead) &&
		   ! (sprite_used[MYMAN] && pause_timer)) ||
		  ((winning < ONESEC) && winning &&
		   (ISDOT(c) || ISPELLET(c) || ((winning & 8) && ISDOOR(c)))))
		c = ' ';
	      else if ((winning < ONESEC) && (winning & 8) && ! pause_timer) {
#if USE_COLOR
		if (colorize)
		  a = pen[0xF];
		else
#endif
		  c = ' ';
	      }
	      c = tile[(c * TILE_H
			+ j % TILE_H) * TILE_W
		      + i % TILE_W];
	    }
	  } else
	    c = ' ';
	}
        if (! c)
	  c = ' ';
	my_addch(CHAR_TRANS(c)|(a));
      }
    }
#if USE_ATTR || USE_COLOR
    attrset(0);
#endif
    ;
    if (sprite_used[FRUIT] && (LINES > 6) && ! sound) {
      static char *msg[] = {" <  <N>  > ",
			    "<  <ONU>  >",
			    "  <BONUS>  ",
			    " < BONUS > ",
			    "<  BONUS  >",
			    " > BONUS < ",
			    "  >BONUS<  ",
			    ">  >ONU<  <"};
      
      mvprintw(LINES - 1, 1, msg[(cycles >> 1) & 7]);
    }
    if (pellet_timer && ! pause_timer) {
      int s, blue, mean;

      if (! -- pellet_timer) {
	for (s = 0; s < GHOSTS; s ++)
	  if (sprite_used[(blue = BLUE(s))]) {
	    sprite_used[(mean = MEAN(s))] = 1;
	    sprite_used[blue] = 0;
	    sprite_used[s] = VISIBLE_EYES;
	    sprite_x[s] =
	      (sprite_x[mean] = sprite_x[blue]);
	    sprite_y[s] =
	      (sprite_y[mean] = sprite_y[blue]);
	    sprite_frame[s] = (ghost_dir[s] = s + 1) - 1;
	  }
      } else if (pellet_timer <= TWOSECS)
	for (s = 0; s < GHOSTS; s ++)
	  sprite[BLUE(s)] = ((pellet_timer / 10) & 1)
	    ? STILE_BLUE
	    : STILE_WHITE;
    }
    if ((LINES > 6) && (COLS > 46))
      mvprintw(LINES - 1, COLS - 46, " Level: %-10u Lives: %d Score: %-10u ", level, lives, score);
    refresh();
    cycles ++;
  }
#if USE_ATTR || USE_COLOR
  attrset(0);
#endif
  clear();
#if USE_COLOR
  if (colorize) {
    standout();
    mvprintw(LINES ? 1 : 0, 0, " ");
    standend();
    refresh();
    destroy_pen();
    mvprintw(LINES ? 1 : 0, 0, " ");    
    addch('\n');
  }
#endif
  refresh();
  echo();
  endwin();
  fprintf(stderr, "%s: scored %d points, reached level %d\n",
	  argv[0], score, level);
  return 0;
}
