/************************************************************************/
/************************************************************************/
/*																		*/
/*	Name :		DRAW.C													*/
/*	Project :	Tomb of Drewan in 'C'									*/
/*	Author :	Paul Robson												*/
/*	Created :	23rd April 2001											*/
/*	Function :	Additional functions related to video display			*/
/*	Changes :															*/
/*																		*/
/************************************************************************/
/************************************************************************/

#include "drewan.h"

static void _DRAWChar(int x,int y,int Char,int Colour);
static int _DRAWToward(int Pos,int Tgt);
void DRAWRefresh(int x,int y);

struct _MapEntry { int Char,Colour; };		/* Struct represents one cell */
static struct _MapEntry Map[22][23];		/* The VIC Screen */
static int Printing = 0;					/* 1 using DRAWPrint (no tiles) */

#define ONSCREEN(x,y) ((x) >= 0 && (x) < 22 && (y) >= 0 && (y) < 23)

/************************************************************************/
/*																		*/
/*						Draw a single character on the map				*/
/*																		*/
/************************************************************************/

void DRAWClear(void)						/* Prototyping for DRAW.C */
{
	int x,y;
	for (x = 0;x < 22;x++)					/* Draw space in all screen */
		for (y = 0;y < 23;y++)
		{
			Map[x][y].Char = 0;				/* Use direct call to IODraw() */
			Map[x][y].Colour = 0;			/* because of hook stuff */
			IODraw(x*8,y*8,' ',COL_WHITE);
		}
}

/************************************************************************/
/*																		*/
/*						Draw a single character on the map				*/
/*																		*/
/************************************************************************/

void DRAWChar(int x,int y,int Char,int Colour)
{
	if (ONSCREEN(x,y))							/* If a legit location */
	{
		_DRAWChar(x*8,y*8,Char,Colour);			/* Draw it */
		Map[x][y].Char = Char;					/* Save on the map */
		Map[x][y].Colour = Colour;
	}
}

/************************************************************************/
/*																		*/
/*						Draw a single character on the map				*/
/*																		*/
/************************************************************************/

int	 DRAWGet(int x,int y)
{
	int n = -1;
	if (ONSCREEN(x,y)) n = Map[x][y].Char;
	return n;
}

/************************************************************************/
/*																		*/
/*				Draw a string (possibly reversed) on the map			*/
/*																		*/
/************************************************************************/

void DRAWPrint(int x,int y,int Colour,char *Text)
{
	int c;
	Printing = 1;							/* Printing now */
	while(*Text != '!' && *Text != '\0')	/* Go through the string */
	{
		c = *Text++;						/* Read character */
		if (isalpha(c)) c=toupper(c)-64;	/* Adjust for character set */
		if ((Colour & COL_REVTEXT) != 0)
				if (c % 64 <= 26 || c == 0x20) c = c ^ 0x40;
		if (c == '.')
			Colour = (*Text++) - '0';       /* Change colour */
		else
			DRAWChar(x++,y,c,Colour);		/* Draw it */
	}
	Printing = 0;							/* No longer printing */
}

/************************************************************************/
/*																		*/
/*						Draw a single character on the map				*/
/*																		*/
/************************************************************************/

static void _DRAWChar(int x,int y,int Char,int Colour)
{
	int ASCII,IsOnMap;
	IsOnMap = (y <= 17) && (Printing == 0);	/* Maybe special map tiles ? */

	if (IOIsClassic())						/* Classic is just that.... */
	{
		IODraw(x,y,Char,Colour);
		return;
	}

	if (IOXDraw(x,y,Char,Colour) != 0) return;

	if (isdigit(Char & 0xFF) || (Char&0x3F) <= 26)	/* if alphanum or reversed alpha */
	{
		ASCII = isdigit(Char & 0xFF) ?		/* Convert to ASCII */
						Char : Char | 0x40;
		if (Char & 0x40) ASCII |= 0x80;		/* If was reversed,set bit 7 */
		if (IOXAlphaDraw(x,y,				/* Try to draw it */
				ASCII,Colour) != 0) return;
	}
	if (IsOnMap)
		if (IOXMapTileDraw(x,y,Char,Colour)) return;

	IODraw(x,y,Char,Colour);
}

/************************************************************************/
/*																		*/
/*				  Animate movement from Start to End					*/
/*																		*/
/************************************************************************/

void DRAWAnimate(POINT *pStart,POINT *pEnd,int Char,int Colour,int Time)
{
	long NextMove;
	POINT Start,End,Last;
	int Anim,m = 8;
	Start = *pStart;End = *pEnd;			/* Retrieve points */
	if (!IOIsClassic())						/* If pixel level */
	{
		Start.x*=8;End.x*=8;				/* Scale up cells->pixels */
		Start.y*=8;End.y*=8;
		Time = Time / 8;					/* Cell time -> pixel time */
		m = 1;								/* Coords will be in pix units */
	}
	NextMove = IOTimer()+Time;
	while (Start.x != End.x ||				/* Wait till we reach the end */
						Start.y != End.y)
	{
		Anim = GRAN_UPDOWN;					/* Work out animation bits */
		if (End.x < Start.x) Anim = GRAN_LEFT;
		if (End.x > Start.x) Anim = GRAN_RIGHT;
		Char = (Char & 0x3FFF) | Anim;		/* Set anim bits */
		Last = Start;						/* Save position */
		_DRAWChar(Start.x*m,Start.y*m,		/* Draw it */
								Char,Colour);
		Start.x = DRAWToward(Start.x,End.x);/* Move start towards end */
		Start.y = DRAWToward(Start.y,End.y);
		while (IOTimer() < NextMove) {}		/* Wait */
		NextMove = NextMove+Time;			/* Set up the next time out */

		if (Last.x % 8)						/* Off centre horizontal */
		{
			DRAWRefresh(Last.x*m+8,Last.y*m);
			if (Last.y*m % 8) DRAWRefresh(Last.x*m+8,Last.y*m+8);
		}
		if (Last.y*m % 8)					/* Off centre vertical */
			DRAWRefresh(Last.x*m,Last.y*m+8);
		DRAWRefresh(Last.x*m,Last.y*m);	/* Always do the basic one */
	}
}

/************************************************************************/
/*																		*/
/*				Refresh the cell at PIXEL address x,y					*/
/*																		*/
/************************************************************************/

void DRAWRefresh(int x,int y)
{
	x = x/8;y = y/8;						/* Back to cells, then draw */
	DRAWChar(x,y,Map[x][y].Char,Map[x][y].Colour);
}

/************************************************************************/
/*																		*/
/*						Move POS towards TGT							*/
/*																		*/
/************************************************************************/

int DRAWToward(int Pos,int Tgt)
{
	if (Pos == Tgt) return Pos;
	return (Pos < Tgt) ? Pos+1 : Pos-1;
}

/************************************************************************/
/*																		*/
/*							Delay function								*/
/*																		*/
/************************************************************************/

void DRAWDelay(int Time)
{
	long t = IOTimer()+(long)Time;
	while (IOTimer() < t) {}
}

/************************************************************************/
/*																		*/
/*							Draw Message Box							*/
/*																		*/
/************************************************************************/

void DRAWMessage(int Cmd,char *Message)
{
	int Col,x,y,Lines,c;
	if (IOXMessage(Cmd,Message)) return;
	for (x = 0;x < 22;x++)					/* Blank the message area */
		for (y = 19;y < 23;y++)
				DRAWChar(x,y,' ',0);
	Lines = 1;								/* Count the lines */
	for (x = 0;x < strlen(Message);x++)
		if (Message[x] == '!') Lines++;
	y = 21-Lines/2;Col = COL_WHITE;			/* Start point */
	while (*Message != '\0')
	{
		c = x = 0;
		while (Message[x] != '\0' && Message[x] != '!')
		{
			if (Message[x++] == '.') c--; else c++;
		}
		DRAWPrint(11-c/2,y++,Col,Message);
		Message = Message + x;if (*Message == '!') Message++;
	}
	if (Cmd & MB_DELAY)
		DRAWDelay(PARAM(PR_MSGDELAY,2000));
}

/************************************************************************/
/*																		*/
/*						Restore after Message Box						*/
/*																		*/
/************************************************************************/

void DRAWRemMessage(PLAYER *p,ROOM *r)
{
	if (IOXRemMessage(p,r)) return;
	PANELDraw(p);
}