/************************************************************************/
/************************************************************************/
/*																		*/
/*	Name :		ROOM.C													*/
/*	Project :	Tomb of Drewan in 'C'									*/
/*	Author :	Paul Robson												*/
/*	Created :	23rd April 2001											*/
/*	Function :	Draw the room,treasures,monsters etc.					*/
/*				Contains values from the original VIC 20 program		*/
/*	Changes :															*/
/*																		*/
/************************************************************************/
/************************************************************************/

#include "drewan.h"

static void _ROOMDrawStandard(int);
static void _ROOMDrawType(int Type);
static void _ROOMInitTreasure(ROOM *r);
static void _ROOMIdentifyTreasure(PLAYER *p,ROOM *r);
static void _ROOMCreateMonsters(ROOM *r);
static void _ROOMShowMonsters(ROOM *r);

/************************************************************************/
/*																		*/
/*					Draw a room, initialisation optional				*/
/*																		*/
/************************************************************************/

void ROOMDraw(PLAYER *p,ROOM *r,int Initialise)
{
	int i;
	IOFrame(COL_YELLOW*16+COL_BLUE);
	r->Desc = &(p->Room[p->RoomID]);		/* Make the descriptor point */
	i = ROOMStyle(p->RoomID);
	if (i == MPT_GARDEN) IOFrame(COL_YELLOW*16+COL_GREEN);
	if (i == MPT_DUNGEON) IOFrame(COL_BLACK*16+COL_RED);
	_ROOMDrawStandard(p->RoomID);			/* Draw the outline */
	_ROOMDrawType(r->Desc->LayoutType);		/* Cut the bits out */
	if (p->RoomID < MAPSIZE ||				/* Top/Bottom frame */
		p->RoomID >= MAPSIZE*(MAPSIZE-1))
			for (i = 0;i < 22;i++)
				DRAWChar(i,(p->RoomID < MAPSIZE) ? 0:17,GRX_OUTSIDEWALL,COL_BLUE);
	if (p->RoomID%MAPSIZE == 0 ||			/* Left/Right frame */
		p->RoomID%MAPSIZE == MAPSIZE-1)
			for (i = 0;i < 18;i++)
				DRAWChar((p->RoomID%MAPSIZE) ? 21:0,i,GRX_OUTSIDEWALL,COL_BLUE);
	_ROOMInitTreasure(r);					/* Set up the treasure */
	if (Initialise)                     	/* Initialising ? */
	{
		_ROOMIdentifyTreasure(p,r);			/* Decide what they are */
		_ROOMCreateMonsters(r);				/* Create the monsters */
		if (p->Has[HF_AMULET])				/* Show any monsters if you */
				_ROOMShowMonsters(r);		/* have the amulet */
	}
	DRAWChar(p->Pos.x,p->Pos.y,				/* Draw the player */
						p->Char,p->Colour);
}

/************************************************************************/
/*																		*/
/*						Show monsters, briefly (amulet)					*/
/*																		*/
/************************************************************************/

static void _ROOMShowMonsters(ROOM *r)
{
	int i;
	for (i = 0;i < MCOUNT;i++)
	{
		if (r->Monster[i].State != MS_DEAD)
		{
			DRAWChar(r->Monster[i].Pos.x,r->Monster[i].Pos.y,
						r->Monster[i].Graphic,r->Monster[i].Colour);
			DRAWDelay(PARAM(PR_AMULETVIEWTIME,500));
			DRAWChar(r->Monster[i].Pos.x,
						r->Monster[i].Pos.y,' ',COL_WHITE);
		}
	}
}

/************************************************************************/
/*																		*/
/*					Create the monsters in this room					*/
/*																		*/
/************************************************************************/

static int _MTypes[19] = { 1,2,3,4,5,6,4,5,6,3,2,1,6,5,3,4,2,1,3 };

static unsigned char _MPos[56] = {
	1,4,15,1,2,14,13,18,3,2,13,8,4,16,11,19,
	5,5,15,5,6,12,11,13,2,7,11,8,8,12,9,20,7,4,16,7,4,15,11,13,
	6,10,11,5,6,12,12,12,4,7,13,8,4,11,10,16 };

static void _ROOMCreateMonsters(ROOM *r)
{
	int i = sizeof(_MPos);
	for (i = 0;i < MCOUNT;i++)
	{
		r->Monster[i].State = MS_DEAD;		/* Initially dead */
		if (r->Desc->MonsterDead[i] == 0)	/* If not killed, create it */
		{
			r->Monster[i].State = MS_INACTIVE;
			r->Monster[i].Pos.x = _MPos[(r->Desc->MonsterLayoutType-1)%7*8+i*2+1];
			r->Monster[i].Pos.y = _MPos[(r->Desc->MonsterLayoutType-1)%7*8+i*2];
			r->Monster[i].Type = _MTypes[(r->Desc->MonsterType-1)%15+i];
			MONSTGetInfo(&(r->Monster[i]));
		}
	}
}

/************************************************************************/
/*																		*/
/*							Identify treasures							*/
/*																		*/
/************************************************************************/

static int TType_Data[19] = { 1,5,6,4,3,2,7,8,9,10,11,8,6,3,10,5,8,1,7 };

static void _ROOMIdentifyTreasure(PLAYER *p,ROOM *r)
{
	int n,i;
	for (i = 0;i < TCOUNT;i++)				/* Get the treasure name */
	{
		n = TType_Data[i+(r->Desc->TreasureType-1)%15];
		r->Treasure[i].TreasureID = n;
		strcpy(r->Treasure[i].Name,ROOMObject(n));
		r->Treasure[i].Gold = rand()%20+2;
	}
	for (i = 0;i < TCOUNT;i++)				/* Check for the "big" treasure */
		if (p->RoomID == p->MajorTreasure[i])
			{
			n = r->Treasure[i].TreasureID = HF_WATERSTONE+i;
			strcpy(r->Treasure[i].Name,ROOMObject(n));
			}
	if (p->RoomID == p->MajorTreasure[4])	/* Amulet check */
		{
			r->Treasure[i].TreasureID = HF_AMULET;
			strcpy(r->Treasure[i].Name,ROOMObject(-5));
		}
}

/************************************************************************/
/*																		*/
/*			Initialise the treasure positions, display them				*/
/*																		*/
/************************************************************************/

static unsigned char TreasureLayout[] =	{
		1,1,2,17,9,4,14,16,3,4,6,19,14,13,15,7,1,1,
		6,19,10,13,16,1,3,4,6,19,14,13,15,7,
		2,1,1,20,16,2,14,16,1,2,1,19,9,4,16,
		14,9,4,1,15,8,9,15,13 };

static void _ROOMInitTreasure(ROOM *r)
{
	int i;
	for (i = 0;i < TCOUNT;i++)				/* There are four treasures */
	{
		r->Treasure[i].Enabled = 0;			/* No treasure */
		if (r->Desc->TreasureTaken[i] == 0)	/* Have not got it yet */
		{
			r->Treasure[i].Enabled = 1;		/* Treasure on screen */
			r->Treasure[i].Pos.x =			/* Get its location */
					TreasureLayout[(r->Desc->TreasureLayoutType%7)*8+i*2+1];
			r->Treasure[i].Pos.y =
					TreasureLayout[(r->Desc->TreasureLayoutType%7)*8+i*2];
			DRAWChar(r->Treasure[i].Pos.x,
					 r->Treasure[i].Pos.y,
					 GR_TREASURE,COL_MAGENTA);
		}
	}
}

/************************************************************************/
/*																		*/
/*	  Remove bits from the room, according to the type of the room		*/
/*																		*/
/************************************************************************/

static unsigned char _RemoveBits[] = {
        4,1,4,4,8,6,7,8,12,5,15,9,10,10,12,16,1,16,14,1,11,12,11,4,12,17,12,7,14,1,
        3,3,1,6,3,17,6,6,6,17,8,1,8,9,8,20,12,5,16,4,16,9,16,12,16,17,15,20,3,18,
        1,3,2,16,3,8,4,5,13,6,6,6,9,3,9,6,9,13,10,11,12,8,13,9,13,15,14,3,15,20,
        2,6,3,17,4,2,8,20,5,20,9,3,10,9,10,10,10,11,12,3,12,5,12,8,12,18,14,1,14,15,
        1,3,3,6,3,20,12,5,7,7,7,16,8,6,8,20,12,3,12,8,13,12,15,9,15,20,16,4,11,12,
        4,2,4,4,2,16,7,12,8,1,9,6,9,9,10,9,11,9,12,5,12,13,12,16,13,12,15,4,16,9,
        1,3,1,6,2,11,3,20,4,5,6,11,7,11,10,11,10,12,12,8,12,13,13,9,15,4,16,6,16,17,
		1,3,2,12,3,18,3,6,7,12,8,2,8,20,10,5,11,12,12,8,13,6,13,19,14,9,15,4,15,16 };

static void _ROOMDrawType(int Type)
{
	int i,x,y;
	Type = ((Type-1)%8)*30;					/* There are 8 lots of 15 pairs */
	for (i = 0;i < 15;i++)					/* Go through each pair */
		{
		y = _RemoveBits[Type++];			/* Read position and blank it */
		x = _RemoveBits[Type++];
		DRAWChar(x,y,32,COL_BLACK);
		}
}

/************************************************************************/
/*																		*/
/*					Draw the constant part of the room					*/
/*																		*/
/************************************************************************/

static char *RoomLayout[] = {
		"***********X***********  *  *9*4**  *  * ******  *4**",
		"  *  * *4**************** *66**",
		"5*4*   ********  *  *    *     *   **",
		"  *  ************   *****  *  *9****  *  *  *  ******** **",
		"  ****  ****8**   *    *  *8***********  ******** *X",
		"5*  *  *  *   * X***** *  *  *  *   * **   * *  *  *  *** ****",
		"   * *  *  *    *   ************X**********",
		NULL };

static void _ROOMDrawStandard(int r)
{
	int Pos = 0;
	int n,c,i = 0;
	char *s;
	c = GR_BRICK;n = ROOMStyle(r);
	if (n == MPT_GARDEN) c = GRX_TREE;
	if (n == MPT_DUNGEON) c = GRX_STONE;
	do
	{
		s = RoomLayout[i++];				/* Read next descriptor */
		while (*s != '\0')					/* Work through it */
		{
			if (*s == ' ' || isdigit(*s))	/* Leave spaces */
			{
				n = 1;
				if (isdigit(*s)) n = (*s - '0');
				while (n-- > 0)
				{
					DRAWChar(Pos%22,Pos/22,' ',COL_WHITE);
					Pos++;
				}
			}
			if (*s == '*')					/* Draw one brick */
			{
				DRAWChar(Pos%22,Pos/22,c,COL_WHITE);
				Pos++;
			}
			if (*s == 'X')					/* Draw one door */
			{
				DRAWChar(Pos%22,Pos/22,GR_DOOR,COL_WHITE);
				Pos++;
			}
			s++;
		}
	} while (RoomLayout[i] != NULL);		/* Until drawn the lot */
}

/************************************************************************/
/*																		*/
/*						Return the name of the object					*/
/*																		*/
/************************************************************************/

char *OName[11] = {
	 "a jump spell","a remover spell","a golden arrow","a chaos spell",
	"an elixir","a water spell","a fire spell","pounds of gold",
	"a petrify spell", "a vampire spell","a mirror spell"  };

char *TName[5] = { "water","earth","fire","air","amulet" };

char *ROOMObject(int ID)
{
	if (ID >= HF_WATERSTONE)				/* Handle big treasures */
		return TName[ID-HF_WATERSTONE];
	return OName[ID];						/* Other stuff */
}

/************************************************************************/
/*																		*/
/*									Map type							*/
/*																		*/
/************************************************************************/

int ROOMStyle(int n)
{
	int x,y,w;
	if (IOIsClassic()) return MPT_ROOM;			/* Not in classic mode */
	if (PARAM(PR_NOALTROOM,0)) return MPT_ROOM;
	x = n % MAPSIZE;y = n / MAPSIZE;
	w = MAPSIZE/4;
	if (n == 0) return MPT_GARDEN;
	if (x % w == 0 && x != 0) return MPT_GARDEN;
	if (y % w == 0 && y != 0) return MPT_GARDEN;
	srand(n*417);if (rand()%6 == 0) return MPT_DUNGEON;
	return MPT_ROOM;
}



