/* extuill.c -- Extra utility functions.  */

#include "defs.h"

#ifndef __TEST_
# include <curses.h>
#endif /* testing */

#include <math.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>

#ifndef __TEST__
# include "graphics.h"
#endif /* __TEST__ */

void
fatal (const char *errstr)
{
  /* Display error message and quit */
#ifndef __TEST__
  if (stdscr)			/* if not empty */
    restore_terminal ();
#endif /* __TEST__ */
  fputs (errstr, stderr);
  fputs ("!\n", stderr);
  exit (EXIT_FAILURE);
}

inline void *
xmalloc (size_t size)
{
  /* Safe mem allocation routine */
  register void *value = malloc (size);
  if (value == 0)
    fatal ("Out of memory");
  return value;
}

inline void *
xrealloc (void *ptr, size_t size)
{
  /* Safe mem reallocation routine */
  register void *value = realloc (ptr, size);
  if (value == 0)
    fatal ("Out of memory");
  return value;
}

inline void
xfree (void *ptr)
{
  /* Safe memory-freeing routine */

  if (ptr == 0)
    return;

  free (ptr);
  ptr = 0;
}

int
getrand (int min, int max)
{
  /* Return a random number between or including min and max.  */
  static int rand_init = 0;
  int num;
  if (!rand_init)
    {
      srand (time (0));		/* initialize rand number generator */
      rand_init = 1;		/*   if this is the first run */
    }
  num = (int) rand ();		/* get rand number */
  num = num % (abs (max - min) + 1) + 1; /* take mod to cut range */
  return num;
}

int
lookup (char *item, char *list, char term, int *pos)
{
  /* Return the number corresponding with an item's position in a
     list, starting from 0, or -1 if not found. 'term' represents the
     character that terminates each entry in list. The starting
     character of item in list is returned through 'pos'.  */
  int i = 0, j = 0, cnt = 0;
  if (pos)
    *pos = 0;

  /* Refuse to lookup a nul item or list */
  if (item[0] == 0 || list[0] == 0)
    return -1;

  for (;;)
    {
      if (list[i] != item[0])
	{
	  while (list[i] && list[i] != term)
	    i++;
	}
      else
	{
	  while (list[i] && list[i] != term && item[j] != 0
		  && list[i] == item[j])
	    {
	      i++;
	      j++;
	    }
	  if (item[i] == 0)	/* item found */
	    {
	      if (pos)
		*pos = i - j;
	      return cnt;
	    }
	}
      if (list[i] == 0)
	break;
      cnt++;
    }

  return -1;			/* item not found */
}

#ifdef NEED_MVWHLINE
inline int
mvwhline (WINDOW * win, int row, int col, chtype ch, int n)
{
  /* Substitute mvwhline for pdcurses under DJGPP.  */

  if (wmove (win, row, col) == ERR)	/* did error occur? */
    return ERR;			/* abort */
  return whline (win, ch, n);
}
#endif /* NEED_MVWHLINE */

#ifdef NEED_STRNLEN
/*  Originally from:
 * 
 *  linux/lib/string.c
 *  (from Linux 2.4 sources)
 * 
 * Copyright (C) 1991, 1992  Linus Torvalds
 * 
 */

/**
 * strnlen - Find the length of a length-limited string
 * @s: The string to be sized
 * @count: The maximum number of bytes to search
 */
size_t strnlen (const char *s, size_t count)
{
  const char *sc;

  for (sc = s; count-- && *sc != '\0'; ++sc)
    /* nothing */ ;
  return sc - s;
}
#endif /* need strnlen func */

void
shandler (int sig)
{
  /* Handle signals that we are given.  */

  switch (sig)
    {
      case SIGINT:
	fatal ("Terminated with SIGINT");
	break;
      case SIGQUIT:
	fatal ("Terminated with SIGQUIT");
      case SIGTERM:
	fatal ("Terminated with SIGTERM");
    }
}
