/* game.c -- Contains a listing of all decision-making parts of
             the game.
*/

#include "defs.h"

#ifndef __TEST__
# include <curses.h>
#endif /* __TEST__ */
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef __LINUX__
# include <signal.h>
#endif /* Linux */

#ifndef __TEST__
# include "graphics.h"
#endif /* __TEST__ */
#include "extutil.h"

/* Globals */

/* Difficulty window */
WINDOW *difficulty_win;

int
is_correct_guess (char guess, char *word)
{
  /* Determine if the guess was correct. */
  int i;

  for (i = 0; i < word_length; i++)
    {				/* process all letters in word */
      if (word[i] == guess)
	return 1;
    }
  return 0;			/* letter not found in word */
}

int
add_correct_letter (char guess, char *guessed_word, char *word)
{
  /* Add guessed letter(s) to replace '_' chars, and
     return the score to be awarded. */
  int i, score = 0;

  for (i = 0; i < word_length; i++)
    {
      if (word[i] == guess)
	{
	  guessed_word[i] = guess;
	  score += 10;		/* 10 points per replaced letter */
	}
    }

  return score;
}

void
process_guess (WINDOW * ascii_win, char guess, char *letter_list,
	       char *guessed_word, char *word)
{
  /* Process the user's guess and take action. */
  if (guess == 27)
    {				/* did they hit ESC? */
      lives = -1;
      return;
    }

  if (!isalpha (guess))
    {				/* is guess an alphabetic char */
      mvaddstr (2, 2, "You used an illegal character!    ");
      guess = 0;
      return;
    }

  guess = toupper (guess);	/* convert to uppercase */
  addch (guess);		/* output character typed */
  if (letter_list[guess - 'A'] == '*')
    {				/* is letter used already? */
      mvaddstr (2, 2, "You've already tried that!        ");
      guess = 0;
      return;
    }

  letter_list[guess - 'A'] = '*';	/* blank out guessed letter */

  if (is_correct_guess (guess, word))
    {				/* is guess correct? */
      mvaddstr (2, 2, "You made a correct guess!         ");

      /* Now increment score and add guessed letters. */
      score += add_correct_letter (guess, guessed_word, word);
      return;
    }

  /* Hmm, the guess must be incorrect. */
  mvaddstr (2, 2, "This letter is not in the word!   ");
  lives--;
  score -= 5;
  print_art (ascii_win, lives);	/* draw the next hanging stage */
  wnoutrefresh (ascii_win);	/* refresh the hanging 'window' */
}

#ifdef __LINUX__
void
diff_resize_screen ()
{
  /* Handle resizing the difficulty window.  */
  int maxlines, maxcols;
  
  global_resize_screen ();

  getmaxyx (stdscr, maxlines, maxcols);
  if (mvwin (difficulty_win, (maxlines - 7) / 2, (maxcols - 15) / 2)
      != OK)
    fatal ("Could not move difficulty window");
  
  wmove (difficulty_win, 5, 11);
  wnoutrefresh (difficulty_win);
  doupdate ();
}
#endif /* linux */

int
get_difficulty (void)
{
  /* Prompt the user concerning the difficulty level,
     Returning 1=easy, 2=normal, or 3=hard. Normal is default.  */
  char difficulty = '2';
  int maxlines, maxcols;
  getmaxyx (stdscr, maxlines, maxcols);

  /* Now draw the window with the difficulty question. */
  difficulty_win = subwin (stdscr, 7, 15, (maxlines - 7) / 2,
			   (maxcols - 15) / 2);
  box (difficulty_win, 0, 0);
  mvwaddstr (difficulty_win, 1, 2, "1. Easy");
  mvwaddstr (difficulty_win, 2, 2, "2. Normal");
  mvwaddstr (difficulty_win, 3, 2, "3. Star Wars");
  mvwaddstr (difficulty_win, 5, 2, "Choice: ");
  wnoutrefresh (difficulty_win);
  doupdate ();

#ifdef __LINUX__
  signal (SIGWINCH, diff_resize_screen);
#endif /* linux */
  
  difficulty = wgetch (difficulty_win);	/* fetch keypress */

#ifdef __LINUX__
  signal (SIGWINCH, SIG_IGN);
#endif

  wclear (difficulty_win);
  wnoutrefresh (difficulty_win);
  delwin (difficulty_win);	/* remove difficulty window */

  if (difficulty == 27)
    {				/* ESC pressed */
      restore_terminal ();
      exit (EXIT_FAILURE);
    }

  if (difficulty == '1')
    return 1;
  else if (difficulty == '3')
    return 3;
  else
    return 2;
}

int
end_game (char *word)
{
  /* End of game routine. */
  char ch;
  int maxlines, maxcols;
  WINDOW *game_end_win;

  game_end_win = subwin (stdscr, 4, 40, 6, 4);	/* win for "game over" string */
  box (game_end_win, 0, 0);	/* draw box around it */

  if (lives > 0)
    {				/* user won the game */
      print_welldone (stdscr);
      mvwaddstr (game_end_win, 1, 2, "You saved him!");
    }
  else
    {				/* they lost or quit */
      print_gameover (stdscr);	/* shows "GameOver" graphic on screen */
      score = score > 0 ? 0 : score;	/* score = 0 if positive */
      print_score ();
      print_underscores (word);	/* display the word */

      if (lives == 0)
	{			/* lost */
	  mvwaddstr (game_end_win, 1, 2, "He was hanged!");
	  mvwaddstr (game_end_win, 2, 4, "You lose.");
	}
      else
	{			/* quit */
	  mvwaddstr (game_end_win, 1, 2, "Quitter! I hope you don't give up");
	  mvwaddstr (game_end_win, 2, 4, "this quickly on your homework!");
	}
    }

  getmaxyx (stdscr, maxlines, maxcols);
  mvaddstr (maxlines - 4, 2,
	    "Press any key to play again, ESC exits...");
  wnoutrefresh (game_end_win);	/* make sure BOTH of these windows get updated */
  refresh ();			/*   in the correct order */
  ch = wgetch (game_end_win);	/* get a keypress */
  wclear (game_end_win);
  wnoutrefresh (game_end_win);
  delwin (game_end_win);	/* remove end of game window */
  wnoutrefresh (stdscr);	/* refresh the new top window */

  return ch;
}
