-----------------------------------------------------------------------------
-- rpg.ex                                                                  --
-- A simple RPG game engine developed by Mike Wever.                       --
-----------------------------------------------------------------------------

include graphics.e
include image.e
include get.e
include wildcard.e
include file.e

include rpgglob.e
include rpgspcl.e
include rpgtrn.e
include rpgmap.e
include rpgscrn.e
include rpgwin.e
include rpgitem.e
include rpgchar.e
include rpgmnstr.e
include rpggen.e
include rpggame.e

include mainmap.e


-- set resolution of time() to 1/100th of a second
tick_rate(100)

Map = mainmap
MapSize = {length(Map), length(Map[1])}

sequence KeyBuffer
KeyBuffer = repeat(' ',20)

-----------------------------------------------------------------------------
-- Wait for the user to press a key.  Store the key in a buffer to check   --
-- for "cheat codes".                                                      --
-----------------------------------------------------------------------------
function waitForKey()
  integer k
  integer s

  k = wait_key()
  KeyBuffer = KeyBuffer[2..20] & k
  if KeyBuffer[20] = #119 then -- ALT-P
    s = PrintScreen()
    if s = BMP_SUCCESS then
      MessageBox(1, 1, "Screenshot saved as scrnshot.bmp")
    else
      MessageBox(1, 1, sprintf("Error creating screenshot. (%d)",{s}))
    end if
  elsif equal(KeyBuffer[18..20],"xxx") then
    MessageBox(0,0,sprintf("FP %d(%d)  WS %d  EP %d  DMG: %d  PROT %d",
                           {CharRec[CH_FP],CharRec[CH_FP_MAX],
                            CharRec[CH_WS],CharRec[CH_EXP],
                            CharRec[CH_DMG],CharRec[CH_PROT]}))
  end if
  return k
end function
waitForKeyID = routine_id("waitForKey")

-----------------------------------------------------------------------------
-- Draw a tile at a given location                                         --
-- Parameters:                                                             --
--   sequence: screen location to draw at {x,y}                            --
--   integer:  terrain type to draw                                        --
-----------------------------------------------------------------------------
procedure drawTile(sequence loc, integer t)
  for i = 0 to length(Terrain[t]) - 1 do
    SetPixels(BACK_SCREEN,Terrain[t][i + 1],{loc[1],loc[2] + i})
  end for
end procedure

-----------------------------------------------------------------------------
-- Paint the terrain and character on the screen.                          --
-----------------------------------------------------------------------------
procedure paintScreen()
  integer startX, endX, startY, endY
  integer x, y
  
  startY = CharPos[2] - (TilesPerHalfScreen[2])
  if startY > MapSize[2] - (TilesPerScreen[2] - 1) then
    startY = MapSize[2] - (TilesPerScreen[2] - 1)
  end if
  if startY < 1 then
    startY = 1
  end if
  endY = startY + (TilesPerScreen[2] - 1)
  if endY > MapSize[2] then
    endY = MapSize[2]
  end if
  startX = CharPos[1] - (TilesPerHalfScreen[1])
  if startX > MapSize[1] - (TilesPerScreen[1] - 1) then
    startX = MapSize[1] - (TilesPerScreen[1] - 1)
  end if
  if startX < 1 then
    startX = 1
  end if
  endX = startX + (TilesPerScreen[1] - 1)
  if endX > MapSize[1] then
    endX = MapSize[1]
  end if
  x = 0
  y = 0
  for i = startY to endY do
    for ii = startX to endX do
      drawTile({x,y},Map[i][ii][MAP_TERRAIN])
      x += TILE_SIZE[1]
    end for
    x = 0
    y += TILE_SIZE[2]
  end for
  x = (CharPos[1] - startX) * TILE_SIZE[1]
  y = (CharPos[2] - startY) * TILE_SIZE[2]
  DrawCharacter({x,y})
end procedure
paintScreenID = routine_id("paintScreen")

-----------------------------------------------------------------------------
-- Determine what to do for a keypress.                                    --
-- Parameters:                                                             --
--   integer: key pressed                                                  --
-----------------------------------------------------------------------------
procedure handleKey(integer key)
  integer k
  sequence inv
  integer item
  integer fn

  k = lower(key)  
  if k    = #0148 then -- up arrow
    -- Move up
    MoveCharacter({0,-1})
  elsif k = #0150 then -- down arrow
    -- Move down
    MoveCharacter({0,+1})
  elsif k = #014B then -- left arrow
    -- Move left
    MoveCharacter({-1,0})
  elsif k = #014D then -- right arrow
    -- Move right
    MoveCharacter({+1,0})
  elsif k = #0068 then -- 'h'
    -- Display help
    MessageBox(10, 10, "Movement.......cursor keys\n"
                     & "Drop Item......D key\n"
                     & "Help...........H key\n"
                     & "Inventory......I key\n"
                     & "Save Game......S key\n"
                     & "Use Item.......U key\n"
                     & "Close Windows..Space bar\n"
                     & "Screen Shot....ALT-P\n"
                     & "\n"
                     & "Quit...........Escape key (ESC)")
  end if
  if k = #0064 then -- 'd'
    DropItem()
  end if
  if k = #0069 then -- 'i'
    -- Display the character's inventory
    inv = "You are carrying..."
    for i = 1 to length(CharRec[CH_INV]) do
      inv &= "\n" & Items[CharRec[CH_INV][i]][ITEM_NAME]
    end for
    inv &= sprintf("\n%s (weapon)\n%s (shield)\n%d gold bit",
                   {CharRec[CH_WPN],CharRec[CH_SHLD],CharRec[CH_BITS]})
    if CharRec[CH_BITS] != 1 then
      inv &= 's'
    end if
    MessageBox(10, 10, inv)
  end if
  if k = #0073 then -- 's'
    -- Save the game
    fn = open("savegame.dat","w")
    if fn = -1 then
      MessageBox(10, 10, "Can not write the saved game file.")
    else
      print(fn,CharRec)
      puts(fn,"\n")
      print(fn,CharPos)
      puts(fn,"\n")
      print(fn,SpecialFlags)
      close(fn)
      MessageBox(10, 10, "Game saved.")
    end if
  end if
  if k = #0075 then -- 'u'
    -- Use an item from the character's inventory
    UseItem({10,10})
  end if
end procedure

-----------------------------------------------------------------------------
-- Main logic for the game.                                                --
-----------------------------------------------------------------------------
procedure doGame()
  integer exitFlag
  integer key
  atom last
  
  hWinCFp = LoadWindow(ScreenWidth - 24, 5, ScreenWidth - 5, 9)
  DrawCharFP(hWinCFp)
  exitFlag = 0
  key = 0
  last = time()
  paintScreen()
  while not exitFlag and not GameWon do
    if key = #001B then -- ESC key exits the program
      exitFlag = 1
      exit
    end if
    handleKey(key)
    if CharRec[CH_FP] < 1 then
      exitFlag = 1
      exit
    end if
    if DoSpecial then
      if SpecialRoutines[DoSpecial] != -1 then
        call_proc(SpecialRoutines[DoSpecial],{})
      end if
      DoSpecial = 0
    end if
    if CharRec[CH_FP] < 1 then
      exitFlag = 1
      exit
    end if
    FlipScreen()
    -- enforce minimum action time of 1/15th of a second
    while time() - last < .067 do
    end while
    key = waitForKey()
    last = time()
    -- flush out the key buffer; we don't want to let keys accumulate
    while get_key() != -1 do
    end while
  end while
  if CharRec[CH_FP] < 1 then
    ClearWorkScreen()
    FlipScreen()
    MessageBox(10,10,"You are dead.")
  elsif GameWon then
    ClearWorkScreen()
    MessageBox(10, 10, "You won!  Yay for you!  Now play\n"
                     & "again and again until you have\n"
                     & "unlocked every secret.  Go on, do\n"
                     & "it.  You know you want to.")
  end if
end procedure

-----------------------------------------------------------------------------
-- Show the opening sequence                                               --
-----------------------------------------------------------------------------
procedure openingSequence()
  MessageBox(10, 10, "Once there lived in the Graun Forest a simple\n"
                   & "man named Mike.  Mike loved plants more than\n"
                   & "people, so he moved to the farthest corner of\n"
                   & "the forest in order that there would be no one\n"
                   & "around to distract him from his true passion,\n"
                   & "which was growing carrots.\n"
                   & "\n"
                   & "Mike was a master carrot grower.  His carrots\n"
                   & "grew in size each year, and he always took\n"
                   & "home the ribbon for best carrots at the\n"
                   & "county fair.")
  MessageBox(10, 10, "As one might expect on a carrot farm, many\n"
                   & "rabbits lived around Mike's house.  Usually\n"
                   & "Mike loved to have the cute, furry creatures\n"
                   & "around.  But recently they began to act\n"
                   & "strangely.  They were not satisfied with the\n"
                   & "substandard carrots that Mike left for them\n"
                   & "to eat, though the carrots were the size of a\n"
                   & "man's arm.  The rabbits rose up, determined to\n"
                   & "take the carrots that they felt rightfully\n"
                   & "belonged to them.")
  MessageBox(10, 10, "Mike remembered the story of a magic flute\n"
                   & "that was lost in the woods near his house ages\n"
                   & "ago.  When you played the flute, the story\n"
                   & "went, it calmed the heart of any wild beasts\n"
                   & "that meant you harm.  It was his only chance.\n"
                   & "\n"
                   & "So Mike grabbed his largest kitchen knife, and\n"
                   & "he headed out in search of the flute...")
end procedure

-----------------------------------------------------------------------------
-- Load a saved game                                                       --
-----------------------------------------------------------------------------
procedure loadGame()
  integer fn
  sequence s
  
  fn = open("savegame.dat","r")
  if fn = -1 then
    MessageBox(10, 10, "Can not read the saved game file.")
  else
    s = get(fn)
    if s[1] = GET_SUCCESS then
      CharRec = s[2]
      s = get(fn)
      if s[1] = GET_SUCCESS then
        CharPos = s[2]
        s = get(fn)
        if s[1] = GET_SUCCESS then
          SpecialFlags = s[2]
          for i = 1 to length(SpecialFlags) do
            if SpecialFlagIsOn(i) then
              DoSpecialFlagLoadRtn(i)
            end if
          end for
        end if
      end if
    end if
    close(fn)
    if s[1] != GET_SUCCESS then
      MessageBox(10, 10, "Saved game file corrupted!  Can not load the game.")
    else
      MessageBox(10, 10, "Game loaded.")
    end if
  end if
end procedure

-----------------------------------------------------------------------------
-- Opening Menu                                                            --
-----------------------------------------------------------------------------
integer choice

InitScreen()

set_active_page(1)
choice = 0
while choice != 3 do
  choice = SelectBox(10, 10, "Welcome to Mike's Story.",
                     {"Start a new game","Continue a saved game","Quit"})
  if choice = 1 then
    InitChar()
    InitSpecial()
    CharPos = {3,3}
    openingSequence()
  elsif choice = 2 then
    loadGame()
  end if
  if choice = 1 or choice = 2 then
    DrawMode = MAIN_GAME
    doGame()
    DrawMode = START_MENU
    ClearWorkScreen()
  end if
end while
DefaultScreen()