-----------------------------------------------------------------------------
-- rpgscrn.e                                                               --
-- Routines for manipulating the screen for rpg.ex                         --
--                                                                         --
-- written by Mike Wever                                                   --
-----------------------------------------------------------------------------

include image.e

-- Define available colors
global constant RPG_BLACK = 0,
                RPG_RED = 1,
                RPG_GREEN = 2,
                RPG_BLUE = 3,
                RPG_YELLOW = 4,
                RPG_GRAY = 5,
                RPG_DARK_GREEN = 6,
                RPG_WHITE = 7,
                RPG_BROWN = 8,
                RPG_LIGHT_BLUE = 10

global sequence Colors
Colors = 
{ {  0,  0,  0},  -- 0 = Black
  { 63,  0,  0},  -- 1 = Red
  {  0, 63,  0},  -- 2 = Green
  {  0,  0, 63},  -- 3 = Blue
  { 63, 63,  0},  -- 4 = Yellow
  { 48, 48, 48},  -- 5 = Gray
  {  0, 32,  0},  -- 6 = Dark Green
  { 63, 63, 63},  -- 7 = White
  { 48, 32,  0},  -- 8 = Brown
  { 32, 32, 63},  -- 9 = Light Blue
  {  0,  0, 48},  -- 10 = Dark Blue
  {255,255,255},  -- 11 = Undefined
  {255,255,255},  -- 12 = Undefined
  {255,255,255},  -- 13 = Undefined
  {255,255,255},  -- 14 = Undefined
  {255,255,255},  -- 15 = Undefined
  {  0,  0,  0}}  -- 16 = Black (color 16 is border in some graphics modes)




global integer DisplayPage
global atom paintScreenID

-- The work screen that all windows are drawn to before writing to video.
sequence WorkScreen

-- List of window to be displayed on the screen
sequence Windows

-- Window 1 is the main drawing screen or "back screen"
global constant BACK_SCREEN = 1

-- Define screen info
global integer MinX, MaxX, MinY, MaxY
global integer ScreenWidth, ScreenHeight
global sequence TilesPerScreen, TilesPerHalfScreen

-- Define screen drawing modes
global constant START_MENU = 0
global constant MAIN_GAME = 1

global integer DrawMode
DrawMode = START_MENU

-----------------------------------------------------------------------------
-- Initialize the list of windows to be displayed.                         --
-----------------------------------------------------------------------------
procedure InitWindows()
  Windows = {{{0, 0}, repeat(repeat(RPG_BLACK, ScreenWidth), ScreenHeight)}}
end procedure

-----------------------------------------------------------------------------
-- Put into 320 x 200 with 256 colors, 1 page                              --
-- This is the fastest graphics mode                                       --
-----------------------------------------------------------------------------
global procedure InitScreen()
  sequence vc

  if graphics_mode(19) then
    puts(1, "Graphics failed to initialize.\n")
    abort(1)
  end if
  all_palette(Colors)
  vc = video_config()
  MinX = 0
  MaxX = vc[VC_XPIXELS] - TILE_SIZE[1]
  ScreenWidth = vc[VC_XPIXELS]
  MinY = 0
  MaxY = vc[VC_YPIXELS] - TILE_SIZE[2]
  ScreenHeight = vc[VC_YPIXELS]
  TilesPerScreen = {floor(vc[VC_XPIXELS] / TILE_SIZE[1]),
                    floor(vc[VC_YPIXELS] / TILE_SIZE[2])}
  TilesPerHalfScreen = {floor(TilesPerScreen[1] / 2),
                        floor(TilesPerScreen[2] / 2)}
  DisplayPage = get_display_page()
  InitWindows()
  WorkScreen = repeat(repeat(RPG_BLACK, ScreenWidth), ScreenHeight)
end procedure

-----------------------------------------------------------------------------
-- Return to standard text mode                                            --
-----------------------------------------------------------------------------
global procedure DefaultScreen()
  if graphics_mode(-1) then
    puts(1, "Failed to return to default screen mode.\n")
    abort(1)
  end if
end procedure

-----------------------------------------------------------------------------
-- Load a window into the list to be displayed.                            --
-- Parameters:                                                             --
--   atom: left (x coordinate of top left corner)                          --
--   atom: top (y coordinate of top left corner)                           --
--   atom: right (x coordinate of bottom right corner)                     -- 
--   atom: bottom (y coordinate of bottom right corner)                    --
-- Returns:                                                                --
--   atom: handle for the window (position in the sequence)                --
-----------------------------------------------------------------------------
global function LoadWindow(atom left, atom top, atom right, atom bottom)
  atom width, height
  
  width = (right - left) + 1
  height = (bottom - top) + 1
  Windows = append(Windows, {{left,top},repeat(repeat(0,width),height)})
  return length(Windows)
end function

-----------------------------------------------------------------------------
-- Give dimensions of a window.                                            --
-- Parameters:                                                             --
--   atom: handle for the window                                           --
-- Returns:                                                                --
--   sequence: dimensions {width, height}                                  --
-----------------------------------------------------------------------------
global function WindowDims(atom hWin)
  atom width, height
  
  height = length(Windows[hWin][2])
  width = length(Windows[hWin][2][1])

  return {width, height}
end function

-----------------------------------------------------------------------------
-- Unload a window from the list to be displayed.                          --
-- Parameters:                                                             --
--   atom: window handle                                                   --
-----------------------------------------------------------------------------
global procedure UnloadWindow(atom handle)
  -- Prevent error if user tries to unload non-existent window
  -- Don't allow window 1 (the main screen) to be unloaded
  if handle > 1 and handle <= length(Windows) then
    Windows = Windows[1..handle - 1] & Windows[handle + 1..length(Windows)]
  end if
end procedure

-----------------------------------------------------------------------------
-- Unload all windows from the list to be displayed.                       --
-----------------------------------------------------------------------------
global procedure UnloadAllWindows()
  InitWindows()
end procedure

-----------------------------------------------------------------------------
-- Clear the work screen                                                   --
-----------------------------------------------------------------------------
global procedure ClearWorkScreen()
  InitWindows()
end procedure

-----------------------------------------------------------------------------
-- "Flip" the screen by switching active and display pages.  This is a way --
-- to prevent screen flicker caused by drawing while the screen is being   --
-- refreshed on the monitor.                                               --
-- Note: Switching active and display pages does nothing in graphics modes --
--       with only one page.                                               --
-----------------------------------------------------------------------------
global procedure FlipScreen()
  integer x, y, initX

  DisplayPage += 1
  if DisplayPage > 1 then
    DisplayPage = 0
  end if
  -- Draw all windows to a work screen before writing to the screen.  This
  -- should minimize flicker in single-page graphic modes.
  for i = 1 to length(Windows) do
    x = Windows[i][1][1] + 1
    y = Windows[i][1][2] + 1
    initX = x
    for ii = 1 to length(Windows[i][2]) do
      for iii = 1 to length(Windows[i][2][ii]) do
        WorkScreen[y][x] = Windows[i][2][ii][iii]
        x += 1
      end for
      x = initX
      y += 1
    end for
  end for
  for i = 0 to ScreenHeight - 1 do
    pixel(WorkScreen[i + 1],{0,i})
  end for
  set_display_page(DisplayPage)
  if DisplayPage = 1 then
    set_active_page(0)
  else
    set_active_page(1)
  end if
end procedure

-----------------------------------------------------------------------------
-- Set a pixel or line of pixels on the work screen.  This routine does    --
-- not perform clipping, to increase the efficiency.                       --
-- Parameters:                                                             --
--   atom:     handle of window to draw on                                 --
--   object:   an integer containing the color value to set the pixel to   --
--             or a sequence of such integers                              --
--   sequence: a sequence holding the first position to set {x,y}          --
-----------------------------------------------------------------------------
global procedure SetPixels(atom hWin, object val,sequence loc)
  integer x, y

  x = loc[1] + 1
  y = loc[2] + 1
  if atom(val) then
    Windows[hWin][2][y][x] = val
  else
    for i = 1 to length(val) do
      Windows[hWin][2][y][x] = val[i]
      x += 1
    end for
  end if
end procedure

-----------------------------------------------------------------------------
-- Wrapper to allow paintScreen to be called from anywhere                 --
-----------------------------------------------------------------------------
global procedure PaintScreen()
  call_proc(paintScreenID,{})
end procedure

-----------------------------------------------------------------------------
-- Save a bitmap file with the current screen image.                       --
-- Return:                                                                 --
--  integer: return value from save_bitmap()                               --
-----------------------------------------------------------------------------
global function PrintScreen()
  sequence pal
  atom x, y, initX

  -- Get the current palette and convert to ranges expected by save_bitmap()
  pal = get_all_palette()
  for i = 1 to length(pal) do
    for ii = 1 to length(pal[i]) do
      pal[i][ii] *= 4
    end for
  end for
  
  -- Draw all windows to a work screen for the bitmap.
  for i = 1 to length(Windows) do
    x = Windows[i][1][1] + 1
    y = Windows[i][1][2] + 1
    initX = x
    for ii = 1 to length(Windows[i][2]) do
      for iii = 1 to length(Windows[i][2][ii]) do
        WorkScreen[y][x] = Windows[i][2][ii][iii]
        x += 1
      end for
      x = initX
      y += 1
    end for
  end for

  -- Save the bitmap
  return save_bitmap({pal, WorkScreen}, "scrnshot.bmp")
end function