-----------------------------------------------------------------------------
-- rpgwin.e                                                                --
-- Routines for drawing windows for rpg.ex                                 --
--                                                                         --
-- written by Mike Wever                                                   --
-----------------------------------------------------------------------------

-- Define font size
global constant FONT_WIDTH = 6
global constant FONT_HEIGHT = 10

global integer FontColor
FontColor = RPG_BLACK

sequence fontIndex
-- High Byte  0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F
fontIndex = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  -- 0  Low Byte
              0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  -- 1
              0,65,68, 0, 0, 0, 0,67,69,70, 0, 0,63,71,64, 0,  -- 2
             53,54,55,56,57,58,59,60,61,62, 0, 0, 0, 0, 0,66,  -- 3
              0,27,28,29,30,31,32,33,34,35,36,37,38,39,40,41,  -- 4
             42,43,44,45,46,47,48,49,50,51,52, 0, 0, 0, 0, 0,  -- 5
              0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,15,  -- 6
             16,17,18,19,20,21,22,23,24,25,26, 0, 0, 0, 0, 0,  -- 7
              0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  -- 8
              0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  -- 9
              0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  -- A
              0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  -- B
              0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  -- C
              0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  -- D
              0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,  -- E
              0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 } -- F

sequence font
font = 
{ { "      ",
    "      ",
    " XX   ",
    "X  X  ",
    "X  X  ",
    "X  X  ",
    " XX X ",
    "      ",
    "      " },

  { "X     ",
    "X     ",
    "XXXX  ",
    "X   X ",
    "X   X ",
    "X   X ",
    "XXXX  ",
    "      ",
    "      " },

  { "      ",
    "      ",
    " XXXX ",
    "X     ",
    "X     ",
    "X     ",
    " XXXX ",
    "      ",
    "      " },

  { "    X ",
    "    X ",
    " XXXX ",
    "X   X ",
    "X   X ",
    "X   X ",
    " XXXX ",
    "      ",
    "      " },
 
  { "      ",
    "      ",
    " XXX  ",
    "X   X  ",
    "XXXXX ",
    "X     ",
    " XXXX ",
    "      ",
    "      " },
 
  { "   X  ",
    "  X X ",
    "  X   ",
    " XXX  ",
    "  X   ",
    "  X   ",
    "  X   ",
    "      ",
    "      " },
 
  { "      ",
    "      ",
    " XXX  ",
    "X   X ",
    "X   X ",
    "X   X ",
    " XXXX ",
    "    X ",
    "XXXX  " },

  { "X     ",
    "X     ",
    "XXXX  ",
    "X   X ",
    "X   X ",
    "X   X ",
    "X   X ",
    "      ",
    "      " },
 
  { "  X   ",
    "      ",
    "  X   ",
    "  X   ",
    "  X   ",
    "  X   ",
    "  X   ",
    "      ",
    "      " },
 
  { "    X ",
    "      ",
    "    X ",
    "    X ",
    "    X ",
    "    X ",
    "    X ",
    "    X ",
    " XXX  " },
 
  { "X     ",
    "X     ",
    "X   X ",
    "X  X  ",
    "XXX   ",
    "X  X  ",
    "X   X ",
    "      ",
    "      " },
 
  { "  X   ",
    "  X   ",
    "  X   ",
    "  X   ",
    "  X   ",
    "  X   ",
    "  X   ",
    "      ",
    "      " },
 
  { "      ",
    "      ",
    " X X  ",
    "X X X ",
    "X   X ",
    "X   X ",
    "X   X ",
    "      ",
    "      " },
 
  { "      ",
    "      ",
    " XXX  ",
    " X  X ",
    " X  X ",
    " X  X ",
    " X  X ",
    "      ",
    "      " },
 
  { "      ",
    "      ",
    " XXX  ",
    "X   X ",
    "X   X ",
    "X   X ",
    " XXX  ",
    "      ",
    "      " },
 
  { "      ",
    "      ",
    "XXXX  ",
    "X   X ",
    "X   X ",
    "X   X ",
    "XXXX  ",
    "X     ",
    "X     " },
 
  { "      ",
    "      ",
    " XXXX ",
    "X   X ",
    "X   X ",
    "X   X ",
    " XXXX ",
    "    X ",
    "    X " },
 
  { "      ",
    "      ",
    " XXX  ",
    " X  X ",
    " X    ",
    " X    ",
    " X    ",
    "      ",
    "      " },
 
  { "      ",
    "      ",
    " XXXX ",
    "X     ",
    " XXX  ",
    "    X ",
    "XXXX  ",
    "      ",
    "      " },
 
  { "      ",
    "  X   ",
    "XXXXX ",
    "  X   ",
    "  X   ",
    "  X   ",
    "  X   ",
    "      ",
    "      " },
 
  { "      ",
    "      ",
    "X  X  ",
    "X  X  ",
    "X  X  ",
    "X  X  ",
    " XX X ",
    "      ",
    "      " },
 
  { "      ",
    "      ",
    "X   X ",
    "X   X ",
    "X   X ",
    " X X  ",
    "  X   ",
    "      ",
    "      " },
 
  { "      ",
    "      ",
    "X   X ",
    "X   X ",
    "X   X ",
    "X X X ",
    " X X  ",
    "      ",
    "      " },
 
  { "      ",
    "      ",
    "X   X ",
    " X X  ",
    "  X   ",
    " X X  ",
    "X   X ",
    "      ",
    "      " },
 
  { "      ",
    "      ",
    "X   X ",
    "X   X ",
    "X   X ",
    "X   X ",
    " XXXX ",
    "    X ",
    "XXXX  " },
 
  { "      ",
    "      ",
    "XXXXX ",
    "   X  ",
    "  X   ",
    " X    ",
    "XXXXX ",
    "      ",
    "      " },
 
  { "  X   ",
    " X X  ",
    "X   X ",
    "XXXXX ",
    "X   X ",
    "X   X ",
    "X   X ",
    "      ",
    "      " },

  { "XXXX  ",
    "X   X ",
    "X   X ",
    "XXXX  ",
    "X   X ",
    "X   X ",
    "XXXX  ",
    "      ",
    "      " },

  { " XXXX ",
    "X     ",
    "X     ",
    "X     ",
    "X     ",
    "X     ",
    " XXXX ",
    "      ",
    "      " },

  { "XXXX  ",
    "X   X ",
    "X   X ",
    "X   X ",
    "X   X ",
    "X   X ",
    "XXXX  ",
    "      ",
    "      " },

  { "XXXXX ",
    "X     ",
    "X     ",
    "XXX   ",
    "X     ",
    "X     ",
    "XXXXX ",
    "      ",
    "      " },

  { "XXXXX ",
    "X     ",
    "X     ",
    "XXX   ",
    "X     ",
    "X     ",
    "X     ",
    "      ",
    "      " },

  { " XXX  ",
    "X   X ",
    "X     ",
    "X     ",
    "X  XX ",
    "X   X ",
    " XXX  ",
    "      ",
    "      " },

  { "X   X ",
    "X   X ",
    "X   X ",
    "XXXXX ",
    "X   X ",
    "X   X ",
    "X   X ",
    "      ",
    "      " },

  { " XXX  ",
    "  X   ",
    "  X   ",
    "  X   ",
    "  X   ",
    "  X   ",
    " XXX  ",
    "      ",
    "      " },

  { "    X ",
    "    X ",
    "    X ",
    "    X ",
    "    X ",
    "X   X ",
    " XXX  ",
    "      ",
    "      " },

  { "X   X ",
    "X  X  ",
    "X X   ",
    "XX    ",
    "X X   ",
    "X  X  ",
    "X   X ",
    "      ",
    "      " },

  { "X     ",
    "X     ",
    "X     ",
    "X     ",
    "X     ",
    "X     ",
    "XXXXX ",
    "      ",
    "      " },

  { "X   X ",
    "XX XX ",
    "XX XX ",
    "X X X ",
    "X   X ",
    "X   X ",
    "X   X ",
    "      ",
    "      " },

  { "X   X ",
    "XX  X ",
    "XX  X ",
    "X X X ",
    "X X X ",
    "X  XX ",
    "X   X ",
    "      ",
    "      " },

  { " XXX  ",
    "X   X ",
    "X   X ",
    "X   X ",
    "X   X ",
    "X   X ",
    " XXX  ",
    "      ",
    "      " },

  { "XXXX  ",
    "X   X ",
    "X   X ",
    "XXXX  ",
    "X     ",
    "X     ",
    "X     ",
    "      ",
    "      " },

  { " XXX  ",
    "X   X ",
    "X   X ",
    "X   X ",
    "X X X ",
    "X  X  ",
    " XX X ",
    "      ",
    "      " },

  { "XXXX  ",
    "X   X ",
    "X   X ",
    "XXXX  ",
    "X X   ",
    "X  X  ",
    "X   X ",
    "      ",
    "      " },

  { " XXXX ",
    "X     ",
    "X     ",
    " XXX  ",
    "    X ",
    "    X ",
    "XXXX  ",
    "      ",
    "      " },

  { "XXXXX ",
    "  X   ",
    "  X   ",
    "  X   ",
    "  X   ",
    "  X   ",
    "  X   ",
    "      ",
    "      " },

  { "X   X ",
    "X   X ",
    "X   X ",
    "X   X ",
    "X   X ",
    "X   X ",
    " XXX  ",
    "      ",
    "      " },

  { "X   X ",
    "X   X ",
    "X   X ",
    "X   X ",
    "X   X ",
    " X X  ",
    "  X   ",
    "      ",
    "      " },

  { "X   X ",
    "X   X ",
    "X   X ",
    "X X X ",
    "XX XX ",
    " X X  ",
    " X X  ",
    "      ",
    "      " },

  { "X   X ",
    "X   X ",
    " X X  ",
    "  X   ",
    " X X  ",
    "X   X ",
    "X   X ",
    "      ",
    "      " },

  { "X   X ",
    "X   X ",
    " X X  ",
    "  X   ",
    "  X   ",
    "  X   ",
    "  X   ",
    "      ",
    "      " },

  { "XXXXX ",
    "    X ",
    "   X  ",
    "  X   ",
    " X    ",
    "X     ",
    "XXXXX ",
    "      ",
    "      " },

  { " XXX  ",
    "X   X ",
    "X   X ",
    "X X X ",
    "X   X ",
    "X   X ",
    " XXX  ",
    "      ",
    "      " },

  { "  X   ",
    " XX   ",
    "  X   ",
    "  X   ",
    "  X   ",
    "  X   ",
    " XXX  ",
    "      ",
    "      " },

  { " XXX  ",
    "X   X ",
    "    X ",
    "   X  ",
    "  X   ",
    " X    ",
    "XXXXX ",
    "      ",
    "      " },

  { " XXX  ",
    "X   X ",
    "    X ",
    "  XX  ",
    "    X ",
    "X   X ",
    " XXX  ",
    "      ",
    "      " },

  { "X  X  ",
    "X  X  ",
    "X  X  ",
    "XXXXX ",
    "   X  ",
    "   X  ",
    "   X  ",
    "      ",
    "      " },

  { "XXXXX ",
    "X     ",
    "X     ",
    "XXXX  ",
    "    X ",
    "    X ",
    "XXXX  ",
    "      ",
    "      " },

  { " XXX  ",
    "X   X ",
    "X     ",
    "XXXX  ",
    "X   X ",
    "X   X ",
    " XXX  ",
    "      ",
    "      " },

  { "XXXXX ",
    "    X ",
    "   X  ",
    "  X   ",
    " X    ",
    " X    ",
    " X    ",
    "      ",
    "      " },

  { " XXX  ",
    "X   X ",
    "X   X ",
    " XXX  ",
    "X   X ",
    "X   X ",
    " XXX  ",
    "      ",
    "      " },

  { " XXX  ",
    "X   X ",
    "X   X ",
    " XXXX ",
    "    X ",
    "X   X ",
    " XXX  ",
    "      ",
    "      " },

  { "      ",
    "      ",
    "      ",
    "      ",
    "      ",
    "      ",
    "  X   ",
    " X    ",
    "      " },

  { "      ",
    "      ",
    "      ",
    "      ",
    "      ",
    "      ",
    "  X   ",
    "      ",
    "      " },

  { "  X   ",
    " XXX  ",
    " XXX  ",
    " XXX  ",
    "  X   ",
    "      ",
    "  X   ",
    "      ",
    "      " },

  { " XXX  ",
    "X   X ",
    "X   X ",
    "   X  ",
    "  X   ",
    "      ",
    "  X   ",
    "      ",
    "      " } ,

  { "      ",
    "  X   ",
    "  X   ",
    "      ",
    "      ",
    "      ",
    "      ",
    "      ",
    "      " } ,

  { " X X  ",
    " X X  ",
    "      ",
    "      ",
    "      ",
    "      ",
    "      ",
    "      ",
    "      " },

  { "   X  ",
    "  X   ",
    " X    ",
    " X    ",
    " X    ",
    "  X   ",
    "   X  ",
    "      ",
    "      " },

  { " X    ",
    "  X   ",
    "   X  ",
    "   X  ",
    "   X  ",
    "  X   ",
    " X    ",
    "      ",
    "      " },

  { "      ", -- 71
    "      ",
    "      ",
    " XXX  ",
    "      ",
    "      ",
    "      ",
    "      ",
    "      " } }


-- Will point to function for catching keystrokes
global atom waitForKeyID

-- Define text box sequence entries
global constant TEXT_BOX_TL = 1 -- top left
global constant TEXT_BOX_TR = 2 -- top right
global constant TEXT_BOX_BL = 3 -- bottom left
global constant TEXT_BOX_BR = 4 -- bottom right
global constant TEXT_BOX_T  = 5 -- top size
global constant TEXT_BOX_B  = 6 -- bottom side
global constant TEXT_BOX_L  = 7 -- left side
global constant TEXT_BOX_R  = 8 -- right side

sequence TextBoxPieces
TextBoxPieces = 
  -- top left
{ { {  9, 9, 9, 9 },
    {  9, 3, 3, 3 },
    {  9, 3, 3, 3 },
    {  9, 3, 3, 3 } },
  -- top right
  { {  9, 9, 9, 3 },
    {  3, 3, 3,10 },
    {  3, 3, 3,10 },
    {  3, 3, 3,10 } },
  -- bottom left
  { {  9, 3, 3, 3 },
    {  9, 3, 3, 3 },
    {  9, 3, 3, 3 },
    {  3,10,10,10 } },
  -- bottom right
  { {  3, 3, 3,10 },
    {  3, 3, 3,10 },
    {  3, 3, 3,10 },
    { 10,10,10,10 } },
  -- top
  {  9,
     3,
     7,
     7 },
  -- bottom
  {  7,
     7,
     3,
    10 },
  -- left
  {  9, 3, 7, 7 },
  -- right
  {  7, 7, 3,10 } }


------------------------------------------------------------------------
-- Write text to the screen                                           --
--                                                                    --
-- Parameters:                                                        --
--   atom:     Handle of window to write in                           --
--   sequence: Location to write at {x,y}                             --
--   sequence: Text to write                                          --
------------------------------------------------------------------------
global procedure WriteText(atom hWin, sequence loc, sequence text)
  integer off2
  sequence line
  integer c
  integer y
  
  line = repeat(repeat(0,length(text) * FONT_WIDTH),FONT_HEIGHT - 1)
  off2 = 0
  for i = 1 to length(text) do
    c = fontIndex[text[i] + 1]
    for ii = 1 to FONT_HEIGHT - 1 do
      for iii = 1 to FONT_WIDTH do
        off2 += 1
        if c != 0 and font[c][ii][iii] != ' ' then
          line[ii][off2] = FontColor
        else
          line[ii][off2] = 7
        end if
      end for
      if ii < FONT_HEIGHT - 1 then
        off2 -= FONT_WIDTH
      end if
    end for
  end for
  y = loc[2]
  for i = 1 to FONT_HEIGHT - 1 do
    SetPixels(hWin,line[i],{loc[1],y})
    y += 1
  end for
end procedure

-----------------------------------------------------------------------------
-- Split a sequence into lines of text.                                    --
-- Parameters:                                                             --
--   sequence: data to split                                               --
-- Returns:                                                                --
--   sequence: first entry is length of longest line, other entries are    --
--             list of strings of text                                     --
-----------------------------------------------------------------------------
function splitText(sequence input)
  sequence lines
  integer b, e, len, longestLine

  -- Break the text into lines, keeping track of the longest line
  lines = {}
  longestLine = 0
  b = 1
  for c = 1 to length(input) do
    if input[c] = '\n' then
      e = c - 1
      lines = append(lines,input[b..e])
      len = (e - b) + 1
      if len > longestLine then
        longestLine = len
      end if
      b = c + 1
    end if
  end for
  if b <= length(input) then
    lines = append(lines,input[b..length(input)])
    len = (length(input) - b) + 1
    if len > longestLine then
      longestLine = len
    end if
  end if
  return {longestLine,lines}
end function

-----------------------------------------------------------------------------
-- Draw a text box on the screen at a given location                       --
--   atom: handle of the window that is a text box                         --
-----------------------------------------------------------------------------
global procedure DrawTextBox(atom hWin)
  integer width, height
  integer topLen, sideLen
  integer x, y
  sequence line
  sequence wDims

  wDims = WindowDims(hWin)
  width = wDims[1]
  height = wDims[2]
  topLen = width - 8
  if topLen < 0 then
    topLen = 0
  end if
  sideLen = height - 8
  if sideLen < 0 then
    sideLen = 0
  end if

  x = 0
  y = 0
  for i = 1 to 4 do
    line = TextBoxPieces[TEXT_BOX_TL][i]
           & repeat(TextBoxPieces[TEXT_BOX_T][i],topLen)
           & TextBoxPieces[TEXT_BOX_TR][i]
    SetPixels(hWin,line,{x,y})
    y += 1
  end for
  for i = 1 to sideLen do
    line = TextBoxPieces[TEXT_BOX_L]
           & repeat(7,topLen)
           & TextBoxPieces[TEXT_BOX_R]
    SetPixels(hWin,line,{x,y})
    y += 1
  end for
  for i = 1 to 4 do
    line = TextBoxPieces[TEXT_BOX_BL][i]
           & repeat(TextBoxPieces[TEXT_BOX_B][i],topLen)
           & TextBoxPieces[TEXT_BOX_BR][i]
    SetPixels(hWin,line,{x,y})
    y += 1
  end for
end procedure

-----------------------------------------------------------------------------
-- Draw a message box in a window.                                         --
-- Parameters:                                                             --
--   atom:     left side of box (x coord of top left corner)               --
--   atom:     top of box (y coord of top left corner)                     --
--   sequence: text to display                                             --
-- Returns:                                                                --
--   atom: handle of the window that was created                           --
-----------------------------------------------------------------------------
global function DrawMessageBox(atom left, atom top, sequence text)
  atom hWin
  sequence lines
  integer longestLine
  integer x, y
  integer right, bottom
  sequence s

  s = splitText(text)
  longestLine = s[1]
  lines = s[2]

  -- Build the text box and write the text in it
  right = left + (longestLine * FONT_WIDTH) + 12
  bottom = top + (length(lines) * FONT_HEIGHT)  + 10
  hWin = LoadWindow(left, top, right, bottom)
  DrawTextBox(hWin)
  x = 6
  y = 6
  for i = 1 to length(lines) do
    WriteText(hWin, {x, y}, lines[i])
    y += FONT_HEIGHT
  end for
  return hWin
end function

-----------------------------------------------------------------------------
-- Create a new window that is a message box.  Display the window until    --
-- the user presses the right key to dismiss it.                           --
-- Parameters:                                                             --
--   atom:     left side of box (x coord of top left corner)               --
--   atom:     top of box (y coord of top left corner)                     --
--   sequence: text to display                                             --
--   atom:     key to wait for before closing the text box                 --
-----------------------------------------------------------------------------
global procedure MessageBox(atom left, atom top, sequence text)
  atom hWin
  integer k

  -- Create the message box
  hWin = DrawMessageBox(left,top,text)

  -- Display the message box
  FlipScreen()
  k = call_func(waitForKeyID,{})
  -- space bar or enter will close the text box
  while k != ' ' and k != #000D do
    k = call_func(waitForKeyID,{})
  end while

  -- Unload the message box so it won't display again
  UnloadWindow(hWin)
end procedure

-----------------------------------------------------------------------------
-- Draw a box on the screen containing a list and allow the user to select --
-- an item from the list.  Selection is chosen with the space bar.  The    --
-- user can leave without making a selection by pressing the ESC key.      --
-- Parameters:                                                             --
--   integer:  left side of box (x coord of top left corner)               --
--   integer:  top of box (y coord of top left corner)                     --
--   sequence: list of sequences containing names of items to select from  --
-----------------------------------------------------------------------------
global function SelectBox(integer left, integer top, sequence prompt,
                          sequence options)
  atom hWin
  sequence lines
  integer longestLine
  integer x, y
  integer right, bottom
  integer selMin, selMax
  integer sel
  integer k
  sequence s

  -- Break the text into lines, keeping track of the longest line
  lines = {}
  s = splitText(prompt)
  longestLine = s[1]
  lines = append(s[2],"")
  selMin = length(lines) + 1
  selMax = selMin - 1
  for i = 1 to length(options) do
    if length(options[i]) > longestLine then
      longestLine = length(options[i])
    end if
    lines = append(lines,options[i])
    selMax += 1
  end for

  sel = selMin
  -- Build the select box window
  right = left + (longestLine * FONT_WIDTH) + 12
  bottom = top + (length(lines) * FONT_HEIGHT)  + 10
  hWin = LoadWindow(left, top, right, bottom)
  while 1 = 1 do
    -- Re-draw the select box window
    DrawTextBox(hWin)
    x = 6
    y = 6
    for i = 1 to length(lines) do
      if i = sel then
        FontColor = RPG_GREEN
        WriteText(hWin, {x, y}, lines[i])
        FontColor = RPG_BLACK
      else
        WriteText(hWin, {x, y}, lines[i])
      end if
      y += FONT_HEIGHT
    end for
    FlipScreen()
    k = call_func(waitForKeyID,{})
    if k = ' ' or k = #000D then -- space bar or enter
      sel -= (selMin - 1)
      exit
    elsif k = #0148 then  -- up arrow
      if sel > selMin then
        sel -= 1
      end if
    elsif k = #0150 then  -- down arrow
      if sel < selMax then
        sel += 1
      end if
    elsif k = #001B then  -- ESC key
      sel = 0
      exit
    end if
  end while

  -- Unload the select box window so it won't appear again
  UnloadWindow(hWin)
  return sel
end function