\Textris.XPL	20-Apr-2009	Loren Blaney	loren.blaney@idcomm.com
\Tetris crossed with Scrabble.
\Original game by Thomas G. Hanlin.
\Compile with 32-bit XPL0. TASM runs out of memory so use MASM.
\
\This program is free software; you can redistribute it and/or modify it under
\ the terms of the GNU General Public License version 2 as published by the
\ Free Software Foundation.
\This program is distributed in the hope that it will be useful, but WITHOUT
\ ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
\ FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
\ details.
\You should have received a copy of the GNU General Public License along with
\ this program (in the file LICENSE.TXT); if not, write to the Free Software
\ Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

include c:\cxpl\codes;		\intrinsic routine declarations

def	FieldHeight = 9+1,	\play field dimensions, including borders
	FieldWidth = 9+2,	\ (character cells)
	FieldX = 45,		\upper-left coordinate of play field
	FieldY = 15,		\ (in pixels)
	FrameWidth = 15,	\wooden frame width (pixels)
	HistX = 61,		\upper-left corner of history window
	HistY = 7,		\ (character cells)
	HiScoreX=11, HiScoreY=9,\Hall-of-Fame window (character cells)
	LineMax = 28,		\maximum line number in history window
	ScreenWidth=640, ScreenHeight=480, \graphic screen dimensions (pixels)
	TileWidth=44, TileHeight=50; \letter tile image dimensions (pixels)

def	Blank = $20,		\blank tile
	ShortDelay = 500;	\delay for fast fall (microseconds)

def	Black = 0,		\color
	BRed = $C0,		\bright red
	Wood = $6A;		\wood color

int	CursorX, CursorY,	\screen cursor position in history window
	DoSound,		\flag: toggles sound on and off
	FastFall,		\flag: drop tile quickly
	Handle,			\output file handle for GAME.TXT
	HiPoints,		\point value of highest scoring word
	Letter,			\falling letter tile (A..Z or blank)
	LetterX, LetterY,	\falling letter's pixel coords in play field
	Score,			\total accumulated points
	SpeedDelay,		\speed of fall (microseconds per pixel)
	WordCount;		\word counter for GAME.TXT

char	HiWord(7);		\highest scoring word

int	Field(FieldWidth, FieldHeight);	\play field, including borders

def	DictSize = 55000,	\maximum number of words allowed in dictionary
	HashBoxSize = 256,	\number of hash boxes (must be power of 2)
	HashMask = HashBoxSize-1,
	Nil = -1;		\indicator for unused index in HashBox

int	DictInx,		\index to next entry in Dictionary
	HashLink(DictSize),	\list of hash linkage pointers
	HashBox(HashBoxSize),	\contains linked list headers
	Dictionary(DictSize, 2);\table of words (2 = MaxWordLen/IntSize +1)

def	FontHeight = 14;	\height of font character cell (pixels, bytes)

char	Font,			\256-character font table
	TileImage,		\320x240 array of tile images, etc.
	Palette;		\256 RGB values (6-bit)

def	Bel=$07, BS=$08, Tab=$09, LF=$0A, FF=$0C,	\control chars
	CR=$0D, Esc=$1B, Sp=$20, Ctrl=$40;
def	UpArrow=$48, DnArrow=$50, LtArrow=$4B, RtArrow=$4D,	\scan codes
	Home=$47, End=$4F, Delete=$53, Func1=$3B;



proc	SetFont;
\The font is included here, near the beginning of the code, because BIOS needs 
\ it in the first 64k of our data segment.
int	CpuReg,		\address of CPU register array (from GetReg)
	DataSeg;	\segment address of our data (for BIOS calls)
begin
include Font;		\(must reside within first 64K of data segment)

CpuReg:= GetReg;
DataSeg:= CpuReg(12);

\set interrupt vector $43 to point to our font table
Poke(0, $43*4, Font);
Poke(0, $43*4+1, Font>>8);
Poke(0, $43*4+2, DataSeg);
Poke(0, $43*4+3, DataSeg>>8);
Poke($40, $85, FontHeight);
Poke($40, $85+1, FontHeight>>8);
end;	\SetFont



proc	Fatal(Msg);		\Fatal error handler
char	Msg;
begin
SetVid(3);			\standard text mode (clears screen too)
Text(0, Msg);
CrLf(0);
ChOut(0, Bel);
exit;
end;	\Fatal



func	GetTimeX;		\because WinXP is so incredibly $#@&ED UP!!!
return GetTime & $0000FFFF;



proc    Delay(D);		\Delay D microseconds
int     D;
int     T;
begin
T:= GetTimeX;
repeat until ((GetTimeX-T) & $0000FFFF) >= D;
end;	\Delay



func	GetKey;			\Get a keystroke
int	K;
begin
K:= ChIn(1);			\device 0 won't input Esc
if K = 0 then K:= -ChIn(1);	\return scan codes as negative values
if K = Esc then			\Esc terminates program
	[SetVid(3);   exit];	\restore standard text mode and bail out
return K;
end;	\GetKey



func	ToUpper(Ch);		\Convert letter to uppercase
int	Ch;
return if Ch>=^a & Ch<=^z then Ch&$DF else Ch;



func	IntLen(N);		\Return number of digits in N, a decimal integer
int	N;			\Adds 1 for a minus sign if N is negative
int	I;
for I:= if N>=0 then 1 else 2, 20 do
	[N:= N/10;  if N = 0 then return I];



proc	JustOut(Dev, N, Places, Fill); \Output a right-justified integer
\Always outputs correct value of N regardless of Places
int	Dev,	\output device number
	N,	\integer (negative numbers should be filled with spaces)
	Places,	\size of field in characters (right-justifies)
	Fill;	\character to fill background of field. Usually space (Sp) or ^0
int	I;	\ If not a space then beware of negative numbers.
begin
for I:= 1, Places-IntLen(N) do ChOut(Dev, Fill);
IntOut(Dev, N);
end;	\JustOut



func	StrLen(Str);		\Returns the number of characters in a string
char	Str;
int	I;
for I:= 0, 32000 do
	if Str(I) >= $80 then return I+1;



proc	StrCopy(S1, S2);	\Copy string S1 into S2
char	S1, S2;
int	I;
begin
for I:= 0, 32000 do
	begin
	S2(I):= S1(I);
	if S1(I) > $7F then return;
	end;
end;	\StrCopy



func	StrEqual(S1, S2);	\Compare strings, return 'true' if they're equal
char	S1, S2;
int	I;
begin
for I:= 0, 32000 do
	begin
	if S1(I) # S2(I) then return false;
	if S1(I) > $7F then return true;
	end;
end;	\StrEqual



proc	TextN(Dev, Str, N);	\Output a string N bytes long
int	Dev;	\output device
char	Str;
int	N;
int	I;
begin
for I:= 0, N-1 do
	ChOut(Dev, Str(I));
end;	\TextN



proc	StrNFill(Str, N, Pat);	\Fill string with pattern
char	Str;	\string
int	N,	\number of bytes to fill
	Pat;	\pattern
int	I;
begin
for I:= 0, N-1 do Str(I):= Pat;
end;	\StrNFill



proc	GetFld(Field, N, X, Y);	\Input a field (string) of chars
char	Field;	\address of field
int	N,	\number of characters in field
	X, Y;	\field position on screen
int	Ch,
	M,	\maximum index of Field
	I, J;
begin
M:= N-1;
\Fill field with underlines. Unused spaces in the field are converted to
\ underlines, which are converged from both ends.
I:= 0;   while Field(I)=SP & I<=M do [Field(I):= ^_;   I:= I+1];
I:= M;   while Field(I)=SP & I>=0 do [Field(I):= ^_;   I:= I-1];

J:= 0;				\our position in the field
loop	begin
	Cursor(X, Y);		\display what we have
	TextN(6, Field, N);

	Attrib(BRed<<8+Black);	\use inverse video to make a cursor
	Cursor(X+J, Y);
	ChOut(6, Field(J));
	Ch:= GetKey;		\wait for keystroke
	Attrib(Black<<8+BRed);	\turn cursor off
	Cursor(X+J, Y);
	ChOut(6, Field(J));

	case Ch of
	  -LtArrow,-UpArrow:if J > 0 then J:= J-1;
	  -RtArrow,-DnArrow:if J < M then J:= J+1;
	  -Home:J:= 0;
	  -End:	J:= M;
	  FF:	[for I:= 0, M do Field(I):= ^_;	\clear the entire field
		J:= 0];				\our position in the field
	  -Delete:
		begin
		for I:= J, M-1 do		\shift field left
			Field(I):= Field(I+1);
		Field(M):= ^_;
		end;
	  BS:	begin
		if J > 0 then
			[J:= J-1;
			for I:= J, M-1 do	\shift field left
				Field(I):= Field(I+1);
			Field(M):= ^_];
		end;
	  CR:	quit;
	  Esc:	quit
	other	begin
		if Ch >= $20 then		\ignore control chars
			begin
			for I:= -(M-1), -J do	\shift rest of line right
				Field(-I+1):= Field(-I);
			Field(J):= Ch;
			if J < M then J:= J+1;	\move cursor right
			end;
		end;
	end;

J:= 0;				\put spaces back in place of underlines
while J<=M & Field(J)=^_ do [Field(J):= SP;	J:= J+1];
J:= M;
while J>=0 & Field(J)=^_ do [Field(J):= SP;	J:= J-1];

Cursor(X, Y);			\re-display field without underlines
TextN(6, Field, N);
end;	\GetFld

\-------------------------------------------------------------------------------

proc	DrawRectangle(X, Y, W, H, C, F);	\Draw a filled or unfilled box
int	X, Y,	\coordinates of upper-left corner (pixels)
	W, H,	\width and height
	C, F;	\color and fill flag
int	J;
begin
if F then					\draw a filled (solid) box
	begin
	for J:= Y, Y+H-1 do
		[Move(X, J);   Line(X+W-1, J, C)];
	end
else	begin				       \draw a non-filled (outlined) box
	Move(X, Y);
	Line(X+W-1, Y, C);
	Line(X+W-1, Y+H-1, C);
	Line(X, Y+H-1, C);
	Line(X, Y, C);
	end;
end;	\DrawRectangle



proc	DrawSprite(X0, Y0, Spr);	\Draw a sprite
int	X0, Y0;		\coordinates of upper-left corner (pixels)
char	Spr;		\address of sprite data
int	X, Y, I, C, W, H;
begin
W:= Spr(0);		\get width and height (in pixels)
H:= Spr(1);
I:= 2;
for Y:= Y0, Y0+H-1 do
    for X:= X0, X0+W-1 do
	begin
	C:= Spr(I);		\get pixel's color
	if C then		\background (0) is transparent
		Point(X, Y, C);
	I:= I+1;
	end;
end;	\DrawSprite



proc	DrawFrame;		\Draw wooden frame around play field
int	X0, Y0;			\coordinates in TileImage data array
int	I, X, Y, W, H;
char	Spr(2+FrameWidth*240);
begin
\make a sprite from image data
W:= TileWidth*9/2 + FrameWidth;
H:= FrameWidth;
Spr(0):= W;
Spr(1):= H;

X0:= 0;			\draw left and right pieces of top border
Y0:= 200;		\coordinates in TileImage data
I:= 2;			\initialize index past dimensions in sprite array
for Y:= Y0, Y0+H-1 do
    for X:= X0, X0+W-1 do
	begin
	Spr(I):= TileImage(X + Y*320);
	I:= I+1;
	end;
DrawSprite(FieldX-FrameWidth, 0, Spr);

I:= 2;
for Y:= Y0, Y0+H-1 do
    for X:= -(X0+W-1), -X0 do
	begin
	Spr(I):= TileImage(-X + Y*320);
	I:= I+1;
	end;
DrawSprite(FieldX-FrameWidth+W, 0, Spr);

X0:= 320-W;		\draw right and left pieces of bottom border
I:= 2;
for Y:= Y0, Y0+H-1 do
    for X:= X0, X0+W-1 do
	begin
	Spr(I):= TileImage(X + Y*320);
	I:= I+1;
	end;
DrawSprite(FieldX-FrameWidth+W, FieldY+TileHeight*9, Spr);

I:= 2;
for Y:= Y0, Y0+H-1 do
    for X:= -(X0+W-1), -X0 do
	begin
	Spr(I):= TileImage(-X + Y*320);
	I:= I+1;
	end;
\DrawSprite(FieldX-FrameWidth, ScreenHeight-FrameWidth, Spr);
DrawSprite(FieldX-FrameWidth, FieldY+TileHeight*9, Spr);

\width and height of border image data
W:= TileHeight*9/2 + FrameWidth;
H:= FrameWidth;
Spr(0):= H;		\sprite's width
Spr(1):= W;		\sprite's height

X0:= 0;			\draw bottom and top pieces of left border
I:= 2;
for Y:= -(X0+W-1), -X0 do
    for X:= Y0, Y0+H-1 do
	begin
	Spr(I):= TileImage(X*320 - Y);
	I:= I+1;
	end;
DrawSprite(FieldX-FrameWidth, 240, Spr);

I:= 2;
for Y:= X0, X0+W-1 do
    for X:= Y0, Y0+H-1 do
	begin
	Spr(I):= TileImage(X*320 + Y);
	I:= I+1;
	end;
DrawSprite(FieldX-FrameWidth, 0, Spr);

X0:= 320-W;		\draw top and bottom pieces of right border
I:= 2;
for Y:= -(X0+W-1), -X0 do
    for X:= Y0, Y0+H-1 do
	begin
	Spr(I):= TileImage(X*320 - Y);
	I:= I+1;
	end;
DrawSprite(FieldX-FrameWidth+FrameWidth+TileWidth*9, 0, Spr);

I:= 2;
for Y:= X0, X0+W-1 do
    for X:= Y0, Y0+H-1 do
	begin
	Spr(I):= TileImage(X*320 + Y);
	I:= I+1;
	end;
DrawSprite(FieldX-FrameWidth+FrameWidth+TileWidth*9, 240, Spr);
end;	\DrawFrame



proc	DrawTileDigit(XX, YY, Digit);
int	XX, YY, Digit;	\where to draw tile on screen; (non ASCII) digit
int	X0, Y0;		\coordinates in TileImage data
int	I, X, Y;
char	Spr(2+18*19);
begin
\make a sprite from image data
Spr(0):= 18;	\width
Spr(1):= 19;	\height
X0:= Digit*18;
Y0:= 215;
I:= 2;			\initialize index past dimensions in sprite array
for Y:= Y0, Y0+19-1 do
    for X:= X0, X0+18-1 do
	begin
	Spr(I):= TileImage(X + Y*320);
	I:= I+1;
	end;
DrawSprite(XX, YY, Spr);
end;	\DrawTileDigit



proc	DrawTile(XX, YY, Let);
int	XX, YY, Let;	\coordinates where to draw tile on screen; letter
int	X0, Y0;		\coordinates in TileImage data
int	I, X, Y, C;
char	Spr(TileWidth*TileHeight);
begin
if Let = 0 then		\background (empty)
	begin
	X0:= 6*TileWidth;
	Y0:= 3*TileHeight
	end
else if Let = Blank then
	begin
	X0:= 5*TileWidth;
	Y0:= 3*TileHeight
	end
else if Let>=^A & Let<=^Z then
	begin
	Y0:= (Let-^A)/7 * TileHeight;
	X0:= rem(0) * TileWidth;
	end
else Fatal("Invalid tile letter");

I:= 0;			\initialize index past dimensions in sprite array
for Y:= Y0, Y0+TileHeight-1 do
    for X:= X0, X0+TileWidth-1 do
	begin
	Spr(I):= TileImage(X + Y*320);
	I:= I+1;
	end;
I:= 0;
for Y:= YY, YY+TileHeight-1 do
    for X:= XX, XX+TileWidth-1 do
	begin
	C:= Spr(I);			\get pixel's color
	if C & Y>=FieldY then		\background (0) is transparent
		Point(X, Y, C);		\clip to top edge of field
	I:= I+1;
	end;
end;	\DrawTile



proc	DrawField;		\Draw the play field
int	X, Y;			\field square coordinates (including borders)
begin
for Y:= 0, FieldHeight-2 do
    for X:= 1, FieldWidth-2 do
	DrawTile(FieldX + (X-1)*TileWidth,  FieldY + Y*TileHeight,  Field(X,Y));
end;	\DrawField



proc	DrawGradients;		\Draw left and right red gradient backgrounds
int	W, C, X, Y;
begin
W:= FieldX - FrameWidth;
C:= BRed - 3*W/2;
for X:= 0, W-1 do
	begin
	Move(X, 0);  Line(X, ScreenHeight-1, if X&3 then C else C-8);
	C:= C+2;
	end;
C:= BRed;
for X:= FieldX+9*TileWidth+FrameWidth, ScreenWidth-1 do
	begin
	Move(X, 0);  Line(X, ScreenHeight-1, C);
	Move(X+1, 0);  Line(X+1, ScreenHeight-1, C);
	Move(X+2, 0);  Line(X+2, ScreenHeight-1, C);
	Move(X+3, 0);  Line(X+3, ScreenHeight-1, C-8);
	C:= C-1;
	X:= X+3;
	end;
end;	\DrawGradients



proc	LoadPalette;
int	I;
begin
port($3C8):= 0;				\starting color register 
for I:= 0, 256*3-1 do
	port($3C9):= Palette(I);
end;	\LoadPalette

\-------------------------------------------------------------------------------

proc	Explode(X, Y, W, H, Steps);	\Rapidly grow a window
int	X, Y, W, H, Steps;
int	SW, SH, SX, SY, S;
begin
for S:= 1, Steps do
	begin
	SW:= W * S/Steps;
	SH:= H * S/Steps;
	SX:= X + (W-SW)/2;
	SY:= Y + (H-SH)/2;
	Delay(16666);
	DrawRectangle(SX, SY, SW, SH, Black, true);
	DrawRectangle(SX+1, SY+1, SW-2, SH-2, Wood, false);
	end;
end;	\Explode



proc	HighScores;		\Show high-score window and maybe add this score
\Inputs Score
def	ListSize = 10,	\number of items in list
	NameSize = 16;	\number of characters provided for a name
int	ListScore(ListSize),	\player's score
	ListName(ListSize, NameSize),	\player's name
	ListWord(ListSize, 7),	\word with highest point value
	ListPoints(ListSize);	\number of points
int	H, I, J, K;
char	Str, Str2;
begin
\show the list with new score (if any) in its place
Explode(HiScoreX*8-1, HiScoreY*FontHeight-1, 39*8+2, (ListSize+3)*FontHeight+2, 8);

Trap(false);		\read list from disk
H:= FOpen("TEXTRIS.DAT", 0);
FSet(H, ^I);
OpenI(3);
if GetErr then
	begin		\if there's no list on the disk then start one
	for I:= 0, ListSize-1 do
		begin
		ListScore(I):= 0;
		StrNFill(ListName(I), NameSize, ^_);
		ListPoints(I):= 0;
		StrCopy("       ", ListWord(I));
		end;
	end
else	begin		\read in list from disk
	Trap(true);
	for I:= 0, ListSize-1 do
		begin
		ListScore(I):= IntIn(3);
		Str:= ListName(I);		\access bytes, not integers
		for J:= 0, NameSize-1 do
			Str(J):= ChIn(3);
		ListPoints(I):= IntIn(3);
		Str:= ListWord(I);
		for J:= 0, 7-1 do
			Str(J):= ChIn(3);
		end;
	FClose(H);	\close file handle that was successfully opened
	end;
Trap(true);

\scan down for first entry in list that's less than or equal to Score
K:= -1;
for I:= 0, ListSize-1 do
    if ListScore(I) <= Score then	\if on list then insert new score
	begin
	Str:= ListName(ListSize-1);	\save discarded pointers for new entry
	Str2:= ListWord(ListSize-1);
	for J:= -(ListSize-2), -I do	\shift existing entries down one
		begin
		ListScore(1-J):= ListScore(-J);
		ListName(1-J):= ListName(-J);
		ListPoints(1-J):= ListPoints(-J);
		ListWord(1-J):= ListWord(-J);
		end;
	ListScore(I):= Score;
	ListName(I):= Str;
	StrNFill(Str, NameSize, ^_);	\blank the line
	ListPoints(I):= HiPoints;
	ListWord(I):= Str2;
	for J:= 0, 7-1 do Str2(J):= HiWord(J);
	K:= I;				\save index to new entry
	I:= ListSize;			\exit 'for' loop
	end;

\ 012345678901234567890123456789012345678
\|  12345  xxxxxxxxxxxxxxxx 1234 xxxxxxx |

Cursor(HiScoreX, HiScoreY);		\show title of window
Attrib(Wood<<8+Black);
Text(6, " Hall of Fame              High Word   ");
for J:= 0, ListSize-1 do
	begin				\show the list
	Attrib(if J = K then BRed else Wood);
	Cursor(HiScoreX+2, HiScoreY+2+J);
	JustOut(6, ListScore(J), 5, ^0);
	Cursor(HiScoreX+9, HiScoreY+2+J);
	TextN(6, ListName(J), NameSize);
	Cursor(HiScoreX+26, HiScoreY+2+J);
	JustOut(6, ListPoints(J), 4, ^ );
	Cursor(HiScoreX+31, HiScoreY+2+J);
	Text(6, ListWord(J));
	end;

\allow player's name to be entered if Score was added to the list
if K >= 0 then
	begin
	Attrib(BRed);
	GetFld(Str, NameSize, HiScoreX+9, HiScoreY+2+K);
	for J:= 0, NameSize-1 do	\write entry to GAME.TXT
		ChOut(3, Str(J));
	Crlf(3);

	H:= FOpen("TEXTRIS.DAT", 1);	\write list to disk
	FSet(H, ^O);
	OpenO(3);
	for I:= 0, ListSize-1 do
		begin
		IntOut(3, ListScore(I));
		ChOut(3, Sp);
		Str:= ListName(I);
		for J:= 0, NameSize-1 do
			ChOut(3, Str(J));
		ChOut(3, Sp);
		IntOut(3, ListPoints(I));
		ChOut(3, Sp);
		Str:= ListWord(I);
		for J:= 0, 7-1 do
			ChOut(3, Str(J));
		Crlf(3);
		end;
	Close(3);
	FClose(H);
	FSet(Handle, ^o);		\reassign GAME.TXT handle to device 3
	end;
end;	\HighScores

\-------------------------------------------------------------------------------

func	HashCode(Str);		\Return string's hash code
char	Str;
int	I, H;
begin
H:= 0;
for I:= 0, 32000 do
	begin
	H:= H + Str(I);
	if Str(I) > $7F then return H & HashMask;
	end;
end;	\HashCode



func	LookUp(Word);		\Look up a word and return 'true' if it's found
char	Word;
int	I;
begin
I:= HashBox(HashCode(Word));	\follow the chain of list linkages until word is
loop	begin			\ found or end of list
	if I = Nil then return false;
	if StrEqual(Word, Dictionary(I)) then return true;
	I:= HashLink(I);
	end;
end;	\LookUp



proc	Insert(Word);		\Insert a word into the Dictionary
char	Word;
int	H;
begin
if DictInx >= DictSize then Fatal("Too many words");
StrCopy(Word, Dictionary(DictInx));
H:= HashCode(Word);
HashLink(DictInx):= HashBox(H);
HashBox(H):= DictInx;
DictInx:= DictInx+1;
end;	\Insert

\-------------------------------------------------------------------------------

proc	ShowScore;		\Display the total score using sprite digits
int	I, X, Y;

	proc	NumOut(N);	\Display a positive integer using tiles
	int	N;
	int	Q, R;
	begin
	Q:= N/10;
	R:= rem(0);
	if Q \>0\ then NumOut(Q);	\recurse
	DrawTileDigit(X, Y, R);
	X:= X + 19;
	end;	\NumOut

begin	\ShowScore
X:= HistX*8;
Y:= 27*16;
for I:= 1, 5-IntLen(Score) do
	begin
	DrawTileDigit(X, Y, 0);
	X:= X + 19;
	end;
NumOut(Score);
end;	\ShowScore



func	WordPoints(Word);	\Return the number of points for a word
char	Word;			\(assumes letters A..Z)
int	Tbl, Points, I;
begin	\Scrabble tile values
\	 A B C D E F G H I J K L M N O P Q  R S T U V W X Y Z
Tbl:=	[1,3,3,2,1,4,2,4,1,8,5,1,3,1,1,3,10,1,1,1,1,4,4,8,4,10];

Points:= 0;
for I:= 0, StrLen(Word)-1 do
	begin
	Letter:= Word(I) & $7F;
	Points:= Points + Tbl(Letter-^A);
	end;
for I:= 4, StrLen(Word) do	\letters beyond 3 are all on triple word scores
	Points:= Points * 3;

if Points >= HiPoints then	\record highest scoring word
	begin
	HiPoints:= Points;
	for I:= 0, 7-1 do HiWord(I):= Word(I);
	end;

return Points;
end;	\WordPoints



proc	ScrollWind;		\Scroll up the characters in the history window
int	X, Y;
begin
for Y:= HistY*FontHeight, LineMax*FontHeight-1 do
    for X:= HistX*8, (HistX+7+1+4)*8-1 do	\word (7), space (1), points (4)
	Point(X, Y, ReadPix(X, Y+FontHeight));
for Y:= LineMax*FontHeight, (LineMax+1)*FontHeight-1 do
    for X:= HistX*8, (HistX+7+1+4)*8-1 do	\blank bottom line of text
	Point(X, Y, 0);
end;	\ScrollWind



proc	Record(Word);		\Show a record of the Word in the history window
char	Word;
begin
Cursor(CursorX, CursorY);
Attrib(Wood<<8);
Text(6, Word);
Cursor(CursorX+8, CursorY);
Attrib(Wood);
JustOut(6, WordPoints(Word), 4, ^ );
CursorY:= CursorY+1;
if CursorY > LineMax then
	[CursorY:= LineMax;  ScrollWind];

WordCount:= WordCount+1;
JustOut(3, WordCount, 5, ^ );
Text(3, ".  ");
Text(3, Word);				\write word to GAME.TXT
ChOut(3, Tab);
JustOut(3, WordPoints(Word), 4, ^ );	\points
ChOut(3, Tab);
JustOut(3, Score, 4, ^ );		\total score so far
CrLf(3);
end;	\Record



proc	Message(Str);		\Show a message in the history window
char	Str;			\Warning: last character not shown
int	I, Ch;
begin
I:= 0;
loop	begin
	Ch:= Str(I);
	if Ch >= $80 then quit;
	I:= I+1;
	if Ch = CR then CursorX:= HistX
	else if Ch = LF then
		begin
		CursorY:= CursorY+1;
		if CursorY > LineMax then
			[CursorY:= LineMax;  ScrollWind];
		end
	else	begin
		Cursor(CursorX, CursorY);
		ChOut(6, Ch);
		CursorX:= CursorX+1;
		end;
	end;
end;	\Message

\-------------------------------------------------------------------------------

proc	DrawBkgndPoint(X, Y);	\Draw a tile's background pixel
int	X, Y;			\coordinates relative to FieldX, and FieldY
int	RX, RY, C;
begin
RX:= rem(X/TileWidth);
RY:= rem(Y/TileHeight);
C:= $0A;			\background color
if RX=0 ! RX=TileWidth-1 ! RY=0 ! RY=TileHeight-1 then
	C:= $0B;		\color of background lines
if Y >= 0 then Point(X+FieldX, Y+FieldY, C);	\clip to top of field
end;	\DrawBkgndPoint



proc	DrawBkgndEdge(X, Y);	\Draw background along top edge for falling tile
int	X, Y;			\coordinates relative to FieldX, and FieldY
int	I;
begin
DrawBkgndPoint(X, Y+2);
DrawBkgndPoint(X+1, Y+1);
for I:= X+2, X+TileWidth-2-1 do DrawBkgndPoint(I, Y);
DrawBkgndPoint(X+TileWidth-2, Y+1);
DrawBkgndPoint(X+TileWidth-1, Y+2);
end;	\DrawBkgndEdge



proc	DropTiles;		\Drop tiles that have nothing under them
int	Fell, I, J, X, Y, YY, T;
int	Falling(FieldWidth, FieldHeight);
loop begin
for J:= 0, FieldHeight-1 do
    for I:= 0, FieldWidth-1 do
	Falling(I,J):= 0;

Fell:= false;
for J:= -(FieldHeight-3), 0 do			\scan from the bottom up
    for I:= 1, FieldWidth-2 do			\ skipping borders
	begin
	T:= Field(I,-J);
	if T#0 & Field(I,1-J)=0 then
		begin				\have tile with nothing under it
		Fell:= true;			\indicate that something fell
		Field(I,1-J):= T;		\move tile down
		Field(I,-J):= 0;
		Falling(I,-J):= T;		\mark falling tile's origin
		end;
	end;

if Fell then
    begin	\move tile down a pixel at a time for each tile that is falling
    for YY:= 0, TileHeight-1 do
	begin
	for J:= -(FieldHeight-3), 0 do		\scan from the bottom up
	    for I:= 1, FieldWidth-2 do		\ skipping borders
		begin
		T:= Falling(I,-J);
		if T # 0 then
			begin			\show falling tile
			X:= (I-1)*TileWidth;
			Y:= YY - J*TileHeight;
			DrawBkgndEdge(X, Y);
			DrawTile(FieldX+X, FieldY+Y+1, T);
			end;
		end;

	if KeyHit then GetKey;			\Esc aborts immediately
	Delay(if FastFall then ShortDelay else SpeedDelay);
	end;	\for
    end
else quit;
end;	\DropTiles



func	CrunchCol(Len);		\If a word was made then crunch the letters down
int	Len;			\length of word (characters)
int	X, Y, I, J, K, Ch;
char	Word(8);
begin
for Y:= 0, FieldHeight-1-Len do		\for top row down to bottom row-(Len-1)
    begin
    for X:= 1, FieldWidth-2 do		\scan vertically for Len-letter words
	begin				\ skipping borders
	for I:= 0, Len-1 do
		begin
		Ch:= Field(X,Y+I);
		if Ch<^A ! Ch>^Z then I:= 10 	\exit for loop if empty
		else Word(I):= Ch;
		end;
	if I = Len then
		begin				\got Len many letters
		Word(Len-1):= Word(Len-1) ! $80;\terminate string
		if LookUp(Word) then		\see if it's a word
			begin			\found it!
			for K:= 0, 7-1 do	\flash it
			    begin
			    for J:= 0, Len-1 do
				DrawTile(FieldX+(X-1)*TileWidth,
					 FieldY+(Y+J)*TileHeight,
					 if K&1 then Word(J)&$7F else 0);
			    Sound(DoSound, 1, (7-K)*1000);
			    end;
			Score:= Score + WordPoints(Word);
			ShowScore;
			Record(Word);
			for J:= Y, Y+Len-1 do Field(X,J):= 0;	\erase word
			DropTiles;
			return true;
			end;
		end;
	end;
    end;
return false;
end;	\CrunchCol



func	CrunchRow(Len);		\If a word was made then crunch the letters down
int	Len;			\length of word (characters)
int	X, Y, I, J, K, Ch;
char	Word(8);
begin
for Y:= 0, FieldHeight-2 do		\from the top row down to the bottom row
    begin
    for X:= 1, FieldWidth-1-Len do	\scan for Len-letter words
	begin				\ skipping borders
	for I:= 0, Len-1 do
		begin
		Ch:= Field(X+I,Y);
		if Ch<^A ! Ch>^Z then I:= 10 	\exit for loop if empty
		else Word(I):= Ch;
		end;
	if I = Len then
		begin				\got Len letters
		Word(Len-1):= Word(Len-1) ! $80; \terminate string
		if LookUp(Word) then		\see if it's a word
			begin			\found it!
			for K:= 0, 7-1 do	\flash it
			    begin
			    for I:= 0, Len-1 do
				DrawTile(FieldX+(X-1+I)*TileWidth,
					 FieldY+Y*TileHeight,
					 if K&1 then Word(I)&$7F else 0);
			    Sound(DoSound, 1, (7-K)*1000);
			    end;
			Score:= Score + WordPoints(Word);
			ShowScore;
			Record(Word);
			for I:= X, X+Len-1 do Field(I,Y):= 0;	\erase word
			DropTiles;
			return true;
			end;
		end;
	end;
    end;
return false;
end;	\CrunchRow



proc	DropLetter;		\Drop letters and handling keyboard commands
int	StartTime,
	Blocked,		\flag: letter cannot move in current direction
	NewX, NewY,		\tenative new position (pixels)
	Ch;


	proc	DoMove;		\If tenative move is legal then actually do it
	int	X, Y;
	begin
	Blocked:=
		Field(NewX/TileWidth+1, (NewY+TileHeight-1)/TileHeight) # 0;
	if not Blocked then
		begin
		\restore background pixels at old location
		if NewX = LetterX then	\just moved down a pixel
			DrawBkgndEdge(LetterX, LetterY)
		else	begin		\moved to another column
			for Y:= LetterY, LetterY+TileHeight-1 do
			    for X:= LetterX, LetterX+TileWidth-1 do
				DrawBkgndPoint(X, Y);
			end;
		LetterX:= NewX;  LetterY:= NewY;	\make move
		DrawTile(FieldX+LetterX, FieldY+LetterY, Letter);
		end;
	end;	\DoMove


begin	\DropLetter
DrawTile(FieldX+LetterX, FieldY+LetterY, Letter);
FastFall:= false;
StartTime:= GetTimeX;

repeat	if KeyHit then		\loop to delay for moving down one pixel
		begin
		NewY:= LetterY;
		Ch:= ToUpper(GetKey);
		case Ch of
		  -LtArrow:
			[NewX:= LetterX-TileWidth;  DoMove];
		  -RtArrow:
			[NewX:= LetterX+TileWidth;  DoMove];
		  -DnArrow, Sp:
			FastFall:= true;
		  -Func1:
			DoSound:= not DoSound
		other	if Letter=Blank & Ch>=^A & Ch<=^Z then
				begin
				Letter:= Ch;	\show it immediately
				DrawTile(FieldX+LetterX, FieldY+LetterY, Letter);
				end;
		end;

	\delay for balance of time interval
	repeat until ((GetTimeX-StartTime) & $0000FFFF) >=
			(if FastFall then ShortDelay else SpeedDelay);
	StartTime:= GetTimeX;

	\Attempt to move letter down one pixel
	NewX:= LetterX;   NewY:= LetterY+1;
	DoMove;
until Blocked;

Sound(false, 1, 1);	\sync to system clock so click always sounds the same
Sound(DoSound, 1, 4100);
end;	\DropLetter



proc	PlayGame;		\Play until letters overflow top of play field
char	TileSet;
int	X, Y, W, H, Len, PreLetter;
begin
TileSet:=
"  AAAAAAAAABBCCDDDDEEEEEEEEEEEEFFGGGHHIIIIIIIIIJKLLLLMMNNNNNNOOOOOOOOPPQRRRRRRSSSSTTTTTTUUUUVVWWXYYZ ";

Score:= 0;
ShowScore;
WordCount:= 0;
HiPoints:= 0;
StrCopy("       ", HiWord);

for Y:= 0, FieldHeight-1 do		\set up empty field with borders
    for X:= 0, FieldWidth-1 do
	Field(X, Y):=
	  if X=0 ! X=FieldWidth-1 ! Y=FieldHeight-1 then ^# \border\ else 0;
DrawField;

\set up history window
X:= HistX;
Y:= HistY;
W:= 7+1+4;		\word (7), space (1), points (4)
H:= LineMax-Y+1;
DrawRectangle(X*8-3, Y*FontHeight-3, W*8+6, H*FontHeight+6, Black, true);
DrawRectangle(X*8-2, Y*FontHeight-2, W*8+4, H*FontHeight+4, Wood, false);
CursorX:= X;   CursorY:= Y;

PreLetter:= TileSet(Ran(100));		\randomly select letter for preview
loop	begin
	Letter:= PreLetter;		\drop the preview letter
	PreLetter:= TileSet(Ran(100));	\show next (preview) letter
	W:= TileWidth+6;		\width with frame
	H:= TileHeight+6;
	X:= HistX*8 + (7+1+4)*8/2 - W/2;\word (7), space (1), points (4)
	Y:= FieldY-3;
	DrawRectangle(X, Y, W, H, 0, true);
	DrawRectangle(X, Y, W, H, Wood, false);
	DrawTile(X+3, Y+3, PreLetter);

	LetterX:= 4*TileWidth;		\start letter in center column above top
	LetterY:= -TileHeight;
	if Field(LetterX/TileWidth+1, (LetterY+TileHeight-1)/TileHeight) # 0 then quit; \lost

	SpeedDelay:= 25000;
	if Score > 1000 then SpeedDelay:= 25_000_000/Score;

	OpenI(1);			\dispose of any pending keystroke
	DropLetter;

	\if tile is still blank then select a letter at random
	if Letter = Blank then
		begin
		Letter:= TileSet(Ran(98)+2);	\skip the blank tiles
		DrawTile(FieldX+LetterX, FieldY+LetterY, Letter);
		end;

	Field(LetterX/TileWidth+1, LetterY/TileHeight):= Letter;

	\scan play field for any 7-letter words down to any 3-letter words
	Len:= 7;
	repeat	if CrunchRow(Len) then Len:= 7
		else if CrunchCol(Len) then Len:= 7
		else Len:= Len-1;
	until Len < 3;
	end;
end;	\PlayGame

\-------------------------------------------------------------------------------

proc	InitDict;		\Set up word list (dictionary)
int	I, Len, Ch, Num, Inx;
char	Word(8);
char	LastWord(8);
char	Dict;
begin
include	Dict;			\compressed dictionary string

DictInx:= 0;
for I:= 0, HashBoxSize-1 do HashBox(I):= Nil;

Inx:= 0;
loop	begin
	Ch:= Dict(Inx);
	Inx:= Inx+1;
	if Ch >= $80 then return;
	if Ch<^0 ! Ch>^9 then Fatal("Number missing in Dict.xpl");
	Num:= Ch & $0F;
	for I:= 0, Num-1 do Word(I):= LastWord(I);
	Len:= I;
	loop	begin		\read word chars
		Ch:= Dict(Inx);
		Inx:= Inx+1;
		if Ch<^A ! Ch>^Z then quit;
		Word(Len):= Ch;
		Len:= Len+1;
		end;
	Inx:= Inx-1;		\backup

	for I:= 0, Len-1 do LastWord(I):= Word(I);

	if I > 0 then
		begin
		Word(I-1):= Word(I-1) ! $80;
		Insert(Word);
		end
	else	Fatal("Zero length word in Dict.xpl");
	end;
end;	\InitDict



begin	\Main
include Tiles;
SetVid($101);			\640x480x8 (256 colors)
SetFont;
LoadPalette;
DrawGradients;
DrawFrame;

InitDict;

Handle:= FOpen("GAME.TXT", 1);	\save a list of the words in an output file
FSet(Handle, ^o);		\small output buffer closes upon exit
OpenO(3);

TrapC(true);			\only allow a clean exit (that resets graphics)
DoSound:= true;
loop	begin
	PlayGame;
	HighScores;

	Attrib(Wood);
	Message("
 Hit Enter
    for
another game
    ...
 or hit Esc
  to exit.
");
	CrLf(3);
	repeat until GetKey = CR;
	end;
end;	\Main
