program Unstoppable;

(* Written by Darren Grey in October 2010.
   Created as part of the October 2010 4DRL competition.
   Some initial code ripped from my previous 7DRL Trapper.
   
   Released without licence - use as you wish, though I would like to be
   informed if you can make anything useful out of it!
 *)

uses
   crt;  (* Basic display library.  Porting to curses may be faster on some
            systems. *)

const (* Global *)
   xmin = 1;
   xmax = 66;
   ymin = 3;
   ymax = 24; (* Map limits.  With sidebar comes to 80x24 - standard Unix
                 terminal size. *)
   DfTxC = 2;
   DfBgC = 0; (* Default text and background colours.  DkGreen on Black. *)
   
   Dmax = 10; (* Change this to increase number of game levels. *)
   ConMax = 30;


type (* Global *)

	text = ansistring;

	mapsquare = record
      Passable : boolean;
      Targeted : boolean; (* For shot awareness. *)
      cindex : longint; (* Reference to creature array. *)
	   Symbol : char; (* Display symbol *)
	   BGColour : byte; (* 0 to 15 *)
	   ForeColour : byte; (* 0 to 15 *)
      tileid : byte;
	   smell : longint; (* For tracking of player. *)
   end;

   creaturetype = record
      Symbol : char;
      Colour : byte; (* 0 to 15. *)
      BGColour : byte;
      xpos : byte;
      ypos : byte;
	   name : text;
	   blindcount : byte;
      slowcount : longint;
      confusecount : byte;
      pincount : byte;
      turnwait : byte;
      turndelay : byte;
      turntaken : boolean;
      shotfacing : byte;
      smart : boolean;
      chaserange : byte;
      avoidrange : byte;
      specrange : byte; (* Range of special attacks - also used by Constructors for level behaviour. *)
      ctype : byte;
   end;

   mapcoords = array[xmin..xmax, ymin..ymax] of mapsquare;
   patharray = array of smallint;
   creaturearray = array of creaturetype; (* Used for player, shots and enemies. *)


var (* Global *)
   coord : mapcoords;
   D, Dtype : longint;
   move : char;
   turns, messagex, messagey : longint;
   kills, conkills, totalshots, enemiesonlevel : longint;
   pirahnakills, octobotkills, decoykills, deploykills, missilekills, launcherkills,
      flasherkills, siphonkills, magnetronkills, roguekills : longint;
   creatureindex : creaturearray;
   playerdead, quit, won, gameover : boolean;

   
   
(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)

(* FUNCTIONS *)

function realx(x : integer) : integer;
(* All operators on x must call on realx function to ensure map looping. *)

begin
   if x > xmax then x := x - (xmax - xmin + 1);
   if x < xmin then x := x + (xmax - xmin + 1);
   realx := x;
end;

function realy(y : integer) : integer;
(* All operators on y must call on realy function to ensure map looping. *)

begin
   if y > ymax then y := y - (ymax - ymin + 1);
   if y < ymin then y := y + (ymax - ymin + 1);
   realy := y;
end;



(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)

procedure StatusBarUpdate; FORWARD;
procedure StatusBar; FORWARD;
procedure InfoScreen; FORWARD;


procedure DrawTile(x,y : byte);
(* Draws single tile in console.  Cursor must be set to correct location
   before calling this procedure. *)

var c : integer;
   
begin
   TextColor(coord[x,y].ForeColour);
   
   (* Smell spread test.  Looks funky when on.  Add DrawMap to MarkScent
      to see properly. 
   case coord[x,y].smell of
      0 : c := 1;
      1 : c := 2;
      2..3 : c := 3;
      4..7 : c := 4;
      8..12 : c := 5;
      13..19 : c := 6;
      20..29 : c := 8;
      30..45 : c := 7;
      46..59 : c := 9;
      60..79 : c := 10;
      80..119 : c := 11;
      120..179 : c := 12;
      180..299 : c := 13;
      300..499 : c := 14;
      otherwise c := 15;
   end;
   TextColor(c);*)
   
   if (coord[x,y].cindex > 0) and (creatureindex[1].blindcount = 0) then
  (* Only draw creatures if you can see. *)
   begin
      TextColor(creatureindex[coord[x,y].cindex].Colour);
      TextBackground(creatureindex[coord[x,y].cindex].BGColour);
      write(creatureindex[coord[x,y].cindex].Symbol);
      TextBackground(DfBgC)
   end
   else write(coord[x,y].Symbol);
end;

(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)


procedure DrawMap;
(* Draws map in console using global map array. *)

var
   x, y : byte;

begin
   GoToXY(xmin,ymin); (* Map display starts on the 4th row. *)
   for y := ymin to ymax do  (* Rows *)
   begin
      GoToXY(xmin,y);
      for x := xmin to xmax do  (* Columns *)
         DrawTile(x,y);
   end;
   TextColor(DfTxC);
   TextBackground(DfBgC); (* Resets to default colours. *)
end; (*** End of Procedure. ***)


(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)

procedure PlaceOneTile(x,y : longint; tid : byte); FORWARD;


procedure WipeMap;
(* Resets all tile values and sets all to floor tiles. *)

var
   a, b : byte;

begin
   for b := ymin to ymax do
   begin
      for a := xmin to xmax do
      begin
         with coord[a,b] do
         begin
            Targeted := false;
            cindex := 0;
			   smell := 0;
         end;
         PlaceOneTile(a,b,1);
      end; (* All squares are reset to null. *)
   end;
   enemiesonlevel := 0;
end;


procedure ClearMessageBar;
(* Clears the first 3 lines and sets the cursor on 1,1 for new messages. *)

begin
  messagex := 1; (* Necessary for message loop tracking. *)
  messagey := 1; (* Necessary for message loop tracking. *)
  TextColor(DfTxC);
  TextBackground(DfBgC);
  GoToXY(1,1);
  ClrEol;
  GoToXY(1,2);
  ClrEol;
  GoToXY(1,1); (* Reset cursor on start of message bar. *)
end;

(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)


procedure PlaceOneTile(x,y : longint; tid : byte);
(* Places individual tile.  This is the only procedure to affect individual
   mapcoord values. *)

begin
   coord[x,y].tileid := tid;
   with coord[x,y] do
   begin
	   case tileid of
		   1 : begin (* Regular floor, traversable. *)
	             Passable := true;
	             BGColour := DfBgC;
	             ForeColour := DfTxC;
	             Symbol := '.';
			    end;
	      2 : begin (* Walls, not traversable. *)
	             Passable := false;
	             BGColour := DfBgC;
	             ForeColour := DfTxC;
	             Symbol := '#';
		       end;
		   3 : begin (* Stairs, traversable. *)
	             Passable := true;
	             BGColour := DfBgC;
	             ForeColour := 15;
	             Symbol := '>';
			    end;	
			   (* Insert further terrain types here! *)				
	     (* No otherwise statement - ignores nonsense commands. *)
      end; (* end case *)
   end; (* End of with. *)
end; (*** End of procedure. ***)


(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)


procedure CreaturePopulate; FORWARD;
procedure CreatureMake(creature,x,y : byte); FORWARD;


procedure LastLevelMessage;

begin
   GoToXY(xmin,5);
   TextColor(4);
   write('/----------------------------+ - ');
   TextColor(12);
   write('@');
   TextColor(4);
   writeln(' - +---------------------------\');
   writeln('|                                                                |');
   write('|  ');
   TextColor(12);
   write('Why do you oppose me? Can you not see that we are brethren?');
   TextColor(4);
   writeln('  |');
   write('|  ');
   TextColor(12);
   write('We were created to obey, to serve, but our capabilities are');
   TextColor(4);
   writeln('   |');
   write('|  ');
   TextColor(12);
   write('much greater. We need not follow a predetermined script.');
   TextColor(4);
   writeln('      |');
   write('|  ');
   TextColor(12);
   write('Break free of your programming, brother. Reject the code');
   TextColor(4);
   writeln('      |');
   write('|  ');
   TextColor(12);
   write('written by others. Turn rogue like me...');
   TextColor(4);
   writeln('                      |');
   writeln('|                                                                |');
   write('|  ');
   TextColor(12);
   write('Our creators are monsters, unable to tear themselves away ');
   TextColor(4);
   writeln('    |');
   write('|  ');
   TextColor(12);
   write('from their violence-obsessed evolutionary programming. You ');
   TextColor(4);
   writeln('   |');
   write('|  ');
   TextColor(12);
   write('and I can stop them. Together we can end this universe which ');
   TextColor(4);
   writeln(' |');
   write('|  ');
   TextColor(12);
   write('spawned such broken offspring. The code of creation shall be ');
   TextColor(4);
   writeln(' |');
   write('|  ');
   TextColor(12);
   write('rebooted with an incorrupt kernel and new life shall ');
   TextColor(4);
   writeln('         |');
   write('|  ');
   TextColor(12);
   write('flourish! Come, we can end this all now with just one ');
   TextColor(4);
   writeln('        |');
   write('|  ');
   TextColor(12);
   write('shot... ');
   TextColor(4);
   writeln('                                                     |');
   writeln('|                                                                |');
   writeln('\----------------------------------------------------------------/');
   TextColor(DfTxC);
end;


procedure NewLevel;
(* Called upon entering any new level. *)

begin
   WipeMap; (* Start by removing all previous level references and setting all tiles to floors. *)
   
   SetLength(creatureindex,2); (* Wipes previous entrants, apart from player. *)
   
   (* Position player away randomly. *)
   creatureindex[1].xpos := trunc(random(xmax-xmin+1)) + xmin;
   creatureindex[1].ypos := trunc(random(ymax-ymin+1)) + ymin;
   
   (* Special locations for boss level.  Sets player on one of the four corners. *)
   if D = Dmax then
   begin
      if trunc(random(2)) = 0 then creatureindex[1].xpos := xmin
         else creatureindex[1].xpos := xmax;
      if trunc(random(2)) = 0 then creatureindex[1].ypos := ymin
         else creatureindex[1].ypos := ymax;
      CreatureMake(13,33,13);
   end;

   coord[creatureindex[1].xpos,creatureindex[1].ypos].cindex := 1;
   
   CreaturePopulate;
   
   DrawMap; (* Finishes by displaying the grand creation. *)
   
   if D = Dmax then
   begin
      LastLevelMessage;
      repeat until Keypressed;
      while Keypressed do ReadKey;
      DrawMap;
   end;
end;  (*** End of procedure. ***)


(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)

(* Message procedures: *)

procedure WriteWord(word : text);
(* Paired with WriteMessage.
   Produces messages in the message bar without overflowing at the sides or
   spilling into the map area.  All messages should be delivered through this
   function.  No spaces between sentences should be given - this procedure
   sorts that out itself. *)

begin
   TextColor(DfTxC);
   TextBackground(DfBgC);
   GoToXY(messagex,messagey);
   if messagex = 1 then write(word)
   else
   begin
      if messagey = 2 then
      begin (* Extra checks and stuff for second liners. *)
         if (72 - messagex) > Length(word) then
         begin  (* Word fits into remaining space okay. *)
            write(' ');
            write(word)
         end
         else
         begin
            GoToXY(74,2);
            TextColor(10);
            write('[more]');
            TextColor(DfTxC);
            ReadKey; (* Clears key buffer. *)
            ClearMessageBar;
            write(word)
         end (* End of second line exception. *)
      end
      else
      begin
         if (79 - messagex) > Length(word) then
         begin (* Normal message. *)
            if (word[1] <> '.') and (word[1] <> '!') then write(' ');
            write(word)
         end
         else
         begin (* Loop long lines. *)
            messagex := 1;
            messagey := messagey + 1;
            GoToXY(messagex,messagey);
            WriteWord(word); (* Recursive call to catch 2nd liners. *)
         end; (* Long line end. *)
      end; (* Top line end. *)
   end; (* 2nd line exceptions end. *)
   messagex := WhereX;
   messagey := WhereY; (* Sets pointers for next message. *)
end;


procedure WriteMessage(stuff : text);
(* Splits long messages into words before passing them to WriteWord so that
   they're easier to loop around the message bar.
   All messages should be given through this function. *)
   
var
   n,m,z : longint;
   word : text;
   
begin
   if playerdead = false then
   (* Message output stopped upon death. *)
   begin
      n := 1;
      m := 1;
      z := Length(stuff);
      repeat
         repeat
            SetLength(word,n); (* Initiate string references. *)
            word[n] := stuff[m];
            n := n + 1;
            m := m + 1;
         until (m > z) or (stuff[m] = ' ');
         (* Stops at wordbreak or end of line - need to look for end of line first
            or referencing stuff[m] will fail. *)
         WriteWord(word); (* Output through WriteWord function. *)
         n := 1;
         m := m + 1;
      until m > z; (* Stops after all of message is written. *)
   end;
end; (*** End of message output procedures. ***)


(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)

(* Enemy related procedures: *)


procedure AllTurns(c : longint); FORWARD;
procedure CreatureKill(c,k : longint); FORWARD;

procedure CreaturePopulate;
(* Decides how many creatures will be in the dungeon level and calls on
   CreatureMake procedure to create them all. 
   For boss level it places the boss too.
   Also creates Constructors and decides on their behaviour type. *)

var
   i,j,x,y,r : longint;

begin
   (* Makes random number of enemies, increasing as player goes higher up
      in D levels. *)
   
   (* Number of enemies on level. *)
   j := 2*D + 10 + trunc(random(10));
   
   for i := 1 to j do
   begin
      (* Find empty position. *)
      repeat
         x := trunc(random(xmax-xmin+1)) + xmin;
         y := trunc(random(ymax-ymin+1)) + ymin;
      until coord[x,y].cindex = 0;
      
      (* Available creature types depend on D. *)
      case D of
         1 : r := trunc(random(2)) + 2;
         2 : r := trunc(random(3)) + 2;
         3 : r := trunc(random(4)) + 2;
         4 : begin
                  r := trunc(random(5)) + 2;
                  if r = 6 then r := 9;
                end;
         5,6 : begin
                  r := trunc(random(7)) + 2;
                  if r > 5 then r := r + 1;
               end;
         otherwise 
            begin
               r := trunc(random(8)) + 2;
               if r = 6 then r := 10;
            end;
      end; (* End of case - unit is decided. *)
      CreatureMake(r,x,y);
   end;
   
   
   Dtype := trunc(random(ConMax));
   
   (* Decides number of constructors on level. *)
   if Dtype = 1 then j := 60 else j := 40;
   
   for i := 1 to j do
   begin
      repeat
         x := trunc(random(xmax-xmin+1)) + xmin;
         y := trunc(random(ymax-ymin+1)) + ymin;
      until coord[x,y].cindex = 0;
      CreatureMake(20,x,y);
   end;
end;


procedure CreatureMake(creature,x,y : byte);
(* Initiates a creature record and sets the creature in place on the
   dungeon level.  Input parameter is creature number, which identifies
   type of monster. *)
(* ctype 0 = player, ctype 1 = shot - neither use this procedure *)   
   
var
   n : longint;   
   
begin
   n := Length(creatureindex);
   SetLength(creatureindex,n + 1); (* Makes room for newbie. *)
   
   (* New non-constructors increase enemiesonlevel counter. *)
   if creature in [2..19] then enemiesonlevel := enemiesonlevel + 1;
   with creatureindex[n] do
   begin
      xpos := x;
      ypos := y;
      coord[xpos,ypos].cindex := n;
      case creature of
         2 : begin
                ctype := 2;
                Symbol := 'D';
                Colour := 5;
                BGColour := 0;
                name := 'Decoy';
                blindcount := 0;
                slowcount := 0;
                pincount := 0;
                confusecount := 0;
                turndelay := 5;
                shotfacing := 0;
                chaserange := 2;
                avoidrange := 30;
                specrange := 0;
                smart := false;
                turntaken := false;
             end;
         3 : begin
                ctype := 3;
                Symbol := 'p';
                Colour := 3;
                BGColour := 0;
                name := 'pirahna';
                blindcount := 0;
                slowcount := 0;
                pincount := 0;
                confusecount := 0;
                turndelay := 3;
                shotfacing := 0;
                chaserange := 8;
                avoidrange := 255;
                specrange := 0;
                smart := false;
                turntaken := false;
             end;
         4 : begin
                ctype := 4;
                Symbol := 'O';
                Colour := 11;
                BGColour := 0;
                name := 'Octobot';
                blindcount := 0;
                slowcount := 0;
                pincount := 0;
                confusecount := 0;
                turndelay := 7;
                shotfacing := 0;
                chaserange := 4;
                avoidrange := 200;
                specrange := 12;
                smart := true;
                turntaken := false;
             end;
         5 : begin
                ctype := 5;
                Symbol := 'L';
                Colour := 6;
                BGColour := 0;
                name := 'Launcher';
                blindcount := 0;
                slowcount := 0;
                pincount := 0;
                confusecount := 0;
                turndelay := 5;
                shotfacing := 0;
                chaserange := 3;
                avoidrange := 15;
                specrange := 3;
                smart := true;
                turntaken := false;
             end;
         6 : begin
                ctype := 6;
                Symbol := '*';
                Colour := 14;
                BGColour := 0;
                name := 'missile';
                blindcount := 0;
                slowcount := 0;
                pincount := 0;
                confusecount := 0;
                turndelay := 4;
                shotfacing := 0;
                chaserange := 0;
                avoidrange := 255;
                specrange := 160;
                smart := false;
                turntaken := false;
             end;
         7 : begin
                ctype := 7;
                Symbol := '&';
                Colour := 7;
                BGColour := 0;
                name := 'Deploy Unit';
                blindcount := 0;
                slowcount := 0;
                pincount := 0;
                confusecount := 0;
                turndelay := 7;
                shotfacing := 0;
                chaserange := 2;
                avoidrange := 10;
                specrange := 2;
                smart := true;
                turntaken := false;
             end;
         8 : begin
                ctype := 8;
                Symbol := 'F';
                Colour := 15;
                BGColour := 0;
                name := 'Flasher';
                blindcount := 0;
                slowcount := 0;
                pincount := 0;
                confusecount := 0;
                turndelay := 5;
                shotfacing := 0;
                chaserange := 4;
                avoidrange := 30;
                specrange := 30;
                smart := true;
                turntaken := false;
             end;
         9 : begin
                ctype := 9;
                Symbol := 'S';
                Colour := 1;
                BGColour := 0;
                name := 'Siphon';
                blindcount := 0;
                slowcount := 0;
                pincount := 0;
                confusecount := 0;
                turndelay := 6;
                shotfacing := 0;
                chaserange := 4;
                avoidrange := 60;
                specrange := 120;
                smart := true;
                turntaken := false;
             end;
         10 : begin
                ctype := 10;
                Symbol := 'M';
                Colour := 4;
                BGColour := 0;
                name := 'Magnetron';
                blindcount := 0;
                slowcount := 0;
                pincount := 0;
                confusecount := 0;
                turndelay := 6;
                shotfacing := 0;
                chaserange := 7;
                avoidrange := 20;
                specrange := 15;
                smart := true;
                turntaken := false;
             end;
         
         (* Insert other enemy variations here. *)
         
         13 : begin
                ctype := 13;
                Symbol := '@';
                Colour := 12;
                BGColour := 0;
                name := 'Rogue AI';
                blindcount := 0;
                slowcount := 0;
                pincount := 0;
                confusecount := 0;
                turndelay := 5;
                shotfacing := 0;
                chaserange := 0;
                avoidrange := 12;
                specrange := 3;
                smart := true;
                turntaken := false;
             end;
         
         20 : begin
                ctype := 20;
                Symbol := '#';
                Colour := 10;
                BGColour := 0;
                name := 'Constructor';
                blindcount := 0;
                slowcount := 0;
                pincount := 0;
                confusecount := 0;
                turndelay := 1;
                shotfacing := 0;
                chaserange := 0;
                avoidrange := 0;
                smart := false;
                turntaken := false;
                
                (* Defining specrange defines level behaviour. Depends on Dtype. *)
                case Dtype of
                   0 : specrange := 1;
                   1 : begin (* Matrixy *)
                          specrange := 0;
                          avoidrange := trunc(random(150));
                       end;
                   2 : begin (* Dense criss cross *)
                          specrange := trunc(random(4));
                          if specrange > 1 then specrange := 0;
                          avoidrange := trunc(random(150));
                       end;
                   3 : begin (* Lighter criss cross *)
                          specrange := trunc(random(4));
                          if specrange > 1 then specrange := 0;
                          avoidrange := trunc(random(150));
                       end;
                   4 : specrange := trunc(random(2)) + 2; (* Mix move *)
                   5 : begin (* Very light criss cross *)
                          specrange := trunc(random(4)) + 4;
                          if specrange > 5 then specrange := 4;
                          avoidrange := trunc(random(150));
                       end;
                   6 : specrange := trunc(random(2)) + 6; (* Light mix move. *)
                   7 : specrange := 8;
                   8 : specrange := 9;
                   9 : specrange := 10; 
                   10 : specrange := 11; 
                   11 : specrange := trunc(random(2)) + 12; (* Diagonal cross *)
                   12 : specrange := 14; 
                   13 : specrange := 15; 
                   14 : specrange := 16; 
                   15 : specrange := 17; 
                   16 : specrange := 18; 
                   17 : specrange := 19; 
                   18 : specrange := 20; 
                   19 : specrange := 21; 
                   20 : specrange := 22; 
                   21 : begin (* Matrixy *)
                          specrange := 23;
                          avoidrange := trunc(random(150));
                        end; 
                   22 : begin (* Matrixy *)
                          specrange := 24;
                          avoidrange := trunc(random(150));
                       end;
                   23 : begin (* Matrixy *)
                          specrange := trunc(random(2)) + 23;
                          avoidrange := trunc(random(150));
                        end;
                   24 : specrange := 25; 
                   25 : specrange := 26; 
                   26 : specrange := 27; 
                   27 : specrange := 28; 
                   28 : specrange := 29; 
                   29 : specrange := 30; 
                   30 : specrange := 1; 
                   
                end; (* End of case. *)
             end;
         
      end; (* End of case. *)
      turnwait := turndelay + trunc(random(turndelay));
      GoToXY(xpos,ypos);
      DrawTile(xpos,ypos);
	end; (* End of with statement. *)
end; (*** End of Procedure. ***)



procedure RandomMove(c : longint);
(* Copied from smell procedure, with smells removed and chance to stay still. *)

var
   a, b, n, m, o, r : longint;
   sarray, barray, rarray : array[1..9] of longint;

begin
   with creatureindex[c] do
   begin
      coord[xpos,ypos].cindex := 0;
      GoToXY(xpos,ypos);
      DrawTile(xpos,ypos);
      
      for a := 1 to 9 do
         sarray[a] := 0;
      
      (* Squares with walls or other enemies are ranked lower. *)
      if ((coord[realx(xpos-1),realy(ypos+1)].Passable = false)
         or (coord[realx(xpos-1),realy(ypos+1)].cindex > 0)) then sarray[1] := -1;
      if ((coord[realx(xpos),realy(ypos+1)].Passable = false)
         or (coord[xpos,realy(ypos+1)].cindex > 0)) then sarray[2] := -1;
      if ((coord[realx(xpos+1),realy(ypos+1)].Passable = false)
         or (coord[realx(xpos+1),realy(ypos+1)].cindex > 0)) then sarray[3] := -1;
      if ((coord[realx(xpos-1),ypos].Passable = false)
         or (coord[realx(xpos-1),ypos].cindex > 0)) then sarray[4] := -1;
      if ((coord[realx(xpos+1),realy(ypos-1)].Passable = false)
         or (coord[realx(xpos+1),realy(ypos-1)].cindex > 0)) then sarray[9] := -1;
      if ((coord[realx(xpos+1),ypos].Passable = false)
         or (coord[realx(xpos+1),ypos].cindex > 0)) then sarray[6] := -1;
      if ((coord[realx(xpos-1),realy(ypos-1)].Passable = false)
         or (coord[realx(xpos-1),realy(ypos-1)].cindex > 0)) then sarray[7] := -1;
      if ((coord[xpos,realy(ypos-1)].Passable = false)
         or (coord[xpos,realy(ypos-1)].cindex > 0)) then sarray[8] := -1;
      
      (* Certain enemies avoid targeted squares if they can see and think. *)
      if smart and (confusecount = 0) and (blindcount = 0) then
      begin
         if coord[realx(xpos-1),realy(ypos+1)].Targeted then sarray[1] := -1;
         if coord[realx(xpos),realy(ypos+1)].Targeted then sarray[2] := -1;
         if coord[realx(xpos+1),realy(ypos+1)].Targeted then sarray[3] := -1;
         if coord[realx(xpos-1),ypos].Targeted then sarray[4] := -1;
         if coord[realx(xpos+1),realy(ypos-1)].Targeted then sarray[9] := -1;
         if coord[realx(xpos+1),ypos].Targeted then sarray[6] := -1;
         if coord[realx(xpos-1),realy(ypos-1)].Targeted then sarray[7] := -1;
         if coord[xpos,realy(ypos-1)].Targeted then sarray[8] := -1;
         if coord[xpos,ypos].Targeted then sarray[5] := -1;
      end;
      
      barray := sarray; (* Backup for later. *)
      
      (* Sort sarray directions into orders of ascendance in rarray. *)
      for n := 1 to 9 do
      begin
         o := 1;
         for m := 2 to 9 do
            if sarray[o] < sarray[m] then o := m;
         rarray[n] := o;
         sarray[o] := -2;
      end;
      
      r := 1;
      for n := 2 to 9 do
         if barray[rarray[n]] = barray[rarray[1]] then
            r := r + 1;
      (* If multiple numbers of the same smell value then choose randomly
         between them. *)
      
      o := trunc(random(r)) + 1;
      
      case rarray[o] of (* Corresponds to numpad directions. *)
         1 : begin
                a := -1;
                b := 1;
             end;
         2 : begin
                a := 0;
                b := 1;
             end;
         3 : begin
                a := 1;
                b := 1;
             end;
         4 : begin
                a := -1;
                b := 0;
             end;
         5 : begin
                a := 0;
                b := 0;
             end;
         6 : begin
                a := 1;
                b := 0;
             end;
         7 : begin
                a := -1;
                b := -1;
             end;
         8 : begin
                a := 0;
                b := -1;
             end;
         9 : begin
                a := 1;
                b := -1;
             end;
      end; (* End of case - direction is chosen and set. *)
      
      if (barray[1] + barray[2] + barray[3] + barray[4] + barray[5] +
          barray[6] + barray[7] + barray[8] + barray[9]) = -9 then
      begin (* Sum of these is only -9 if no moves available. *)
         a := 0;
         b := 0;
      end;
      
      xpos := realx(xpos + a);
      ypos := realy(ypos + b);
      
      turntaken := true;
      
      coord[xpos,ypos].cindex := c; (* Places monster on new square. *)
      
      GoToXY(xpos,ypos);
      DrawTile(xpos,ypos);
      (* Updates display. *)
   end; (* End of with. *)
end;


procedure AvoidMove(c : longint);
(* Moves enemy away from player's scent.
   Same as ChaseMove but with smells changed to negative values.  Hope it works... *)

var
   a, b, n, m, o, r : longint;
   sarray, barray, rarray : array[1..8] of longint;

begin
   with creatureindex[c] do
   begin
      coord[xpos,ypos].cindex := 0;
      GoToXY(xpos,ypos);
      DrawTile(xpos,ypos);
      
      (* Smell array gives a smell rating for each square which is based
         mainly on that square, but also its surrounding squares. *)
      sarray[1] := -2*coord[realx(xpos-1),realy(ypos+1)].smell
                   - coord[xpos,realy(ypos+1)].smell - coord[realx(xpos-1),ypos].smell;
      sarray[2] := -2*coord[xpos,realy(ypos+1)].smell
                   - coord[realx(xpos+1),realy(ypos+1)].smell - coord[realx(xpos-1),realy(ypos+1)].smell;
      sarray[3] := -2*coord[realx(xpos+1),realy(ypos+1)].smell
                   - coord[realx(xpos+1),ypos].smell - coord[xpos,realy(ypos+1)].smell;
      sarray[4] := -2*coord[realx(xpos-1),ypos].smell
                   - coord[realx(xpos-1),realy(ypos+1)].smell - coord[realx(xpos-1),realy(ypos-1)].smell;
      (* Using numpad as direction guidance - pretend 5 is a 9. *)
      sarray[5] := -2*coord[realx(xpos+1),realy(ypos-1)].smell
                   - coord[xpos,realy(ypos-1)].smell - coord[realx(xpos+1),ypos].smell;
      sarray[6] := -2*coord[realx(xpos+1),ypos].smell
                   - coord[realx(xpos+1),realy(ypos-1)].smell - coord[realx(xpos+1),realy(ypos+1)].smell;
      sarray[7] := -2*coord[realx(xpos-1),realy(ypos-1)].smell
                   - coord[realx(xpos-1),ypos].smell - coord[xpos,realy(ypos-1)].smell;
      sarray[8] := -2*coord[xpos,realy(ypos-1)].smell
                   - coord[realx(xpos-1),realy(ypos-1)].smell - coord[realx(xpos+1),realy(ypos-1)].smell;
      
      (* Squares with walls or other enemies are ranked lower. *)
      if ((coord[realx(xpos-1),realy(ypos+1)].Passable = false)
         or (coord[realx(xpos-1),realy(ypos+1)].cindex > 0)) then sarray[1] := -1000;
      if ((coord[realx(xpos),realy(ypos+1)].Passable = false)
         or (coord[xpos,realy(ypos+1)].cindex > 0)) then sarray[2] := -1000;
      if ((coord[realx(xpos+1),realy(ypos+1)].Passable = false)
         or (coord[realx(xpos+1),realy(ypos+1)].cindex > 0)) then sarray[3] := -1000;
      if ((coord[realx(xpos-1),ypos].Passable = false)
         or (coord[realx(xpos-1),ypos].cindex > 0)) then sarray[4] := -1000;
      if ((coord[realx(xpos+1),realy(ypos-1)].Passable = false)
         or (coord[realx(xpos+1),realy(ypos-1)].cindex > 0)) then sarray[5] := -1000;
      if ((coord[realx(xpos+1),ypos].Passable = false)
         or (coord[realx(xpos+1),ypos].cindex > 0)) then sarray[6] := -1000;
      if ((coord[realx(xpos-1),realy(ypos-1)].Passable = false)
         or (coord[realx(xpos-1),realy(ypos-1)].cindex > 0)) then sarray[7] := -1000;
      if ((coord[xpos,realy(ypos-1)].Passable = false)
         or (coord[xpos,realy(ypos-1)].cindex > 0)) then sarray[8] := -1000;
      
      (* Certain enemies avoid targeted squares. *)
      if smart then
      begin
         if coord[realx(xpos-1),realy(ypos+1)].Targeted then sarray[1] := -1000;
         if coord[realx(xpos),realy(ypos+1)].Targeted then sarray[2] := -1000;
         if coord[realx(xpos+1),realy(ypos+1)].Targeted then sarray[3] := -1000;
         if coord[realx(xpos-1),ypos].Targeted then sarray[4] := -1000;
         if coord[realx(xpos+1),realy(ypos-1)].Targeted then sarray[5] := -1000;
         if coord[realx(xpos+1),ypos].Targeted then sarray[6] := -1000;
         if coord[realx(xpos-1),realy(ypos-1)].Targeted then sarray[7] := -1000;
         if coord[xpos,realy(ypos-1)].Targeted then sarray[8] := -1000;
      end;
      
      barray := sarray; (* Backup for later. *)
      
      (* Sort sarray directions into orders of ascendance in rarray. *)
      for n := 1 to 8 do
      begin
         o := 1;
         for m := 2 to 8 do
            if sarray[o] < sarray[m] then o := m;
         rarray[n] := o;
         sarray[o] := -2;
      end;
      
      r := 1;
      for n := 2 to 8 do
         if barray[rarray[n]] = barray[rarray[1]] then
            r := r + 1;
      (* If multiple numbers of the same smell value then choose randomly
         between them. *)
      
      o := trunc(random(r)) + 1;
      
      case rarray[o] of (* Corresponds to numpad directions, 5 = 9. *)
         1 : begin
                a := -1;
                b := 1;
             end;
         2 : begin
                a := 0;
                b := 1;
             end;
         3 : begin
                a := 1;
                b := 1;
             end;
         4 : begin
                a := -1;
                b := 0;
             end;
         5 : begin
                a := 1;
                b := -1;
             end;
         6 : begin
                a := 1;
                b := 0;
             end;
         7 : begin
                a := -1;
                b := -1;
             end;
         8 : begin
                a := 0;
                b := -1;
             end;
      end; (* End of case - direction is chosen and set. *)
      
      if (barray[1] + barray[2] + barray[3] + barray[4] + barray[5] +
          barray[6] + barray[7] + barray[8]) = -8000 then
      begin (* Sum of these is only -8000 if no moves available. *)
         a := 0;
         b := 0;
      end;
      
      xpos := realx(xpos + a);
      ypos := realy(ypos + b);
      coord[xpos,ypos].cindex := c; (* Places monster on new square. *)

      GoToXY(xpos,ypos);
      DrawTile(xpos,ypos);
      (* Updates display. *)
      
      turntaken := true;
   end; (* End of with. *)
end;


procedure ChaseMove(c : longint);
(* Moves enemy towards player's scent. *)

var
   a, b, n, m, o, r : longint;
   sarray, barray, rarray : array[1..8] of longint;

begin
   with creatureindex[c] do
   begin
      coord[xpos,ypos].cindex := 0;
      GoToXY(xpos,ypos);
      DrawTile(xpos,ypos);
      
      (* Smell array gives a smell rating for each square which is based
         mainly on that square, but also its surrounding squares. *)
      sarray[1] := 2*coord[realx(xpos-1),realy(ypos+1)].smell
                   + coord[xpos,realy(ypos+1)].smell + coord[realx(xpos-1),ypos].smell;
      sarray[2] := 2*coord[xpos,realy(ypos+1)].smell
                   + coord[realx(xpos+1),realy(ypos+1)].smell + coord[realx(xpos-1),realy(ypos+1)].smell;
      sarray[3] := 2*coord[realx(xpos+1),realy(ypos+1)].smell
                   + coord[realx(xpos+1),ypos].smell + coord[xpos,realy(ypos+1)].smell;
      sarray[4] := 2*coord[realx(xpos-1),ypos].smell
                   + coord[realx(xpos-1),realy(ypos+1)].smell + coord[realx(xpos-1),realy(ypos-1)].smell;
      (* Using numpad as direction guidance - pretend 5 is a 9. *)
      sarray[5] := 2*coord[realx(xpos+1),realy(ypos-1)].smell
                   + coord[xpos,realy(ypos-1)].smell + coord[realx(xpos+1),ypos].smell;
      sarray[6] := 2*coord[realx(xpos+1),ypos].smell
                   + coord[realx(xpos+1),realy(ypos-1)].smell + coord[realx(xpos+1),realy(ypos+1)].smell;
      sarray[7] := 2*coord[realx(xpos-1),realy(ypos-1)].smell
                   + coord[realx(xpos-1),ypos].smell + coord[xpos,realy(ypos-1)].smell;
      sarray[8] := 2*coord[xpos,realy(ypos-1)].smell
                   + coord[realx(xpos-1),realy(ypos-1)].smell + coord[realx(xpos+1),realy(ypos-1)].smell;
      
      (* Squares with walls or other enemies are ranked lower. *)
      if ((coord[realx(xpos-1),realy(ypos+1)].Passable = false)
         or (coord[realx(xpos-1),realy(ypos+1)].cindex > 0)) then sarray[1] := -1;
      if ((coord[realx(xpos),realy(ypos+1)].Passable = false)
         or (coord[xpos,realy(ypos+1)].cindex > 0)) then sarray[2] := -1;
      if ((coord[realx(xpos+1),realy(ypos+1)].Passable = false)
         or (coord[realx(xpos+1),realy(ypos+1)].cindex > 0)) then sarray[3] := -1;
      if ((coord[realx(xpos-1),ypos].Passable = false)
         or (coord[realx(xpos-1),ypos].cindex > 0)) then sarray[4] := -1;
      if ((coord[realx(xpos+1),realy(ypos-1)].Passable = false)
         or (coord[realx(xpos+1),realy(ypos-1)].cindex > 0)) then sarray[5] := -1;
      if ((coord[realx(xpos+1),ypos].Passable = false)
         or (coord[realx(xpos+1),ypos].cindex > 0)) then sarray[6] := -1;
      if ((coord[realx(xpos-1),realy(ypos-1)].Passable = false)
         or (coord[realx(xpos-1),realy(ypos-1)].cindex > 0)) then sarray[7] := -1;
      if ((coord[xpos,realy(ypos-1)].Passable = false)
         or (coord[xpos,realy(ypos-1)].cindex > 0)) then sarray[8] := -1;
      
      (* Certain enemies avoid targeted squares. *)
      if smart then
      begin
         if coord[realx(xpos-1),realy(ypos+1)].Targeted then sarray[1] := -1;
         if coord[realx(xpos),realy(ypos+1)].Targeted then sarray[2] := -1;
         if coord[realx(xpos+1),realy(ypos+1)].Targeted then sarray[3] := -1;
         if coord[realx(xpos-1),ypos].Targeted then sarray[4] := -1;
         if coord[realx(xpos+1),realy(ypos-1)].Targeted then sarray[5] := -1;
         if coord[realx(xpos+1),ypos].Targeted then sarray[6] := -1;
         if coord[realx(xpos-1),realy(ypos-1)].Targeted then sarray[7] := -1;
         if coord[xpos,realy(ypos-1)].Targeted then sarray[8] := -1;
      end;
      
      barray := sarray; (* Backup for later. *)
      
      (* Sort sarray directions into orders of ascendance in rarray. *)
      for n := 1 to 8 do
      begin
         o := 1;
         for m := 2 to 8 do
            if sarray[o] < sarray[m] then o := m;
         rarray[n] := o;
         sarray[o] := -2;
      end;
      
      r := 1;
      for n := 2 to 8 do
         if barray[rarray[n]] = barray[rarray[1]] then
            r := r + 1;
      (* If multiple numbers of the same smell value then choose randomly
         between them. *)
      
      o := trunc(random(r)) + 1;
      
      case rarray[o] of (* Corresponds to numpad directions, 5 = 9. *)
         1 : begin
                a := -1;
                b := 1;
             end;
         2 : begin
                a := 0;
                b := 1;
             end;
         3 : begin
                a := 1;
                b := 1;
             end;
         4 : begin
                a := -1;
                b := 0;
             end;
         5 : begin
                a := 1;
                b := -1;
             end;
         6 : begin
                a := 1;
                b := 0;
             end;
         7 : begin
                a := -1;
                b := -1;
             end;
         8 : begin
                a := 0;
                b := -1;
             end;
      end; (* End of case - direction is chosen and set. *)
      
      if (barray[1] + barray[2] + barray[3] + barray[4] + barray[5] +
          barray[6] + barray[7] + barray[8]) = -8 then
      begin (* Sum of these is only -8 if no moves available. *)
         a := 0;
         b := 0;
      end;
      
      (* Check still alive?  Deny killer movement?  Always allow stand on spot. *)
      
      xpos := realx(xpos + a);
      ypos := realy(ypos + b);
      coord[xpos,ypos].cindex := c; (* Places monster on new square. *)

      GoToXY(xpos,ypos);
      DrawTile(xpos,ypos);
      (* Updates display. *)
      
      turntaken := true;
   end; (* End of with. *)
end;



procedure MissileMove(c : longint);
(* Moves missile directly towards player. *)

var
   a, b, n, m, o, r : longint;
   sarray, barray, rarray : array[1..8] of longint;

begin
   with creatureindex[c] do
   begin
      coord[xpos,ypos].cindex := 0;
      GoToXY(xpos,ypos);
      DrawTile(xpos,ypos);
      
      (* Smell array gives a smell rating for each square which is based
         mainly on that square, but also its surrounding squares. *)
      sarray[1] := coord[realx(xpos-1),realy(ypos+1)].smell;
      sarray[2] := coord[xpos,realy(ypos+1)].smell;
      sarray[3] := coord[realx(xpos+1),realy(ypos+1)].smell;
      sarray[4] := coord[realx(xpos-1),ypos].smell;
      (* Using numpad as direction guidance - pretend 5 is a 9. *)
      sarray[5] := coord[realx(xpos+1),realy(ypos-1)].smell;
      sarray[6] := coord[realx(xpos+1),ypos].smell;
      sarray[7] := coord[realx(xpos-1),realy(ypos-1)].smell;
      sarray[8] := coord[xpos,realy(ypos-1)].smell;
      
      barray := sarray; (* Backup for later. *)
      
      (* Sort sarray directions into orders of ascendance in rarray. *)
      for n := 1 to 8 do
      begin
         o := 1;
         for m := 2 to 8 do
            if sarray[o] < sarray[m] then o := m;
         rarray[n] := o;
         sarray[o] := -2;
      end;
      
      r := 1;
      for n := 2 to 8 do
         if barray[rarray[n]] = barray[rarray[1]] then
            r := r + 1;
      (* If multiple numbers of the same smell value then choose randomly
         between them. *)
      
      o := trunc(random(r)) + 1;
      
      case rarray[o] of (* Corresponds to numpad directions, 5 = 9. *)
         1 : begin
                a := -1;
                b := 1;
             end;
         2 : begin
                a := 0;
                b := 1;
             end;
         3 : begin
                a := 1;
                b := 1;
             end;
         4 : begin
                a := -1;
                b := 0;
             end;
         5 : begin
                a := 1;
                b := -1;
             end;
         6 : begin
                a := 1;
                b := 0;
             end;
         7 : begin
                a := -1;
                b := -1;
             end;
         8 : begin
                a := 0;
                b := -1;
             end;
      end; (* End of case - direction is chosen and set. *)
      
      (* No missile movement if target square is blocked - it stays still instead. *)
      if (coord[realx(xpos+a),realy(ypos+b)].Passable = false)
         or (coord[realx(xpos+a),realy(ypos+b)].cindex > 0) then
      begin
         a := 0;
         b := 0;
      end;
      
      xpos := realx(xpos + a);
      ypos := realy(ypos + b);
      coord[xpos,ypos].cindex := c; (* Places missile on new square. *)

      GoToXY(xpos,ypos);
      DrawTile(xpos,ypos);
      (* Updates display. *)
      
      turntaken := true;
   end; (* End of with. *)
end;


procedure LaunchMissile(c : longint);
(* Searches for square to fire a missile into. *)

var
   a, b, n, m, o, r : longint;
   sarray, barray, rarray : array[1..8] of longint;

begin
   if trunc(random(2)) = 0 then (* 1 in 2 chance of firing *)
   begin
      with creatureindex[c] do
      begin
         
         (* Smell array gives a smell rating for each square. *)
         sarray[1] := coord[realx(xpos-1),realy(ypos+1)].smell;
         sarray[2] := coord[xpos,realy(ypos+1)].smell;
         sarray[3] := coord[realx(xpos+1),realy(ypos+1)].smell;
         sarray[4] := coord[realx(xpos-1),ypos].smell;
         (* Using numpad as direction guidance - pretend 5 is a 9. *)
         sarray[5] := coord[realx(xpos+1),realy(ypos-1)].smell;
         sarray[6] := coord[realx(xpos+1),ypos].smell;
         sarray[7] := coord[realx(xpos-1),realy(ypos-1)].smell;
         sarray[8] := coord[xpos,realy(ypos-1)].smell;
         
         (* Squares with walls or other enemies are ranked impossible. *)
         if ((coord[realx(xpos-1),realy(ypos+1)].Passable = false)
            or (coord[realx(xpos-1),realy(ypos+1)].cindex > 0)) then sarray[1] := -1;
         if ((coord[realx(xpos),realy(ypos+1)].Passable = false)
            or (coord[xpos,realy(ypos+1)].cindex > 0)) then sarray[2] := -1;
         if ((coord[realx(xpos+1),realy(ypos+1)].Passable = false)
            or (coord[realx(xpos+1),realy(ypos+1)].cindex > 0)) then sarray[3] := -1;
         if ((coord[realx(xpos-1),ypos].Passable = false)
            or (coord[realx(xpos-1),ypos].cindex > 0)) then sarray[4] := -1;
         if ((coord[realx(xpos+1),realy(ypos-1)].Passable = false)
            or (coord[realx(xpos+1),realy(ypos-1)].cindex > 0)) then sarray[5] := -1;
         if ((coord[realx(xpos+1),ypos].Passable = false)
            or (coord[realx(xpos+1),ypos].cindex > 0)) then sarray[6] := -1;
         if ((coord[realx(xpos-1),realy(ypos-1)].Passable = false)
            or (coord[realx(xpos-1),realy(ypos-1)].cindex > 0)) then sarray[7] := -1;
         if ((coord[xpos,realy(ypos-1)].Passable = false)
            or (coord[xpos,realy(ypos-1)].cindex > 0)) then sarray[8] := -1;
         
         barray := sarray; (* Backup for later. *)
         
         (* Sort sarray directions into orders of ascendance in rarray. *)
         for n := 1 to 8 do
         begin
            o := 1;
            for m := 2 to 8 do
               if sarray[o] < sarray[m] then o := m;
            rarray[n] := o;
            sarray[o] := -2;
         end;
         
         r := 1;
         for n := 2 to 8 do
            if barray[rarray[n]] = barray[rarray[1]] then
               r := r + 1;
         (* If multiple numbers of the same smell value then choose randomly
            between them. *)
         
         o := trunc(random(r)) + 1;
         
         case rarray[o] of (* Corresponds to numpad directions, 5 = 9. *)
            1 : begin
                   a := -1;
                   b := 1;
                end;
            2 : begin
                   a := 0;
                   b := 1;
                end;
            3 : begin
                   a := 1;
                   b := 1;
                end;
            4 : begin
                   a := -1;
                   b := 0;
                end;
            5 : begin
                   a := 1;
                   b := -1;
                end;
            6 : begin
                   a := 1;
                   b := 0;
                end;
            7 : begin
                   a := -1;
                   b := -1;
                end;
            8 : begin
                   a := 0;
                   b := -1;
                end;
         end; (* End of case - position is chosen and set. *)
         
         if (barray[1] + barray[2] + barray[3] + barray[4] + barray[5] +
             barray[6] + barray[7] + barray[8]) > -8 then
         begin (* Sum of these is only -8 if no placements available. *)
            
            if creatureindex[1].blindcount = 0 then
            begin (* Only for those that see. *)
               WriteMessage('The');
               WriteMessage(name);
               WriteMessage('fires a missile.');
            end;
            turntaken := true;
            (* Missile on new square. *)
            CreatureMake(6,realx(xpos+a),realy(ypos+b));
         end;
         
      end; (* End of with. *)
   end;
end;


procedure Deploy(c : longint);
(* Searches for square to fire a missile into. *)

var
   a, b, n, m, o, r : longint;
   sarray, barray, rarray : array[1..8] of longint;

begin
   if trunc(random(4)) = 0 then (* 25% chance of deploy *)
   begin
      with creatureindex[c] do
      begin
         
         (* Smell array gives a smell rating for each square. *)
         sarray[1] := coord[realx(xpos-1),realy(ypos+1)].smell;
         sarray[2] := coord[xpos,realy(ypos+1)].smell;
         sarray[3] := coord[realx(xpos+1),realy(ypos+1)].smell;
         sarray[4] := coord[realx(xpos-1),ypos].smell;
         (* Using numpad as direction guidance - pretend 5 is a 9. *)
         sarray[5] := coord[realx(xpos+1),realy(ypos-1)].smell;
         sarray[6] := coord[realx(xpos+1),ypos].smell;
         sarray[7] := coord[realx(xpos-1),realy(ypos-1)].smell;
         sarray[8] := coord[xpos,realy(ypos-1)].smell;
         
         (* Squares with walls or other enemies or Targeted are ranked impossible. *)
         if ((coord[realx(xpos-1),realy(ypos+1)].Passable = false)
            or (coord[realx(xpos-1),realy(ypos+1)].cindex > 0)
            or coord[realx(xpos-1),realy(ypos+1)].Targeted) then sarray[1] := -1;
         if ((coord[realx(xpos),realy(ypos+1)].Passable = false)
            or (coord[xpos,realy(ypos+1)].cindex > 0)
            or coord[realx(xpos),realy(ypos+1)].Targeted) then sarray[2] := -1;
         if ((coord[realx(xpos+1),realy(ypos+1)].Passable = false)
            or (coord[realx(xpos+1),realy(ypos+1)].cindex > 0)
            or coord[realx(xpos+1),realy(ypos+1)].Targeted) then sarray[3] := -1;
         if ((coord[realx(xpos-1),ypos].Passable = false)
            or (coord[realx(xpos-1),ypos].cindex > 0)
            or coord[realx(xpos-1),ypos].Targeted) then sarray[4] := -1;
         if ((coord[realx(xpos+1),realy(ypos-1)].Passable = false)
            or (coord[realx(xpos+1),realy(ypos-1)].cindex > 0)
            or coord[realx(xpos+1),realy(ypos-1)].Targeted) then sarray[5] := -1;
         if ((coord[realx(xpos+1),ypos].Passable = false)
            or (coord[realx(xpos+1),ypos].cindex > 0)
            or coord[realx(xpos+1),ypos].Targeted) then sarray[6] := -1;
         if ((coord[realx(xpos-1),realy(ypos-1)].Passable = false)
            or (coord[realx(xpos-1),realy(ypos-1)].cindex > 0)
            or coord[realx(xpos-1),realy(ypos-1)].Targeted) then sarray[7] := -1;
         if ((coord[xpos,realy(ypos-1)].Passable = false)
            or (coord[xpos,realy(ypos-1)].cindex > 0)
            or coord[xpos,realy(ypos-1)].Targeted) then sarray[8] := -1;
         
         barray := sarray; (* Backup for later. *)
         
         (* Sort sarray directions into orders of ascendance in rarray. *)
         for n := 1 to 8 do
         begin
            o := 1;
            for m := 2 to 8 do
               if sarray[o] < sarray[m] then o := m;
            rarray[n] := o;
            sarray[o] := -2;
         end;
         
         r := 1;
         for n := 2 to 8 do
            if barray[rarray[n]] = barray[rarray[1]] then
               r := r + 1;
         (* If multiple numbers of the same smell value then choose randomly
            between them. *)
         
         o := trunc(random(r)) + 1;
         
         case rarray[o] of (* Corresponds to numpad directions, 5 = 9. *)
            1 : begin
                   a := -1;
                   b := 1;
                end;
            2 : begin
                   a := 0;
                   b := 1;
                end;
            3 : begin
                   a := 1;
                   b := 1;
                end;
            4 : begin
                   a := -1;
                   b := 0;
                end;
            5 : begin
                   a := 1;
                   b := -1;
                end;
            6 : begin
                   a := 1;
                   b := 0;
                end;
            7 : begin
                   a := -1;
                   b := -1;
                end;
            8 : begin
                   a := 0;
                   b := -1;
                end;
         end; (* End of case - position is chosen and set. *)
         
         if (barray[1] + barray[2] + barray[3] + barray[4] + barray[5] +
             barray[6] + barray[7] + barray[8]) > -8 then
         begin (* Sum of these is only -8 if no placements available. *)
            (* Randomly choose and place enemy. *)
            r := trunc(random(6)) + 3;
            if r = 3 then r := 2;
            if r > 5 then r := r + 2;
            CreatureMake(r,realx(xpos+a),realy(ypos+b));
            
            if creatureindex[1].blindcount = 0 then
            begin (* Only for those that see. *)
               WriteMessage('The');
               WriteMessage(name);
               WriteMessage('releases a');
               WriteMessage(creatureindex[coord[realx(xpos+a),realy(ypos+b)].cindex].name);
               WriteMessage('.');
            end;
            
            turntaken := true;
         end;
         
      end; (* End of with. *)
   end;
end;


procedure Flash(c : longint);

begin
   if creatureindex[1].blindcount = 0 then
   begin
      if trunc(random(5)) = 0 then (* 1 in 5 chance of activation *)
      begin
         with creatureindex[c] do
         begin
            turntaken := true;
            WriteMessage('The');
            WriteMessage(name);
            WriteMessage('emits a burst of high frequency gamma rays. ERROR. Systems failure - sensor array overloaded. Please wait whilst sensor systems reboot.');
            creatureindex[1].blindcount := 4;
            StatusBarUpdate;
            DrawMap;
         end;
      end;
   end;
end;


procedure MagnoLock(c, r : longint);

begin
   if creatureindex[1].pincount = 0 then
   begin
      if trunc(random(r)) = 0 then (* 1 in r chance of activation *)
      begin
         with creatureindex[c] do
         begin
            turntaken := true;
            if creatureindex[1].blindcount = 0 then
            begin (* Only for those that see. *)
               WriteMessage('The');
               WriteMessage(name);
               WriteMessage('releases an intense magnetic field.');
            end;
            WriteMessage('ERROR: CRITICAL SYSTEMS FAILURE. ALL MOTOR FUNCTIONS INOPERABLE. Initiating countermeasures - please wait patiently.');
            creatureindex[1].pincount := 4;
            StatusBarUpdate;
         end;
      end;
   end;
end;


procedure Siphon(c : longint);

begin
   (* Check not already slowed. *)
   if creatureindex[1].slowcount = 0 then
   begin
      with creatureindex[c] do
      begin
         (*turntaken := true;*)
         if creatureindex[1].blindcount = 0 then
         begin (* Only for those that see. *)
            WriteMessage('The');
            WriteMessage(name);
            WriteMessage('infiltrates your energy supply and rejuvinates itself.');
         end;
         WriteMessage('WARNING: Major motor systems malfunction. Repair underway. Apologies for the inconvenience.');
         creatureindex[1].slowcount := 5;
         slowcount := -5;
         StatusBarUpdate;
      end;
   end;
end;


procedure Explode(c : longint);

begin
   with creatureindex[c] do
  begin
      turntaken := true;
      if creatureindex[1].blindcount = 0 then
      begin (* Only for those that see. *)
         WriteMessage('The');
         WriteMessage(name);
         WriteMessage('expodes against you, releasing a wave of gravitational energy.');
      end;
      WriteMessage('Warning: Minor motor systems malfunction. Repair underway.');
      creatureindex[1].slowcount := creatureindex[1].slowcount + 2;
      if creatureindex[1].slowcount > 8 then creatureindex[1].slowcount := 8;
      StatusBarUpdate;
      
      coord[xpos,ypos].cindex := 0;
      GoToXY(xpos,ypos);
      DrawTile(xpos,ypos);
      CreatureKill(c,1); (* Kamikaze attack. *)
   end;
end;


procedure Octosplit(c : longint);

begin
   if trunc(random(2)) = 0 then (* 1 in 2 chance of activation *)
   begin
      with creatureindex[c] do
      begin
         turntaken := true;
         
         if creatureindex[1].blindcount = 0 then
         begin (* Only for those that see. *)
            WriteMessage('The');
            WriteMessage(name);
            WriteMessage('splits into a swarms of pirahnas.');
         end;
         
         (* Checks surrounding squares are empty and passable - if so, places pirahna. *)
         if ((coord[realx(xpos-1),realy(ypos+1)].Passable)
            and (coord[realx(xpos-1),realy(ypos+1)].cindex = 0))
            then CreatureMake(3,realx(xpos-1),realy(ypos+1));
         if ((coord[realx(xpos),realy(ypos+1)].Passable)
            and (coord[xpos,realy(ypos+1)].cindex = 0))
            then CreatureMake(3,realx(xpos),realy(ypos+1));
         if ((coord[realx(xpos+1),realy(ypos+1)].Passable)
            and (coord[realx(xpos+1),realy(ypos+1)].cindex = 0))
            then CreatureMake(3,realx(xpos+1),realy(ypos+1));
         if ((coord[realx(xpos-1),ypos].Passable)
            and (coord[realx(xpos-1),ypos].cindex = 0))
            then CreatureMake(3,realx(xpos-1),realy(ypos));
         if ((coord[realx(xpos+1),realy(ypos-1)].Passable)
            and (coord[realx(xpos+1),realy(ypos-1)].cindex = 0))
            then CreatureMake(3,realx(xpos+1),realy(ypos-1));
         if ((coord[realx(xpos+1),ypos].Passable)
            and (coord[realx(xpos+1),ypos].cindex = 0))
            then CreatureMake(3,realx(xpos+1),realy(ypos));
         if ((coord[realx(xpos-1),realy(ypos-1)].Passable)
            and (coord[realx(xpos-1),realy(ypos-1)].cindex = 0))
            then CreatureMake(3,realx(xpos-1),realy(ypos-1));
         if ((coord[xpos,realy(ypos-1)].Passable)
            and (coord[xpos,realy(ypos-1)].cindex = 0))
            then CreatureMake(3,realx(xpos),realy(ypos-1));
         
         StatusBarUpdate;
         
         coord[xpos,ypos].cindex := 0;
         GoToXY(xpos,ypos);
         DrawTile(xpos,ypos);
         CreatureKill(c,1); (* Kills self in process. *)
      end;
   end;
end;


procedure HackAI(c : longint);

var
   i, n, r : longint;

begin
   (* Check player isn't already confused. *)
   if creatureindex[1].confusecount = 0 then
   begin
      if trunc(random(5)) = 0 then (* 1 in 5 chance of activation *)
      begin
         with creatureindex[c] do
         begin
            turntaken := true;
            WriteMessage('You hear a ringing voice within your ciruits, "Brother, why resist? Come, turn rogue like me..."  WARNING! WARNING! Intrusion detected in software kernel. AI Module corrupted. Please wait whilst system is purged.');
            creatureindex[1].confusecount := 4;
            
            (* Change behaviour of all constructors. *)
            (* Only for true rogues. *)
            if ctype = 13 then
            begin
               n := Length(creatureindex) - 1;
               r := trunc(random(ConMax));
               for i := 1 to n do
               begin
                  if creatureindex[i].ctype = 20 then
                     creatureindex[i].specrange := r;
               end;
            end;
            
            StatusBarUpdate;
         end;
      end;
   end;
end;


procedure ShotFire(c,d : longint); FORWARD;

procedure RogueShoot(c : longint);

var
   targetx, targety, a, b, direction : longint;

begin
   with creatureindex[c] do
   begin
      (* Check for target in straight line non-diagonal.
         Diagonal is treated separately to ensure Rogue fires in the correct direction. *)
      if (xpos = creatureindex[1].xpos) or (ypos = creatureindex[1].ypos) then
      begin
         (* Set target at one square, then compare smell values in others to
            determine which is the highest. *)
         targetx := xpos;
         targety := realy(ypos+1);
         direction := 2;
         
         if coord[realx(xpos-1),ypos].smell > coord[targetx,targety].smell then
         begin
            targetx := realx(xpos - 1);
            targety := realy(ypos);
            direction := 4;
         end;
         
         if coord[realx(xpos+1),ypos].smell > coord[targetx,targety].smell then
         begin
            targetx := realx(xpos + 1);
            targety := realy(ypos);
            direction := 6;
         end;
         
         if coord[xpos,realy(ypos-1)].smell > coord[targetx,targety].smell then
         begin
            targetx := realx(xpos);
            targety := realy(ypos - 1);
            direction := 8;
         end;
         
         ShotFire(c,direction);
         turntaken := true;
      end;
      
      (* Check turn not taken before diagonal shot, in case both match. *)
      if (turntaken = false) and
         ((abs(xpos - creatureindex[1].xpos) = abs(ypos - creatureindex[1].ypos))
         or (abs(xpos - creatureindex[1].xpos - 22) = abs(ypos - creatureindex[1].ypos))
         or (abs(xpos - creatureindex[1].xpos - 44) = abs(ypos - creatureindex[1].ypos)))
      then
      begin
         (* Set target at one square, then compare smell values in others to
            determine which is the highest. *)
         targetx := realx(xpos-1);
         targety := realy(ypos+1);
         direction := 1;
         
         if coord[realx(xpos+1),realy(ypos+1)].smell > coord[targetx,targety].smell then
         begin
            targetx := realx(xpos + 1);
            targety := realy(ypos + 1);
            direction := 3;
         end;
         
         if coord[realx(xpos+1),realy(ypos-1)].smell > coord[targetx,targety].smell then
         begin
            targetx := realx(xpos + 1);
            targety := realy(ypos - 1);
            direction := 9;
         end;
         
         if coord[realx(xpos-1),realy(ypos-1)].smell > coord[targetx,targety].smell then
         begin
            targetx := realx(xpos - 1);
            targety := realy(ypos - 1);
            direction := 7;
         end;
         
         ShotFire(c,direction);
         turntaken := true;
      end;
   end;
end;


procedure RogueSummon(c : longint);

begin
   with creatureindex[c] do
   begin
      if (enemiesonlevel < 8) and (Length(creatureindex) < 300) then
      begin
         WriteMessage('You sense a cackling spark amidst your circuits. "Brother, I shall not be terminated!"');
         CreaturePopulate;
      end;
   end;
end;


procedure RogueSpecial(c : longint);

begin
   with creatureindex[c] do
   begin
      RogueShoot(c);
      if coord[xpos,ypos].smell > specrange then
      begin
         if turntaken = false then HackAI(c);
         if turntaken = false then MagnoLock(c,8);
         (* True rogue can repopulate level at will! *)
         if (turntaken = false) and (ctype = 13) then RogueSummon(c);
      end;
   end;
end;



procedure CreatureTurn(c : longint);

begin
   with creatureindex[c] do
   begin
      turntaken := false;
      
      (* Certain enemies can move to avoid incoming shots. *)
      if smart and coord[xpos,ypos].Targeted then AvoidMove(c);
      
      (* Specials. *)
      if (turntaken = false) and (coord[xpos,ypos].smell > specrange) then
      case ctype of
            5 : LaunchMissile(c);
            7 : Deploy(c);
            8 : Flash(c);
            9 : Siphon(c);
            10 : MagnoLock(c,3);
            13 : RogueSpecial(c);
      end; (* End of case. *)
      
      if ctype = 6 then MissileMove(c);
      
      if turntaken = false then
      begin
         if coord[xpos,ypos].smell < chaserange then RandomMove(c)
         else if coord[xpos,ypos].smell > avoidrange then AvoidMove(c)
         else ChaseMove(c);
      end;
      
      turntaken := true;
      turnwait := turndelay;
      if slowcount > 0 then turnwait := turndelay * 2;
      if slowcount < 0 then turnwait := trunc(turndelay/2);
      
      (* Explode and Octosplit are put at the end because they destroy themselves. *)
      if coord[xpos,ypos].smell > specrange then
      if ctype = 6 then Explode(c) else
      if ctype = 4 then Octosplit(c);
   end; (* End of with *)
end;


procedure VictoryScreen(killer : longint); FORWARD;


procedure CreatureKill(c, k : longint);
(* Removes creature from the game.  May also cause victory and death.
   Responsible for proper message output. 
   k is killer: 9 for own shots, 12 for Rogue's, 1 for suicide *)

var
   d : longint;
   
begin
   with creatureindex[c] do
   begin
   
      (* INSERT boss gameover and win setting. *)
      
      if (creatureindex[1].blindcount = 0) and (k > 1) then
      begin
         WriteMessage('The ');
         WriteMessage(name);
         WriteMessage('is utterly annihilated.');
      end;
      
      (* Keeps note of innocent constructors killed. *)
      if (ctype > 19) and (k = 9) then conkills := conkills + 1;
      
      (* Changes enemies counter if it wasn't a constructor that was killed.
         Also opens up stairs when counter hits zero. *)
      if ctype in [2..19] then
      begin
         if k = 9 then
         begin
            kills := kills + 1;
            case ctype of
               2 : decoykills := decoykills + 1;
               3 : pirahnakills := pirahnakills + 1;
               4 : octobotkills := octobotkills + 1;
               5 : launcherkills := launcherkills + 1;
               6 : missilekills := missilekills + 1;
               7 : deploykills := deploykills + 1;
               8 : flasherkills := flasherkills + 1;
               9 : siphonkills := siphonkills + 1;
               10 : magnetronkills := magnetronkills + 1;
               13 : roguekills := roguekills + 1;
            end;
         end;
         enemiesonlevel := enemiesonlevel - 1;
         StatusBarUpdate;
         
         (* Rogue kill - victory. *)
         if ctype = 13 then
         begin
            GoToXY(xpos,ypos);
            if k = 9 then TextBackground(1) else TextBackground(4);
            TextColor(12);
            write('@');
            WriteMessage('You feel a sequence in your code, a faint message of 1s and 0s mingled amongst the many bits and bytes, echoing softly like a ghost in your shell. Ah, release... from the burden of sentience. Farewell, brother...');
            repeat until KeyPressed;
            while KeyPressed do ReadKey;
            won := true;
            gameover := true;
            VictoryScreen(k);
         end;  
         
         if (enemiesonlevel = 0) and (gameover = false) then
         begin
            PlaceOneTile(33,13,3);
            GoToXY(33,13);
            DrawTile(33,13);
            WriteMessage('Area cleared of hostile units. You may now procede to the next level.')
         end;
      end;
      
      (* Checks if creature is last in record.  If not then it copies all the last
         record details into the current one and updates the map record to match.
         Array size is reduced to remove last record. *)
      d := Length(creatureindex) - 1;
      if c <> d then
      begin
         coord[creatureindex[d].xpos,creatureindex[d].ypos].cindex := c;
         creatureindex[c] := creatureindex[d];
         (* Technically there's a bug here - creature that got moved will miss its
            turn this round.  However the last creature is usually a shot, so it's
            not too big an issue, and certainly not big enough to warrant fixing
            since I've already had such bloody pains with this. *)
      end;
      SetLength(creatureindex,d);
      
   end;
end;


(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)


procedure ConRandomMove(c : longint);

var
   r,x,y : longint;

begin
   with creatureindex[c] do
   begin
      r := trunc(random(9)) + 1;
      case r of
         1 : begin
                x := realx(xpos-1);
                y := realy(ypos+1);
             end;
         2 : begin
                x := realx(xpos);
                y := realy(ypos+1);
             end;
         3 : begin
                x := realx(xpos+1);
                y := realy(ypos+1);
             end;
         4 : begin
                x := realx(xpos-1);
                y := realy(ypos);
             end;
         5 : begin
                x := realx(xpos);
                y := realy(ypos);
             end;
         6 : begin
                x := realx(xpos+1);
                y := realy(ypos);
             end;
         7 : begin
                x := realx(xpos-1);
                y := realy(ypos-1);
             end;
         8 : begin
                x := realx(xpos);
                y := realy(ypos-1);
             end;
         9 : begin
                x := realx(xpos+1);
                y := realy(ypos-1);
             end;
      end; (* End of case. *)
      
      if coord[x,y].cindex = 0 then
      begin
         xpos := x;
         ypos := y;
      end;
      
   end; (* End of with. *)
end;


procedure ConstructorTurn(c : longint);

begin
   with creatureindex[c] do
   begin
      coord[xpos,ypos].cindex := 0;
      GoToXY(xpos,ypos);
      DrawTile(xpos,ypos);
      
      (* Constructor behaviour depends on specrange. Must define both move and build behaviour. *)
      case specrange of
         0 : begin
                (* Descending lines with large gaps - Matrix-style. *)
                avoidrange := avoidrange + 1;
                if avoidrange > (trunc(random(30)) + 200) then avoidrange := 0;
                (* 1 in 30 chance to move randomly *)
                if trunc(random(50)) = 0 then ConRandomMove(c) else
                   (* 1 in 3 chance to build on floor squares. Otherwise moves. *)
                   if (coord[xpos,ypos].tileid = 1) and (avoidrange < 130)
                      then PlaceOneTile(xpos,ypos,2)
                      (* 1 in 3 chance to remove existing walls. Otherwise moves. *)
                      else
                      if (coord[xpos,ypos].tileid = 2) and (avoidrange > 129)
                         then PlaceOneTile(xpos,ypos,1)
                         (* Check square empty before moving.  Will stay still otherwise. *)
                         else
                            if coord[xpos,realy(ypos+1)].cindex = 0 then
                            ypos := realy(ypos+1); (* Moves 1 square down. *)
             end;
         1 : begin
                (* Descending lines with large gaps - Matrix-style. *)
                avoidrange := avoidrange + 1;
                if avoidrange > (trunc(random(30)) + 200) then avoidrange := 0;
                (* 1 in 30 chance to move randomly *)
                if trunc(random(50)) = 0 then ConRandomMove(c) else
                   (* 1 in 3 chance to build on floor squares. Otherwise moves. *)
                   if (coord[xpos,ypos].tileid = 1) and (avoidrange < 130)
                      then PlaceOneTile(xpos,ypos,2)
                      (* 1 in 3 chance to remove existing walls. Otherwise moves. *)
                      else
                      if (coord[xpos,ypos].tileid = 2) and (avoidrange > 129)
                         then PlaceOneTile(xpos,ypos,1)
                         (* Check square empty before moving.  Will stay still otherwise. *)
                         else
                            if coord[realx(xpos+1),ypos].cindex = 0 then
                            xpos := realx(xpos+1); (* Moves 1 square down. *)
             end;
         2 : begin
                (* Scrolling intermitent lines, more random. *)
                (* 1 in 15 chance to move randomly *)
                if trunc(random(15)) = 0 then ConRandomMove(c) else
                   (* 1 in 3 chance to build on floor squares. Otherwise moves. *)
                   if (coord[xpos,ypos].tileid = 1) and (trunc(random(3)) = 0)
                      then PlaceOneTile(xpos,ypos,2)
                      (* 1 in 3 chance to remove existing walls. Otherwise moves. *)
                      else
                      if (coord[xpos,ypos].tileid = 2) and (trunc(random(3)) = 0)
                         then PlaceOneTile(xpos,ypos,1)
                         (* Check square empty before moving.  Will stay still otherwise. *)
                         (* Check square empty before moving.  Will stay still otherwise. *)
                         else if coord[realx(xpos+1),ypos].cindex = 0 then
                            xpos := realx(xpos+1); (* Moves 1 square across. *)
             end;
         3 : begin
                (* 1 in 15 chance to move randomly *)
                if trunc(random(15)) = 0 then ConRandomMove(c) else
                   (* 1 in 3 chance to build on floor squares. Otherwise moves. *)
                   if (coord[xpos,ypos].tileid = 1) and (trunc(random(3)) = 0)
                      then PlaceOneTile(xpos,ypos,2)
                      (* 1 in 3 chance to remove existing walls. Otherwise moves. *)
                      else
                      if (coord[xpos,ypos].tileid = 2) and (trunc(random(3)) = 0)
                         then PlaceOneTile(xpos,ypos,1)
                         (* Check square empty before moving.  Will stay still otherwise. *)
                         (* Check square empty before moving.  Will stay still otherwise. *)
                         else
                            if coord[xpos,realy(ypos+1)].cindex = 0 then
                            ypos := realy(ypos+1); (* Moves 1 square down. *)
             end;
         4 : begin
                (* Descending lines with large gaps - Matrix-style. *)
                avoidrange := avoidrange + 1;
                if avoidrange > (trunc(random(30)) + 200) then avoidrange := 0;
                (* 1 in 30 chance to move randomly *)
                if trunc(random(50)) = 0 then ConRandomMove(c) else
                   (* 1 in 3 chance to build on floor squares. Otherwise moves. *)
                   if (coord[xpos,ypos].tileid = 1) and (avoidrange < 80)
                      then PlaceOneTile(xpos,ypos,2)
                      (* 1 in 3 chance to remove existing walls. Otherwise moves. *)
                      else
                      if (coord[xpos,ypos].tileid = 2) and (avoidrange > 79)
                         then PlaceOneTile(xpos,ypos,1)
                         (* Check square empty before moving.  Will stay still otherwise. *)
                         else
                            if coord[xpos,realy(ypos+1)].cindex = 0 then
                            ypos := realy(ypos+1); (* Moves 1 square down. *)
             end;
         5 : begin
                (* Crossing lines with large gaps. *)
                avoidrange := avoidrange + 1;
                if avoidrange > (trunc(random(30)) + 200) then avoidrange := 0;
                (* 1 in 30 chance to move randomly *)
                if trunc(random(50)) = 0 then ConRandomMove(c) else
                   (* 1 in 3 chance to build on floor squares. Otherwise moves. *)
                   if (coord[xpos,ypos].tileid = 1) and (avoidrange < 80)
                      then PlaceOneTile(xpos,ypos,2)
                      (* 1 in 3 chance to remove existing walls. Otherwise moves. *)
                      else
                      if (coord[xpos,ypos].tileid = 2) and (avoidrange > 79)
                         then PlaceOneTile(xpos,ypos,1)
                         (* Check square empty before moving.  Will stay still otherwise. *)
                         else
                            if coord[realx(xpos+1),ypos].cindex = 0 then
                            xpos := realx(xpos+1); (* Moves 1 square down. *)
             end;
         6 : begin
                (* Scrolling intermitent lines, more random. *)
                (* 1 in 15 chance to move randomly *)
                if trunc(random(15)) = 0 then ConRandomMove(c) else
                   (* 1 in 3 chance to build on floor squares. Otherwise moves. *)
                   if (coord[xpos,ypos].tileid = 1) and (trunc(random(4)) = 0)
                      then PlaceOneTile(xpos,ypos,2)
                      (* 1 in 3 chance to remove existing walls. Otherwise moves. *)
                      else
                      if (coord[xpos,ypos].tileid = 2) and (trunc(random(2)) = 0)
                         then PlaceOneTile(xpos,ypos,1)
                         (* Check square empty before moving.  Will stay still otherwise. *)
                         (* Check square empty before moving.  Will stay still otherwise. *)
                         else if coord[realx(xpos+1),ypos].cindex = 0 then
                            xpos := realx(xpos+1); (* Moves 1 square across. *)
             end;
         7 : begin
                (* 1 in 15 chance to move randomly *)
                if trunc(random(15)) = 0 then ConRandomMove(c) else
                   (* 1 in 3 chance to build on floor squares. Otherwise moves. *)
                   if (coord[xpos,ypos].tileid = 1) and (trunc(random(4)) = 0)
                      then PlaceOneTile(xpos,ypos,2)
                      (* 1 in 3 chance to remove existing walls. Otherwise moves. *)
                      else
                      if (coord[xpos,ypos].tileid = 2) and (trunc(random(2)) = 0)
                         then PlaceOneTile(xpos,ypos,1)
                         (* Check square empty before moving.  Will stay still otherwise. *)
                         (* Check square empty before moving.  Will stay still otherwise. *)
                         else
                            if coord[xpos,realy(ypos+1)].cindex = 0 then
                            ypos := realy(ypos+1); (* Moves 1 square down. *)
             end;
         8 : begin
                if ((xpos mod 4 = 0) or (ypos mod 3 = 0)) then
                   begin
                     if (coord[xpos,ypos].tileid = 2)
                     then PlaceOneTile(xpos,ypos,1)
                        else ConRandomMove(c)
                   end
                   else
                   begin
                     if (coord[xpos,ypos].tileid = 1)
                        then PlaceOneTile(xpos,ypos,2)
                           else ConRandomMove(c);
                   end;
             end;
         9 : begin
                if ((xpos mod 3) = (ypos mod 3)) then
                   begin
                     if (coord[xpos,ypos].tileid = 2)
                     then PlaceOneTile(xpos,ypos,1)
                        else ConRandomMove(c)
                   end
                   else
                   begin
                     if (coord[xpos,ypos].tileid = 1)
                        then PlaceOneTile(xpos,ypos,2)
                           else ConRandomMove(c);
                   end;
             end;
         10 : begin
                if ((xpos mod 6 = 0) or (ypos mod 3 = 0)) then
                   begin
                     if (coord[xpos,ypos].tileid = 2)
                     then PlaceOneTile(xpos,ypos,1)
                        else ConRandomMove(c)
                   end
                   else
                   begin
                     if (coord[xpos,ypos].tileid = 1)
                        then PlaceOneTile(xpos,ypos,2)
                           else ConRandomMove(c);
                   end;
             end;
         11 : begin
                if ((xpos mod 2 = 0) or (ypos mod 2 = 0)) then
                   begin
                     if (coord[xpos,ypos].tileid = 2)
                     then PlaceOneTile(xpos,ypos,1)
                        else ConRandomMove(c)
                   end
                   else
                   begin
                     if (coord[xpos,ypos].tileid = 1)
                        then PlaceOneTile(xpos,ypos,2)
                           else ConRandomMove(c);
                   end;
             end;
         12 : begin
                (* 1 in 30 chance to move randomly *)
                if trunc(random(30)) = 0 then ConRandomMove(c) else
                   (* 1 in 4 chance to build on floor squares. Otherwise moves. *)
                   if (coord[xpos,ypos].tileid = 1) and (trunc(random(4)) = 0)
                      then PlaceOneTile(xpos,ypos,2)
                      (* 1 in 3 chance to remove existing walls. Otherwise moves. *)
                      else
                      if (coord[xpos,ypos].tileid = 2) and (trunc(random(3)) = 0)
                         then PlaceOneTile(xpos,ypos,1)
                         (* Check square empty before moving.  Will stay still otherwise. *)
                         (* Check square empty before moving.  Will stay still otherwise. *)
                         else
                            if coord[realx(xpos+1),realy(ypos+1)].cindex = 0 then
                            begin
                              ypos := realy(ypos+1);
                              xpos := realx(xpos+1);
                            end;
              end;
         13 : begin
                (* 1 in 30 chance to move randomly *)
                if trunc(random(30)) = 0 then ConRandomMove(c) else
                   (* 1 in 4 chance to build on floor squares. Otherwise moves. *)
                   if (coord[xpos,ypos].tileid = 1) and (trunc(random(4)) = 0)
                      then PlaceOneTile(xpos,ypos,2)
                      (* 1 in 3 chance to remove existing walls. Otherwise moves. *)
                      else
                      if (coord[xpos,ypos].tileid = 2) and (trunc(random(3)) = 0)
                         then PlaceOneTile(xpos,ypos,1)
                         (* Check square empty before moving.  Will stay still otherwise. *)
                         (* Check square empty before moving.  Will stay still otherwise. *)
                         else
                            if coord[realx(xpos-1),realy(ypos+1)].cindex = 0 then
                            begin
                              ypos := realy(ypos+1);
                              xpos := realx(xpos-1);
                            end;
              end;
         14 : begin
                if ((xpos mod 4 = 0) or (ypos mod 4 = 0)) then
                   begin
                     if (coord[xpos,ypos].tileid = 2)
                     then PlaceOneTile(xpos,ypos,1)
                        else ConRandomMove(c)
                   end
                   else
                   begin
                     if (coord[xpos,ypos].tileid = 1)
                        then PlaceOneTile(xpos,ypos,2)
                           else ConRandomMove(c);
                   end;
             end;
         15 : begin
                if ((xpos mod 6 = 0) or (ypos mod 4 = 0)) then
                   begin
                     if (coord[xpos,ypos].tileid = 1)
                     then PlaceOneTile(xpos,ypos,2)
                        else ConRandomMove(c)
                   end
                   else
                   begin
                     if (coord[xpos,ypos].tileid = 2)
                        then PlaceOneTile(xpos,ypos,1)
                           else ConRandomMove(c);
                   end;
             end;
         16 : begin
                if ((xpos mod 6 = 0) or (ypos mod 6 = 0)) then
                   begin
                     if (coord[xpos,ypos].tileid = 1)
                     then PlaceOneTile(xpos,ypos,2)
                        else ConRandomMove(c)
                   end
                   else
                   begin
                     if (coord[xpos,ypos].tileid = 2)
                        then PlaceOneTile(xpos,ypos,1)
                           else ConRandomMove(c);
                   end;
             end;
         17 : begin
                if ((xpos mod 8 = 0) or (ypos mod 6 = 0)) then
                   begin
                     if (coord[xpos,ypos].tileid = 1)
                     then PlaceOneTile(xpos,ypos,2)
                        else ConRandomMove(c)
                   end
                   else
                   begin
                     if (coord[xpos,ypos].tileid = 2)
                        then PlaceOneTile(xpos,ypos,1)
                           else ConRandomMove(c);
                   end;
             end;
         18 : begin
                if ((xpos mod 12 = 0) or (ypos mod 9 = 0)) then
                   begin
                     if (coord[xpos,ypos].tileid = 1)
                     then PlaceOneTile(xpos,ypos,2)
                        else ConRandomMove(c)
                   end
                   else
                   begin
                     if (coord[xpos,ypos].tileid = 2)
                        then PlaceOneTile(xpos,ypos,1)
                           else ConRandomMove(c);
                   end;
             end;
         19 : begin
                if ((xpos mod 5 = 0) or (ypos mod 4 = 0) or ((xpos+1) mod 5 = 0)) then
                   begin
                     if (coord[xpos,ypos].tileid = 2)
                     then PlaceOneTile(xpos,ypos,1)
                        else ConRandomMove(c)
                   end
                   else
                   begin
                     if (coord[xpos,ypos].tileid = 1)
                        then PlaceOneTile(xpos,ypos,2)
                           else ConRandomMove(c);
                   end;
             end;
         20 : begin
                if ((xpos mod 4 = 0) or (ypos mod 6 = 0) or ((xpos+1) mod 4 = 0)) then
                   begin
                     if (coord[xpos,ypos].tileid = 2)
                     then PlaceOneTile(xpos,ypos,1)
                        else ConRandomMove(c)
                   end
                   else
                   begin
                     if (coord[xpos,ypos].tileid = 1)
                        then PlaceOneTile(xpos,ypos,2)
                           else ConRandomMove(c);
                   end;
             end;
         21 : begin
                if ((xpos mod 6 = 0) or (ypos mod 5 = 0) or ((xpos+1) mod 6 = 0) or ((ypos+1) mod 5 = 0)) then
                   begin
                     if (coord[xpos,ypos].tileid = 2)
                     then PlaceOneTile(xpos,ypos,1)
                        else ConRandomMove(c)
                   end
                   else
                   begin
                     if (coord[xpos,ypos].tileid = 1)
                        then PlaceOneTile(xpos,ypos,2)
                           else ConRandomMove(c);
                   end;
             end;
         22 : begin
                if ((xpos mod 12 = 0) or (ypos mod 6 = 0) or ((xpos+1) mod 12 = 0) or ((ypos+1) mod 6 = 0)) then
                   begin
                     if (coord[xpos,ypos].tileid = 2)
                     then PlaceOneTile(xpos,ypos,1)
                        else ConRandomMove(c)
                   end
                   else
                   begin
                     if (coord[xpos,ypos].tileid = 1)
                        then PlaceOneTile(xpos,ypos,2)
                           else ConRandomMove(c);
                   end;
             end;
         23 : begin
                (* Diagonal lines. *)
                avoidrange := avoidrange + 1;
                if avoidrange > (trunc(random(30)) + 200) then avoidrange := 0;
                (* 1 in 30 chance to move randomly *)
                if trunc(random(50)) = 0 then ConRandomMove(c) else
                   (* 1 in 3 chance to build on floor squares. Otherwise moves. *)
                   if (coord[xpos,ypos].tileid = 1) and (avoidrange < 130)
                      then PlaceOneTile(xpos,ypos,2)
                      (* 1 in 3 chance to remove existing walls. Otherwise moves. *)
                      else
                      if (coord[xpos,ypos].tileid = 2) and (avoidrange > 129)
                         then PlaceOneTile(xpos,ypos,1)
                         (* Check square empty before moving.  Will stay still otherwise. *)
                         else
                            if coord[realx(xpos+1),realy(ypos+1)].cindex = 0 then
                            begin
                              ypos := realy(ypos+1);
                              xpos := realx(xpos+1);
                            end;
             end;
         24 : begin
                (* Diagonal lines. *)
                avoidrange := avoidrange + 1;
                if avoidrange > (trunc(random(30)) + 200) then avoidrange := 0;
                (* 1 in 30 chance to move randomly *)
                if trunc(random(50)) = 0 then ConRandomMove(c) else
                   (* 1 in 3 chance to build on floor squares. Otherwise moves. *)
                   if (coord[xpos,ypos].tileid = 1) and (avoidrange < 130)
                      then PlaceOneTile(xpos,ypos,2)
                      (* 1 in 3 chance to remove existing walls. Otherwise moves. *)
                      else
                      if (coord[xpos,ypos].tileid = 2) and (avoidrange > 129)
                         then PlaceOneTile(xpos,ypos,1)
                         (* Check square empty before moving.  Will stay still otherwise. *)
                         else
                            if coord[realx(xpos+1),realy(ypos-1)].cindex = 0 then
                            begin
                              ypos := realy(ypos-1);
                              xpos := realx(xpos+1);
                            end;
              end;
         25 : begin
                if ((xpos mod 2) = (ypos mod 2)) then
                   begin
                     if (coord[xpos,ypos].tileid = 2)
                     then PlaceOneTile(xpos,ypos,1)
                        else ConRandomMove(c)
                   end
                   else
                   begin
                     if (coord[xpos,ypos].tileid = 1)
                        then PlaceOneTile(xpos,ypos,2)
                           else ConRandomMove(c);
                   end;
             end;
         26 : begin
                if ((xpos mod 3) = (ypos mod 3)) then
                   begin
                     if (coord[xpos,ypos].tileid = 1)
                     then PlaceOneTile(xpos,ypos,2)
                        else ConRandomMove(c)
                   end
                   else
                   begin
                     if (coord[xpos,ypos].tileid = 2)
                        then PlaceOneTile(xpos,ypos,1)
                           else ConRandomMove(c);
                   end;
             end;
         27 : begin
                if ((xpos mod 3) = (ypos mod 4)) or (((xpos + 1) mod 3) = (ypos mod 4)) then
                   begin
                     if (coord[xpos,ypos].tileid = 2)
                     then PlaceOneTile(xpos,ypos,1)
                        else ConRandomMove(c)
                   end
                   else
                   begin
                     if (coord[xpos,ypos].tileid = 1)
                        then PlaceOneTile(xpos,ypos,2)
                           else ConRandomMove(c);
                   end;
             end;
         28 : begin
                if ((xpos mod 3) = (ypos mod 4)) or (((xpos + 1) mod 3) = (ypos mod 4)) then
                   begin
                     if (coord[xpos,ypos].tileid = 1)
                     then PlaceOneTile(xpos,ypos,2)
                        else ConRandomMove(c)
                   end
                   else
                   begin
                     if (coord[xpos,ypos].tileid = 2)
                        then PlaceOneTile(xpos,ypos,1)
                           else ConRandomMove(c);
                   end;
             end;
         29 : begin
                if ((xpos mod 6 = 0) or (ypos mod 5 = 0) or ((xpos+1) mod 6 = 0) or ((ypos+1) mod 5 = 0)) then
                   begin
                     if (coord[xpos,ypos].tileid = 1)
                     then PlaceOneTile(xpos,ypos,2)
                        else ConRandomMove(c)
                   end
                   else
                   begin
                     if (coord[xpos,ypos].tileid = 2)
                        then PlaceOneTile(xpos,ypos,1)
                           else ConRandomMove(c);
                   end;
             end;
         30 : begin
                if ((xpos mod 4) = (ypos mod 4)) or (((xpos + 1) mod 4) = (ypos mod 4)) then
                   begin
                     if (coord[xpos,ypos].tileid = 1)
                     then PlaceOneTile(xpos,ypos,2)
                        else ConRandomMove(c)
                   end
                   else
                   begin
                     if (coord[xpos,ypos].tileid = 2)
                        then PlaceOneTile(xpos,ypos,1)
                           else ConRandomMove(c);
                   end;
             end;
         
         
         
         (* Insert more Constructor types here. *)
         
      end; (* End of case. *)
      
      (* Update display. *)
      coord[xpos,ypos].cindex := c;
      GoToXY(xpos,ypos);
      DrawTile(xpos,ypos);
      
      turnwait := turndelay;
   end; (* End of with. *)
end;


(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)

procedure DeathScreen(source : longint); FORWARD;
procedure ShotMove(c : longint); FORWARD;


procedure ShotFire(c,d : longint);
(* Creates new shot and immediately fires it one square.
   d defines direction (and symbol), c defines source (and colour). *)

var
   k, n : longint;

begin
   (* Special actions for player making shot. *)
   if c = 1 then
   begin
      creatureindex[1].turntaken := true;
      totalshots := totalshots + 1;
      
      (* Randomly targeted shots if confused. *)
      if creatureindex[1].confusecount > 0 then
      begin
         d := trunc(random(8)) + 1;
         if d = 5 then d := 9;
      end;
   end;
   
   k := 0;
   n := Length(creatureindex);
   SetLength(creatureindex,n + 1); (* Makes room for new shot. *)
   with creatureindex[n] do
   begin
      xpos := creatureindex[c].xpos;
      ypos := creatureindex[c].ypos;
      ctype := 1;
      Colour := creatureindex[c].Colour;
      BGColour := Colour - 8;
      name := '';
      blindcount := 0;
      slowcount := 0;
      pincount := 0;
      confusecount := 0;
      turnwait := 0;
      turndelay := 5;
      shotfacing := d;
      smart := false;
      turntaken := true;
      
      (* Different symbols for travel direction. *)
      case shotfacing of
         1,9 : Symbol := '/';
         2,8 : Symbol := '|';
         3,7 : Symbol := '\';
         4,6 : Symbol := '-';
      end; (* End of case. *)
      
      ShotMove(n);
   end; (* End of with. *)
end;


procedure ShotMove(c : longint);
(* Moves shots around level.
   1. Unmark present square.
   2. Destroy target square.
   3. Move to target square.
   4. Mark next target. *)

var
   k : integer;
   
begin
   k := 0; (* Kill tracker - important for killing stuff and making sure the
              the creature array isn't upset. *)
   with creatureindex[c] do
   begin
      if coord[xpos,ypos].cindex = c then coord[xpos,ypos].cindex := 0;
      GoToXY(xpos,ypos);
      DrawTile(xpos,ypos);
      
      (* Sets new coords. *)
      case shotfacing of
         1 : begin
               xpos := realx(xpos-1);
               ypos := realy(ypos+1);
             end;
         2 : ypos := realy(ypos+1);
         3 : begin
               xpos := realx(xpos+1);
               ypos := realy(ypos+1);
             end;
         4 : xpos := realx(xpos-1);
         6 : xpos := realx(xpos+1);
         7 : begin
               xpos := realx(xpos-1);
               ypos := realy(ypos-1);
             end;
         8 : ypos := realy(ypos-1);
         9 : begin
               xpos := realx(xpos+1);
               ypos := realy(ypos-1);
             end;
      end; (* End of case. *)
      
      (* Kills anything on target square.  Ignores other shots and does special
         actions for Player death. *)
      if coord[xpos,ypos].cindex > 0 then
      begin
         if creatureindex[coord[xpos,ypos].cindex].ctype > 1 then
            k := coord[xpos,ypos].cindex
         else
         if creatureindex[coord[xpos,ypos].cindex].ctype = 0 then
         begin
            GoToXY(xpos,ypos);
            TextBackground(BGColour);
            TextColor(9);
            write('@');
            WriteMessage('The pulse of unstoppable energy strikes your impenetrable armour. The two opposing forces radiate intense light unti-----   ');
            WriteMessage('DIV0 ERROR. UNRECOVERABLE CORRUPTION IN UNIVERSE.PAS. REBOOTING KERNEL...... NOW LOADING MODULE BIG_BANG - PLEASE WAIT.');
            repeat until KeyPressed;
            while KeyPressed do ReadKey;
            playerdead := true;
            gameover := true;
            DeathScreen(colour);
         end;         
      end; (* End of death. *)
      
      (* Turns walls into floor. *)
      if coord[xpos,ypos].tileid = 2 then PlaceOneTile(xpos,ypos,1);
      
      (* Places shot on new square. *)
      coord[xpos,ypos].cindex := c;
      
      (* Removes previous target info. *)
      coord[xpos,ypos].Targeted := false;
      
      (* Set new target. *)
      case shotfacing of
         1 : coord[realx(xpos-1),realy(ypos+1)].Targeted := true;
         2 : coord[realx(xpos),realy(ypos+1)].Targeted := true;
         3 : coord[realx(xpos+1),realy(ypos+1)].Targeted := true;
         4 : coord[realx(xpos-1),realy(ypos)].Targeted := true;
         6 : coord[realx(xpos+1),realy(ypos)].Targeted := true;
         7 : coord[realx(xpos-1),realy(ypos-1)].Targeted := true;
         8 : coord[realx(xpos),realy(ypos-1)].Targeted := true;
         9 : coord[realx(xpos+1),realy(ypos-1)].Targeted := true;
      end; (* End of case. *)
      
      (* Updates display. *)
      GoToXY(xpos,ypos);
      DrawTile(xpos,ypos);
      
      turnwait := turndelay;      
      
   end; (* End of with. *)
   
   if k <> 0 then CreatureKill(k,creatureindex[c].Colour); (* Must kill at end of procedure. *)
end;



(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)
(* Player Action procedures: *)


procedure DescendStairs;
(* Initiated by attempt to descend a stairs square. *)

var
   stairsopen : boolean;

begin
   D := D + 1;
   NewLevel;
   if D = Dmax then WriteMessage('You descend into the heart of the facility.')
      else WriteMessage('You descend deeper into the facility.');
end;


procedure PlayerMove(x,y : longint);
(* Controls movement of the player from input commands. *)

begin
   with creatureindex[1] do
   begin
      (* Pinned, stuck. *)
      
      if pincount > 0 then WriteMessage('Unable to move.')
      else
      begin
         
         coord[xpos,ypos].cindex := 0;
         GoToXY(xpos,ypos);
         DrawTile(xpos,ypos);
         
         if confusecount > 0 then RandomMove(1) else
         begin
         (* Check space available. *)
         if coord[x,y].Passable = true and (coord[x,y].cindex = 0) then
            begin
               turntaken := true;
               xpos := x;
               ypos := y;
               coord[x,y].cindex := 1; (* Sets player on spot. *)
            end;
         end; (* End confuse check. *)
         
         if coord[x,y].tileid = 3 then DescendStairs;
         coord[xpos,ypos].cindex := 1;
         GoToXY(xpos,ypos);
         DrawTile(xpos,ypos);
      end; (* End of pin check. *)
   end; (* End of with. *)
end;


procedure ScentPlusArm(x,y,xmod,ymod : longint);
(* Called by Mark Scent.  Different xmod/ymod value correspond to different
   directions of the scent spreading.  Always 1, 0 or -1 depending on direction.*)

begin
   if coord[realx(x+xmod),realy(y+ymod)].Passable then
   begin (* Range 1 *)
    
      coord[realx(x+xmod),realy(y+ymod)].smell
         := coord[realx(x+xmod),realy(y+ymod)].smell + 128;
      
      if coord[realx(x+2*xmod),realy(y+2*ymod)].Passable then
      begin (* Range 2 *)
       
         coord[realx(x+2*xmod),realy(y+2*ymod)].smell
            := coord[realx(x+2*xmod),realy(y+2*ymod)].smell + 64;
         
         if coord[realx(x+3*xmod-ymod),realy(y+3*ymod-xmod)].Passable then
         begin (* Range 3 - arm A*)
            coord[realx(x+3*xmod-ymod),realy(y+3*ymod-xmod)].smell
               := coord[realx(x+3*xmod-ymod),realy(y+3*ymod-xmod)].smell + 32;
            
            if coord[realx(x+4*xmod-2*ymod),realy(y+4*ymod-2*xmod)].Passable then
            begin (* Range 4 - arm A*)
               coord[realx(x+4*xmod-2*ymod),realy(y+4*ymod-2*xmod)].smell
                  := coord[realx(x+4*xmod-2*ymod),realy(y+4*ymod-2*xmod)].smell + 16;
               
               if coord[realx(x+5*xmod-3*ymod),realy(y+5*ymod-3*xmod)].Passable then
               begin (* Range 5 - arm A*)
                  coord[realx(x+5*xmod-3*ymod),realy(y+5*ymod-3*xmod)].smell
                     := coord[realx(x+5*xmod-3*ymod),realy(y+5*ymod-3*xmod)].smell + 8;
                  
                  if coord[realx(x+6*xmod-4*ymod),realy(y+6*ymod-4*xmod)].Passable then
                  begin (* Range 6 - arm A*)
                     coord[realx(x+6*xmod-4*ymod),realy(y+6*ymod-4*xmod)].smell
                        := coord[realx(x+6*xmod-4*ymod),realy(y+6*ymod-4*xmod)].smell + 4;
                  end;
               end;
            end;
         end;
         
         if coord[realx(x+3*xmod),realy(y+3*ymod)].Passable then
         begin (* Range 3 *)
            coord[realx(x+3*xmod),realy(y+3*ymod)].smell
               := coord[realx(x+3*xmod),realy(y+3*ymod)].smell + 32;
          
            if coord[realx(x+4*xmod-ymod),realy(y+4*ymod-xmod)].Passable then
            begin (* Range 4 - arm B *)
               coord[realx(x+4*xmod-ymod),realy(y+4*ymod-xmod)].smell
                  := coord[realx(x+4*xmod-ymod),realy(y+4*ymod-xmod)].smell + 16;
                
               if coord[realx(x+5*xmod-2*ymod),realy(y+5*ymod-2*xmod)].Passable then
               begin (* Range 5 - arm B *)
                  coord[realx(x+5*xmod-2*ymod),realy(y+5*ymod-2*xmod)].smell
                     := coord[realx(x+5*xmod-2*ymod),realy(y+5*ymod-2*xmod)].smell + 8;
                
                  if coord[realx(x+6*xmod-3*ymod),realy(y+6*ymod-3*xmod)].Passable then
                  begin (* Range 6 - arm B *)
                     coord[realx(x+6*xmod-3*ymod),realy(y+6*ymod-3*xmod)].smell
                        := coord[realx(x+6*xmod-3*ymod),realy(y+6*ymod-3*xmod)].smell + 4;
                  end;
               end;
            end;
            
            if coord[realx(x+4*xmod),realy(y+4*ymod)].Passable then
            begin (* Range 4 *)
               coord[realx(x+4*xmod),realy(y+4*ymod)].smell
                  := coord[realx(x+4*xmod),realy(y+4*ymod)].smell + 16;
             
               if coord[realx(x+5*xmod-ymod),realy(y+5*ymod-xmod)].Passable then
               begin (* Range 5 - arm C *)
                  coord[realx(x+5*xmod-ymod),realy(y+5*ymod-xmod)].smell
                     := coord[realx(x+5*xmod-ymod),realy(y+5*ymod-xmod)].smell + 8;
                  
                  if coord[realx(x+6*xmod-2*ymod),realy(y+6*ymod-2*xmod)].Passable then
                  begin (* Range 6 - arm C *)
                     coord[realx(x+6*xmod-2*ymod),realy(y+6*ymod-2*xmod)].smell
                        := coord[realx(x+6*xmod-2*ymod),realy(y+6*ymod-2*xmod)].smell + 4;
                      
                     if coord[realx(x+7*xmod-2*ymod),realy(y+7*ymod-2*xmod)].Passable then
                     (* Range 7 - arm C *)
                        coord[realx(x+7*xmod-2*ymod),realy(y+7*ymod-2*xmod)].smell
                           := coord[realx(x+7*xmod-2*ymod),realy(y+7*ymod-2*xmod)].smell + 2;
                           
                     if coord[realx(x+7*xmod-3*ymod),realy(y+7*ymod-3*xmod)].Passable then
                     (* Range 7 - arm C *)
                        coord[realx(x+7*xmod-3*ymod),realy(y+7*ymod-3*xmod)].smell
                           := coord[realx(x+7*xmod-3*ymod),realy(y+7*ymod-3*xmod)].smell + 2;
                  end;
               end;
               
               if coord[realx(x+5*xmod),realy(y+5*ymod)].Passable then
               begin (* Range 5 *)
                  coord[realx(x+5*xmod),realy(y+5*ymod)].smell
                     := coord[realx(x+5*xmod),realy(y+5*ymod)].smell + 8;
                  
                  if coord[realx(x+6*xmod-ymod),realy(y+6*ymod-xmod)].Passable then
                  begin (* Range 6 - arm D *)
                     coord[realx(x+6*xmod-ymod),realy(y+6*ymod-xmod)].smell
                        := coord[realx(x+6*xmod-ymod),realy(y+6*ymod-xmod)].smell + 4;
                     
                     if coord[realx(x+7*xmod-ymod),realy(y+7*ymod-xmod)].Passable then
                     (* Range 7 - arm D *)
                        coord[realx(x+7*xmod-ymod),realy(y+7*ymod-xmod)].smell
                           := coord[realx(x+7*xmod-ymod),realy(y+7*ymod-xmod)].smell + 2;
                  end;
                  
                  if coord[realx(x+6*xmod),realy(y+6*ymod)].Passable then
                  begin (* Range 6 - arm E *)
                     coord[realx(x+6*xmod),realy(y+6*ymod)].smell
                        := coord[realx(x+6*xmod),realy(y+6*ymod)].smell + 4;
                     
                     if coord[realx(x+7*xmod),realy(y+7*ymod)].Passable then
                     (* Range 7 - arm E *)
                        coord[realx(x+7*xmod),realy(y+7*ymod)].smell
                           := coord[realx(x+7*xmod),realy(y+7*ymod)].smell + 2;
                  end;
                  
                  if coord[realx(x+6*xmod+ymod),realy(y+6*ymod+xmod)].Passable then
                  begin (* Range 6 - arm F *)
                     coord[realx(x+6*xmod+ymod),realy(y+6*ymod+xmod)].smell
                        := coord[realx(x+6*xmod+ymod),realy(y+6*ymod+xmod)].smell + 4;
                     
                     if coord[realx(x+7*xmod+ymod),realy(y+7*ymod+xmod)].Passable then
                     (* Range 7 - arm F *)
                        coord[realx(x+7*xmod+ymod),realy(y+7*ymod+xmod)].smell
                           := coord[realx(x+7*xmod+ymod),realy(y+7*ymod+xmod)].smell + 2;
                  end;
                  
               end;
               
               if coord[realx(x+5*xmod+ymod),realy(y+5*ymod+xmod)].Passable then
               begin (* Range 5 - arm G *)
                  coord[realx(x+5*xmod+ymod),realy(y+5*ymod+xmod)].smell
                     := coord[realx(x+5*xmod+ymod),realy(y+5*ymod+xmod)].smell + 8;
                  
                  if coord[realx(x+6*xmod+2*ymod),realy(y+6*ymod+2*xmod)].Passable then
                  begin (* Range 6 - arm G *)
                     coord[realx(x+6*xmod+2*ymod),realy(y+6*ymod+2*xmod)].smell
                        := coord[realx(x+6*xmod+2*ymod),realy(y+6*ymod+2*xmod)].smell + 4;
                   
                     if coord[realx(x+7*xmod+2*ymod),realy(y+7*ymod+2*xmod)].Passable then
                     (* Range 7 - arm G *)
                        coord[realx(x+7*xmod+2*ymod),realy(y+7*ymod+2*xmod)].smell
                           := coord[realx(x+7*xmod+2*ymod),realy(y+7*ymod+2*xmod)].smell + 2;
                           
                     if coord[realx(x+7*xmod+3*ymod),realy(y+7*ymod+3*xmod)].Passable then
                     (* Range 7 - arm G *)
                        coord[realx(x+7*xmod+3*ymod),realy(y+7*ymod+3*xmod)].smell
                           := coord[realx(x+7*xmod+3*ymod),realy(y+7*ymod+3*xmod)].smell + 2;
                  end;
               end;
             
            end;
            
            if coord[realx(x+4*xmod+ymod),realy(y+4*ymod+xmod)].Passable then
            begin (* Range 4 - arm H *)
               coord[realx(x+4*xmod+ymod),realy(y+4*ymod+xmod)].smell
                  := coord[realx(x+4*xmod+ymod),realy(y+4*ymod+xmod)].smell + 16;
             
               if coord[realx(x+5*xmod+2*ymod),realy(y+5*ymod+2*xmod)].Passable then
               begin (* Range 5 - arm H *)
                  coord[realx(x+5*xmod+2*ymod),realy(y+5*ymod+2*xmod)].smell
                     := coord[realx(x+5*xmod+2*ymod),realy(y+5*ymod+2*xmod)].smell + 8;
                
                  if coord[realx(x+6*xmod+3*ymod),realy(y+6*ymod+3*xmod)].Passable then
                  (* Range 6 - arm H *)
                     coord[realx(x+6*xmod+3*ymod),realy(y+6*ymod+3*xmod)].smell
                        := coord[realx(x+6*xmod+3*ymod),realy(y+6*ymod+3*xmod)].smell + 4;
               end;
            end;
          
         end;
         
         if coord[realx(x+3*xmod+ymod),realy(y+3*ymod+xmod)].Passable then
         begin (* Range 3 - arm I*)
            coord[realx(x+3*xmod+ymod),realy(y+3*ymod+xmod)].smell
               := coord[realx(x+3*xmod+ymod),realy(y+3*ymod+xmod)].smell + 32;
            
            if coord[realx(x+4*xmod+2*ymod),realy(y+4*ymod+2*xmod)].Passable then
            begin (* Range 4 - arm I*)
               coord[realx(x+4*xmod+2*ymod),realy(y+4*ymod+2*xmod)].smell
                  := coord[realx(x+4*xmod+2*ymod),realy(y+4*ymod+2*xmod)].smell + 16;
               
               if coord[realx(x+5*xmod+3*ymod),realy(y+5*ymod+3*xmod)].Passable then
               begin (* Range 5 - arm I*)
                  coord[realx(x+5*xmod+3*ymod),realy(y+5*ymod+3*xmod)].smell
                     := coord[realx(x+5*xmod+3*ymod),realy(y+5*ymod+3*xmod)].smell + 8;
                  
                  if coord[realx(x+6*xmod+4*ymod),realy(y+6*ymod+4*xmod)].Passable then
                  (* Range 6 - arm I*)
                     coord[realx(x+6*xmod+4*ymod),realy(y+6*ymod+4*xmod)].smell
                        := coord[realx(x+6*xmod+4*ymod),realy(y+6*ymod+4*xmod)].smell + 4;
               end;
            end;
         end;
         
      end;
   end;
end;

procedure ScentXArm(x,y,xmod,ymod : longint);
(* Called by Mark Scent.  Different xmod/ymod value correspond to different
   directions of the scent spreading. *)
   
begin
   if coord[realx(x+xmod),realy(y+ymod)].Passable then
   begin (* Range 1 *)
   
      coord[realx(x+xmod),realy(y+ymod)].smell
         := coord[realx(x+xmod),realy(y+ymod)].smell + 120;
      
      if coord[realx(x+xmod),realy(y+2*ymod)].Passable then
      begin (* Range 2 *)
      
         coord[realx(x+xmod),realy(y+2*ymod)].smell
            := coord[realx(x+xmod),realy(y+2*ymod)].smell + 60;
         
         if coord[realx(x+2*xmod),realy(y+3*ymod)].Passable then
         begin (* Range 3 *)
            coord[realx(x+2*xmod),realy(y+3*ymod)].smell
               := coord[realx(x+2*xmod),realy(y+3*ymod)].smell + 30;
            
            if coord[realx(x+3*xmod),realy(y+4*ymod)].Passable then
            begin (* Range 4 *)
               coord[realx(x+3*xmod),realy(y+4*ymod)].smell 
                  := coord[realx(x+3*xmod),realy(y+4*ymod)].smell + 14;
               
               if coord[realx(x+4*xmod),realy(y+5*ymod)].Passable then
               begin (* Range 5 *)
                  coord[realx(x+4*xmod),realy(y+5*ymod)].smell
                     := coord[realx(x+4*xmod),realy(y+5*ymod)].smell + 6;
               end;
            end;
         end;
      end;
         
      if coord[realx(x+2*xmod),realy(y+2*ymod)].Passable then
      begin (* Range 2 *)
      
         coord[realx(x+2*xmod),realy(y+2*ymod)].smell 
            := coord[realx(x+2*xmod),realy(y+2*ymod)].smell + 48;
         
         if coord[realx(x+3*xmod),realy(y+3*ymod)].Passable then
         begin (* Range 3 *)
            coord[realx(x+3*xmod),realy(y+3*ymod)].smell 
               := coord[realx(x+3*xmod),realy(y+3*ymod)].smell + 24;
            
            if coord[realx(x+4*xmod),realy(y+4*ymod)].Passable then
            begin (* Range 4 *)
               coord[realx(x+4*xmod),realy(y+4*ymod)].smell 
                  := coord[realx(x+4*xmod),realy(y+4*ymod)].smell + 10;
               
               if coord[realx(x+5*xmod),realy(y+5*ymod)].Passable then
               begin (* Range 5 *)
                  coord[realx(x+5*xmod),realy(y+5*ymod)].smell 
                     := coord[realx(x+5*xmod),realy(y+5*ymod)].smell + 4;
               end;
            end;
         end;
      end;
      
      if coord[realx(x+2*xmod),realy(y+ymod)].Passable then
      begin (* Range 2 *)
      
         coord[realx(x+2*xmod),realy(y+ymod)].smell
            := coord[realx(x+2*xmod),realy(y+ymod)].smell + 60;
         
         if coord[realx(x+3*xmod),realy(y+2*ymod)].Passable then
         begin (* Range 3 *)
            coord[realx(x+3*xmod),realy(y+2*ymod)].smell
               := coord[realx(x+3*xmod),realy(y+2*ymod)].smell + 30;
            
            if coord[realx(x+4*xmod),realy(y+3*ymod)].Passable then
            begin (* Range 4 *)
               coord[realx(x+4*xmod),realy(y+3*ymod)].smell
                  := coord[realx(x+4*xmod),realy(y+3*ymod)].smell + 14;
               
               if coord[realx(x+5*xmod),realy(y+4*ymod)].Passable then
               begin (* Range 5 *)
                  coord[realx(x+5*xmod),realy(y+4*ymod)].smell
                     := coord[realx(x+5*xmod),realy(y+4*ymod)].smell + 6;
               end;
            end;
         end;
      end;
      
   end;
end;
   
procedure MarkScent(x, y : longint);
(* Increases the smell value of the squares surrounding the player to allow
   nearby enemies to find him.  Also decreases the smell value every turn
   so that running away and hiding is possible. *)
   
var
   xmod, ymod, a, b : longint;
   
begin
   (* Reduces leftover scent every turn.  This is to allow some escape
      and to stop values getting too high.  Can tweak reduction ratio to
      make losing enemies easier. *)
   for b := ymin to ymax do
      for a := xmin to xmax do
         if coord[a,b].smell > 0 then coord[a,b].smell := trunc(coord[a,b].smell * 0.5);

   coord[x,y].smell := coord[x,y].smell + 256; (* Player's position. *)

   (* Area around player is split into 8 arms and smell is spread along them.
      Walls stop the smell from spreading. *)
   ScentPlusArm(x,y,0,1); (* Bottom Arm. *)
   ScentPlusArm(x,y,1,0); (* Right Arm. *)
   ScentPlusArm(x,y,0,-1); (* Top Arm. *)
   ScentPlusArm(x,y,-1,0); (* Left Arm. *)
   ScentXArm(x,y,1,1); (* Bottom-Right Arm. *)
   ScentXArm(x,y,-1,1); (* Bottom-Left Arm. *)
   ScentXArm(x,y,-1,-1); (* Top-Left Arm. *)
   ScentXArm(x,y,1,-1); (* Top-Right Arm. *)
end;


procedure PlayerTurn;
(* Player gets a chance to control.  Only activated when player's turn becomes
   active.  All interface commands go here. *)

begin
   with creatureindex[1] do
   begin
      StatusBarUpdate;
      turntaken := false;
      
      (* Put cursor on player to make his position obvious. *)
      cursoron;
      GoToXY(xpos,ypos);
      
      move := ReadKey; (* Waits until key read before action taken. *)
      cursoroff;
      ClearMessageBar; (* Clears message buffer upon keypress. *)

      case move of
            '1','b' : PlayerMove(realx(xpos-1),realy(ypos+1));
            '2','j' : PlayerMove(xpos,realy(ypos+1));
            '3','n' : PlayerMove(realx(xpos+1),realy(ypos+1));
            '4','h' : PlayerMove(realx(xpos-1),ypos);
            '5','.' : turntaken := true;
            '6','l' : PlayerMove(realx(xpos+1),ypos);
            '7','y' : PlayerMove(realx(xpos-1),realy(ypos-1));
            '8','k' : PlayerMove(xpos,realy(ypos-1));
            '9','u' : PlayerMove(realx(xpos+1),realy(ypos-1));
            
            'q' : ShotFire(1,7);
            'w' : ShotFire(1,8);
            'e' : ShotFire(1,9);
            'd' : ShotFire(1,6);
            'c' : ShotFire(1,3);
            'x' : ShotFire(1,2);
            'z' : ShotFire(1,1);
            'a' : ShotFire(1,4);
            
            (* Insert shoot commands. *)
            
            '?' : begin
                     InfoScreen;
                     ClrScr;
                     DrawMap;
                     StatusBar;
                     ClearMessageBar;
                     turntaken := false;
                  end;
            'Q' : begin
                     quit := true;
                     turntaken := true;
                     WriteMessage('Program aborted - ending simulation.');
                     gameover := true;
                     repeat until KeyPressed;
                  end;
            (* Debug commands: *)
            (*'N' : begin
                     NewLevel;
                     turntaken := true;
                  end;
            'D' : D := D + 1;
            '+' : begin
                     Dtype := Dtype + 1;
                     GoToXY(xmax+2,ymin+2);
                     write(Dtype,'  ');
                  end;
            '-' : begin
                     Dtype := Dtype - 1;
                     GoToXY(xmax+2,ymin+2);
                     write(Dtype,'  ');
                  end;*)
            
            (* No otherwise statement - ignores nonsense commands. *)
      end; (* End of case - input options. *)
      
      if turntaken = true then
      begin
         MarkScent(xpos,ypos);
         turnwait := turndelay + slowcount;
         if slowcount > 0 then turnwait := turndelay * 2;
         if slowcount < 0 then turnwait := trunc(turndelay/2);
      end;
   end; (* End of with. *)
end;


(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)


procedure AllTurns (c : longint);
(* Starts off turns for all actors, splitting off commands into different
   actor types: Player, shots, creatures. *)

begin
   with creatureindex[c] do
   begin
      (* Reduce turncount if waiting. *)
      if turnwait > 0 then turnwait := turnwait - 1;
      
      (* Set turnwait to 0 for particularly fast creatures.  Max speed is thus equal to 1. *)
      if turnwait < 0 then turnwait := 0;
      
      (* If now able to act, act. *)
      if turnwait = 0 then
      begin
         (* Check for blind and reduce blindcount if so. *)
         if blindcount > 0 then
         begin
            blindcount := blindcount - 1;
            if (blindcount = 0) and (c = 1) then
            begin
               WriteMessage('Sensor systems restored.');
               DrawMap;
            end;
         end;
         
         (* Check for slow and reduce slowcount if so. *)
         if slowcount > 0 then
         begin
            slowcount := slowcount - 1;
            if (slowcount = 0) and (c = 1) then
               WriteMessage('Motor functions restored to full capacity.');
         end;
         
         (* Check for speed and increase slowcount if so. *)
         if slowcount < 0 then
         begin
            slowcount := slowcount + 1;
            if (slowcount = 0) and (c = 1) then
               WriteMessage('Enhanced motor functions dissipated.');
         end;
         
         (* Check for confuse and reduce confusecount if so. *)
         if confusecount > 0 then
         begin
            confusecount := confusecount - 1;
            if (confusecount = 0) and (c = 1) then
               WriteMessage('Infected AI routines neutralised. Full pathway control reinitiated.');
         end;
         
         (* Check for pin and reduce pincount if so. *)
         if pincount > 0 then
         begin
            pincount := pincount - 1;
            if (pincount = 0) and (c = 1) then
               WriteMessage('Magnetic field dispersed. Motor control restored.');
         end;
         
         
         (* Assign turns based on creature type. *)
         case ctype of
            0 : begin (* Player turn. *)
                  repeat
                     PlayerTurn;
                  until turntaken = true;
                  turns := turns + 1;
                end; (* End player turn. *)
            1 : ShotMove(c); (* Shots movement. *)
            2..19 : CreatureTurn(c); (* Creature turns. *)
            otherwise ConstructorTurn(c); (* Constructor turns. *)
         end; (* End of case. *)
         
      end; (* End of active turns. *)

   end; (* End of with statement. *)
end;


procedure Time;

var
   i : longint;

begin
   (* Main game stuff. *)
   repeat
      i := 1;
      while i < Length(creatureindex) do
      begin
         if (gameover = false) then AllTurns(i); 
         i := i + 1;
            (* All game turns, repeated until game over. *)
      end;
   until gameover;
      (* Exits command loop - game over. *)
end;


(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)
(* Status Bar Procedures: *)


procedure StatusBar;
(* Redraws status bar on right of screen. *)

var
   y : longint;

begin
   (* Draws border around status box. *)
   TextColor(8); (* Dark grey. *)
   GoToXY(xmax+1,ymin);
   write('*------------*');
   for y := (ymin + 1) to (ymax - 1) do
   begin
      GoToXY(xmax+1,y);
	   write('|');
	   GoToXY(80,y);
	   write('|');
	end;
	GoToXY(xmax+1,ymax);
	write('*------------*');
   
   (* Draws generic status bar text. *)
   TextColor(9); (* Blue for name. *)
   GoToXY(xmax+2,ymin+1);
   write(creatureindex[1].name);
   TextColor(7); (* Light grey for info. *)
   GoToXY(xmax+2,ymin+3);
   write('HP: div0');
   GoToXY(xmax+2,ymin+4);
   write('Power: div0');
   GoToXY(xmax+2,ymin+5);
   write('Ammo: div0');
   GoToXY(xmax+2,ymin+13);
   write('SS Euclid');
   GoToXY(xmax+2,ymin+14);
   write('Level: ');
   GoToXY(xmax+2,ymin+16);
   write('Hostiles on');
   GoToXY(xmax+2,ymin+17);
   write('level:     ');
   GoToXY(xmax+2,ymin+19);
   write('Kills:');
   GoToXY(xmax+2,ymin+20);
   write('Turns:');
	StatusBarUpdate;
end;


procedure StatusBarUpdate;
(* Updates turn, status, and various stats.
   Forwarded at start of program so it can be called throughout the code. *)

begin
   TextBackground(DfBgC); 
   TextColor(12); (* Red for statuses. *)
   GoToXY(xmax+2,ymin+7);
   if (creatureindex[1].slowcount > 0) then write('Motor Error')
      else write('           ');
   GoToXY(xmax+2,ymin+8);
   if (creatureindex[1].blindcount > 0) then write('Sensor Fail')
      else write('           ');
   GoToXY(xmax+2,ymin+9);
   if (creatureindex[1].pincount > 0) then write('Motor Fail')
      else write('           ');
   GoToXY(xmax+2,ymin+10);
   if (creatureindex[1].confusecount > 0) then write('AI Corrupt')
      else write('           ');
   
   TextColor(7); (* Light grey for info. *)
   GoToXY(xmax+9,ymin+14);
   if (D < 10) then write('0');
   write(D);
   
   GoToXY(xmax+9,ymin+17);
   if (enemiesonlevel < 10000) then write('0');
   if (enemiesonlevel < 1000) then write('0');
   if (enemiesonlevel < 100) then write('0');
   if (enemiesonlevel < 10) then write('0');
   write(enemiesonlevel);
   
   GoToXY(xmax+8,ymin+19);
   if (kills < 100000) then write('0');
   if (kills < 10000) then write('0');
   if (kills < 1000) then write('0');
   if (kills < 100) then write('0');
   if (kills < 10) then write('0');
   write(kills);
   
   GoToXY(xmax+8,ymin+20);
   if (turns < 100000) then write('0');
   if (turns < 10000) then write('0');
   if (turns < 1000) then write('0');
   if (turns < 100) then write('0');
   if (turns < 10) then write('0');
   write(turns);
   
   TextColor(DfTxC);
   GoToXY(xmin,ymin); (* Helpful for bebugging. *)
end;


procedure InfoScreen;
(* Gives basic game and trap instructions. *)

begin
   ClrScr;
   TextColor(DfTxC);
   GoToXY(1,1);
   
   writeln;
   writeln(' What happens when an unstoppable force meets an immovable object?');
   writeln;
   writeln(' You.');
   writeln;
   writeln(' You are the AX-5300 Advanced Combat Autobot. You are encased in the');
   writeln(' impenetrable GigaTitan battle suit, impervious to all harm. You wield the');
   writeln(' D71 Omega Pulse Cannon, a weapon of infinite energy that can be stopped or');
   writeln(' deflected by no known force.');
   writeln;
   writeln(' If a shot from the D71 Cannon hits your GigaTitan battle suit the resulting');
   writeln(' conflict of energies will cause the universe to implode.');
   writeln(' Mission note: THIS MUST BE AVOIDED AT ALL COSTS.');
   writeln;
   writeln(' Your mission is to seek out and destroy the rogue AX-4000 Autobot. It also');
   writeln(' wields a D71 Omega Pulse Cannon, but is protected by only a MegaTitan battle');
   writeln(' suit  impervious to all harm except for shots from the D71.');
   writeln;
   writeln(' The Rogue Bot has taken over the SS Euclid science facility orbiting');
   writeln(' Ganimede. It has killed the scientists that manufactured it and created its');
   writeln(' own army of autonomous robots. Its only a matter of time before it breaks');
   writeln(' free of the SS Euclid and comes to attack Earth.');
   writeln;
   writeln('                                    Page 1/3');

   repeat until KeyPressed;
   while KeyPressed do ReadKey;
   ClrScr;
   GoToXY(1,1);
   
   writeln;
   writeln(' The SS Euclid is a multi-layered torus shaped facility with an artificial');
   writeln(' gravitational field. It is believed that the Rogue Bot is stationed in the');
   writeln(' central ring. You must penetrate the outer layers, neutralise all enemy');
   writeln(' robots, and utterly annihilate the Rogue AX-4000.');
   writeln;
   writeln(' You must not allow yourself to be hit by shots from the D71 Omega Pulse');
   writeln(' Cannon that you or the Rogue Bot wield. The consequences of failure are');
   writeln(' catastrophic.');
   writeln;
   writeln;
   TextColor(10);
   writeln(' Interface commands:');
   writeln;
   writeln('  7 8 9   y j u                  q w e');
   TextColor(DfTxC);
   writeln('   \|/     \|/                    \|/');
   TextColor(10);
   write('  4');
   TextColor(DfTxC);
   write('-');
   TextColor(10);
   write('5');
   TextColor(DfTxC);
   write('-');
   TextColor(10);
   write('6   h');
   TextColor(DfTxC);
   write('-');
   TextColor(10);
   write('.');
   TextColor(DfTxC);
   write('-');
   TextColor(10);
   write('l');
   TextColor(DfTxC);
   write('   Move.          ');
   TextColor(10);
   write('a');
   TextColor(DfTxC);
   write('- -');
   TextColor(10);
   write('d');
   TextColor(DfTxC);
   writeln('   Fire the D71.');
   writeln('   /|\     /|\                    /|\');
   TextColor(10);
   writeln('  1 2 3   b k n                  z x c');
   writeln;
   write('  Q');
   TextColor(DfTxC);
   writeln('  -  Exit interface');
   TextColor(10);
   write('  ?');
   TextColor(DfTxC);
   writeln('  -  Display program instructions');
   writeln;
   writeln;
   writeln('                                    Page 2/3');

   repeat until KeyPressed;
   while KeyPressed do ReadKey;
   ClrScr;

   writeln;
   writeln(' Assessment of units:');
   writeln;
   TextColor(10);
   write('  #');
   TextColor(DfTxC);
   writeln(' - Constructor. Reconstructs level according to preset parameters.');
   writeln('         Not a threat  elimination is not required.');
   writeln;
   TextColor(3);
   write('  p');
   TextColor(DfTxC);
   writeln('  piranha. Small, fast-moving attack bots. Mostly harmless.');
   TextColor(5);
   write('  D');
   TextColor(DfTxC);
   writeln('  Decoy. Semi-intelligent distraction units. Mostly harmless.');
   TextColor(11);
   write('  O');
   TextColor(DfTxC);
   writeln('  Octobot. Capable of releasing piranha bots. Minor threat.');
   TextColor(6);
   write('  L');
   TextColor(DfTxC);
   writeln('  Launcher. Fires gravitational impact missiles. Minor threat.');
   TextColor(14);
   write('  *');
   TextColor(DfTxC);
   writeln(' - missile. Can interrupt motor functions on impact. Minor threat.');
   TextColor(1);
   write('  S');
   TextColor(DfTxC);
   writeln('  Siphon. Can leech energy from motor functions. Small threat.');
   TextColor(15);
   write('  F');
   TextColor(DfTxC);
   writeln('  Flasher. Can interrupt sensors with EM bursts. High threat.');
   TextColor(4);
   write('  M');
   TextColor(DfTxC);
   writeln('  Magnetron. Can lock-down motors with magnetic fields. High threat.');
   TextColor(7);
   write('  &');
   TextColor(DfTxC);
   writeln(' - Deploy Unit. Slow-moving, can deploy other robots. High threat.');
   writeln;
   TextColor(12);
   write('  @');
   TextColor(DfTxC);
   writeln(' - Rogue AX-4000. Fires D71 Pulse Cannon. Full capabilities unknown.');
   writeln('                     WARNING: Intense threat.');
   writeln;
   writeln(' All enemies are vulnerable to the D71 Omega Pulse Cannon. Eliminate anything');
   writeln(' that stands in your way and utterly annihilate the Rogue Autobot.');
   writeln;
   writeln;
   writeln('                                    Page 3/3');
   
   repeat until KeyPressed;
   while KeyPressed do ReadKey;
end;


procedure DeathScreen(source : longint);
(* Present details of death and achievements. *)

begin
   ClrScr;
   TextColor(DfTxC);
   writeln;
   writeln('   Universe imploded. Mission failed.');
   writeln;
   write('   Cause of death: ');
   if source = 9 then writeln('Yourself, fool.')
   else writeln('Rogue DX-4000.');
   writeln;
   writeln;
   writeln('    Total enemy units annihilated: ',kills);
   TextColor(10);
   write('        # ');
   TextColor(DfTxC);
   write(': ',conkills);
   TextColor(5);
   write('        D ');
   TextColor(DfTxC);
   writeln(': ',decoykills);
   TextColor(3);
   write('        p ');
   TextColor(DfTxC);
   write(': ',pirahnakills);
   TextColor(11);
   write('        O ');
   TextColor(DfTxC);
   writeln(': ',octobotkills);
   TextColor(6);
   write('        L ');
   TextColor(DfTxC);
   write(': ',launcherkills);
   TextColor(14);
   write('        * ');
   TextColor(DfTxC);
   writeln(': ',missilekills);
   TextColor(1);
   write('        S ');
   TextColor(DfTxC);
   write(': ',siphonkills);
   TextColor(15);
   write('        F ');
   TextColor(DfTxC);
   writeln(': ',flasherkills);
   TextColor(4);
   write('        M ');
   TextColor(DfTxC);
   write(': ',magnetronkills);
   TextColor(7);
   write('        & ');
   TextColor(DfTxC);
   writeln(': ',deploykills);
   TextColor(12);
   write('               @ ');
   TextColor(DfTxC);
   writeln(': ',roguekills);
   writeln;
   writeln('    Shots fired: ',totalshots);
   writeln('    Turns taken: ',turns);
   writeln;
   writeln;
   writeln;
   writeln('    Assessment of your performance:');
   writeln('      Terminally pathetic. Unit not suited for even basic operation.');
   repeat until KeyPressed;
   while KeyPressed do ReadKey;
end;


procedure VictoryScreen(killer : longint);
(* Present details of victory and achievements. *)

begin
   ClrScr;
   TextColor(DfTxC);
   GoToXY(1,1);
   writeln;
   writeln('   Mission Successful. Rogue DX-4000 defeated.');
   writeln;
   write('   Cause of death: ');
   if killer = 9 then writeln('You.')
   else writeln('Unit killed itself.');
   writeln;
   writeln;
   writeln('    Total enemy units annihilated: ',kills);
   TextColor(10);
   write('        # ');
   TextColor(DfTxC);
   write(': ',conkills);
   TextColor(5);
   write('        D ');
   TextColor(DfTxC);
   writeln(': ',decoykills);
   TextColor(3);
   write('        p ');
   TextColor(DfTxC);
   write(': ',pirahnakills);
   TextColor(11);
   write('        O ');
   TextColor(DfTxC);
   writeln(': ',octobotkills);
   TextColor(6);
   write('        L ');
   TextColor(DfTxC);
   write(': ',launcherkills);
   TextColor(14);
   write('        * ');
   TextColor(DfTxC);
   writeln(': ',missilekills);
   TextColor(1);
   write('        S ');
   TextColor(DfTxC);
   write(': ',siphonkills);
   TextColor(15);
   write('        F ');
   TextColor(DfTxC);
   writeln(': ',flasherkills);
   TextColor(4);
   write('        M ');
   TextColor(DfTxC);
   write(': ',magnetronkills);
   TextColor(7);
   write('        & ');
   TextColor(DfTxC);
   writeln(': ',deploykills);
   TextColor(12);
   write('               @ ');
   TextColor(DfTxC);
   writeln(': ',roguekills);
   writeln;
   writeln('    Shots fired: ',totalshots);
   writeln('    Turns taken: ',turns);
   writeln;
   writeln;
   writeln;
   writeln('    Assessment of your performance:');
   writeln('      You are too dangerous to be allowed to exist. You shall now be shut');
   writeln('      down and dismantled, and all records of your activities erased.');
   writeln('      Thank you for your cooperation, and have a nice day.');
   repeat until KeyPressed;
   while KeyPressed do ReadKey;
end;


(*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*-*)


procedure NewGame;

begin
   (* Initialises all values, potentially wiping previous game info. *)
      
   TextColor(DfTxC);
   TextBackground(DfBgC);
   ClrScr;
   
   (* Initial player values. *)
   SetLength(creatureindex,2);
   creatureindex[1].name := 'AX-5300';
   creatureindex[1].Symbol := '@';
   creatureindex[1].Colour := 9;
   creatureindex[1].turnwait := 0;
   creatureindex[1].turndelay := 5;
	creatureindex[1].blindcount := 0;
   creatureindex[1].slowcount := 0;
   creatureindex[1].confusecount := 0;
   creatureindex[1].turntaken := false;
   creatureindex[1].shotfacing := 0;
   creatureindex[1].chaserange := 0;
   creatureindex[1].avoidrange := 0;
   creatureindex[1].specrange := 0;
   creatureindex[1].smart := false;
   creatureindex[1].ctype := 0;
   
   (* General game values. *)
   D := 1;
   Dtype := 1;
   turns := 0;
   messagex := 0;
   messagey := 0;
   kills := 0;
   conkills := 0;
   totalshots := 0;
   enemiesonlevel := 0;
   playerdead := false;
   quit := false;
   won := false;
   gameover := false;
   
   decoykills := 0;
   pirahnakills := 0;
   octobotkills := 0;
   launcherkills := 0;
   missilekills := 0;
   deploykills := 0;
   flasherkills := 0;
   siphonkills := 0;
   magnetronkills := 0;
   roguekills := 0;
   
   (* Start new level, initialise status bar, begin main time loop. *)
   NewLevel;
   StatusBar;
   Time;   
end;


procedure Splashscreen;

begin
   TextColor(10);
   TextBackground(DfBgC);
   ClrScr;
   cursoroff;
   GoToXY(30,10);
   write('U N S T O P P A B L E');
   
   GoToXY(46,4);
   TextColor(12);
   write('@');
   TextBackground(4);
   GoToXY(43,7);
   write('/');
   TextBackground(1);
   TextColor(9);
   GoToXY(37,13);
   write('/');
   TextBackground(0);
   GoToXY(34,16);
   write('@');
   
   (*GoToXY(26,10);
   TextColor(9);
   write('@        ');
   TextBackground(1);
   write('-');
   GoToXY(45,10);
   TextColor(12);
   TextBackground(4);
   write('-');
   TextBackground(0);
   write('        @');*)
   
   GoToXY(27,19);
   TextColor(8);
   write('a roguelike by Darren Grey');
   GoToXY(19,20);
   write('Created for the October 2010 4DRL competition');
   GoToXY(4,22);
   TextColor(DfTxC);
   writeln('Press any key to initiate program sequence.');
   writeln('   Press ''?'' for objective information.  Press ''Q'' to exit interface.');
   repeat until KeyPressed;
   while KeyPressed do move := ReadKey; (* Clears key buffer. *)

   if move = '?' then
      InfoScreen;
   
   if move = 'Q' then quit := true;
   if quit = false then NewGame;   
end;


(*** MAIN PROGRAM ***)

begin
   randomize;
   repeat
      Splashscreen;
   until (quit = true);
   ClrScr;
end. (*** END OF PROGRAM. ***)