\8ACROSS.XPL	JUL-01-99
\8-Across Solitaire Game
\Written by Loren Blaney
\Inspired by Margret Rosenberg

\REVISIONS:
\JUN-27-99, Minor changes for XPL 2.4: Added Clear to prevent flashing card
\ images, win screen goes all the way to bottom.
\JUL-01-99, Swap Line and Text lines so that Nvidia chip works.


\This uses 800x600 graphics, which is mode $6A for VESA-compatible display cards
\Since BIOS doesn't provide a mouse pointer for this mode, this code provides it

inc	C:\CXPL\CODESI;		\Include code definitions for intrinsic routines

eproc	LoadLBM;		\External procedure to load an .LBM file
ext	ImageSize(X0, Y0, X1, Y1, Z),		\Get size of image (paragraphs)
	DrawImage(X, Y, BufSeg, Op),		\Draw image in BufSeg at X,Y
	SaveImage(X0, Y0, X1, Y1, Z, BufSeg),	\Copy image at X,Y into BufSeg
	SaveMask(X0, Y0, X1, Y1, Z, MaskBuf);	\Save copy of mask for the image

int	Cpureg;		\Address of CPU register array (Getreg). This must be
			\ declared first, in same location as in LOADLBM.
\Dimensions:
def	CardWidth = 71,		\Playing card dimensions in pixels
	CardHeight = 96,
	CardSpacing = 9,	\Space between cards (pixels)
	HomeSpacing = 16,	\Vertical space between cards in home position
	EraserHeight = CardHeight + HomeSpacing,
	EraserWidth = CardWidth + CardSpacing,
	PointerWidth = 15,	\Mouse pointer dimensions
	PointerHeight = 25,
	ButtonWidth = 63,	\Push button dimensions
	ButtonHeight = 23,
	ButtonSpacing = 30;	\(Center-to-center spacing, not space between)

\Locations:
def	PileX = 130,	PileY = 30,	\Pixel coordinates of upper-left corner
	HomeX = 30,	HomeY = 30,	\Home position
	UndoX = 34,	UndoY = 500,	\Buttons
	MsgX = 2,	MsgY = 0,	\Messages (text coordinates)
	SlopXMax = CardSpacing-2,	\Sloppiness with which cards are stacked
	SlopYMax = 4;			\ Set to zero for neat stacks

int	BackImageBuf,	\Background image overwritten by mouse pointer
	ButtonBuf(6),	\Array of buffers holding button images
	ButtonMaskBuf,	\Buffer holding image mask for button
	ButtonSize,	\Number of paragraphs required to hold a button image
	CardBuf(54),	\Array of buffers holding card images (including back)
	CardMaskBuf,	\Buffer holding mask for card image
	CardSize,	\Number of paragraphs required to hold a card image
	Counter,	\Number of cards remaining in the pile
	EraserBuf,	\Buffer holding blank area for an eraser
	EraserSize,	\Number of paragraphs required to hold blank area
	HaveMouse,	\Flag: Mouse driver installed and mouse works
	Home(4),	\Cards moved to home position (removed from pile)
	MouseX0,	\Mouse coordinates used for displaying pointer
	MouseY0,
	CardOverlap(8),	\Number of vertical pixels cards in each pile overlap
	Pile(8,52),	\Array to hold cards in pile formation
			\1..52 = card; 0 = no card; -1..-52 = faces down
	PointerBuf,	\Buffer holding mouse pointer image
	PointerMaskBuf,	\Buffer holding mask for pointer image
	PointerSize,	\Number of paragraphs required to hold a pointer image
	SaveCounter,	\Saves for Undo command
	SaveHome(4),	\ Used to restore values in corresponding variables
	SavePile(8,52),
	VidMode;	\Initial video mode to be restored upon exit

def	Esc = $1B;	\Escape control character code

\===============================================================================

func	CallInt(Int, AX, BX, CX, DX, BP, DS, ES); \Call software interrupt
int	Int, AX, BX, CX, DX, BP, DS, ES;  \(Unused arguments need not be passed)
begin
Cpureg(0):= AX;
Cpureg(1):= BX;
Cpureg(2):= CX;
Cpureg(3):= DX;
Cpureg(6):= BP;
Cpureg(9):= DS;
Cpureg(11):= ES;
Softint(Int);
return Cpureg(0);		\Return AX register
end;	\CallInt



func	GetVid;		\Return the current video mode
return CallInt($10, $0F00) & $FF; \Function $0F



proc	AlignPalette;	\Make palette registers correspond to VGA DAC registers
int	N;
for N:= 0, 15 do
	CallInt($10, $1000, N<<8 !N);	\Function $10, subfunction $00

\------------------------------- MOUSE ROUTINES --------------------------------

func	OpenMouse;	\Initialize mouse; return 'false' if it fails
begin			\Pointer is set to center of screen but is hidden
CallInt($21, $3533);	\Make sure mouse vector ($33) points to something
HaveMouse:= false;
if Cpureg(1)=0 & Cpureg(11)=0 then return false;
HaveMouse:= CallInt($33, $0000); \Reset mouse and get status
if HaveMouse then HaveMouse:= true; \(Beware of 'not' operator in 32-bit XPL)
return HaveMouse;		\Return 'false' if failure
end;	\OpenMouse



func	GetMousePosition(N); \Return position of specified mouse coordinate
int	N;	\0 = X coordinate; 1 = Y coordinate
\For video modes $0-$E and $13 the maximum coordinates are 639x199, minus
\ the size of the pointer. For modes $F-$12 the coordinates are the same as
\ the pixels. For 80-column text modes divide the mouse coordinates by 8 to
\ get the character cursor position.
\For mode $6A (800x600) BIOS returns the correct X coordinate, but the Y
\ coordinate must be multiplied by 3.
begin
if ~HaveMouse then return 0;
CallInt($33, $0003);
return if N\#0\ then Cpureg(3)*3 else Cpureg(2);
end;	\GetMousePosition



func	GetMouseButton(N);	\Return 'true' if specified mouse button is down
int	N;	\Button number: 0 = left; 1 = right (or middle)
begin
if ~HaveMouse then return false;
CallInt($33, $0003);
return if N then (Cpureg(1)&2)=2 else (Cpureg(1)&1)=1;
end;	\GetMouseButton



proc	MoveMouse(X, Y);	\Move mouse pointer to X,Y
int	X, Y;
if HaveMouse then
	CallInt($33, $0004, 0, X, Y);



proc	DrawMousePointer;	\At MouseX0, MouseY0 (BIOS doesn't do mode $6A)
begin
\Save background image at mouse position
SaveImage(MouseX0, MouseY0, MouseX0+PointerWidth-1, MouseY0+PointerHeight-1,
	$0F, BackImageBuf);
\Draw mouse pointer at mouse position
DrawImage(MouseX0, MouseY0, PointerMaskBuf, 1\AND\);	\Mask out a hole
DrawImage(MouseX0, MouseY0, PointerBuf, 2\OR\);		\Draw in the pointer
end;	\DrawMousePointer



proc	ShowMouse(On);		\Turn mouse pointer on or off
\The video mode should be set before calling this routine. The pointer should
\ be turned off before drawing over it.
int	On;	\Flag: True = pointer on; False = pointer off
begin
if On
then DrawMousePointer
else DrawImage(MouseX0, MouseY0, BackImageBuf, 0\COPY\);   \Restore background
end;	\ShowMouse



proc	SetMouseLimits;		\Limit the travel of the mouse pointer on screen
begin
\BIOS thinks the mouse ranges from 0 to 799 horizontally and 0 to 199 vertically
CallInt($33, $07, 0, 0, 800-PointerWidth);
CallInt($33, $08, 0, 0, (600-PointerHeight)/3);
end;	\MouseLimits

\-------------------------------------------------------------------------------

proc	Exit;		\Make a clean exit
begin
SetVid(VidMode);	\Restore original video mode (clears screen)
exit;
end;	\Exit



func	GetKey;		\Get character from keyboard (wait if necessary)
int	SC, Ch;			\This is a low-level routine with no echo,
begin				\ and no cursor.
SC:= CallInt($16, $0000);	\Function $00
Ch:= SC & $FF;
if Ch = 0 then Ch:= -(SC>>8);	\Return non-ascii chars as negative scan code
if Ch = $03 \Ctrl-C\ then Exit;
return Ch;
end;	\GetKey



proc	WaitVB;		 \Wait for beginning of monitor's next vertical blank
int	SR;
begin
SR:= Peek($40, $63) + $306;	\Port address of status register
if (SR & $39A) # $39A then return;
while Pin(SR, 0) & $08 do;	\Wait for no vertical blank
repeat until Pin(SR, 0) & $08;	\Wait for vertical blank
end;	\WaitVB;

\===============================================================================

proc	ClearColumn(Col);	\Erase a column of cards in the pile
int	Col;
int	X, Y;
begin
X:= PileX + Col*(CardWidth+CardSpacing);
Y:= PileY;
loop	begin
	DrawImage(X, Y, EraserBuf, 0\COPY\);
	Y:= Y + EraserHeight;
	if Y + EraserHeight >= 600 then quit;
	end;
DrawImage(X, 600-EraserHeight, EraserBuf, 0\COPY\);
end;	\ClearColumn



proc	ShowCard(N, X, Y);	\Display card N at location X,Y
int	N,	\Card number (1-52), 0 is blank, negative numbers are face down
	X, Y;	\Graphic screen coordinates
int	SlopX, SlopY;
begin
SlopX:= Ran(SlopXMax+1);
SlopY:= Ran(SlopYMax+1);
DrawImage(X+SlopX, Y+SlopY, CardMaskBuf, 1\AND\);   \Make a hole (round corners)
DrawImage(X+SlopX, Y+SlopY, CardBuf(if N < 0 then 53 \back\ else N), 2\OR\);
end;	\ShowCard;

\-------------------------------------------------------------------------------

proc	ShowCards(Locs);	\Display cards in pile and home position
int	Locs;	\Bit position indicates columns and home positions to update
int	H, C, X, Y, T, B, X1, Y1, N;
begin
\Show home positions
for H:= 0, 4-1 do
    if 1<<(H+8) & Locs then
	begin
	Y:= HomeY + H*(CardHeight+HomeSpacing);

	\Clear area around home box
	DrawImage(HomeX, Y, EraserBuf, 0\COPY\);

	\Draw box
	DrawImage(HomeX+SlopXMax/2, Y+SlopYMax/2, CardBuf(0), 0\COPY\);

	\Draw all cards up to the top card
	T:= Home(H);		\Get top card
	B:= Rem((T-1)/13);	\Get the rank 0..12
	B:= T - B;		\The bottom card (the ace in the correct suit)
	for C:= B, T do
		ShowCard(C, HomeX, Y);
	end;

\Reduce the amount of overlap in columns that have lots of cards
for X:= 0, 8-1 do
    begin
    if 1<<X & Locs then
	begin
	N:= 0;			\Count the number of cards in this column (N)
	for Y:= 0, 52-1 do
		begin
		C:= Pile(X,Y);		\Get card
		if C # 0 then N:= N + 1
		else Y:= 52;
		end;
	CardOverlap(X):= if N <= 15 then 30
			 else (600-PileY-CardHeight-10) / (N-1);
	end;
    end;

\Show all the cards in the pile
for X:= 0, 8-1 do
    if 1<<X & Locs then
	begin
	ClearColumn(X);
	X1:= PileX + X*(CardWidth+CardSpacing);
	DrawImage(X1+SlopXMax/2, PileY+SlopYMax/2, CardBuf(0), 0\COPY\);
	for Y:= 0, 52-1 do
		begin
		C:= Pile(X,Y);		\Get card
		if C # 0 then
			begin
			Y1:= PileY + Y*CardOverlap(X);
			ShowCard(C, X1, Y1);
			end
		else Y:= 52;
		end;
	end;
end;	\ShowCards

\===============================================================================

proc	Exch(A, B);	\Exchange the contents of locations A and B
int	A, B;
int	T;
begin
T:= A(0);
A(0):= B(0);
B(0):= T;
end;	\Exch



proc	Undo;		\Undo last move (toggles)
int	X, Y;
begin
Exch(addr Counter, addr SaveCounter);
for Y:= 0, 4-1 do Exch(addr Home(Y), addr SaveHome(Y));
for X:= 0, 8-1 do
    for Y:= 0, 52-1 do
	Exch(addr Pile(X,Y), addr SavePile(X,Y));
ShowCards($FFF);
end;	\Undo



proc	DoDeal;		\Restart program if Deal button was pressed
Restart;



proc	DrawButton(N, Down);	\Draw button N in either its up or down position
int	N, Down;
\ N  Button
\ 0  Undo
\ 1  Deal
\ 2  Exit
begin
ShowMouse(false);
DrawImage(UndoX, UndoY+ButtonSpacing*N, ButtonBuf(N*2+(if Down then 1 else 0)),
	0\COPY\);
ShowMouse(true);
end;	\DrawButton



proc	HandleMousePointer;	\Display mouse pointer
int	MX, MY;
begin
MX:= GetMousePosition(0);
MY:= GetMousePosition(1);
if MX#MouseX0 ! MY#MouseY0 then
	begin					\Mouse moved
	\Restore background at old mouse position
	DrawImage(MouseX0, MouseY0, BackImageBuf, 0\COPY\);
	MouseX0:= MX;   MouseY0:= MY;		\Get new mouse position
	DrawMousePointer;
	end;
end;	\HandleMousePointer



proc	HandleKeyCommands;	\Check for and handle shortcut keystrokes
int	Ch;
begin
if Chkkey then
	begin
	Ch:= GetKey;
	ShowMouse(false);	\(Necessary for keyboard "U" command)
	case Ch of
	  ^U,^u,-22:	Undo;	\(Alt+U = -22)
	  ^D,^d,-32:	DoDeal;
	  ^X,^x,-45,Esc:Exit
	other	  [];
	ShowMouse(true);
	end;
end;	\HandleKeyCommands



proc	GetMouseClick;		\Wait for the mouse to click on something
int	N, N0, Pressed;
begin
loop begin
ShowMouse(true);
repeat	begin
	HandleMousePointer;
	HandleKeyCommands;
	end;
until GetMouseButton(0) ! GetMouseButton(1);	\A mouse button is pressed

\Get soft button number (N0) (if any) that mouse pointer is on
N0:= 0;
for N:= 0, 3-1 do
	begin
	\If mouse was on button N then record button number in N0
	if MouseX0>=UndoX & MouseX0<=UndoX+ButtonWidth-1 &
	   MouseY0>=UndoY+N*ButtonSpacing &
	   MouseY0<=UndoY+N*ButtonSpacing+ButtonHeight-1 then
	   	N0:= N;
	end;

Pressed:= false;
repeat	begin
	HandleMousePointer;
	HandleKeyCommands;

	\If mouse pointer is on original soft button (N0) then show it pressed
	if MouseX0>=UndoX & MouseX0<=UndoX+ButtonWidth-1 &
	   MouseY0>=UndoY+N0*ButtonSpacing &
	   MouseY0<=UndoY+N0*ButtonSpacing+ButtonHeight-1 then
		begin
		if not Pressed then		\Only draw button if it changes
			begin
			DrawButton(N0, true);	\Draw pressed button
			Pressed:= true;		\Indicate it has been drawn
			end;
		end
	else	begin	\Mouse pointer is not on original button: show button up
		if Pressed then
			begin
			DrawButton(N0, false);	\Draw unpressed button
			Pressed:= false;
			end;
		end;
	end;
until not GetMouseButton(0) & not GetMouseButton(1);	\Both buttons released

\If a soft button is down then execute its function
if Pressed then
	begin
	DrawButton(N0, false);		\Display unpressed button
	ShowMouse(false);		\(Necessary for Undo command)
	case N0 of
	  0:	Undo;
	  1:	DoDeal;
	  2:	Exit
	other	[];
	end
else quit;
end;	\loop

ShowMouse(false);
end;	\GetMouseClick



proc	Wait(T);	\Wait T 60ths of a second or until a mouse click
int	T;
begin
ShowMouse(true);
loop	begin
	HandleMousePointer;
	HandleKeyCommands;
	WaitVB;
	T:= T -1;
	if T <= 0 then quit;
	if GetMouseButton(0) ! GetMouseButton(1) then quit;
	end;
ShowMouse(false);
end;	\Wait



proc	ShowWin;	\Show "1000 card pickup" for winning screen
int	N, X, Y, T;
begin
T:= 1000;
loop	begin
	HandleKeyCommands;
	if T < 940 then		\Beware of double clicking
		if GetMouseButton(0) ! GetMouseButton(1) then quit;
	WaitVB;
	T:= T -1;
	if T <= 0 then quit;
	N:= Ran(52) + 1;	\1..52
	if Ran(5) = 0 then N:= -N;
	X:= Ran(800-CardWidth);
	Y:= Ran(600-CardHeight-PileY) + PileY;
	DrawImage(X, Y, CardMaskBuf, 1\AND\);   \Make a hole (round corners)
	DrawImage(X, Y, CardBuf(if N < 0 then 53 \back\ else N), 2\OR\);
	end;
end;	\ShowWin

\-------------------------------------------------------------------------------

proc	ShowMsg(Str);	\Display an error message
char	Str;
begin
Cursor(MsgX, MsgY);
Text(0, Str);
end;	\ShowMsg



proc	Beep;		\A not-too-obnoxious beep
begin
Sound(false, 1, 1000);	\Synchronize with system timer to make tone a
Sound(true, 1, 3000);	\ consistent duration and a consistent sound.
end;	\Beep



proc	ErrorMsg(Str);	\Display an error message
char	Str;
begin
Beep;
ShowMsg(Str);
end;	\ErrorMsg



proc	SaveState;	\Save current state for Undo command
int	X, Y;
begin
SaveCounter:= Counter;
for Y:= 0, 4-1 do SaveHome(Y):= Home(Y);
for X:= 0, 8-1 do
    for Y:= 0, 52-1 do
	SavePile(X,Y):= Pile(X,Y);
end;	\SaveState

\-------------------------------------------------------------------------------

proc	MakeMove;	\Make a (legal) move
int	MX, MY,		\Mouse position in pixel coordinates
	X0, Y0,		\Pile's column and row coordinates (top card to move)
	X1, Y1,		\Column and row coordinates (-1) of destination
	C0, C1,		\Card values at source and destination
	N;		\Scratch


	proc	MoveStack;	\Move stack of cards from X0,Y0 to X1,Y1
	begin
	N:= 0;
	while Pile(X0,Y0+N) # 0 do		\Last card has been moved
		begin
		Pile(X1,Y1+N):= Pile(X0,Y0+N);	\Move card to new location
		Pile(X0,Y0+N):= 0;		\Remove card from old location
		N:= N + 1;			\Next card
		end;
	end;	\MoveStack


begin	\MakeMove
\Ignore invalid moves and display feedback messages indicating reason for error
loop	begin
	loop	begin				\Click on card(s) to move
		ShowMsg("Click on a card to move     ");
		GetMouseClick;
		MX:= GetMousePosition(0);
		MY:= GetMousePosition(1);

		\Ignore click if it's not on the pile
		if MX<PileX ! MX>PileX+8*(CardWidth+CardSpacing)-1 !
		   MY<PileY then
			ErrorMsg("Select a card from the pile ")

		else	begin
			\Convert pixel coordinates to pile's column and row
			X0:= (MX - PileX) / (CardWidth + CardSpacing);
			Y0:= (MY - PileY) / CardOverlap(X0);

			\If clicking on bottom part of bottom card then move up
			for N:= 0, CardHeight/CardOverlap(X0) do
				if Pile(X0,Y0)=0 & Y0>0 then Y0:= Y0-1;

			\Ignore click if it's not on a card
			C0:= Pile(X0,Y0);		\Get card's value
			if C0 = 0 then
				ErrorMsg("Select a card               ")

			\Ignore click if the card is not facing up
			else if C0 < 0 then
				ErrorMsg("Only face-up cards can move ")

			\Otherwise exit loop with valid card selected from pile
			else quit;
			end;

		Wait(120);			\Delay to read message
		end;	\loop - repeat until valid card is selected

	\Provide feedback to player
	ShowMsg("Click on a column to move to");

	\Click on target (move-to) location
	GetMouseClick;
	MX:= GetMousePosition(0);
	MY:= GetMousePosition(1);

	\If target location is in the pile then...
	if MX>=PileX & MX<=PileX+8*(CardWidth+CardSpacing)-1 &
	   MY>=PileY then
		begin
		\Get the card at the bottom of the stack
		X1:= (MX - PileX) / (CardWidth + CardSpacing);

		\Move Y1 up to first card at bottom of stack or to top of column
		Y1:= 52-1;
		while Pile(X1,Y1)=0 & Y1>0 do Y1:= Y1 - 1;
		C1:= Pile(X1,Y1);	\Get card value

		\If target column contains a card then...
		if C1 # 0 then
			begin
			\If target column = source column then...
			if X1 = X0 then
			    begin
			    \If source card (C0) is the bottom card then...
			    if Pile(X0,Y0+1) = 0 then
				begin
				\If there's a legal move to the home posn then
				for Y1:= 0, 4-1 do
				\(Don't put ace of spades on king of clubs, etc)
					begin
					if C0=Home(Y1)+1 & Rem(C0/13)#1\ace\ !
					   Rem(C0/13)=1\ace\ & Home(Y1)=0 then
						begin	\Move card to home posn
						SaveState;
						Home(Y1):= C0;	\Move card
						Pile(X0,Y0):= 0;
						Counter:= Counter - 1;
						X1:= Y1 + 8;	\For ShowCards
						quit;
						end;
					end;
				end;
			    ErrorMsg("Can't move to same column   ");
			    end
			\Else if source card = target card +/-1 then...
			else if C0=C1-1 & Rem(C1/13)#1 \not an ace\  !
				C0=C1+1 & Rem(C1/13)#0 \not a king\ then
				begin
				SaveState;
				Y1:= Y1 + 1;	\Move to below target card
				MoveStack;
				quit;
				end
			else	ErrorMsg("Cards must be in sequence   ");
			end
		else	begin			\Target column is blank
			\If source card is not a king then error
			if Rem(C0/13) # 0 then
				ErrorMsg("Only a king can move here   ")

			\Else if source card is not off bottom or top then error
			else if Pile(X0,Y0+1)#0 & Y0#0 then
				ErrorMsg("King must be at the bottom  ")

			\Else move card(s) and quit loop
			else	begin
				SaveState;
				MoveStack;
				quit;
				end;
			end;
		end

	\Else if target location is in the home position then...
	else if MX>=HomeX & MX<=HomeX+4*(CardWidth+CardSpacing)-1 &
		MY>=HomeY & MY<=HomeY+4*(CardHeight+HomeSpacing)-1 then
		begin
		Y1:= (MY - HomeY) / (CardHeight + HomeSpacing);
		C1:= Home(Y1);				\Get card's value

		\If source card is not from the bottom of the stack then error
		if Pile(X0,Y0+1) # 0 then
			ErrorMsg("Card must be at the bottom  ")

		\Else if target location contains card & source card does not
		\ equal target card+1 (or top already has a king) then error
		else if C1#0 & (C0#C1+1 ! Rem(C1/13)=0\king\) then
			ErrorMsg("Cards must be in sequence   ")

		\Else if target location is blank & source card is not an ace
		\ then error
		else if C1=0 & Rem(C0/13)#1\ace\ then
			ErrorMsg("Only an ace can move here   ")

		\Else move card to home position
		else	begin
			SaveState;
			Home(Y1):= C0;		\Move card
			Pile(X0,Y0):= 0;
			Counter:= Counter - 1;
			X1:= Y1 + 8;		\For ShowCards
			quit;
			end;
		end

	\Else target location is illegal
	else ErrorMsg("Move to column or home stack");

	Wait(120);
	end;	\loop - repeat until valid move is made

ShowMsg("                            ");	\Erase prompt message for safety

\If there is a face-down card at the bottom of the stack, turn it up
if Y0 > 0 then
	if Pile(X0,Y0-1) < 0 then Pile(X0,Y0-1):= -Pile(X0,Y0-1);

ShowCards(1<<X0 ! 1<<X1);		\Show result
end;	\MakeMove

\-------------------------------------------------------------------------------

proc	Deal;		\Set up cards in initial pile (tableau) configuration
\			  and initialize Home array and Counter
\
\     0  1  2  3  4  5  6  7  X
\ 0   U  D  U  D  U  D  U  D		U = Face up
\ 1   D  U  D  U  D  U  D  U		D = Face down
\ 2   U  D  U  D  U  D  U  D
\ 3      U  D  U  D  U  D  U
\ 4         U  D  U  D  U  D
\ 5            U  D  U  D  U
\ 6               U  D  U  D
\ 7                  U  D  U
\ 8                     U  D
\ 9                        U
\ Y
\
int	X, Y;		\Coordinates in the pile
char	Deck(52);	\Array of card values 0..51
int	T;


	func	GetCard;	\Randomly select a card (1-52)
	int	Loc, Card, I;
	begin
	Loc:= Ran(Counter);		\Randomly select a location in the deck
	Card:= Deck(Loc);		\Get card value at this location
	for I:= Loc, Counter-2 do	\Remove selected card by moving all
		Deck(I):= Deck(I+1);	\ cards down
	Counter:= Counter - 1;		\One less card location to select from
	return Card;			\Return card value (1..52)
	end;	\GetCard


begin	\Deal
Counter:= 52;			\Set up deck with all 52 card values (1..52)
for X:= 0, 52-1 do Deck(X):= X+1;

for Y:= 0, 52-1 do		\Initialize Pile with "no cards"
    for X:= 0, 8-1 do
	Pile(X,Y):= 0;

for Y:= 0, 9 do			\Set up Pile with starting configuration
    for X:= 0, 8-1 do		\ (shown above)
	begin
	CardOverlap(X):= 30;
	if Y <= X+2 then
		begin
		Pile(X,Y):= if (X+Y) & 1 then -GetCard else GetCard;
		for T:= 0, 4-1 do WaitVB;	\Go slow for effect
		ShowCard(Pile(X,Y), PileX+X*(CardWidth+CardSpacing),
			PileY+Y*CardOverlap(X));
		end;
	end;

for X:= 0, 4-1 do Home(X):= 0;	\No cards in Home positions
end;	\Deal

\-------------------------------------------------------------------------------

proc	PlayGame;		\Play one game of Solitaire
int	N, H, Y;
begin
Clear;
Move(0, 18);   Line(800-1, 18, 3\black\);
Cursor(40, 0);
Text(0, "8-Across Solitaire");

for N:= 0, 8-1 do
	begin
	ClearColumn(N);		\Clear column then draw empty card box
	DrawImage(PileX+N*(CardWidth+CardSpacing)+SlopXMax/2, PileY+SlopYMax/2,
		CardBuf(0), 0\COPY\);
	end;

for H:= 0, 4-1 do			\Show home positions
	begin
	Y:= HomeY + H*(CardHeight+HomeSpacing);

	\Clear area around home box
	DrawImage(HomeX, Y, EraserBuf, 0\COPY\);

	DrawImage(HomeX+SlopXMax/2, HomeY+H*(CardHeight+HomeSpacing)+SlopYMax/2,
		CardBuf(0), 0\COPY\);	\Draw empty card box
	end;

for N:= 0, 3-1 do			\Show buttons
	DrawImage(UndoX, UndoY+ButtonSpacing*N, ButtonBuf(N*2), 0\COPY\);

Deal;

Counter:= 52;
SaveState;			\Make sure Undo command doesn't restore garbage
repeat MakeMove until Counter <= 0;
ShowMsg("Congratulations!            ");
ShowWin;
GetMouseClick;
end;	\PlayGame

\-------------------------------------------------------------------------------

proc	Initialize;
int	I, X, Y, X0, Y0;
begin
Cpureg:= Getreg;
TrapC(true);		\Force user to make a clean exit, restoring video mode

I:= Equip;
if (I(2)&$08) = 0 then			\Test for VGA display
	[Text(0, "

This program requires an SVGA display.  Sorry.
");   exit];

\Initialize mouse
Text(0, "Looking for mouse...
");
if not OpenMouse then
	[Text(0, "This program requires a mouse.  Sorry.
");   exit];

VidMode:= GetVid;
Setvid($6A);				\800x600x16, VESA-compatible
CallInt($10, $1201, $36);		\Disable video display
WaitVB;					\Avoid any possible screen glitch
AlignPalette;
LoadLBM("8ACROSS.LBM");			\Load card images into VGA display mem

\Save card images to main memory
CardSize:= ImageSize(0, 0, CardWidth-1, CardHeight-1, $0F);
I:= 0;
for Y:= 0, 6-1 do
    for X:= 0, 9-1 do
	begin				\CardBuf(0) = Empty card box
	CardBuf(I):= Malloc(CardSize);	\CardBuf(53) = Card back
	SaveImage(X*CardWidth, Y*CardHeight,
		 (X+1)*CardWidth-1, (Y+1)*CardHeight-1, $0F, CardBuf(I));
	I:= I + 1;
	end;

\Use card back for image mask. BEWARE: Dark green images are transparent.
CardMaskBuf:= Malloc(CardSize);
X:= 8*CardWidth;
Y:= 5*CardHeight;
SaveMask(X, Y, X+CardWidth-1, Y+CardHeight-1, $0F, CardMaskBuf);

\Save a blank area to use as an eraser
X:= 9*CardWidth;
Y:= 100;
EraserSize:= ImageSize(X, Y, X+EraserWidth-1, Y+EraserHeight-1, $0F);
EraserBuf:= Malloc(EraserSize);
SaveImage(X, Y, X+EraserWidth-1, Y+EraserHeight-1, $0F, EraserBuf);

\Save mouse pointer image to memory (BIOS doesn't provide one in mode $6A)
X:= 9*CardWidth;
Y:= 0;
PointerSize:= ImageSize(X, Y, X+PointerWidth-1, Y+PointerHeight-1, $0F);
PointerBuf:= Malloc(PointerSize);
SaveImage(X, Y, X+PointerWidth-1, Y+PointerHeight-1, $0F, PointerBuf);
PointerMaskBuf:= Malloc(PointerSize);
SaveMask(X, Y, X+PointerWidth-1, Y+PointerHeight-1, $0F, PointerMaskBuf);

\Initialize mouse pointer
BackImageBuf:= Malloc(PointerSize);
MouseX0:= GetMousePosition(0);
MouseY0:= GetMousePosition(1);
MoveMouse(800/2, 200/2);		\Show mouse pointer in center of screen
SetMouseLimits;

\Save button images to memory
ButtonSize:= ImageSize(0, 0, ButtonWidth-1, ButtonHeight-1, $0F);
X0:= 639;   Y0:= 28;			\Upper-left corner of button images
I:= 0;
for Y:= 0, 3-1 do
    for X:= 0, 2-1 do
	begin
	ButtonBuf(I):= Malloc(ButtonSize);
	SaveImage(X0+X*ButtonWidth, Y0+Y*ButtonHeight,
		  X0+(X+1)*ButtonWidth-1, Y0+(Y+1)*ButtonHeight-1,
		  $0F, ButtonBuf(I));
	I:= I + 1;
	end;

\Save button mask
ButtonMaskBuf:= Malloc(ButtonSize);
SaveMask(X0, Y0, X0+ButtonWidth-1, Y0+ButtonHeight-1, $0F, ButtonMaskBuf);

Clear;
WaitVB;					\Avoid any possible screen glitch
CallInt($10, $1200, $36);		\Reenable video display
end;	\Initialize

\===============================================================================

begin	\Main
if not Rerun then Initialize;		\"Deal" button restarts program
loop PlayGame;
end;	\Main
