unit tdyGraph;

interface

uses
	jfunc;

const
	videoCard_mda=0;
	videoCard_cga=1;
	videoCard_pcJr=2;
	videoCard_tandy1000=3;
	videoCard_tandySLTL=4;
	videoCard_ega=5;
	videoCard_vga=6;
	videoCard_mcga=7;
	videoCardName:packed array[0..videoCard_mcga] of string[11]=(
		'MDA','CGA','PCJr','Tandy 1000','Tandy TL/SL','EGA','VGA','MCGA'
	);

type
	pTileData=^byte;

	pTileSet=^tileSet;
	tileSet=object
		size:word;
		name:dosName;
		dataStart,
		dataEnd:pTileData;
		constructor init(dataFile:dosName);
		destructor term;
	end;

	writeTileProc=procedure(x,y:byte; tileSet:pointer; tile:byte);
	backgroundProc=procedure(x,y:byte; data:pTileData);
	bufferShowProc=procedure(x,y:byte);

	pFontData=^tFontData;
	tFontData=array[0..127,0..7] of byte;
	tKernData=array[0..127] of byte;

	pfontSet=^fontSet;
	fontSet=object
		fontData:tFontData;
		kernData:tKernData;
		constructor init(dataFile:dosName);
		function textLength(st:st40):word;
		procedure outText(x,y:byte; st:st40; color:byte);
		procedure outTextRight(x,y:byte; st:st40; color:byte);
		procedure outTextCentered(x,y:byte; st:st40; color:byte);

		{ decimal functions erase background to color zero!!! }
		procedure outDecNonPadded(x,y:byte; n:longint; color:byte);
		procedure outDecPadded(x,y:byte; n:longint; color:byte);
		procedure outHex(x,y:byte; n:longint; color:byte);
	end;

	pSprite=^Sprite;
	sprite=object
		{ these values the user can change directly }
		currentX,currentY,currentTile,
		bufferX,bufferY:integer;

		{ from here out this should be left to the code to manage }
		oldX,oldY,sTileSize:integer;
		sTiles:pTileSet;

		{ function handlers, prevents needing a 'case' statement inside the loop }
		bufferWriteTile:writeTileProc;
		bufferShow:bufferShowProc;

	  { pointers for list control }
		prev,next:pSprite;

		constructor init(parent:pSprite; x,y:byte; tile,tileSize:integer; tiles:pTileSet);
		procedure bufferHide;
		procedure bufferRender;
	end;

	pUpdateBlock=^updateBlock;
	updateBlock=object
		x,y:byte;
		update:boolean;
		next:pUpdateBlock;
		constructor init(ix,iy:integer);
	end;

	updateBlockList=object
		first,last:pUpdateBlock;
		constructor init;
		procedure add(ix,iy:integer);
		destructor term;
	end;

	pPackedBuffer=^tPackedBuffer;
	tPackedBuffer=array[0..7999] of byte;

var
	sourceBuffer,destBuffer,
	backgroundBuffer,renderBuffer:pPackedBuffer;
	updateList:updateBlockList;
	videoCard:word;

procedure tg_init;
procedure tg_term;

procedure tg_clear(color:byte);

procedure tg_putPixel(x,y,color:byte);
function tg_getPixel(x,y:byte):byte;

procedure tg_bar(sx,sy,ex,ey:integer; color:byte);
procedure tg_rectangle(sx,sy,ex,ey:integer; color:byte);

procedure tg_tile3(x,y:byte; tileSet:pointer; tile:byte);
procedure tg_tile5(x,y:byte; tileSet:pointer; tile:byte);

procedure tg_writeChar(x,y,chr,color:byte; data:pFontData);
procedure tg_writeNum(x,y,digit,color,bgColor:byte; data:pFontData);

procedure tg_waitRetrace;

function buffer_addSprite(x,y,tile,tileSize:integer; tiles:pTileSet):pSprite;

procedure buffer_sourceBackground; { also sets dest to render }
procedure buffer_sourceRender;     { also sets dest to background }
procedure buffer_sourceClear(color:byte);
procedure buffer_copySource2Dest;
procedure buffer_copyDest2Screen;
procedure buffer_copySourceDest8x8(x,y:byte); { source -> display, 8x8 byte-bound }
procedure buffer_show4x3(x,y:byte);           { dest -> display, 8x8 byte-bound }
procedure buffer_show8x6(x,y:byte);            { dest -> display, 8x8 byte-bound }
procedure buffer_tile3(x,y:byte; tileSet:pointer; tile:byte);  { tile -> dest }
procedure buffer_tile5(x,y:byte; tileSet:pointer; tile:byte);  { tile -> dest }
procedure buffer_nullTile4x3(x,y:byte);        { erase source buffer section }
procedure buffer_showSprites;
procedure buffer_hideSprites;
procedure buffer_updateSprites;

procedure buffer_destPutPixel(x,y,color:byte);
function buffer_sourceGetPixel(x,y:byte):byte;


implementation


const
	pageSize=$3E80;
	pageSizeWords=pageSize shr 1;

var
	firstSprite,lastSprite:pSprite;
	textSegment:word;
	oldMode:byte;
	oldTgExitProc:pointer;

	textGraphOn:boolean;

constructor updateBlock.init(ix,iy:integer);
begin
	x:=ix;
	y:=iy;
	update:=false;
	next:=nil;
end;

constructor updateBlockList.init;
begin
	first:=nil;
	last:=nil;
end;

procedure updateBlockList.add(ix,iy:integer);
begin
	if (first=nil) then begin
		new(first,init(ix,iy));
		last:=first;
	end else begin
		new(last^.next,init(ix,iy));
		last:=last^.next;
	end;
end;

destructor updateBlockList.term;
begin
	while not(first=nil) do begin
		last:=first^.next;
		dispose(first);
		first:=last;
	end;
end;

constructor sprite.init(parent:pSprite; x,y:byte; tile,tileSize:integer; tiles:pTileSet);
begin
	currentX:=x;
	oldX:=x;
	currentY:=y;
	oldY:=y;
	currentTile:=tile;
	sTiles:=tiles;
	sTileSize:=tileSize;
	case sTileSize of
		3:begin
			bufferWriteTile:=buffer_tile3;
			bufferShow:=buffer_show4x3;
		end;
		5:begin
			bufferWriteTile:=buffer_tile5;
			bufferShow:=buffer_show8x6;
		end;
	end;
	prev:=parent;
	next:=nil;
end;

procedure sprite.bufferHide;
begin
	buffer_copySourceDest8x8(oldX,oldY);
end;

procedure sprite.bufferRender;
begin
	bufferWriteTile(currentX,currentY,sTiles^.dataStart,currentTile);
	oldX:=currentX;
	oldY:=currentY;
end;

function buffer_addSprite(x,y,tile,tileSize:integer; tiles:pTileSet):pSprite;
begin
	if firstSprite=nil then begin
		new(firstSprite,init(nil,x,y,tile,tileSize,tiles));
		lastSprite:=firstSprite;
	end else begin
		new(lastSprite^.next,init(lastSprite,x,y,tile,tileSize,tiles));
		lastSprite:=lastSprite^.next;
	end;
	buffer_addSprite:=lastSprite;
end;

procedure discardSprites;
begin
	while not(lastSprite=nil) do begin
		firstSprite:=lastSprite^.prev;
		dispose(lastSprite);
		lastSprite:=firstSprite;
	end;
end;

procedure rot4(dstart:pointer; size:word); assembler;
asm
	mov  cx,size
	les  di,dstart
@loop:
	mov  al,es:[di]
	ror  al,1
	ror  al,1
	ror  al,1
	ror  al,1
	stosb
	loop @loop
end;

constructor tileSet.init(dataFile:dosName);
var
	f:file;
	pdata:pointer;
begin
	name:=datafile;
	assign(f,name+'.DAT');
	reset(f,1);
	size:=filesize(f);
	getmem(dataStart,size);
	dataEnd:=dataStart;
	inc(dataEnd,size);
	blockread(f,dataStart^,size);
	close(f);
	pdata:=datastart; { asm can't see object elements }
	write('attempting ',name);
	{ tandy/jr 16 color backwards from TXT }
	if not(name='MAP') then rot4(dataStart,size);
	writeln(' - done');
end;

destructor tileSet.term;
begin
	freemem(dataStart,size);
end;

constructor fontSet.init(dataFile:dosName);
var
	f:file;
begin
	assign(f,dataFile+'.FNT');
	reset(f,1);
	blockread(f,fontData,sizeof(tFontData));
	blockread(f,kernData,sizeof(tKernData));
	close(f);
end;

function fontSet.textLength(st:st40):word;
var
	t,sx:word;
begin
	sx:=0;
	t:=0;
	while (t<length(st)) do begin
		inc(t);
		case st[t] of
			'\':begin
				inc(t);
				if(st[t]='\') then sx:=sx+kernData[ord('\')];
			end;
			else begin
				sx:=sx+kernData[ord(st[t]) and $7F];
			end;
		end;
	end;
	textLength:=sx;
end;

procedure fontSet.outText(x,y:byte; st:st40; color:byte);
var
	sx,t:word;
	b,c:byte;

begin
	sx:=x;
	t:=0;
	c:=color;
	while (t<length(st)) do begin
		inc(t);
		case st[t] of
			'\':begin
				inc(t);
				b:=ord(st[t]);
				case st[t] of
					'0'..'9':c:=b-48;
					'A'..'F':c:=b-55;
					'a'..'F':c:=b-87;
					'\':begin
						tg_writeChar(sx,y,b,c,@fontData);
						inc(sx,kernData[b]);
					end;
				end;
			end;
			else begin
				b:=ord(st[t]);
				tg_writeChar(sx,y,b,c,@fontData);
				inc(sx,kernData[b]);
			end;
		end;
	end;
end;

procedure fontSet.outTextCentered(x,y:byte; st:st40; color:byte);
var
	sx:word;
begin
	sx:=x-textLength(st) div 2;
	outText(sx,y,st,color);
end;

procedure fontSet.outTextRight(x,y:byte; st:st40; color:byte);
var
	sx:word;
begin
	sx:=x-textLength(st);
	outText(sx,y,st,color);
end;

procedure fontSet.outDecNonPadded(x,y:byte; n:longint; color:byte);
var
	sx:word;
	d:longint;
	b:byte;
	started:boolean;
begin
	d:=10000000;
	sx:=x;
	started:=false;
	while (d>0) do begin
		b:=((n div d) mod 10);
		d:=d div 10;
		if (b>0) or started then begin
			tg_writeNum(sx,y,b,color,0,@fontData);
			inc(sx,4);
			started:=true;
		end;
	end;
end;


procedure fontSet.outDecPadded(x,y:byte; n:longint; color:byte);
var
	sx:word;
	d:longint;
	b:byte;
begin
	d:=10000000;
	sx:=x;
	while (d>0) do begin
		b:=((n div d) mod 10);
		d:=d div 10;
		tg_writeNum(sx,y,b,color,0,@fontData);
		inc(sx,4);
	end;
end;

procedure fontSet.outHex(x,y:byte; n:longint; color:byte);
var
	sx:word;
	d:longint;
	b:byte;
begin
	d:=$10000000;
	sx:=x;
	while (d>0) do begin
		b:=(n div d) and $0F;
		case b of
			0..9:inc(n,48);
			10..15:inc(n,55);
		end;
		d:=d div $10;
		tg_writeChar(sx,y,b,color,@fontData);
		inc(sx,kernData[b]);
	end;
end;

procedure tg_waitRetrace; assembler;
asm
	mov  dx,$3DA
	mov  ah,$08
@loop:
	in   al,dx
	and  al,ah
	jz   @loop
end;

{
	Detecting which video card is present is kinda tricky...
	but thankfully they did something smart with int $10.
	Calls to unknown subfunctions just RET leaving registers
	intact, so if you call a VGA function that you know changes
	a register, and the register doesn't change, it's not a VGA.
	Call a EGA function ditto, ditto... finally check if we're in
	a monochrome display mode, that's MDA.

	Unfortunately there's no known reliable check for a CGA since
	newer cards pretend to be one -- but if we eliminate
	'everything else' from the list, it must be CGA.
}

function detectCard:byte; assembler;
asm
	mov  ax,$1200
	mov  bl,$32       { VGA only enable video }
	int  $10
	cmp  al,$12       { VGA returns $12, all others leave it unmodified! }
	jne  @notVGA      { not a vga, test for EGA }
	                  { VGA, or is it? test for MCGA }
	xor  bl,bl        { null BL so it's set up for non-PS/2 }
	mov  ax,$1A00
	int  $10
	cmp  bl,$0A       { MCGA returns $0A..$0C }
	jb   @isVGA
	cmp  bl,$0C
	jg   @isVGA
	mov  al,videoCard_mcga
	ret
@isVGA:
	mov  al,videoCard_vga
	ret
@notVGA:           { We eliminated VGA, so an EGA/VGA true must be EGA }
	mov  ah,$12
	mov  bl,$10       { EGA/VGA get configuration info }
	int  $10
	and  bl,$03       { EGA/VGA returns a 0..3 value here }
	jz   @notEGA      { not a VGA, test for MDA }
	mov  al,videoCard_ega
	ret
@notEGA:            { MDA all we need to detect is video mode 7 }
	mov  ah,$0F       { get Video mode }
	int  $10
	cmp  al,$07
	jne  @notMDA
	mov  al,videoCard_mda
	ret
@notMDA:            { not MDA, check for Jr. }
	mov  ax,$FFFF
	mov  es,ax
	mov  di,$000E     { second to last byte PCjr/Tandy BIOS info area }
	mov  al,$FD       { ends up $FD only on the Jr. }
	cmp  es:[di],al
	jne  @notJr
	mov  al,videoCard_pcJr
	ret
@notJr:             { not junior, test for tandy }
	mov  al,$FF       { all tandy's return $FF here }
	cmp  es:[di],al
	jne  @notTandy
	mov  ax,$FC00
	mov  es,ax
	xor  di,di
	mov  al,$21
	cmp  es:[di],al
	jne  @notTandy
	mov  ah,$C0       { test for SL/TL }
	int  $15          { Get System Environment }
	jnc  @tandySLTL     { early Tandy's leave the carry bit set, TL/SL does not }
	mov  al,videoCard_tandy1000
	ret
@tandySLTL:
	mov  al,videoCard_tandySLTL
	ret
@notTandy:
	mov  al,videoCard_cga { all other cards eliminated, must be CGA }
end;

function getVideoMode:byte; assembler;
asm
	mov  ax,$0F00;
	int  $10;
end;

procedure setVideoMode(mode:byte); assembler;
asm
	xor ah,ah
	mov al,mode
	int $10;
end;

procedure tg_putPixel(x,y,color:byte); assembler;
asm
	mov  ax,$B800
	mov  es,ax

	mov  ah,y
	xor  al,al

	shr  ax,1
	shr  ax,1
	mov  di,ax
	shr  ax,1
	shr  ax,1
	add  di,ax

	xor  ah,ah
	mov  al,x
	mov  bx,ax
	shr  ax,1
	add  di,ax

	mov  ah,color
	mov  al,es:[di]

	and  bx,$01
	jnz   @xEven
	mov  cl,4
	shl  ah,cl
	and  al,$0F
	jmp  @writeIt
@xEven:
	and  al,$F0
  and  ah,$0F
@writeIt:
	or   al,ah
	mov  es:[di],al
	add  di,$2000
	mov  es:[di],al
end;

function tg_getPixel(x,y:byte):byte; assembler;
asm
	mov  ax,$B800
	mov  es,ax

	mov  ah,y
	xor  al,al

	shr  ax,1
	shr  ax,1
	mov  di,ax
	shr  ax,1
	shr  ax,1
	add  di,ax

	xor  ah,ah
	mov  al,x
	mov  bx,ax
	shr  ax,1
	add  di,ax

	mov  al,es:[di]

	and  dx,$01
	jz   @xEven

	mov  cl,4
	shr  al,cl

@xEven:
	and  al,$0F
end;

procedure buffer_destPutPixel(x,y,color:byte); assembler;
asm
	les  di,destBuffer
	xor  al,al
	mov  ah,y
	shr  ax,1
	mov  bx,ax
	shr  ax,1
	shr  ax,1
	add  bx,ax

	xor  dh,dh
	mov  dl,x
	add  bx,dx
	shr  bx,1
	add  di,bx

	mov  al,color
	mov  bl,es:[di]

	and  dx,1
	jnz  @xOdd

	and  bl,$F0
	jmp  @xDone

@xOdd:
	and  bl,$0F
	mov  cl,4
	shl  al,cl

@xDone:
	or   al,bl
	mov  es:[di],al
end;

function buffer_sourceGetPixel(x,y:byte):byte; assembler;
asm
	les  di,sourceBuffer
	xor  al,al
	mov  ah,y
	shr  ax,1
	mov  bx,ax
	shr  ax,1
	shr  ax,1
	add  bx,ax

	xor  dh,dh
	mov  dl,x
	add  bx,dx
	shr  bx,1
	add  di,bx

	mov  al,es:[di]

	and  dx,$01
	jz   @xEven

	mov  cl,4
	shr  al,cl

@xEven:
	and  al,$0F
end;


procedure hLine(x,y,disX:byte; color:byte); assembler;
asm

	mov  ah,y
	xor  al,al
	shr  ax,1
	shr  ax,1
	mov  di,ax
	shr  ax,1
	shr  ax,1
	add  di,ax

	xor  ah,ah
	mov  al,x
	mov  bx,ax
	shr  ax,1
	add  di,ax

	xor  dh,dh
	mov  dl,disX

	mov  ah,color
	and  ah,$0F
	mov  bh,ah
	mov  cl,4
	shl  bh,cl

	push  ds
	mov		cx,$B800
	mov		es,cx
	add   cx,$0200
	mov   ds,cx

	test bl,$0001
	jz   @longRun

	mov  al,es:[di]
	and  al,$F0
	or   al,ah
	mov  ds:[di],al
	stosb
	dec  dx
	jz   @done


@longrun:
	cmp  dx,2
	jl   @lastPixel
	mov  al,ah
	or   al,bh
	mov  cx,dx
	shr  cx,1
@longLoop:
	mov  ds:[di],al
	stosb
	loop @longLoop
	and  dx,$0001
	jz   @done

@lastPixel:
	mov  al,es:[di]
	and  al,$0F
	or   al,bh
	mov  es:[di],al
	mov  ds:[di],al
@done:
	pop  ds
end;

procedure vLine(x,y,disY:byte; color:byte); assembler;
asm
	mov  ah,y
	xor  al,al
	shr  ax,1
	shr  ax,1
	mov  di,ax
	shr  ax,1
	shr  ax,1
	add  di,ax

	xor  ah,ah
	mov  al,x
	mov  dx,ax
	shr  ax,1
	add  di,ax

	mov  bl,color
	and  bl,$0F
	mov  bh,$F0
	{
		It's faster to set BH and then change it on condition
		than it is to add all sorts of test and jump logic to
		only set $F0 as needed
	}
	test dx,$0001
	jnz   @vSetDistance

	mov  cl,4
	shl  bl,cl
	mov  bh,$0F

@vSetDistance:
	xor  ch,ch
	mov  cl,disY

	mov  ax,$B800
	mov  es,ax
	add  ax,$0200
	mov  dx,ds
	mov  ds,ax

@vLineLoop:
	mov  al,es:[di]
	and  al,bh
	or   al,bl
	mov  es:[di],al
	mov  ds:[di],al
	add  di,$50
	loop @vLineLoop

	mov  ds,dx
end;

procedure tg_bar(sx,sy,ex,ey:integer; color:byte);
var
	y,distance:integer;
begin
	distance:=(ex-sx);
	if (distance=0) then begin
		vLine(sx,sy,(ey-sy)+1,color);
	end else begin
		y:=sy;
		inc(distance);
		while (y<=ey) do begin
			hLine(sx,y,distance,color);
			inc(y);
		end;
	end;
end;

procedure tg_rectangle(sx,sy,ex,ey:integer; color:byte);
var
	distance:integer;
begin
	distance:=(ex-sx)+1;
	hLine(sx,sy,distance,color);
	hLine(sx,ey,distance,color);
	distance:=(ey-sy)+1;
	vLine(sx,sy,distance,color);
	vLine(ex,sy,distance,color);
end;


procedure tg_tile3(x,y:byte; tileSet:pointer; tile:byte); assembler;
asm
	{ set video address for STOSW }
	mov  ax,$B800
	mov  es,ax

	mov  ah,y
	xor  al,al
	shr  ax,1
	shr  ax,1
	mov  di,ax
	shr  ax,1
	shr  ax,1
	add  di,ax

	xor  ah,ah
	mov  al,x
	mov  cx,ax
	shr  ax,1
	add  di,ax

	{ calculate tile offset }
	xor  ax,ax
	mov  al,tile
	shl  ax,1
	shl  ax,1
	shl  ax,1
	mov  bx,ax
	add  ax,bx
	add  ax,bx
	and  cx,1
	jnz  @oddtile
	add  ax,12
@oddTile:
	mov  dx,ds
	lds  si,tileSet
	add  si,ax

	mov  cx,3

@tileLoop:

	lodsw
	mov  bx,es:[di]
	and  bl,ah
	or   bl,al
	lodsw
	and  bh,ah
	or   bh,al
	mov  ax,bx
	mov  es:[di],ax
	add  di,$2000
	stosw
	sub  di,$1FB2

	loop @tileLoop

	mov ds,dx
end;

procedure tg_tile5(x,y:byte; tileSet:pointer; tile:byte); assembler;
asm
	{ set video address for STOSW }
	mov  ax,$B800
	mov  es,ax

	mov  ah,y
	xor  al,al
	shr  ax,1
	shr  ax,1
	mov  di,ax
	shr  ax,1
	shr  ax,1
	add  di,ax

	xor  ah,ah
	mov  al,x
	mov  cx,ax
	shr  ax,1
	add  di,ax

	{ calculate tile offset }
	xor  ax,ax
	mov  al,tile
	mov  bx,60
	mul  bx
	and  cx,1

	jnz  @oddtile
	add  ax,30

@oddTile:
	push ds

	lds  si,tileSet
	add  si,ax

	mov  dx,$2000
	mov  cx,5

@tileLoop:

	lodsw
	mov  bx,es:[di]
	and  bl,ah
	or   bl,al
	lodsw
	and  bh,ah
	or   bh,al
	mov  ax,bx
	mov  es:[di],ax
	add  di,dx
	stosw

	lodsw
	mov  bl,es:[di]
	and  bl,ah
	or   bl,al
	mov  al,bl
	mov  es:[di],al
	sub  di,dx
	stosb
	add  di,77

	loop @tileLoop

	pop  ds
end;

procedure tg_writeChar(x,y,chr,color:byte; data:pFontData); assembler;
asm
	mov  ax,$B800
	mov  es,ax { es == vid segment }

	mov  ah,y
	xor  al,al
	shr  ax,1
	shr  ax,1
	mov  di,ax
	shr  ax,1
	shr  ax,1
	add  di,ax

	mov  al,x
	xor  ah,ah
	mov  ch,al  { ch now holds x for later }
	shr  ax,1
	add  di,ax


	xor  ah,ah
	mov  al,chr
	shl  ax,1
	shl  ax,1
	shl  ax,1   { ax = chr * 8 }


	mov  bl,color
	and  bl,$0F
	mov  bh,bl
	mov  cl,4
	shl  bh,cl { bx == cccc:0000:0000:cccc }

	push ds
	lds  si,data
	add  si,ax  { now pointing at font map offset }

	mov  cl,ch
	and  cl,1   { amount to shift the bitplanes }
	mov  ch,$08
	mov  dx,$F00F

@loopRow:
	lodsb
	shr  al,cl { align to bitplane }
	mov  ah,al { move to ah so al is free for stosb's }

	mov  al,es:[di]
	rcl  ah,1
	jnc  @skip1
	and  al,dl
	or   al,bh
@skip1:
	rcl  ah,1
	jnc  @skip2
	and  al,dh
	or   al,bl
@skip2:
	mov  es:[di],al
	add  di,$2000
	stosb

	mov  al,es:[di]
	rcl  ah,1
	jnc  @skip3
	and  al,dl
	or   al,bh
@skip3:
	rcl  ah,1
	jnc  @skip4
	and  al,dh
	or   al,bl
@skip4:
	mov  es:[di],al
	sub  di,$2000
	stosb

	mov  al,es:[di]
	rcl  ah,1
	jnc  @skip5
	and  al,dl
	or   al,bh
@skip5:
	rcl  ah,1
	jnc  @skip6
	and  al,dh
	or   al,bl
@skip6:
	mov  es:[di],al
	add  di,$2000
	stosb

	mov  al,es:[di]
	rcl  ah,1
	jnc  @skip7
	and  al,dl
	or   al,bh
@skip7:
	rcl  ah,1
	jnc  @skip8
	and  al,dh
	or   al,bl
@skip8:
	mov  es:[di],al
	sub  di,$2000
	stosb
	add  di,76

	dec  ch
	jnz  @loopRow

	pop  ds

end; {tg_writeChar color only }

procedure tg_writeNum(x,y,digit,color,bgColor:byte; data:pFontData); assembler;
asm
	mov  ax,$B800
	mov  es,ax { es == vid segment }

	mov  ah,y
	xor  al,al
	shr  ax,1
	shr  ax,1
	mov  di,ax
	shr  ax,1
	shr  ax,1
	add  di,ax

	mov  al,x
	xor  ah,ah
	mov  ch,al  { ch now holds x for later }
	shr  ax,1
	add  di,ax

	xor  ah,ah
	mov  al,digit
	add  al,$30
	shl  ax,1
	shl  ax,1
	shl  ax,1   { ax = chr * 8 }

	mov  dl,bgcolor
	and  dl,$0F
	mov  dh,dl
	mov  cl,4
	shl  dh,cl
	or   dl,dh

	mov  bl,color
	and  bl,$0F
	mov  bh,bl
	mov  cl,4
	shl  bh,cl { bx == cccc:0000:0000:cccc }

	push ds
	lds  si,data
	add  si,ax  { now pointing at font map offset }

	mov  cl,dl
	mov  ch,$05
	mov  dx,$F00F

@loopRow:
	lodsb
	mov  ah,al

	mov  al,cl
	rcl  ah,1
	jnc  @skip1
	and  al,dl
	or   al,bh
@skip1:
	rcl  ah,1
	jnc  @skip2
	and  al,dh
	or   al,bl
@skip2:
	mov  es:[di],al
	add  di,$2000
	stosb

	mov  al,cl
	rcl  ah,1
	jnc  @skip3
	and  al,dl
	or   al,bh
@skip3:
	rcl  ah,1
	jnc  @skip4
	and  al,dh
	or   al,bl
@skip4:
	mov  es:[di],al
	sub  di,$2000
	stosb
	add  di,78

	dec  ch
	jnz  @loopRow

	pop  ds

end; { tg_writechar color,bgcolor }



procedure buffer_sourceClear(color:byte); assembler;
asm
	les  di,sourceBuffer
	mov  al,color
	mov  ah,al
	mov  cl,4
	shl  al,cl
	or   al,ah
	mov  ah,al
	mov  cx,4000
	rep  stosw
end;

procedure buffer_copySource2Dest; assembler;
asm
	mov  dx,ds { faster than a push/pop to just use dx }
	mov  cx,4000
	les  di,destBuffer
	lds  si,sourceBuffer
	rep  movsw
	mov  ds,dx
end;

procedure buffer_copyDest2Screen; assembler;
asm
	mov  ax,$B800
	mov  es,ax
	xor  di,di
	mov  dx,ds { faster than a push/pop to just use dx }
	lds  si,destBuffer
	mov  bx,si
	mov  cx,4000
	rep  movsw
	mov  di,$2000
	mov  si,bx
	mov  cx,4000
	rep  movsw
	mov  ds,dx
end;

{
	restore copies a 8x8 section from sourceBuffer to destBuffer
	ON THE BYTE BOUNDARY!!!
}
procedure buffer_copySourceDest8x8(x,y:byte); assembler;
asm
	mov  ah,y
	xor  al,al
	shr  ax,1  { =y*128 }
	mov  bx,ax
	shr  ax,1
	shr  ax,1  { =y*32 }
	add  bx,ax { 32+128 = 160 }
	shr  bx,1  { =y*80! }

	mov  al,x
	xor  ah,ah
	shr  ax,1
  add  bx,ax { bx now holds starting offset }

  les  di,destBuffer
  mov  dx,ds
	lds  si,sourceBuffer
	add  di,bx
	add  si,bx

	mov  bx,$004C
	mov  cx,9

@loop:
	movsw
	movsw
	add  si,bx
	add  di,bx
	loop @loop

	mov  ds,dx
end;

{
	show copies an 4x3 section from destBuffer to display
	ON THE BYTE BOUNDARY!!!
}
procedure buffer_show4x3(x,y:byte); assembler;
asm
	mov  ax,$B800
	mov  es,ax

	mov  ah,y
	xor  al,al
	shr  ax,1
	shr  ax,1
	mov  di,ax
	shr  ax,1
	shr  ax,1
	add  di,ax

	xor  ah,ah
	mov  al,x
	mov  bx,ax
	shr  ax,1
	add  di,ax

	mov  bx,di

	mov  dx,ds
	lds  si,destBuffer
	add  si,bx
	mov  cx,80
	mov  bx,$2000

	mov  ax,ds:[si]
	add  si,cx
	mov  es:[di],ax
	add  di,bx
	mov  es:[di],ax
	sub  di,bx

	mov  ax,ds:[si]
	add  si,cx
	mov  es:[di],ax
	add  di,bx
	mov  es:[di],ax
	sub  di,bx

	mov  ax,ds:[si]
	mov  es:[di],ax
	add  di,bx
	mov  es:[di],ax

	mov  ds,dx
end;

{
	show copies an 8x6 section from destBuffer to display
	ON THE BYTE BOUNDARY!!!
}
procedure buffer_show8x6(x,y:byte); assembler;
asm
	mov  ax,$B800
	mov  es,ax

	mov  ah,y
	xor  al,al
	shr  ax,1
	shr  ax,1
	mov  di,ax
	shr  ax,1
	shr  ax,1
	add  di,ax

	xor  ah,ah
	mov  al,x
	mov  bx,ax
	shr  ax,1
	add  di,ax

	push ds
	lds  si,destBuffer
	add  si,di

	mov  cx,8
	mov  bx,$2000
	mov  dx,76

@loop:

	lodsw
	mov  es:[di],ax
	add  di,bx
	stosw

	lodsw
	mov  es:[di],ax
	sub  di,bx
	stosw

	add  di,dx
	add  si,dx

	loop @loop

	pop ds
end;


{
	draws a 3x3 tile to the dest buffer
}
procedure buffer_tile3(x,y:byte; tileSet:pointer; tile:byte); assembler;
asm
	{ set video address for STOSW }
	mov  ah,y
	xor  al,al
	shr  ax,1
	mov  bx,ax
	shr  ax,1
	shr  ax,1
	add  bx,ax
	mov  al,x
	xor  ah,ah
	mov  cx,ax { cx holds x }
	add  bx,ax
	shr  bx,1  { bx holds *80 }

	les  di,destBuffer
	add  di,bx

	{ calculate tile offset }
	mov  al,tile
	xor  ah,ah
	shl  ax,1
	shl  ax,1
	shl  ax,1
	mov  bx,ax
	add  ax,bx
	add  ax,bx

	and  cx,1
	jnz  @oddtile
	add  ax,12
@oddTile:
	mov  dx,ds
	lds  si,tileSet
	add  si,ax

	mov  bx,$004E
{
	Unrolling this may have been silly, but it's fast
}
	lodsw
	and  ah,es:[di]
	or   al,ah
	stosb

	lodsw
	and  ah,es:[di]
	or   al,ah
	stosb

	add di,bx

	lodsw
	and  ah,es:[di]
	or   al,ah
	stosb

	lodsw
	and  ah,es:[di]
	or   al,ah
	stosb

	add di,bx

	lodsw
	and  ah,es:[di]
	or   al,ah
	stosb

	lodsw
	and  ah,es:[di]
	or   al,ah
	stosb

	mov ds,dx
end;

{
	draws a 5x5 tile to the dest buffer
}
procedure buffer_tile5(x,y:byte; tileSet:pointer; tile:byte); assembler;
asm
	{ set video address for STOSW }
	mov  ah,y
	xor  al,al
	shr  ax,1  { =*128 }
	mov  bx,ax
	shr  ax,1
	shr  ax,1  { =*32 }
	add  bx,ax
	mov  al,x
	xor  ah,ah
	mov  cx,ax { cx holds x }
	add  bx,ax
	shr  bx,1  { bx holds *80 }
	les  di,destBuffer
	add  di,bx

	{ calculate tile offset }
	xor  ax,ax
	mov  al,tile
	mov  bx,60
	mul  bx
	and  cx,1
	jnz  @oddtile
	add  ax,30
@oddTile:
	mov  dx,ds
	lds  si,tileSet
	add  si,ax

	mov  cx,5
	mov  bx,$004D

{ I'm NOT unrolling this one! }

@loop:

	lodsw
	and  ah,es:[di]
	or   al,ah
	stosb

	lodsw
	and  ah,es:[di]
	or   al,ah
	stosb

	lodsw
	and  ah,es:[di]
	or   al,ah
	stosb

	add  di,bx

	loop @loop

	mov ds,dx
end;

{
	erases a 4x3 area on the byte boundary from sourceBuffer
	byte-bound
}
procedure buffer_nullTile4x3(x,y:byte); assembler;
asm
	{ set video address for STOSW }
	les  di,sourceBuffer

	mov  ah,y
	xor  al,al
	shr  ax,1
	mov  bx,ax
	shr  ax,1
	shr  ax,1
	add  bx,ax

	mov  al,x
	xor  ah,ah
	add  bx,ax

	shr  bx,1  { bx holds *80 }
	add  di,bx

	xor  ax,ax
	mov  es:[di],ax
	mov  es:[di+$50],ax
	mov  es:[di+$A0],ax
end;

procedure buffer_showSprites;
var
	tSprite:pSprite;
begin
	tSprite:=firstSprite;
	buffer_sourceBackground;
	while not(tSprite=nil) do with tSprite^ do begin
		bufferShow(currentX,currentY);
		oldX:=currentX;
		oldY:=currentY;
		tSprite:=next;
	end;
end;

procedure buffer_hideSprites;
var
	tSprite:pSprite;
begin
	buffer_sourceBackground;
	tSprite:=firstSprite;
	while not(tSprite=nil) do with tSprite^ do begin
		buffer_copySourceDest8x8(oldX,oldY);
		tSprite:=next;
	end;
	tSprite:=firstSprite;
	while not(tSprite=nil) do with tSprite^ do begin
		bufferShow(oldX,oldY);
		tSprite:=next;
	end;
end;

procedure buffer_updateSprites;
var
	tSprite:pSprite;
	pUpdate:pUpdateBlock;
begin
	buffer_sourceBackground;
	tSprite:=firstSprite;
	while not(tSprite=nil) do with tSprite^ do begin
		bufferWriteTile(currentX,currentY,sTiles^.dataStart,currentTile);
		tSprite:=next;
	end;

	tSprite:=firstSprite;
	while not(tSprite=nil) do with tSprite^ do begin
		if currentX<oldX then bufferX:=currentX else bufferX:=oldX;
		oldX:=currentX;
		if currentY<oldY then bufferY:=currentY else bufferY:=oldY;
		oldY:=currentY;
		bufferShow(bufferX,bufferY);
		tSprite:=next;
	end;

	pUpdate:=updateList.first;
	while not(pUpdate=nil) do with pUpdate^ do begin
		buffer_Show4x3(x,y);
		pUpdate:=next;
	end;

	tSprite:=firstSprite;
	while not(tSprite=nil) do with tSprite^ do begin
		buffer_copySourceDest8x8(currentX,currentY);
		tSprite:=next;
	end;

	pUpdate:=updateList.first;
	while not(pUpdate=nil) do with pUpdate^ do begin
		buffer_copySourceDest8x8(x,y);
		pUpdate:=next;
	end;
end;

procedure buffer_sourceBackground;
begin
	sourceBuffer:=backgroundBuffer;
	destBuffer:=renderBuffer;
end;

procedure buffer_sourceRender;
begin
	sourceBuffer:=renderBuffer;
	destBuffer:=backgroundBuffer;
end;

procedure tg_clear(color:byte); assembler;
asm { clear video memory }
	mov  ax,$B800
	mov  es,ax
	xor  di,di
	mov  al,color
	mov  ah,al
	and  ah,$0F
	mov  cl,4
	shl  al,cl
	or   al,ah
	mov  ah,al
	mov  cx,4000
	rep  stosw
end;

procedure tg_init;
var
	t:word;
begin
	oldMode:=getVideoMode;
	setVideoMode($08);
	buffer_sourceBackground;
	buffer_sourceClear(0);
	buffer_copySource2Dest;
	buffer_copyDest2Screen;
	textGraphOn:=true;
end;

procedure tg_term;
begin
	setVideoMode(oldMode);
	textGraphOn:=false;
end;

{
	adding our own custom exitproc is a safeguard so that
	if someone using this library forgets to call tg_term
	the program will still exit to text mode. I WISH that
	execution breaks or failures would call exitproc.

	A more robust unit would add a custom error handler.
}
procedure tg_exitProc; far;
begin
	if textGraphOn then tg_term;
	discardSprites;
	dispose(renderBuffer);
	dispose(backgroundBuffer);
	updateList.term;
	exitProc:=oldTgExitProc;
end;

begin
	write('Video card detected: ');
	videoCard:=detectCard;
	writeln(videoCardName[videoCard]);
	case videoCard of
		videoCard_pcJr,
		videoCard_tandy1000,
		videoCard_tandySLTL:begin
			firstSprite:=nil;
			lastSprite:=nil;
			textSegment:=$B800;
			textGraphOn:=false;
			updateList.init;
			new(backgroundBuffer);
			new(renderBuffer);
			oldTgExitProc:=exitProc;
			exitProc:=@tg_exitProc;
		end;
		else begin
			writeln('This program requires Tandy 1000 or PC Jr. Graphics');
			halt;
		end;
	end;
end.