unit tdySound;

interface

uses
	jfunc,
	timer; { timer included for PIT constants }

type
	tSynthLine=object
		dataStart,dataEnd,dataOffset:pChar;
		note,noteTimer,
		volume,tempo,voice,priority,
		tempoCalc:integer;
		done:boolean;
		constructor init(data:pChar; wTempo,wVoice,wPriority:word);
		procedure reset;
		procedure playStep;
	end;

procedure startSound;
procedure killSound;
procedure outFreq(iChannel,iFreq,iLevel,iPriority:word);
procedure killVoices;

implementation

const
	tandyFreq=3579545;
	tandyDivider=tandyFreq/32;
	tandyIntDivider=round(tandyDivider);

	noteOffset:array[65..71] of word=(0,2,3,5,7,8,10);

	notes:array[1..84] of word=(
	{   C   C#,Db   D   D#,Eb   E     F   F#,Gb   G   G#,Ab   A   A#,Bb   B   }
		0065, 0070, 0073, 0078, 0082, 0087, 0093, 0098, 0104, 0110, 0117, 0123,
		0131, 0139, 0147, 0156, 0165, 0175, 0185, 0196, 0208, 0220, 0233, 0247,
		0262, 0277, 0294, 0311, 0330, 0349, 0370, 0392, 0415, 0440, 0466, 0494,
		0523, 0554, 0587, 0622, 0659, 0698, 0740, 0784, 0831, 0880, 0932, 0987,
		1047, 1109, 1175, 1245, 1329, 1397, 1480, 1568, 1661, 1760, 1865, 1976,
		2093, 2217, 2349, 2489, 2637, 2794, 2960, 3136, 3322, 3520, 3729, 3951,
		4186, 4435, 4699, 4978, 5274, 5588, 5920, 6272, 6645, 7040, 7459, 7902
	);

var
	soundActive:boolean;
	oldExitProc:pointer;

procedure tandySound(voice:byte; tone:word; level:byte); assembler;
asm
	mov  dx,$C0
	mov  cl,voice
	mov  al,$10
	shl  al,cl
	and  al,$60
	or   al,$80
	mov  ch,al { save channel for level set }
	mov  bx,tone
	mov  ah,bl
	and  ah,$0F
	or   al,ah
	out  dx,al
	mov  cl,4
	shr  bx,cl
	mov  al,bl
	out  dx,al
	mov  al,level
	and  al,$0F
	or   al,ch
	or   al,$10
	out  dx,al
end;

function tandyDetect:boolean; assembler;
asm
	mov  bx,$FFFF
	mov  es,bx
	mov  di,$000E
	mov  al,$FD
	cmp  es:[di],al
	jne  @notJr
	mov  ax,bx
	ret
@notJr:
	mov  al,$FF
	cmp  es:[di],al
	jne  @notTandy
	mov  ax,$FC00
	mov  es,ax
	xor  di,di
	mov  al,$21
	cmp  es:[di],al
	jne  @notTandy
	mov  ax,bx
	ret
@notTandy:
	xor  ax,ax
end;

procedure outFreq(iChannel,iFreq,iLevel,iPriority:word);
begin
	if (iFreq>0) then begin
		tandySound(
			iChannel,
			tandyIntDivider div iFreq,
			15-iLevel
		);
	end else tandySound(iChannel,0,15);
end;

procedure killVoices;
var
	t:word;
begin
	for t:=0 to 2 do outFreq(t,0,0,0);
end;

constructor tSynthLine.init;
var
	b:byte;
begin
	b:=byte(data^);
	dataStart:=data;
	inc(dataStart);
	dataEnd:=dataStart;
	inc(dataEnd,b);
	dataOffset:=dataStart;
	tempo:=wTempo;
	voice:=wVoice;
	priority:=wPriority;
	tempoCalc:=900 div tempo; { max accuracy 32th notes... kinda }
	reset;
end;

procedure tSynthLine.reset;
begin
	dataOffset:=dataStart;
	volume:=32;
	noteTimer:=0;
	done:=false;
	outFreq(voice,0,0,0);
end;

procedure tSynthLine.playStep;
begin
	if (noteTimer>0) then begin
		dec(noteTimer);
	end else if not(dataOffset=dataEnd) then begin
		case dataOffset^ of
			'0'..'9':begin
				note:=(ord(dataOffset^)-48)*12-3;
				inc(dataOffset);
				case dataOffset^ of
					'A'..'G':begin
						note:=note+noteOffset[ord(dataOffset^)];
						inc(dataOffset);
						if (dataOffset^='#') then begin
							inc(note);
							inc(dataOffset);
						end;
						if (dataOffset^='b') then begin
							dec(note);
							inc(dataOffset);
						end;
						case dataOffset^ of
							'0'..'9':noteTimer:=(ord(dataOffset^)-48)*tempoCalc-1;
							'A'..'Z':noteTimer:=(ord(dataOffset^)-55)*tempoCalc-1;
							'a'..'z':noteTimer:=(ord(dataOffset^)-87)*tempoCalc-1;
						end;
						inc(dataOffset);
						if (note>0) and (note<85) then begin
							note:=notes[note];
							outFreq(voice,note,volume,priority);
						end else outFreq(voice,0,0,0);
					end;
				end;
			end; { note }
			'v','V':begin
				inc(dataOffset);
				case dataOffset^ of
					'0'..'9':volume:=(ord(dataOffset^)-48);
					'A'..'F':volume:=(ord(dataOffset^)-55);
					'a'..'f':volume:=(ord(dataOffset^)-87);
					else volume:=0;
				end;
				outFreq(voice,note,volume,priority);
				inc(dataOffset);
			end; { volume }
			'r','R':begin
				outFreq(voice,0,0,0);
				inc(dataOffset);
				case dataOffset^ of
					'0'..'9':noteTimer:=(ord(dataOffset^)-48)*tempoCalc-1;
					'A'..'Z':noteTimer:=(ord(dataOffset^)-55)*tempoCalc-1;
					'a'..'z':noteTimer:=(ord(dataOffset^)-87)*tempoCalc-1;
				end;
				inc(dataOffset);
			end; { rest }
		end; { case }
		if dataOffset^=',' then inc(dataOffset);
	end else if not(done) then begin
		outFreq(voice,0,0,0);
		done:=true;
	end;
end;

procedure startSound;
begin
	asm
		{ 1k/jr }
		mov  ax,$8003
		int  $1A
	 	{ manually set for older machines just in case }
		mov  al,$6C
		out  $61,al
		mov  ax,$8003
		int  $1A
		mov  al,1
		mov  soundActive,al
	end;
	killVoices;
end;

procedure killSound;
begin
	killVoices;
	asm
		mov  dx,$C0
		mov  al,$9F
		out  dx,al
		mov  al,$BF
		out  dx,al
		mov  al,$DF
		out  dx,al
		xor  al,al
		mov  soundActive,al
	end;
end;

procedure safeExit; far;
begin
	if soundActive then killSound;
	exitProc:=oldExitProc;
end;

begin
	soundActive:=false;
	if not(tandyDetect) then begin
		writeln('This program requires Tandy or PC Jr. Sound');
		halt;
	end;
	oldExitProc:=exitproc;
	exitproc:=@safeExit;
end.