/*
 * This file is part of the zmiy project.
 * Copyright (C) 2013 Mateusz Viste
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <dos.h>   /* provides int86() along with the union REGS type */
#include <time.h>  /* clock() */
#include "io.h"    /* include self headers for control */



void cursor_set(int startscanline, int endscanline) {
  union REGS regs;
  regs.h.ah = 0x01;
  regs.h.ch = startscanline;
  regs.h.cl = endscanline;
  int86(0x10, &regs, &regs);
}


void locate(int row, int column) {
  union REGS regs;
  regs.h.ah = 0x02;
  regs.h.bh = 0;
  regs.h.dh = row;
  regs.h.dl = column;
  int86(0x10, &regs, &regs);
}


void printchar(char c, int attr) {
  union REGS regs;
  regs.h.ah = 0x09; /* Write char and attribute */
  regs.h.al = c;
  regs.h.bh = 0;    /* display page */
  regs.h.bl = attr;
  regs.x.cx = 1;    /* write it 1 time */
  int86(0x10, &regs, &regs);
}



/* waits for a keypress and return it. Returns 0 for extended keystroke, then
   function must be called again to return scan code. */
int getkey(void) {
  union REGS regs;
  int res;
  regs.h.ah = 0x08;  
  int86(0x21, &regs, &regs);
  res = regs.h.al;
  if (res == 0) { /* extended key - poll again */
    regs.h.ah = 0x08;  
    int86(0x21, &regs, &regs);
    res = regs.h.al | 0x100;
  }
  return(res);
}


/* poll the keyboard, and return the next input key if any, or -1 */
int getkey_ifany(void) {
  union REGS regs;
  regs.h.ah = 0x0B;  
  int86(0x21, &regs, &regs);
  if (regs.h.al == 0xFF) {
      return(getkey());
    } else {
      return(-1);
  }
}


void setvideomode_80(int rows) {
  union REGS regs;
  /* first set up a 80x25 video mode */
  regs.h.ah = 0x00;  /* set video mode */
  regs.h.al = 0x03;  /* 80x25 */
  int86(0x10, &regs, &regs);
  if (rows == 50) {
    /* now select a 8x8 font */
    regs.h.ah = 0x11;
    regs.h.al = 0x12;
    regs.h.bl = 0;
    int86(0x10, &regs, &regs);
  }
}


#define INT_HICLOCKS_PER_SEC ((unsigned long)CLK_TCK << 8)
#define hiclock() ((unsigned long)clock() << 8)

void milisleep_apm(int ms) { /* Turbo C provides a 'delay()' function for */
  clock_t lMillisec = ms;    /* this, but I am trying to be "green" here  */
  static clock_t finish = 0; /* and not waste CPU in a dumb busy loop.    */
  clock_t curclock = hiclock();
  union REGS regs;
  /* set it to be in the future */
  for (;;) {
    finish += (lMillisec * INT_HICLOCKS_PER_SEC) / 1000;
    if (finish > curclock) break; /* detect anomalies */
    finish = curclock;
  }
  while (finish > hiclock()) {
    regs.x.ax = 0x5305; /* APM v1.0+ CPU IDLE */
    int86(0x15, &regs, &regs);
  }
}


/* a simple wrapper on the Turbo C 'delay' function. */
void milisleep_delay(int ms) {
  delay(ms);
}


/* The function below is relying on the BIOS 'WAIT' service, but it doesn't work on
 * some emulators: makes DOSBox crazy, and makes DOSemu keyboard controls laggy */
void milisleep_bios(int ms) {
  union REGS regs;
  long us = ms;
  long us_hi, us_lo;
  us *= 1000; /* convert ms into us */
  us_hi = (us >> 16) & 0xFFFF;
  us_lo = us & 0xFFFF;
  regs.h.ah = 0x86;    /* 0x86 is the BIOS WAIT service */
  regs.x.cx = (unsigned int)us_hi;  /* CX contains the high word */
  regs.x.dx = (unsigned int)us_lo;  /* DX contains the low word */
  int86(0x15, &regs, &regs);
}
