/*
 * Zmiy is a snake-like game for DOS and 8086.
 * Copyright (C) 2013 Mateusz Viste
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "io.h"
#include "levels.h"


#define PVER "0.85"
#define PDATE "2013"


/* Color scheme       BG FG*/
#define COLOR_BAR      0x70
#define COLOR_SNAKE1   0xE0
#define COLOR_SNAKE2   0x00
#define COLOR_WALL     0x60
#define COLOR_ITEM     0x1A
#define COLOR_BG       0x11
#define COLOR_MSG_GOOD 0x2F
#define COLOR_MSG_BAD  0x4F
#define COLOR_MSG_INFO 0x8F

#define MSG_GOODJOB  0
#define MSG_TRYAGAIN 1
#define MSG_GAMEOVER 2
#define MSG_PAUSE    3

/* This global variable is a function pointer to the milisleep() implementation */
void (*milisleep)(int);


void PrintScoreBar(struct gamestruct *game) {
  int x = 0;
  char tmpstr[32];
  char *tmpstrptr;
  /* draw the 'SCORE' string */
  for (tmpstrptr = " SCORE: "; *tmpstrptr != 0; tmpstrptr += 1) {
    locate(0, x++);
    printchar(*tmpstrptr, COLOR_BAR);
  }
  /* draw the score itself */
  ltoa(game->score, tmpstr, 10);
  for (tmpstrptr = tmpstr; *tmpstrptr != 0; tmpstrptr += 1) {
    locate(0, x++);
    printchar(*tmpstrptr, COLOR_BAR);
  }
  /* Draw some background */
  for (; x < 35; x++) {
    locate(0, x);
    printchar(' ', COLOR_BAR);
  }
  /* draw the 'LEVEL' string */
  for (tmpstrptr = "LEVEL: "; *tmpstrptr != 0; tmpstrptr += 1) {
    locate(0, x++);
    printchar(*tmpstrptr, COLOR_BAR);
  }
  /* draw the level itself */
  itoa(game->level, tmpstr, 10);
  tmpstrptr = tmpstr;
  for (tmpstrptr = tmpstr; *tmpstrptr != 0; tmpstrptr += 1) {
    locate(0, x++);
    printchar(*tmpstrptr, COLOR_BAR);
  }
  /* Draw some background */
  for (; x < 71; x++) {
    locate(0, x);
    printchar(' ', COLOR_BAR);
  }
  /* draw the 'SPEED' string */
  for (tmpstrptr = "SPEED: "; *tmpstrptr != 0; tmpstrptr += 1) {
    locate(0, x++);
    printchar(*tmpstrptr, COLOR_BAR);
  }
  /* draw the speed itself */
  itoa(game->speed, tmpstr, 10);
  tmpstrptr = tmpstr;
  for (tmpstrptr = tmpstr; *tmpstrptr != 0; tmpstrptr += 1) {
    locate(0, x++);
    printchar(*tmpstrptr, COLOR_BAR);
  }
  /* draw the rest of the background */
  for (; x < 80; x++) {
    locate(0, x);
    printchar(' ', COLOR_BAR);
  }
}


/* flush the content of the keyboard buffer */
void keybflush(void) {
  while (getkey_ifany() != -1); /* flush the keyboard buffer */
}


/* Output a message on screen */
void OutMessage(int msgid) {
  int x, linenum;
  int msglen;
  int msghoffset, msgvoffset;
  int msgcolor;
  /* define the 'good job' text */
  char *line1_goodjob = "  XXX  XX   XX  XXX       X  XX  XXX   X ";
  char *line2_goodjob = " X    X  X X  X X  X      X X  X X  X  X ";
  char *line3_goodjob = " X XX X  X X  X X  X      X X  X XXX   X ";
  char *line4_goodjob = " X  X X  X X  X X  X   X  X X  X X  X    ";
  char *line5_goodjob = "  XXX  XX   XX  XXX     XX   XX  XXX   X ";
  /* define the 'game over' text */
  char *line1_gameover = "  XXX  XX   X X  XXX    XX  X  X XXX XXX   X ";
  char *line2_gameover = " X    X  X X X X X     X  X X  X X   X  X  X ";
  char *line3_gameover = " X XX XXXX X X X XXX   X  X X  X XXX XXX   X ";
  char *line4_gameover = " X  X X  X X X X X     X  X X X  X   X  X    ";
  char *line5_gameover = "  XXX X  X X X X XXX    XX   X   XXX X  X  X ";
  /* define the 'try again' text */
  char *line1_tryagain = " XXXXX XXX  X  X    XX   XXX  XX  XXX X  X  X ";
  char *line2_tryagain = "   X   X  X X  X   X  X X    X  X  X  XX X  X ";
  char *line3_tryagain = "   X   XXX   XXX   XXXX X XX XXXX  X  X XX  X ";
  char *line4_tryagain = "   X   X  X    X   X  X X  X X  X  X  X  X    ";
  char *line5_tryagain = "   X   X  X  XX    X  X  XXX X  X XXX X  X  X ";
  /* define the 'pause' text */
  char *line1_pause = " XXX   XX  X  X  XXX XXX ";
  char *line2_pause = " X  X X  X X  X X    X   ";
  char *line3_pause = " XXX  XXXX X  X  XX  XXX ";
  char *line4_pause = " X    X  X X  X    X X   ";
  char *line5_pause = " X    X  X  XX  XXX  XXX ";
  /* link pointers to correct message */
  char *line[5];
  if (msgid == MSG_GOODJOB) {
      line[0] = line1_goodjob;
      line[1] = line2_goodjob;
      line[2] = line3_goodjob;
      line[3] = line4_goodjob;
      line[4] = line5_goodjob;
      msgcolor = COLOR_MSG_GOOD;
    } else if (msgid == MSG_TRYAGAIN) {
      line[0] = line1_tryagain;
      line[1] = line2_tryagain;
      line[2] = line3_tryagain;
      line[3] = line4_tryagain;
      line[4] = line5_tryagain;
      msgcolor = COLOR_MSG_BAD;
    } else if (msgid == MSG_GAMEOVER) {
      line[0] = line1_gameover;
      line[1] = line2_gameover;
      line[2] = line3_gameover;
      line[3] = line4_gameover;
      line[4] = line5_gameover;
      msgcolor = COLOR_MSG_BAD;
    } else if (msgid == MSG_PAUSE) {
      line[0] = line1_pause;
      line[1] = line2_pause;
      line[2] = line3_pause;
      line[3] = line4_pause;
      line[4] = line5_pause;
      msgcolor = COLOR_MSG_INFO;
    } else {
      line[0] = "";
      line[1] = "";
      line[2] = "";
      line[3] = "";
      line[4] = "";
      msgcolor = COLOR_MSG_BAD;
  }
  for (msglen = 0; line[0][msglen] != 0; msglen++);  /* a simpler (and faster) way than strlen() */
  msghoffset = 40 - (msglen / 2);
  msgvoffset = 20;
  for (x = 0; line[0][x] != 0; x++) {
    locate(msgvoffset - 1, msghoffset + x);
    printchar(' ', msgcolor);
    locate(msgvoffset + 5, msghoffset + x);
    printchar(' ', msgcolor);
    for (linenum = 0; linenum < 5; linenum++) {
      locate(msgvoffset + linenum, msghoffset + x);
      if (line[linenum][x] == ' ') {
          printchar(' ', msgcolor);
        } else {
          printchar(' ', msgcolor << 4);
      }
    }
  }
}


void pausegame(struct gamestruct *game) {
  int x, y;
  OutMessage(MSG_PAUSE);
  milisleep(500);
  keybflush();
  getkey();
  milisleep(100);
  /* mark all fields as 'to be refreshed' */
  for (y = 0; y < 50; y++) {
    for (x = 0; x < 80; x++) {
      game->playfield[x][y] |= BLOCK_REFRESH_FLAG;
    }
  }
}


int PlayGame(struct gamestruct *game) {
  int x, y;
  int blockvalue, blocktype;
  int lastkey;
  int exitflag = EXITGAME_SUCCESS;
  keybflush(); /* flush the keyboard buffer */
  PrintScoreBar(game); /* draw the score bar */
  for (;;) {
    for (y = game->playfieldvoffset; y < game->playfieldvoffset + game->playfieldheight; y++) {
      for (x = 0 ; x < game->playfieldwidth; x++) {
        /* extract the TYPE and VALUE of the block */
        blocktype = game->playfield[x][y] & BLOCK_TYPE;
        blockvalue = game->playfield[x][y] & BLOCK_VALUE;
        /* if it's part of a snake, increment it */
        if ((blocktype == BLOCK_SNAKE1) || (blocktype == BLOCK_SNAKE2)) {
          if (blockvalue >= game->snakelen) {
              game->playfield[x][y] = BLOCK_EMPTY | BLOCK_REFRESH_FLAG;
              blocktype = BLOCK_EMPTY;
              blockvalue = 0;
            } else {
              game->playfield[x][y] += 1;
          }
        }
        /* check if it needs refreshing - and if so, display it */
        if ((game->playfield[x][y] & BLOCK_REFRESH_FLAG) == 0) continue;  /* skip if no refresh needed */
        locate(y, x);
        switch (blocktype) {
          case BLOCK_SNAKE1:
            printchar(' ', COLOR_SNAKE1);
            break;
          case BLOCK_SNAKE2:
            printchar(' ', COLOR_SNAKE2);
            break;
          case BLOCK_WALL:
            printchar(' ', COLOR_WALL);
            break;
          case BLOCK_EMPTY:
            if (blockvalue > 0) {
                printchar('0' + blockvalue, COLOR_ITEM);
              } else {
                printchar(' ', COLOR_BG);
            }
            break;
        }
        /* mark the block as refreshed */
        game->playfield[x][y] &= ~BLOCK_REFRESH_FLAG;
      }
    }

    milisleep(50 + ((10 - game->speed) << 3));
    for (;;) { /* process input keys as long as there are keys in the queue, or an action happen */
      lastkey = getkey_ifany();
      if (lastkey == -1) { /* no key pressed */
          break;
        } else if (lastkey == 0x150) { /* DOWN */
          if ((game->snakedirection == SNAKEDIR_LEFT) || (game->snakedirection == SNAKEDIR_RIGHT)) {
            game->snakedirection = SNAKEDIR_DOWN;
            break;
          }
        } else if (lastkey == 0x148) { /* UP */
          if ((game->snakedirection == SNAKEDIR_LEFT) || (game->snakedirection == SNAKEDIR_RIGHT)) {
            game->snakedirection = SNAKEDIR_UP;
            break;
          }
        } else if (lastkey == 0x14B) { /* LEFT */
          if ((game->snakedirection == SNAKEDIR_UP) || (game->snakedirection == SNAKEDIR_DOWN)) {
            game->snakedirection = SNAKEDIR_LEFT;
            break;
          }
        } else if (lastkey == 0x14D) { /* RIGHT */
          if ((game->snakedirection == SNAKEDIR_UP) || (game->snakedirection == SNAKEDIR_DOWN)) {
            game->snakedirection = SNAKEDIR_RIGHT;
            break;
          }
        } else if (lastkey == 0x1B) { /* ESC */
          exitflag = EXITGAME_ESCAPE;
          break;
        } else if ((lastkey == 0x0D) || (lastkey == 'p') || (lastkey == 'P')) { /* ENTER or 'P' */
          pausegame(game);
          break;
      }
    }
    if (exitflag != 0) break;
    if (game->snakedirection == SNAKEDIR_UP) {
        if (game->snakeposy > game->playfieldvoffset) {
            game->snakeposy -= 1;
          } else {
            game->snakeposy = game->playfieldheight + game->playfieldvoffset - 1;
        }
      } else if (game->snakedirection == SNAKEDIR_RIGHT) {
        if (game->snakeposx + 1 < game->playfieldwidth) {
            game->snakeposx += 1;
          } else {
            game->snakeposx = 0;
        }
      } else if (game->snakedirection == SNAKEDIR_DOWN) {
        if (game->snakeposy + 1 < (game->playfieldheight + game->playfieldvoffset)) {
            game->snakeposy += 1;
          } else {
            game->snakeposy = game->playfieldvoffset;
        }
      } else if (game->snakedirection == SNAKEDIR_LEFT) {
        if (game->snakeposx > 0) {
            game->snakeposx -= 1;
          } else {
            game->snakeposx = game->playfieldwidth - 1;
        }
    }
    if ((game->playfield[game->snakeposx][game->snakeposy] & BLOCK_TYPE) == BLOCK_EMPTY) {
        blockvalue = game->playfield[game->snakeposx][game->snakeposy] & BLOCK_VALUE;
        if (blockvalue > 0) {
          game->score += 10;
          PrintScoreBar(game);
          if (blockvalue == 9) return(EXITGAME_SUCCESS);
          game->snakelen = blockvalue * 20;
          PutItem(game, blockvalue + 1);
        }
        game->playfield[game->snakeposx][game->snakeposy] = (BLOCK_SNAKE1 + 1) | BLOCK_REFRESH_FLAG;
      } else {
        if (game->score >= 100) {
            game->score -= 100;
            exitflag = EXITGAME_RESTART;
          } else {
            exitflag = EXITGAME_GAMEOVER;
        }
    }
  }
  return(exitflag);
}


void AdjustSpeedToScore(struct gamestruct *game) {
  if (game->score >= 2000) {
      game->speed = 9;
    } else if (game->score >= 1500) {
      game->speed = 8;
    } else if (game->score >= 1200) {
      game->speed = 7;
    } else if (game->score >= 900) {
      game->speed = 6;
    } else if (game->score >= 700) {
      game->speed = 5;
    } else if (game->score >= 500) {
      game->speed = 4;
    } else if (game->score >= 300) {
      game->speed = 3;
    } else if (game->score >= 50) {
      game->speed = 2;
    } else {
      game->speed = 1;
  }
}


void showhelp(void) {
  puts("Zmiy v" PVER " Copyright (C) Mateusz Viste " PDATE);
  puts("");
  puts(" /timer=apm    use the 'int15 AX=5305h' APM call for timing (default)");
  puts(" /timer=bios   use the 'int15 AH=86h' BIOS call for timing");
  puts(" /timer=delay  use a busy loop for timing");
  puts("");
}


int main(int argc, char **argv) {
  struct gamestruct *game;
  enum exitstatus exitflag = EXITGAME_SUCCESS;
  int n;

  /* Parse command line params*/
  milisleep = milisleep_apm;  /* use the APM sleep by default */
  for (n = 1; n < argc; n++) {
    if (strcmp(argv[n], "/timer=bios") == 0) {
        milisleep = milisleep_bios;
      } else if (strcmp(argv[n], "/timer=delay") == 0) {
        milisleep = milisleep_delay;
      } else if (strcmp(argv[n], "/timer=apm") == 0) {
        milisleep = milisleep_apm;
      } else {
        showhelp();
        return(0);
    }
  }

  /* allocate memory for the game */
  game = malloc(sizeof(struct gamestruct));
  if (game == NULL) {
    puts("Error: out of memory!");
    return(1);
  }

  setvideomode_80(50); /* set up 80x50 */
  cursor_set(0x0F, 0x0E); /* hide the cursor */
  srand((unsigned int)time(NULL));   /* feed the random generator with a seed */

  game->score = 0;
  game->level = 1;

  for (;;) {
    AdjustSpeedToScore(game);
    LoadLevel(game, game->level);
    exitflag = PlayGame(game);
    if ((exitflag == EXITGAME_GAMEOVER) || (exitflag == EXITGAME_ESCAPE)) break;
    if (exitflag == EXITGAME_SUCCESS) {
        OutMessage(MSG_GOODJOB);
        game->level += 1;
      } else {
        OutMessage(MSG_TRYAGAIN);
    }
    milisleep(500);
    keybflush();
    getkey();
  }

  if (exitflag == EXITGAME_GAMEOVER) {
    OutMessage(MSG_GAMEOVER);
    milisleep(500);
    keybflush();
    getkey();
  }

  free(game);

  setvideomode_80(25);
  puts("Zmiy v" PVER " Copyright (C) Mateusz Viste " PDATE);
  puts("");
  puts("This program is free software: you can redistribute it and/or modify");
  puts("it under the terms of the GNU General Public License as published by");
  puts("the Free Software Foundation, either version 3 of the License, or");
  puts("(at your option) any later version.");
  puts("");
  return(0);
}
