/*
    Sudoku86

  Copyright (c) 2014, Mateusz Viste
  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "keyb.h"      /* keyboard routines */
#include "mouse.h"     /* mouse routines */
#include "video.h"     /* video routines */

#define PVER "1.0.1"
#define PDATE "2014"

#include "font.c"

#define fieldoffx 90
#define fieldoffy 17
#define fieldwidth 22
#define fieldheight 19
#define paneloffx 35
#define paneloffy 11

/* The playfield grid */
#define VGACOL_CELL0_FILL 30
#define VGACOL_CELLB_FILL 28
#define VGACOL_CELL_BORDER 8
#define CGACOL_CELL0_FILL 3
#define CGACOL_CELLB_FILL 1
#define CGACOL_CELL_BORDER 0
/* Left panel with selectable digits */
#define VGACOL_PANELCELL_BORDER 7
#define VGACOL_PANELCELL_FILL 28
#define VGACOL_PANELCELL_SEL_BORDER 6
#define VGACOL_PANELCELL_SEL_FILL 15
#define CGACOL_PANELCELL_BORDER 0
#define CGACOL_PANELCELL_FILL 3
#define CGACOL_PANELCELL_SEL_BORDER 2
#define CGACOL_PANELCELL_SEL_FILL 3

#define RUNFLAG_NODBUF 1
#define RUNFLAG_CGA 2

#define HINTBLOCKED 512
#define HINTREFRESH 1024


struct game_t {
  unsigned char playfield[9][9];
  unsigned short hints[9][9]; /* bitfield with hints */
  int selected;
  unsigned short refreshselectedbar;
};

/* releases a CPU 'slice' */
static void releasecpu(void) {
  union REGS regs;
  regs.x.ax = 0x1680;
  int86(0x2F, &regs, &regs);
}

static void resetgame(struct game_t *game) {
  memset(game, 0, sizeof(struct game_t));  /* set all game data to 0 */
  game->refreshselectedbar = 511; /* mark all buttons of the selection bar to be refreshed */
}

/* checks whether a game is solved. returns 0 if not, non-zero otherwise. */
static int checksolution(struct game_t *game) {
  unsigned short digitsbitfield;
  int x, y, offsetx, offsety;
  /* check that every row and every column contains exactly all digits 1-9 */
  for (y = 0; y < 9; y++) {
    /* check rows */
    digitsbitfield = 0;
    for (x = 0; x < 9; x++) digitsbitfield |= (1 << game->playfield[x][y]);
    if (digitsbitfield != 1022) return(0);
    /* check columns */
    digitsbitfield = 0;
    for (x = 0; x < 9; x++) digitsbitfield |= (1 << game->playfield[y][x]);
    if (digitsbitfield != 1022) return(0);
  }
  /* check that every 3x3 sub-grid contains all digits 1-9 */
  for (offsety = 0; offsety < 9; offsety += 3) {
    for (offsetx = 0; offsetx < 9; offsetx += 3) {
      digitsbitfield = 0;
      for (y = 0; y < 3; y++) {
        for (x = 0; x < 3; x++) {
          digitsbitfield |= (1 << game->playfield[x + offsetx][y + offsety]);
        }
      }
      if (digitsbitfield != 1022) return(0);
    }
  }
  /* if we got here, then the playfield is all good */
  return(1);
}

static void redrawscreen(struct video_handler *videohandler, struct game_t *game) {
  int i, ii, x, y;
  unsigned char *bigfont[9];
  int col_cell_border, col_cellb_fill, col_cell0_fill, col_panelcell_border, col_panelcell_fill, col_panelcell_sel_border, col_panelcell_sel_fill;
  if (videohandler->mode == 0x13) {
      col_cell_border = VGACOL_CELL_BORDER;
      col_cellb_fill = VGACOL_CELLB_FILL;
      col_cell0_fill = VGACOL_CELL0_FILL;
      col_panelcell_sel_border = VGACOL_PANELCELL_SEL_BORDER;
      col_panelcell_sel_fill = VGACOL_PANELCELL_SEL_FILL;
      col_panelcell_border = VGACOL_PANELCELL_BORDER;
      col_panelcell_fill = VGACOL_PANELCELL_FILL;
    } else {
      col_cell_border = CGACOL_CELL_BORDER;
      col_cellb_fill = CGACOL_CELLB_FILL;
      col_cell0_fill = CGACOL_CELL0_FILL;
      col_panelcell_sel_border = CGACOL_PANELCELL_SEL_BORDER;
      col_panelcell_sel_fill = CGACOL_PANELCELL_SEL_FILL;
      col_panelcell_border = CGACOL_PANELCELL_BORDER;
      col_panelcell_fill = CGACOL_PANELCELL_FILL;
  }
  bigfont[0] = big1_pgm;
  bigfont[1] = big2_pgm;
  bigfont[2] = big3_pgm;
  bigfont[3] = big4_pgm;
  bigfont[4] = big5_pgm;
  bigfont[5] = big6_pgm;
  bigfont[6] = big7_pgm;
  bigfont[7] = big8_pgm;
  bigfont[8] = big9_pgm;
  if ((videohandler->flags & VIDEO13H_DBUF) == 0) mouse_hide();

  for (i = 0; i < 9; i++) {
    x = fieldoffx + i * (fieldwidth - 1) + (i / 3);
    for (ii = 0; ii < 9; ii++) {
      /* first of all, check if this field really needs a refresh */
      if ((game->hints[i][ii] & HINTREFRESH) == 0) continue;
      game->hints[i][ii] ^= HINTREFRESH;  /* reset the 'refresh' flag, since we will redraw the field right now */
      /* compute the y coordinate */
      y = fieldoffy + ii * (fieldheight - 1) + (ii / 3);
      /* draw a cell of the grid */
      video_rect(videohandler, x, y, fieldwidth, fieldheight, col_cell_border);
      /* now draw what's inside a single cell */
      if (game->hints[i][ii] & HINTBLOCKED) { /* readonly field */
          video_rectfill(videohandler, x + 1, y + 1, fieldwidth - 2, fieldheight - 2, col_cellb_fill);
          video_putsprite(videohandler, bigfont[game->playfield[i][ii] - 1], x + 7, y + 3, 9, 13, 16, 15, 12);
        } else { /* 'normal' playable field */
          video_rectfill(videohandler, x + 1, y + 1, fieldwidth - 2, fieldheight - 2, col_cell0_fill);
          if ((game->playfield[i][ii] > 0) && (game->playfield[i][ii] < 10)) {
              video_putsprite(videohandler, bigfont[game->playfield[i][ii] - 1], x + 7, y + 3, 9, 13, 16, 15, 255);
            } else if (game->hints != 0) { /* check for hints */
              int nhint;
              for (nhint = 0; nhint < 9; nhint++) {
                if (game->hints[i][ii] & (1 << nhint)) {
                  video_putsprite(videohandler, smallfont[nhint], x + 2 + (nhint % 3) * 6, y + 1 + (nhint / 3) * 6, 4, 5, 16, 15, 255);
                }
              }
          }
      }
    }
  }

  /* draw buttons with selectable digits on the left, if the bar changed */
  for (i = 0; i < 9; i++) {
    if (((game->refreshselectedbar >> i) & 1) == 0) continue; /* skip buttons that don't need a refresh */
    game->refreshselectedbar ^= (1 << i);
    if (game->selected == i + 1) {
        video_rect(videohandler, paneloffx, paneloffy + i * 20, 20, 17, col_panelcell_sel_border);
        video_rectfill(videohandler, paneloffx + 1, paneloffy + 1 + i * 20, 18, 15, col_panelcell_sel_fill);
        video_putsprite(videohandler, bigfont[i], paneloffx + 6, paneloffy + 2 + i * 20, 9, 13, 16, 15, 255);
      } else {
        video_rect(videohandler, paneloffx, paneloffy + i * 20, 20, 17, col_panelcell_border);
        video_rectfill(videohandler, paneloffx + 1, paneloffy + 1 + i * 20, 18, 15, col_panelcell_fill);
        video_putsprite(videohandler, bigfont[i], paneloffx + 6, paneloffy + 2 + i * 20, 9, 13, 16, 15, 12);
    }
  }

  if ((videohandler->flags & VIDEO13H_DBUF) != 0) {
    video_waitvblank();
    mouse_hide();
    video_flip(videohandler);
  }
  mouse_show();
}

/* processes mouse events. returns 0 if nothing done, non-zero otherwise (then a screen refresh would be welcome) */
static int processmouse(struct game_t *game, int mouseb, int mousex, int mousey) {
  int i, ii;
  if ((mouseb < 1) || (mouseb > 2)) return(0);
  /* might be a selectable button on the left */
  if ((mousex > paneloffx) && (mousex < paneloffx + 20) && (mouseb == 1)) {
    for (i = 0; i < 9; i++) {
      if ((mousey > paneloffy + i * 20) && (mousey < paneloffy + 18 + i * 20)) {
        if (game->selected > 0) game->refreshselectedbar |= (1 << (game->selected - 1)); /* remember to redraw the selection bar later */
        game->selected = i + 1;
        game->refreshselectedbar |= (1 << i); /* remember to redraw the selection bar later */
        return(1);
      }
    }
  }
  /* check if it's a playfield click */
  for (i = 0; i < 9; i++) {
    int x = fieldoffx + i * (fieldwidth - 1) + (i / 3);
    for (ii = 0; ii < 9; ii++) {
      int y = fieldoffy + ii * (fieldheight - 1) + (ii / 3);
      if ((mousex > x) && (mousey > y) && (mousex < x + fieldwidth) && (mousey < y + fieldheight)) {
        /* if it went on a readonly field, ignore the click */
        if (game->hints[i][ii] & HINTBLOCKED) return(0);
        /* */
        game->hints[i][ii] |= HINTREFRESH; /* mark this field to be redrawn later */
        if (mouseb == 1) {
            game->playfield[i][ii] = game->selected;
            return(1);
          } else if (mouseb == 2) {
            if (game->playfield[i][ii] != 0) {
                game->playfield[i][ii] = 0;
              } else if (game->selected > 0) { /* if a digit is selected, switch it on/off in hints */
                int bitmask;
                bitmask = 1 << (game->selected - 1);
                game->hints[i][ii] ^= bitmask;
            }
            return(1);
        }
      }
    }
  }
  return(0);
}

static void loadsdmstring(struct game_t *game, char *sdm, int revflag) {
  int x, y, parsedir = 1;
  if (revflag != 0) {
    parsedir = -1;
    sdm += 80;
  }
  for (y = 0; y < 9; y++) {
    for (x = 0; x < 9; x++) {
      if (*sdm != 0) {
        if ((*sdm >= '1') && (*sdm <= '9')) {
          game->playfield[x][y] = *sdm - '0';
          game->hints[x][y] = HINTBLOCKED;
        }
        game->hints[x][y] |= HINTREFRESH; /* set the field as "to be refreshed" */
        sdm += parsedir;
      }
    }
  }
}

static int loadrndgamefromfile(struct game_t *game, char *file) {
  long filesize, levelscount, levelnum;
  int x;
  unsigned char sdm[82];
  char sdmbin[41];
  FILE *fd;
  fd = fopen(file, "rb");
  if (fd == NULL) return(-1);
  /* get file's size */
  fseek(fd, 0, SEEK_END);
  filesize = ftell(fd);
  /* from file size we know how many levels there are */
  if (filesize % 41 != 0) { /* check that the filesize is okay */
    fclose(fd);
    return(-2);
  }
  levelscount = filesize / 41;
  /* choose a random level from the pool of all levels */
  levelnum = rand() % levelscount;
  /* load the chosen level */
  fseek(fd, levelnum * 41, SEEK_SET);
  fread(sdmbin, 1, 41, fd);
  fclose(fd);
  for (x = 0; x < 81; x++) {
    sdm[x] = sdmbin[x >> 1];
    if (x % 2 == 0) {
        sdm[x] >>= 4;
      } else {
        sdm[x] &= 0x0F;
    }
    sdm[x] += '0';
  }
  sdm[x] = 0;
  loadsdmstring(game, (char*)sdm, rand() % 2);
  return(0);
}

/* wait for any keypress or a mouse click */
static void wait_key_or_click(void) {
  int checkstate;
  for (;;) {
    checkstate = keyb_getkey_ifany();
    if (checkstate >= 0) break;
    checkstate = mouse_fetchrelease(NULL, NULL);
    if (checkstate > 0) break;
    releasecpu();
  }
}

/* fetch the directory where the program has been launched from, and return its length. Result never goes beyond 128. */
static int exepath(char *result) {
  char far *psp, *env;
  unsigned int envseg, pspseg, x, i;
  int lastsep;
  union REGS regs;
  /* get the PSP segment */
  regs.h.ah = 0x62;
  int86(0x21, &regs, &regs),
  pspseg = regs.x.bx;
  /* compute a far pointer that points to the top of PSP */
  psp = MK_FP(pspseg, 0);
  /* fetch the segment address of the environment */
  envseg = psp[0x2D];
  envseg <<= 8;
  envseg |= psp[0x2C];
  /* compute the env pointer */
  env = MK_FP(envseg, 0);
  /* skip all environment variables */
  x = 0;
  for (;;) {
    x++;
    if (env[x] == 0) { /* end of variable */
      x++;
      if (env[x] == 0) break; /* end of list */
    }
  }
  x++;
  /* read the WORD that indicates string that follow */
  if (env[x] < 1) {
    result[0] = 0;
    return(0);
  }
  x += 2;
  /* else copy the EXEPATH to our return variable, and truncate after last '\' */
  lastsep = -1;
  for (i = 0;; i++) {
    result[i] = env[x++];
    if (result[i] == '\\') lastsep = i;
    if (result[i] == 0) break; /* end of string */
    if (i >= 126) break;       /* this DOS string should never go beyond 127 chars! */
  }
  result[lastsep + 1] = 0;
  return(lastsep + 1);
}

static void preparefilename(char *exedir, int exedirlen, char *filename) {
  int i = 0;
  while (exedirlen < 126) {
    if (filename[i] == 0) break;
    exedir[exedirlen++] = filename[i++];
  }
  exedir[exedirlen] = 0;
}

/* parse command line params and return flags */
int parsecmdline(int argc, char **argv) {
  int i, res = 0;
  for (i = 1; i < argc; i++) {
    if (strcmp(argv[i], "/nodbuf") == 0) {
        res |= RUNFLAG_NODBUF;
      } else if (strcmp(argv[i], "/cga") == 0) {
        res |= RUNFLAG_CGA;
      } else { /* got unknown option */
        return(-1);
    }
  }
  return(res);
}

static void printstring(int row, int column, char *str) {
  union REGS regs;
  regs.h.ah = 2;  /* set cursor position */
  regs.h.dh = row;
  regs.h.dl = column;
  regs.h.bh = 0;
  int86(0x10, &regs, &regs);
  printf("%s", str);
}

int main(int argc, char **argv) {
  char exedirectory[128];
  int exedirectorylen;
  int exitflag = 0;
  int runflags;
  struct game_t game;
  struct video_handler *videohandler;

  runflags = parsecmdline(argc, argv);
  if (runflags < 0) {
    puts("Sudoku86 v" PVER "\n"
         "Copyright (C) Mateusz Viste " PDATE "\n"
         "\n"
         "  Usage: SUDOKU86 [options]\n"
         "\n"
         "Available options:\n"
         "  /nodbuf  Disable video double buffering. Use this if you don't\n"
         "           have enough conventional memory or if your CPU is very slow.\n"
         "\n"
         "  /cga     Force CGA mode\n");
    return(0);
  }

  /* seed the randmon generator */
  srand((unsigned int)time(NULL));

  /* fetch the directory we have been launched from */
  exedirectorylen = exepath(exedirectory);

  resetgame(&game);
  preparefilename(exedirectory, exedirectorylen, "sudoku86.lev");
  if (loadrndgamefromfile(&game, exedirectory) != 0) {
    puts("Error: failed to load level data");
    return(0);
  }

  /* check for VGA presence - fallback to CGA if no VGA */
  if (video_detectvga() == 0) runflags |= RUNFLAG_CGA;

  /* check for mouse presence */
  if (mouse_init() == 0) {
    puts("Error: no mouse detected!");
    return(0);
  }

  /* init video mode */
  if (runflags & RUNFLAG_CGA) {
      videohandler = video_open(0x04, 0);
    } else {
      if (runflags & RUNFLAG_NODBUF) {
          videohandler = video_open(0x13, 0);
        } else {
          videohandler = video_open(0x13, VIDEO13H_DBUF);
      }
  }
  if (videohandler == NULL) {
    puts("Error: failed to init video mode");
    return(0);
  }

  /* load our palette and background image if not running on CGA */
  if ((runflags & RUNFLAG_CGA) == 0) {
      preparefilename(exedirectory, exedirectorylen, "sudoku86.dat");
      video_loadpalfromfile(exedirectory, 32702l, 240, 16);
      video_file2screen(videohandler, exedirectory);
    } else { /* if running CGA, then generate a simple background */
      int x, y;
      for (y = 0; y < 200; y += 1) {
        for (x = (y & 3); x < 320; x += 3) {
          video_putpixel(videohandler, x, y, 2);
        }
      }
  }

  mouse_show();
  redrawscreen(videohandler, &game);
  keyb_flush(); /* flush keyboard buffer */

  while (exitflag == 0) {
    int mousex, mousey, mouseb;
    releasecpu();
    if (keyb_getkey_ifany() == 27) exitflag = 1;
    mouseb = mouse_fetchrelease(&mousex, &mousey);
    if (mouseb == 0) continue;
    mousex >>= 1; /* divide X coordinate by 2 because we are in mode 13h */
    if (processmouse(&game, mouseb, mousex, mousey) != 0) { /* if something meaningfull have been clicked, do a few things.. */
      redrawscreen(videohandler, &game);
      if (checksolution(&game) != 0) {
        preparefilename(exedirectory, exedirectorylen, "sudoku86.dat");
        if (runflags & RUNFLAG_CGA) {
            printstring(10, 16, "              ");
            printstring(11, 16, "  WELL DONE!  ");
            printstring(12, 16, "              ");
          } else {
            video_putspritefromfile(videohandler, exedirectory, 33422l, 116, 60, 142, 62, 0, 31, 255);
        }
        if ((runflags & RUNFLAG_NODBUF) == 0) { /* if double buffering is used, refresh screen now */
          mouse_hide();
          video_flip(videohandler);
          mouse_show();
        }
        wait_key_or_click(); /* wait for a keypress or a click */
        resetgame(&game);
        preparefilename(exedirectory, exedirectorylen, "sudoku86.lev");
        loadrndgamefromfile(&game, exedirectory);
        redrawscreen(videohandler, &game);
      }
    }
  }

  mouse_hide();
  video_close(videohandler);

  puts("Sudoku86 v" PVER "\n"
       "Copyright (C) " PDATE " Mateusz Viste\n"
       "\n"
       " http://sudoku86.sourceforge.net\n"
       "\n"
       "Sudoku86 is free software, released under the BSD 2-Clause license.\n"
       "See readme.txt for details.\n");

  return(0);
}
