/*
   video library - provides a few functions for mode 4 and 13h programming.

   Copyright (c) 2014, Mateusz Viste
   All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

#include <dos.h>       /* provides int86() along with the union REGS type */
#include <stdio.h>     /* FILE */
#include <stdlib.h>
#include <string.h>    /* memcpy() */
#include "video.h"  /* include self for control */

#ifdef __WATCOMC__
  #include <conio.h>
  #define inportb inp
#endif

/* uncompress a RLE stream from a file into a buffer */
static void video_rlefile2buff(char *file, unsigned char *buff, long foffset, long explen) {
  unsigned char rlebitlen, bytebuff, rlebitmask;
  unsigned int uncompressedbytes = 0;
  int rle;
  FILE *fd;
  fd = fopen(file, "rb");
  if (fd == NULL) return;
  if (foffset > 0) fseek(fd, foffset, SEEK_SET);
  /* read and validate the RLEBITLEN parameter */
  fread(&rlebitlen, 1, 1, fd);
  if ((rlebitlen > 7) || (rlebitlen < 1)) {
    fclose(fd);
    return;
  }
  rlebitmask = 255;
  rlebitmask <<= rlebitlen;
  rlebitmask &= 0xff;
  /* decompress the stream */
  for (;;) {
    fread(&bytebuff, 1, 1, fd);
    if ((bytebuff & rlebitmask) != rlebitmask) { /* if it's a raw byte, save it once */
        rle = 1;
      } else { /* if it's a RLE marker, read the next byte */
        rle = (bytebuff ^ rlebitmask) + 1; /* reset RLE marker bits and increment */
        fread(&bytebuff, 1, 1, fd);
    }
    for (; rle > 0; rle--) {
      buff[uncompressedbytes++] = bytebuff;
      if (uncompressedbytes >= explen) break;
    }
    if (uncompressedbytes >= explen) break;
  }
  fclose(fd);
}

/* returns 1 if VGA has been detected, zero otherwise */
int video_detectvga(void) {
  union REGS regs;
  regs.x.ax = 0x1A00; /* can be used to detect the presence of VGA. */
  int86(0x10, &regs, &regs);
  if (regs.h.al == 0x1A) return(1);  /* VGA supported */
  return(0); /* else it's not vga */
}

/* init video mode */
struct video_handler *video_open(int mode, int flags) {
  union REGS regs;
  struct video_handler *result;
  /* validate the mode */
  if ((mode != 0x04) && (mode != 0x13)) return(NULL);
  result = malloc(sizeof(struct video_handler));
  if (result == NULL) return(NULL);
  result->flags = flags;
  result->mode = mode;
  if (mode == 0x13) {  /* VGA mode 0x13 */
      if (flags & VIDEO13H_DBUF) { /* VGA with double buffering */
          result->dbuf = calloc(64000u, 1);
          if (result->dbuf == NULL) {
            free(result);
            return(NULL);
          }
        } else { /* VGA without double buffering */
          result->dbuf = (unsigned char far*)0xA0000000L;
      }
    } else if (mode == 0x04) {  /* CGA mode 0x04 */
      result->dbuf = (unsigned char far*)0xB8000000l;
  }
  /* first save the current mode, to be able to restore it later */
  regs.h.ah = 0x0F;
  int86(0x10, &regs, &regs);
  result->lastmode = regs.h.al;
  /* now switch to the selected video mode */
  regs.h.ah = 0;
  regs.h.al = mode;
  int86(0x10, &regs, &regs);
  return(result);
}

/* renders a sprite of width and height dimensions onscreen, starting at specified x/y location
   coloffset is an offset to add to color indexes, while transp is the index of the transparent color (set to -1 if none) */
void video_putsprite(struct video_handler *handler, unsigned char *sprite, int x, int y, int width, int height, int coloffset, int transp, int maxcol) {
  int pixelx, pixely, pixelval;
  if ((maxcol > 255) || (maxcol < 0)) maxcol = 255;
  for (pixely = 0; pixely < height; pixely++) {
    for (pixelx = 0; pixelx < width; pixelx++) {
      pixelval = sprite[pixely * width + pixelx];
      if (handler->mode == 0x04) if (pixelval < 7) pixelval = 0; else pixelval = transp; /* nasty hack for antialiased text */
      if (pixelval == transp) continue;
      if (pixelval > maxcol) pixelval = maxcol;
      video_putpixel(handler, x + pixelx, y + pixely, pixelval + coloffset);
    }
  }
}

/* same as video_putsprite(), but reads the sprite from a file */
void video_putspritefromfile(struct video_handler *handler, char *file, long foffset, int x, int y, int width, int height, int coloffset, int transp, int maxcol) {
  unsigned char *buff;
  buff = malloc(width * height);
  if (buff == NULL) return;
  video_rlefile2buff(file, buff, foffset, width * height);
  video_putsprite(handler, buff, x, y, width, height, coloffset, transp, maxcol);
  free(buff);
}

/* reads a screen dump from file and puts it to the screen buffer */
void video_file2screen(struct video_handler *handler, char *file) {
  video_rlefile2buff(file, handler->dbuf, 0, 64000u);
}

/* load count colors of palette from array of rgb triplets */
void video_loadpal(unsigned char *pal, int count, int offset) {
  int i;
  video_waitvblank();
  for (i = 0; i < count; i++) {
    if (i + offset > 255) break;
    video_setpalette(i + offset, pal[i+i+i], pal[i+i+i+1], pal[i+i+i+2]);
  }
}

void video_loadpalfromfile(char *file, long filepos, int count, int offset) {
  unsigned char *rgb;
  FILE *fd;
  fd = fopen(file, "rb");
  if (fd == NULL) return;
  rgb = malloc(count * 3);
  if (rgb == NULL) {
    fclose(fd);
    return;
  }
  if (filepos > 0) fseek(fd, filepos, SEEK_SET);
  fread(rgb, 1, count * 3, fd);
  fclose(fd);
  video_loadpal(rgb, count, offset);
  free(rgb);
}

/* Wait until VBLANK - works both in mode 0x04 and mode 0x13 */
void video_waitvblank(void) {
  unsigned char Status;
  do {
    Status = inportb(0x3DA);
  } while((Status & 0x08));
  do {
    Status = inportb(0x3DA);
  } while(!(Status & 0x08));
}

/* update the screen */
void video_flip(struct video_handler *handler) {
  unsigned char far *vram = (unsigned char far*)0xA0000000l;
  /* copy screen buffer to vram only if we are using double buffering */
  if (handler->flags & VIDEO13H_DBUF) memcpy(vram, handler->dbuf, 64000u);
}

/* clear screen using color */
void video_cls(struct video_handler *handler, unsigned char color) {
  if (handler->mode == 0x13) { /* VGA mode 0x13 */
      memset(handler->dbuf, color, 64000u);
    } else if (handler->mode == 0x04) { /* CGA mode 0x04 */
      int colbyte;
      color &= 3;
      colbyte = color | (color << 2) | (color << 4) | (color << 6);
      memset(handler->dbuf, colbyte, 8000);
      memset(handler->dbuf + 0x2000, colbyte, 8000);
  }
}

void video_close(struct video_handler *handler) {
  union REGS regs;
  /* restore the initial video mode */
  regs.h.ah = 0;
  regs.h.al = handler->lastmode;
  int86(0x10, &regs, &regs);
  /* free the double buffering memory, if any */
  if (handler->flags & VIDEO13H_DBUF) free(handler->dbuf);
  free(handler);
}

void video_putpixel(struct video_handler *handler, int x, int y, unsigned char col) {
  if (handler->mode == 0x13) {
      handler->dbuf[((y << 8) + (y << 6)) + x] = col;
    } else if (handler->mode == 0x04) {
      unsigned int offset = (y & 1) * 0x2000 + (y >> 1) * 80 + (x >> 2);
      unsigned char mask[4] = {63, 207, 243, 252};
      int bitshl[4] = {6, 4, 2, 0};
      col &= 3;
      col <<= bitshl[x % 4];
      handler->dbuf[offset] &= mask[x % 4]; /* reset bits */
      handler->dbuf[offset] |= col;
  }
}

/* render a horizontal line of length len starting at x/y */
void video_hline(struct video_handler *handler, int x, int y, int len, unsigned char color) {
  if (handler->mode == 0x13) { /* optimized for VGA mode 0x13 */
      memset(handler->dbuf + ((y << 8) + (y << 6)) + x, color, len);
    } else { /* not optimized for other modes */
      int i;
      for (i = 0; i < len; i++) video_putpixel(handler, x + i, y, color);
  }
}

void video_line(struct video_handler *handler, int x1, int y1, int x2, int y2, unsigned char color) {
  unsigned l;
  unsigned long x, y;
  int dx, dy;
  x = x1;
  x <<= 9;
  x += 256;
  y = y1;
  y <<= 9;
  y += 256;
  dx = x2 - x1;
  dy = y2 - y1;
  for (l = 0; l < 512; l++) {
    video_putpixel(handler, (int)(x >> 9), (int)(y >> 9), color);
    x += dx;
    y += dy;
  }
}

void video_rect(struct video_handler *handler, int x, int y, int width, int height, unsigned char color) {
  video_hline(handler, x, y, width, color);                                     /*  +--------+  */
  video_line(handler, x, y, x, (y + height - 1), color);                        /*  |           */
  video_line(handler, x + width - 1, y, x + width - 1, y + height - 1, color);  /*           |  */
  video_hline(handler, x, y + height - 1, width, color);                        /*  +--------+  */
}

/* renders a rectangle on screen, at position x/y, filled with color */
void video_rectfill(struct video_handler *handler, int x, int y, int width, int height, unsigned char color) {
  int i;
  for (i = y; i < y + height; i++) {
    video_hline(handler, x, i, width, color);
  }
}

void video_setpalette(int index, unsigned char r, unsigned char g, unsigned char b) {
  outp(0x3C8, index);
  outp(0x3C9, r);
  outp(0x3C9, g);
  outp(0x3C9, b);
}
