/*
    Sudoku86

  Copyright (c) 2014, Mateusz Viste
  All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

#include <dos.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "keyb.h"      /* keyboard routines */
#include "mouse.h"     /* mouse routines */
#include "video.h"     /* video routines */

#define PVER "1.0.2"
#define PDATE "2014"

#include "font.c"

#define fieldoffx 90
#define fieldoffy 17
#define fieldwidth 22
#define fieldheight 19
#define paneloffx 35
#define paneloffy 11

/* The playfield grid */
#define VGACOL_CELL0_FILL 30
#define VGACOL_CELLB_FILL 28
#define VGACOL_CELL_BORDER 8
#define CGACOL_CELL0_FILL 3
#define CGACOL_CELLB_FILL 1
#define CGACOL_CELL_BORDER 0
/* Left panel with selectable digits */
#define VGACOL_PANELCELL_BORDER 7
#define VGACOL_PANELCELL_FILL 28
#define VGACOL_PANELCELL_SEL_BORDER 6
#define VGACOL_PANELCELL_SEL_FILL 15
#define CGACOL_PANELCELL_BORDER 0
#define CGACOL_PANELCELL_FILL 3
#define CGACOL_PANELCELL_SEL_BORDER 2
#define CGACOL_PANELCELL_SEL_FILL 3

#define RUNFLAG_NODBUF 1
#define RUNFLAG_CGA 2
#define RUNFLAG_CUSTLEV 4

#define HINTBLOCKED 512
#define HINTREFRESH 1024


struct game_t {
  unsigned char playfield[9][9];
  unsigned short hints[9][9]; /* bitfield with hints */
  int selected;
  unsigned short refreshselectedbar;
};

/* releases a CPU 'slice' */
static void releasecpu(void) {
  union REGS regs;
  regs.x.ax = 0x1680;
  int86(0x2F, &regs, &regs);
}

static void resetgame(struct game_t *game) {
  memset(game, 0, sizeof(struct game_t));  /* set all game data to 0 */
  game->refreshselectedbar = 511; /* mark all buttons of the selection bar to be refreshed */
}

/* clears a game to its initial state */
static void cleargame(struct game_t *game) {
  int x, y;
  /* mark all fields on the playfield as to be 'refreshed' */
  for (y = 0; y < 9; y++) {
    for (x = 0; x < 9; x++) {
      if ((game->hints[x][y] & HINTBLOCKED) == 0) game->playfield[x][y] = 0;
      game->hints[x][y] &= HINTBLOCKED;
      game->hints[x][y] |= HINTREFRESH;
    }
  }
}

static void printstring(int row, int column, char *str) {
  union REGS regs;
  regs.h.ah = 2;  /* set cursor position */
  regs.h.dh = row;
  regs.h.dl = column;
  regs.h.bh = 0;
  int86(0x10, &regs, &regs);
  printf("%s", str);
}

/* checks whether a game is solved. returns 0 if not, non-zero otherwise. */
static int checksolution(struct game_t *game) {
  unsigned short digitsbitfield;
  int x, y, offsetx, offsety;
  /* check that every row and every column contains exactly all digits 1-9 */
  for (y = 0; y < 9; y++) {
    /* check rows */
    digitsbitfield = 0;
    for (x = 0; x < 9; x++) digitsbitfield |= (1 << game->playfield[x][y]);
    if (digitsbitfield != 1022) return(0);
    /* check columns */
    digitsbitfield = 0;
    for (x = 0; x < 9; x++) digitsbitfield |= (1 << game->playfield[y][x]);
    if (digitsbitfield != 1022) return(0);
  }
  /* check that every 3x3 sub-grid contains all digits 1-9 */
  for (offsety = 0; offsety < 9; offsety += 3) {
    for (offsetx = 0; offsetx < 9; offsetx += 3) {
      digitsbitfield = 0;
      for (y = 0; y < 3; y++) {
        for (x = 0; x < 3; x++) {
          digitsbitfield |= (1 << game->playfield[x + offsetx][y + offsety]);
        }
      }
      if (digitsbitfield != 1022) return(0);
    }
  }
  /* if we got here, then the playfield is all good */
  return(1);
}

static void redrawscreen(struct video_handler *videohandler, struct game_t *game) {
  int i, ii, x, y;
  unsigned char *bigfont[9];
  int col_cell_border, col_cellb_fill, col_cell0_fill, col_panelcell_border, col_panelcell_fill, col_panelcell_sel_border, col_panelcell_sel_fill;
  if (videohandler->mode == 0x13) {
      col_cell_border = VGACOL_CELL_BORDER;
      col_cellb_fill = VGACOL_CELLB_FILL;
      col_cell0_fill = VGACOL_CELL0_FILL;
      col_panelcell_sel_border = VGACOL_PANELCELL_SEL_BORDER;
      col_panelcell_sel_fill = VGACOL_PANELCELL_SEL_FILL;
      col_panelcell_border = VGACOL_PANELCELL_BORDER;
      col_panelcell_fill = VGACOL_PANELCELL_FILL;
    } else {
      col_cell_border = CGACOL_CELL_BORDER;
      col_cellb_fill = CGACOL_CELLB_FILL;
      col_cell0_fill = CGACOL_CELL0_FILL;
      col_panelcell_sel_border = CGACOL_PANELCELL_SEL_BORDER;
      col_panelcell_sel_fill = CGACOL_PANELCELL_SEL_FILL;
      col_panelcell_border = CGACOL_PANELCELL_BORDER;
      col_panelcell_fill = CGACOL_PANELCELL_FILL;
  }
  bigfont[0] = big1_pgm;
  bigfont[1] = big2_pgm;
  bigfont[2] = big3_pgm;
  bigfont[3] = big4_pgm;
  bigfont[4] = big5_pgm;
  bigfont[5] = big6_pgm;
  bigfont[6] = big7_pgm;
  bigfont[7] = big8_pgm;
  bigfont[8] = big9_pgm;
  if ((videohandler->flags & VIDEO_DBUF) == 0) mouse_hide();

  for (i = 0; i < 9; i++) {
    x = fieldoffx + i * (fieldwidth - 1) + (i / 3);
    for (ii = 0; ii < 9; ii++) {
      /* first of all, check if this field really needs a refresh */
      if ((game->hints[i][ii] & HINTREFRESH) == 0) continue;
      game->hints[i][ii] ^= HINTREFRESH;  /* reset the 'refresh' flag, since we will redraw the field right now */
      /* compute the y coordinate */
      y = fieldoffy + ii * (fieldheight - 1) + (ii / 3);
      /* draw a cell of the grid */
      video_rect(videohandler, x, y, fieldwidth, fieldheight, col_cell_border);
      /* now draw what's inside a single cell */
      if (game->hints[i][ii] & HINTBLOCKED) { /* readonly field */
          video_rectfill(videohandler, x + 1, y + 1, fieldwidth - 2, fieldheight - 2, col_cellb_fill);
          video_putsprite(videohandler, bigfont[game->playfield[i][ii] - 1], x + 7, y + 3, 9, 13, 16, 15, 12);
        } else { /* 'normal' playable field */
          video_rectfill(videohandler, x + 1, y + 1, fieldwidth - 2, fieldheight - 2, col_cell0_fill);
          if ((game->playfield[i][ii] > 0) && (game->playfield[i][ii] < 10)) {
              video_putsprite(videohandler, bigfont[game->playfield[i][ii] - 1], x + 7, y + 3, 9, 13, 16, 15, 255);
            } else if (game->hints != 0) { /* check for hints */
              int nhint;
              for (nhint = 0; nhint < 9; nhint++) {
                if (game->hints[i][ii] & (1 << nhint)) {
                  video_putsprite(videohandler, smallfont[nhint], x + 2 + (nhint % 3) * 6, y + 1 + (nhint / 3) * 6, 4, 5, 16, 15, 255);
                }
              }
          }
      }
    }
  }

  /* draw buttons with selectable digits on the left, if the bar changed */
  for (i = 0; i < 9; i++) {
    if (((game->refreshselectedbar >> i) & 1) == 0) continue; /* skip buttons that don't need a refresh */
    game->refreshselectedbar ^= (1 << i);
    if (game->selected == i + 1) {
        video_rect(videohandler, paneloffx, paneloffy + i * 20, 20, 17, col_panelcell_sel_border);
        video_rectfill(videohandler, paneloffx + 1, paneloffy + 1 + i * 20, 18, 15, col_panelcell_sel_fill);
        video_putsprite(videohandler, bigfont[i], paneloffx + 6, paneloffy + 2 + i * 20, 9, 13, 16, 15, 255);
      } else {
        video_rect(videohandler, paneloffx, paneloffy + i * 20, 20, 17, col_panelcell_border);
        video_rectfill(videohandler, paneloffx + 1, paneloffy + 1 + i * 20, 18, 15, col_panelcell_fill);
        video_putsprite(videohandler, bigfont[i], paneloffx + 6, paneloffy + 2 + i * 20, 9, 13, 16, 15, 12);
    }
  }

  if ((videohandler->flags & VIDEO_DBUF) != 0) {
    video_waitvblank();
    mouse_hide();
    video_flip(videohandler);
  }
  mouse_show();
}

/* processes mouse events. returns 0 if nothing done, non-zero otherwise (then a screen refresh would be welcome) */
static int processmouse(struct game_t *game, int mouseb, int mousex, int mousey) {
  int i, ii;
  if ((mouseb < 1) || (mouseb > 2)) return(0);
  /* might be a selectable button on the left */
  if ((mousex > paneloffx) && (mousex < paneloffx + 20) && (mouseb == 1)) {
    for (i = 0; i < 9; i++) {
      if ((mousey > paneloffy + i * 20) && (mousey < paneloffy + 18 + i * 20)) {
        if (game->selected > 0) game->refreshselectedbar |= (1 << (game->selected - 1)); /* remember to redraw the selection bar later */
        game->selected = i + 1;
        game->refreshselectedbar |= (1 << i); /* remember to redraw the selection bar later */
        return(1);
      }
    }
  }
  /* check if it's a playfield click */
  for (i = 0; i < 9; i++) {
    int x = fieldoffx + i * (fieldwidth - 1) + (i / 3);
    for (ii = 0; ii < 9; ii++) {
      int y = fieldoffy + ii * (fieldheight - 1) + (ii / 3);
      if ((mousex > x) && (mousey > y) && (mousex < x + fieldwidth) && (mousey < y + fieldheight)) {
        /* if it went on a readonly field, ignore the click */
        if (game->hints[i][ii] & HINTBLOCKED) return(0);
        /* */
        game->hints[i][ii] |= HINTREFRESH; /* mark this field to be redrawn later */
        if (mouseb == 1) {
            game->playfield[i][ii] = game->selected;
            return(1);
          } else if (mouseb == 2) {
            if (game->playfield[i][ii] != 0) {
                game->playfield[i][ii] = 0;
              } else if (game->selected > 0) { /* if a digit is selected, switch it on/off in hints */
                int bitmask;
                bitmask = 1 << (game->selected - 1);
                game->hints[i][ii] ^= bitmask;
            }
            return(1);
        }
      }
    }
  }
  return(0);
}

static void loadsdmstring(struct game_t *game, char *sdm, int revflag) {
  int x, y, parsedir = 1;
  if (revflag != 0) {
    parsedir = -1;
    sdm += 80;
  }
  for (y = 0; y < 9; y++) {
    for (x = 0; x < 9; x++) {
      if (*sdm != 0) {
        if ((*sdm >= '1') && (*sdm <= '9')) {
          game->playfield[x][y] = *sdm - '0';
          game->hints[x][y] = HINTBLOCKED;
        }
        game->hints[x][y] |= HINTREFRESH;
        sdm += parsedir;
      }
    }
  }
}

static int loadrndgamefromfile(struct game_t *game, char *file) {
  long filesize, levelscount, levelnum;
  int x;
  unsigned char sdm[82];
  char sdmbin[41];
  FILE *fd;
  fd = fopen(file, "rb");
  if (fd == NULL) return(-1);
  /* get file's size */
  fseek(fd, 0, SEEK_END);
  filesize = ftell(fd);
  /* from file size we know how many levels there are */
  if (filesize % 41 != 0) { /* check that the filesize is okay */
    fclose(fd);
    return(-2);
  }
  levelscount = filesize / 41;
  /* choose a random level from the pool of all levels */
  levelnum = rand() % levelscount;
  /* load the chosen level */
  resetgame(game);
  fseek(fd, levelnum * 41, SEEK_SET);
  fread(sdmbin, 1, 41, fd);
  fclose(fd);
  for (x = 0; x < 81; x++) {
    sdm[x] = sdmbin[x >> 1];
    if (x % 2 == 0) {
        sdm[x] >>= 4;
      } else {
        sdm[x] &= 0x0F;
    }
    sdm[x] += '0';
  }
  sdm[x] = 0;
  loadsdmstring(game, (char*)sdm, rand() % 2);
  return(0);
}

static void loadstate(struct game_t *game, char *file) {
  int x, y;
  unsigned int bufval;
  FILE *fd;
  fd = fopen(file, "rb");
  if (fd == NULL) return;
  resetgame(game);
  game->selected = fgetc(fd);
  for (y = 0; y < 9; y++) {
    for (x = 0; x < 9; x++) {
      /* read and decode the packed value */
      bufval = fgetc(fd) << 8;
      bufval |= fgetc(fd);
      game->playfield[x][y] = bufval & 15;
      bufval >>= 4;
      game->hints[x][y] = bufval | HINTREFRESH;
    }
  }
  fclose(fd);
}

static void savestate(struct game_t *game, char *file) {
  int x, y;
  unsigned int bufval;
  FILE *fd;
  fd = fopen(file, "wb");
  if (fd == NULL) return;
  fprintf(fd, "%c", game->selected);
  for (y = 0; y < 9; y++) {
    for (x = 0; x < 9; x++) {
      /* prepared the packed value */
      bufval = game->hints[x][y] & 1023; /* save only lower 10 bits, because these contain actual digits and the 'blocked' marker */
      bufval <<= 4; /* make room for the currently selected digit */
      bufval |= game->playfield[x][y];
      /* save the value to file, MSB first */
      fprintf(fd, "%c", bufval >> 8);
      fprintf(fd, "%c", bufval & 0xff);
    }
  }
  fclose(fd);
}

static void preparefilename(char *exedir, int exedirlen, char *filename) {
  int i = 0;
  while (exedirlen < 126) {
    if (filename[i] == 0) break;
    exedir[exedirlen++] = filename[i++];
  }
  exedir[exedirlen] = 0;
}

static void markallplayfieldtorefresh(struct game_t *game) {
  int x, y;
  for (y = 0; y < 9; y++) {
    for (x = 0; x < 9; x++) {
      game->hints[x][y] |= HINTREFRESH;
    }
  }
}

/* wait for any keypress or a mouse click */
static void wait_key_or_click(void) {
  int checkstate;
  for (;;) {
    checkstate = keyb_getkey_ifany();
    if (checkstate >= 0) break;
    checkstate = mouse_fetchrelease(NULL, NULL);
    if (checkstate > 0) break;
    releasecpu();
  }
}

/* processes mouse events. returns 0 if nothing done, non-zero otherwise (then a screen refresh would be welcome) */
static int processkeyb(struct game_t *game, int keypress, char *levelfile, char *exedirectory, int exedirectorylen) {
  switch (keypress) {
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
      if (game->selected > 0) game->refreshselectedbar |= (1 << (game->selected - 1)); /* remember to redraw the selection bar later */
      game->selected = keypress - '0';
      game->refreshselectedbar |= (1 << (keypress - '1')); /* remember to redraw the selection bar later */
      return(1);
    case 'c':   /* clear the level */
    case 'C':
      cleargame(game);
      return(1);
    case 'n':   /* new game */
    case 'N':
      loadrndgamefromfile(game, levelfile);
      return(1);
    case ' ': /* select next digit */
      if (game->selected > 0) game->refreshselectedbar |= (1 << (game->selected - 1)); /* remember to redraw the selection bar later */
      game->selected += 1;
      if (game->selected > 9) game->selected = 1;
      game->refreshselectedbar |= (1 << (game->selected - 1)); /* remember to redraw the selection bar later */
      return(1);
    case 0x13b:  /* F1 - help */
      mouse_hide();
      printstring( 3, 12, "----------------[HELP]-");
      printstring( 4, 12, "                       ");
      printstring( 5, 12, " Left click:           ");
      printstring( 6, 12, "  select a digit       ");
      printstring( 7, 12, "                       ");
      printstring( 8, 12, " Right click:          ");
      printstring( 9, 12, "  erase or put a note  ");
      printstring(10, 12, "                       ");
      printstring(11, 12, " Keyb:                 ");
      printstring(12, 12, "  1-9   Select digit   ");
      printstring(13, 12, "  SPACE Next digit     ");
      printstring(14, 12, "  C     Clear game     ");
      printstring(15, 12, "  N     New game       ");
      printstring(16, 12, "  F1    Help           ");
      printstring(17, 12, "  F5/F7 Save/Load      ");
      printstring(18, 12, "  ESC   Quit           ");
      printstring(19, 12, "                       ");
      printstring(20, 12, "-----------------------");
      mouse_show();
      markallplayfieldtorefresh(game);
      wait_key_or_click();
      return(1);
    case 0x13f:  /* F5 - save */
      preparefilename(exedirectory, exedirectorylen, "sudoku86.sav");
      savestate(game, exedirectory);
      mouse_hide();
      printstring(10, 16, "              ");
      printstring(11, 16, "  GAME SAVED  ");
      printstring(12, 16, "              ");
      mouse_show();
      markallplayfieldtorefresh(game);
      sleep(1);
      return(1);
    case 0x141:  /* F7 - load */
      preparefilename(exedirectory, exedirectorylen, "sudoku86.sav");
      loadstate(game, exedirectory);
      return(1);
  }
  return(0);
}

/* fetch the directory where the program has been launched from, and return its length. Result never goes beyond 128. */
static int exepath(char *result) {
  char far *psp, far *env;
  unsigned int envseg, pspseg, x, i;
  int lastsep;
  union REGS regs;
  /* get the PSP segment */
  regs.h.ah = 0x62;
  int86(0x21, &regs, &regs),
  pspseg = regs.x.bx;
  /* compute a far pointer that points to the top of PSP */
  psp = MK_FP(pspseg, 0);
  /* fetch the segment address of the environment */
  envseg = psp[0x2D];
  envseg <<= 8;
  envseg |= psp[0x2C];
  /* compute the env pointer */
  env = MK_FP(envseg, 0);
  /* skip all environment variables */
  x = 0;
  for (;;) {
    x++;
    if (env[x] == 0) { /* end of variable */
      x++;
      if (env[x] == 0) break; /* end of list */
    }
  }
  x++;
  /* read the WORD that indicates string that follow */
  if (env[x] < 1) {
    result[0] = 0;
    return(0);
  }
  x += 2;
  /* else copy the EXEPATH to our return variable, and truncate after last '\' */
  lastsep = -1;
  for (i = 0;; i++) {
    result[i] = env[x++];
    if (result[i] == '\\') lastsep = i;
    if (result[i] == 0) break; /* end of string */
    if (i >= 126) break;       /* this DOS string should never go beyond 127 chars! */
  }
  result[lastsep + 1] = 0;
  return(lastsep + 1);
}

/* parse command line params and return flags */
static int parsecmdline(int argc, char **argv, char **levelfile) {
  int i, res = 0;
  for (i = 1; i < argc; i++) {
    if (strcmp(argv[i], "/nodbuf") == 0) {
        res |= RUNFLAG_NODBUF;
      } else if (strcmp(argv[i], "/cga") == 0) {
        res |= RUNFLAG_CGA;
      } else if ((argv[i][0] != '/') && ((res & RUNFLAG_CUSTLEV) == 0)) { /* if anything else that doesn't start with a '/', it's a level file */
        *levelfile = argv[i];
        res |= RUNFLAG_CUSTLEV;
      } else { /* got unknown option */
        return(-1);
    }
  }
  return(res);
}

/* returns the current dos time, as a number of seconds */
static long getdostime(void) {
  long result;
  union REGS regs;
  regs.h.ah = 0x2C; /* get system time */
  int86(0x21, &regs, &regs),
  result = regs.h.ch * 3600; /* hour (0-23) */
  result += regs.h.cl * 60;  /* minutes (0-59) */
  result += regs.h.dl;       /* seconds (0-59) */
  return(result);
}

int main(int argc, char **argv) {
  char exedirectory[128];
  int exedirectorylen;
  int exitflag = 0;
  int runflags;
  char *levelfile = NULL;
  struct game_t game;
  struct video_handler *videohandler;

  runflags = parsecmdline(argc, argv, &levelfile);
  if (runflags < 0) {
    puts("Sudoku86 v" PVER "\n"
         "Copyright (C) " PDATE " Mateusz Viste\n"
         "\n"
         "  Usage: SUDOKU86 [options] [levels.lev]\n"
         "\n"
         "Available options:\n"
         "  /nodbuf  Disable video double buffering. Use this if you don't have\n"
         "           enough conventional memory or if your CPU is very slow.\n"
         "\n"
         "  /cga     Force CGA mode\n");
    return(0);
  }

  /* seed the random generator */
  srand((unsigned int)getdostime());

  /* fetch the directory we have been launched from */
  exedirectorylen = exepath(exedirectory);

  if (levelfile == NULL) {
    preparefilename(exedirectory, exedirectorylen, "sudoku86.lev");
    levelfile = strdup(exedirectory);
  }
  if (loadrndgamefromfile(&game, levelfile) != 0) {
    printf("Error: failed to load level data '%s'\n", levelfile);
    return(1);
  }

  /* check for VGA presence - fallback to CGA if no VGA */
  if (video_detectvga() == 0) runflags |= RUNFLAG_CGA;

  /* check for mouse presence */
  if (mouse_init() == 0) {
    puts("Error: no mouse detected!");
    return(0);
  }

  /* init video mode */
  {
    int videoflags = 0, videomode = 0x13;
    if ((runflags & RUNFLAG_NODBUF) == 0) videoflags = VIDEO_DBUF;
    if (runflags & RUNFLAG_CGA) videomode = 0x04;
    videohandler = video_open(videomode, videoflags);
  }
  if (videohandler == NULL) {
    puts("Error: failed to init video mode");
    return(0);
  }

  /* load our palette and background image if not running on CGA */
  if ((runflags & RUNFLAG_CGA) == 0) {
      preparefilename(exedirectory, exedirectorylen, "sudoku86.dat");
      video_loadpalfromfile(exedirectory, 32702l, 240, 16);
      video_file2screen(videohandler, exedirectory);
    } else { /* if running CGA, then generate a simple background */
      int x, y;
      for (y = 0; y < 200; y += 1) {
        for (x = (y & 3); x < 320; x += 3) {
          video_putpixel(videohandler, x, y, 2);
        }
      }
  }

  mouse_show();
  redrawscreen(videohandler, &game);
  keyb_flush(); /* flush keyboard buffer */

  while (exitflag == 0) {
    int mousex, mousey, mouseb, keypress;
    releasecpu();
    keypress = keyb_getkey_ifany();
    if (keypress == 27) exitflag = 1;
    mouseb = mouse_fetchrelease(&mousex, &mousey);
    if ((mouseb == 0) && (keypress == -1)) continue;
    mousex >>= 1; /* divide X coordinate by 2 because we are in mode 13h */
    if ((processkeyb(&game, keypress, levelfile, exedirectory, exedirectorylen) != 0) || (processmouse(&game, mouseb, mousex, mousey) != 0)) { /* if something meaningfull have been clicked or pressed, do a few things.. */
      redrawscreen(videohandler, &game);
      if (checksolution(&game) != 0) {
        preparefilename(exedirectory, exedirectorylen, "sudoku86.dat");
        if (runflags & RUNFLAG_CGA) {
            mouse_hide();
            printstring(10, 16, "              ");
            printstring(11, 16, "  WELL DONE!  ");
            printstring(12, 16, "              ");
            mouse_show();
          } else {
            video_putspritefromfile(videohandler, exedirectory, 33422l, 116, 60, 142, 62, 0, 31, 255);
            if ((runflags & RUNFLAG_NODBUF) == 0) { /* if double buffering is used, refresh screen now */
              mouse_hide();
              video_flip(videohandler);
              mouse_show();
            }
        }
        wait_key_or_click(); /* wait for a keypress or a click */
        loadrndgamefromfile(&game, levelfile);
        redrawscreen(videohandler, &game);
      }
    }
  }

  mouse_hide();
  video_close(videohandler);

  if ((runflags & RUNFLAG_CUSTLEV) != 0) free(levelfile);

  puts("Sudoku86 v" PVER "\n"
       "Copyright (C) " PDATE " Mateusz Viste\n"
       "\n"
       " http://sudoku86.sourceforge.net\n"
       "\n"
       "Sudoku86 is free software, released under the BSD 2-Clause license.\n"
       "See readme.txt for details.\n");

  return(0);
}
