/**************************************************************
*    htron.c                                   	              *
*    H-Tron - a DOS Tron game for the Hercules graphics card  *
*    written in Borland Turbo C 2.01                          *
*    Author: RobertK (RobertK@psc.at), December 2016          *
*    Version 2016-12-26                                       *
*                                                             *
*    Controls:                                                *
*    P1/P2 turn left/right: A/D and Cursor left/right         *
*    (or cursor left/right in one-player mode)                *
*    Pause game: P                                            *
*    End game: Escape                                         *
**************************************************************/

#include <graphics.h>
#include <stdlib.h>

#define POINTS_TO_WIN 5
#define AI_MAX_DISTANCE_TO_CHECK 15

/* Playfield array that holds the information whether
   there is an obstacle at this x,y coordinate.
   Playfield dimensions: 90 x 58
   x=0 to 89
   y=0 to 57
*/
int field[90][58];

/* indicates whether the player has hit an obstacle */
int p1Crashed;
int p2Crashed;

/* player coordinate variables */
int p1x,p1y,p2x,p2y;

/* player direction variables in radian degrees */
/* 0 (or 360) = right, 90 = up, 180 = left, 270 = down */
int p1dir,p2dir;

/* player score variables */
int p1score,p2score;

/* One or two players */
int numberOfPlayers;


void DrawBlock(int x, int y)
/* x,y: block coordinates */
/* x=0 to 89, y=0 to 57 */
{
  int xCoord;
  int yCoord;

  if (x<0 || y<0 || x>89 || y>57) return;

  /* Transform to graphics coordinates */
  xCoord=x*8;
  yCoord=y*6;

  /* draw block */
  rectangle(xCoord,yCoord,xCoord+7,yCoord+5);

  /* set obstacle marker */
  field[x][y]=1;
}


void InitPlayfield()
{
  int x,y;
  char textP1[20];
  char textP2[20];
  char p1TxtScore[5];
  char p2TxtScore[5];
  char far * textScore = "S C O R E";


  /* clear playfield array */
  for(x=0;x<90;x++)
   for(y=0;y<58;y++)
    field[x][y]=0;

  /* Draw border */
  rectangle(0,0,719,347);  /* outer border */
  rectangle(7,6,712,18);   /* score area */
  rectangle(7,23,712,342); /* inner border (playfield) */

  /* draw P1 and P2 score */
  /*
  if (p1score>0)
    for(x=0;x<p1score;x++)
     line(15+x*4,9,15+x*4,15);
  if (p2score>0)
    for(x=0;x<p2score;x++)
      line(704-x*4,9,704-x*4,15);
  */
  if (numberOfPlayers==1)
  {
   strcpy(textP1,"Computer: ");
   strcpy(textP2,"Human: ");
  }
  else
  {
   strcpy(textP1,"Player One: ");
   strcpy(textP2,"Player Two: ");
  }
  itoa(p1score,p1TxtScore,10);
  itoa(p2score,p2TxtScore,10);
  strcat(textP1,p1TxtScore);
  strcat(textP2,p2TxtScore);
  outtextxy(30,9,textP1);
  outtextxy(691-textwidth(textP2),9,textP2);
  outtextxy(360-(textwidth(textScore)/2),9,textScore);

  /* Mark border as obstacles in playfield array */
  for(x=0;x<90;x++)
  {
    field[x][3]=1;
    field[x][57]=1;
  }
  for(y=0;y<58;y++)
  {
    field[0][y]=1;
    field[89][y]=1;
  }

  /* Initiate player positions, directions and crash indicators */
  p1x=15;
  p1y=29;
  p2x=74;
  p2y=29;
  p1dir=0;
  p2dir=180;
  p1Crashed=0;
  p2Crashed=0;

  /* Draw players at their start positions */
  DrawBlock(p1x,p1y);
  DrawBlock(p2x,p2y);
}


void MovePlayers()
{
  switch(p1dir)
  {
    case 0:   p1x=p1x+1; break;
    case 90:  p1y=p1y-1; break;
    case 180: p1x=p1x-1; break;
    case 270: p1y=p1y+1; break;
  }
  switch(p2dir)
  {
    case 0:   p2x=p2x+1; break;
    case 90:  p2y=p2y-1; break;
    case 180: p2x=p2x-1; break;
    case 270: p2y=p2y+1; break;
  }

  /* Collision detection */
  if (field[p1x][p1y]==1)
  {
    p1Crashed=1;
    p2score++;
  }
  if (field[p2x][p2y]==1)
  {
    p2Crashed=1;
    p1score++;
  }
  /* no points when both players crashed */
  if (p1Crashed==1 && p2Crashed==1)
  {
   p1score--;
   p2score--;
  }

  /* Draw players at their new positions */
  /* (unless a collision has been detected) */
  if (p1Crashed==0) DrawBlock(p1x,p1y);
  if (p2Crashed==0) DrawBlock(p2x,p2y);

}

void AIPlayerOne()
{
  /* Let the computer player decide whether to turn left or right,
     or continue straight on */

  /* number of empty grid blocks before an obstacle
     from the AI player's point of view */
  int distanceFront,distanceLeft,distanceRight;

  int maxDistanceToChk;
  maxDistanceToChk=AI_MAX_DISTANCE_TO_CHECK;

  /* To make the AI player's movements less predictable,
  we occasionally make him a little short-sighted */
  maxDistanceToChk=maxDistanceToChk-random(10);

  /* Check the distance from the AI player's position
     to the next obstacle in the front, left and right
     direction from the AI player's point of view. According
     to the players direction, we have to use the respective
     grid direction */
  switch(p1dir)
  {
    case 0: /* right */
       distanceFront=AICheckDistanceEast(p1x,p1y,maxDistanceToChk);
       distanceLeft=AICheckDistanceNorth(p1x,p1y,maxDistanceToChk);
       distanceRight=AICheckDistanceSouth(p1x,p1y,maxDistanceToChk);
       break;
    case 90: /* up */
       distanceFront=AICheckDistanceNorth(p1x,p1y,maxDistanceToChk);
       distanceLeft=AICheckDistanceWest(p1x,p1y,maxDistanceToChk);
       distanceRight=AICheckDistanceEast(p1x,p1y,maxDistanceToChk);
       break;
    case 180: /* left */
       distanceFront=AICheckDistanceWest(p1x,p1y,maxDistanceToChk);
       distanceLeft=AICheckDistanceSouth(p1x,p1y,maxDistanceToChk);
       distanceRight=AICheckDistanceNorth(p1x,p1y,maxDistanceToChk);
       break;
    case 270: /* down */
       distanceFront=AICheckDistanceSouth(p1x,p1y,maxDistanceToChk);
       distanceLeft=AICheckDistanceEast(p1x,p1y,maxDistanceToChk);
       distanceRight=AICheckDistanceWest(p1x,p1y,maxDistanceToChk);
       break;
  }

  /* Now we decide what the AI player shall do */
  if ((distanceFront>=distanceLeft) && (distanceFront>=distanceRight))
  {
    /* clear sailing ahead, nothing to do */
  }
  else if ((distanceFront<distanceLeft) || (distanceFront<distanceRight))
  {
    /* now we know that it would be safer to turn either left or right */
    /* let us check which direction would be better */
    if (distanceLeft>distanceRight)
    {
    	/* turn left */
    	p1dir = p1dir+90;
    	if (p1dir>=360) p1dir=0;
    }
    else if (distanceLeft<distanceRight)
    {
        /* turn right */
	p1dir = p1dir-90;
	if (p1dir<0) p1dir=270;
    }
    else /* distanceLeft == distanceRight */
    {
	/* randomly turn either left or right */
	if (random(100)<50)
	{
	  /* turn left */
	  p1dir = p1dir+90;
	  if (p1dir>=360) p1dir=0;
	}
	else
	{
	  /* turn right */
	  p1dir = p1dir-90;
	  if (p1dir<0) p1dir=270;
	}
    }
  }
}

/* The following four functions calculate the distance from the
   given position to the next obstacle in grid map direction */
int AICheckDistanceEast(int x,int y,int maxDistanceToCheck)
{
  int distance;
  distance=1;
  while(field[x+distance][y]==0 && distance<=maxDistanceToCheck)
  {
    distance++;
  }
  return distance;
}

int AICheckDistanceWest(int x,int y,int maxDistanceToCheck)
{
  int distance;
  distance=1;
  while(field[x-distance][y]==0 && distance<=maxDistanceToCheck)
  {
    distance++;
  }
  return distance;
}

int AICheckDistanceNorth(int x,int y,int maxDistanceToCheck)
{
  int distance;
  distance=1;
  while(field[x][y-distance]==0 && distance<=maxDistanceToCheck)
  {
    distance++;
  }
  return distance;
}

int AICheckDistanceSouth(int x,int y,int maxDistanceToCheck)
{
  int distance;
  distance=1;
  while(field[x][y+distance]==0 && distance<=maxDistanceToCheck)
  {
    distance++;
  }
  return distance;
}


void main()
{
  int x,y,i,j,exitgame;
  int key, scancode;
  int notPlayAgain;
  int grd, grm;
  int gresult;

  notPlayAgain=0;

  clrscr();
  printf("*** H-Tron ***\n");
  printf("A DOS game for the Hercules graphics card,\n");
  printf("written in December 2016 by RobertK (RobertK@psc.at).\n\n");
  printf("This is the classical overhead view \"Tron\" game from movie of the same title\n");
  printf("(motorcycles that leave \"wall trails\" behind them). If you crash into a wall,\n");
  printf("your opponent scores a point. The player who first reaches a score of %i wins.\n\n",POINTS_TO_WIN);

  printf("Controls:\n");
  printf("P1/P2 turn left/right: A/D and cursor left/right\n");
  printf("(or cursor left/right in one-player mode)\n");
  printf("Pause game: P, End Game: Escape\n");
  printf("Press Space to start a round.\n\n");
  printf("One or two players (1/2)?");
  do
  {
    key=getch();
    if (key=='1') numberOfPlayers=1;
    if (key=='2') numberOfPlayers=2;
    if (key==27) /* Escape */
    {
      clrscr();
      return;
    }
  }
  while(key!='1' && key!='2');

  /* game restart loop */
  while(notPlayAgain<1)
  {
    p1score=0;
    p2score=0;
    exitgame=0;

    /* === switch to graphics mode: beginning of section === */

    /* Detect the graphics driver and mode */
    detectgraph(&grd,&grm);

    /* initialize the graphics mode with initgraph */
    initgraph(&grd, &grm, "");
    gresult = graphresult();
    if(gresult != grOk)
    {
      printf(grapherrormsg(gresult));
      getch();
      return;
    }

    /* getgraphmode returns the current graphics mode */
    /* grm = getgraphmode(); */
    /* printf("current mode: %d", grm); */
    /* getch(); */

    /* setgraphmode sets the system to graphics mode, clears the screen */
    setgraphmode(HERCMONOHI);

    /* === switch to graphics mode: end of section === */

    /* round restart loop */
    while(p1score<POINTS_TO_WIN && p2score<POINTS_TO_WIN && exitgame<1)
    {
	cleardevice();
	InitPlayfield();

  	/* workaround to avoid pause immediately after the game starts */
  	delay(10);

	/* while(!kbhit()); */
	do
	{
	  key=getch();
	  if (key==27) exitgame=1;
	} while (key!=32 && key!=27);

	/* main game loop */
  	while(exitgame<1 && p1Crashed<1 && p2Crashed<1)
  	{
    	  if (kbhit())
    	  {
     	    key=bioskey(0);
      	    scancode = key >> 8;

      	    switch(scancode)
      	    {
		case 75: /* Left Key (P2 Left) */
	  	  p2dir = p2dir+90;
          	  if (p2dir>=360) p2dir=0;
          	  break;
        	case 77: /* Right Key (P2 Right) */
          	  p2dir = p2dir-90;
          	  if (p2dir<0) p2dir=270;
          	  break;
        	case 30: /* A Key (P1 Left) */
		  if (numberOfPlayers==2)
		  {
          	    p1dir = p1dir+90;
          	    if (p1dir>=360) p1dir=0;
		  }
          	  break;
        	case 32: /* D Key (P1 Right) */
		  if (numberOfPlayers==2)
		  {
          	    p1dir = p1dir-90;
          	    if (p1dir<0) p1dir=270;
		  }
          	  break;
        	case 25: /* P Key (Pause) */
          	  while(!kbhit());
         	  break;
        	case 1: /* Escape key */
          	  exitgame=1;
          	  break;
      	     }
    	  }

          /* let the computer player decide what to do */
	  if (numberOfPlayers==1) AIPlayerOne();

    	  MovePlayers();

	  /* Sleep for 50 milliseconds.
	     note: the delay() function was introduced in Turbo C 2.0
	     (it is not available in Turbo C 1.0) */
	  delay(50);
        }

	/* wait a quarter of a second after a crash has
	   happened before clearing the screen */
	if (p1Crashed || p2Crashed) delay(250);
    }

    restorecrtmode(); /* switch back to text mode */
    closegraph(); /* to clean up memory */
    clrscr();

    printf("Final Score: %i : %i\n",p1score,p2score);
    if (p1score>p2score)
	if (numberOfPlayers==2)
	  printf("Player One wins!");
	else
	  printf("Computer wins!");
    if (p1score<p2score)
	if (numberOfPlayers==2)
	  printf("Player Two wins!");
	else
	  printf("You win!");
    if (p1score==p2score)
    	printf("A draw!");
    printf("\nPlay again (y/n)?");
    do
    {
    	key=getch();
    	if (key=='n' || key=='N') notPlayAgain=1;

    } while(key!='y' && key!='Y' && key!='n' && key!='N');

  }
  clrscr();
  return;
}