/*	@(#)route.c 1.1 86/02/03 SMI; from UCB 6.3 83/12/15	*/
/*	@(#)route.c	2.1 86/04/15 NFSSRC */

#include "sys/types.h"
#include "sys/param.h"
#include "sys/errno.h"
#include "sys/time.h"
#ifdef PAGING
#include "sys/mmu.h"
#include "sys/seg.h"
#include "sys/page.h"
#endif PAGING
#include "sys/systm.h"
#include "sys/mbuf.h"
#include "sys/socket.h"
#include "sys/protosw.h"
#include "sys/ioctl.h"
#include "netinet/in.h"
#include "netinet/in_systm.h"
#include "net/if.h"
#include "net/af.h"

#include "net/route.h"
#include "errno.h"
#include "sys/signal.h"
#include "sys/user.h"

int	rttrash;		/* routes not in table but not freed */
struct	sockaddr wildcard;	/* zero valued cookie for wildcard searches */
int	rthashsize = RTHASHSIZ;	/* for netstat, etc. */

/*
 * Packet routing routines.
 */
rtalloc(ro)
	register struct route *ro;
{
	register struct rtentry *rt;
	register struct mbuf *m;
	register u_long hash;
	struct sockaddr *dst = &ro->ro_dst;
	int (*match)();
	register int doinghost;
	struct afhash h;
	u_int af = dst->sa_family;
	register struct rtentry *rtmin;
	struct mbuf **table;

	if (ro->ro_rt && ro->ro_rt->rt_ifp && (ro->ro_rt->rt_flags & RTF_UP))
		return;				/* XXX */
	if (af >= AF_MAX)
		return;
	(*afswitch[af].af_hash)(dst, &h);
	rtmin = 0;
	match = afswitch[af].af_netmatch;
	hash = h.afh_hosthash, table = rthost, doinghost = 1;
again:
	for (m = table[hash % RTHASHSIZ]; m; m = m->m_next) {
		rt = mtod(m, struct rtentry *);
		if (rt->rt_hash != hash)
			continue;
		if ((rt->rt_flags & RTF_UP) == 0 ||
		    (rt->rt_ifp->if_flags & IFF_UP) == 0)
			continue;
		if (doinghost) {
			if (bcmp((caddr_t)&rt->rt_dst, (caddr_t)dst,
			    sizeof (*dst)))
				continue;
			 /* With hosts we must look at all routes to find best */
			if (rtmin == 0 || rt->rt_use < rtmin->rt_use)
				rtmin = rt;
		} else {
			if (rt->rt_dst.sa_family != af ||
			    !(*match)(&rt->rt_dst, dst))
				continue;
			/*
			 * First matching entry should be the only entry.
			 * Therefore, it is the best entry.
			 */
			rtmin = rt;
			break;
		}
	}
	/*
	 * If we have not found an entry, maybe change what we are looking for.
	 */
	if (rtmin == 0) {
		if (doinghost) {
			doinghost = 0;
			hash = h.afh_nethash, table = rtnet;
			goto again;
		}
		/*
		 * Check for wildcard gateway, by convention network 0.
		 */
		if (dst != &wildcard) {
			dst = &wildcard, hash = 0;
			rtstat.rts_wildcard++;
			goto again;
		}
		/*
		 * Nothing worked; give up...
		 */
		rtstat.rts_wildcard--;  /* undo the above increment */
		rtstat.rts_unreach++;
	} else {
		rtmin->rt_refcnt++;
	}
	ro->ro_rt = rtmin;
}

rtfree(rt)
	register struct rtentry *rt;
{

	if (rt == 0)
		panic("rtfree");
	rt->rt_refcnt--;
	if (rt->rt_refcnt == 0 && (rt->rt_flags&RTF_UP) == 0) {
		rttrash--;
		(void) m_free(dtom(rt));
	}
}

/*
 * Force a routing table entry to the specified
 * destination to go through the given gateway.
 * Normally called as a result of a routing redirect
 * message from the network layer.
 *
 * N.B.: must be called at splnet or higher
 *
 * Should notify all parties with a reference to
 * the route that it's changed (so, for instance,
 * current round trip time estimates could be flushed),
 * but we have no back pointers at the moment.
 */
rtredirect(dst, gateway, flags)
	struct sockaddr *dst, *gateway;
	int flags;
{
	struct route ro;
	register struct rtentry *rt;

	/* verify the gateway is directly reachable */
	if (ifa_ifwithnet(gateway) == 0) {
		rtstat.rts_badredirect++;
		return;
	}
	ro.ro_dst = *dst;
	ro.ro_rt = 0;
	rtalloc(&ro);
	rt = ro.ro_rt;
	/*
	 * Create a new entry if we just got back a wildcard entry
	 * or the the lookup failed.  This is necessary for hosts
	 * which use routing redirects generated by smart gateways
	 * to dynamically build the routing tables.
	 */
	if (rt &&
	    (*afswitch[dst->sa_family].af_netmatch)(&wildcard, &rt->rt_dst)) {
		rtfree(rt);
		rt = 0;
	}
	if (rt == 0) {
		rtinit(dst, gateway, (int)SIOCADDRT,
		       (flags & RTF_HOST) | RTF_GATEWAY | RTF_DYNAMIC);
		rtstat.rts_dynamic++;
		return;
	}
	if (((rt->rt_flags & RTF_HOST) == 0) && (flags & RTF_HOST)) {
		/*
		 * Changing from route to gateway => route to host.
		 * Create new route, rather than smashing route to net.
		 */
		rtfree(rt);
		rtinit(dst, gateway, (int)SIOCADDRT, flags | RTF_DYNAMIC);
		rtstat.rts_newgateway++;
	} else {
		/*
		 * Smash the current notion of the gateway to
		 * this destination.  This is probably not right,
		 * as it's conceivable a flurry of redirects could
		 * cause the gateway value to fluctuate wildly during
		 * dynamic routing reconfiguration.
		 */
		rt->rt_gateway = *gateway;
		rtfree(rt);
		rtstat.rts_newgateway++;
	}
}

/*
 * Routing table ioctl interface.
 */
rtioctl(cmd, data)
	int cmd;
	caddr_t data;
{

	if (cmd != SIOCADDRT && cmd != SIOCDELRT)
		return (EINVAL);
	if (!suser())
		return (u.u_error);
	return (rtrequest(cmd, (struct rtentry *)data));
}

/*
 * Carry out a request to change the routing table.  Called by
 * interfaces at boot time to make their ``local routes'' known,
 * for ioctl's, and as the result of routing redirects.
 */
rtrequest(req, entry)
	int req;
	register struct rtentry *entry;
{
	register struct mbuf *m, **mprev;
	struct mbuf **mfirst;
	register struct rtentry *rt;
	struct afhash h;
	int s, error = 0, (*match)();
	u_int af;
	u_long hash;
	struct ifaddr *ifa;
	struct ifaddr *ifa_ifwithdstaddr();

	af = entry->rt_dst.sa_family;
	if (af >= AF_MAX || af == 0)
		return (EAFNOSUPPORT);
	(*afswitch[af].af_hash)(&entry->rt_dst, &h);
	if (entry->rt_flags & RTF_HOST) {
		hash = h.afh_hosthash;
		mprev = &rthost[hash % RTHASHSIZ];
	} else {
		hash = h.afh_nethash;
		mprev = &rtnet[hash % RTHASHSIZ];
	}
	match = afswitch[af].af_netmatch;
	s = splimp();
	for (mfirst = mprev; m = *mprev; mprev = &m->m_next) {
		rt = mtod(m, struct rtentry *);
		if (rt->rt_hash != hash)
			continue;
		if (entry->rt_flags & RTF_HOST) {
#define	equal(a1, a2) \
	(bcmp((caddr_t)(a1), (caddr_t)(a2), sizeof (struct sockaddr)) == 0)
			if (!equal(&rt->rt_dst, &entry->rt_dst))
				continue;
		} else {
			if (rt->rt_dst.sa_family != entry->rt_dst.sa_family ||
			    (*match)(&rt->rt_dst, &entry->rt_dst) == 0)
				continue;
		}
		if (equal(&rt->rt_gateway, &entry->rt_gateway))
			break;
	}
	switch (req) {

	case SIOCDELRT:
		if (m == 0) {
			error = ESRCH;
			goto bad;
		}
		*mprev = m->m_next;
		if (rt->rt_refcnt > 0) {
			rt->rt_flags &= ~RTF_UP;
			rttrash++;
			m->m_next = 0;
		} else
			(void) m_free(m);
		break;

	case SIOCADDRT:
		if (m) {
			error = EEXIST;
			goto bad;
		}
/*???*/		if (af != entry->rt_gateway.sa_family) {
			error = EAFNOSUPPORT;
			goto bad;
		}
		if ((entry->rt_flags & RTF_GATEWAY) == 0) {
			/*
			 * If we are adding a route to an interface,
			 * and the interface is a pt to pt link
			 * we should search for the destination
			 * as our clue to the interface.  Otherwise
			 * we can use the local address.
			 */
			ifa = (struct ifaddr *)0;
			if (entry->rt_flags & RTF_HOST) 
				ifa = ifa_ifwithdstaddr(&entry->rt_dst);
			if (ifa == (struct ifaddr *)0)
				ifa = ifa_ifwithaddr(&entry->rt_gateway);
		} else {
			/*
			 * If we are adding a route to a remote net
			 * or host, the gateway may still be on the
			 * other end of a pt to pt link.
			 */
			ifa = ifa_ifwithdstaddr(&entry->rt_gateway);
		}
		if (ifa == (struct ifaddr *)0) {
			ifa = ifa_ifwithnet(&entry->rt_gateway);
			if (ifa == (struct ifaddr *)0) {
				error = ENETUNREACH;
				goto bad;
			}
		}
		m = m_get(M_DONTWAIT, MT_RTABLE);
		if (m == 0) {
			error = ENOBUFS;
			goto bad;
		}
		m->m_next = *mfirst;
		*mfirst = m;
		m->m_off = MMINOFF;
		m->m_len = sizeof (struct rtentry);
		rt = mtod(m, struct rtentry *);
		rt->rt_hash = hash;
		rt->rt_dst = entry->rt_dst;
		rt->rt_gateway = entry->rt_gateway;
		rt->rt_flags = RTF_UP |
		    (entry->rt_flags & (RTF_HOST|RTF_GATEWAY|RTF_DYNAMIC));
		rt->rt_refcnt = 0;
		rt->rt_use = 0;
		rt->rt_ifp = ifa->ifa_ifp;
		break;
	}
bad:
	(void) splx(s);
	return (error);
}

/*
 * Set up a routing table entry, normally
 * for an interface.
 */
rtinit(dst, gateway, cmd, flags)
	struct sockaddr *dst, *gateway;
	int cmd, flags;
{
	struct rtentry route;
	
	bzero((caddr_t)&route, sizeof (route));
	route.rt_dst = *dst;
	route.rt_gateway = *gateway;
	route.rt_flags = flags;
	(void) rtrequest(cmd, &route);
}
