//
// Copyright 1994, Cray Research, Inc.
//                 
// Permission to use, copy, modify and distribute this software and
// its accompanying documentation (the "Software") is granted without
// fee, provided that the above copyright notice and this permission
// notice appear in all copies of the Software and all supporting
// documentation, and the name of Cray Research, Inc. not be used in
// advertising or publicity pertaining to distribution of the 
// Software without the prior specific, written permission of Cray
// Research, Inc.  The Software is a proprietary product of Cray
// Research, Inc., and all rights not specifically granted by this
// license shall remain in Cray Research, Inc.  No charge may be made
// for the use or distribution of the Software.  The Software may be
// distributed as a part of a different product for which a fee is
// charged, if (i) that product contains or provides substantial
// functionality that is additional to, or different from, the
// functionality of the Software, and (ii) no separate, special or
// direct charge is made for the Software.
//         
// THE SOFTWARE IS MADE AVAILABLE "AS IS", AND ALL EXPRESS AND
// IMPLIED WARRANTIES, INCLUDING THE IMPLIED WARRANTIES OF FITNESS
// FOR A PARTICULAR PURPOSE, MERCHANTABILITY, AND FREEDOM FROM
// VIOLATION OF THIRD PARTY INTELLECTUAL PROPERTY RIGHTS, ARE HEREBY
// DISCLAIMED AND EXCLUDED BY CRAY RESEARCH, INC.  CRAY RESEARCH,
// INC. WILL NOT BE LIABLE IN ANY EVENT FOR ANY CONSEQUENTIAL,
// SPECIAL, INCIDENTAL, OR INDIRECT DAMAGES ARISING OUT OF OR IN
// CONNECTION WITH THE PERFORMANCE OF THE SOFTWARE OR ITS USE BY ANY
// PERSON, OR ANY FAILURE OR NEGLIGENCE ON THE PART OF CRAY RESEARCH,
// INC., EXCEPT FOR THE GROSS NEGLIGENCE OR WILLFUL MISCONDUCT OF
// CRAY RESEARCH.
// 
// This License Agreement shall be governed by, and interpreted and
// construed in accordance with, the laws of the State of Minnesota,
// without reference to its provisions on the conflicts of laws, and
// excluding the United Nations Convention of the International Sale
// of Goods.
//
//	USMID %Z%%M%	%I%	%G% %U%
//	Chart.h++,v 1.1.1.1 1996/08/21 08:27:05 prb Exp
//

#ifndef _Cvo_Chart_
#define _Cvo_Chart_

#include <Cvo/Frame.h++>
#include <Cvo/Label.h++>
#include <Cvo/TextBoard.h++>
#include <Cvo/Mark.h++>
#include <Cvo/OptionMenu.h++>

enum Cvo_ChartOrient {
    Cvo_ChartHorizontal     = 0x01,
    Cvo_ChartVertical       = 0x02,
    Cvo_ChartVerticalSide   = 0x04,
    Cvo_ChartHorizontalSide = 0x08
};

class Cvo_Chart;

struct OrientData {
    Cvo_Chart* 		c;
    Cvo_ChartOrient	o;
};

class Cvo_ChartMark : public Cvo_RootedList {
public:
    double              value;
    char                *string;

    Cvo_ChartMark(Cvo_ChartMark **, double, char *);
    ~Cvo_ChartMark()			{ delete[] string; };
    Cvo_ChartMark *Next()		{ return((Cvo_ChartMark *)(next)); }
    Cvo_ChartMark *Prev()		{ return((Cvo_ChartMark *)(prev)); }
};

class Cvo_GraphMark : public Cvo_RootedList {
public:
    Cvo_ChartMark       *mark;
    Cvo_TextBoardItem   *item;

    Cvo_GraphMark(Cvo_GraphMark **, Cvo_ChartMark *);
    Cvo_GraphMark *Next()		{ return((Cvo_GraphMark *)(next)); }
    Cvo_GraphMark *Prev()		{ return((Cvo_GraphMark *)(prev)); }
};

class Cvo_GraphObject : public Cvo_Frame {
    friend Cvo_Chart;
protected:
    Cvo_Chart			*chartob;
    Cvo_Window			*graph;
    BOOL			pixmapbacking : 1;
    BOOL			dirty : 1;
private:
    virtual Cvo_GraphObject *CloneMe(Cvo_Object *, Cvo_Chart *) = 0;
    CVO_DEFINE_REGISTER_FUNCTIONS(Cvo_GraphObject)
protected:
    virtual void DrawIt(XEvent *, void *) = 0;
    virtual void DrawMarks() = 0;
    void _Init(Cvo_Chart *);

    inline Cvo_ChartMark *Marks();
    inline Cvo_ChartMark *Cmarks();
    inline double **Columns();
    inline int Ncolumns();
    inline int Ncolors();
    inline Cvo_Color *Colors();
    inline Cvo_Color Linecolor();
    inline Cvo_Color Backcolor();

    virtual void SetText(wchar_t *) = 0;
    virtual void SetColumnText(wchar_t *) = 0;

    virtual void HandleEvent(XEvent *, void *) = 0;

public:
    CONSTRUCDECL_1ARG(Cvo_GraphObject,Cvo_Chart*)

    inline void CreateColumns(int);

    virtual Cvo_GraphMark *AddMark(Cvo_ChartMark *) = 0;
    virtual Cvo_GraphMark *AddColumnMark(Cvo_ChartMark *) = 0;

    virtual void EvaluateMarks() = 0;
    virtual void EvaluateColumnMarks() = 0;

    virtual void ClearMarks() = 0;
    virtual void ClearColumnMarks() = 0;

    virtual void SetMinimum(double) = 0;
    virtual double GetMinimum() = 0;

    virtual void Redraw() = 0;

    friend void _Cvo_GraphObject_Expose_Handler(Cvo_Object *, XEvent *, void *);
};

class Cvo_ChartLegend : public Cvo_Frame {
    friend Cvo_Chart;

    Cvo_Label	*label;
    Cvo_Window	*box;
    void        _Init(const Cvo_Color &, char *);
    void        _Init(const Cvo_Color &, wchar_t *);
public:
    CONSTRUCDECL_2ARG(Cvo_ChartLegend, const Cvo_Color &, char *)
    CONSTRUCDECL_2ARG(Cvo_ChartLegend, const Cvo_Color &, wchar_t *)
};

class Cvo_Chart : public Cvo_Window {
    friend class Cvo_GraphObject;
private:
    Cvo_Window      *legend;
    Cvo_ChartLegend **legends;
    Cvo_GraphObject *grafik;
    Cvo_Frame       *frame1;
    Cvo_Frame       *frame2;
    Cvo_Frame       *frame3;
    Cvo_Label       *label;
    Cvo_ChartOrient orientation;
protected:
    Cvo_ChartMark   *marks;
    Cvo_ChartMark   *cmarks;
    double          **columns;
    Cvo_Color       *colors;
    Cvo_Color       linecolor;
    Cvo_Color       backcolor;
    int             ncolumns;
    int             ncolors;
    Cvo_OptionMenu  *menu;
    Cvo_MenuItem    *orientlist;
    OrientData      *odata;
    wchar_t         *marklabel;
    wchar_t         *cmarklabel;
private:
    void _Init(int, Cvo_ChartOrient, Cvo_GraphObject *);
    void CloneEvent(XEvent *, void *);
    CVO_DEFINE_REGISTER_FUNCTIONS(Cvo_Chart)
public:
    CONSTRUCDECL_3ARG(Cvo_Chart, int, Cvo_ChartOrient=Cvo_ChartVertical, Cvo_GraphObject * = NULL)
    void SetTitle(char *);
    void SetTitle(wchar_t *);
    void SetLegend(int, char *);
    void SetLegend(int, wchar_t *);
    Cvo_GraphObject *Grafik()           { return(grafik); }

    void CreateColumns(int);
    void Set(unsigned i, unsigned j, double v)
        { if (i < ncolumns && j < ncolors && v >= 0.0 ) {
		columns[i][j] = v;
		if (grafik)
		    grafik->dirty = True;
	  } }
    void SetOrientation(Cvo_ChartOrient);
    Cvo_ChartOrient Orientation()	{ return(orientation); }

    Cvo_ChartMark *AddMark(char *, double);
    Cvo_ChartMark *AddColumnMark(char *, int);
    void RemoveMark(Cvo_ChartMark *sm);
    void RemoveColumnMark(Cvo_ChartMark *sm);

    void EvaluateMarks()		{ grafik->EvaluateMarks(); }
    void EvaluateColumnMarks()		{ grafik->EvaluateColumnMarks(); }

    void Clear();
    void ClearMarks();
    void ClearColumnMarks();
	
    void SetText(char *);
    void SetText(wchar_t *);
    void SetColumnText(char *);
    void SetColumnText(wchar_t *);

    void SetMinimum(double d)		{ grafik->SetMinimum(d); }
    double GetMinimum()			{ return(grafik->GetMinimum()); }

    void Redraw()      			{ Cvo_Window::Redraw();
					  if (grafik->pixmapbacking)
					    grafik->DrawIt(0, 0);
                          		  grafik->Redraw();
                          		  label->Redraw(); }
    void AddPeelOff();
    void AddPin();
    void AddOrientSelect(long);

    Cvo_Object *Clone(Cvo_Object *);

};

inline void
Cvo_GraphObject::CreateColumns(int i)
{
    chartob->CreateColumns(i);
}

inline Cvo_ChartMark *
Cvo_GraphObject::Marks()
{
    return(chartob->marks);
}

inline Cvo_ChartMark *
Cvo_GraphObject::Cmarks ()
{
    return (chartob->cmarks);
}

inline double **
Cvo_GraphObject::Columns ()
{
    return (chartob->columns);
}

inline int
Cvo_GraphObject::Ncolumns ()
{
    return (chartob->ncolumns);
}

inline int
Cvo_GraphObject::Ncolors ()
{
    return (chartob->ncolors);
}

inline Cvo_Color *
Cvo_GraphObject::Colors()
{
    return (chartob->colors);
}

inline Cvo_Color
Cvo_GraphObject::Linecolor()
{
    return (chartob->linecolor);
}

inline Cvo_Color
Cvo_GraphObject::Backcolor()
{
    return (chartob->backcolor);
}

class Cvo_VerticalGraph : public Cvo_GraphObject {
protected:
    friend          Cvo_Chart;
    int             mincolwidth;
    double          mincolheight;
    int             colwidth;
    int             coldepth;
    int             colseparation;
    int             fudge;
    double          scale;

    Cvo_Frame       *frame1;
    Cvo_Frame       *frame2;
    Cvo_TextBoard   *ticks;
    Cvo_TextBoard   *cticks;
    Cvo_GraphMark   *marklist;
    Cvo_GraphMark   *cmarklist;
    Cvo_Label	    *marklabel;
    Cvo_Label	    *cmarklabel;
private:
    void _Init(Cvo_Chart *);
public:
    void DrawIt(XEvent *, void *);
    CVO_DEFINE_REGISTER_FUNCTIONS(Cvo_VerticalGraph)
protected:
    Cvo_GraphObject *CloneMe(Cvo_Object *,Cvo_Chart*);
    void DrawBox(int x, int y, int h, const Cvo_Color &c);

    void SetText(wchar_t *);
    void SetColumnText(wchar_t *);

    void HandleEvent(XEvent*, void*);
public:
    CONSTRUCDECL_1ARG(Cvo_VerticalGraph, Cvo_Chart*)
    ~Cvo_VerticalGraph()
        { delete graph; delete frame1; delete frame2;
          ClearMarks(); ClearColumnMarks(); delete ticks; delete
          cticks;}

    void FillRhombus(Cvo_Window *,int,int,int,int,int=0,int=0);

    void DrawMarks();
    void EvaluateMarks();
    void EvaluateColumnMarks();

    Cvo_GraphMark* AddMark(Cvo_ChartMark*);
    Cvo_GraphMark* AddColumnMark(Cvo_ChartMark*);

    void ClearMarks();
    void ClearColumnMarks();
    void RemoveMark(Cvo_GraphMark *sm)
        { ticks->RemoveString(sm->item);
          sm->Unlink((Cvo_RootedList**)&marklist);
          delete sm; };
    void RemoveColumnMark(Cvo_GraphMark *sm)
        { cticks->RemoveString(sm->item);
          sm->Unlink((Cvo_RootedList**)&cmarklist);
          delete sm; };

    void Redraw() { Cvo_Window::Redraw();
                    graph->Redraw();
                    ticks->Redraw();
                    cticks->Redraw(); }

    void SetMinimum(double d) { mincolheight = (int)d; }
    double GetMinimum() { return((double)mincolheight); }

    int ComputeX(int i);
    CARD16 CalculateWidth();
    CARD16 CalculateHeight();
    void SetMinColumnWidth(int i)   { mincolwidth = i; }
    void SetMinColumnHeight(double d)  { mincolheight = d; }
};

class Cvo_HorizontalGraph : public Cvo_GraphObject {
protected:
    friend          Cvo_Chart;
    int             mincolwidth;
    double          mincolheight;
    int             colwidth;
    int             coldepth;
    int             colseparation;
    int             fudge;
    double          scale;

    Cvo_Frame       *frame1;
    Cvo_Frame       *frame2;
    Cvo_TextBoard   *ticks;
    Cvo_TextBoard   *cticks;
    Cvo_GraphMark   *marklist;
    Cvo_GraphMark   *cmarklist;
    Cvo_Label	    *marklabel;
    Cvo_Label	    *cmarklabel;

private:
    void _Init(Cvo_Chart *);
public:
    void DrawIt(XEvent *, void *);
    CVO_DEFINE_REGISTER_FUNCTIONS(Cvo_HorizontalGraph)
protected:
    Cvo_GraphObject *CloneMe(Cvo_Object *, Cvo_Chart*);
    void DrawBox(int x, int y, int h, const Cvo_Color &c);

    void SetText(wchar_t *);
    void SetColumnText(wchar_t *);

    void HandleEvent(XEvent*, void*);
public:
    CONSTRUCDECL_1ARG(Cvo_HorizontalGraph, Cvo_Chart*)
    ~Cvo_HorizontalGraph()
        { delete graph; delete frame1; delete frame2;
          ClearMarks(); ClearColumnMarks(); delete ticks; delete
          cticks; }

    Cvo_GraphMark* AddMark(Cvo_ChartMark*);
    Cvo_GraphMark* AddColumnMark(Cvo_ChartMark*);

    void FillRhombus(Cvo_Window *,int,int,int,int,int=0,int=0);

    void DrawMarks();
    void EvaluateMarks();
    void EvaluateColumnMarks();

    void ClearMarks();
    void ClearColumnMarks();
    void RemoveMark(Cvo_GraphMark *sm)
        { ticks->RemoveString(sm->item);
          sm->Unlink((Cvo_RootedList**)&marklist);
          delete sm; };
    void RemoveColumnMark(Cvo_GraphMark *sm)
        { cticks->RemoveString(sm->item);
          sm->Unlink((Cvo_RootedList**)&cmarklist);
          delete sm; };

    void Redraw() { Cvo_Window::Redraw();
                    graph->Redraw();
                    ticks->Redraw();
                    cticks->Redraw(); }

    int DHeight()                   { return(graph->Height() - coldepth - 1); }
    int DWidth()                    { return(graph->Width() - coldepth - 1); }
    double Scale()                  { return(scale); }
    int ComputeX(int i);
    int ColumnWidth()               { return(colwidth); }
    CARD16 CalculateWidth();
    CARD16 CalculateHeight();

    void SetMinimum(double d) { mincolheight = (int)d; }
    double GetMinimum() { return((double)mincolheight); }

    void SetMinColumnWidth(int i)   { mincolwidth = i; }
    void SetMinColumnHeight(double d)  { mincolheight = d; }
};

class Cvo_VerticalSideGraph : public Cvo_VerticalGraph {
private:
    void _Init(Cvo_Chart *);
public:
    CONSTRUCDECL_1ARG(Cvo_VerticalSideGraph, Cvo_Chart *)
    Cvo_GraphObject *CloneMe(Cvo_Object *,  Cvo_Chart *);
    void DrawIt(XEvent *, void *);
protected:
    void DrawBox(int, int, int, int, int, const Cvo_Color &);

    void HandleEvent(XEvent *, void *);
};

class Cvo_HorizontalSideGraph : public Cvo_HorizontalGraph {
private:
    void _Init(Cvo_Chart *);
public:
    CONSTRUCDECL_1ARG(Cvo_HorizontalSideGraph, Cvo_Chart *)
    Cvo_GraphObject *CloneMe(Cvo_Object *, Cvo_Chart *);
    void DrawIt(XEvent *, void *);
protected:
    void DrawBox(int, int, int, int, int, const Cvo_Color &);

    void HandleEvent(XEvent*, void*);
};

#define CvoChartEvent		(Cvo_ChartEventType + InternalEvent0)

struct Cvo_ChartEvent: Cvo_AnyEvent {
    int		bar;			// column pressed
    int		color;			// color of the pressed column
    double	value;			// value associated with the column
    int		button;			// button pressed
    unsigned int state;			// X State when button released
};

#endif
