# Copyright (c) 1996 Berkeley Software Design, Inc.
# All rights reserved.
# The Berkeley Software Design Inc. software License Agreement specifies
# the terms and conditions for redistribution.
#
#       BSDI PPP.pm,v 1.10 1998/09/24 16:13:50 chrisk Exp
#
# TODO: Eval around the code to catch errors and restore umask().

package AdminWEB::PPP;
require Exporter;
@ISA = qw(Exporter);
@EXPORT = qw(
    set_nif
    numif
    ppp_interfaces
    auth_protos
    set_pppauth
    local_ipaddr
    set_localaddr
    set_addrmap
    assign_pool
    set_chap_secret
    set_pppsys
);

use AdminWEB::Support;

sub set_nif {
    my ($type, $numif, $doit) = @_;
    my $MIB = "net.link_layer.generic.$type.numif";
    my ($rc, $rctmp, $found) = ("/etc/rc.local", "/etc/rc.local.tmp");

    open(RCIN, "< $rc") or die "$rc: $!\n";
    my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
	$atime,$mtime,$ctime,$blksize,$blocks) = stat($rc);
    my ($omask) = umask(0177);
    open(RCOUT, "> $rctmp") or die "$rctmp: $!\n";
    chown($uid, $gid, $rctmp);
    chmod($mode, $rctmp);
    while (<RCIN>) {
	# fixup any existing sysctl that matches what we are doing
	$found = 1 if s/^(#\s*)?(.*sysctl\s+-w\s*\Q${MIB}\E=).*/$2$numif/;
	print RCOUT $_;
    }
    if (!$found) {
	# didn't find an existing sysctl, add one ourselves
	print RCOUT "echo -n \"setting number of '$type' interfaces: \"; ";
	print RCOUT "sysctl -w ${MIB}=$numif\n";
    }
    close(RCIN);
    close(RCOUT) or die "Error writting $rctmp: $!\n";
    rename($rctmp, $rc) or die "Error moving $rctmp into $rc: $!\n";
    if ($doit) {
	my ($status, $output) =
	    run_backend('sysctl', '-w', "${MIB}=$numif");
	die "Failed running system -w ${MIB}=$numif\n@$output\n"
		unless $status == 0;
    }
    umask($omask);
}

sub numif {
    my ($type) = @_;
    my $nif;

    my ($status, $output) =
	run_backend('sysctl', '-n', "net.link_layer.generic.${type}.numif");
    $nif = $output->[0] + 0 if $status == 0;	# ensure numeric representation
    return $nif;				# undef on error
}

sub ppp_interfaces {
    my @list;
    @list = split(/[\r\n]/, `ifconfig -a`);
    @list = grep(s/^(ppp\d+):.*/$1/, @list);
    return \@list;
}

# XXX: needs more work to be robust
sub auth_protos {
    my ($status, $output) = run_backend('cat', '/etc/ppp.sys');
    my ($disabled, @auths);
    if ($status == 0) {
	$disabled = grep(/^# MaxIM disabled pap and chap/, @$output);

	@$output = grep(!/^#/, @$output);	# remove comments
	if (grep(/^ppp_direct:/, @$output)) {
	    push(@auths, 'PAP') if grep(/:pap:/, @$output);
	    push(@auths, 'CHAP') if grep(/:chap:/, @$output);
	}
	else {
	    # leave @auths empty if it's disabled;
	    # else default to both assuming that it's a new config
	    push(@auths, 'PAP', 'CHAP') unless $disabled;
	}
    }
    else {
	die "/etc/ppp.sys appears to be missing!\n";
    }
    return \@auths;
}

sub set_pppauth {
    my ($auths) = @_;
    my ($in, $outtmp) = ("/etc/ppp.sys", "/etc/ppp.sys.tmp");

    open(IN, "< $in") or die "$in: $!\n";
    my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
	$atime,$mtime,$ctime,$blksize,$blocks) = stat($in);
    my ($omask) = umask(0177);
    open(OUT, "> $outtmp") or die "$outtmp: $!\n";
    chown($uid, $gid, $outtmp);
    chmod($mode, $outtmp);
    # put it at the top of the file
    if (defined $auths && $auths ne '') {
	print OUT "ppp_direct:dialin:${auths}:auth-retries=3:\n";
    }
    else {
	print OUT "# MaxIM disabled pap and chap\n";
    }
    while (<IN>) {
	if (/^ppp_direct:/) {
	    while (/\\$/) { $_ = <IN>; }	# eat cont. lines
	    next;
	}
	print OUT $_;
    }
    close(IN);
    close(OUT) or die "Error writting $outtmp: $!\n";
    rename($outtmp, $in) or die "Error moving $outtmp into $in: $!\n";
    umask($omask);
}

use AdminWEB::Netstart;
sub local_ipaddr {
    my $ip;
    $ip = `grep ^LOCAL /etc/netscripts/addr-map`;
    chomp $ip;
    $ip =~ s/^LOCAL\s*//;
    if ($ip eq '') {
	eval {
	    $netstart = new AdminWEB::Netstart;
	    $AdminWEB::Netstart::inet{$netstart->primary} =~ m/^\s*([\d.]+)\s+/;
	    $ip = $1;
	};
    }
    $ip;				# undef if any problems
}

sub set_localaddr {
    my ($ip) = @_;
    set_addrmap('LOCAL', $ip, 1);
}

sub set_addrmap {
    my ($key, $value, $top) = @_;
    my ($map, $maptmp) = ("/etc/netscripts/addr-map",
	"/etc/netscripts/addr-map.tmp");

    open(MAPIN, "< $map") or die "$map: $!\n";
    my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
	$atime,$mtime,$ctime,$blksize,$blocks) = stat($map);
    my ($omask) = umask(0177);
    open(MAPOUT, "> $maptmp") or die "$maptmp: $!\n";
    chown($uid, $gid, $maptmp);
    chmod($mode, $maptmp);
    print MAPOUT "$key\t$value\n" if $top;
    while (<MAPIN>) {
	next if /^$key/;
	print MAPOUT $_;
    }
    print MAPOUT "$key\t$value\n" if !$top;
    close(MAPIN);
    close(MAPOUT) or die "Error writting $maptmp: $!\n";
    rename($maptmp, $map) or die "Error moving $maptmp into $map: $!\n";
    umask($omask);
}

sub assign_pool {
    my($firstip,$startif,$numif) = @_;
    my ($ppp, @new);
    my ($in, $outtmp) = ("/etc/netscripts/addr-map", 
        "/etc/netscripts/addr-map.tmp");
    open(MAPIN, "< $in") or die "$in: $!\n";  
    my (@map) = <MAPIN>;
    close(MAPIN);

    my $curppp = $startif; $curppp =~ s/^ppp//;
    my $count = $numif;
    my $curip = $firstip;
    while ($count--) {
	push(@new, "ppp$curppp $curip\n");
	$curppp++;
	$curip =~ s/(\d+)$/($1+1)/e;	# incr curip
    }

    # remove the old lines
    foreach $_ (@new) {
	($ppp) = (m/^(ppp\d+)/);
	@map = grep(!/^${ppp}\b/, @map);
    }
    # add the new ones
    push(@map, @new);

    my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
	$atime,$mtime,$ctime,$blksize,$blocks) = stat($in);
    my ($omask) = umask(0177);
    open(MAPOUT, "> $outtmp") or die "$outtmp: $!\n";
    chown($uid, $gid, $outtmp);
    chmod($mode, $outtmp);
    foreach $_ (@map) {
	print MAPOUT $_;
    }
    close(MAPOUT) or die "Error writting $outtmp: $!\n";
    rename($outtmp, $in) or die "Error moving $outtmp into $in: $!\n";  
    umask($omask);
}

sub set_chap_secret {
    my($name, $secret) = @_;

    my ($in) = '/etc/chap_md5_secrets';
    my ($outtmp) = $in . '.tmp';
    open(IN, "< $in") or die "$in: $!\n";  
    my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
	$atime,$mtime,$ctime,$blksize,$blocks) = stat($in);
    my ($omask) = umask(0177);
    open(OUT, "> $outtmp") or die "$outtmp: $!\n";
    chown($uid, $gid, $outtmp);
    chmod($mode, $outtmp);
    while (<IN>) {
	next if /^${name}:/;
	print OUT $_;
    }
    print OUT "$name:ascii:$secret:\n";
    close(IN);
    close(OUT) or die "Error writting $outtmp: $!\n";
    rename($outtmp, $in) or die "Error moving $outtmp into $in: $!\n";  
    umask($omask);
}

sub set_pppsys {
    my ($key, $value) = @_;
    my $in = "/etc/ppp.sys";
    my $out = $in . "tmp";

    open(IN, "< $in") or die "$in: $!\n";
    my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,
	$atime,$mtime,$ctime,$blksize,$blocks) = stat($in);
    my ($omask) = umask(0177);
    open(OUT, "> $out") or die "$out: $!\n";
    chown($uid, $gid, $out);
    chmod($mode, $out);
    while (<IN>) {
	if (/^${key}:/) {
	    while (/\\$/) { $_ = <IN>; }	# eat cont. lines
	    next;
	}
	print OUT $_;
    }
    print OUT "$key:$value\n";
    close(IN);
    close(OUT) or die "Error writting $out: $!\n";
    rename($out, $in) or die "Error moving $out into $in: $!\n";
    umask($omask);
}

1;
