# Copyright (c) 1996 Berkeley Software Design, Inc.
# All rights reserved.
# The Berkeley Software Design Inc. software License Agreement specifies
# the terms and conditions for redistribution.
#
#	BSDI Properties.pm,v 1.8 1999/01/13 23:26:25 tks Exp
#

=head1 NAME

DB::Properties - Simple Text-Format Database

=head1 SYNOPSIS

require DB::Properties;
$properties = Create DB::Properties($file);
$properties->writable or die "$file is read-only";
$properties->set('Property', 'Value');
print $properties->value('Property');
$properties->store;

=head1 DESCRIPTION

Manages a text-format database for storing simple strings.
The Properties file is locked while the object exists.

By leaving off $file you can manage properties in memory
without an underlying save file.  You can also use
the noautosave method to disable saving the file so you
can load properties from a file but not write them out.
Autosave is automatically disabled if the file is read-only.

=cut

package DB::Properties;
use IO::LockingFile;
use Carp;

=head1 METHODS

=over 4

=item Create([$file[, $timeout]]) [STATIC]

Creates an object of type DB::Properties.
Loads $file if present.
$timeout is in seconds (default 30).

=cut
sub Create {
    my $self = bless { }, shift;
    my ($filename, $timeout) = @_;

    $self->{'Filename'} = '';
    $self->{'FileHandle'} = undef;
    $self->{'Properties'} = { };
    $self->{'LOADED'} = 0;			# file been loaded?
    $self->{'READONLY'} = 1;			# file read-only?
    $self->{'CHANGED'} = 0;			# have we changed?
    $self->{'AUTOSAVE'} = 1;			# save on DESTROY?
    $self->load($filename, $timeout) if $filename;
    $self;
}

=item load($file[, $timeout])

Loads the object from $file.  Sets save file to $file.
$timeout is in seconds (default 30).

=cut
sub load {
    my $self = shift;
    my ($filename, $timeout) = @_;
    my ($record);

    $self->setfile($filename, $timeout);

    my $fh = $self->{'FileHandle'};
    {
	local($/) = undef;	# snarf entire file
	$record = <$fh>;
    }
    undef $fh;
    $self->parserec($record) if defined $record && $record ne '';
    $self->{'LOADED'} = 1;	# file has been loaded.
    $self;
}

=item store

Stores the object into a file.  You can specify an alternate file
for the object by calling setfile().  Errors are fatal.

=cut
sub store {
    my $self = shift;

    croak "Attempted to store DB::Properties object with no file."
	unless defined $self->{'FileHandle'} && defined $self->{'Filename'};

    # write tmp file and then move into place for atomic updates
    my $newfile = $self->{'Filename'} . '.new';
    unlink($newfile);
    my $fh = new IO::LockingFile($newfile, O_WRONLY|O_CREAT|O_EXCL);
    croak "$newfile: $!" unless defined $fh;

    print $fh "# Simple-Text-Database\n";
    print $fh "# Format is: <Property>:<tab><Value>\n";
    print $fh $self->stringify;

    # swap newfile for oldfile
    $fh->close or die "$newfile: $!\n"; undef $fh;
    rename $newfile, $self->{'Filename'} or
         die "rename $newfile " . $self->{'Filename'} . ": $!\n";

    $self->{'FileHandle'} = undef;

    $self;
}

=item setfile($filename[,$timeout])

Sets up $filename to be the object store.  File is locked.
Locking $timeout is in seconds (default 30).

=cut
sub setfile {
    my $self = shift;
    my $filename = shift;
    my $timeout = shift || 30;
    my $fh;

    delete $self->{'FileHandle'};

    $self->{'READONLY'} = 0;
    $fh = new IO::LockingFile($filename, O_RDONLY|O_CREAT);
    croak "$filename: $!" unless defined $fh;
    $fh->exclusive($timeout) or croak "locking $filename: $!";
    $self->{'FileHandle'} = $fh;
    $self->{'Filename'} = $filename;
    undef $fh;
    $self;
}

=item writable

Returns true if the loaded object is writable.

=cut
sub writable {
    my $self = shift;
    $self->{'FileHandle'} && ! $self->{'READONLY'};
}

### PRIVATE METHOD: parserec($record)
###
### Parses the record into the object.
sub parserec {
    my $self = shift;
    my ($record) = @_;
    my ($prop, $val, $line, @lines);

    my $props = $self->{'Properties'};		# HASH REF
    @lines = split(/\n/, $record);
    foreach $line (@lines) {
	next if $line =~ /^$/ || $line =~ /^#/;
	($prop, $val) = split(/:/, $line, 2);
	$val =~ s/^\s*//;
	$self->append($prop, defined $val ? $val : '');
    }
    $self;
}

### PRIVATE METHOD: stringify()
###
### convert object state into a string (parseable with parserec()).
sub stringify {
    my $self = shift;
    my ($str, $prop) = '';
    my $props = $self->{'Properties'};		# HASH REF
    foreach $prop (sort keys %$props) {
	if (ref $props->{$prop} eq 'ARRAY') {
	    ### prop is an array of entries
	    map {
		$str .= join('', $prop, ":\t", $_, "\n");
	    } @{$props->{$prop}};
	}
	else {
	    ### simple scalar
	    $str .= join('', $prop, ":\t", $props->{$prop}, "\n");
	}
    };
    $str;
}

=item properties

Returns list of the defined properties for this object.

=cut
sub properties {
    my $self = shift;
    keys %{$self->{'Properties'}};
}

=item exists($property)

Returns true if the property exists for this object.

=cut
sub exists {
    my $self = shift;
    my ($property) = @_;
    exists $self->{'Properties'}->{$property};
}

=item defined($property)

Returns true if the property is defined for this object.

=cut
sub defined {
    my $self = shift;
    my ($property) = @_;
    defined $self->{'Properties'}->{$property};
}

=item islist($property)

Returns true if the value of $property is an ARRAY REF.

=cut
sub islist {
    my $self = shift;
    my ($property) = @_;
    my $props = $self->{'Properties'};		# HASH REF
    ref $props->{$property} eq 'ARRAY';
}

=item value($property)

Returns the value associated with the property.  If islist($property)
is true then we return a LIST of values.

=cut
sub value {
    my $self = shift;
    my ($property) = @_;
    my $props = $self->{'Properties'};		# HASH REF
    if (wantarray) {
	$self->islist($property) ?
	    @{$props->{$property}} : $props->{$property};
    }
    else {
	$self->islist($property) ?
	    join(' ', @{$props->{$property}}) : $props->{$property};
    }
}

=item set($property, $value[, ...])

Set $property to $value clearing any existing values.
A list of values can be used.

=cut
sub set {
    my $self = shift;
    my ($property, @values) = @_;
    my $props = $self->{'Properties'};		# HASH REF
    if (@values > 1) {
	$props->{$property} = [ @values ];
    }
    else {
	$props->{$property} = $values[0];
    }
    $self->{'CHANGED'} = 1 if $self->{'LOADED'};
    $self;
}

=item append($property, $value[, ...])

Appends the value(s) to the property.

=cut
sub append {
    my $self = shift;
    my ($property, @values) = @_;
    my $props = $self->{'Properties'};		# HASH REF

    ### lazy stack creation is to save memory for the most common case
    if (defined $props->{$property}) {
	if (ref $props->{$property} eq 'ARRAY') {
	    ### push onto the stack
	    push (@{$props->{$property}}, @values);
	}
	else {
	    ### change single value into a stack and append
	    $props->{$property} = [ $props->{$property}, @values ];
	}
    }
    else {
	$self->set($property, @values);
    }
    $self->{'CHANGED'} = 1 if $self->{'LOADED'};
    $self;
}

=item noautosave

Prevents auto-store on DESTROY.

=cut
sub noautosave {
    my $self = shift;
    $self->{'AUTOSAVE'} = 0;
}

=item autosave

Enables auto-store on DESTROY [the default if a writable file is available].

=cut
sub autosave {
    my $self = shift;
    $self->{'AUTOSAVE'} = 1;
}

### PRIVATE METHOD: DESTROY
###
### Auto-saves if something changed, save
### is enabled, we aren't memory-only,
### and an underlying file exists for this object.
sub DESTROY {
    my $self = shift;
    $self->store
	if $self->{'CHANGED'} && $self->{'AUTOSAVE'} && $self->writable;
    # $self->{'FileHandle'}->close;
    delete $self->{'FileHandle'};
}

=back

=cut

1;
