# Copyright (c) 1995, 1996 Berkeley Software Design, Inc.
# All rights reserved.
# The Berkeley Software Design Inc. software License Agreement specifies
# the terms and conditions for redistribution.
#
#       BSDI Linkable.pm,v 1.3 1996/08/28 20:17:31 sanders Exp
#
# TODO: search, equal, mark, return

package DataType::Linkable;
use Carp;

=head1 NAME

DataType::Linkable - linkable object module

=head1 SYNOPSIS

    require DataType::Linkable;
    @ISA = qw(DataType::Linkable);

=head1 DESCRIPTION

Manage a tree-structured object hierarchy.

The following methods create and manage objects:
Create, Initialize, Clone, Copy

The following methods return object state information:
empty, position, offleft, offright, children, child, all_children.

The following methods modify state information and return $self:
first, last, next, previous, i_th.

The following methods manipulate the object hierarchy and return $self:
append, prefix, before, after, delete, delete_all, replace, swapleft, swapright.

=head1 METHODS

=over 4

=cut

# =============================================================================
# Methods

sub DESTROY {
    # nuke the ARRAY so that references are freed.
    my $self = shift;
    delete $self->{'__DataType::Linkable::children'};
}


=item Create I<static>

Create a new object, calls B<Initialize>.

=cut
sub Create {
    my $self = bless { }, shift;
    $self->Initialize(@_);
}

=item Initialize

Called by Create and Clone to initialize the new object.

=cut
sub Initialize {
    my $self = shift;
    $self->{'__DataType::Linkable::position'} = 0;
    $self->{'__DataType::Linkable::children'} = [ ];
    $self;
}

=item $object->Clone

Clone an existing object hierarchy.

    $new_object = $old_object->Clone;

First, it blesses the newly created C<$new_object> and then
calls C<$new_object->Copy($old_object);> to do the copy.

=cut
sub Clone {
    my $self = shift;
    my $clone = bless { }, ref $self;
    $clone->Initialize(@_);
    $clone->Copy($self);
    $clone;
}

=item $dest_object->Copy(I<source_object>)

Copy the contents of I<$source_object> into I<$dest_object>.

=cut
sub Copy {
    my $self = shift;
    my $from = shift;
    map { $self->append($_->Clone) } $from->all_children;
    # copy our private state
    $self->{'__DataType::Linkable::position'} =
	$from->{'__DataType::Linkable::position'};
    $self;
}

=item position

Returns the current child index (which may or may not be valid
depending on the state of the object).

=cut
sub position {
    my $self = shift;
    $self->{'__DataType::Linkable::position'};
}

=item offleft

Returns true if B<position> is off to the left (too small) of being valid.

=cut
sub offleft {
    my $self = shift;
    ($self->empty) || ($self->position < 0);
}

=item offright

Returns true if B<position> is off to the right (too large) of being valid.

=cut
sub offright {
    my $self = shift;
    ($self->empty) || ($self->position >= $self->children);
}

=item children

Returns the number of children in this container.

=cut
sub children {
    my $self = shift;
    scalar(@{$self->{'__DataType::Linkable::children'}});
}

=item empty

Returns true if this container has no children.

=cut
sub empty {
    my $self = shift;
    ($self->children == 0);
}

=item append ($object[,$object...])

Appends objects to the list of children

=cut
# XXX: should append automatically set position: $self->last;
sub append {
    my $self = shift;
    push (@{$self->{'__DataType::Linkable::children'}}, @_);
    $self;
}
# link() is an alias for append()
use vars qw(&link);
*link = \&append;

=item prefix ($object[,$object...])

Prepends the objects to the list of children

=cut
sub prefix {
    my $self = shift;
    unshift (@{$self->{'__DataType::Linkable::children'}}, @_);
    $self;
}

=item before ($object[,$object...])

Insert objects before current B<position>.

=cut
sub before {
    my $self = shift;
    splice(@{$self->{'__DataType::Linkable::children'}}, $self->position, 0, @_);
    $self;
}
use vars qw(&insert);
*insert = \&before;

=item after ($object[,$object...])

Appends objects after current B<position>.

=cut
sub after {
    my $self = shift;
    splice(@{$self->{'__DataType::Linkable::children'}}, $self->position + 1, 0, @_);
    $self;
}

=item delete

Removes the object pointed to by B<position> from the list.

=cut
sub delete {
    my $self = shift;
    splice(@{$self->{'__DataType::Linkable::children'}}, $self->position, 1);
    $self;
}

=item delete_all

Removes all objects from the list of children.

=cut
sub delete_all {
    my $self = shift;
    $self->{'__DataType::Linkable::children'} = [ ];
    $self->{'__DataType::Linkable::position'} = 0;
    $self;
}

=item replace ($object[,$object...])

Replaces the current object pointed to by B<position> with the
given objects.

=cut
sub replace {
    my $self = shift;
    splice(@{$self->{'__DataType::Linkable::children'}}, $self->position + 1, 1, @_);
    $self;
}

=item swapleft

Swaps the current object pointed to by B<position> with
the object to our left.

=cut
sub swapleft {
    my $self = shift;
    my $pos = $self->position;
    my $save = $self->{'__DataType::Linkable::children'}->[$pos];
    $self->{'__DataType::Linkable::children'}->[$pos] = $self->{'__DataType::Linkable::children'}->[$pos-1];
    $self->{'__DataType::Linkable::children'}->[$pos-1] = $save;
    $self;
}

=item swapright

Swaps the current object pointed to by B<position> with
the object to our right.

=cut
sub swapright {
    my $self = shift;
    my $pos = $self->position;
    my $save = $self->{'__DataType::Linkable::children'}->[$pos];
    $self->{'__DataType::Linkable::children'}->[$pos] = $self->{'__DataType::Linkable::children'}->[$pos+1];
    $self->{'__DataType::Linkable::children'}->[$pos+1] = $save;
    $self;
}

=item first

Sets B<position> to point to the first child.

=cut
sub first {
    my $self = shift;
    $self->{'__DataType::Linkable::position'} = 0;
    $self;
}

=item last

Sets B<position> to point to the last child.

=cut
sub last {
    my $self = shift;
    $self->{'__DataType::Linkable::position'} = $self->children - 1;
    $self;
}

=item next

Increments B<position> to point to the next child.

=cut
sub next {
    my $self = shift;
    $self->{'__DataType::Linkable::position'}++;
    $self;
}

=item previous

Decrements B<position> to point to the previous child.

=cut
sub previous {
    my $self = shift;
    $self->{'__DataType::Linkable::position'}--;
    $self;
}

=item i_th (position)

Sets B<position> to point to the i'th child.

=cut
sub i_th {
    my $self = shift;
    $self->{'__DataType::Linkable::position'} = shift;
    $self;
}

=item child

Returns the current child object pointed to by B<position>.

=cut
sub child {
    my $self = shift;
    defined $self->{'__DataType::Linkable::children'}->[$self->{'__DataType::Linkable::position'}]
	? $self->{'__DataType::Linkable::children'}->[$self->{'__DataType::Linkable::position'}] : undef;
}

=item all_children

Returns an array of all the children.

=cut
sub all_children {
    my $self = shift;
    @{$self->{'__DataType::Linkable::children'}}
}

=back

=cut

1;
