# Copyright (c) 1996 Berkeley Software Design, Inc.
# All rights reserved.
# The Berkeley Software Design Inc. software License Agreement specifies
# the terms and conditions for redistribution.
#
#       BSDI NamedTree.pm,v 1.4 1996/08/28 20:19:29 sanders Exp

package DataType::NamedTree;
use Carp;

=head1 NAME

DataType::NamedTree - named tree (recursive)

=head1 SYNOPSIS

    require DataType::NamedTree;
    @ISA = qw(DataType::NamedTree);

=head1 DESCRIPTION

Manage a tree-structured object hierarchy of named objects.

The following methods create and manage objects:
    Create, Initialize

The following methods return object state information:
    childrens_names, exists, isleaf, leaves, maxdepth, name,
    namedchild, notleaf

The following methods manipulate the object hierarchy and return $self:
    delete, delete_all, set


=head1 METHODS

=over 4

=cut

# =============================================================================
# Methods

sub DESTROY {
    # nuke the HASH so that references are freed.
    my $self = shift;
    delete $self->{'__DataType::NamedTree::children'};
}

=item Create ($name) I<static>

Create a new object and calls B<Initialize> for it.

=cut
sub Create {
    my $self = bless { }, shift;
    $self->Initialize(@_);
}

=item Initialize [($name)]

Called by Create to initialize the new object.

=cut
sub Initialize {
    my $self = shift;
    my $name = shift;
    $self->{'__DataType::NamedTree::name'} = $name;	# our name
    $self->{'__DataType::NamedTree::parent'} = undef;	# our parent (if any)
    $self->{'__DataType::NamedTree::children'} = { };	# child HASH
    $self->{'__DataType::NamedTree::leaves'} = 0;	# count leaf children
    $self->{'__DataType::NamedTree::maxdepth'} = 0;	# tree depth
    $self;
}

=item childrens_names

Returns the names of all stored children.

=cut
sub childrens_names {
    my $self = shift;
    keys %{$self->{'__DataType::NamedTree::children'}};
}

=item namedchild ($name)

Returns the object for the named child (undef if it doesn't exist).

=cut
sub namedchild {
    my $self = shift;
    my ($name) = @_;
    $self->{'__DataType::NamedTree::children'}->{$name};
}

=item exists ($name)

Returns true if this object has a child named $name.

=cut
sub exists {
    my $self = shift;
    my ($childname) = @_;
    exists $self->{'__DataType::NamedTree::children'}->{$childname};
}

=item maxdepth

Returns the depth of the object tree under this object.

=cut
sub maxdepth {
    my $self = shift;
    $self->{'__DataType::NamedTree::maxdepth'};
}

=item name

Returns the objects name which must be set at creation time and cannot
be changed during the life of the object.

=cut
sub name {
    my $self = shift;
    $self->{'__DataType::NamedTree::name'};
}

=item leaves

Returns the number of leaf children under this object.

=cut
sub leaves {
    my $self = shift;
    $self->{'__DataType::NamedTree::leaves'};
}

# increments #leaves
sub addleaves { ## PRIVATE
    my $self = shift;
    my $count = shift || 1;
    $self->{'__DataType::NamedTree::leaves'} += $count;
    $self;
}

# decrements #leaves
sub subleaves { ## PRIVATE
    my $self = shift;
    my $count = shift || 1;
    $self->{'__DataType::NamedTree::leaves'} -= $count;
    $self;
}

=item isleaf

Returns true if this object is a leaf object (i.e., contains no children).

=cut
sub isleaf {
    my $self = shift;
    $self->{'__DataType::NamedTree::leaves'} == 0;
}

=item notleaf

Returns true if this object has children.

=cut
sub notleaf {
    my $self = shift;
    $self->{'__DataType::NamedTree::leaves'} > 0;
}

=item parent ($parental_unit)

Returns the parent object of the object.  Root objects have
no parent and return C<undef>.

=cut
sub parent {
    my $self = shift;
    $self->{'__DataType::NamedTree::parent'};
}

# Sets the object's parent to be $parental_unit.
sub setparent { ## PRIVATE
    my $self = shift;
    my ($parental_unit) = @_;
    $self->{'__DataType::NamedTree::parent'} = $parental_unit;
    $self;
}

=item delete($name)

Removes the named object from our tree.

=cut
sub delete {
    my $self = shift;
    my ($name) = @_;
    my $object = $self->namedchild($name);

    # adjust the tree's maxdepth and leaf count.

    my $odepth = $object->maxdepth;	# depth of $object's tree
    my $oleaves = $object->leaves;	# # leaves of $object we're loosing
    $oleaves++ if $object->isleaf;	# or we're loosing it as a leaf
    # and if *we* just became a leaf then we get one back
    $oleaves-- if $self->leaves == $oleaves;

    if ($oleaves > 0) {
	# adjust by $oleaves all the way up the tree
	my $parents = $self;
	while ($parents) {
	    $parents->subleaves($oleaves);
	    $parents = $parents->parent;
	}
    }

    # delete our reference to $object
    $object->setparent(undef);		# uproot the object
    delete $self->{'__DataType::NamedTree::children'}->{$name};

    # adjust maxdepth if we deleted our deepest child
    if ($self->maxdepth == $omaxdepth + 1) {
	# find our deepest remaining child
	my ($depth, $child);
	foreach $child (values %{$self->{'__DataType::NamedTree::children'}}) {
	    $depth = $child->maxdepth if $child->maxdepth > $depth;
	}
	# carry on if we find another child at the same depth
	if ($depth < $omaxdepth) {
	    # otherwise, we'll have to adjust the tree
	    # we loop through our parents incrementing maxdepth until we
	    # hit a parent with a deeper tree somewhere else.
	    my $p = $self;
	    $self->{'__DataType::NamedTree::maxdepth'} = ++$depth;
	    for($p=$p->parent; $p; $p=$p->parent) {
		if (++$depth > $p->{'__DataType::NamedTree::maxdepth'}) {
		    $p->{'__DataType::NamedTree::maxdepth'} = $depth;
		}
		else {
		    last;
		}
	    }
	}

    }

    $self;
}

=item delete_all

Removes all objects from the list of children.

=cut
sub delete_all {
    my $self = shift;
    map { $self->delete($_) } $self->childrens_names;
    $self;
}

=item set ($object)

Sets the named object in the list of children for this object.
The object's name (as determined by C<$object->name>) must be not
be the same as an existing child in this object (of course, objects
at different places in the tree may well have the same name, they
just cannot have the same parent).

=cut
sub set {
    my $self = shift;
    my ($object) = @_;

    croak("Cannot overwrite duplicate child")
	if $self->exists($object->name);

    ### update maxdepth attributes for the tree
    # if the new object's depth is deeper than we currently hold
    # then we need to update the tree.
    my $p = $self;
    if ($object->maxdepth + 1 > $p->maxdepth) {
	my $depth = $object->maxdepth + 1;
	$p->{'__DataType::NamedTree::maxdepth'} = $depth;

	# loop through our parents incrementing maxdepth until we
	# hit a parent with a deeper tree somewhere else.
	for($p=$p->parent; $p; $p=$p->parent) {
	    if (++$depth > $p->{'__DataType::NamedTree::maxdepth'}) {
	    	$p->{'__DataType::NamedTree::maxdepth'} = $depth;
	    }
	    else {
		last;
	    }
	}
    }

    # add the object and set it's parent to us
    $self->{'__DataType::NamedTree::children'}->{$object->name} = $object;
    $object->setparent($self);

    # figure the number of leaves being gained (might be 0);
    my $leaves = $object->leaves;
    $leaves++ if $object->isleaf;	# you win some...
    if ($self->isleaf) {
	# we always get all of our childs leaves
	$self->addleaves($leaves);
	# but our parents are losing a leaf (us)
	$leaves--;
    }
    else {
	$self->addleaves($leaves);
    }

    if ($leaves > 0) {
	# adjust by $leaves all the way up the tree
	my $parents = $self->parent;
	while ($parents) {
	    $parents->addleaves($leaves);
	    $parents = $parents->parent;
	}
    }

    $self;
}

=back

=cut

1;
