# Copyright (c) 1995, 1996 Berkeley Software Design, Inc.
# All rights reserved.
# The Berkeley Software Design Inc. software License Agreement specifies
# the terms and conditions for redistribution.
#
#       BSDI Calendar.pm,v 1.1 1996/08/06 23:20:53 sanders Exp

package FileFormat::Calendar;
use Carp;

=head1 NAME

B<Calendar> - process unix calendar files

=head1 SYNOPSIS

Usage:
    require FileFormat::Calendar;
    $cal = Create FileFormat::Calendar;
    $cal->entry("6/15 Payday");
    $cal->readfile("$ENV{HOME}/calendar");
    print $cal->messages_for("MM/DD");		# MM is 1-12/DD is 1-31
    print $cal->messages_for("daily/DW");	# DW is 0-6 (sun-sat)
    print $cal->messages_for("monthly/DD");	# DD is 1-31

    Where messages_for -> "msg1\nmsg2\n...\n"

=head1 FORMATS

Date formats supported:
    6/15     ... June 15 (if ambiguous uses month/day).
    Jun. 15  ... June 15.
    15 June  ... June 15.
    Thursday ... Every Thursday.
    June     ... Every June 1st.
    15 *     ... 15th of every month.

=cut

# Package Globals: $system_calendar %days $days %months $months
BEGIN {
    $system_calendar = "/usr/share/calendar/";
    %days = (
	'Sunday',	0,
	'Monday',	1,
	'Tuesday',	2,
	'Wednesday',	3,
	'Thursday',	4,
	'Friday',	5,
	'Saturday',	6,
	'Sun',		0,
	'Mon',		1,
	'Tue',		2,
	'Wed',		3,
	'Thu',		4,
	'Fri',		5,
	'Sat',		6,
    );
    $days = join('|', keys %days);

    %months = (
	'January',	1,
	'February',	2,
	'March',	3,
	'April',	4,
	'May',		5,
	'June',		6,
	'July',		7,
	'August',	8,
	'September',	9,
	'October',	10,
	'November',	11,
	'December',	12,
	'Jan',		1,
	'Feb',		2,
	'Mar',		3,
	'Apr',		4,
	'May',		5,
	'Jun',		6,
	'Jul',		7,
	'Aug',		8,
	'Sep',		9,
	'Oct',		10,
	'Nov',		11,
	'Dec',		12,
    );
    $months = join('|', keys %months);
}

# Create FileFormat::Calendar([$file])
sub Create {
    my $self = bless { }, shift;
    $self->Initialize(@_);
}

sub Initialize {
    my $self = shift;
    my $file = shift;
    $self->{'msg'} = { };
    $self->{'includes'} = { };
    $self->readfile($file) if defined $file;
    $self;
}

sub msg {
    my $self = shift;
    $self->{'msg'};
}

sub messages_for {
    my $self = shift;
    $self->{'msg'}->{@_};
}

sub readfile {
    my $self = shift;
    my $calendar = shift;

    $self->{'includes'}->{$calendar} = 1;	# been there...

    local(*CAL, $_);
    if (open(CAL, "< $calendar")) {
	while ($_ = <CAL>) {
	    next if /^\s*$/;
	    if (/^#include <(.*)>/) {
		my $file = $1;
		$file = $system_calendar . $1 unless $file =~ m,^/,;
		next if defined $self->{'includes'}->{$file};	# ...done that
		$self->readfile($file);
	    }
	    if (/^#include "(.*)"/) {
		my $file = $1;
		$file = $ENV{'HOME'} . "/" . $1 unless $file =~ m,^/,;
		next if defined $self->{'includes'}->{$file};	# ...done that
		$self->readfile($file);
	    }
	    next if /^#/;

	    $self->entry($_);
	}
    }
    close(CAL);
}

sub entry {
    my $self = shift;
    local($_) = shift;

    my $digits = '[0-9][0-9]?';

    if (m/^\s*0*($digits)\/0*($digits)\*?\s+(.*)/i) {
	# 6/15     ... June 15 (default to month/day).
	local($a, $b, $msg) = ($1, $2, $3);
	if (&monthok($a) && &dayok($b, $a)) {
	    $self->{'msg'}->{$a . "/" . $b} .= $msg . "\n";
	}
	elsif (&monthok($b) && &dayok($a, $b)) {
	    $self->{'msg'}->{$b . "/" . $a} .= $msg . "\n";
	}
	else {
	    carp "syntax error (month/day, day/month): $_\n";
	}
	next;
    }
    elsif (m/^\s*($months)\.?\s+0*($digits)\*?\s+(.*)/i) {
	# Jun. 15  ... June 15.
	local($a, $b, $msg) = ($1, $2, $3);
	$a = $months{$a};
	if (!&dayok($b, $a)) {
	    carp "syntax error (Month day): $_\n";
	    next;
	}
	$self->{'msg'}->{$a . "/" . $b} .= $msg . "\n";
	next;
    }
    elsif (m/^\s*0*($digits)\s+($months)\*?\s+(.*)/i) {
	# 15 June  ... June 15.
	local($a, $b, $msg) = ($1, $2, $3);
	$b = $months{$b};
	if (!&dayok($a, $b)) {
	    carp "syntax error (day Month): $_\n";
	    next;
	}
	$self->{'msg'}->{$b . "/" . $a} .= $msg . "\n";
	next;
    }
    elsif (m/^\s*($days)\*?\s+(.*)/i) {
	# Thursday ... Every Thursday.
	local($a, $msg) = ($1, $2);
	$self->{'msg'}->{"daily/" . $days{$a}} .= $msg . "\n";
	next;
    }
    elsif (m/^\s*($months)\s+(.*)/i) {
	# June     ... Every June 1st.
	local($a, $msg) = ($1, $2);
	$a = $months{$a};
	$self->{'msg'}->{$a . "/1"} .= $msg . "\n";
	next;
    }
    elsif (m/^\s*0*($digits)\s+\*\s+(.*)/i) {
	# 15 *     ... 15th of every month.
	local($a, $msg) = ($1, $2);
	if (!&dayok($a, 1)) {
	    carp "syntax error (day *): $_\n";
	    next;
	}
	$self->{'msg'}->{"monthly/" . $a} .= $msg . "\n";
	next;
    }
    else {
	# we just ignore it
	# carp "Unknown calendar format: $_\n";
	next;
    }
}

#######################################################################
##### Private Functions
#######################################################################

sub monthok {
    local($month) = @_;
    $month > 0 && $month <= 12;
}
sub dayok {
    local($day, $month) = @_;
    local(@days) = (31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31);
    $day > 0 && $day <= $days[$month-1];
}

1;
