# Copyright (c) 1995, 1996 Berkeley Software Design, Inc.
# All rights reserved.
# The Berkeley Software Design Inc. software License Agreement specifies
# the terms and conditions for redistribution.
#
#       BSDI Cal.pm,v 1.2 1996/08/09 00:25:37 sanders Exp
#
# Cal.pm -- generate an HTML table in the form of a calendar
#
# TODO: Class documentation

package HTML::Markup::Cal;
use Carp;

BEGIN {
    $cal_default = "/usr/bin/cal";

    $empty = "<BR><BR><BR><BR><BR>";

    %months = (
	'January',	 1, 'February',	 2, 'March',	 3,
	'April',	 4, 'May',	 5, 'June',	 6,
	'July',		 7, 'August',	 8, 'September', 9,
	'October',	10, 'November',	11, 'December',	12,
	'Jan',		 1, 'Feb',	 2, 'Mar',	 3,
	'Apr',		 4, 'May',	 5, 'Jun',	 6,
	'Jul',	 	 7, 'Aug',	 8, 'Sep',	 9,
	'Oct',		10, 'Nov',	11, 'Dec',	12,
    );
}

# =============================================================================
# Public Methods

# Create HTML::Markup::Cal($month, $year[, \%msg[, $calcmd]]);
sub Create {
    my $self = bless { }, shift;
    $self->Initialize(@_);
    $self;
}

# TODO: export more proporties like the title string
sub Initialize {
    my $self = shift;
    my $month = shift;
    my $year = shift;
    my $msg = shift;
    my $cal = shift || $cal_default;
    my $calmonth, @dates, @days;

    # dates holds the references for entry to add new items to the table
    $self->{'dates'} = [ ];

    # `cal` output format:
    #     October 1995
    #  S  M Tu  W Th  F  S
    #  1  2  3  4  5  6  7
    #  8  9 10 11 12 13 14
    # 15 16 17 18 19 20 21
    # 22 23 24 25 26 27 28
    # 29 30 31

    local(*CAL, $_);
    open(CAL, "-|") || exec $cal, $month, $year || die "$cal: $!\n";

    # Extract the month and year from the first line
    $_ = <CAL>;
    s/^\s*(.*)\s*/$1/;
    ($month, $year) = split(/ /, $_);
    $month =~ y/a-z/A-Z/;
    $calmonth = $months{"\u\L$month"};

    # Skip the weekday line ``S  M Tu  W Th  F  S''
    $_ = <CAL>;

    # For each weekline of the calendar, grab the dates into @dates
    # Empty dates are stored as a zero and there are *always* 7 days
    while ($_ = <CAL>) {
	next if /^\s*$/;
	$_ .= " " x 20;				# make sure it's long enough
	push(@dates, unpack("A2xA2xA2xA2xA2xA2xA2", $_));
    }
    map { $_ += 0 } @dates;			# convert to numeric form

    close(CAL);

    # Generate the HTML table to build the calendar
    $self->new_table($month, $year);
    while (scalar(@dates)) {
	@days = splice(@dates, 0, 7);		# 7 days at a time...
	$self->dates(@days);
	$self->entries($msg, $calmonth, @days);
    }

    $self;
}

# TODO:
sub Copy {
    croak "Not Implemented Yet";
}

sub Clone {
    croak "Not Implemented Yet";
}

sub table {
    my $self = shift;
    $self->{'table'};
}

sub realize {
    my $self = shift;
    my $td;

    # Build table entries for empty dates
    foreach $td ( @{$self->{'dates'}} ) {
	$td->Literal($empty) if $td->empty;
    }

    # And then realize the table itself...
    ($self->{'table'})->realize;
}

sub entry {
    my $self = shift;
    my $day = shift;
    my $string = shift;
    my $td = $self->{'dates'}->[$day - 1];
    my $text = $td->first->child;
    $text->append($string, "\n");

    $self;
}

# =============================================================================
# Private Methods

sub new_table {
    my $self = shift;
    my ($month, $year) = @_;
    my $tr, $td, $header, $day;

    $self->{'table'} = Create HTML::Markup::HTML2('TABLE');
    $self->{'table'}->width("100%")->border(2)->attribute('cols', 7);

    # <!-- HEADER _____________________________ HEADER -->

    $tr = $self->{'table'}->TR;
    $td = $tr->TH->align("center")->valign("center")->colspan(7);
    $header = $td->H1;
    $header->BR;
    $header->I->Text("$month $year");
    $header->BR;
    $header->BR;

    # <!-- WEEKDAYS _________________________ WEEKDAYS -->

    $tr = $self->{'table'}->TR;
    foreach $day ("Sunday", "Monday", "Tuesday", "Wednesday",
	    "Thursday", "Friday", "Saturday") {
	$tr->TH->align("center")->Text($day);
    }

    $self;
}

sub dates {
    my $self = shift;
    my @days = @_;
    my $tr, $day;

    # <!-- DATES ________________________________ DATES -->
    $tr = $self->{'table'}->TR;
    foreach $day (@days) {
	# non-days are left empty
	if ($day) {
	    $tr->TD->width(75)->Text($day);
	}
	else {
	    # an empty cell
	    $tr->TD->width(75)->BR;
	}
    }

    $self;
}

sub entries {
    my $self = shift;
    local($msg, $month, @days) = @_;
    my $tr;

    # <!-- ENTRIES ____________________________ ENTRIES -->

    $tr = $self->{'table'}->TR;
    $self->init_entry($tr, $msg, 0, $month, shift @days);
    $self->init_entry($tr, $msg, 1, $month, shift @days);
    $self->init_entry($tr, $msg, 2, $month, shift @days);
    $self->init_entry($tr, $msg, 3, $month, shift @days);
    $self->init_entry($tr, $msg, 4, $month, shift @days);
    $self->init_entry($tr, $msg, 5, $month, shift @days);
    $self->init_entry($tr, $msg, 6, $month, shift @days);

    $self;
}

sub init_entry {
    my $self = shift;
    local($tr, $msg, $dayweek, $month, $day) = @_;
    my $string;

    my $td = $tr->TD->align("center");

    if ($day != 0) {
	$string .= $msg->{"$month/$day"}
	    if defined($msg->{"$month/$day"});
	$string .= $msg->{"daily/$dayweek"}
	    if defined($msg->{"daily/$dayweek"});
	$string .= $msg->{"monthly/$day"}
	    if defined($msg->{"monthly/$day"});

	# Add only if it's a real day, the result is that the
	# days are added in order.
	push(@{$self->{'dates'}}, $td);
	if (defined $string) {
	    chomp($string);
	    $string = join('', HTML::Markup::Text::_escape($string));
	    $string =~ s/[\r\n]+/ <B>\&#164;<\/B> /gm;
	    $td->Literal($string);
	}
    } else {
	$td->Literal($empty);
    }

    $self;
}

1;
