# Copyright (c) 1995, 1996 Berkeley Software Design, Inc.
# All rights reserved.
# The Berkeley Software Design Inc. software License Agreement specifies
# the terms and conditions for redistribution.
#
#       BSDI Entities.pm,v 1.2 1996/08/28 20:24:41 sanders Exp
#
# Object-Oriented HTML
#
# NOTE: This information was extracted from the HTML Level 2 DTD

=head1 NAME

HTML::Markup::Entities - HTML Entity methods exporter

=head1 SYNOPSIS

use HTML::Markup::Entities;

=head1 DESCRIPTION

Defines entity methods for all the standard HTML and
ISO 8859-1/Latin 1 Entities.

=head1 METHODS

The follow methods are exported.

=over 4

=item Common HTML escapes:

quot amp lt gt

=item Marks and signs:

nbsp iexcl cent pund curren yen brvbar sect uml copy ordf laquo
not shy reg macr deg plusmn sup2 sup3 acute micro para middot
cedil sup1 ordm raquo frac14 frac12 frac34 iquest times divide

=item Accented characters:

Aacute aacute Acirc acirc AElig aelig Agrave agrave Aring aring
Atilde atilde Auml auml Ccedil ccedil Eacute eacute Ecirc ecirc
Egrave egrave ETH eth Euml euml Iacute iacute Icirc icirc Igrave
igrave Iuml iuml Ntilde ntilde Oacute oacute Ocirc ocirc Ograve
ograve Oslash oslash Otilde otilde Ouml ouml szlig THORN thorn
Uacute uacute Ucirc ucirc Ugrave ugrave Uuml uuml Yacute yacute
yuml

=back

=cut

package HTML::Markup::Entities;
use Carp;

require Exporter;
@ISA = qw(Exporter);

BEGIN {
    # ISO 8859-1/Latin 1 Entities
    my $entities = {
	# common HTML escapes
	'quot'   => '&quot;',    # &#034; double quote
	'amp'    => '&amp;',     # &#038; ampersand
	'lt'     => '&lt;',      # &#060; left chevron, less-than
	'gt'     => '&gt;',      # &#062; right chevron, greater-than

	# marks and signs
	'nbsp'   => '&#160;',    # &#160; non-breaking space
	'iexcl'  => '&#161;',    # &#161; inverted exclamation mark
	'cent'   => '&#162;',    # &#162; cent sign
	'pund'   => '&#163;',    # &#163; pound sign
	'curren' => '&#164;',    # &#164; general currency sign
	'yen'    => '&#165;',    # &#165; yen sign
	'brvbar' => '&#166;',    # &#166; broken vertical line
	'sect'   => '&#167;',    # &#167; section sign
	'uml'    => '&#168;',    # &#168; umlaut/dieresis
	'copy'   => '&#169;',    # &#169; copyright sign
	'ordf'   => '&#170;',    # &#170; ordinal indicator, fem
	'laquo'  => '&#171;',    # &#171; angle quotation, left
	'not'    => '&#172;',    # &#172; not sign
	'shy'    => '&#173;',    # &#173; soft hyphen
	'reg'    => '&#174;',    # &#174; registered trademark sign
	'macr'   => '&#175;',    # &#175; macron
	'deg'    => '&#176;',    # &#176; degree sign
	'plusmn' => '&#177;',    # &#177; plus/minus sign
	'sup2'   => '&#178;',    # &#178; superscript 2
	'sup3'   => '&#179;',    # &#179; superscript 3
	'acute'  => '&#189;',    # &#189; acute accent
	'micro'  => '&#181;',    # &#181; micro sign
	'para'   => '&#182;',    # &#182; pilcrow/paragraph mark
	'middot' => '&#183;',    # &#183; middle dot
	'cedil'  => '&#184;',    # &#184; cedilla
	'sup1'   => '&#185;',    # &#185; superscript 1
	'ordm'   => '&#186;',    # &#186; ordinal indicator, male
	'raquo'  => '&#187;',    # &#187; angle quotation, right
	'frac14' => '&#188;',    # &#188; fraction 1/4
	'frac12' => '&#189;',    # &#189; fraction 1/2
	'frac34' => '&#190;',    # &#190; fraction 3/4
	'iquest' => '&#191;',    # &#191; inverted question mark
	'times'  => '&#215;',    # &#215; multiplication sign
	'divide' => '&#247;',    # &#247; division sign

	# accented characters
	'Aacute' => '&#193;',    # &#193; capital A, acute accent
	'aacute' => '&#225;',    # &#225; small a, acute accent
	'Acirc'  => '&#194;',    # &#194; capital A, circumflex accent
	'acirc'  => '&#226;',    # &#226; small a, circumflex accent
	'AElig'  => '&#198;',    # &#198; capital AE diphthong (ligature)
	'aelig'  => '&#230;',    # &#230; small ae diphthong (ligature)
	'Agrave' => '&#192;',    # &#192; capital A, grave accent
	'agrave' => '&#224;',    # &#224; small a, grave accent
	'Aring'  => '&#197;',    # &#197; capital A, ring
	'aring'  => '&#229;',    # &#229; small a, ring
	'Atilde' => '&#195;',    # &#195; capital A, tilde
	'atilde' => '&#227;',    # &#227; small a, tilde
	'Auml'   => '&#196;',    # &#196; capital A, dieresis or umlaut mark
	'auml'   => '&#228;',    # &#228; small a, dieresis or umlaut mark
	'Ccedil' => '&#199;',    # &#199; capital C, cedilla
	'ccedil' => '&#231;',    # &#231; small c, cedilla
	'Eacute' => '&#201;',    # &#201; capital E, acute accent
	'eacute' => '&#233;',    # &#233; small e, acute accent
	'Ecirc'  => '&#202;',    # &#202; capital E, circumflex accent
	'ecirc'  => '&#234;',    # &#234; small e, circumflex accent
	'Egrave' => '&#200;',    # &#200; capital E, grave accent
	'egrave' => '&#232;',    # &#232; small e, grave accent
	'ETH'    => '&#208;',    # &#208; capital Eth, Icelandic
	'eth'    => '&#240;',    # &#240; small eth, Icelandic
	'Euml'   => '&#203;',    # &#203; capital E, dieresis or umlaut mark
	'euml'   => '&#235;',    # &#235; small e, dieresis or umlaut mark
	'Iacute' => '&#205;',    # &#205; capital I, acute accent
	'iacute' => '&#237;',    # &#237; small i, acute accent
	'Icirc'  => '&#206;',    # &#206; capital I, circumflex accent
	'icirc'  => '&#238;',    # &#238; small i, circumflex accent
	'Igrave' => '&#204;',    # &#204; capital I, grave accent
	'igrave' => '&#236;',    # &#236; small i, grave accent
	'Iuml'   => '&#207;',    # &#207; capital I, dieresis or umlaut mark
	'iuml'   => '&#239;',    # &#239; small i, dieresis or umlaut mark
	'Ntilde' => '&#209;',    # &#209; capital N, tilde
	'ntilde' => '&#241;',    # &#241; small n, tilde
	'Oacute' => '&#211;',    # &#211; capital O, acute accent
	'oacute' => '&#243;',    # &#243; small o, acute accent
	'Ocirc'  => '&#212;',    # &#212; capital O, circumflex accent
	'ocirc'  => '&#244;',    # &#244; small o, circumflex accent
	'Ograve' => '&#210;',    # &#210; capital O, grave accent
	'ograve' => '&#242;',    # &#242; small o, grave accent
	'Oslash' => '&#216;',    # &#216; capital O, slash
	'oslash' => '&#248;',    # &#248; small o, slash
	'Otilde' => '&#213;',    # &#213; capital O, tilde
	'otilde' => '&#245;',    # &#245; small o, tilde
	'Ouml'   => '&#214;',    # &#214; capital O, dieresis or umlaut mark
	'ouml'   => '&#246;',    # &#246; small o, dieresis or umlaut mark
	'szlig'  => '&#223;',    # &#223; small sharp s, German (sz ligature)
	'THORN'  => '&#222;',    # &#222; capital THORN, Icelandic
	'thorn'  => '&#254;',    # &#254; small thorn, Icelandic
	'Uacute' => '&#218;',    # &#218; capital U, acute accent
	'uacute' => '&#250;',    # &#250; small u, acute accent
	'Ucirc'  => '&#219;',    # &#219; capital U, circumflex accent
	'ucirc'  => '&#251;',    # &#251; small u, circumflex accent
	'Ugrave' => '&#217;',    # &#217; capital U, grave accent
	'ugrave' => '&#249;',    # &#249; small u, grave accent
	'Uuml'   => '&#220;',    # &#220; capital U, dieresis or umlaut mark
	'uuml'   => '&#252;',    # &#252; small u, dieresis or umlaut mark
	'Yacute' => '&#221;',    # &#221; capital Y, acute accent
	'yacute' => '&#253;',    # &#253; small y, acute accent
	'yuml'   => '&#255;',    # &#255; small y, dieresis or umlaut mark
    };

    my ($eval, $entity, $escape) = '';
    $eval .= join(' ', '@EXPORT = qw(', keys %$entities, ");\n");
    while (($entity, $escape) = each %$entities) {
	$eval .= qq{
	    sub $entity { my \$self = shift; \$self->entity('${escape}'); } };
    }
    eval $eval;
    croak $@ if $@;
}
