# Copyright (c) 1996, 1997 Berkeley Software Design, Inc.
# All rights reserved.
# The Berkeley Software Design Inc. software License Agreement specifies
# the terms and conditions for redistribution.
#
#	BSDI Lock.pm,v 1.6 1997/04/17 21:23:31 sanders Exp
#
# IO::Lock -- file locking methods
#
# IO::Lock does not have a Constructor, it's a virtual base class

package IO::Lock;
require 5.003;
use vars qw($VERSION);
use strict;
use Carp;
# XXX: fix in perl5.004: use Fcntl qw(:flock);
sub LOCK_SH () { 1; }
sub LOCK_EX () { 2; }
sub LOCK_NB () { 4; }
sub LOCK_UN () { 8; }
$VERSION = '1.01';

### METHOD: trylock(LOCK_EX|LOCK_SH)
###
### Non-blocking lock attempt.  Returns false if the filehandle is
### locked, else it locks it.  This is just a shortcut for using LOCK_NB.

sub trylock {
    my $self = shift;
    flock($self, LOCK_NB | $_[0]);
}

### METHOD: polled($lock_flags,[$timeout[, $rate]])
###
### Locks the filehandle with $lock_flags (LOCK_EX or LOCK_SH).
### $timeout, if present, indicates number of seconds to wait.
### $rate is the number of seconds between tries.

sub polled {
    my $self = shift;
    my $flags = shift;
    my $timeout = shift || 0;
    my $rate = shift || 1;
    my $locked;

    if ($timeout > 0) {
	while ($timeout > 0) {
	    return $locked if $locked = $self->trylock(LOCK_EX);
	    sleep($rate);
	    $timeout -= $rate;
	}
	$! = 60 if $timeout <= 0;
	return undef;
    }
    else {
	return flock($self, $flags);
    }
}

### METHOD: signaled($lock_flags,$timeout)
###
### Locks the filehandle with $lock_flags (LOCK_EX or LOCK_SH).
### $timeout, if present, indicates number of seconds to wait.
### If $timeout isn't present or is 0 then alarm signals are disabled.

sub signaled {
    my $self = shift;
    my $flags = shift;
    my $timeout = shift || 0;

    # disable alarms, saving old value for later
    my $oldalarm = alarm(0);
    my $oldsig = $SIG{ALRM};
    my $time = time;

    $SIG{ALRM} = sub { die "TIMEDOUT\n"; };

    # eval catches death on SIGALRM
    eval {
	alarm($timeout);
	flock($self, $flags);
	alarm(0);
    };
    # either alarms are disabled or the alarm already tripped

    # check whether or not we actually got the lock
    my $flock = $self->trylock($flags);
    my $errno = ($! + 0);

    # restore any previous alarm state
    $SIG{ALRM} = $oldsig;
    if ($oldalarm > 0) {
	$oldalarm = $oldalarm - (time - $time);
	$oldalarm = 1 if $oldalarm < 1;
	alarm($oldalarm);
    }

    $! = $errno;
    return $flock;
}

### METHOD: exclusive([$timeout | -polled[,$timeout[,$rate]]])
###
### Locks the filehandle.
### $timeout, if present, indicates number of seconds to wait.
### $rate is the number of seconds between tries.

sub exclusive {
    my $self = shift;

    if ($_[0] eq '-polled') {
	shift;
	return $self->polled(LOCK_EX, @_);
    }
    else {
	return $self->signaled(LOCK_EX, @_);
    }
}

### METHOD: shared([$timeout | -polled[,$timeout[,$rate]]])
###
### Locks the filehandle.
### $timeout, if present, indicates number of seconds to wait.
### $rate is the number of seconds between tries.
sub shared {
    my $self = shift;

    if ($_[0] eq '-polled') {
	shift;
	return $self->polled(LOCK_SH, @_);
    }
    else {
	return $self->signaled(LOCK_SH, @_);
    }
}

### unlock()
###
### Removes any lock on the filehandle.
sub unlock {
    my $self = shift;
    return flock($self, LOCK_NB|LOCK_UN);
}

1;
