package Sound::Voxware::Mixer;
use English;
use Carp;

use IO::File;

BEGIN {
    # Hardware interface specs derived from voxware soundcard.h

    # sets
    $SOUND_MIXER_RECSRC = 0xff;
    $SOUND_MIXER_DEVMASK = 0xfe;
    $SOUND_MIXER_RECMASK = 0xfd;
    $SOUND_MIXER_CAPS = 0xfc;
	$SOUND_CAP_EXCL_INPUT = 0x1;
    $SOUND_MIXER_STEREODEVS = 0xfb;

    $SOUND_MIXER_READ_RECSRC = 0x40044dff;
    $SOUND_MIXER_READ_DEVMASK = 0x40044dfe;
    $SOUND_MIXER_READ_RECMASK = 0x40044dfd;
    $SOUND_MIXER_READ_STEREODEVS = 0x40044dfb;
    $SOUND_MIXER_READ_CAPS = 0x40044dfc;

    $SOUND_MIXER_WRITE_RECSRC = 0xc0044dff;

    # for converting #'s into an $attributes index
    $SOUND_DEVICE_NAMES = [
	'volume',	'bass',		'treble',	'synth',
	'pcm',		'speaker',	'line',		'microphone',
	'cd',		'imix',		'altpcm',	'rec',
	'igain',	'ogain',	'line1',	'line2',
	'line3',	'digi1',	'digi2',	'digi3',
	'phin',		'phout',	'video',	'radio',
	'monitor',	'depth',	'center',	'midi',
	undef,		undef,		undef,		undef,
    ];
    $OSOUND_DEVICE_NAMES = [
	'vol',		'bass',		'treble',	'synth',
	'pcm',		'speaker',	'line',		'mic',
	'cd',		'mix',		'pcm2',		'rec',
	'igain',	'ogain',	'line1',	'line2',
	'line3',	undef,		undef,		undef,
	undef,		undef,		undef,		undef,
	undef,		undef,		undef,		undef,
	'mute',		'enhance',	'loud',		undef,
    ];

    # attribute => [ canonical, label, description, code, read, write, ],
    $CANON = 0;
    $LABEL = 1;
    $DESC  = 2;
    $CODE  = 3;
    $READ  = 4;
    $WRITE = 5;
    $attributes = {
	"volume"	=> [ "volume", "Master", "Master volume",
				0x0, 0x40044d00, 0xc0044d00, ],
	"vol"		=> [ "volume", "Master", "Master volume",
				0x0, 0x40044d00, 0xc0044d00, ],
	"bass"		=> [ "bass", "Bass  ", "Bass",
				0x1, 0x40044d01, 0xc0044d01, ],
	"treble"	=> [ "treble", "Treble", "Treble",
				0x2, 0x40044d02, 0xc0044d02, ],
	"synth"		=> [ "synth", "Synth ", "On-board Sythesizer",
				0x3, 0x40044d03, 0xc0044d03, ],
	"pcm"		=> [ "pcm", "DSP   ", "Primary DAC",
				0x4, 0x40044d04, 0xc0044d04, ],
	"speaker"	=> [ "speaker", "PCSpkr", "PC speaker",
				0x5, 0x40044d05, 0xc0044d05, ],
	"line"		=> [ "line", "Line  ", "Main line in",
				0x6, 0x40044d06, 0xc0044d06, ],
	"microphone"	=> [ "microphone", "Mic   ", "Microphone",
				0x7, 0x40044d07, 0xc0044d07, ],
	"mic"		=> [ "mic", "Mic   ", "Microphone",
				0x7, 0x40044d07, 0xc0044d07, ],
	"cd"		=> [ "cd", "CD    ", "CD Player",
				0x8, 0x40044d08, 0xc0044d08, ],
	"imix"		=> [ "imix", "Mon   ", "Recording monitor",
				0x9, 0x40044d09, 0xc0044d09, ],
	"mix"		=> [ "mix", "Mon   ", "Recording monitor",
				0x9, 0x40044d09, 0xc0044d09, ],
	"altpcm"	=> [ "altpcm", "PCM2  ", "Secondary DAC",
				0xa, 0x40044d0a, 0xc0044d0a, ],
	"pcm2"		=> [ "altpcm", "PCM2  ", "Secondary DAC",
				0xa, 0x40044d0a, 0xc0044d0a, ],
	"rec"		=> [ "rec", "Rec   ", "Record level",
				0xb, 0x40044d0b, 0xc0044d0b, ],
	"igain"		=> [ "igain", "IGain ", "Input gain",
				0xc, 0x40044d0c, 0xc0044d0c, ],
	"ogain"		=> [ "ogain", "OGain ", "Output gain",
				0xd, 0x40044d0d, 0xc0044d0d, ],
	"line1"		=> [ "line1", "Line 1 ", "Line input 1",
				0xe, 0x40044d0e, 0xc0044d0e, ],
	"line2"		=> [ "line2", "Line 2 ", "Line input 2",
				0xf, 0x40044d0f, 0xc0044d0f, ],
	"line3"		=> [ "line3", "Line 3 ", "Line input 3",
				0x10, 0x40044d10, 0xc0044d10, ],
	"digi1"		=> [ "digi1", "Digi 1 ", "Digital input 1",
				0x11, 0x40044d11, 0xc0044d11, ],
	"digi2"		=> [ "digi2", "Digi 2 ", "Digital input 2",
				0x12, 0x40044d12, 0xc0044d12, ],
	"digi3"		=> [ "digi3", "Digi 3 ", "Digital input 3",
				0x13, 0x40044d13, 0xc0044d13, ],
	"phin"		=> [ "phin", "Phone In ", "Phone input",
				0x14, 0x40044d14, 0xc0044d14, ],
	"phout"		=> [ "phout", "Phone Out ", "Phone output",
				0x15, 0x40044d15, 0xc0044d15, ],
	"video"		=> [ "video", "Video ", "Videio input",
				0x16, 0x40044d16, 0xc0044d16, ],
	"radio"		=> [ "radio", "Radio ", "Radio input",
				0x17, 0x40044d17, 0xc0044d17, ],
	"monitor"	=> [ "monitor", "Monitor ", "Monitor input",
				0x18, 0x40044d18, 0xc0044d18, ],
	"depth"		=> [ "depth", "Depth ", "3D Depth",
				0x19, 0x40044d19, 0xc0044d19, ],
	"center"	=> [ "center", "Center ", "3D Center",
				0x1a, 0x40044d1a, 0xc0044d1a, ],
	"midi"		=> [ "midi", "MIDI ", "MIDI",
				0x1b, 0x40044d1b, 0xc0044d1b, ],
	# rdev is a special case
	"rdev"		=> [ "rdev", "Source", "Recording source",
				$SOUND_MIXER_RECSRC,
				$SOUND_MIXER_READ_RECSRC,
				$SOUND_MIXER_WRITE_RECSRC, ],
    };
}

sub Create { ## public
    my $self = bless { }, shift;
    $self->Initialize(@ARG);
    $self;
}

sub Initialize { ## public
    my $self = shift;
    my $mixer_dev = shift;
    $self->{"Sound::Voxware::Mixer::device"} = ($mixer_dev || "/dev/mixer");
    $self->{"Sound::Voxware::Mixer::attrs"} = { };
    $self->{"Sound::Voxware::Mixer::devmask"} = 0;
    $self->{"Sound::Voxware::Mixer::stereomask"} = 0;
    $self->{"Sound::Voxware::Mixer::recmask"} = 0;
    $self->{"Sound::Voxware::Mixer::MIXERFH"} = new IO::File;
    $self->open_mixer;		# assumes multiple access is ok
    $self->load;
    $self;
}

################################################################
###   mixer filehandle   #######################################
################################################################

sub mixer_fh { ## private
    my $self = shift;
    return $self->{"Sound::Voxware::Mixer::MIXERFH"};
}

sub DESTROY { ## implicit
    my $self = shift;
    $self->mixer_fh->close;
    $self->{"Sound::Voxware::Mixer::MIXERFH"} = undef;
}

sub open_mixer { ## private
    my $self = shift;
    my $MIXER = $self->{"Sound::Voxware::Mixer::device"};
    $self->mixer_fh->open("+< $MIXER\0")
	or croak "$MIXER: $!";
}

sub close_mixer { ## private
    my $self = shift;
    $self->mixer_fh->close;
}

################################################################
###   device settings   ########################################
################################################################

sub attribute { ## public
    my $self = shift;
    my $modified = '1';
    $self->{"Sound::Voxware::Mixer::attrs"}->{$_[0]} = $_[1] . $modified
	if defined $_[1];
    my $val = $self->{"Sound::Voxware::Mixer::attrs"}->{$_[0]};
    chop $val;		# remove state byte
    $val;
}

sub attrs { ## public
    my $self = shift;
    keys %{$self->{"Sound::Voxware::Mixer::attrs"}};
}

sub changed { ## private
    my $self = shift;
    my $dev = shift;
    # check if $dev value has been changed
    substr($self->{"Sound::Voxware::Mixer::attrs"}->{$dev},-1,1) eq '1';
}

sub updated { ## private
    my $self = shift;
    my $dev = shift;
    # clear the changed bit for $dev
    substr($self->{"Sound::Voxware::Mixer::attrs"}->{$dev},-1,1) = '0';
}

sub is_stereo { ## public
    my $self = shift;
    my $dev = shift;
    $dev = $self->code($dev) if defined $attributes->{$dev};
    $self->{"Sound::Voxware::Mixer::stereomask"} & (1 << $dev);
}

sub rdevs { ## public
    my $self = shift;
    mask2names($self->{"Sound::Voxware::Mixer::recmask"});
}

sub devices { ## public
    my $self = shift;
    mask2names($self->{"Sound::Voxware::Mixer::devmask"});
}

################################################################
###   attribute structure   ####################################
################################################################

sub label { ## public
    my $self = shift;
    $attributes->{$_[0]}->[$LABEL];
}

sub desc { ## public
    my $self = shift;
    $attributes->{$_[0]}->[$DESC];
}

sub code { ## private
    my $self = shift;
    $attributes->{$_[0]}->[$CODE];
}

sub read { ## private
    my $self = shift;
    $attributes->{$_[0]}->[$READ];
}

sub write { ## private
    my $self = shift;
    $attributes->{$_[0]}->[$WRITE];
}

sub mask2names { ## private
    my $mask = shift;
    my($idx, @devs);
    $idx = 0;
    while ($mask != 0) {
	push(@devs, $SOUND_DEVICE_NAMES->[$idx]) if (($mask & 0x1) != 0);
	$mask >>= 1;
	$idx++;
    }
    @devs;
}

sub names2mask { ## private
    my $mask;
    map { $mask |= 1 << $attributes->{$_}->[$CODE] } @ARG;
    $mask;
}

################################################################
###   load/store   #############################################
################################################################

sub get_mixer_state { ## private
    my $self = shift;
    my($idx, $mask, $dev, $desc, $left, $right, $val);

    $val = pack('i', 0);
    ioctl($self->mixer_fh, $SOUND_MIXER_READ_DEVMASK, $val)
	or croak "ioctl SOUND_MIXER_READ_DEVMASK: $!";
    $self->{"Sound::Voxware::Mixer::devmask"} = unpack('i', $val);

    $val = pack('i', 0);
    ioctl($self->mixer_fh, $SOUND_MIXER_READ_STEREODEVS, $val)
	or croak "ioctl SOUND_MIXER_READ_STEREODEVS: $!";
    $self->{"Sound::Voxware::Mixer::stereomask"} = unpack('i', $val);

    $val = pack('i', 0);
    ioctl($self->mixer_fh, $SOUND_MIXER_READ_RECMASK, $val)
	or croak "ioctl SOUND_MIXER_READ_RECMASK: $!";
    $self->{"Sound::Voxware::Mixer::recmask"} = unpack('i', $val);

    # Handle 'rdev' special case
    $val = pack('i', 0);
    ioctl($self->mixer_fh, $SOUND_MIXER_READ_RECSRC, $val)
	or croak "ioctl SOUND_MIXER_READ_RECSRC: $!";
    $val = unpack('i', $val);
    $self->attribute('rdev', join(":",mask2names($val)));
    $self->updated('rdev');

    # XXX: handle $SOUND_MIXER_READ_CAPS, this will tell us if we can
    # have multiple recording devices or not.

    # build exported attributes
    foreach $dev ($self->devices) {
	$val = pack('i', 0);
	ioctl($self->mixer_fh, $self->read($dev), $val)
	    or croak "ioctl read($dev): $!";
	$val = unpack('i', $val);

	$left = $val & 0xff;
	$right = $val >> 8 & 0xff;
	$self->attribute($dev, $self->is_stereo($dev)?"$left:$right":$left);
	$self->updated($dev);
    }
}

sub load { ## public
    my $self = shift;
    $self->get_mixer_state;
}

sub store { ## public
    my $self = shift;
    my($dev,$val,$left,$right);

    foreach $dev ($self->attrs) {
	next unless $self->changed($dev);
	$self->updated($dev);

	if ($dev eq 'rdev') {
	    $val = pack('i', names2mask(split(/:/,$self->attribute($dev))));
	}
	else {
	    ($left, $right) = split(/:/, $self->attribute($dev));
	    $right = $left unless $self->is_stereo($dev);
	    $val = pack('i', $right << 8 | $left);
	}
	ioctl($self->mixer_fh, $self->write($dev), $val)
	    or croak "ioctl: $!";
    }

    # reload to make sure things got set right
    $self->get_mixer_state;
}

1;
