# Copyright (c) 1997 Tony Sanders <sanders@earth.com>
package Cdrom;

use strict;
use Carp;
use vars qw($VERSION @ISA @EXPORT_OK $AUTOLOAD);

require Exporter;
require DynaLoader;
require AutoLoader;

@ISA = qw(Exporter DynaLoader);
# Items to export into callers namespace by default. Note: do not export
# names by default without a very good reason. Use EXPORT_OK instead.
# Do not simply export all your public functions/methods/constants.
@EXPORT_OK = qw(
	UNKNOWN
	STOPPED
	PLAYING
	PAUSED
	frame_to_msf
	msf_to_frame
);
$VERSION = '0.01';

sub AUTOLOAD {
    # This AUTOLOAD is used to 'autoload' constants from the constant()
    # XS function.  If a constant is not found then control is passed
    # to the AUTOLOAD in AutoLoader.

    my $constname;
    ($constname = $AUTOLOAD) =~ s/.*:://;
    my $val = constant($constname, @_ ? $_[0] : 0);
    if ($! != 0) {
	if ($! =~ /Invalid/) {
	    $AutoLoader::AUTOLOAD = $AUTOLOAD;
	    goto &AutoLoader::AUTOLOAD;
	}
	else {
		croak "Your vendor has not defined Cdrom macro $constname";
	}
    }
    eval "sub $AUTOLOAD { $val }";
    goto &$AUTOLOAD;
}

bootstrap Cdrom $VERSION;

$Cdrom::states[&UNKNOWN] = "unknown";
$Cdrom::states[&STOPPED] = "stopped";
$Cdrom::states[&PLAYING] = "playing";
$Cdrom::states[&PAUSED]  = "paused";

# Preloaded methods go here.

sub pause {
    my $self = shift;
    $self->cdpause(0);
}
sub resume {
    my $self = shift;
    $self->cdpause(1);
}

#### cdstatus interfaces

sub state {
    my $self = shift;
    return ($self->cdstatus)[0];
}

sub control {
    my $self = shift;
    return ($self->cdstatus)[1];
}

sub cur_track {
    my $self = shift;
    return ($self->cdstatus)[2];
}

sub cur_index {
    my $self = shift;
    return ($self->cdstatus)[3];
}

sub rel_frame {
    my $self = shift;
    return ($self->cdstatus)[4];
}

sub abs_frame {
    my $self = shift;
    return ($self->cdstatus)[5];
}

sub track_info {
    my $self = shift;
    $self->cdtrk(@_);
}

# Autoload methods go after =cut, and are processed by the autosplit program.

1;
__END__

=head1 NAME

Cdrom - Perl extension for BSD/OS libcdrom

=head1 SYNOPSIS

  use Cdrom;
  use Cdrom qw(UNKNOWN STOPPED PLAYING PAUSED frame_to_msf msf_to_frame);

=head1 DESCRIPTION

The Cdrom package exports the interfaces to libcdrom on BSD/OS
in an object-oriented fashion.

=head2 Public Methods

=over

=item Cdrom->new([$cdrom_device])

Opens the CD device and returns a Cdrom object reference.
$cdrom_device defaults to _PATH_CDROM from cdrom.h
(typically /dev/rsr0c).

=item $cd->tracks()

Number of tracks on the disk.

=item $cd->frames()

Total number of frames on disk.

=item $cd->track_info($track)

Returns track information for track $track:

    ($start_frame, $nframes, $control)

=item $cd->cdplay($start, $stop)

Play from track $start to track $stop.

=item $cd->pause()

Pause playing.

=item $cd->resume()

Resume from pause.

=item $cd->cdstop()

Stop the CD.

=item $cd->cdeject()

Eject the CD, if possible.

=item $cd->cdload()

Load the CD, if possible.

=item $cd->cdvolume(volume)

Set the CD volume, if possible.

=item $cd->state()

State returns one of:

	Cdrom::PAUSED()
	Cdrom::PLAYING()
	Cdrom::STOPPED()
	Cdrom::UNKNOWN()

You can also choose to import those symbols:

	use Cdrom qw(PAUSED, PLAYING, STOPPED, UNKNOWN);

=item $cd->control()

See cdrom(3)

=item $cd->cur_track()

Current track.

=item $cd->cur_index()

Current index.

=item $cd->rel_frame()

Current frame, track relative.

=item $cd->abs_frame()

Current absolute frame.

=item Cdrom::frame_to_msf(frm)

Convert an absolute frame to a (minute, second, frame) spec.

=item Cdrom::msf_to_frame(minute, second, frame)

Convert a (minute, second, frame) spec to an absolute frame.

=head1 AUTHOR

Tony Sanders, <sanders@earth.com>

=head1 SEE ALSO

cdctl(1), cdrom(3)

=cut
