#
# Routines to manage the DNS files
#

package	AdminWEB::DNS;
require Exporter;
@ISA = qw(Exporter);
@EXPORT = qw(
        convert_conf
        convert_domain
	delete_domain
	$conf_file
	$resolv_file
	$RESOLV_DOMAIN
	@RESOLV_NS
	@RESOLV_SRCH
	$def_dbdir
	$dbdir
	@sortlist
	%PRIMARY
	%SECONDARY
	@forwarders
	$slave
	%DBFILES
	%SOA
	%A_RECS
	%ALIASES
	%MX_RECORDS
	%PTRS
	%NS
	%HINFO
	%TXT
	%WKS
        read_resolver
        write_resolver
        init_resolver
        read_config
	write_config
        init_config
	init_domain
        read_dbfile
        write_dbfile
	set_dbfile
	unset_dbfile
        set_soa
        unset_soa
        set_ns
        unset_ns
        set_arec
        unset_arec
        set_cname
        unset_cname
        set_mx
        unset_mx
        set_ptr
        unset_ptr
        set_hinfo 
        unset_hinfo
        set_txt
        unset_txt
        set_wks
        unset_wks
        set_reverse
        unset_reverse
	ip_2_inaddrarpa
        net_2_inaddrarpa
	get_myinfo
        mkdirp
	clear_all_recs
        reload_dns
	kill_dns
	byorigin
	mxpreference
	check_domain
	check_ip
	cleanup_name
	);

$conf_file="/etc/named.boot";
$resolv_file="/etc/resolv.conf";
$netstart_file="/etc/netstart";
$def_dbdir="/etc/namedb";
$nsipaddr="0.0.0.0";
$named_pid_file="/var/run/named.pid";
$reload = 1;

# XXX: need to convert the remaining interfaces to the new style
use FileFormat::DNS::Paths;
use FileFormat::DNS::NamedBoot;
use FileFormat::DNS::DomainDB;

sub convert_domain {
    my ($domain, $file) = @_;
    my $dom = new FileFormat::DNS::DomainDB($domain, $file);
    $dom->{'CHANGED'} = 1;
    $dom->close;
}

sub convert_conf {
    my $file = $_[0];
    # XXX tell it not to look at named.boot/named.conf equivalence...
    my $named = new FileFormat::DNS::NamedBoot($file, 0, 1);
    $named->{'CHANGED'} = 1;
    $named->close;
}

# XXX:
sub delete_domain {
    my $domain = $_[0];
    &read_config($conf_file);
    if ( $domain =~ /^[0-9.]*$/ ) {
	$domain = &net_2_inaddrarpa($domain);
    }
    $domain =~ s/\.*$/./;
    if ( $domain eq "127.in-addr.arpa." ) {
	print "Will not delete loopback domain $domain\n";
	exit 1;
    }
    if ( defined $SECONDARY{$domain} ) {
	($dbfile,@primaddr) = split(/,/,$SECONDARY{$domain});
	delete $SECONDARY{$domain};
	&write_config($conf_file);
	$dbfile = "$dbdir/$dbfile" if $dbdir ne "";
	unlink $dbfile;
    }
    elsif ( defined $PRIMARY{$domain} ) {
	$dbfile = $PRIMARY{$domain};
	$dbfile = "$dbdir/$dbfile" if $dbdir ne "";
	&unset_dbfile($dbfile,$domain);
	foreach $dbfile (keys %DBFILES) {
	    &write_dbfile($dbfile);
	}
	delete $PRIMARY{$domain};
	&write_config($conf_file);
    }
}

sub set_soa {
local($host,@soafields) = @_;

	$SOA{$host} = join (':',@soafields);
	}

sub set_ns {
local($host,$ttl,$nserver) = @_;
local($ottl,$ns,$i,@NS);

# search the current list and if it's on it update it
	@NS = split ( ':', $NS{$host});
	for ( $i = 0; $i <= $#NS; $i++) {
		($ottl, $ns) = split (',',$NS[$i]);
		if ( $ns eq $nserver ) {
			$ttl = $ottl if $ttl eq "" && $ottl ne "";
			$NS[$i] = join (',',$ttl,$nserver);
			$i--;
			last;
			}
		}

# If it wasn't on the list add it to the list
	if ( $i > $#NS ) {
		push ( @NS, "$ttl,$nserver");
		}

	$NS{$host} = join ( ':',@NS);
	}

sub set_arec {
local($host,$ttl,$addr) = @_;
local($ottl,$ipaddr,$i,@AREC);

# search the current list and if it's on it update it
	@AREC = split ( ':', $A_RECS{$host});
	for ( $i = 0; $i <= $#AREC; $i++) {
		($ottl, $ipaddr) = split (',',$AREC[$i]);
		if ( $ipaddr eq $addr ) {
			$ttl = $ottl if $ttl eq "" && $ottl ne "";
			$AREC[$i] = join (',',$ttl,$addr);
			$i--;
			last;
			}
		}

# If it wasn't on the list add it to the list
	if ( $i > $#AREC ) {
		push ( @AREC, "$ttl,$addr");
		}

	$A_RECS{$host} = join ( ':',@AREC);
	}

sub set_cname {
local($host,$ttl,$alias) = @_;
local($ottl,$cname,$i,@CNAME);

# search the current list and if it's on it update it
	@CNAME = split ( ':', $ALIASES{$host});
	for ( $i = 0; $i <= $#CNAME; $i++) {
		($ottl, $cname) = split (',',$CNAME[$i]);
		if ( $cname eq $alias ) {
			$ttl = $ottl if $ttl eq "" && $ottl ne "";
			$CNAME[$i] = join (',',$ttl,$alias);
			$i--;
			last;
			}
		}

# If it wasn't on the list add it to the list
	if ( $i > $#CNAME ) {
		push ( @CNAME, "$ttl,$alias");
		}

	$ALIASES{$host} = join ( ':',@CNAME);
	}

sub mxpreference {
local($ttl,$a_pref,$b_pref);

	($ttl,$a_pref) = split(',',$a);
	($ttl,$b_pref) = split(',',$b);
	$a_pref <=> $b_pref;
	}

sub set_mx {
local($host,$ttl,$mxpref,$mailer) = @_;
local($ottl,$mx,$pref,$i,@MX,@MXSORTED);

# search the current list and if it's on it update it
	@MX = split ( ':', $MX_RECORDS{$host});
	for ( $i = 0; $i <= $#MX; $i++) {
		($ottl,$pref,$mx) = split (',',$MX[$i]);
		if ( $mx eq $mailer ) {
			$ttl = $ottl if $ttl eq "" && $ottl ne "";
			$NS[$i] = join (',',$ttl,$mxpref,$mailer);
			$i--;
			last;
			}
		}

# If it wasn't on the list add it to the list
	if ( $i > $#MX ) {
		push ( @MX, "$ttl,$mxpref,$mailer");
		}

	@MXSORTED = sort mxpreference @MX;

	$MX_RECORDS{$host} = join ( ':',@MXSORTED);
	}

sub set_ptr {
local($host,$ttl,$fwdhost) = @_;
local($ottl,$oldhost,$i,@PTR);

	$fwdhost =~ s/^@\.//;
# search the current list and if it's on it update it
	@PTR = split ( ':', $PTRS{$host});
	for ( $i = 0; $i <= $#PTR; $i++) {
		($ottl, $oldhost) = split (',',$PTR[$i]);
		if ( $oldhost eq $fwdhost ) {
			$ttl = $ottl if $ttl eq "" && $ottl ne "";
			$PTR[$i] = join (',',$ttl,$fwdhost);
			$i--;
			last;
			}
		}

# If it wasn't on the list add it to the list
	if ( $i > $#PTR ) {
		push ( @PTR, "$ttl,$fwdhost");
		}

	$PTRS{$host} = join ( ':',@PTR);
	}

sub set_hinfo {
local($host,$ttl,$hw,$sw) = @_;

#
# only one hinfo per hostname, strings may have variety of chars so
# I chose | as a seperator
#
	$HINFO{$host} = "$ttl:$hw:$sw";
	}

sub set_txt {
local($host,$ttl,$txt) = @_;

#
# only one txt per hostname, strings may have variety of chars so
# I chose | as a seperator
#
	$TXT{$host} = "$ttl:$txt";
	}

sub set_wks {
local($host,$ttl,$ip,$protocol,@services) = @_;
local($i,$j,$ottl,$ipaddr,$prot,$servlist,@WKS,@SERVLIST);

	@WKS = split ( ':',$WKS{$host} ) ;
	for ( $i = 0; $i <= $#WKS; $i++) {
		($ottl, $ipaddr, $prot, $servlist) = split (',',$WKS[$i]);
		if ( $ipaddr eq $ip && $prot eq $protocol ) {
			$ttl = $ottl if $ttl eq "" && $ottl ne "";
			for ( $j = 0; $j <= $#services; $j++ ) {
				if ( index($servlist,$services[$j]) < 0 ) {
					$servlist = join ( ' ',$servlist,$services[$j]);
					}
				}
			$WKS[$i] = join ( ',',$ttl,$ipaddr,$prot,$servlist);
			$i--;
			last;
			}
		}

# If it wasn't on the list add it to the list
	if ( $i > $#WKS ) {
		$serv = join ( ' ',@services);
		push ( @WKS, "$ttl,$ip,$protocol,$serv");
		}

	$WKS{$host} = join ( ':',@WKS);

	}

sub unset_soa {
local($domain) = @_;

	delete $SOA{$domain};
	}

sub unset_ns {
local($host,$nserver) = @_;
local($nsttl,$ns,$i,@NS);

	if ( $nserver eq "" ) {
		delete $NS{$host};
		return;
		}
# search the current list and remove all matches on it
	@NS = split ( ':', $NS{$host});
	for ( $i = 0; $i <= $#NS; $i++) {
		($nsttl, $ns) = split (',',$NS[$i]);
		if ( $ns eq $nserver ) {
			splice ( @NS,$i,1 );
			}
		}
	if ( $#NS >= 0 ) {
		$NS{$host} = join ( ':',@NS);
		}
	else {
		delete $NS{$host};
		}
	}

sub unset_arec {
local($host,$addr) = @_;
local($ttl,$ipaddr,$i,@AREC);

	if ( $addr eq "" ) {
		delete $A_RECS{$host};
		return;
		}

# search the current list and remove all matches on it
	@AREC = split ( ':', $A_RECS{$host});
	for ( $i = 0; $i <= $#AREC; $i++) {
		($ttl, $ipaddr) = split (',',$AREC[$i]);
		if ( $ipaddr eq $addr ) {
			splice ( @AREC,$i,1 );
			}
		}
	if ( $#AREC >= 0 ) {
		$A_RECS{$host} = join ( ':',@AREC);
		}
	else {
		delete $A_RECS{$host};
		}
	}

sub unset_cname {
local($host,$alias) = @_;
local($ttl,$cname,$i,@CNAME);

	if ( $alias eq "" ) {
		delete $ALIASES{$host};
		return;
		}
# search the current list and remove all matches on it
	@CNAME = split ( ':', $ALIASES{$host});
	for ( $i = 0; $i <= $#CNAME; $i++) {
		($ttl, $cname) = split (',',$CNAME[$i]);
		if ( $alias eq $cname ) {
			splice ( @CNAME,$i,1 );
			}
		}

	if ( $#CNAME >= 0 ) {
		$ALIASES{$host} = join ( ':',@CNAME);
		}
	else {
		delete $ALIASES{$host};
		}
	}

sub unset_mx {
local($host,$mailer) = @_;
local($ttl,$mx,$pref,$i,@MX);

	if ( $mailer eq "" ) {
		delete $MX_RECORDS{$host};
		return;
		}
# search the current list and remove all matches on it
	@MX = split ( ':', $MX_RECORDS{$host});
	for ( $i = 0; $i <= $#MX; $i++) {
		($ttl,$pref,$mx) = split (',',$MX[$i]);
		if ( $mx eq $mailer ) {
			splice ( @MX,$i,1 );
			}
		}

	if ( $#MX >= 0 ) {
		$MX_RECORDS{$host} = join ( ':',@MX);
		}
	else {
		delete $MX_RECORDS{$host};
		}
	}

sub unset_ptr {
local($host,$fwdhost) = @_;
local($ttl,$oldhost,$i,@PTR);

	if ( $fwdhost eq "" ) {
		delete $PTRS{$host};
		return;
		}

# search the current list and remove all matches on it
	@PTR = split ( ':', $PTRS{$host});
	for ( $i = 0; $i <= $#PTR; $i++) {
		($ttl, $oldhost) = split (',',$PTR[$i]);
		if ( $oldhost eq $fwdhost ) {
			splice ( @PTR,$i,1 );
			}
		}
	if ( $#PTR >= 0 ) {
		$PTRS{$host} = join ( ':',@PTR);
		}
	else {
		delete $PTRS{$host};
		}
	}

sub unset_hinfo {
local($host) = @_;

	delete $HINFO{$host};
	}

sub unset_txt {
local($host) = @_;

	delete $TXT{$host};
	}

sub unset_wks {
local($host,$ip,$protocol,@services) = @_;
local($i,$j,$ottl,$ipaddr,$prot,$servlist,@WKS,@WKSORIG,@servlist);

	# If only host is passed delete all records
	if ( $ip eq "" ) {
		delete $WKS{$host};
		return;
		}
	@WKS = split ( ':',$WKS{$host} );
	@WKSORIG = @WKS;
	for ( $i = 0; $i <= $#WKSORIG; $i++) {
		($ottl,$ipaddr,$prot,$servlist) = split (',',$WKSORIG[$i]);
		if ( $ip eq $ipaddr ) {
			# If IP  but no protocol remove all records of given IP
			if ( $protocol eq "" ) {
				splice (@WKS,$i,1);
				next;
				}
			if ( $prot eq $protocol ) {
				# If no services specified remove all services
                                # of given IP and protocol
				if ( $#services < 0 ) {
					splice (@WKS,$i,1);
					next;
					}
				@servlist = split ( ' ',$servlist);
				for ( $j = 0; $j <= $#services; $j++ ) {
					for ( $k = 0; $k <= $#servlist ; $k++) {
						if ( $services[$j] eq $servlist[$k] ){
							splice ( @servlist,$k,1 );
							}
						}
					}
				if ( $#servlist < 0 ) {
					splice (@WKS,$i,1);
					}
				else {
					$servlist = join ( ' ', @servlist);
					$WKS[$i] = join ( ',',$ttl,$ipaddr,$prot,$servlist);
					}
				}
			}
		}
	$WKS{$host} = join ( ':',@WKS);
	}

sub set_reverse {
local($ip,$host) = @_;
local($ipdbfile,$iphost,$relip,$ipdomain,$ttl,$fqdn);

	($iphost) = &ip_2_inaddrarpa($ip);
	$ipdomain = "";
	$iphost =~ /^(\d{1,3})\.(.*)$/;
	if ( defined $PRIMARY{$2} ) {
		$relip = $1;
		$ipdomain = $2;
		}
	else {
		$iphost =~ /^(\d{1,3}\.\d{1,3})\.(.*)$/;
		if ( defined $PRIMARY{$2} ) {
			$relip = $1;
			$ipdomain = $2;
			}
		else {
			$iphost =~ /^(\d{1,3}\.\d{1,3}\.\d{1,3})\.(.*)$/;
			if ( defined $PRIMARY{$2} ) {
				$relip = $1;
				$ipdomain = $2;
				}
			}
		}

	if ( $ipdomain ne "" ) {
		$fqdn = &triplet2fqdn($host);
		$ipdbfile = $PRIMARY{$ipdomain};
		$ipdbfile = "$dbdir/$ipdbfile" if $dbdir ne "";
		if ( !defined $DBFILES{$ipdbfile} ) {
			&read_dbfile($ipdbfile,$ipdomain);
			}
		&set_ptr("$ipdbfile,$ipdomain,$relip",$ttl,$fqdn);
                return 0;
		}
        return 1;
	}

sub unset_reverse {
local($host) = @_;
local($ttl,$addr,$ipaddr,$ipdbfile,$relip,$ipdomain,$fqdn);

	if ( ! defined $A_RECS{$host} ) {
		return 0;
		}
	$fqdn = &triplet2fqdn($host);
	# Remove all reverse records for host
	@AREC = split ( ':',$A_RECS{$host});
	foreach $AREC (@AREC) {
		($ttl,$addr) = split ( ',',$AREC);
		# don't remove 127.0.0.1
		next if $addr eq "127.0.0.1";
		($iphost) = &ip_2_inaddrarpa($addr);
		$ipdomain = "";
		$iphost =~ /^(\d{1,3})\.(.*)$/;
		if ( defined $PRIMARY{$2} ) {
			$relip = $1;
			$ipdomain = $2;
			}
		else {
			$iphost =~ /^(\d{1,3}\.\d{1,3})\.(.*)$/;
			if ( defined $PRIMARY{$2} ) {
				$relip = $1;
				$ipdomain = $2;
				}
			else {
				$iphost =~ /^(\d{1,3}\.\d{1,3}\.\d{1,3})\.(.*)$/;
				if ( defined $PRIMARY{$2} ) {
					$relip = $1;
					$ipdomain = $2;
					}
				}
			}

		next if $ipdomain eq "";
		$ipdbfile = $PRIMARY{$ipdomain};
		$ipdbfile = "$dbdir/$ipdbfile" if $dbdir ne "";
		if ( !defined $DBFILES{$ipdbfile} ) {
			&read_dbfile($ipdbfile,$ipdomain);
			&set_dbfile($ipdbfile,$ipdomain);
			}
		&unset_ptr("$ipdbfile,$ipdomain,$relip",$fqdn);
		}
	}

sub set_dbfile {
local ($dbfile,$origin) = @_;
local($i,@origins);

	@origins = split (':',$DBFILES{$dbfile});
	foreach $i (@origins) {
		#
		# It's already on the list
		#
		return if $i eq $origin;
		}
	push ( @origins, $origin );
	$DBFILES{$dbfile} = join ( ':',@origins);
	}

sub unset_dbfile {
local ($dbfile,$origin) = @_;
local ($i,@origins);

	@origins = split (':',$DBFILES{$dbfile});
	foreach $i (@origins) {
		if ( $i eq $origin ) {
			splice ( @origins, $i, 1);
			}
		}
	if ( $#origins >= 0 ) {
		$DBFILES{$dbfile} = join ( ':',@origins);
		}
	else {
		delete  $DBFILES{$dbfile};
		}
	}

sub read_config {
local($file) = @_;
    
# XXX force a call to the real read routine so we catch it when
# they've modified named.conf locally.  This is really gross, but
# it's probably better than duplicating the code and we don't have time
# to completely rewrite it now.
my $named = new FileFormat::DNS::NamedBoot($file, 0, 0);
$named->close;

open(CONFIG_FILE, "<$file") || return 1;
while ( <CONFIG_FILE> ) {
#
# Strip out comments
#
	s/;.*$//;
#
# Break into fields
#
	@line = split;
#
# Check the entry type
#
#
# Sortlist entry
#
	if ( $line[0] =~ /^sortlist/i ) {
		shift @line;
		@sortlist = @line;
		}
#
# Directory entry
#
	elsif ( $line[0] =~ /^directory/i ) {
		$dbdir = $line[1];
		}
#
# Cache entry
#
	elsif ( $line[0] =~ /^cache/i ) {
		$CACHE{$line[1]} = $line[2];
		}
#
# Primary entry
#
	elsif ( $line[0] =~ /^primary/i ) {
		$line[1] = "$line[1]." if $line[1] !~ /.+\.$/;
		$PRIMARY{"\L$line[1]"} = $line[2];
		}
#
# Secondary entry
#
	elsif ( $line[0] =~ /^secondary/i ) {
		$line[1] = "$line[1]." if $line[1] !~ /.+\.$/;
		$line[1] = "\L$line[1]";
		$SECONDARY{$line[1]} = $line[$#line];
		for ( $i = 2; $i <= $#line - 1; $i++) {
			$SECONDARY{$line[1]} ="$SECONDARY{$line[1]},$line[$i]";
			}
		}
#
#Forwarder enrty
#
	elsif ( $line[0] =~ /^forwarders/i ) {
		shift line;
		push(@forwarders,@line);
		}
#
# Slave entry
#
	elsif ( $line[0] =~ /^slave/i ) {
		$slave = "slave";
		}
	
	}
close(CONFIG_FILE);
return 0;
	}

sub init_config {
local($dbfile,$domain);

	$domain = "127.in-addr.arpa.";
	$dbfile = "${domain}db";
	$CACHE{"."}="root.cache";
	#&write_cache("."); DO this outside this routine

	$PRIMARY{$domain} = $dbfile;
	$now = time;
	$dbfile = "$dbdir/$dbfile" if $dbdir ne "";

	&set_soa("$dbfile,$domain,","",$myfqdname,"hostmaster.$myfqdname",$now,"28800","7200","604800","86400");
	&set_ns("$dbfile,$domain,","",$myfqdname);
	&set_ptr("$dbfile,$domain,1.0.0","","localhost.$mydomain");

	&set_dbfile($dbfile,$domain);

	return 0;
	}

sub init_resolver {
	local($file, $ipaddr) = @_;
	local($i);

	open(RESOLVFILE, ">$file") || return 1;
	$i = $mydomain;
	$i =~ s/\.*$//;
	printf (RESOLVFILE "domain	$i\n");
	printf (RESOLVFILE "nameserver	$ipaddr\n");
	close(RESOLVFILE);
	return 0;
	}

sub write_config {
	local($file) = @_;
	local($dbfile,$domain,$sortaddr,$cache,@primaddr,$forwaddr);
	open(CONFIG_FILE, ">$file") || return 1;
#
# Write any existing sortlist
#
	if ( $#sortlist >= 0 ) {
		print CONFIG_FILE "sortlist";
		foreach $sortaddr (@sortlist) {
			print CONFIG_FILE " $sortaddr";
			}
		print CONFIG_FILE "\n";
		}

#
# Write the directory entry
#
	print CONFIG_FILE ";\n;Directory for db files\n;\n";
	print CONFIG_FILE "directory	$dbdir\n";
#
# Print any cache entries
#
	print CONFIG_FILE ";\n;Cache files\n;\n";
	foreach $cache (sort( keys %CACHE)) {
		print CONFIG_FILE "cache	$cache	$CACHE{$cache}\n";
		}
#
# Print any domains We are the primary for
#
	print CONFIG_FILE ";\n;Primary domains\n;\n";
	foreach $domain (sort( keys %PRIMARY)) {
		$dbfile = $PRIMARY{$domain};
		$domain =~ s/(.*)\.$/$1/;
		print CONFIG_FILE "primary		$domain 	$dbfile\n";
		}	
#
# Print any domains we are a secondary for
#
	printf(CONFIG_FILE ";\n;Secondary domains\n;\n");
	foreach $domain (sort( keys %SECONDARY)) {
		($dbfile,@primaddr) = split(/,/,$SECONDARY{$domain});
		$domain =~ s/(.*)\.$/$1/;
		print CONFIG_FILE "secondary	$domain";
		foreach $primaddr (@primaddr) {
			print CONFIG_FILE " $primaddr";
			}
		print CONFIG_FILE "	$dbfile\n";
		}
#
# Print any forwarders
#
	if ( $#forwarders >= 0 ) {
		print CONFIG_FILE ";\n;Forwarder list\n;\n";
		print CONFIG_FILE "forwarders";
		foreach $forwaddr (@forwarders) {
			print CONFIG_FILE " $forwaddr";
			}
		print CONFIG_FILE "\n";
		}
#
# Print any Slave entry
#
	if ( $slave eq "slave" ) {
		print CONFIG_FILE ";\n;Slave option\n;\n";
		print CONFIG_FILE "$slave\n";
		}
#
# Finish up
#
	close(CONFIG_FILE);

	# XXX:
	convert_conf($file);

	return 0;
	}

sub read_dbfile {
	local($dbfile,$domain) = @_;
	local($ip,$origin,$neworigin,$dum,$linecnt);

	if ( ! open (DOMAINFILE, "<$dbfile")) {
		push(@errors,"Open of dbfile $dbfile Failed");
		return 1;
		}

	$origin = $domain;
	&set_dbfile($dbfile,$origin);
	$linecnt = 0;
	while ( <DOMAINFILE> ) {
		$linecnt++;
		chop;
		#
		# Strip out comments
		#
		s/;.*$//;
		#
		# If it's a blank line, read next line
		#
		if ( /^$/ ) {
			next;
			}
		#
		# IF there is a line continuation, concatenate lines
		#
		if ( /\(/ ) {
			while ( ! /\(.*\)/ ) {
				$oldline = $_;
				chop ($_ = <DOMAINFILE>);
				$linecnt++;
				s/;.*$//;
				$_ = join(' ',$oldline,$_);
				}
			#
			# remove the '(' and ')'
			#
			s/\(//;
			s/\)//;
			}
		#
		# Check for a $INCLUDE or $ORIGIN or $TTL line
		#
		if ( /^\$ORIGIN/i ) {
			($dum,$neworigin) = split;
			if ( $neworigin !~ /\.$/ ) {
				$origin = "$neworigin.$origin";
				}
			else {
				$origin = $neworigin;
				}
			&set_dbfile($dbfile,$origin);
			next;
			}
		if ( /^\$INCLUDE/i ) {
			push(@errors,"\$INCLUDE lines currently not supported");
			return 1;
			}
		if ( /^\$TTL/i ) {
			($dum, $xttl) = split;
			next;
			}
		#
		# If the line doesn't begin with a whitespace get the hostname
		#
		if ( /^[\@a-zA-Z0-9_-]/ ) {
			@restline = &parse_line($_);
			$owner = shift @restline;
			$owner = "\L$owner";
			#
			# If the entry is the domain name or @
			#
			if ( $owner eq "@" ||$owner =~ /^$origin/ ) {
				$owner = "";
				}
			#
			# If the entry does have a .origin at the end
			# (Is fully qualified name in this origin)
			#
			elsif ( $owner =~ /(.+).$origin/ ) {
				$owner = $1;
				}
			#
			# Is not a fully qualified domain
			# Or is fully qualified, but found in this domain
			# was found in this $ORIGIN of the file
			else {
				$owner = $owner;
				}
			}
		else {
			#
			# Strip out leading whitespaces
			#
			s/^[\s]*//;
			if ( ((s/\"/\"/g)%2) ne 0 ) {
				push(@errors,"Unbalanced \" near line $linecnt\n");
				return 1;
				}
			@restline = &parse_line($_);
			}

		$ttl = "";
		#
		# Check for a TTL value for this record
		#
		if ( $restline[0] =~ /^\d+$/ ) {
			$ttl = $restline[0];
			shift @restline;
			}
		if ( $restline[0] ne "IN" ) {
			push(@errors,"Unknown record type line $linecnt: $restline[0]\n");
			return 1;
			next;
			}
		if ( $restline[1] eq "A" ) {
			&set_arec("$dbfile,$origin,$owner",$ttl,$restline[2]);
			next;
			}
		if ( $restline[1] eq "PTR" ) {
			&set_ptr("$dbfile,$origin,$owner",$ttl,"\L$restline[2]");
			next;
			}
		if ( $restline[1] eq "CNAME" ) {
			&set_cname("$dbfile,$origin,$owner",$ttl,"\L$restline[2]");
			next;
			}
		if ( $restline[1] eq "MX" ) {
			&set_mx("$dbfile,$origin,$owner",$ttl,$restline[2],"\L$restline[3]");
			next;
			}
		if ( $restline[1] eq "HINFO" ) {
			&set_hinfo("$dbfile,$origin,$owner",$ttl,$restline[2],$restline[3]);
			next;
			}
		if ( $restline[1] eq "TXT" ) {
			&set_txt("$dbfile,$origin,$owner",$ttl,$restline[2]);
			next;
			}
		if ( $restline[1] eq "WKS" ) {
			shift @restline;
			shift @restline;
			$ip = shift @restline;
			$protocol = shift @restline;
			for ($lpvar = 0; $lpvar <= $#restline; $lpvar++) {
				$restline[$lpvar] = "\L$restline[$lpvar]";
				}
			&set_wks("$dbfile,$origin,$owner",$ttl,$ip,$protocol,@restline);
			next;
			}
		if ( $restline[1] eq "NS" ) {
			&set_ns("$dbfile,$origin,$owner",$ttl,"\L$restline[2]");
			next;
			}
		if ( $restline[1] eq "SOA" ) {
			&set_soa("$dbfile,$origin,$owner",$ttl,"\L$restline[2]","\L$restline[3]",$restline[4],$restline[5],$restline[6],$restline[7],$restline[8]);
			next;
			}
		}

	close DOMAINFILE;
	return 0;
	}


sub write_cache {
	local($cache) = @_;

	open (CACHEFILE, ">$dbdir/$CACHE{$cache}") || return 1;

	print CACHEFILE <<"EOF";
;
;       This file holds the information on root name servers needed to
;       initialize cache of Internet domain name servers 
;       (e.g. reference this file in the "cache  .  <file>"
;       configuration file of BIND domain name servers).
;
;       This file is made available by InterNIC registration services
;       under anonymous FTP as
;           file                /domain/named.root
;           on server           FTP.RS.INTERNIC.NET
;       -OR- under Gopher at    RS.INTERNIC.NET
;           under menu          InterNIC Registration Services (NSI)
;              submenu          InterNIC Registration Archives
;           file                named.root
;
;       last update:    Oct 5, 1994
;       related version of root zone:   1994100500
;
.                        99999999 IN  NS    NS.INTERNIC.NET.
NS.INTERNIC.NET.         99999999     A     198.41.0.4
.                        99999999     NS    NS1.ISI.EDU. 
NS1.ISI.EDU.             99999999     A     128.9.0.107
.                        99999999     NS    C.PSI.NET.
C.PSI.NET.               99999999     A     192.33.4.12
.                        99999999     NS    TERP.UMD.EDU.
TERP.UMD.EDU.            99999999     A     128.8.10.90
.                        99999999     NS    NS.NASA.GOV.
NS.NASA.GOV.             99999999     A     128.102.16.10
                         99999999     A     192.52.195.10
.                        99999999     NS    NS.ISC.ORG.
NS.ISC.ORG.              99999999     A     192.5.5.241
.                        99999999     NS    NS.NIC.DDN.MIL.
NS.NIC.DDN.MIL.          99999999     A     192.112.36.4
.                        99999999     NS    AOS.ARL.ARMY.MIL.
AOS.ARL.ARMY.MIL.        99999999     A     128.63.4.82
                         99999999     A     192.5.25.82
.                        99999999     NS    NIC.NORDU.NET.
NIC.NORDU.NET.           99999999     A     192.36.148.17
; End of File
EOF

	close CACHEFILE;
	return 0;
	}

sub print_dbfile {
local($dbfile) = @_;
local($real_domain);
local(@origin,$origin);

if ( ! open(DOMAINFILE, ">$dbfile") ) {
	push(@error,"Couldn't open dbfile $dbfile for writing\n");
	return 1;
	}

print DOMAINFILE <<EOC;
;
; This file was Generated by config_mdns. It can be directly edited if you
;follow the following rules:
;
; Only the following record types will be preserved across subsequent runs
;
;	SOA    Record  (Can only be one per origin)
;	NS     Records
;	A      Records
;	MX     Records
;	CNAME  Records
;	HINFO  Records (Only 1 per host, no ':' in hardware or software fields)
;	TXT    Records (Only 1 per host)
;	WKS    Records
;	PTR    Records (Should only be in Reverse Domains)
;
; Any other type of Record should be put below the DO NOT DELETE comment
; line near the bottom of the file, or it will not be lost after running
; config_mdns.
; The config_mdns converts all name to lower case, and uses relative
; hostnames.
;
EOC

$xttl = 28800 if $xttl == 0;
print DOMAINFILE "\$TTL $xttl\n";

@origin = split (':',$DBFILES{$dbfile});

print DOMAINFILE "; This dbfile supports the following origins\n";

foreach $origin ( @origin ) {
	print DOMAINFILE "; $origin\n";
	}

print DOAMINFILE ";\n";

foreach $origin ( @origin ) {

	print DOMAINFILE ";\n\$ORIGIN $origin\n;\n";

	if ( defined $SOA{"$dbfile,$origin,"} ) {
		# XXX:
		$real_domain = $origin;
		print DOMAINFILE "@	IN	SOA	";
		($ttl,$phost,$email,$oserial,$refresh,$retry,$expire,$minimum) = split(':',$SOA{"$dbfile,$origin,"});
		printf(DOMAINFILE "$phost	$email \(\n");
		$serial = time;
		if ( $oserial > $serial ) {
			$serial = $oserial + 1;
			}
		printf(DOMAINFILE "		$serial ; Serial Number\n");
		printf(DOMAINFILE "		$refresh ; Refresh\n");
		printf(DOMAINFILE "		$retry ; Retry\n");
		printf(DOMAINFILE "		$expire ; Expire\n");
		printf(DOMAINFILE "		$minimum \) ; Minimum TTL\n");
		}

	printf(DOMAINFILE ";\n; DNS Servers, A Record and MX Records for this domain\n;\n");
	@NS = split ( ':',$NS{"$dbfile,$origin,"});
	for ( $i = 0; $i <= $#NS; $i++ ) {
		($ttl,$ns) = split (',',$NS[$i]);
		printf(DOMAINFILE "	$ttl	IN	NS	$ns\n");
		}
	if ( defined $A_RECS{"$dbfile,$origin,"} ) {
		@A_RECS = split (':',$A_RECS{"$dbfile,$origin,"});
		for ( $i = 0; $i <= $#A_RECS; $i++ ) {
			($ttl,$ipaddr) = split(',',$A_RECS[$i]);
			printf(DOMAINFILE "	$ttl	IN	A	$ipaddr\n");
			}
		}

	if ( defined $MX_RECORDS{"$dbfile,$origin,"} ) {
		@MX_RECORDS = split(':',$MX_RECORDS{"$dbfile,$origin,"});
		for ( $i = 0; $i <= $#MX_RECORDS; $i++ ) {
			($ttl,$preference,$mxhost) = split(',',$MX_RECORDS[$i]);
			printf(DOMAINFILE "	$ttl	IN	MX	$preference	$mxhost\n");
			}
		}

	if ( defined $HINFO{"$dbfile,$origin,"} ) {
		($ttl,$hw,$sw) = split(':',$HINFO{"$dbfile,$origin,"},3);
		printf(DOMAINFILE "	$ttl	IN	HINFO	\"$hw\"	\"$sw\"\n");
		}

	if ( defined $TXT{"$dbfile,$origin,"} ) {
		($ttl,$txt) = split(':',$TXT{"$dbfile,$origin,"},2);
		printf(DOMAINFILE "	$ttl	IN	TXT	\"$txt\"\n");
		}

	if ( defined $WKS{"$dbfile,$origin,"} ) {
		@WKS_RECS = split ( ':', $WKS{"$dbfile,$origin,"} );
		for ( $i = 0; $i <= $#WKS_RECS; $i++ ) {
			($ttl,$ip,$protocol,$services) = split(',',@WKS_RECS[$i]);
			printf(DOMAINFILE "\t$ttl\tIN\tWKS\t$ip\t$protocol");
			@services = split ( ' ',$services);
			if ( $#services == 0 ) {
				printf(DOMAINFILE "\t$services[0]\n");
				}
			else {
				printf(DOMAINFILE " \(\t$services[0]\n");
				for ( $j = 1; $j <= $#services; $j++ ) {
					printf(DOMAINFILE "\t\t\t\t\t$services[$j]");
					if ( $j == $#services ) {
						printf(DOMAINFILE " \)\n");
						}
					else {
						printf(DOMAINFILE "\n");
						}
					}
				}
			}
		}

	$havewritten = 0;
	foreach $host ( sort ( keys %NS ) ) {
		next if $host !~ /^$dbfile,$origin,(.*)/;
		$relhost = $1;
		next if ($relhost eq "");

		if ( !$havewritten ) {
			printf(DOMAINFILE ";\n; NS Records for Sub-Domains of domain $domain\n;\n");
			$havewritten = 1;
			}

		@NS = split (':',$NS{$host});
		printf(DOMAINFILE "$relhost");
		for ( $i = 0; $i <= $#NS; $i++ ) {
			($ttl,$nshost) = split(',',$NS[$i]);
			printf(DOMAINFILE "	$ttl	IN	NS	$nshost\n");
			}
		}

	$havewritten = 0;
	if ( defined $A_RECS{"$dbfile,$origin,localhost"} ) {
		printf(DOMAINFILE ";\n; A Records for hosts in origin $origin\n;\n");
		$havewritten = 1;
		@A_RECS = split (':',$A_RECS{"$dbfile,$origin,localhost"});
		printf(DOMAINFILE "localhost");
		for ( $i = 0; $i <= $#A_RECS; $i++ ) {
			($ttl,$ipaddr) = split(',',$A_RECS[$i]);
			printf(DOMAINFILE "	$ttl	IN	A	$ipaddr\n");
			}
		printf(DOMAINFILE ";\n");
		}
	if ( defined $HINFO{"$dbfile,$origin,localhost"} ) {
		($ttl,$hw,$sw) = split(':',$HINFO{"$dbfile,$origin,localhost"},3);
		printf(DOMAINFILE "	$ttl	IN	HINFO	\"$hw\"	\"$sw\"\n");
		}
	if ( defined $TXT{"$dbfile,$origin,localhost"} ) {
		($ttl,$txt) = split(':',$TXT{"$dbfile,$origin,localhost"},2);
		printf(DOMAINFILE "	$ttl	IN	TXT	\"$txt\"\n");
		}

	if ( defined $WKS{"$dbfile,$origin,localhost"} ) {
		@WKS_RECS = split ( ':', $WKS{"$dbfile,$origin,localhost"} );
		for ( $i = 0; $i <= $#WKS_RECS; $i++ ) {
			($ttl,$ip,$protocol,$services) = split(',',@WKS_RECS[$i]);
			printf(DOMAINFILE "\t$ttl\tIN\tWKS\$ip\t$protocol");
			@services = split ( ' ',$services);
			if ( $#services == 0 ) {
				printf(DOMAINFILE "\t$services[0]\n");
				}
			else {
				printf(DOMAINFILE " \(\t$services[0]\n");
				for ( $j = 1; $j <= $#services; $j++ ) {
					printf(DOMAINFILE "\t\t\t\t\t$services[$j]");
					if ( $j == $#services ) {
						printf(DOMAINFILE " \)\n");
						}
					else {
						printf(DOMAINFILE "\n");
						}
					}
				}
			}
		}

	foreach $host ( sort ( keys %A_RECS ) ) {
		next if $host !~ /^$dbfile,$origin,(.*)/;
		$relhost = $1;
		next unless  ($relhost ne "" && $relhost ne "localhost");
		if ( !$havewritten ) {
			printf(DOMAINFILE ";\n; A Records for hosts in origin $origin\n;\n");
			$havewritten = 1;
			}
		
		@A_RECS = split (':',$A_RECS{$host});
		printf(DOMAINFILE "$relhost");
		for ( $i = 0; $i <= $#A_RECS; $i++ ) {
			($ttl,$ipaddr) = split(',',$A_RECS[$i]);
			printf(DOMAINFILE "	$ttl	IN	A	$ipaddr\n");
			}
		if ( defined $MX_RECORDS{$host} ) {
			@MX_RECORDS = split(':',$MX_RECORDS{$host});
			for ( $i = 0; $i <= $#MX_RECORDS; $i++ ) {
				($ttl,$preference,$mxhost) = split(',',$MX_RECORDS[$i]);
				printf(DOMAINFILE "	$ttl	IN	MX	$preference	$mxhost\n");
				}
			}
		if ( defined $HINFO{$host} ) {
			($ttl,$hw,$sw) = split(':',$HINFO{$host},3);
			printf(DOMAINFILE "	$ttl	IN	HINFO	\"$hw\"	\"$sw\"\n");
			}
		if ( defined $TXT{$host} ) {
			($ttl,$txt) = split(':',$TXT{$host},2);
			printf(DOMAINFILE "	$ttl	IN	TXT	\"$txt\"\n");
			}
		if ( defined $WKS{"$host"} ) {
			@WKS_RECS = split ( ':', $WKS{"$host"} );
			for ( $i = 0; $i <= $#WKS_RECS; $i++ ) {
				($ttl,$ip,$protocol,$services) = split(',',@WKS_RECS[$i]);
				printf(DOMAINFILE "\t$ttl\tIN\tWKS\t$ip\t$protocol");
				@services = split ( ' ',$services);
				if ( $#services == 0 ) {
					printf(DOMAINFILE "\t$services[0]\n");
					}
				else {
					printf(DOMAINFILE " \(\t$services[0]\n");
					for ( $j = 1; $j <= $#services; $j++ ) {
						printf(DOMAINFILE "\t\t\t\t\t\t$services[$j] ");
						if ( $j == $#services ) {
							printf(DOMAINFILE " \)\n");
							}
						else {
							printf(DOMAINFILE "\n");
							}
						}
					}
				}
			}

		printf(DOMAINFILE ";\n");
		}

	$havewritten = 0;
	foreach $host ( sort ( keys %ALIASES ) ) {
		next if $host !~ /^$dbfile,$origin,(.*)/;
		$relhost = $1;
		next unless  ($relhost ne "" && $relhost ne "localhost");

		if ( !$havewritten ) {
			printf(DOMAINFILE ";\n;Aliases for this domain\n;\n");
			$havewritten = 1;
			}

		($ttl,$alias) = split ( ',',$ALIASES{$host});
		printf(DOMAINFILE "$relhost	$ttl	IN	CNAME	$alias\n");

		if ( defined $HINFO{$host} ) {
			($ttl,$hw,$sw) = split(':',$HINFO{$host},3);
			printf(DOMAINFILE "	$ttl	IN	HINFO	\"$hw\"	\"$sw\"\n");
			}
		if ( defined $TXT{$host} ) {
			($ttl,$txt) = split(':',$TXT{$host},2);
			printf(DOMAINFILE "	$ttl	IN	TXT	\"$txt\"\n");
			}
		if ( defined $WKS{"$host"} ) {
			@WKS_RECS = split ( ':', $WKS{"$host"} );
			for ( $i = 0; $i <= $#WKS_RECS; $i++ ) {
				($ttl,$ip,$protocol,$services) = split(',',@WKS_RECS[$i]);
				printf(DOMAINFILE "\t$ttl\tIN\tWKS\t$ip\t$protocol");
				@services = split ( ' ',$services);
				if ( $#services == 0 ) {
					printf(DOMAINFILE "\t$services[0]\n");
					}
				else {
					printf(DOMAINFILE " \(\t$services[0]\n");
					for ( $j = 1; $j <= $#services; $j++ ) {
						printf(DOMAINFILE "\t\t\t\t\t$services[$j]");
						if ( $j == $#services ) {
							printf(DOMAINFILE " \)\n");
							}
						else {
							printf(DOMAINFILE "\n");
							}
						}
					}
				}
			}
		}

	$havewritten = 0;
	foreach $ip (sort ( keys %PTRS ) ) {
		next if $ip !~ /^$dbfile,$origin,(.*)/;
		$relip = $1;

		if ( !$havewritten ) {
			printf(DOMAINFILE ";\n; PTR Records for domain $domain\n;\n");
			$havewritten = 1;
			}
		@PTRS = split ( ':', $PTRS{$ip});
		for ( $i = 0; $i <= $#PTRS; $i++ ) {
			($ttl,$ptr) = split ( ',',$PTRS[$i]);
			printf(DOMAINFILE "$relip	$ttl	IN	PTR	$ptr\n");
			}
		}

	$havewritten = 0;
	foreach $host ( sort ( keys %MX_RECORDS ) ) {
		next if $host !~ /^$dbfile,$origin,(.*)/;
		$relhost = $1;
		next if ( defined $A_RECS{$host} || $relhost eq "" ) ;

		if ( !$havewritten ) {
			printf(DOMAINFILE ";\n;MX hosts without an A record\n;\n");
			$havewritten = 1;
			}

		printf(DOMAINFILE "$relhost");
		@MX_RECORDS =split(':',$MX_RECORDS{$host});
		for ( $i = 0; $i <= $#MX_RECORDS; $i++ ) {
			($ttl,$preference,$mxhost) = split(',',$MX_RECORDS[$i]);
			printf(DOMAINFILE "	$ttl	IN	MX	$preference	$mxhost\n");
			}
		printf(DOMAINFILE ";\n");
		}

	$havewritten = 0;
	foreach $host ( sort ( keys %HINFO ) ) {
		next if $host !~ /^$dbfile,$origin,(.*)/;
		$relhost = $1;
		next if ( defined $A_RECS{$host} || $relhost eq "" ) ;

		if ( !$havewritten ) {
			printf(DOMAINFILE ";\n;HINFO records without an A record\n;\n");
			$havewritten = 1;
			}

		($ttl,$hw,$sw) = split(':',$HINFO{$host},3);

		printf(DOMAINFILE "$relhost");
		printf(DOMAINFILE "	$ttl	IN	HINFO	\"$hw\"	\"$sw\"\n");
		printf(DOMAINFILE "\n;\n");
		}

	$havewritten = 0;
	foreach $host ( sort ( keys %TXT ) ) {
		next if $host !~ /^$dbfile,$origin,(.*)/;
		$relhost = $1;
		next if ( defined $A_RECS{$host} || $relhost eq "" ) ;

		if ( !$havewritten ) {
			printf(DOMAINFILE ";\n;TXT records without an A record\n;\n");
			$havewritten = 1;
			}

		($ttl,$txt) = split(':',$TXT{$host},2);

		printf(DOMAINFILE "$relhost");
		printf(DOMAINFILE "	$ttl	IN	TXT	\"$txt\"\n");
		printf(DOMAINFILE "\n;\n");
		}

	$havewritten = 0;
	foreach $host ( sort ( keys %WKS ) ) {
		next if $host !~ /^$dbfile,$origin,(.*)/;
		$relhost = $1;
		next if ( defined $A_RECS{$host} || $relhost eq "" ) ;

		if ( !$havewritten ) {
			printf(DOMAINFILE ";\n;WKS records without an A record\n;\n");
			$havewritten = 1;
			}

		printf(DOMAINFILE "$relhost");
		@WKS_RECS = split ( ':', $WKS{$host} );
		for ( $i = 0; $i <= $#WKS_RECS; $i++ ) {
			($ttl,$ip,$protocol,$services) = split(',',@WKS_RECS[$i]);
			printf(DOMAINFILE "\t$ttl\tIN\tWKS\t$ip\t$protocol");
			@services = split (' ',$services);
			if ( $#services == 0 ) {
				printf(DOMAINFILE "\t$services[0]\n");
				}
			else {
				printf(DOMAINFILE " \(\t$services[0]\n");
				for ( $j = 1; $j <= $#services; $j++ ) {
					printf(DOMAINFILE "\t\t\t\t\t$services[$j]");
					if ( $j == $#services ) {
						printf(DOMAINFILE " \)\n");
						}
					else {
						printf(DOMAINFILE "\n");
						}
					}
				}
			}
		printf(DOMAINFILE "\n;\n");
		}
	}
printf (DOMAINFILE ";\n; DO NOT DELETE THIS LINE -- place local changes below here\n");

close DOMAINFILE;
# XXX:
convert_domain($real_domain, $dbfile);
}

sub ip_2_inaddrarpa {
	local($ip) = @_;

	$ip =~ /^(\d+)\.(\d+)\.(\d+)\.(\d+)$/;

	#
	# Class A network
	#
	if ( $1 < 128 ) {  
		return ("$4.$3.$2.$1.in-addr.arpa.","$1.in-addr.arpa.","$4.$3.$2" );
		}
	#
	# Class B network
	#
	elsif ( $1 > 127 && $1 < 192 ) {
		return ("$4.$3.$2.$1.in-addr.arpa.","$2.$1.in-addr.arpa.","$4.$3" );
		}
	#
	# Class C network
	#
	return ("$4.$3.$2.$1.in-addr.arpa.","$3.$2.$1.in-addr.arpa.","$4" );
	}

sub net_2_inaddrarpa {
	local($net) = @_;

	@bytes = split(/\./,$net);

        return "$bytes[0].in-addr.arpa." if ( $bytes[0] < 128 );
        return "$bytes[1].$bytes[0].in-addr.arpa." if ( $bytes[0] > 127 && $bytes[0] < 192 );
        return "$bytes[2].$bytes[1].$bytes[0].in-addr.arpa.";


	}

sub byorigin {
	join('.',reverse split('.',$a)) cmp join('.',reverse split('.',$b));	
	}

sub clear_all_recs {
	undef %DBFILES;
	undef %PRIMARY;
	undef %SECONDARY;
	undef %SOA;
	undef %A_RECS;
	undef %ALIASES;
	undef %MX_RECORDS;
	undef %PTRS;
	undef %NS;
	undef %HINFO;
	undef %TXT;
	undef %WKS;
	}

sub mkdirp {
        local($dir,$mode) = @_;
        local($nxtpath,$pathname);

        @PATH = split('/',$dir);
        $pathname = shift(@PATH);

        for(;;) {
                (mkdir($pathname,$mode) || return 1) if ! -d $pathname;
                last unless defined ($nxtpath = shift(@PATH));
                $pathname = join('/',$pathname,$nxtpath);
                }
        }

sub get_myinfo {
	local($sys_hostname,$netstart_hostname,$resolv_domain,$dotcnt);

	$resolv_domain = "";
	if ( open (RESOLV, "<$resolv_file") ) {
		while ( <RESOLV> ) {
			chop;
			if (/^domain\s+(.+)/i ) {
				$resolv_domain = "\L$1";
				$resolv_domain =~ s/\.*$/./;
				last;
				}
			}
		close RESOLV;
		}
	$netstart_hostname = "";
	$myipaddr = "";
	if ( open (NETSTART, "<$netstart_file") ) {
		while ( <NETSTART> ) {
			chop ;
			$netstart_hostname = "\L$1" if /^hostname=(.+)/i;
			$myipaddr = $1 if /^ipaddr=(.+)/i;
			last if $netstart_hostname ne "" && $myipaddr ne "";
			}
		close NETSTART;
		}
	chop ($sys_hostname = `/bin/hostname`);
	$sys_hostname = "\L$sys_hostname";

	$mydomain = $resolv_domain;

	if ( $sys_hostname =~ /$mydomain$/i || "$sys_hostname." =~ /$mydomain$/i ) {
		$myfqdname = $sys_hostname;
		$myfqdname =~ s/\.*$/\./;
		}
	elsif ( $sys_hostname =~ /\.$/ ) {
		$myfqdname = $sys_hostname;
		}
	else {
		$myfqdname = "$sys_hostname.$mydomain";
		}
	return($myfqdname,$mydomain,$myipaddr);
	}

sub reload_dns {
	local($pid,);

	if ( open (NAMEDRUN, "<$named_pid_file")) {
		chop ($pid = <NAMEDRUN>);
		close NAMEDRUN;
		return 0 if kill ('HUP', $pid) == 1;
		}
	system ( '/usr/sbin/named' );
	return 0;
	}
	 
sub kill_dns {
	local($pid,);

	if ( open (NAMEDRUN, "<$named_pid_file")) {
		chop ($pid = <NAMEDRUN>);
		close NAMEDRUN;
		return 0 if kill ('TERM', $pid) == 1;
		}
	}

sub read_resolver {
local ( @dummy );

	if ( open (RESOLV, "<$resolv_file") ) {

		while ( <RESOLV> ) {
			chomp;
			if (/^domain\s+(.+)/i ) {
				$RESOLV_DOMAIN = "\L$1";
				$RESOLV_DOMAIN =~ s/\.*$/./;
				}
			if (/^nameserver\s+(.+)/i ) {
				push(@RESOLV_NS, $1);
				}
			if (/^search\s+.+/i ) {
				@dummy = split;
				shift @dummy;
				push(@RESOLV_SRCH,@dummy );
				}
			}
		close RESOLV;
		return 0;
		}
	push (@error,"Couldn't open resolver file $resolv_file for reading\n");
	return 1;
	}

sub write_resolver {
local ( @dummy, $n );
	if ( open (RESOLV, ">$resolv_file") ) {
		if ( defined $RESOLV_DOMAIN ) {
			$RESOLV_DOMAIN =~ s/\.*$//;
			print RESOLV "domain\t\t$RESOLV_DOMAIN\n";
			}
		if ( $#RESOLV_NS >= 0) {
			for $n (@RESOLV_NS) {
				print RESOLV "nameserver\t$n\n";
				}
			}
		if ( $#RESOLV_SRCH >= 0 ) {
			print RESOLV "search";
			for $n (@RESOLV_NS) {
				print RESOLV " $n";
				}
			print RESOLV "\n";
			}
		return 0;
		}
	push (@error,"Couldn't open resolver file $resolv_file for writing/n");
	return 1;
	}

sub write_dbfile {
local($dbfile) = @_;
local($dbtmp);
	
	if ( -f "$dbfile" ) {
		#&mkdirp ("$dbdir/old_dns",0755) if ! -d "$dbdir/old_dns";
		unlink "${dbfile}.old" ;
		rename($dbfile,"${dbfile}.old");
		}
	else {
		$dbtmp = $dbfile;
		$dbtmp =~ s/(.*)\/.+$/$1/;
		&mkdirp ($dbtmp,0755);
		}
	&print_dbfile($dbfile);

	if ( -f "${dbfile}.old" ) {
		open( OLDDB, "<${dbfile}.old");
		while (<OLDDB>) {
			if ( /^; DO NOT DELETE THIS LINE -- place local changes below here/ ) {
				open ( NEWDB, ">>$dbfile");
				while ( <OLDDB> ) {
					print NEWDB;
					}
				close NEWDB;
				}
			}
		close OLDDB;
		}
	return 0;
	}

sub init_domain {
local($domain,$defmailer,@nslist) = @_;

    # Setup the configuration file
    mkdirp($dbdir,0755) if $dbdir ne '' && ! -d $dbdir;
    $PRIMARY{$domain} = "${domain}db";

    $dbfile = $PRIMARY{$domain};
    $dbfile = "$dbdir/$dbfile" if $dbdir ne "";
    set_dbfile($dbfile,$domain);

    # Setup the DBfile
    my $ttl = "";
    my $now = time;

    # set the SOA record
    set_soa("$dbfile,$domain,",$ttl,$myfqdname,"hostmaster.$myfqdname",$now,"28800","7200","604800","86400");

    # Set the NS records
    my $sec;
    foreach $sec (@nslist) {
        # if no . assume it's in this domain
        $sec = "$sec.$domain" if ($sec =~ s/\./\./g == 0);
        $sec =~ s/\.*$/\./;
        set_ns("$dbfile,$domain,",$ttl,$sec);
    }

    # IF a forward domain set localhost and domain's MX record
    if ( ! ($domain =~ /in-addr\.arpa\.$/) ) {
        set_arec("$dbfile,$domain,localhost",$ttl,"127.0.0.1");
        if ( $defmailer !~ /^$/ ) {
            # If no . in it assume it's in local domain
            if ( ($defmailer =~ s/\./\./g) == 0) {
                set_mx("$dbfile,$domain,",$ttl,5,"$defmailer.$domain");
	    }
                # If it doesn't end in a . add one
	    else {
                $defmailer =~ s/\.*$/\./;
                set_mx("$dbfile,$domain,",$ttl,5,$defmailer);
	    }
	}
    }
}
sub triplet2fqdn {
local($host) = @_;
local($relhost, $origin,$dbfile,$fqdn);

	($dbfile,$origin,$relhost) = split (',',$host);
	#
	# Hostname is just the origin
	#
	if ( $relhost eq "" ) {
		$fqdn = $origin;
		}
	#
	# Hostname is already fully qualified
	#
	elsif ( $relhost =~ /.*\.$/ ) {
		$fqdn = $relhost;
		}
	#
	# Hostname is relative to origin so stick on origin
	#
	else	{
		$fqdn = "$relhost.$origin";
		}
	}

sub parse_line {
	local($inline) = @_;
	local($i,$j,@array,@list,@restline);

	@array = split('"',$inline);

	for ( $i = 0; $i <= $#array; $i++ ) {
		if ( $i % 2 == 0 ) {
			@list = split (/[ \t\n]+/,$array[$i]);
			for ( $j = 0; $j <= $#list; $j++ ) {
				push(@restline,$list[$j]) if $list[$j] ne "";
				}
			}
		else {
			push(@restline,$array[$i]);
			}
		}
	@restline;
	}

sub cleanup_name {
    local($name) = @_;

    $name =~ s/^\s//;         # Cleanup leading white space
    $name =~ s/\s$//;         # Cleanup trailing white spaces

    return "\L$name";
}

sub check_ip {
    my ($stat,$name,$ip) = @_;

    if ( $ip !~ /\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}/ ) {
        $stat->error("$name => contains invalid character(s) $ip");
        return 1;
    }
    return 0;
}

sub check_dname {
    my ($stat,$name) = @_;

    if ( $name =~ /^\./ ) {
        return 1;
    }
}

1;
